/*
** Definitions for Brecis security engine (second hardware version).
** User API portion.
**
** Copyright 2002 Brecis Communications
**
*/

#ifndef MSP_SECV2_H
#define MSP_SECV2_H

/* old file included mostly for definition of SECDEV */
#include <brecis/msp_sec.h>

/*
** details that are of use outside of the driver
*/

/*
** The security association structure
*/
typedef struct {
	unsigned int flags ;
	unsigned int esp_spi ;
	unsigned int esp_sequence ;
	unsigned int hash_chain_a[5] ;
	unsigned int crypt_keys[8] ;
	unsigned int hash_chain_b[5] ;
	unsigned int hash_init_len[2] ;
	unsigned int crypt_iv[4] ;
	unsigned int proto_ip[5] ;
} MSP_SEC2_SA ;

/* flags field values for SA struct */
#define SAFLG_MODE_MASK		0x7
#define SAFLG_MODE_ESP_IN	0
#define SAFLG_MODE_ESP_OUT	1
#define SAFLG_MODE_HMAC		2
#define SAFLG_MODE_HASH_PAD	3
#define SAFLG_MODE_HASH		4
#define SAFLG_MODE_CRYPT	5

#define SAFLG_SI		0x80 /* increment sequence number */
#define SAFLG_CRI		0x100 /* Create IV */
#define SAFLG_CPI		0x200 /* Compare ICV */
#define SAFLG_EM		0x400 /* ESP Manual Mode */
#define SAFLG_CV		0x800 /* Use Chaining Variables */

#define SAFLG_HASH_MASK		0xe000
#define SAFLG_MD5_96		0x0000
#define SAFLG_MD5		0x2000
#define SAFLG_SHA1_96		0x4000
#define SAFLG_SHA1		0x6000
#define SAFLG_HASHNULL		0x8000

#define SAFLG_KEYS_MASK		0x70000
#define SAFLG_DES_K1_DECRYPT	0x10000
#define SAFLG_AES_DECRYPT	SAFLG_DES_K1_DECRYPT /* same bit */
#define SAFLG_DES_K2_DECRYPT	0x20000
#define SAFLG_DES_K3_DECRYPT	0x40000

#define SAFLG_BLK_MASK		0x380000
#define SAFLG_ECB		0
#define SAFLG_CTR		0x080000
#define SAFLG_CBC_ENCRYPT	0x100000
#define SAFLG_CBC_DECRYPT	0x180000
#define SAFLG_CFB_ENCRYPT	0x200000
#define SAFLG_CFB_DECRYPT	0x280000
#define SAFLG_OFB		0x300000

#define SAFLG_CRYPT_TYPE_MASK	0x1C00000
#define SAFLG_DES		0
#define SAFLG_3DES		0x0400000
#define SAFLG_AES_128		0x0800000
#define SAFLG_AES_192		0x0C00000
#define SAFLG_AES_256		0x1000000
#define SAFLG_CRYPTNULL		0x1400000


/*
** Defines for workq entry
*/
/* control word */
#define SEC2_WE_CTRL_SZ		0x0ff
#define SEC2_WE_CTRL_CQ		0x100
#define SEC2_WE_CTRL_GI		0x800
#define SEC2_WE_CTRL_AKO	0x8000
#define SEC2_WE_CTRL_NXTHDR_SHF	16
#define SEC2_WE_CTRL_PADLEN_SHF	24

#define CBK_NONE	( (void (*)(void*, unsigned int)) 0 )
#define CBK_POLL	( (void (*)(void*, unsigned int)) 1 )
#define CBK_SLEEP	( (void (*)(void*, unsigned int)) 2 )

#define BLK_NONE	0
#define BLK_POLL	1
#define BLK_SLEEP	2

/* scatter/gather flags */
#define SEC2_WE_SG_SCATTER	0x80000000
#define SEC2_WE_SG_SOP		0x40000000
#define SEC2_WE_SG_EOD		0x20000000
#define SEC2_WE_SG_EOP		0x10000000
#define SEC2_WE_SG_SIZE		0x00001FFF

/* for calls to the driver */
#define SG_GATHER		0
#define SG_SCATTER		0x80000000
/*
** Userland access.
**
** Linux has no "callback" facility to user space.  At least nothing
** lightweight.  So, submitting requests to the kernel will block
** until they complete.
**
** The SA structure will exist in user space, and the user will
** write directly to it.  The user will form a work queue entry
** (with SA pointing to its own memory space) and will send a pointer
** to it down to the kernel via an ioctl call.
**
** The kernel simply copies the work queue entry into the work queue,
** substituting its own callback function, then goes to sleep or polls
** for completion.  When awakened, or when the poll returns successfully,
** the ioctl returns to user space.
*/

/* IOCTL values */

#define MSP_SEC_CTLV2		('B' << 8 | 0x0003)
#define MSP_SEC_SET_HMAC	('B' << 8 | 0x0004)
#define MSP_SEC_SET_AES		('B' << 8 | 0x0005)

/*
** The work queue entry structure.  It is built on the fly in the
** kernel API, so user level API the only one that actually needs the
** definition.
*/
struct work_queue_sg {
	void		*addr ;
	unsigned int	size_flags ;
} ;

#define MSP_SEC2_SGE_MAX	16

typedef struct work_queue_entry {
	int wqn ;		/* work queue number */
	int next_sge ;		/* next sge entry  */
	int status ;		/* status returned from kernel */
	MSP_SEC2_SA		*sa ;
	unsigned int		control ;
	unsigned int		sw0 ;	/* software use word 0 */
	unsigned int		sw1 ;   /* software use word 1 */
	struct work_queue_sg	sge[MSP_SEC2_SGE_MAX] ;
} MSP_SEC2_WQE ;

typedef struct {
	MSP_SEC2_SA		*sa ;
	char			*key ;
	int			keylen ;
	int			workq ;
	int			compq ;
} MSP_SEC2_SET_HMAC_ARGS ;

typedef struct {
	MSP_SEC2_SA		*sa ;
	int			workq ;
	int			compq ;
} MSP_SEC2_SET_AES_ARGS ;
		

#ifndef __KERNEL__
int __sec_fd ;

static inline int
MSP_SEC_ENQUIRE(int *info)
{
	if (__sec_fd <= 0)
		__sec_fd = open(SECDEV,0) ;
	if (__sec_fd <= 0)
	{
		perror(SECDEV) ;
		return -1 ;
	}

	return (ioctl(__sec_fd, MSP_SEC_ENQ, info)) ;
}	

static inline void
MSP_SEC2_NEW_REQUEST(MSP_SEC2_WQE *wqep,
		     int work_q,
		     MSP_SEC2_SA *sa,
		     int control)
{
	wqep->wqn = work_q ;
	wqep->next_sge = 0 ;
	wqep->sa = sa ;
	wqep->control = (control) ;
}

static inline int
MSP_SEC2_ADD_SG(MSP_SEC2_WQE *wqep,
		int scatter,
		void *address,
		int size)
{
	if (wqep->next_sge >= MSP_SEC2_SGE_MAX)
		return -1 ;

	wqep->sge[wqep->next_sge].addr = address ; 
	if (scatter) { 
		size |= SEC2_WE_SG_SCATTER ; 
	}
	wqep->sge[wqep->next_sge].size_flags = size ; 
	wqep->next_sge++ ;

	return 0 ;
}

static inline int
MSP_SEC2_END_REQUEST(MSP_SEC2_WQE *wqep)
{
	if (__sec_fd <= 0)
		__sec_fd = open(SECDEV,0) ;
	if (__sec_fd <= 0)
	{
		perror(SECDEV) ;
		return -1 ;
	}
	return ioctl(__sec_fd, MSP_SEC_CTLV2, wqep) ;
}

#define MSP_SEC2_ABORT_REQUEST(wqep)


static inline int
MSP_SEC2_SET_HMAC_KEY(		MSP_SEC2_SA *sa,
				unsigned char *key,
				int keylen,
				int workq,
				int compq)
{
	MSP_SEC2_SET_HMAC_ARGS args ;

	if (__sec_fd <= 0)
		__sec_fd = open(SECDEV,0) ;
	if (__sec_fd <= 0)
	{
		perror(SECDEV) ;
		return -1 ;
	}

	args.sa     = sa ;
	args.key    = key ;
	args.keylen = keylen ;
	args.workq  = workq ;
	args.compq  = compq ;

	return ioctl(__sec_fd, MSP_SEC_SET_HMAC, &args) ;
}

static inline int
MSP_SEC2_SET_AES_DECRYPT_KEY(   MSP_SEC2_SA *sa,
				int workq,
				int compq)
{
	MSP_SEC2_SET_AES_ARGS args ;

	if (__sec_fd <= 0)
		__sec_fd = open(SECDEV,0) ;
	if (__sec_fd <= 0)
	{
		perror(SECDEV) ;
		return -1 ;
	}

	args.sa     = sa ;
	args.workq  = workq ;
	args.compq  = compq ;

	return ioctl(__sec_fd, MSP_SEC_SET_AES, &args) ;
}

#endif  /* ! __KERNEL__ */

#ifdef __KERNEL__
/*
** Kernel level access
*/

/*
** It is necessary that only one process be placing work requests in
** the queue at a time.  Perhaps you know it's a uniprocessor, and you
** know that you only write the queue from task level in kernel space,
** so you will never be interrupted while writing to the queue, then
** you need no locking.  But maybe you need to do things from
** interrupt level sometimes.  If that's the case, you need to mask
** interrupts while creating the queue entry.
**
** The driver will by default mask interrupts between calls to
** msp_sec2_new_request and msp_sec2_end_request (or abort).
** If you know you don't need this masking, you can use
** msp_sec2_set_q_options to turn it off.
*/

/*
** msp_sec2_set_q_options:  set options for a queue
**
** input:
**    work_q           Which queue to put this request on
**    opt              Which option to set
**    val              What value to set the option to
**
** returns:
**    0 -- success.
**    1 -- unknown option number 'opt'
**    2 -- Value 'val' is out of range for option 'opt'
*/

int
msp_sec2_set_q_options(		int work_q,
				int opt,
				int val ) ;

#define WQ_OPT_MASKINT	0

/*
** msp_sec2_new_request:  start adding a new request to a work queue
**
** input:
**    work_q           Which queue to put this request on
**    n_sg_entries     How many Scatter/Gather entries this request will use
**    sa               Security association to use with this request
**    control          Control word to use for this request
**                         (descriptor size is set by these functions, though)
**    callback_fn      function to call when operation completes
**    cbk_prm          parameter passed to callback function
**    block            BLK_NONE, BLK_POLL, or BLK_SLEEP -- return, poll,
**                         or sleep when no space available
**
** returns:
**    0 -- success.
**    other - failure: not enough space or other failure
**
** note:
**    This is the only time space is checked for (with
**    n_sg_entries).  You must not call msp_sec2_add_sg() more times
**    than you indicated when you call this function.  You may,
**    however, call it fewer than n_sg_entries; the descriptor will
**    be adjusted.
*/

int
msp_sec2_new_request(	int work_q, 
			int n_sg_entries ,
			MSP_SEC2_SA *sa, 
			int control,
			void (*callback_fn)(void *,unsigned int),
			void *cbk_prm,
			int block) ;



/*
** msp_sec2_add_sg: Add a scatter/gather entry to the descriptor being built.
**
** Scatter is SG_SCATTER if this is a scatter descriptor, SG_GATHER if a
** gather descriptor.  Address is the address.  Size is the size;
** this must be <= 4095 as the hardware only uses 12 bits; this
** routine will not check the length, do it yourself.
**
** You must only call this function between msp_sec2_new_request() and
** either msp_sec2_end_request() or msp_sec2_abort_request(), with any
** necessary locking held the whole time.
*/

void
msp_sec2_add_sg(	int work_q, 
			int scatter,
			void * address,
			int size ) ;

/*
** msp_sec2_end_request: Finish building work entry and start hardware.
**
** You SHOULD write your software to assume that the callback
** function may be called even before this routine returns.  (This
** may be necessary if we ever try to emulate the new API on old
** hardware.)
**
** Once this routine returns, you are safe to release any necessary
** locking.
*/

void
msp_sec2_end_request( int work_q ) ;

/*
** msp_sec2_abort_request: abandons the work request being built
**
** I don't expect this will be used, as you're better off making the
** decision ahead of time and avoiding starting to build the
** descriptor in the first place.  But, just in case, it's here.
**
** Once this routine returns, you are safe to release any necessary
** locking.
*/

void
msp_sec2_abort_request( int work_q ) ;

/* msp_sec2_set_hmac_key: set up an SA with the partial hashes
** necessary for HMAC (and ESP with HMAC) operation.
**
** input:
**    sa              sa structure to be filled with partial hashes
**                    [setting of hash type filed in sa->flags is
**                    used to determine which type of hash (MD5 or
**                    SHA1) is used for the partial hashes]
**    key             pointer to key data
**    keylen          length (in bytes) of the key data
**    workq           which work queue to use for the hashing operations
**    compq           which completion queue to use for the operations
**    sleep           If zero, busy wait/poll for results; if non-zero,
**                    put task to sleep waiting for results.
** returns:
**    0 -- success
**    -1 -- failure for any reason (bad key pointer, bad keylen value,
**          bad hash type in sa->flags, etc.)
**    On return, sa->hash_chain_a, sa->hash_chain_b, and sa->hash_init_len
**    are filled in with correct values for HMAC operations.
**
** note:
**    Calls with parameter 'sleep' != 0 must be done from task level!
*/

int
msp_sec2_set_hmac_key(		MSP_SEC2_SA *sa,
				unsigned char *key,
				int keylen,
				int workq,
				int compq,
				int sleep) ;


/* msp_sec2_set_aes_decrypt_key -- do aes decrypt key pre-processing.
**
** input:
**    sa              sa structure to be filled with AES decrypt key
**                    (sa->flags encryption mode must be one of the
**                    AES modes, and sa->crypt_keys field should
**                    be filled with the original key material;
**                    the crypt_keys field will be overwritten with
**                    the correct decrypt key when this preprocessing
**                    is complete.
**    workq           which work queue to use for the operation
**    compq           which completion queue to use for the operation
**    sleep           If zero, busy wait/poll for results; if non-zero,
**                    put task to sleep waiting for results.
** returns:
**    0 -- success
**    -1 -- failure for any reason (bad crypt type in sa->flags, etc.)
**    On return, sa->crypt_keys is filled in with the decrypt key.
**
** note:
**    Calls with parameter 'sleep' != 0 must be done from task level!
*/

int
msp_sec2_set_aes_decrypt_key(	MSP_SEC2_SA *sa,
				int workq,
				int compq,
				int sleep) ;

/* msp_sec2_poll_completion_queues: emptys the completion queues.
**
** Can be called from an interrupt routine, from a convenient place to
** poll (like an idle loop), or from a busy-waiting request.
**
** Queue_mask should have a bit set for each completion queue you wish
** to poll.  0x01 is the first completion queue, and 0x02 is the
** second.
**
** If max_req is non-zero, at most max_req completion records will be
** acted on and removed from the completion queue(s).  If it is zero,
** the selected queues will be repeatedly processed until they are
** empty.
**
** Three possible return values:
**
** -1 - no work to be done on entering
**  0 - some work was done, queue(s) are now empty
**  1 - some work was done, still something left to do
**
*/

int
msp_sec2_poll_completion_queues(int queue_mask, int max_req) ;

#endif

#endif /* MSP_SECV2_H */
