/*
** Definitions for Brecis security engine.  Kernel-only portion.
**
** Copyright 2001,2002 Brecis Communications
**
*/
#ifndef BRECIS_MSP_SEC_KERN_H
#define BRECIS_MSP_SEC_KERN_H


#define	SEC_BASE	0xB8800000	/* Base address to Security Engine */
#define	DES_NDA_REG   (SEC_BASE + 0x000)  /* Pointer to next descriptor    */
#define	HSH_NDA_REG   (SEC_BASE + 0x050)  /* Pointer to next hash desc.    */
#define SEC_RNG_CNF   (SEC_BASE + 0x084)  /* random number gen config */
#define SEC_RNG_VAL   (SEC_BASE + 0x094)  /* random number gen value */

/*************************************************************
** Saved Current Descriptor Address
*************************************************************/
#define	DES_CDA_REG   (volatile unsigned int *) (SEC_BASE + 0x098)
#define	HSH_CDA_REG   (volatile unsigned int *) (SEC_BASE + 0x09C)


/*************************************************************
** Main DES and HASH register layouts
*************************************************************/
typedef struct {
	volatile void                    *next_desc_address ;
	volatile unsigned long           byte_count ;
	volatile unsigned char           *source_address ;
	volatile unsigned char           *dest_address ;
	volatile unsigned long           mode_control ;
	volatile unsigned long           descriptor_id ;
	volatile unsigned long           init_vector_high ;
	volatile unsigned long           init_vector_low ;
	volatile unsigned long           key1_high ;
	volatile unsigned long           key1_low ;
	volatile unsigned long           key2_high ;
	volatile unsigned long		 key2_low ;
	volatile unsigned long           key3_high ;
	volatile unsigned long		 key3_low ;
	volatile unsigned long           interface_control ;
	volatile unsigned long           interface_status ;
	volatile unsigned long           input_data_high ;
	volatile unsigned long           input_data_low ;
	volatile unsigned long           output_data_high ;
	volatile unsigned long           output_data_low ;
	
} msp_sec_des_regs ;

typedef struct {
	volatile void                    *next_desc_address ;
	volatile unsigned long           byte_count ;
	volatile unsigned char           *source_address ;
	volatile unsigned char           *dest_address ;
	volatile unsigned long           mode_control ;
	volatile unsigned long           descriptor_id ;
	volatile unsigned long           error_header1 ;
	volatile unsigned long           error_header2 ;
	volatile unsigned long           chain_var_a ;
	volatile unsigned long           chain_var_b ;
	volatile unsigned long           chain_var_c ;
	volatile unsigned long		 chain_var_d ;
	volatile unsigned long           chain_var_e ;
	volatile unsigned long		 reserved1 ;
	volatile unsigned long           interface_control ;
	volatile unsigned long           interface_status ;
	volatile unsigned long           input_data ;
	volatile unsigned long           reserved2 ;
	
} msp_sec_hsh_regs ;


/*--------------------------------------------------------------------------+
| DES Next Descriptor Address Register:
|
|     Bit 0    =  Address Valid
|                 0 = Address is not valid
|                 1 = Address is valid
|
|     Bit 1    =  Continuation
|                 0 = Next descriptor starts a new segment
|                 1 = Next descriptor is a continuation of the current seg.
|
|     Bit 31:2 =  Next descriptor address aligned to word boundary.
+--------------------------------------------------------------------------*/
#define DNDA_VLD	0x1		/* Address Valid indicator         */
#define DNDA_CNT	0x2		/* NDA Continuation indicator      */


/*--------------------------------------------------------------------------+
|
| DES Interface Control Register:
|
|     Bit 0    =  DMA start/continue (has priority over PIO enable)
|                 0 = No operation
|                 1 = Start/continue DMA
|
|     Bit 1    =  PIO Enable
|                 0 = PIO Mode Disabled
|                 1 = PIO Mode Enabled
|
|     Bit 2    =  DES Engine Reset
|                 0 = No reset
|                 1 = Reset DES Engine
|
|     Bit 3    =  Halt DMA
|                 0 = No halt
|                 1 = Halt DMA operation
|
|     Bit 4    =  Stop DMA upon Segment/message complete
|                 0 = Disable
|                 1 = Stop DMA engine after current
|                     segment/message completed
|
|     Bit 5    =  Interrupt on DMA Error
|                 0 = Disable DMA error interrupt
|                 1 = Force interrupt when DMA error condition is dected
|                     and global interrupt is enable
|
|     Bit 7    =  Force Interrupt When Segment Done  (current descriptor)
|                 0 = Disable
|                 1 = Force interrupt when Segment Complete
|                     and global interrupt is enable
|
|     Bit 6    =  Force Interrupt When Message Done (entire link list)
|                 0 = Disable
|                 1 = Force interrupt when message Complete
|                     and global interrupt is enable
|
|     Bit 8    =  Global Interrupt Enable
|                 0 = Disable all interrupt
|                 1 = Enable interrupts
|
+--------------------------------------------------------------------------*/
#define	DIFC_DMA_START	0x0001          /* DMA Start/Continue              */
#define	DIFC_PIO_MODE	0x0002          /* PIO Mode                        */
#define	DIFC_DES_RST	0x0004          /* DES Engine reset                */

#define	DIFC_DMA_HLT	0x0008          /* DMA Halt Control                */
#define	DIFC_DMA_STP	0x0010          /* Stop DMA when current op done   */

#define	DIFC_INT_DER	0x0020          /* Interrupt upon DMA Error        */
#define	DIFC_INT_MC	0x0040          /* Interrupt upon Message Complete */
#define	DIFC_INT_SC	0x0080          /* Interrupt upon Segment Complete */
#define	DIFC_INT_EN	0x0100          /* Global Interrupt Enable         */


/*--------------------------------------------------------------------------+
| DES Interface Status Register:  Read Only Register
|
|     Bit 0    =  DES Engine Busy
|                 0 = DES Engine idle
|                 1 = DES Engine busy (PIO mode or DMA not reset)
|
|     Bit 1    =  DES Output Data Valid
|                 This bit is set to indicate DES has completed its
|                 operation and the DES Output Data registers hold the
|                 results. It is reset when it detects both read requests
|                 to Output Data registers.
|
|     Bit 2    =  DES DMA On
|                 0 = DES DMA Idle
|                 1 = DES DMA In Progress
|
|     Bit 3    =  DES Message Complete Indicator
|                 This bit indicates DES operation for the entire descriptor
|                 link list is completed.  It is reset by hardware when
|                 another start DMA is detected in DES IFC.
|
|     Bit 4    =  DES Segment Complete Indicator
|                 This bit indicates DES operation for the current
|                 descriptor block is completed.  It is reset by hardware
|                 when another start DMA is detected in DES IFC.
|
|     Bit 5    =  DES Interrupt Pending on DMA Error
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by DMA Encounterred Error Condition. It must be
|                 cleared by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
|     Bit 6    =  DES Interrupt Pending on Message Completed
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by message complete Condition. It must be cleared
|                 by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
|     Bit 7    =  DES Interrupt Pending on Segment Completed
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by segment complete Condition. It must be cleared
|                 by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
|
|     HASH Shadow status: (identical functions as DES status)
|     -------------------
|
|     Bit 8    =  Hash Engine Busy
|     Bit 9    =  Hash Output Data Valid
|     Bit 10   =  Hash DMA On
|     Bit 11   =  Hash Message Complete Indicator
|     Bit 12   =  Hash Segment Complete Indicator
|     Bit 13   =  Hash Interrupt Pending on DMA Error
|     Bit 14   =  Hash Interrupt Pending on Message Completed
|     Bit 15   =  Hash Interrupt Pending on Segment Completed
|
+--------------------------------------------------------------------------*/
#define DIFS_BSY	0x0001		/* DES Busy indicator              */
#define DIFS_ODV	0x0002		/* DES Output data valid indicator */
#define DIFS_DON	0x0004		/* DES DMA On                      */
#define DIFS_MC		0x0008		/* DES Message Complete            */
#define DIFS_SC		0x0010		/* DES Segment Complete            */
#define DIFS_IDR	0x0020		/* DES IntrPending on DMA Err      */
#define DIFS_IMC	0x0040		/* DES IntrPending on Message Done */
#define DIFS_ISC	0x0080		/* DES IntrPending on Segment Done */

#define DIFS_HBSY	0x0100		/* Hsh Busy indicator              */
#define DIFS_HODV	0x0200		/* Hsh Output data valid indicator */
#define DIFS_HDON	0x0400		/* Hsh DMA On                      */
#define DIFS_HMC	0x0800		/* Hsh Message Complete            */
#define DIFS_HSC	0x1000		/* Hsh Segment Complete            */
#define DIFS_HIDR	0x2000		/* Hsh IntrPending on DMA Err      */
#define DIFS_HIMC	0x4000		/* Hsh IntrPending on Message Done */
#define DIFS_HISC	0x8000		/* Hsh IntrPending on Segment Done */

#define DIFS_IIM	0x10000000	/* DES IntrPending Incomplete Msg  */
#define DIFS_BAE	0x80000000	/* DES/HSH Bad Address             */

#define DIFS_RBC_MASK	0x0FFF0000	/* DES Remaining Byte Count        */




/*--------------------------------------------------------------------------+
| HASH Nex Descriptor Address Register:
|
|     Bit 0    =  Address Valid
|                 0 = Address is not valid
|                 1 = Address is valid
|
|     Bit 1    =  Continuation
|                 0 = Next descriptor starts a new segment
|                 1 = Next descriptor is a continuation of the current seg.
|
|     Bit 31:2 =  Next descriptor address aligned to word boundary.
+--------------------------------------------------------------------------*/
#define HNDA_VLD	0x1		/* Address Valid indicator         */
#define HNDA_CNT	0x2		/* NDA Continuation indicator      */


/*--------------------------------------------------------------------------+
|
| Hash Interface Control Register:
|
|     Bit 0    =  DMA start/continue (has priority over PIO enable)
|                 0 = No operation
|                 1 = Start/continue DMA
|
|     Bit 1    =  PIO Enable
|                 0 = PIO Mode Disabled
|                 1 = PIO Mode Enabled
|
|     Bit 2    =  Hash Engine Reset
|                 0 = No reset
|                 1 = Reset Hash Engine
|
|     Bit 3    =  Halt DMA
|                 0 = No halt
|                 1 = Halt DMA operation
|
|     Bit 4    =  Stop DMA upon Segment/message complete
|                 0 = Disable
|                 1 = Stop DMA engine after current
|                     segment/message completed
|
|     Bit 5    =  Interrupt on DMA Error
|                 0 = Disable DMA error interrupt
|                 1 = Force interrupt when DMA error condition is dected
|                     and global interrupt is enable
|
|     Bit 6    =  Force Interrupt When Message Done (entire link list)
|                 0 = Disable
|                 1 = Force interrupt when message Complete
|                     and global interrupt is enable
|
|     Bit 7    =  Force Interrupt When Segment Done  (current descriptor)
|                 0 = Disable
|                 1 = Force interrupt when Segment Complete
|                     and global interrupt is enable
|
|     Bit 8    =  Global Interrupt Enable
|                 0 = Disable all interrupt
|                 1 = Enable interrupts
|
+--------------------------------------------------------------------------*/
#define	HIFC_DMA_START	0x0001          /* DMA Start/Continue              */
#define	HIFC_PIO_MODE	0x0002          /* PIO Mode                        */
#define	HIFC_HSH_RST	0x0004          /* Hash Engine reset               */

#define	HIFC_DMA_HLT	0x0008          /* DMA Halt Control                */
#define	HIFC_DMA_STP	0x0010          /* Stop DMA when current op done   */

#define	HIFC_INT_DER	0x0020          /* Interrupt upon DMA Error        */
#define	HIFC_INT_MC	0x0040          /* Interrupt upon Message Complete */
#define	HIFC_INT_SC	0x0080          /* Interrupt upon Segment Complete */
#define	HIFC_INT_EN	0x0100          /* Global Interrupt Enable         */



/*--------------------------------------------------------------------------+
| Hash Interface Status Register:  Read Only Register
|
|     DES Shadow status: (indical in function to Hash status)
|     -----------------
|     Bit 0    =  DES Engine Busy
|     Bit 1    =  DES Output Data Valid
|     Bit 2    =  DES DMA On
|     Bit 3    =  DES Message Complete Indicator
|     Bit 4    =  DES Segment Complete Indicator
|     Bit 5    =  DES Interrupt Pending on DMA Error
|     Bit 6    =  DES Interrupt Pending on Message Completed
|     Bit 7    =  DES Interrupt Pending on Segment Completed
|
|
|     HASH status:
|     -----------
|
|     Bit 8    =  Hash Engine Busy
|                 0 = DES Engine idle
|                 1 = DES Engine busy (PIO mode or DMA not reset)
|
|     Bit 9    =  Hash Output Data Valid
|                 This bit is set to indicate Hash has completed its
|                 operation and the Hash Variables[A:E] registers hold the
|                 results. It is reset when it detects both read requests
|                 to Output Data registers.
|
|     Bit 10   =  Hash DMA On
|                 0 = Hash DMA Idle
|                 1 = Hash DMA In Progress
|
|     Bit 11   =  Hash Message Complete Indicator
|                 This bit indicates Hash operation for the entire descriptor
|                 link list is completed.  It is reset by hardware when
|                 another start DMA is detected in Hash IFC.
|
|     Bit 12   =  Hash Segment Complete Indicator
|                 This bit indicates Hash operation for the current
|                 descriptor block is completed.  It is reset by hardware
|                 when another start DMA is detected in Hash IFC.
|
|     Bit 13   =  Hash Interrupt Pending on DMA Error
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by DMA Encounterred Error Condition. It must be
|                 cleared by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
|     Bit 14   =  Hash Interrupt Pending on Message Completed
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by message complete Condition. It must be cleared
|                 by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
|     Bit 15   =  Hash Interrupt Pending on Segment Completed
|                 This bit is set by hardware to indicate a pending interrupt
|                 caused by segment complete Condition. It must be cleared
|                 by software, by writing 0 to this bit position, after
|                 the interrupt has been serviced.
|
+--------------------------------------------------------------------------*/
#define HIFS_DBSY	0x0001		/* DES Busy indicator              */
#define HIFS_DODV	0x0002		/* DES Output data valid indicator */
#define HIFS_DDON	0x0004		/* DES DMA On                      */
#define HIFS_DMC	0x0008		/* DES Message Complete            */
#define HIFS_DSC	0x0010		/* DES Segment Complete            */
#define HIFS_DIDR	0x0020		/* DES IntrPending on DMA Err      */
#define HIFS_DIMC	0x0040		/* DES IntrPending on Message Done */
#define HIFS_DISC	0x0080		/* DES IntrPending on Segment Done */

#define HIFS_BSY	0x0100		/* Hsh Busy indicator              */
#define HIFS_ODV	0x0200		/* Hsh Output data valid indicator */
#define HIFS_DON	0x0400		/* Hsh DMA On                      */
#define HIFS_MC		0x0800		/* Hsh Message Complete            */
#define HIFS_SC		0x1000		/* Hsh Segment Complete            */
#define HIFS_IDR	0x2000		/* Hsh IntrPending on DMA Err      */
#define HIFS_IMC	0x4000		/* Hsh IntrPending on Message Done */
#define HIFS_ISC	0x8000		/* Hsh IntrPending on Segment Done */

#define HIFS_IIM	0x40000000	/* Hsh IntrPending Incomplete Msg  */
#define HIFS_BAE	0x80000000	/* DES/HSH Bad Address             */

#define HIFS_RBC_MASK	0x0FFF0000	/* DES Remaining Byte Count        */


/*
** descriptor structure used by the driver
*/
typedef struct sec_desc {
	sec_ioctl                 ioctl ;
	struct sec_desc           *next ;
} sec_desc ;

typedef struct desc_list {
	struct sec_desc *head ;
	struct sec_desc *tail ;
} desc_list ;


typedef struct msp_sec_context
{
	int placeholder ;
} msp_sec_context ;

extern int msp_sec_ioctl_kernel(msp_sec_context *context, unsigned int cmd,
				void *arg) ;


#endif /* BRECIS_MSP_SEC_KERN_H */
