/*==========================================================================+
|                                                                           |
|   Title:   SysRegs.h                                                      |
|                                                                           |
|   Author:  Michael Ngo                                                    |
|                                                                           |
|   Date:    09/10/00.                                                      |
|                                                                           |
+===========================================================================+
|   $Id: BrecisSysRegs.h,v 1.9 2003/04/10 22:54:09 mrustad Exp $
+===========================================================================+
|                                                                           |
|   Description:                                                            |
|     This file defines Triad architecture specific constants.              |
|     add all register definition and macro for all register - Phil Le	    |
|                                                                           |
|   Special Notes:                                                          |
|                                                                           |
|                                                                           |
+==========================================================================*/

#ifndef _BRECIS_SYS_REGS_H
#define _BRECIS_SYS_REGS_H

/*--------------------------------------------------------------------------+
| System/Device Identification Registers
+--------------------------------------------------------------------------*/
#define	SREG_BASE	0xBC000000	/* Base address to SLM Block */



#ifdef __ASSEMBLER__
#define	DEV_ID_REG	(SREG_BASE + 0x000)  /* Device-ID Register         */
#define	FWR_ID_REG	(SREG_BASE + 0x004)  /* Firmware-ID Register       */
#define	SYS_ID_REG0	(SREG_BASE + 0x008)  /* System-ID Register-0       */
#define	SYS_ID_REG1	(SREG_BASE + 0x00C)  /* System-ID Register-1       */
#else
typedef	volatile unsigned int	regint;
#define	_align(x)	__attribute__((__aligned__(x)))
typedef struct slmRegs_t {
	regint	dev_id;		/* +0x00 - Device ID */
	regint	fwr_id;		/* +0x04 - Firmware ID */
	regint	sys_id0;	/* +0x08 - System ID 0 */
	regint	sys_id1;	/* +0x0C - System ID 1 */
	regint	sys_rst;	/* +0x10 RO */
	regint	rst_set;	/* +0x14 WO */
	regint	rst_clr;	/* +0x18 WO */
	regint	unused1;	/* +0x1C */
	regint	pcm0_clk;	/* +0x20 */
	regint	pcm1_clk;	/* +0x24 */
	regint	pcm2_clk;	/* +0x28 */
	regint	pcm3_clk;	/* +0x2C */
	regint	pll_div;	/* +0x30 */
	regint	mips_clk;	/* +0x34 */
	regint	ve_clk;		/* +0x38 */
	regint	fe_clk;		/* +0x3C */
	regint	dvb_clk;	/* +0x40 */
	regint	smac_clk;	/* +0x44 */
	regint	perf_clk;	/* +0x48 */
	regint	wd_ctl;		/* +0x4C - Watchdog timer control */
	regint	wd_svc;		/* +0x50 - Watchdog timer service */
	regint	pit_ctl;	/* +0x54 - Programmable timer control */
	regint	pit_ct0;	/* +0x58 - Programmable timer 0 */
	regint	pit_gct0;	/* +0x5C - Programmable timer gate 0 */
	regint	pit_ct1;	/* +0x60 - Programmable timer 1 */
	regint	pit_gct1;	/* +0x64 - Programmable timer gate 1 */
	regint	pit_ct2;	/* +0x68 - Programmable timer 2 */
	regint	pit_gct2;	/* +0x6C - Programmable timer gate 2 */
	regint	int_sta;	/* +0x70 - Interrupt status */
	regint	int_msk;	/* +0x74 - Interrupt mask */
	regint	ve_mbox;	/* +0x78 - VE mailbox */
	regint	pe_mbox;	/* +0x7C - PE mailbox */
	struct elbregs
	{
		regint	config;	/* Chip select config */
		regint	base_addr;	/* Base address register */
		regint	size_mask;	/* Size mask register */
		regint	access_type;	/* Access type (only in [0]-[5]) */
	} elb[8];			/* +0x80 - +0xFF */
#define	single_pc_enable	/* +0xEC - Single PC card enable */\
	elb[6].access_type
#define	elb_clk_wait_config	/* +0FC - ELB clock/wait config register */\
	elb[7].access_type
	regint	unused2[20];	/* +0x100 - +0x14F UART0 */
	regint	ser_sta;	/* +0x150 - System error status */
	regint	ser_h1l;	/* +0x154 - System error status */
	regint	ser_h2l;	/* +0x158 - System error status */
	regint	sys_ie;		/* +0x15C - System interrupt config */
	regint	unused3[14];	/* +0x160 - +0x197 unused */
	regint	elb_reset;	/* +0x198 - ELB reset */
	regint	boot_status;	/* +0x19C - Boot status */
	regint	elb_ext[8];	/* +0x1A0 - +0x1BF - Extended address registers */
}	slmRegs;

extern slmRegs	msp_slm_regs __attribute__((__section__(".register")));


/*
 * dev_id register bits.
 */

#define	DEV_ID_PCI_DIS	(1 << 26)		/* Set if PCI disabled */
#define	DEV_ID_PCI_HOST	(1 << 20)		/* Set if PCI host */
#define	DEV_ID_SINGLE_PC	(1 << 19)	/* Set if single PC Card mode */


/*
 * PCMCIA definitions.
 */

#define	ELB_CONFIG_PCMCIA_INIT	0x9FE4
#define	ELB_CONFIG_PCMCIA_ACCESS	0x8764
#define	ELB_CONFIG_PCMCIA_WAIT	0x2

#define	ELB_ACCESS_TYPE_MEMORY	0
#define	ELB_ACCESS_TYPE_IO	1
#define	ELB_ACCESS_TYPE_ATTR	2

/* Bits in single_pc_enable register */

#define	PCCARD_32	0x02	/* Set if card is PCCARD 32 (Cardbus) */
#define	SINGLE_PCCARD	0x01	/* Set to enable single PC card mode */

#define	PCMCIA_IRQ	1			/* Use IRQ 1 for PCMCIA */
#define	PCMCIA_ATTR_BASE	0xBE000000	/* Attribute base (CS1) */
#define	PCMCIA_MEM_BASE		0xBE400000	/* Memory base (CS2) */
#define	PCMCIA_IO_BASE		0xBE800000	/* I/O base (CS3) */


#define	DEV_ID_REG	((regint *) (SREG_BASE + 0x000))
#define	FWR_ID_REG	((regint *) (SREG_BASE + 0x004))
#define	SYS_ID_REG0	((regint *) (SREG_BASE + 0x008))
#define	SYS_ID_REG1	((regint *) (SREG_BASE + 0x00C))
#endif

#define	MSP5000_ID	0x5001	/* ID for MSP5000 */


/*--------------------------------------------------------------------------+
| Reset Registers
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	SYS_RST_REG	(SREG_BASE + 0x010)  /* system Reset Status Register RO */
#define	RST_SET_REG	(SREG_BASE + 0x014)  /* System set Reset Register    WO */
#define	RST_CLR_REG	(SREG_BASE + 0x018)  /* System Clear Reset Register  WO */
#else
#define	SYS_RST_REG	((regint *) (SREG_BASE + 0x010))
#define	RST_SET_REG	((regint *) (SREG_BASE + 0x014))
#define	RST_CLR_REG	((regint *) (SREG_BASE + 0x018))
#endif




/*--------------------------------------------------------------------------+
| Clock Generation Registers
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	PCM0_CLK_REG	(SREG_BASE + 0x020)  /* PCM0 Clock Generator       */
#define	PCM1_CLK_REG	(SREG_BASE + 0x024)  /* PCM1 Clock Generator       */
#define	PCM2_CLK_REG	(SREG_BASE + 0x028)  /* PCM2 Clock Generator       */
#define PCM3_CLK_REG	(SREG_BASE + 0x02C)  /* PCM3 Clock Generator	   */
#define	PLL_DIV_REG	(SREG_BASE + 0x030)  /* PLL Divider Value          */
#define	MIPS_CLK_REG	(SREG_BASE + 0x034)  /* MIPS Clock Generator       */
#define	VE_CLK_REG	(SREG_BASE + 0x038)  /* Voice Engine Clock Gen     */
#define	FE_CLK_REG	(SREG_BASE + 0x03C)  /* Framer Engine Clock Gen    */
#define	DVB_CLK_REG	(SREG_BASE + 0x040)  /* DV-Bus Clock Generator     */
#define	SMAC_CLK_REG	(SREG_BASE + 0x044)  /* SEC & MAC Clock Generator  */
#define	PERF_CLK_REG	(SREG_BASE + 0x048)  /* Peripheral & ADPCM Clk Gen */
#else
#define	PCM0_CLK_REG	((regint *) (SREG_BASE + 0x020))
#define	PCM1_CLK_REG	((regint *) (SREG_BASE + 0x024))
#define	PCM2_CLK_REG	((regint *) (SREG_BASE + 0x028))
#define	PCM3_CLK_REG	((regint *) (SREG_BASE + 0x02C))
#define	PLL_DIV_REG	((regint *) (SREG_BASE + 0x030))
#define	MIPS_CLK_REG	((regint *) (SREG_BASE + 0x034))
#define	VE_CLK_REG	((regint *) (SREG_BASE + 0x038))
#define	FE_CLK_REG	((regint *) (SREG_BASE + 0x03C))
#define	DVB_CLK_REG	((regint *) (SREG_BASE + 0x040))
#define	SMAC_CLK_REG	((regint *) (SREG_BASE + 0x044))
#define	PERF_CLK_REG	((regint *) (SREG_BASE + 0x048))
#endif

/*--------------------------------------------------------------------------+
| Watchdog and Programmable Timer/Counter Registers
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	WD_CTL_REG	(SREG_BASE + 0x04C)  /* Watchdog Timer Control Reg */
#define	WD_SVC_REG	(SREG_BASE + 0x050)  /* Watchdog Timer Service Reg */
#define	PIT_CTL_REG	(SREG_BASE + 0x054)  /* ProgTimer Control Reg      */
#define	PIT_CT0_REG	(SREG_BASE + 0x058)  /* Programmable Timer 0 Reg   */
#define	PIT_GCT0_REG	(SREG_BASE + 0x05C)  /* Programmable Timer Gate 0  */
#define	PIT_CT1_REG	(SREG_BASE + 0x060)  /* Programmable Timer 1 Reg   */
#define	PIT_GCT1_REG	(SREG_BASE + 0x064)  /* Programmable Timer Gate 1  */
#define	PIT_CT2_REG	(SREG_BASE + 0x068)  /* Programmable Timer 2 Reg   */
#define	PIT_GCT2_REG	(SREG_BASE + 0x06C)  /* Programmable Timer Gate 2  */
#else
#define	WD_CTL_REG	((regint *) (SREG_BASE + 0x04C))
#define	WD_SVC_REG	((regint *) (SREG_BASE + 0x050))
#define	PIT_CTL_REG	((regint *) (SREG_BASE + 0x054))
#define	PIT_CT0_REG	((regint *) (SREG_BASE + 0x058))
#define	PIT_GCT0_REG	((regint *) (SREG_BASE + 0x05C))
#define	PIT_CT1_REG	((regint *) (SREG_BASE + 0x060))
#define	PIT_GCT1_REG	((regint *) (SREG_BASE + 0x064))
#define	PIT_CT2_REG	((regint *) (SREG_BASE + 0x068))
#define	PIT_GCT2_REG	((regint *) (SREG_BASE + 0x06C))
#endif

/*--------------------------------------------------------------------------+
| Interrupt Controller
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	INT_STA_REG	(SREG_BASE + 0x070)  /* Interrupt Status Register  */
#define	INT_MSK_REG	(SREG_BASE + 0x074)  /* Interrupt Mask Register    */
#else
#define	INT_STA_REG	((regint *) (SREG_BASE + 0x070))
#define	INT_MSK_REG	((regint *) (SREG_BASE + 0x074))
#endif


/*--------------------------------------------------------------------------+
| Voice and Framer Mailbox.
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	VE_MBOX_REG	(SREG_BASE + 0x078)  /* Voice Engine Mailbox Reg   */
#define	FE_MBOX_REG	(SREG_BASE + 0x07C)  /* Framer Engine Mailbox Reg  */
#else
#define	VE_MBOX_REG	((regint *) (SREG_BASE + 0x078))
#define	FE_MBOX_REG	((regint *) (SREG_BASE + 0x07C))
#endif



/*--------------------------------------------------------------------------+
| System Error Registers.
+--------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define	SER_STA_REG	(SREG_BASE + 0x150)  /* System Error Status Reg    */
#define	SER_H1L_REG	(SREG_BASE + 0x154)  /* System Error Status Reg    */
#define	SER_H2L_REG	(SREG_BASE + 0x158)  /* System Error Status Reg    */
#define	SYS_IE_REG	(SREG_BASE + 0x15C)  /* System Interrupt Configuration    */
#else
#define	SER_STA_REG	((regint *) (SREG_BASE + 0x150))
#define	SER_H1L_REG	((regint *) (SREG_BASE + 0x154))
#define	SER_H2L_REG	((regint *) (SREG_BASE + 0x158))
#define	SYS_IE_REG	((regint *) (SREG_BASE + 0x15C))
#endif


/*--------------------------------------------------------------------------+
| Peripheral Block
+--------------------------------------------------------------------------*/

#define	PER_BASE	0xB8400000	/* Base address to Peripheral Reg  */

#ifndef __ASSEMBLER__
typedef struct msp_scc_bd_t {
	struct msp_scc_bd_t	*next;	/* Pointer to next BD */
	regint	ctrl;	/* Control/status with data bytes */
	unsigned char	*data;		/* Pointer to data buffer */
}	msp_scc_bd;

typedef struct sccDmaRegs_t {
	regint		dmaCtrl;		/* DMA control */
	msp_scc_bd	* volatile dmaTxCurrentBd;	/* Tx current BD addr */
	msp_scc_bd	* volatile dmaTxNextBd;	/* Tx next BD addr */
	regint		dmaTxBdCtrl;		/* Tx BD control */
	unsigned char	* volatile dmaTxBuffer;	/* Tx buffer addr */
	msp_scc_bd	* volatile dmaRxCurrentBd;	/* Rx current BD addr */
	msp_scc_bd	* volatile dmaRxNextBd;	/* Tx next BD addr */
	regint		dmaRxBdCtrl;		/* Rx BD control */
	unsigned char	* volatile dmaRxBuffer;	/* Rx buffer addr */
}	sccDmaRegs;

typedef struct sccReg_t {
	volatile unsigned char	ctl;
	unsigned char	unused[3];
	volatile unsigned char	data;
	unsigned char	unused2[3];
}	sccReg;

typedef struct perRegs_t
{
	sccDmaRegs	sccDma[2];
#define	sccaDma	sccDma[0]
#define	sccbDma	sccDma[1]

	regint	per_control _align(16);	/* Peripheral control */
	regint	per_status;		/* +0x54 Peripheral status */
	regint	spi_mpi_tx_size;	/* +0x58 SPI MPI Tx size */
	regint	spi_mpi_rx_size;	/* +0x5C SPI MPI Rx size */
	regint	spi_mpi_control;	/* +0x60 SPI MPI control reg */
	regint	spi_mpi_cs;		/* +0x64 SPI MPI cs reg */

	regint	per_error_bit _align(16); /* +0x70 Error bit */
	regint	per_error_mask;	/* +0x74 Peripheral error mask */
	regint	per_header1;	/* +0x78 Header 1 */
	regint	per_header2;	/* +0x7C Header 2 */

	regint	per_int_bit;	/* +0x80 Peripheral interrupt bit */
	regint	per_int_mask;	/* +0x84 Peripheral interrupt mask */
	regint	gpio_int_bit;	/* +0x88 GPIO interrupt bit */
	regint	gpio_int_mask;	/* +0x8C GPIO interrupt mask */

	sccReg	scc[2];		/* +0x90 - 0x9F - SCC registers */
#define	scca	scc[0]
#define	sccb	scc[1]

	regint	i2c_slave_addr;	/* +0xA0 I2C slave address */
	regint	i2c_data;	/* +0xA4 I2C data */
	regint	i2c_control;	/* +0xA8 I2C control */
	regint	i2c_status;	/* +0xAC I2C status */
	regint	i2c_ext_slave;	/* +0xB0 I2c extended slave */
	regint	unused[2];	/* +0xB4 - 0xBB unused */
	regint	soft_reset;	/* +0xBC soft reset */

	regint	spi_mpi_core_data;	/* +0xC0 SPI MPI core data */
	regint	spi_mpi_core_control;	/* +0xC4 SPI MPI code control */
	regint	spi_mpi_core_status;	/* +0xC8 SPI MPI core status */
	regint	spi_mpi_core_ssel;	/* +0xCC SPI MPI core ssel */
	regint	spi_mpi_data_fifo;	/* +0xD0 SPI MPI data fifo */

	regint	gpio_data _align(16);	/* +0xE0 GPIO data */
	regint	gpio_config1;		/* +0xE4 GPIO config 1 */
	regint	gpio_config2;		/* +0xE8 GPIO config 2 */
	regint	gpio_od_config;		/* +0xEC GPIO output drive config */

	regint	pit0_mode;		/* +0xF0 PIT 0 mode */
	regint	pit0_count;		/* +0xF4 PIT 0 count */
	regint	pit0_date;		/* +0xF8 PIT 0 gate */

	regint	pit1_mode;		/* +0xFC PIT 1 mode */
	regint	pit1_count;		/* +0x100 PIT 1 count */
	regint	pit1_date;		/* +0x104 PIT 1 gate */

	regint	pit2_mode;		/* +0x108 PIT 2 mode */
	regint	pit2_count;		/* +0x10C PIT 2 count */
	regint	pit2_date;		/* +0x110 PIT 2 gate */

	/*
	 * Following TDM registers are terribly incomplete - only
	 * those on mod 16 boundaries are provided to fill out the
	 * structure. Add others as you wish...
	 */

	regint	tdm_splxd _align(16);	/* +0x120 TDM register */

	regint	tdm_splxsmask[2] _align(16);	/* +0x130 - 0x137 TDM reg */

	regint	tdm_tx_start_addr _align(16);	/* +0x140 */

	regint	tdm_rx_end_addr _align(16);	/* +0x150 TDM Rx end address */

	regint	tdm_dma_int_mask _align(16);	/* +0x160 TDM DMA int mask */

	regint	gpio_config3 _align(16);	/* +0x170 GPIO config 3 */
	regint	gpio_data_15_9;		/* +0x174 GPIO data 15:9 */
	regint	gpio_data_23_16;	/* +0x178 GPIO data 23:16 */
	regint	gpio_data_31_24;	/* +0x17C GPIO data 31:24 */
	regint	gpio_data_39_32;	/* +0x180 GPIO data 39:32 */
	regint	gpio_data_47_40;	/* +0x184 GPIO data 47:40 */
	regint	gpio_data_54_48;	/* +0x188 GPIO data 54:48 */
	regint	gpio_config4;		/* +0x18C GPIO config 4 */
	regint	gpio_config5;		/* +0x190 GPIO config 5 */
	regint	gpio_config6;		/* +0x194 GPIO config 6 */
	regint	gpio_config7;		/* +0x198 GPIO config 7 */
	regint	gpio_od_config2;	/* +0x19C GPIO output drive config */

}	perRegs;

extern perRegs msp_per_regs __attribute__((__section__(".register")));


/*********************************************************/
/* SCC register definition   				 */
/*********************************************************/
#define SCCA_CNTL_REG		((regint *)(PER_BASE + 0x00))	/* SCCA Control Reg */
#define SCCA_TX_CBD_ADD_REG	((regint *)(PER_BASE + 0x04))	/* SCCA Tx Current BD Add Reg */
#define SCCA_TX_NBD_ADD_REG	((regint *)(PER_BASE + 0x08))	/* SCCA Tx Next BD Add Reg */
#define SCCA_TX_BD_CNTL_REG	((regint *)(PER_BASE + 0x0C))	/* SCCA Control BD Add Reg */
#define SCCA_TX_BD_DADR_REG	((regint *)(PER_BASE + 0x10))	/* SCCA Tx BD Data Address reg */
#define SCCA_RX_CBD_ADD_REG	((regint *)(PER_BASE + 0x14))	/* SCCA RX Current BD Addr reg */
#define SCCA_RX_NBD_ADD_REG	((regint *)(PER_BASE + 0x18))	/* SCCA RX Next BD Address reg */
#define SCCA_RX_BD_CNTL_REG	((regint *)(PER_BASE + 0x1C))	/* SCCA RX BD Control reg */
#define SCCA_RX_BD_DADR_REG	((regint *)(PER_BASE + 0x20))	/* SCCA RX BD Data Address reg */
#define SCCB_CNTL_REG		((regint *)(PER_BASE + 0x24))	/* SCCB Control Reg  */
#define SCCB_TX_CBD_ADD_REG	((regint *)(PER_BASE + 0x28))	/* SCCB Tx Current BD Add Reg */
#define SCCB_TX_NBD_ADD_REG	((regint *)(PER_BASE + 0x2C))	/* SCCB Tx Next BD Add Reg */
#define SCCB_TX_BD_CNTL_REG	((regint *)(PER_BASE + 0x30))	/* SCCB Control BD Add Reg */
#define SCCB_TX_BD_DADR_REG	((regint *)(PER_BASE + 0x34))	/* SCCB Tx BD Data Address reg */
#define SCCB_RX_CBD_ADD_REG	((regint *)(PER_BASE + 0x38))	/* SCCB RX Current BD Addr reg */
#define SCCB_RX_NBD_ADD_REG	((regint *)(PER_BASE + 0x3C))	/* SCCB RX Next BD Address reg */
#define SCCB_RX_BD_CNTL_REG	((regint *)(PER_BASE + 0x40))	/* SCCB RX BD Control reg */
#define SCCB_RX_BD_DADR_REG	((regint *)(PER_BASE + 0x44))	/* SCCB RX BD Data Address reg */

#define SCCA_CNTL_BD_REG	((regint *)(PER_BASE + 0x0C))	/* SCCA Control BD Add Reg */
#define SCCA_TXBD_DADR_REG	((regint *)(PER_BASE + 0x10))	/* SCCA Tx BD Data Address reg */
#define SCCB_CNTL_BD_REG	((regint *)(PER_BASE + 0x30))	/* SCCB Control BD Add Reg */
#define SCCB_TXBD_DADR_REG	((regint *)(PER_BASE + 0x34))	/* SCCB Tx BD Data Address reg */
/*********************************************************/
/*	Peripheral Block Register / SPI MPI		 */
/*********************************************************/
#define PER_CNTL_REG		((regint *)(PER_BASE + 0x50)) 	/* Peripheral) Control Register */
#define PER_STAT_BIT_REG	((regint *)(PER_BASE + 0x54)) 	/* Status bit register */
#define SPI_MPI_TX_SIZE_REG	((regint *)(PER_BASE + 0x58))	/* SPI MPI Tx Size Register */
#define SPI_MPI_RX_SIZE_REG	((regint *)(PER_BASE + 0x5C))	/* SPI MPI Rx Size Register */
#define SPI_MPI_CNTL_REG	((regint *)(PER_BASE + 0x60))	/* SPI MPI Control Register */
#define SPI_MPI_CS_REG		((regint *)(PER_BASE + 0x64))	/* SPI MPI CS Register */

/**********************************************************/
/* Peripheral Block Register / Error Register 		*/
/**********************************************************/
#define PER_ERR_BIT_REG		((regint *)(PER_BASE + 0x70))	/* Error Bit Register */
#define PER_ERR_MSK_REG		((regint *)(PER_BASE + 0x74))	/* Error Bit Mask Register */
#define PER_HDR1_REG		((regint *)(PER_BASE + 0x78))	/* Error Header 1 Register */
#define PER_HDR2_REG		((regint *)(PER_BASE + 0x7C))	/* Error Header 2 Register */
/**********************************************************/
/* Peripheral Block Register / Interrupt Register	*/
/**********************************************************/
#define PER_INT_BIT_REG		((regint *)(PER_BASE + 0x80))	/* Interrupt Bit register */
#define PER_INT_MSK_REG		((regint *)(PER_BASE + 0x84))	/* Interrupt Mask Register */
#define GPIO_INT_BIT_REG	((regint *)(PER_BASE + 0x88))	/* GPIO interrupt bit reg */
#define GPIO_INT_MSK_REG	((regint *)(PER_BASE + 0x8C))	/* GPIO interrupt MASK Reg */
/**********************************************************/
/* Serial Communications Control Block Registers 	*/
/**********************************************************/
#define SCCA_ACCESS_DNCL_REG	((volatile U8 *)(PER_BASE + 0x90))	/* SCCA access DMC Low Reg		*/
#define SCCA_ACCESS_DNCH_REG	((volatile U8 *)(PER_BASE + 0x94))	/* SCCA access DNC HI  Reg		*/
#define SCCB_ACCESS_DNCL_REG	((volatile U8 *)(PER_BASE + 0x98))	/* SCCB access DMC Low Reg		*/
#define SCCB_ACCESS_DNCH_REG	((volatile U8 *)(PER_BASE + 0x9C))	/* SCCB access DNC HI  Reg		*/

/**********************************************************/
/* I2C Registers					  */
/**********************************************************/
#define I2C_SLAVE_ADD_REG	((regint *)(PER_BASE + 0xA0))	/* I2C Slave Addr Register */
#define I2C_DATA_REG		((regint *)(PER_BASE + 0xA4))	/* I2C Data Register */
#define I2C_CNTL_REG		((regint *)(PER_BASE + 0xA8))	/* I2C Control Register */
#define I2C_STA_REG		((regint *)(PER_BASE + 0xAC))	/* I2C Status Register */
#define I2C_CCR_REG		(I2C_STA_REG) /* I2C CCR Register */
#define I2C_EXT_SLA_REG		((regint *)(PER_BASE + 0xB0))	/* I2C Extended Slave Reg */
#define SOFT_RST_REG		((regint *)(PER_BASE + 0xBC))	/* Software Reset Reg */
/**********************************************************/
/* SPI/MPI Registers					  */
/**********************************************************/
#define SPI_MPI_CORE_DATA_REG    ((regint *)(PER_BASE + 0xC0))	/* SPI/MPI core Data reg */
#define SPI_MPI_CORE_CNTL_REG    ((regint *)(PER_BASE + 0xC4))	/* SPI/MPI core Control Reg */
#define SPI_MPI_CORE_STAT_REG    ((regint *)(PER_BASE + 0xC8))	/* SPI/MPI core Status Reg */
#define SPI_MPI_CORE_SSEL_REG    ((regint *)(PER_BASE + 0xCC))	/* SPI/MPI core Ssel reg */
#define SPI_MPI_DATA_FIFO_REG    ((regint *)(PER_BASE + 0xD0))	/* SPI/MPI Data Fifo Reg */
#define SPI_MPI_DATA_FIFO_REG_U8 ((volatile U8  *)(PER_BASE + 0xD0))	/* SPI/MPI Data Fifo Reg */
/**********************************************************/
/* GPIO registers					  */
/**********************************************************/
#define GPIO_DATA_REG		((regint *)(PER_BASE + 0xE0))	/* GPIO Data reg */
#define GPIO_CFG1_REG		((regint *)(PER_BASE + 0xE4))	/* GPIO CFG1 reg */
#define GPIO_CFG2_REG		((regint *)(PER_BASE + 0xE8))	/* GPIO CFG2 reg */
#define GPIO_OD_CFG_REG		((regint *)(PER_BASE + 0xEC))	/* GPIO output buffer sel reg */
/**********************************************************/
/* Programmable Timer. counter registers		*/
/**********************************************************/
#define PER_PIT0_MODE_REG	((regint *)(PER_BASE + 0xF0))	/* PIT 0 Mode Register */
#define PER_PIT0_CNT_REG	((regint *)(PER_BASE + 0xF4))	/* PIT 0 Count Register */
#define PER_PIT0_GATE_REG	((regint *)(PER_BASE + 0xF8))	/* PIT 0 Gate Register */
#define PER_PIT1_MODE_REG	((regint *)(PER_BASE + 0xFC))	/* PIT 1 Mode Register */
#define PER_PIT1_CNT_REG	((regint *)(PER_BASE + 0x100))	/* PIT 1 Count Register */
#define PER_PIT1_GATE_REG	((regint *)(PER_BASE + 0x104))	/* PIT 1 Gate Register */
#define PER_PIT2_MODE_REG	((regint *)(PER_BASE + 0x108))	/* PIT 2 Mode Register */
#define PER_PIT2_CNT_REG	((regint *)(PER_BASE + 0x10C))	/* PIT 2 Count Register */
#define PER_PIT2_GATE_REG	((regint *)(PER_BASE + 0x110))	/* PIT 2 Gate Register */
/* Following defines for POLO CPU ONLY */
#define GPIO_CFG3_REG	((regint *)(PER_BASE + 0x170))  /* GPIO's [23:16] */
#define GPIO_CFG4_REG	((regint *)(PER_BASE + 0x18c))  /* GPIO's [31:24] */
#define GPIO_CFG5_REG	((regint *)(PER_BASE + 0x190))  /* GPIO's [39:32] */
#define GPIO_CFG6_REG	((regint *)(PER_BASE + 0x194))  /* GPIO's [47:40] */
#define GPIO_CFG7_REG	((regint *)(PER_BASE + 0x198))  /* GPIO's [54:48] */

#define GPIO_DATA2_REG	((regint *)(PER_BASE + 0x174))  /* GPIO's [15:9]  */
#define GPIO_DATA3_REG	((regint *)(PER_BASE + 0x178))  /* GPIO's [23:16] */
#define GPIO_DATA4_REG	((regint *)(PER_BASE + 0x17c))  /* GPIO's [31:24] */
#define GPIO_DATA5_REG	((regint *)(PER_BASE + 0x180))  /* GPIO's [39:32] */
#define GPIO_DATA6_REG	((regint *)(PER_BASE + 0x184))  /* GPIO's [47:40] */
#define GPIO_DATA7_REG	((regint *)(PER_BASE + 0x188))  /* GPIO's [54:48] */
#endif	/* __ASSEMBLER__ */


/****************************************************************/
/* Register Definition: Peripheral Block Control Status reg     */
/****************************************************************/

/****************************/
/* SPI/MPI Mode		*/
/****************************/
#define SPI_MPI_RX_BUSY		0x00008000	/* SPI/MPI Receive Busy */
#define SPI_MPI_FIFO_EMPTY	0x00004000	/* SPI/MPI Fifo Empty */
#define SPI_MPI_TX_BUSY		0x00002000	/* SPI/MPI Transmit Busy */
#define SPI_MPI_FIFO_FULL	0x00001000	/* SPI/MPU FIFO full */

/****************************************************************/
/* Register Definition: SPI . MPI control Register		*/
/****************************************************************/
#define SPI_MPI_RX_START	0x00000004	/* Start receive command */
#define SPI_MPI_FLUSH_Q		0x00000002	/* Flush SPI/MPI Queue */
#define SPI_MPI_TX_START	0x00000001	/* Start Transmit Command */

/****************************************************************/
/* Register Definition: Programmable Timer. counter register    */
/****************************************************************/
#define TIMER_MODE_0 0		/* Interrupt on Terminal Count */
#define TIMER_MODE_1 1          /* Hardware Retriggerable One Shot */
#define TIMER_MODE_2 2          /* Rate Generator */
#define TIMER_MODE_3 3          /* Square Wave Mode */
#define TIMER_MODE_4 4          /* Software Triggered Strobe */
#define TIMER_MODE_5 5          /* Hardware Triggered Strobe */
#define TIMER_MODE_7 0x7        /* Off (Reset State) */

#define TIMER_NO_INVERSION  0x0
#define TIMER_INVERT_OUTPUT 0x8

#define TIMER_GATE_CONTROL_OFF  0x0
#define TIMER_GATE_CONTROL_ON   0x1

/****************************************************************/
/* Register Definition: SPI/MPI Chip Select Register		*/
/****************************************************************/
#define SPI_MPI_CS0	(1 << 0)  /* MPI/SPI chip select 0 */
#define SPI_MPI_CS1	(1 << 1)  /* MPI/SPI chip select 1 */
#define SPI_MPI_CS2	(1 << 2)  /* MPI/SPI chip select 2 */
#define SPI_MPI_CS3	(1 << 3)  /* MPI/SPI chip select 3 */
#define SPI_MPI_CS4	(1 << 4)  /* MPI/SPI chip select 4 */
#define SPI_MPI_CS5	(1 << 5)  /* MPI/SPI chip select 5 */
#define SPI_MPI_CS6	(1 << 6)  /* MPI/SPI chip select 6 */
#define SPI_MPI_CS7	(1 << 7)  /* MPI/SPI chip select 7 */
#define SPI_MPI_CS8	(1 << 8)  /* MPI/SPI chip select 8 */
#define SPI_MPI_CS9	(1 << 9)  /* MPI/SPI chip select 9 */

#define SEL_SPI		(1 << 10) /* Select SPI Mode */

#define SEL_MPI		(0 << 10) /* Select MPI Mode */

/************************************************************************/
/* Register Definition: SPI/MPI Core control register		        */
/* [0]  =	Reserved.						*/
/* [1]  =	MSTENB		Enable SPI master mode			*/
/* [2]  =	WOR		Wire-OR mode enabled			*/
/* [3]  =	CKPOL		Clock Polarity				*/
/*				1=SCK idles high, 0=SCK idles low	*/
/* [4]	=	PHASE		Phase Select				*/
/* [6:5]=	DVD		Clock divide - 00=8, 01=16, 10=32, 11=64 */
/* [7]	=	IRQENB		Interrupt enable			*/
/************************************************************************/


/************************************************************************/
/*  Register Definition: SPI/MPI Core Status Register			*/
/* [7]	=	IRQ Interrupt active				*/
/*			Set at the end of a master mode transfer,	*/
/*			or when SLVSEL goes high on a slave transfer	*/
/* [6]	=	Overrun						*/
/*			This bit is set when the DOUT register is written */
/*			while an SPI transfer is in progress.		*/
/* [5]	=	COL This bit is set when there is a master mode	*/
/*			collision between multiple SPI masters.		*/
/*			It is set when SLVSEL goes low while MSTENB=1.	*/
/* [2:4]=	zero						*/
/* [1]	=	TXRUN						*/
/*			1=Master mode operation underway. 		*/
/*			This bit is read only.				*/
/* [0]	=	SLVSEL						*/
/*			This bit corresponds to the SLVSEL pin on the VSPI */
/*			core (note that this is				*/
/*			normally interted at the IO pin). read only.	*/
/************************************************************************/

/************************************************************************/
/* SSEL[7:5]								*/
/*	Number of bits to shift in master mode, 			*/
/*	000=8 bits, 001=1 bit, 111=7 bits.				*/
/* SSEL[4:0]								*/
/*	5 individual Slave Selects for master mode			*/
/************************************************************************/


/*------------------------------------------------------------------------*/
/* Register access for All triad Register				  */
/*------------------------------------------------------------------------*/
#ifdef __ASSEMBLER__
#define WRITE_TRIAD_REG(reg,value)	\
	li	t0,reg;			\
	li	t1,reg;			\
	sw	t1,0(t0)
#define READ_TRIAD_REG(reg)		\
	li	t0,reg;			\
	lw	t1,0(t0)
#else
#define WRITE_TRIAD_REG(reg,value)	(*reg = value)
#define READ_TRIAD_REG(reg)		(*reg)
#endif

#define ZSP_FE	 1
#define ZSP_VE	 2
#define BOOTOS	 1
#define BOOTMON  0

#endif /* !(_BRECIS_SYS_REGS_H) */
