/* $Id: brecis_pcmcia.c,v 1.3 2003/02/17 19:37:36 rsewill Exp $ */
/*
 * brecis_pcmcia.c - Support for PCMCIA on BRECIS msp platforms.
 *
 ******************************************************************
 * Copyright (c) 2002 BRECIS Communications
 *	This software is the property of BRECIS Communications
 *	and may not be copied or distributed in any form without
 *	a prior licensing arrangement.
 *
 * BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND
 * FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 *
 ******************************************************************
 *
 * Author:
 *	Mark D. Rustad, BRECIS Communications, 10/07/2002
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 */


/*
 * This code provides support for PCMCIA access in BRECIS msp platforms.
 *
 * The PCMCIA support is quite different from full-function PCMCIA
 * host adapters. It supports only 3.3v and does not support hot-
 * swapping.
 */

#include <linux/types.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <asm/errno.h>
#include <linux/irq.h>

#include <asm/io.h>
#include <asm/brecis/BrecisSysRegs.h>
#include <brecis/NanoDelay.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/ss.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/cistpl.h>
#include "cs_internal.h"

#define	MODNAME	"brecis_pcmcia"

#undef	PCMCIA_DEBUG
#define	PCMCIA_DEBUG	1

#ifdef	PCMCIA_DEBUG
static char	version[] __initdata =
	"brecis_pcmcia.c $Revision: 1.3 $ 2002/10/13 (mrustad)";
#define	DBG(args...)	printk(KERN_NOTICE MODNAME ": " args)
#define	IDBG(args...)	pkinit(KERN_NOTICE MODNAME ": " args)
#else
#define	DBG(args...)
#define	IDBG(args...)
#endif	/* PCMCIA_DEBUG */

MODULE_AUTHOR("Mark Rustad <Mark.Rustad@brecis.com>");
MODULE_DESCRIPTION("BRECIS PCMCIA driver");
MODULE_LICENSE("GPL");

#define	MSP_IO_MAP_SIZE	(8*1024*1024)	/* 8 MB */

#define	MSP_INIT_STATE	(SS_DETECT | SS_3VCARD)

typedef struct msp_socket_t
{
	u_int	irq;
	u_long	mem_base;
	u_long	mem_length;
	void	(*handler)(void *info, u_int events);
	void	*handler_info;
	u_int	ier;		/* Simulated IER */
	u_int	status;
	u_int	old_status;
	u_int	event_state;
	u_int	reset;
	u_int	pending_events;
#ifdef	PCMCIA_DEBUG
	u_int	dumped_cis;
#endif	/* PCMCIA_DEBUG */
//	u_int	ctrl_base;
	socket_state_t	state;
	pccard_io_map	io_maps[MAX_IO_WIN];
	pccard_mem_map	mem_maps[MAX_WIN];
//	struct vm_struct	*io_vma;	/* kernel vm for I/O */
}	msp_socket_t;

#define	MSP_MAX_SOCKETS	1

static msp_socket_t	msp_sockets[MSP_MAX_SOCKETS];

static spinlock_t	msp_pending_event_lock = SPIN_LOCK_UNLOCKED;

static int	single_pc_mode;	/* If in single pc mode */

/* Calculate socket number from ptr into msp_sockets[] */
#define	msp_sockno(sp)	(sp - msp_sockets)

static socket_cap_t	msp_socket_cap =
{
	SS_CAP_PCCARD		/* Support 16 bit cards */
	| SS_CAP_STATIC_MAP,	/* Mappings are fixed in host memory */
	0x2,			/* IRQs */
	MSP_IO_MAP_SIZE * 3,	/* 8 MB fixed window size */
	0,			/* no PCI support */
	0,			/* no CardBus support */
	0			/* no bus operations needed */
};


#ifdef	PCMCIA_DEBUG
static void	cis_hex_dump(const unsigned char *x, int len)
{
	int	i;
	
	for (i = 0 ; i < len ; i++)
	{
		if (!(i & 0xf))
			printk("\n%08x", (unsigned)x);
		printk(" %02x", *(volatile unsigned char *)x);
		x += 2;
	}
	printk("\n");
}
#endif	/* PCMCIA_DEBUG */


#define	MSP_NUM_MAPPED_IRQS	1	/* For now at least */


static struct
{
	/* index is mapped irq number */
	msp_socket_t	*sock;
	hw_irq_controller	*old_handler;
}	msp_mapped_irq[MSP_NUM_MAPPED_IRQS];


static void	msp_socket_enable_ireq(msp_socket_t *sp)
{
	slmRegs	*sreg = (slmRegs *)SREG_BASE;

	DBG("msp_socket_enable_ireq(sock=%d)\n", msp_sockno(sp));

	sreg->int_msk |= (1 << 0);
}


static void	msp_socket_disable_ireq(msp_socket_t *sp)
{
	slmRegs	*sreg = (slmRegs *)SREG_BASE;

	DBG("msp_socket_disable_ireq(sock=%d)\n", msp_sockno(sp));

	sreg->int_msk &= ~(1 << 0);
}



static unsigned int	msp_startup_irq(unsigned int irq)
{
	msp_socket_enable_ireq(msp_mapped_irq[irq].sock);
	msp_mapped_irq[irq].old_handler->startup(irq);
	return 0;
}


static void	msp_shutdown_irq(unsigned int irq)
{
	msp_socket_disable_ireq(msp_mapped_irq[irq].sock);
	msp_mapped_irq[irq].old_handler->shutdown(irq);
}


static void	msp_enable_irq(unsigned int irq)
{
	msp_socket_enable_ireq(msp_mapped_irq[irq].sock);
	msp_mapped_irq[irq].old_handler->enable(irq);
}


static void	msp_disable_irq(unsigned int irq)
{
	msp_socket_disable_ireq(msp_mapped_irq[irq].sock);
	msp_mapped_irq[irq].old_handler->disable(irq);
}


extern struct hw_interrupt_type	no_irq_type;


static void	msp_mask_and_ack_irq(unsigned int irq)
{
	msp_socket_disable_ireq(msp_mapped_irq[irq].sock);
	/* ack_none() spuriously complains about an unexpected IRQ */
	if (msp_mapped_irq[irq].old_handler != &no_irq_type)
		msp_mapped_irq[irq].old_handler->ack(irq);
}


static void	msp_end_irq(unsigned int irq)
{
	msp_socket_enable_ireq(msp_mapped_irq[irq].sock);
	msp_mapped_irq[irq].old_handler->end(irq);
}


static struct hw_interrupt_type	msp_ss_irq_type = {
	typename:	"PCMCIA-IRQ",
	startup:	msp_startup_irq,
	shutdown:	msp_shutdown_irq,
	enable:		msp_enable_irq,
	disable:	msp_disable_irq,
	ack:		msp_mask_and_ack_irq,
	end:		msp_end_irq
};


/*
 * This function should only be called with interrupts disabled.
 */
static void	msp_map_irq(msp_socket_t *sp, unsigned int irq)
{
	DBG("msp_map_irq(sock=%d irq=%d)\n", msp_sockno(sp), irq);
	
	if (irq >= MSP_NUM_MAPPED_IRQS)
		return;

	msp_mapped_irq[irq].sock = sp;
	/* insert ourselves as the irq controller */
	msp_mapped_irq[irq].old_handler = irq_desc[irq].handler;
	irq_desc[irq].handler = &msp_ss_irq_type;
}


/*
 * This function should only be called with interrupts disabled.
 */
static void	msp_unmap_irq(msp_socket_t *sp, unsigned int irq)
{
	DBG("msp_unmap_irq(sock=%d irq=%d)\n", msp_sockno(sp), irq);
	
	if (irq >= MSP_NUM_MAPPED_IRQS)
		return;

	/* restore the original irq controller */
	irq_desc[irq].handler = msp_mapped_irq[irq].old_handler;
}



/*
 * Set Vpp and Vcc (in tenths of a Volt).  Does not
 * support the hi-Z state.
 */

static int	msp_set_voltages(msp_socket_t *sp, int Vcc, int Vpp)
{
//	u_int psr;
	u_int vcci = 0;
	u_int sock = msp_sockno(sp);

	DBG("msp_set_voltage(%d, %d, %d)\n", sock, Vcc, Vpp);

	switch (Vcc)
	{
	case 0:  vcci = 0; break;
	case 33: vcci = 1; break;
//	case 50: vcci = 2; break;
	default:
		DBG("msp_set_voltage: Unsupported voltage\n");
		return 0;
	}

	return 1;
}


/*
 * Interrupt handling routine.
 *
 * This uses the schedule_task() technique to cause reportable events
 * such as card insertion and removal to be handled in keventd's
 * process context.
 */

static void	msp_events_bh(void *dummy)
{
	msp_socket_t	*sp;
	u_int	events;
	int	i;

	DBG("msp_events_bh...\n");
	for (i = 0; i < MSP_MAX_SOCKETS; i++) {
		sp = &msp_sockets[i];

		spin_lock_irq(&msp_pending_event_lock);
		events = sp->pending_events;
		sp->pending_events = 0;
		spin_unlock_irq(&msp_pending_event_lock);

		if (sp->handler) {
			DBG("msp_events_bh: Calling handler, events=%04X\n",
				events);
			sp->handler(sp->handler_info, events);
		}
	}
}


static struct tq_struct	msp_events_task = {
	routine:	msp_events_bh
};


/*
 * Drive the RESET line to the card.
 */

static void	msp_reset_socket(msp_socket_t *sp, int on)
{
	slmRegs	* const sreg = (slmRegs *)SREG_BASE;

	DBG("msp_reset_socket: reset=%d\n", on);

	if (on)
		sreg->elb_reset = sp->reset = 1;
	else {
		sreg->elb_reset = sp->reset = 0;
		sp->status &= ~SS_READY;
//		sp->pending_events |= SS_READY;
//		NanoDelay(100000);
//		schedule_task(&msp_events_task);
	}

}


static int	msp_init(unsigned int sock)
{
	msp_socket_t *sp = &msp_sockets[sock];

	DBG("msp_init(%d)\n", sock);

	sp->pending_events = SS_DETECT;
//	sp->state.Vcc = 33;
//	sp->state.Vpp = 33;
//	msp_set_voltages(sp, 0, 0);

//	schedule_task(&msp_events_task);

	return 0;
}


static int	msp_suspend(unsigned int sock)
{
	DBG("msp_suspend(%d)\n", sock);

	/* TODO */

	return 0;
}


static int	msp_register_callback(unsigned int sock,
	void (*handler)(void *, unsigned int), void *info)
{
	msp_socket_t	*sp = &msp_sockets[sock];

	DBG("msp_register_callback(%d)\n", sock);
	sp->handler = handler;
	sp->handler_info = info;
	if (handler == 0) {
		MOD_DEC_USE_COUNT;
	} else {
		MOD_INC_USE_COUNT;
	}
	return 0;
}


static int	msp_inquire_socket(unsigned int sock, socket_cap_t *cap)
{
	DBG("msp_inquire_socket(%d)\n", sock);

	*cap = msp_socket_cap;
	return 0;
}


static int	msp_get_status(unsigned int sock, u_int *value)
{
	msp_socket_t	*sp = &msp_sockets[sock];
#if 0
	u_int	status = 0;
	unsigned int	isr;

	isr = msp_in(sp, ISR);

	/* Card is seated and powered when *both* CD pins are low */
	if ((isr & HD64465_PCCISR_PCD_MASK) == 0)
	{
		status |= SS_DETECT;	/* card present */

		switch (isr & HD64465_PCCISR_PBVD_MASK)
		{
		case HD64465_PCCISR_PBVD_BATGOOD:
			break;
		case HD64465_PCCISR_PBVD_BATWARN:
			status |= SS_BATWARN;
			break;
		default:
			status |= SS_BATDEAD;
			break;
		}

		if (isr & HD64465_PCCISR_PREADY)
			status |= SS_READY;

		if (isr & HD64465_PCCISR_PMWP)
			status |= SS_WRPROT;

		/* Voltage Select pins interpreted as per Table 4-5 of the std.
		 * Assuming we have the TPS2206, the socket is a "Low Voltage
		 * key, 3.3V and 5V available, no X.XV available".
		 */
		switch (isr & (HD64465_PCCISR_PVS2 | HD64465_PCCISR_PVS1))
		{
		case HD64465_PCCISR_PVS1:
			printk(KERN_NOTICE MODNAME
				": cannot handle X.XV card, ignored\n");
			status = 0;
			break;
		case 0:
		case HD64465_PCCISR_PVS2:
			/* 3.3V */
			status |= SS_3VCARD;
			break;
		case HD64465_PCCISR_PVS2 | HD64465_PCCISR_PVS1:
			/* 5V */
			break;
		}

		/* TODO: SS_POWERON */
		/* TODO: SS_STSCHG */
	}

	DBG("msp_get_status(%d) = %x\n", sock, status);
	
	*value = status;
	return 0;
#else
	if (sp->reset == 0)
		sp->status |= SS_READY;
	*value = sp->status;
	DBG("msp_get_status(%d) = %x\n", sock, sp->status);
	return 0;
#endif	/* 0 */
}


static int	msp_get_socket(unsigned int sock, socket_state_t *state)
{
	msp_socket_t	*sp = &msp_sockets[sock];

	DBG("msp_get_socket(%d)\n", sock);

	*state = sp->state;
	return 0;
}


static void	msp_timer(unsigned long dummy);


static struct timer_list	event_timer =
{
	function: msp_timer
};


static void	msp_timer(unsigned long dummy)
{
	event_timer.data = 0;
	schedule_task(&msp_events_task);
}


enum	{init = 0, detected, ready};

static int	msp_set_socket(unsigned int sock, socket_state_t *state)
{
	msp_socket_t	*sp = &msp_sockets[sock];
	u_long	flags;
	u_int	cscier;
	u_int	changed;
	int	addtimer = 0;

	DBG("msp_set_socket(sock=%d, flags=%x, csc_mask=%x, Vcc=%d, io_irq=%d)\n",
		sock, state->flags, state->csc_mask, state->Vcc, state->io_irq);

	save_and_cli(flags);	/* Don't want interrupts happening here */

	if (state->Vpp != sp->state.Vpp ||
		state->Vcc != sp->state.Vcc) {
		if (!msp_set_voltages(sp, state->Vcc, state->Vpp)) {
			restore_flags(flags);
			return -EINVAL;
		}
		if (state->Vcc != 0 && sp->state.Vcc == 0 && sp->reset == 0)
			sp->status |= SS_READY;
	}

	/*
	 * Handle changes in the Card Status Change mask,
	 * by propagating to the CSCR register
	 */
	changed = sp->state.csc_mask ^ state->csc_mask;

	cscier = sp->ier;

	if (changed & SS_DETECT) {
		if (state->csc_mask & SS_DETECT)
			cscier |= SS_DETECT;
		else
			cscier &= ~SS_DETECT;
	}

	if (changed & SS_READY) {
		if (state->csc_mask & SS_READY)
			cscier |= SS_READY;
		else
			cscier &= ~SS_READY;
	}

#if 0
	if (changed & SS_BATDEAD) {
		if (state->csc_mask & SS_BATDEAD)
			cscier |= SS_BATDEAD;
		else
			cscier &= ~SS_BATDEAD;
	}

	if (changed & SS_BATWARN) {
		if (state->csc_mask & SS_BATWARN)
			cscier |= SS_BATWARN;
		else
			cscier &= ~SS_BATWARN;
	}
#endif	/* 0 */

	if (changed & SS_STSCHG) {
		if (state->csc_mask & SS_STSCHG)
			cscier |= SS_STSCHG;
		else
			cscier &= ~SS_STSCHG;
	}

	sp->ier = cscier;
	DBG("ier = %04X\n", cscier);

	if (sp->state.io_irq && !state->io_irq)
		msp_unmap_irq(sp, sp->state.io_irq);
	else if (!sp->state.io_irq && state->io_irq)
		msp_map_irq(sp, state->io_irq);

	/*
	 * Handle changes in the flags field,
	 * by propagating to config registers.
	 */
	changed = sp->state.flags ^ state->flags;

	if (changed & SS_IOCARD) {
		DBG("card type: %s\n",
			(state->flags & SS_IOCARD ? "i/o" : "memory" ));
//			bool_to_regbit(sp, GCR, HD64465_PCCGCR_PCCT,
//				state->flags & SS_IOCARD);
	}

	if (changed & SS_RESET) {
		DBG("%s reset card\n",
			(state->flags & SS_RESET ? "start" : "stop"));
			msp_reset_socket(sp, (state->flags & SS_RESET) != 0);
			if (state->flags & SS_RESET) {
				sp->status &= ~(SS_RESET | SS_READY);
				sp->event_state = detected;
				if (event_timer.data) {
					del_timer_sync(&event_timer);
					event_timer.data = 0;
				}
			}
			else if (state->Vcc != 0 && sp->state.Vcc == 0)
				sp->status |= SS_READY;

	}

	if (changed & SS_OUTPUT_ENA) {
		DBG("%sabling card output\n",
			(state->flags & SS_OUTPUT_ENA ? "en" : "dis"));
//			bool_to_regbit(sp, GCR, HD64465_PCCGCR_PDRV,
//				state->flags & SS_OUTPUT_ENA);
	}

	sp->state = *state;

	if (sp->state.Vcc == 0)
		sp->status = MSP_INIT_STATE;

	restore_flags(flags);

	switch (sp->event_state)
	{
	case init:
		DBG("event_state init\n");
		if (sp->ier & SS_DETECT & sp->status) {
			sp->status |= SS_DETECT;
			sp->pending_events |= sp->old_status ^ sp->status;
			sp->old_status = sp->status;
			sp->event_state = detected;
#if 0
			schedule_task(&msp_events_task);
#else
			if (event_timer.data != 0) {
				del_timer_sync(&event_timer);
				event_timer.data = 0;
			}

			event_timer.expires = jiffies + 10;
			addtimer = 1;
//			add_timer(&event_timer);
#endif	/* 0 */
		}
		break;

	case detected:
		DBG("event_state detected\n");
		if (sp->ier & SS_READY) {
			sp->status |= SS_READY;
			sp->pending_events |= sp->old_status ^ sp->status;
			sp->old_status = sp->status;
			sp->event_state = ready;
#if 0
			schedule_task(&msp_events_task);
#else
			if (event_timer.data != 0)
				del_timer_sync(&event_timer);

			event_timer.expires = jiffies + 10;
			addtimer = 1;
//			add_timer(&event_timer);
#endif	/* 0 */
		}
		break;

	case ready:
		DBG("event_state ready\n");
		break;

	default:
		DBG("event_state unknown\n");
	}

#ifdef PCMCIA_DEBUG
	if ((state->flags & SS_OUTPUT_ENA) && sp->dumped_cis == 0) {
		cis_hex_dump((const unsigned char*)sp->mem_base, 0x100);
		sp->dumped_cis = 1;
	}
#endif	/* PCMCIA_DEBUG */

	if (addtimer) {
		event_timer.data = 1;
		add_timer(&event_timer);
	}

	return 0;
}


static int	msp_get_io_map(unsigned int sock, struct pccard_io_map *io)
{
	msp_socket_t	*sp = &msp_sockets[sock];
	int	map = io->map;

	DBG("msp_get_io_map(%d, %d)\n", sock, map);
	if (map >= MAX_IO_WIN)
		return -EINVAL;

	*io = sp->io_maps[map];
	return 0;
}


static int	msp_set_io_map(unsigned int sock, struct pccard_io_map *io)
{
	msp_socket_t	*sp = &msp_sockets[sock];
	int	map = io->map;
	struct pccard_io_map	*sio;
//	pgprot_t	prot;

	DBG("msp_set_io_map(sock=%d, map=%d, flags=0x%x, speed=%dns, start=0x%04x, stop=0x%04x)\n",
		sock, map, io->flags, io->speed, io->start, io->stop);
	if (map >= MAX_IO_WIN)
		return -EINVAL;
	sio = &sp->io_maps[map];

	/* check for null changes */
	if (io->flags == sio->flags &&
		io->start == sio->start &&
		io->stop == sio->stop)
		return 0;

#if 0
	if (io->flags & MAP_AUTOSZ)
		prot = PAGE_KERNEL_PCC(sock, _PAGE_PCC_IODYN);
	else if (io->flags & MAP_16BIT)
		prot = PAGE_KERNEL_PCC(sock, _PAGE_PCC_IO16);
	else
		prot = PAGE_KERNEL_PCC(sock, _PAGE_PCC_IO8);
#endif	/* 0 */

	/* TODO: handle MAP_USE_WAIT */
	if (io->flags & MAP_USE_WAIT)
		printk(KERN_INFO MODNAME ": MAP_USE_WAIT unimplemented\n");
	/* TODO: handle MAP_PREFETCH */
	if (io->flags & MAP_PREFETCH)
		printk(KERN_INFO MODNAME ": MAP_PREFETCH unimplemented\n");
	/* TODO: handle MAP_WRPROT */
	if (io->flags & MAP_WRPROT)
		printk(KERN_INFO MODNAME ": MAP_WRPROT unimplemented\n");
	/* TODO: handle MAP_0WS */
	if (io->flags & MAP_0WS)
		printk(KERN_INFO MODNAME ": MAP_0WS unimplemented\n");

	if (io->flags & MAP_ACTIVE) {
		unsigned long	pstart, psize, paddrbase;

		paddrbase = virt_to_phys((void*)(sp->mem_base +
			MSP_IO_MAP_SIZE));
//		vaddrbase = (unsigned long)sp->io_vma->addr;
		pstart = io->start & PAGE_MASK;
		psize = ((io->stop + PAGE_SIZE) & PAGE_MASK) - pstart;

#if 0
		/*
		 * Change PTEs in only that portion of the mapping requested
		 * by the caller.  This means that most of the time, most of
		 * the PTEs in the io_vma will be unmapped and only the bottom
		 * page will be mapped.  But the code allows for weird cards
		 * that might want IO ports > 4K.
		 */
		DBG("remap_page_range(vaddr=0x%08lx, paddr=0x%08lx, size=0x%08lxx)\n",
			vaddrbase + pstart, paddrbase + pstart, psize);
		remap_page_range(vaddrbase + pstart, paddrbase + pstart, psize, prot);
#endif	/* 0 */

		/*
		 * Change the mapping used by inb() outb() etc
		 */
//		msp_port_map(io->start,
//			io->stop - io->start + 1,
//			io->start, 0);
	} else {
//		msp_port_unmap(
//			sio->start,
//			sio->stop - sio->start + 1);
		/* TODO: remap_page_range() to mark pages not present ? */
	}

	return 0;
}


static int	msp_get_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	msp_socket_t	*sp = &msp_sockets[sock];
	int	map = mem->map;

	DBG("msp_get_mem_map(%d, %d)\n", sock, map);
	if (map >= MAX_WIN)
		return -EINVAL;

	*mem = sp->mem_maps[map];
	return 0;
}


static int	msp_set_mem_map(unsigned int sock, struct pccard_mem_map *mem)
{
	msp_socket_t	*sp = &msp_sockets[sock];
	struct pccard_mem_map	*smem;
	int	map = mem->map;
	unsigned long	paddr, size;

	DBG("msp_set_mem_map(sock=%d, map=%d, flags=0x%x, sys_start=0x%08lx, sys_end=0x%08lx, card_start=0x%08x)\n",
		sock, map, mem->flags, mem->sys_start, mem->sys_stop,
		mem->card_start);

	if (map >= MAX_WIN)
		return -EINVAL;
	smem = &sp->mem_maps[map];

	size = mem->sys_stop - mem->sys_start + 1;

	paddr = sp->mem_base;		/* base of Attribute mapping */
	if (!(mem->flags & MAP_ATTRIB))
		paddr += MSP_IO_MAP_SIZE;	/* base of Common mapping */
	paddr += mem->card_start;

	/* Because we specified SS_CAP_STATIC_MAP, we are obliged
	 * at this time to report the system address corresponding
	 * to the card address requested. This is how Socket Services
	 * queries our fixed mapping. I wish this fact had been
	 * documented - Greg Banks.
	 */
	mem->sys_start = paddr;
	mem->sys_stop = paddr + size - 1;

	*smem = *mem;

	return 0;
}


static void	msp_proc_setup(unsigned int sock, struct proc_dir_entry *base)
{
	DBG("msp_proc_setup(%d)\n", sock);
}


#if 0
static int	msp_irq_demux(int irq, void *dev)
{
//	msp_socket_t	*sp = (msp_socket_t *)dev;
//	u_int	cscr;

	DBG("msp_irq_demux(irq=%d)\n", irq);

#if 0
	if (sp->state.io_irq &&
		(cscr = hs_in(sp, CSCR)) & HD64465_PCCCSCR_PIREQ) {
		cscr &= ~HD64465_PCCCSCR_PIREQ;
		msp_out(sp, cscr, CSCR);
		return sp->state.io_irq;
	}
#endif	/* 0 */

	return irq;
}
#endif	/* 0 */


#if 0
static void	msp_interrupt(int irq, void *dev, struct pt_regs *regs)
{
	msp_socket_t	*sp = (msp_socket_t *)dev;
	u_int	events = 0;
	u_int	cscr;

	cscr = msp_in(sp, CSCR);

	DBG("msp_interrupt, cscr=%04x\n", cscr);

	/* check for bus-related changes to be reported to Socket Services */
	if (cscr & HD64465_PCCCSCR_PCDC) {
		/* double-check for a 16-bit card, as we don't support CardBus */
		if ((msp_in(sp, ISR) & HD64465_PCCISR_PCD_MASK) != 0) {
			printk(KERN_NOTICE MODNAME
				": socket %d, card not a supported card type or not inserted correctly\n",
			msp_sockno(sp));
			/* Don't do the rest unless a card is present */
			cscr &= ~(HD64465_PCCCSCR_PCDC|
				HD64465_PCCCSCR_PRC|
				HD64465_PCCCSCR_PBW|
				HD64465_PCCCSCR_PBD|
				HD64465_PCCCSCR_PSC);
		} else {
			cscr &= ~HD64465_PCCCSCR_PCDC;
			events |= SS_DETECT;	/* card insertion or removal */
		}
	}
	if (cscr & HD64465_PCCCSCR_PRC) {
		cscr &= ~HD64465_PCCCSCR_PRC;
		events |= SS_READY;	/* ready signal changed */
	}
	if (cscr & HD64465_PCCCSCR_PBW) {
		cscr &= ~HD64465_PCCCSCR_PSC;
		events |= SS_BATWARN;	/* battery warning */
	}
	if (cscr & HD64465_PCCCSCR_PBD) {
		cscr &= ~HD64465_PCCCSCR_PSC;
		events |= SS_BATDEAD;	/* battery dead */
	}
	if (cscr & HD64465_PCCCSCR_PSC) {
		cscr &= ~HD64465_PCCCSCR_PSC;
		events |= SS_STSCHG;	/* STSCHG (status changed) signal */
	}
	
	if (cscr & HD64465_PCCCSCR_PIREQ) {
		cscr &= ~HD64465_PCCCSCR_PIREQ;

		/* This should have been dealt with during irq demux */
		printk(KERN_NOTICE MODNAME ": unexpected IREQ from card\n");
	}

	msp_out(sp, cscr, CSCR);

	if (events) {
		/*
		 * Arrange for events to be reported to the registered
		 * event handler function (from CardServices) in a process
		 * context (keventd) "soon".
		 */
		spin_lock(&msp_pending_event_lock);
		sp->pending_events |= events;
		spin_unlock(&msp_pending_event_lock);

		schedule_task(&msp_events_task);
	}
}
#endif	/* 0 */


static struct pccard_operations	msp_operations = {
	msp_init,
	msp_suspend,
	msp_register_callback,
	msp_inquire_socket,
	msp_get_status,
	msp_get_socket,
	msp_set_socket,
	msp_get_io_map,
	msp_set_io_map,
	msp_get_mem_map,
	msp_set_mem_map,
	msp_proc_setup
};


static int	msp_init_socket(msp_socket_t *sp, int irq,
			unsigned long mem_base/*, unsigned int ctrl_base*/)
{
//	unsigned short	v;
	int	i;	//, err;

	DBG("msp_init_socket: irq=%d, mem_base=%08lX\n", irq, mem_base);
	memset(sp, 0, sizeof(*sp));
	sp->irq = irq;
	sp->mem_base = mem_base;
	sp->mem_length = MSP_IO_MAP_SIZE * 3;	/* 24 MB */
	sp->reset = 1;
//	sp->ctrl_base = ctrl_base;

	for (i = 0 ; i < MAX_IO_WIN ; i++)
		sp->io_maps[i].map = i;
	for (i = 0 ; i < MAX_WIN ; i++)
		sp->mem_maps[i].map = i;

//	if ((sp->io_vma = get_vm_area(MSP_IO_MAP_SIZE, VM_IOREMAP)) == 0)
//		return -ENOMEM;

//	msp_register_irq_demux(sp->irq, msp_irq_demux, sp);

//irq	err = request_irq(sp->irq, msp_interrupt, SA_INTERRUPT, MODNAME, sp);
//irq	if (err < 0)
//irq		return err;
	if (request_mem_region(sp->mem_base, sp->mem_length, MODNAME) == 0) {
		sp->mem_base = 0;
		return -ENOMEM;
	}

	/* According to section 3.2 of the PCMCIA standard, low-voltage
	 * capable cards must implement cold insertion, i.e. Vpp and
	 * Vcc set to 0 before card is inserted.
	 */
	/*msp_set_voltages(sp, 0, 0);*/

	sp->status = MSP_INIT_STATE;

//	msp_reset_socket(sp, 1);

	return 0;
}


static void	msp_exit_socket(msp_socket_t *sp)
{
	slmRegs	* const sreg = (slmRegs *)SREG_BASE;
	u_long	flags;

	save_and_cli(flags);	/* Don't want interrupts happening here */

	/* Turn off interrupts */
	sreg->int_msk &= (1 << PCMCIA_IRQ);

	restore_flags(flags);

	/* Power down the card */
	msp_set_voltages(sp, 0, 0);

	if (sp->mem_base != 0)
		release_mem_region(sp->mem_base, sp->mem_length);
//irq	if (sp->irq != 0) {
//irq		free_irq(sp->irq, msp_interrupt);
//irq		msp_unregister_irq_demux(sp->irq);
//irq	}
//	if (sp->io_vma != 0)
//		vfree(sp->io_vma->addr);
}


static int __init	init_msp(void)
{
	slmRegs	* const sreg = (slmRegs *)SREG_BASE;
	perRegs	* const preg = (perRegs *)PER_BASE;
	servinfo_t	serv;
	int	i;

	pkinit("%s\n", version);

	{
		u32	id = sreg->dev_id;

		switch (id & 0xFF00) {
		case 0x2000:
			IDBG("2000 (Polo?), id = 0x%08X\n", id);
			break;

#ifdef	CONFIG_BRECIS_FPGA
		case 0:
			IDBG("FPGA (Polo?), id = 0x%08X\n", id);
			break;
#endif	/* CONFIG_BRECIS_FPGA */

		default:
			IDBG("No PCMCIA, id = 0x%08X\n", id);
			goto no_pcmcia;
		}

		/* Check for single PC Card mode */
		if (id & DEV_ID_SINGLE_PC) {
			if (sreg->single_pc_enable & PCCARD_32) {
				pkinit("No PCMCIA - Single PC Card is Cardbus\n");
				goto no_pcmcia;
			}
			sreg->single_pc_enable |= SINGLE_PCCARD;
			single_pc_mode = 1;	/* Remember single pc mode */
		}
	}

	/*
	 * Check API version
	 */
	pcmcia_get_card_services_info(&serv);
	if (serv.Revision != CS_RELEASE_CODE) {
		pkinit("Card Services release does not match!\n");
		goto no_pcmcia;
	}

	IDBG("Initial ELB setup...\n");
	for (i = 0; i < 8; ++i) {
		IDBG("CS%d config=%04X, base_addr=%04X, "
			"size_mask=%04X, access_type=%04X\n", i,
			sreg->elb[i].config, sreg->elb[i].base_addr,
			sreg->elb[i].size_mask,
			sreg->elb[i].access_type);
	}

	/* clear ELB reset */
	sreg->elb_reset = 0;

	/* set gpio 31 to be ELB chip select 3 */
	preg->gpio_config4 |= 0x90000000;

	/* set gpio 32 to be ELB chip select 2 */
	preg->gpio_config5 |= 0x9;

	/* Effectively disable CS4-7 */
	for (i = 4; i < 8; ++i) {
		sreg->elb[i].config = 5;
		sreg->elb[i].size_mask = 0x7F;
		sreg->elb[i].base_addr = 0x178 + i;
		sreg->elb[i].access_type = 0;
	}

	/* Assign PCMCIA chip selects */

	sreg->elb[1].config = ELB_CONFIG_PCMCIA_ACCESS;
	sreg->elb[1].size_mask = 0;	/* 8 MB window */
	sreg->elb[1].base_addr = 0;	/* 0xBE000000 - 0xBE7FFFFF */
	sreg->elb[1].access_type = ELB_ACCESS_TYPE_ATTR;

	sreg->elb[2].config = ELB_CONFIG_PCMCIA_ACCESS;
	sreg->elb[2].size_mask = 0;	/* 8 MB window */
	sreg->elb[2].base_addr = 0x80;	/* 0xBE800000 - 0xBEFFFFFF */
	sreg->elb[2].access_type = ELB_ACCESS_TYPE_MEMORY;

	sreg->elb[3].config = ELB_CONFIG_PCMCIA_ACCESS;
	sreg->elb[3].size_mask = 0;	/* 8 MB window */
	sreg->elb[3].base_addr = 0x100;	/* 0xBF000000 - 0xBF7FFFFF */
	sreg->elb[3].access_type = ELB_ACCESS_TYPE_IO;

	/* configure GPIO address bits - ELB ADR 23,24,25 */
	/* 14 = reset, gpio 15 = ELB ADR 23 */
	preg->gpio_config2 |= 0xaa000000;	

	/* gpio 16,17 = ELB ADR 24,25 */
	preg->gpio_config3 |= 0xa0000099;

	/* ELB Chip Select 1 Extended address */
	sreg->elb_ext[1] = 0x07;
	
	/* ELB Chip Select 2 Extended address */
	sreg->elb_ext[2] = 0x07;
	
	/* ELB Chip Select 3 Extended address */
	sreg->elb_ext[3] = 0x07;	

	/* Enable reset output */
	if (!single_pc_mode)
		preg->gpio_config2 |= (2 << 24);

	IDBG("After ELB configuration...\n");
	for (i = 0; i < 8; ++i) {
		IDBG("CS%d config=%04X, base_addr=%04X, "
			"size_mask=%04X, access_type=%04X\n", i,
			sreg->elb[i].config, sreg->elb[i].base_addr,
			sreg->elb[i].size_mask,
			sreg->elb[i].access_type);
	}

	IDBG("GPIO config...\n");
	IDBG("config1@%p=%04X, config2=%04X\n",
		&preg->gpio_config1, preg->gpio_config1, preg->gpio_config2);
	IDBG("config3@%p=%04X, config4=%04X\n",
		&preg->gpio_config3, preg->gpio_config3, preg->gpio_config4);
	IDBG("config5@%p=%04X, config6=%04X\n",
		&preg->gpio_config5, preg->gpio_config5, preg->gpio_config6);
	IDBG("config7@%p=%04X, od_config=%04X, od_config2=%04X\n",
		&preg->gpio_config7, preg->gpio_config7, preg->gpio_od_config,
		preg->gpio_od_config2);

//	msp_set_voltages(&msp_sockets[0], 0, 0);

	/*
	 * Setup msp_sockets[] structures and request system resources.
	 * TODO: on memory allocation failure, power down the socket
	 *       before quitting.
	 */
	i = msp_init_socket(&msp_sockets[0], PCMCIA_IRQ,
		PCMCIA_ATTR_BASE);
	if (i < 0)
		return i;

	if (register_ss_entry(MSP_MAX_SOCKETS, &msp_operations) != 0) {
		for (i = 0; i < MSP_MAX_SOCKETS ; i++)
		msp_exit_socket(&msp_sockets[i]);
		return -ENODEV;
	}

	pkinit(KERN_INFO "BRECIS PCMCIA bridge:\n");
	for (i = 0; i < MSP_MAX_SOCKETS ; i++) {
		msp_socket_t	*sp = &msp_sockets[i];

		pkinit(KERN_INFO " socket %d at 0x%08lx irq %d\n",
			i, sp->mem_base, sp->irq);
	}

	return 0;

no_pcmcia:
	return -ENODEV;
}


static void __exit	exit_msp(void)
{
	u_long	flags;
	int	i;

	save_and_cli(flags);

	/*
	 * Release kernel resources
	 */
	del_timer_sync(&event_timer);
	for (i = 0; i < MSP_MAX_SOCKETS ; i++)
		msp_exit_socket(&msp_sockets[i]);
	unregister_ss_entry(&msp_operations);

	restore_flags(flags);
}

module_init(init_msp);
module_exit(exit_msp);

