/*
 ***************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 *
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ***************************************************************************

    Module Name:
    wpa.c

    Abstract:

    Revision History:
    Who         When            What
    --------    ----------      ----------------------------------------------
    Jan Lee     03-07-22        Initial
*/
#include "rt_config.h"

#define PAIRWISEKEY     1
#define GROUPKEY        0
extern  UCHAR   EAPOL[];

BOOLEAN WPAMsgTypeSubst(
    IN UCHAR    EAPType,
    OUT ULONG   *MsgType) 
{
    switch(EAPType)
    {
        case EAPPacket:
            *MsgType = MACHINE_TYPE_EAPPacket;
            break;

        case EAPOLStart:
            *MsgType = MACHINE_TYPE_EAPOLStart;
            break;

        case EAPOLLogoff:
            *MsgType = MACHINE_TYPE_EAPOLLogoff;
            break;

        case EAPOLKey:
            *MsgType = MACHINE_TYPE_EAPOLKey;
            break;

        case EAPOLASFAlert:
            *MsgType = MACHINE_TYPE_EAPOLASFAlert;
            break;

        default:
            DBGPRINT(RT_DEBUG_TRACE, "  Unknown WPA MsgType :    return FALSE; \n");    
            return FALSE;
    }
    
    return TRUE;
}

/*  
    ==========================================================================
    Description: 
        association state machine init, including state transition and timer init
    Parameters: 
        S - pointer to the association state machine
    ==========================================================================
 */
VOID WpaStateMachineInit(
    IN  PRTMP_ADAPTER       pAd,
    IN  STATE_MACHINE       *S,
    OUT STATE_MACHINE_FUNC  Trans[])
{
    NDIS_STATUS Status = NDIS_STATUS_SUCCESS;

    StateMachineInit(S, (STATE_MACHINE_FUNC*)Trans, MAX_WPA_PTK_STATE, MAX_WPA_MSG, (STATE_MACHINE_FUNC)Drop, WPA_PTK, WPA_MACHINE_BASE);

    StateMachineSetAction(S, WPA_PTK, MACHINE_TYPE_EAPPacket, (STATE_MACHINE_FUNC)WpaEAPPacketAction);
    StateMachineSetAction(S, WPA_PTK, MACHINE_TYPE_EAPOLStart, (STATE_MACHINE_FUNC)WpaEAPOLStartAction);
    StateMachineSetAction(S, WPA_PTK, MACHINE_TYPE_EAPOLLogoff, (STATE_MACHINE_FUNC)WpaEAPOLLogoffAction);
    StateMachineSetAction(S, WPA_PTK, MACHINE_TYPE_EAPOLKey, (STATE_MACHINE_FUNC)WpaEAPOLKeyAction);
    StateMachineSetAction(S, WPA_PTK, MACHINE_TYPE_EAPOLASFAlert, (STATE_MACHINE_FUNC)WpaEAPOLASFAlertAction);
}

/*
    ==========================================================================
    Description:
        Port Access Control Inquiry function. Return entry's Privacy and Wpastate.
        Also check if AP needs to initilize 4-way handshake. 
    Return:
        pEntry 
    ==========================================================================
*/
MAC_TABLE_ENTRY *PACInquiry(
    IN PRTMP_ADAPTER                pAd,
    IN MACADDR                      *pAddr,
    OUT NDIS_802_11_PRIVACY_FILTER  *Privacy,
    OUT WPA_STATE                   *WpaState)
{
    MAC_TABLE_ENTRY *pEntry = (MAC_TABLE_ENTRY*)NULL;
    *Privacy = Ndis802_11PrivFilterAcceptAll;
    *WpaState = AS_NOTUSE;

    if (MAC_ADDR_IS_GROUP(*pAddr)) // mcast & broadcast address
    {
    } 
    else // unicast address
    {
        pEntry = MacTableLookup(pAd, pAddr);
        if (pEntry) 
        {
            *Privacy = pEntry->PrivacyFilter;
            *WpaState = pEntry->WpaState;
            if((*WpaState == AS_INITPSK) || (*WpaState == AS_INITPMK))
            {    
                DBGPRINT(RT_DEBUG_TRACE, "PACInquiry  ===>> \n");
                WPAStart4WayHS(pAd, pEntry);
            }
        } 
    }
    return pEntry;
}

/*
    ==========================================================================
    Description:
       
    Return:
         TRUE if this is EAP frame
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckWPAframe(
    IN PRTMP_ADAPTER    pAdapter,
    IN PUCHAR           pHeader,
    IN ULONG            DataByteCount)
{
    UCHAR   SNAP_802_1H[6] = {0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00};
    UCHAR   EAPOL[2] = {0x88, 0x8e};
    PUCHAR  pData;
    
    if (DataByteCount < (LENGTH_802_11 + LENGTH_802_1_H + LENGTH_EAPOL_H))
        return FALSE;

    DBGPRINT(RT_DEBUG_INFO, "RTMPCheckWPAframe   ====>> \n");
    
    pData = pHeader + LENGTH_802_11;
    DataByteCount -= LENGTH_802_11;
    if (RTMPEqualMemory(SNAP_802_1H, pData, 6)) 
    {
        pData += 6;
    }
    if (RTMPEqualMemory(EAPOL, pData, 2)) 
    {
        pData += 2;
    }
    else
        return FALSE;

    switch (*(pData+1))
    {   
        case EAPPacket:
            DBGPRINT(RT_DEBUG_TRACE, "Receive EAP-Packet frame, TYPE = 0 \n");
            break;

        case EAPOLStart:
            DBGPRINT(RT_DEBUG_TRACE, "Receive EAPOL-Start frame, TYPE = 1 \n");
            break;

        case EAPOLLogoff:
            DBGPRINT(RT_DEBUG_TRACE, "Receive EAPOLLogoff frame, TYPE = 2 \n");
            break;

        case EAPOLKey:
            DBGPRINT(RT_DEBUG_TRACE, "Receive EAPOL-Key frame, TYPE = 3, Length =%x\n", *(pData+2));
            break;

        case EAPOLASFAlert:
            DBGPRINT(RT_DEBUG_TRACE, "Receive EAPOLASFAlert frame, TYPE = 4 \n");
            break;

        default:
            return FALSE;
    }   
    return TRUE;
}

/*
    ==========================================================================
    Description:
       Check invalidity of multicast cipher selection in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckMcast(
    IN PRTMP_ADAPTER    pAdapter,
    PEID_STRUCT         eid_ptr)
{
    if (eid_ptr->Len >= 10)
    {   
        if (!RTMPEqualMemory(&eid_ptr->Octet[6], &pAdapter->PortCfg.RSN_IE[6], 4))
            return FALSE;
    }
    else
        return FALSE;

    return TRUE;
}

/*
    ==========================================================================
    Description:
       Check invalidity of unicast cipher selection in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckUcast(
    IN PRTMP_ADAPTER    pAdapter,
    IN PEID_STRUCT      eid_ptr)
{
    INT i,j;

    DBGPRINT(RT_DEBUG_TRACE, "RTMPCheckUcast  ==>  ucast number = %d\n",eid_ptr->Octet[10]);
    if (eid_ptr->Len >= 16)
    {   
        for (i = 0; i < (INT)eid_ptr->Octet[10]; i++)
        {
            for (j = 0; j < (INT)pAdapter->PortCfg.RSN_IE[10]; j++)
            {
                DBGPRINT(RT_DEBUG_INFO, "  %x %x %x %x ",eid_ptr->Octet[12 + i*4],eid_ptr->Octet[13 + i*4],eid_ptr->Octet[14 + i*4],eid_ptr->Octet[15 + i*4]);
                if (!RTMPEqualMemory(&eid_ptr->Octet[12 + i*4], &pAdapter->PortCfg.RSN_IE[12 + j*4], 4))
                    return FALSE;
                else
                    break;
            }
        }
        DBGPRINT(RT_DEBUG_TRACE, "\n");
   }
    else
        return FALSE;

    return TRUE;
}

/*
    ==========================================================================
    Description:
       Check invalidity of authentication method selection in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckAUTH(
    IN PRTMP_ADAPTER    pAdapter,
    IN PEID_STRUCT      eid_ptr)
{
    UCHAR i;
    
    DBGPRINT(RT_DEBUG_TRACE, "RTMPCheckAUTH ===>> RSN_IE Len = %d, last RSN IE = %d \n",eid_ptr->Len,pAdapter->PortCfg.RSN_IE[21]);
    if (eid_ptr->Len >= 22)
    {   
        for (i = eid_ptr->Len; i >= 22; i--)
        {
            if (RTMPEqualMemory(&eid_ptr->Octet[i-4], &pAdapter->PortCfg.RSN_IE[18], 4))
                return TRUE;
        }
    }
    else
        return FALSE;

    return FALSE;
}

/*
    ==========================================================================
    Description:
        this is state machine function. 
        When receiving EAP packets which is  for 802.1x authentication use. 
        Not use in PSK case
    Return:
    ==========================================================================
*/
VOID WpaEAPPacketAction(
    IN PRTMP_ADAPTER    pAd, 
    IN MLME_QUEUE_ELEM  *Elem) 
{
}

VOID WpaEAPOLStartAction(
    IN PRTMP_ADAPTER    pAd, 
    IN MLME_QUEUE_ELEM  *Elem) 
{
    MAC_TABLE_ENTRY     *pEntry;
    PHEADER_802_11      pHeader;

    DBGPRINT(RT_DEBUG_TRACE, "WpaEAPOLStartAction ====>> \n");            
    
    pHeader = (PHEADER_802_11)Elem->Msg;
    pEntry = MacTableLookup(pAd, &pHeader->Controlhead.Addr2);
    if (pEntry) 
    {
        if((pEntry->WpaState == AS_INITPSK) || (pEntry->WpaState == AS_INITPMK))
        {    
            WPAStart4WayHS(pAd, pEntry);
        }
    } 
}

VOID WpaEAPOLLogoffAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
}

/*
    ==========================================================================
    Description:
        Function to handel countermeasures active attack.  Init 60-sec timer if necessary.
    Return:
    ==========================================================================
*/
VOID HandleCounterMeasure(
    IN PRTMP_ADAPTER    pAd, 
    IN MAC_TABLE_ENTRY  *pEntry) 
{
    INT         i;
    UCHAR       *pData;
    
    pAd->PortCfg.MICFailureCounter++;
    // record which entry causes this MIC error, if this entry sends disauth/disassoc, AP doesn't need to log the CM
    pEntry->CMTimerRunning = TRUE;
    if (pAd->PortCfg.CMTimerRunning == TRUE)
    {
        DBGPRINT(RT_DEBUG_TRACE, "Receive CM Attack Twice  Within 60 seconds  ====>>> \n");            
        // renew GTK
        GenRandom(pAd, pAd->PortCfg.GNonce);
        ApLogEvent(pAd, &pEntry->Addr, EVENT_COUNTER_M);
        del_timer_sync(&pAd->PortCfg.CounterMeasureTimer);
        for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
        {
            // happened twice within 60 sec,  AP SENDS disaccociate all associated STAs.  All STA's transition to State 2
            if (pAd->MacTab.Content[i].Valid == TRUE)
            {
                DisAssocAction(pAd, &pAd->MacTab.Content[i], SUBTYPE_DISASSOC, REASON_MIC_FAILURE);
            }
        }
        // Further,  ban all Class 3 DATA transportation for  a period 0f 60 sec
        // disallow new association , too
        pAd->PortCfg.BANClass3Data = TRUE;        

        // check how many entry left...  should be zero
        pAd->PortCfg.GKeyDoneStations = pAd->MacTab.Size;
        DBGPRINT(RT_DEBUG_TRACE, "  pAd->PortCfg.GKeyDoneStations= %d====>>> \n",pAd->PortCfg.GKeyDoneStations);
    }

    pAd->PortCfg.CounterMeasureTimer.expires = jiffies + 60*MLME_TASK_EXEC_INTV;
    add_timer(&pAd->PortCfg.CounterMeasureTimer);
    pAd->PortCfg.CMTimerRunning = TRUE;
    pAd->PortCfg.PrevaMICFailTime = pAd->PortCfg.aMICFailTime;
    pAd->PortCfg.aMICFailTime = jiffies;
}

/*
    ==========================================================================
    Description:
        This is state machine function. 
        When receiving EAPOL packets which is  for 802.1x key management. 
        Use both in WPA, and WPAPSK case. 
        In this function, further dispatch to different functions according to the received packet.  3 categories are : 
          1.  normal 4-way pairwisekey and 2-way groupkey handshake
          2.  MIC error (Countermeasures attack)  report packet from STA.
          3.  Request for pairwise/group key update from STA
    Return:
    ==========================================================================
*/
VOID WpaEAPOLKeyAction(
    IN PRTMP_ADAPTER    pAdapter, 
    IN MLME_QUEUE_ELEM  *Elem) 
{
    INT                 i;
    BOOLEAN             Cancelled;
    MAC_TABLE_ENTRY     *pEntry;
    PHEADER_802_11      pHeader;
    KEY_DESCRIPTER      *pKeyDesc;
    PEAPOL_PACKET       pMsg;

    pHeader = (PHEADER_802_11)Elem->Msg;
    DBGPRINT(RT_DEBUG_INFO, "WpaEAPOLKeyAction  =====>>> \n");
    pMsg = (PEAPOL_PACKET) &Elem->Msg[LENGTH_802_11 + LENGTH_802_1_H];
    pKeyDesc = (PKEY_DESCRIPTER)&Elem->Msg[(LENGTH_802_11+LENGTH_802_1_H+LENGTH_EAPOL_H)];

#ifdef BIG_ENDIAN
    // pMsg->KeyDesc.Keyinfo and pKeyDesc->Keyinfo both point to the same addr.
    // Thus, it only needs swap once.
    *(USHORT *)&pMsg->KeyDesc.Keyinfo = SWAP16(*(USHORT *)&pMsg->KeyDesc.Keyinfo);
#endif

    if ((pAdapter->PortCfg.AuthMode != Ndis802_11AuthModeWPA) && (pAdapter->PortCfg.AuthMode != Ndis802_11AuthModeWPAPSK))
        return;

    if ((pMsg->ProVer!= EAPOL_VER) || (pMsg->KeyDesc.Type != RSN_KEY_DESC))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Key descripter does not match with WPA rule \n");
        return;
    }

    if ((pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled) && (pMsg->KeyDesc.Keyinfo.KeyDescVer != DESC_TYPE_AES))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Key descripter version not match AES \n");
        return;
    }
    else if ((pAdapter->PortCfg.WepStatus == Ndis802_11Encryption2Enabled) && (pMsg->KeyDesc.Keyinfo.KeyDescVer != DESC_TYPE_TKIP))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Key descripter version not match TKIP \n");
        return;
    }

    pEntry = MacTableLookup(pAdapter, &pHeader->Controlhead.Addr2);
    if ((pEntry) && (pEntry->Sst == SST_ASSOC) && (pEntry->WpaState >= AS_INITPSK))
    {
        if ((pKeyDesc->Keyinfo.KeyMic) && (pKeyDesc->Keyinfo.Error) && (pKeyDesc->Keyinfo.Request))
        {
            // Category 2. 
            DBGPRINT(RT_DEBUG_TRACE, " MIC, ERROR, REQUEST ALL 1'S, ACTIVE COUNTERMEASURE \n");
            HandleCounterMeasure(pAdapter, pEntry);
        }
        else if ((pKeyDesc->Keyinfo.Secure) && !(pKeyDesc->Keyinfo.Error) && !(pKeyDesc->Keyinfo.Request))
        {
#ifdef BIG_ENDIAN
            // recovery original byte order, before forward Elem to another routine
            *(USHORT *)&pMsg->KeyDesc.Keyinfo = SWAP16(*(USHORT *)&pMsg->KeyDesc.Keyinfo);
#endif
            // Category 1
            PeerGroupMsg2Action(pAdapter, pEntry, &Elem->Msg[LENGTH_802_11], (Elem->MsgLen-LENGTH_802_11));
        }
        else if (!(pKeyDesc->Keyinfo.Secure) && !(pKeyDesc->Keyinfo.Error) && !(pKeyDesc->Keyinfo.Request))
        {
#ifdef BIG_ENDIAN
            // recovery original byte order, before forward Elem to another routine
            *(USHORT *)&pMsg->KeyDesc.Keyinfo = SWAP16(*(USHORT *)&pMsg->KeyDesc.Keyinfo);
#endif
            // general 4-way handshake step // SECURE=0, REQUEST=0 ,  ERROR= 0
            // Category 1
            if (pEntry->WpaState == AS_PTKSTART)
                PeerPairMsg2Action(pAdapter, pEntry, Elem);
            else if (pEntry->WpaState == AS_PTKINIT_NEGOTIATING)
                PeerPairMsg4Action(pAdapter, pEntry, Elem);
        }
        else if (!(pKeyDesc->Keyinfo.Error) && (pKeyDesc->Keyinfo.Request))
        {
            // Category 3
            if (pKeyDesc->Keyinfo.KeyType == GROUPKEY)
            {
                DBGPRINT(RT_DEBUG_TRACE, "REQUEST=1 , ERROR= 0, UPDATE Group Key\n");
                GenRandom(pAdapter, pAdapter->PortCfg.GNonce);
                pAdapter->PortCfg.WpaGTKState = SETKEYS;
                pAdapter->PortCfg.GKeyDoneStations = pAdapter->MacTab.Size;
                for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
                {
                    if ((pAdapter->MacTab.Content[i].Valid == TRUE) && (pAdapter->MacTab.Content[i].WpaState == AS_PTKINITDONE))
                    {
                        pAdapter->MacTab.Content[i].GTKState = REKRY_NEGOTIATING;
                        WPAHardTransmit(pAdapter, &pAdapter->MacTab.Content[i]);
                    }
                }
            }
            else
            {
                DBGPRINT(RT_DEBUG_TRACE, "REQUEST=1, ERROR= 0, UPDATE Pairwise Key\n");
                pEntry->PairwiseKey[0].KeyLen = 0;
                pEntry->Sst = SST_ASSOC;
                if (pAdapter->PortCfg.AuthMode == Ndis802_11AuthModeWPA)
                    pEntry->WpaState = AS_INITPMK;  
                else if (pAdapter->PortCfg.AuthMode == Ndis802_11AuthModeWPAPSK)
                    pEntry->WpaState = AS_INITPSK;  
                pEntry->GTKState = REKRY_NEGOTIATING;
                if (pEntry->RetryTimerRunning == FALSE )
                {   // retry up to 3 times at interval 1 second
                    DBGPRINT(RT_DEBUG_TRACE, " init_timer : add_timer Timer ====>>\n");
                    pEntry->RetryTimerRunning = TRUE;
                    WPAStart4WayHS(pAdapter, pEntry);
                    init_timer(&pEntry->RetryTimer);
                    pEntry->RetryTimer.expires = jiffies + WPA_RETRY_EXEC_INTV;
                    pEntry->RetryTimer.data = (unsigned long)pAdapter;
                    pEntry->RetryTimer.function = &WPARetryExec;
                    add_timer(&pEntry->RetryTimer);         
                    pEntry->ReTryCounter = PAIR_HS_RETRY_TIMER_CTR;
                }
            }
        }
        else
        {
                //
        }
    }
}

/*
    ==========================================================================
    Description:
        This is a function to initilize 4way handshake
    Return:
         
    ==========================================================================
*/
VOID WPAStart4WayHS(
    IN PRTMP_ADAPTER    pAdapter,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    UINT            i;
    ULONG           FrameLen = 0;
    UCHAR           *OutBuffer = NULL;
    UCHAR           Header802_3[14];
    EAPOL_PACKET    Packet;

    DBGPRINT(RT_DEBUG_TRACE, "WPAStart4WayHS ==>   \n");
    
    if ((pEntry->WpaState > AS_PTKSTART) || (pEntry->WpaState < AS_INITPMK))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Not expect calling WPAStart4WayHS here ");
        return;
    }

    WPAMake8023Hdr(pAdapter, pEntry->Addr.Octet, Header802_3);
    do
    {
        OutBuffer = kmalloc(MAX_LEN_OF_EAP_HS, GFP_KERNEL);
        if(OutBuffer == NULL)
            return;

        // Increment replay counter by 1  
        i = LEN_KEY_DESC_REPLAY;
        do
        {
            i--;
            pAdapter->PortCfg.R_Counter[i]++;
            if (i == 0)
                break;
        }
        while (pAdapter->PortCfg.R_Counter[i] == 0);
        
        // 0. init Packet and Fill header
        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;
        Packet.Body_Len[1] = 0x5f;
        
        // 1. Fill replay counter
        NdisMoveMemory(pEntry->R_Counter, pAdapter->PortCfg.R_Counter, sizeof(pEntry->R_Counter));
        NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, sizeof(pEntry->R_Counter));
        
        // 2. Fill key version, keyinfo, key len
        Packet.KeyDesc.Type = RSN_KEY_DESC;
        Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        Packet.KeyDesc.Keyinfo.KeyType = PAIRWISEKEY;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;
        Packet.KeyDesc.KeyLength[1] = LEN_TKIP_KEY;
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_AES;
        }

#ifdef BIG_ENDIAN
        *(USHORT *)&Packet.KeyDesc.Keyinfo = SWAP16(*(USHORT *)&Packet.KeyDesc.Keyinfo);
#endif
        
        // 3. Fill Anonce
        GenRandom(pAdapter, pEntry->ANonce);  
        NdisMoveMemory(Packet.KeyDesc.Nonce, pEntry->ANonce, sizeof(pEntry->ANonce));
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                            sizeof(Header802_3),    Header802_3,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);

        RTMPToWirelessSta(pAdapter, OutBuffer, FrameLen);

        pEntry->WpaState = AS_PTKSTART;
    }while(FALSE);    
    kfree(OutBuffer);
    
    DBGPRINT(RT_DEBUG_TRACE, "<== WPAStart4WayHS : pEntry->WpaState= %d , FrameLen= %d\n",pEntry->WpaState,FrameLen);
}

/*
    ==========================================================================
    Description:
        When receiving the second packet of 4-way pairwisekey handshake.
    Return:
    ==========================================================================
*/
VOID PeerPairMsg2Action(
    IN PRTMP_ADAPTER    pAdapter, 
    IN MAC_TABLE_ENTRY  *pEntry,
    IN MLME_QUEUE_ELEM  *Elem) 
{   
    UINT                    i;
    UCHAR                   PTK[80], mic[LEN_KEY_DESC_MIC], SNonce[LEN_KEY_DESC_NONCE];
    ULONG                   FrameLen = 0;
    UCHAR                   *OutBuffer = NULL;
    USHORT                  MICMsgLen, MICOffset;
    PUCHAR                  pSA;
    UCHAR                   Header802_3[LENGTH_802_3];
    EAPOL_PACKET            Packet;
    PEAPOL_PACKET           pMsg2;

    DBGPRINT(RT_DEBUG_TRACE, "PeerPairMsg2Action ====>>\n");
    
    if (Elem->MsgLen < (LENGTH_802_11 + LENGTH_802_1_H + sizeof(EAPOL_PACKET) -MAX_LEN_OF_RSNIE -2))
        return;

    // check Entry in valid State
    if (pEntry->WpaState < AS_PTKSTART)
        return;
    do
    {
        OutBuffer = kmalloc(MAX_LEN_OF_EAP_HS, GFP_KERNEL);
        if(OutBuffer == NULL)
            return;
            
        // Save Data Length to pDesc for receiving packet,   then put in outgoing frame Data Len fields.
        pMsg2 = (PEAPOL_PACKET) &Elem->Msg[LENGTH_802_11 + LENGTH_802_1_H];
            
        // 1. check Replay Counter coresponds to MSG1.,  otherwise discard
        if (!RTMPEqualMemory(pMsg2->KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
        {
            DBGPRINT(RT_DEBUG_ERROR, "Replay Counter Different in Msg 2 of 4-way handshake !! \n");
            DBGPRINT(RT_DEBUG_TRACE, "Receive :  %d   %d  %d   %d  %d   %d  %d   %d   \n", pMsg2->KeyDesc.RCounter[0],pMsg2->KeyDesc.RCounter[1],\
                pMsg2->KeyDesc.RCounter[2],pMsg2->KeyDesc.RCounter[3],pMsg2->KeyDesc.RCounter[4],pMsg2->KeyDesc.RCounter[5],pMsg2->KeyDesc.RCounter[6],pMsg2->KeyDesc.RCounter[7]);
            DBGPRINT(RT_DEBUG_TRACE, "Current :  %d   %d  %d   %d  %d   %d  %d   %d   \n",pEntry->R_Counter[0],pEntry->R_Counter[1],pEntry->R_Counter[2],pEntry->R_Counter[3],\
                pEntry->R_Counter[4],pEntry->R_Counter[5],pEntry->R_Counter[6],pEntry->R_Counter[7]);
            break;
        }

        // 2. Derive necessary info for counting PTK and thus count PTK
        MICMsgLen = pMsg2->Body_Len[1] | ((pMsg2->Body_Len[0]<<8) && 0xff00);
        MICMsgLen += LENGTH_EAPOL_H;
        if (MICMsgLen > (Elem->MsgLen -LENGTH_802_11 - LENGTH_802_1_H))
        {
            DBGPRINT(RT_DEBUG_ERROR, "Receive wrong format EAPOL packets \n");
            break;        
        }
        NdisMoveMemory(SNonce, pMsg2->KeyDesc.Nonce, LEN_KEY_DESC_NONCE);
        pSA = pEntry->Addr.Octet;
        CountPTK(pAdapter->PortCfg.PMK,  pEntry->ANonce, pAdapter->CurrentAddress, SNonce, pSA, PTK, LEN_PTK); 
        NdisMoveMemory(pEntry->PTK, PTK, LEN_PTK);
            
        // 3. verify MSG2 MIC, If not valid, discard.
        MICOffset = sizeof(EAPOL_PACKET)-MAX_LEN_OF_RSNIE-2-LEN_KEY_DESC_MIC;
        NdisMoveMemory(OutBuffer, pMsg2, MICMsgLen);  
        NdisZeroMemory((OutBuffer+MICOffset), LEN_KEY_DESC_MIC);
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption2Enabled)
        {
            hmac_md5(PTK, LEN_EAP_MICK, OutBuffer, MICMsgLen, mic);
        }
        else
        {
            UCHAR digest[80];
            HMAC_SHA1(OutBuffer,  MICMsgLen, PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
        }
        if (!RTMPEqualMemory((PUCHAR)pMsg2+MICOffset, mic, LEN_KEY_DESC_MIC))
        {
            DBGPRINT(RT_DEBUG_ERROR, "MIC  Different in Msg 2 of 4-way handshake!! \n");
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, "MIC  VALID  in Msg 2 of 4-way handshake!! \n");
            
        // 4. check RSN IE, if not match, send MLME-DEAUTHENTICATE.
        if (!RTMPEqualMemory(&pMsg2->KeyDesc.Data[2], pEntry->RSN_IE, pEntry->RSNIE_Len))
        {       
            DBGPRINT(RT_DEBUG_ERROR, "RSN_IE   Different in Msg 2 of 4-way handshake!! \n");
            pEntry->Flag |= DIS_REASON_RSNIE_DIFF;            
            DBGPRINT(RT_DEBUG_TRACE, "Receive : %x %x %x %x ...",pMsg2->KeyDesc.Data[2], pMsg2->KeyDesc.Data[3],pMsg2->KeyDesc.Data[4],pMsg2->KeyDesc.Data[5] );
            DBGPRINT(RT_DEBUG_TRACE, "Current : %x %x %x %x ...",pEntry->RSN_IE[0], pEntry->RSN_IE[1],pEntry->RSN_IE[2],pEntry->RSN_IE[3] );
            break;  
        }
        else
            DBGPRINT(RT_DEBUG_ERROR, "RSN_IE VALID  in Msg 2 of 4-way handshake!! \n");
            
        //verified ok , change state
        if (pEntry->RetryTimerRunning == TRUE)
        {
            DBGPRINT(RT_DEBUG_TRACE, "Cancel Retry Timer ! \n");
            del_timer_sync(&pEntry->RetryTimer);
            pEntry->RetryTimerRunning = FALSE;              
        }
        pEntry->WpaState = AS_PTKINIT_NEGOTIATING;

        // Increment replay counter by 1       
        i = LEN_KEY_DESC_REPLAY;
        do
        {           
            i--;
            pAdapter->PortCfg.R_Counter[i]++;
            if (i == 0)
                break;
        }
        while(pAdapter->PortCfg.R_Counter[i] == 0);
        NdisMoveMemory(pEntry->R_Counter, pAdapter->PortCfg.R_Counter, sizeof(pEntry->R_Counter));
            
        // 5. CONSTRUCT msg3
        // 5-0. Init Packet and Fill header
        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;
        NdisMoveMemory(Packet.Body_Len, pMsg2->Body_Len, 2);
            
        // 5-1, Fill replay counter
        NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, sizeof(pEntry->R_Counter));
          
        // 5-2, Fill key version, keyinfo, key len
        Packet.KeyDesc.Type = RSN_KEY_DESC;
        Packet.KeyDesc.Keyinfo.KeyMic = 1;
        Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        Packet.KeyDesc.Keyinfo.KeyType = PAIRWISEKEY;
        Packet.KeyDesc.Keyinfo.Install = 1;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;
        Packet.KeyDesc.KeyLength[1] = LEN_TKIP_KEY;
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_AES;
        }

#ifdef BIG_ENDIAN
        *(USHORT *)&Packet.KeyDesc.Keyinfo = SWAP16(*(USHORT *)&Packet.KeyDesc.Keyinfo);
#endif

        // 5-3, Fill in Data, Data Len
        NdisMoveMemory(Packet.KeyDesc.Nonce, pEntry->ANonce, sizeof(pEntry->ANonce));
        Packet.KeyDesc.DataLen[1] = (pEntry->RSNIE_Len+2);
        Packet.KeyDesc.Data[0] = IE_WPA_RSN_IE;
        Packet.KeyDesc.Data[1] = pEntry->RSNIE_Len;
        NdisMoveMemory(&Packet.KeyDesc.Data[2], pEntry->RSN_IE, pEntry->RSNIE_Len);

        // 5-4, First make a frame for Counting MIC.
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);
        NdisZeroMemory(mic, sizeof(mic));
            
        // 5-5, count MIC
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled)
        {
            UCHAR digest[80];
            HMAC_SHA1(OutBuffer,  FrameLen, PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
        }
        else
        {
            hmac_md5(PTK,  LEN_EAP_MICK, OutBuffer, FrameLen, mic);
        }
        NdisMoveMemory(&Packet.KeyDesc.MIC, mic, LEN_KEY_DESC_MIC);        
            
        // 5-6, Make outgoing frame
        WPAMake8023Hdr(pAdapter, pEntry->Addr.Octet, Header802_3);
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                            LENGTH_802_3,            Header802_3,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);
            
        DBGPRINT(RT_DEBUG_TRACE, "RTMPToWirelessSta : ETHTYPE = %x %x FrameLen = %d! \n",Header802_3[12],Header802_3[13],FrameLen);

        RTMPToWirelessSta(pAdapter, OutBuffer, FrameLen);
        pEntry->WpaState = AS_PTKINIT_NEGOTIATING;
        pEntry->Flag &= (~DISASSOC_4_WAY_TMOUT);
    }while(FALSE);
    kfree(OutBuffer);
}

/*
    ==========================================================================
    Description:
        countermeasures active attack timer execution
    Return:
    ==========================================================================
*/
VOID CMTimerExec(
    IN  unsigned long data) 
{
    UINT            i, j=0;
    RTMP_ADAPTER    *pAdapter = (RTMP_ADAPTER *)data;
        
    pAdapter->PortCfg.BANClass3Data = FALSE;
    for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
    {
        if ((pAdapter->MacTab.Content[i].Valid == TRUE) && (pAdapter->MacTab.Content[i].CMTimerRunning == TRUE))
        {
            pAdapter->MacTab.Content[i].CMTimerRunning =FALSE;
            j++;
        }
    }
    if (j > 1)
        DBGPRINT(RT_DEBUG_ERROR, "Find more than one entry which generated MIC Fail ..  \n");

    pAdapter->PortCfg.CMTimerRunning = FALSE;
}
    
VOID WPARetryExec(
    IN  unsigned long data) 
{
    UINT                i;
    MACADDR             addr;
    RTMP_ADAPTER        *pAdapter = (RTMP_ADAPTER *)data;
    MAC_TABLE_ENTRY     *pEntry;
    UCHAR               LLC_Len[2];
    UCHAR               EAPOLS[14]={0xaa, 0xaa, 0x3, 0x0, 0x0, 0x0, 0x88, 0x8e, 0x1, 0x1, 0x0,0x0,0x0,0x0 };    
    UCHAR               Header802_3[14];
    struct sk_buff      *skb;
    struct net_device   *net_dev;

    //retry up to 3 times, then send disassoc /deauth 
    DBGPRINT(RT_DEBUG_INFO, "EAPOLRetry Send Timer Execuatation ====>>\n");
    net_dev = pAdapter->net_dev;
    for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
    {
        if ((pAdapter->MacTab.Content[i].Valid == TRUE) && (pAdapter->MacTab.Content[i].RetryTimerRunning == TRUE))
        {
            BOOLEAN         bTimer = TRUE;
            
            pEntry = &pAdapter->MacTab.Content[i];
            pEntry->ReTryCounter++;
            DBGPRINT(RT_DEBUG_TRACE, "Send  Timer ReTryCounter= %d  WpaState=%d \n",pEntry->ReTryCounter,pEntry->WpaState);
            NdisMoveMemory(&addr, &pAdapter->MacTab.Content[i].Addr, MAC_ADDR_LEN);

            switch (pAdapter->PortCfg.AuthMode)
            {
                case Ndis802_11AuthModeWPA:                    

                case Ndis802_11AuthModeWPAPSK:
                    //  1.   Cancel  Timer  if necessary
                    if (pEntry->ReTryCounter > (GRP_HS_RETRY_TIMER_CTR+1))
                    {    
                        bTimer = FALSE;
                        pEntry->RetryTimerRunning = FALSE;  
                        if (pEntry->GTKState == REKRY_NEGOTIATING)          
                            pEntry->Flag |= DISASSOC_GRP_UDATE_TMOUT ;
                        DBGPRINT(RT_DEBUG_TRACE, "del_timer_sync    :  ====>>>pEntry->ReTryCounter %d   \n",pEntry->ReTryCounter);
                    }
                    else if (pEntry->ReTryCounter > GRP_HS_RETRY_TIMER_CTR)
                    {
                    }
                    else if (pEntry->ReTryCounter > (PAIR_HS_RETRY_TIMER_CTR+2))                
                    {   // if(pAdapter->MacTab.Content[i].WpaState==AS_INITPSK   || pAdapter->MacTab.Content[i].WpaState==AS_INITPMK )  
                        bTimer = FALSE;
                        pEntry->RetryTimerRunning = FALSE;  
                        DBGPRINT(RT_DEBUG_TRACE, "Cancel retry timer , pEntry->ReTryCounter = %d\n",pEntry->ReTryCounter);
                        ApLogEvent(pAdapter, &pEntry->Addr, EVENT_INVALID_PSK);
                        pEntry->Flag |= DISASSOC_4_WAY_TMOUT;
                    }

                    //  2.   Invoke retry Transmition 
                    if (((pEntry->WpaState == AS_PTKSTART) || (pEntry->WpaState == AS_INITPSK) || (pEntry->WpaState == AS_INITPMK)) && (pEntry->ReTryCounter < GRP_HS_RETRY_TIMER_CTR))
                    {
                        DBGPRINT(RT_DEBUG_TRACE, "Transmit Start 4-way Handshake in Timer ===>>  \n");
                        WPAStart4WayHS(pAdapter, pEntry);
                    }
                    else if ((pEntry->ReTryCounter >= GRP_HS_RETRY_TIMER_CTR) && (pEntry->GTKState == REKRY_NEGOTIATING) )
                    {
                        DBGPRINT(RT_DEBUG_TRACE, "ReInitilize 2-way group-key Handshake in TimerExecution ===>>   \n");
                        WPAHardTransmit( pAdapter,  pEntry);
                    }
                    break;

                default:
                    break;
            }
        
            if(bTimer)
            {
                pEntry->RetryTimer.expires = jiffies + WPA_RETRY_EXEC_INTV;
                add_timer(&pEntry->RetryTimer);
            }
        }
    }
}

/*
    ==========================================================================
    Description:
        When receiving the last packet of 4-way pairwisekey handshake.
        Initilize 2-way groupkey handshake following.
    Return:
    ==========================================================================
*/
VOID PeerPairMsg4Action(
    IN PRTMP_ADAPTER    pAdapter, 
    IN MAC_TABLE_ENTRY  *pEntry,
    IN MLME_QUEUE_ELEM  *Elem) 
{
    UINT                i;
    UCHAR               mic[LEN_KEY_DESC_MIC];
    ULONG               MICMsgLen;
    EAPOL_PACKET        EAPOLPKT;
    PEAPOL_PACKET       pMsg4;
    NDIS_STATUS         Status;
    
    DBGPRINT(RT_DEBUG_TRACE, "WpaEAPOL Peer PAIR  Msg4 Action ====>>\n");

    if (Elem->MsgLen < (LENGTH_802_11 + LENGTH_802_1_H + sizeof(EAPOL_PACKET) -MAX_LEN_OF_RSNIE -2 ) )
        return;
        
    if (pEntry->WpaState < AS_PTKINIT_NEGOTIATING)
        return;

    pMsg4 = (PEAPOL_PACKET)&Elem->Msg[LENGTH_802_11 + LENGTH_802_1_H];

    // 1.check Replay Counter
    if (!RTMPEqualMemory(pMsg4->KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Replay Counter Different in Msg 4 of 4-way handshake!! \n");
        DBGPRINT(RT_DEBUG_TRACE, "Receive : %d %d %d %d  \n", pMsg4->KeyDesc.RCounter[0],pMsg4->KeyDesc.RCounter[1],pMsg4->KeyDesc.RCounter[2],pMsg4->KeyDesc.RCounter[3]);
        DBGPRINT(RT_DEBUG_TRACE, "Current : %d %d %d %d  \n", pEntry->R_Counter[4],pEntry->R_Counter[5],pEntry->R_Counter[6],pEntry->R_Counter[7]);
        return;
    }

    // 2. check MIC, if not valid, discard silently
    MICMsgLen = pMsg4->Body_Len[1] | ((pMsg4->Body_Len[0]<<8) && 0xff00);
    MICMsgLen += LENGTH_EAPOL_H;
    if (MICMsgLen > (Elem->MsgLen -LENGTH_802_11 - LENGTH_802_1_H))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Receive wrong format EAPOL packets \n");
        return;        
    }
    NdisZeroMemory((PUCHAR)&EAPOLPKT, sizeof(EAPOLPKT));
    NdisMoveMemory((PUCHAR)&EAPOLPKT, pMsg4, MICMsgLen);  
    NdisZeroMemory(EAPOLPKT.KeyDesc.MIC, sizeof(EAPOLPKT.KeyDesc.MIC));

    hmac_md5(pEntry->PTK, LEN_EAP_MICK, (PUCHAR)&EAPOLPKT, (MICMsgLen), mic);
    if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption2Enabled)
    {
        hmac_md5(pEntry->PTK, LEN_EAP_MICK, (PUCHAR)&EAPOLPKT, (MICMsgLen), mic);
    }
    else
    {
        UCHAR digest[80];
        HMAC_SHA1((PUCHAR)&EAPOLPKT,  MICMsgLen, pEntry->PTK, LEN_EAP_MICK, digest);
        NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
    }
    if (!RTMPEqualMemory(pMsg4->KeyDesc.MIC, mic, LEN_KEY_DESC_MIC))
    {
        DBGPRINT(RT_DEBUG_ERROR, "MIC different in Msg 4 of 4-way handshake!! \n");
        return;
    }
    else
        DBGPRINT(RT_DEBUG_TRACE, "MIC valid in Msg 4 of 4-way handshake!! \n");

    // 3. uses the MLME.SETKEYS.request to configure PTK into MAC
    NdisZeroMemory(&pEntry->PairwiseKey[0], sizeof(WPA_KEY));   
    NdisMoveMemory(pEntry->PairwiseKey[0].RxTsc, EAPOLPKT.KeyDesc.RSC, 6);

    pEntry->PairwiseKey[0].KeyLen = LEN_TKIP_EK;
    NdisMoveMemory(pEntry->PairwiseKey[0].Key, &pEntry->PTK[32], LEN_TKIP_EK);
    NdisMoveMemory(pEntry->PairwiseKey[0].RxMic, &pEntry->PTK[TKIP_AP_RXMICK_OFFSET], LEN_TKIP_RXMICK);
    NdisMoveMemory(pEntry->PairwiseKey[0].TxMic, &pEntry->PTK[TKIP_AP_TXMICK_OFFSET], LEN_TKIP_TXMICK);
    
    // 4. upgrade state
    pEntry->PrivacyFilter = Ndis802_11PrivFilterAcceptAll;
    pEntry->WpaState = AS_PTKINITDONE;
    pAdapter->PortCfg.PortSecured = WPA_802_1X_PORT_SECURED;
    pEntry->PortSecured = WPA_802_1X_PORT_SECURED;
        
    // 5. init Group 2-way handshake if necessary.
    if ((Status = WPAHardTransmit(pAdapter, pEntry)) != NDIS_STATUS_SUCCESS)
        return;
    
    if (pEntry->RetryTimerRunning == FALSE)
    {
        DBGPRINT(RT_DEBUG_TRACE, "init_timer : add_timer Timer ===>>\n");
        pEntry->RetryTimerRunning = TRUE;
        init_timer(&pEntry->RetryTimer);
        pEntry->RetryTimer.expires = jiffies + WPA_RETRY_EXEC_INTV;
        pEntry->RetryTimer.data = (unsigned long)pAdapter;
        pEntry->RetryTimer.function = &WPARetryExec;
        add_timer(&pEntry->RetryTimer);
        pEntry->ReTryCounter = GRP_HS_RETRY_TIMER_CTR;
    }

    DBGPRINT(RT_DEBUG_TRACE, "<<==== WpaEAPOL Peer PAIR Msg4 Action \n");
}

/*
    ==========================================================================
    Description:
        When receiving the last packet of 2-way groupkey handshake.
    Return:
    ==========================================================================
*/
VOID PeerGroupMsg2Action(
    IN PRTMP_ADAPTER    pAdapter, 
    IN MAC_TABLE_ENTRY  *pEntry,
    IN VOID             *Msg,
    IN UINT             MsgLen)
{
    UINT                Len,i;
    UCHAR               mic[LEN_KEY_DESC_MIC];
    PUCHAR              pData;
    PEAPOL_PACKET       pMsg2;
    EAPOL_PACKET        EAPOLPKT;

    Len = MsgLen - LENGTH_802_1_H;
    pData = (PUCHAR)Msg;
    NdisZeroMemory((PUCHAR)&EAPOLPKT, sizeof(EAPOLPKT));
    NdisZeroMemory(mic, sizeof(mic));
    DBGPRINT(RT_DEBUG_TRACE, "PeerGroupMsg2Action ====> from MAC : %x %x %x %x %x %x   \n",pEntry->Addr.Octet[0],\
        pEntry->Addr.Octet[1], pEntry->Addr.Octet[2],pEntry->Addr.Octet[3],pEntry->Addr.Octet[4],pEntry->Addr.Octet[5]);

    if (MsgLen < (LENGTH_802_1_H + LENGTH_EAPOL_H + sizeof(KEY_DESCRIPTER) -MAX_LEN_OF_RSNIE -2 ) )
        return;
            
    if (pEntry->WpaState != AS_PTKINITDONE)
        return;
        
    pMsg2 = (PEAPOL_PACKET) (pData + LENGTH_802_1_H);

    // 1. verify the Reaply counter, if not valid,
    if (!RTMPEqualMemory(pMsg2->KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
    {
        DBGPRINT(RT_DEBUG_ERROR, "Replay Counter Different in Msg 2 of GROUP 2-way handshake!!!  \n");
        DBGPRINT(RT_DEBUG_ERROR, "Receive :... %d %d %d %d  ", pMsg2->KeyDesc.RCounter[4], pMsg2->KeyDesc.RCounter[5], pMsg2->KeyDesc.RCounter[6], pMsg2->KeyDesc.RCounter[7]);
        DBGPRINT(RT_DEBUG_ERROR, "Current :... %d %d %d %d  \n",pEntry->R_Counter[4],pEntry->R_Counter[5],pEntry->R_Counter[6],pEntry->R_Counter[7]);
        return;
    }
    else
        DBGPRINT(RT_DEBUG_TRACE, "Replay Counter VALID in Msg 2 of GROUP 2-way handshake!!!   \n");

    // 2. verify MIC, 
    NdisMoveMemory(&EAPOLPKT, pMsg2, Len);  
    NdisZeroMemory(&EAPOLPKT.KeyDesc.MIC, LEN_KEY_DESC_MIC);
    if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption2Enabled)
    {
        hmac_md5(pEntry->PTK,   LEN_EAP_MICK, (UCHAR *)&EAPOLPKT, Len, mic);
    }
    else
    {
        UCHAR digest[80];
        HMAC_SHA1((UCHAR *)&EAPOLPKT,   Len, pEntry->PTK, LEN_EAP_MICK, digest);
        NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
    }
    if (!RTMPEqualMemory(pMsg2->KeyDesc.MIC, mic, LEN_KEY_DESC_MIC))
    {
        DBGPRINT(RT_DEBUG_ERROR, "MIC  Different in Msg 2 of GROUP 2-way handshake!! \n");
        return;
    }
    else
        DBGPRINT(RT_DEBUG_ERROR, "MIC Valid  in Msg 2 of GROUP 2-way handshake!! \n");

    // 3.  upgrade state
    if (pEntry->RetryTimerRunning == TRUE)
    {
        DBGPRINT(RT_DEBUG_TRACE, "Cancel Retry Timer ! \n");
        del_timer_sync(&pEntry->RetryTimer);
        pEntry->RetryTimerRunning = FALSE;              
    }
    pEntry->GTKState = REKEY_ESTABLISHED;
    pEntry->Flag &= (~DISASSOC_GRP_UDATE_TMOUT);    
    
    if (pAdapter->PortCfg.GKeyDoneStations > 0 )
        pAdapter->PortCfg.GKeyDoneStations--;
        
    if (pAdapter->PortCfg.GKeyDoneStations == 0)
    {
        pAdapter->PortCfg.WpaGTKState = SETKEYS_DONE;
        DBGPRINT(RT_DEBUG_TRACE, "AP SETKEYS_DONE!!!\n\n");
    }
}

/*
    ==========================================================================
    Description:
        Only for sending the first packet of 2-way groupkey handshake
    Return:
    ==========================================================================
*/
NDIS_STATUS WPAHardTransmit(
    IN PRTMP_ADAPTER    pAdapter,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    UINT                i;
    UCHAR               GTK[TKIP_GTK_LENGTH],pRc4GTK[TKIP_GTK_LENGTH], mic[LEN_KEY_DESC_MIC];
    UCHAR               ekey[(LEN_KEY_DESC_IV+LEN_EAP_EK)];
    ULONG               FrameLen = 0;
    UCHAR               *OutBuffer = NULL;
    PUCHAR              pDest;
    UCHAR               Header802_3[14];
    EAPOL_PACKET        Packet;

    NdisZeroMemory(ekey, sizeof(ekey));
    NdisZeroMemory(pRc4GTK, sizeof(pRc4GTK));

    do
    {
        OutBuffer = kmalloc(MAX_LEN_OF_EAP_HS, GFP_KERNEL);
        if(OutBuffer == NULL)
            return (NDIS_STATUS_FAILURE);

        // Increment replay counter by 1  
        i = LEN_KEY_DESC_REPLAY;
        do
        {
            i--;
            pAdapter->PortCfg.R_Counter[i]++;
            if (i == 0)
                break;
        }
        while (pAdapter->PortCfg.R_Counter[i] == 0);

        NdisMoveMemory(pEntry->R_Counter, pAdapter->PortCfg.R_Counter, sizeof(pEntry->R_Counter));
        // 0. init Packet and Fill header
        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;
        Packet.Body_Len[1] = LEN_MSG1_2WAY;
        // 2, Fill key version, keyinfo, key len
        Packet.KeyDesc.Type = RSN_KEY_DESC;
        Packet.KeyDesc.Keyinfo.KeyMic = 1;
        Packet.KeyDesc.Keyinfo.Secure = 1;
        Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        Packet.KeyDesc.Keyinfo.KeyType = GROUPKEY;
        Packet.KeyDesc.Keyinfo.KeyIndex = pAdapter->PortCfg.WPAGKeyID;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;
        Packet.KeyDesc.KeyLength[1] = TKIP_GTK_LENGTH;
        Packet.KeyDesc.DataLen[1] = TKIP_GTK_LENGTH;
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.Body_Len[1] -= 8;
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
            Packet.KeyDesc.DataLen[1] = LEN_AES_KEY + 8;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_AES;
        }

#ifdef BIG_ENDIAN
        *(USHORT *)&Packet.KeyDesc.Keyinfo = SWAP16(*(USHORT *)&Packet.KeyDesc.Keyinfo);
#endif

        CountGTK(pAdapter->PortCfg.GMK, (UCHAR*)pAdapter->PortCfg.GNonce, pAdapter->CurrentAddress, GTK, TKIP_GTK_LENGTH);
        DBGPRINT(RT_DEBUG_TRACE, "WPA Group Key ID  = %d\n",pAdapter->PortCfg.WPAGKeyID);
        DBGPRINT(RT_DEBUG_INFO, " %x %x %x %x %x %x %x %x ",GTK[0],GTK[1],GTK[2],GTK[3],GTK[4],GTK[5],GTK[6],GTK[7]);
        DBGPRINT(RT_DEBUG_INFO, "  %x %x %x %x %x %x %x %x \n",GTK[8],GTK[9],GTK[10],GTK[11],GTK[12],GTK[13],GTK[14],GTK[15]);
        DBGPRINT(RT_DEBUG_INFO, "  %x %x %x %x %x %x %x %x",GTK[16],GTK[17],GTK[18],GTK[19],GTK[20],GTK[21],GTK[22],GTK[23]);
        DBGPRINT(RT_DEBUG_INFO, "  %x %x %x %x %x %x  %x %x \n",GTK[24],GTK[25],GTK[26],GTK[27],GTK[28],GTK[29],GTK[30],GTK[31]);
                
        NdisMoveMemory(pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].Key, GTK, LEN_TKIP_EK);
        NdisMoveMemory(pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].TxMic, &GTK[16], LEN_TKIP_TXMICK);
        NdisMoveMemory(pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].RxMic, &GTK[24], LEN_TKIP_RXMICK);            
        WPAMake8023Hdr(pAdapter, pEntry->Addr.Octet, Header802_3);
        NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY);
        NdisMoveMemory(Packet.KeyDesc.Nonce, pAdapter->PortCfg.GNonce, LEN_KEY_DESC_NONCE);
        NdisMoveMemory(Packet.KeyDesc.IV, &pAdapter->PortCfg.GNonce[16], sizeof(LEN_KEY_DESC_IV));
        // Suggest IV be random number plus some number,
        Packet.KeyDesc.IV[15] += 2;
        NdisMoveMemory(Packet.KeyDesc.RSC, pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].TxTsc, 6);
        // Count EAPOL MIC , and encrypt DATA field before Send,   DATA fields includes the encrypted GTK
        if (pAdapter->PortCfg.WepStatus == Ndis802_11Encryption3Enabled)
        {
            UCHAR digest[80];  // 80 to prevent overflow
            AES_GTK_KEY_WRAP(&pEntry->PTK[16], GTK, pRc4GTK);
            // First make a frame  for Countint MIC,
            NdisMoveMemory(Packet.KeyDesc.Data,  pRc4GTK, (LEN_AES_KEY +8));
            MakeOutgoingFrame(OutBuffer,                &FrameLen,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);

            HMAC_SHA1(OutBuffer,  FrameLen, pEntry->PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(&Packet.KeyDesc.MIC, digest, LEN_KEY_DESC_MIC);
            // make  outgoing frame
            MakeOutgoingFrame(OutBuffer,                &FrameLen,
                            LENGTH_802_3,                   &Header802_3,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);
        }
        else
        {
            // PREPARE Encrypted  "Key DATA" field.  (Encrypt GTK with RC4, usinf PTK[16]->[31] as Key, IV-field as IV)
            // put TxTsc in Key RSC field
            pAdapter->PrivateInfo.FCSCRC32 = PPPINITFCS32;   //Init crc32.

            // ekey is the contanetion of IV-field, and PTK[16]->PTK[31]
            NdisMoveMemory(ekey, Packet.KeyDesc.IV, LEN_KEY_DESC_IV);
            NdisMoveMemory(&ekey[LEN_KEY_DESC_IV], &pEntry->PTK[16], LEN_EAP_EK);
            ARCFOUR_INIT(&pAdapter->PrivateInfo.WEPCONTEXT, ekey, sizeof(ekey));  //INIT SBOX, KEYLEN+3(IV)
            pAdapter->PrivateInfo.FCSCRC32 = RTMP_CALC_FCS32(pAdapter->PrivateInfo.FCSCRC32, GTK, TKIP_GTK_LENGTH);
            WPAARCFOUR_ENCRYPT(&pAdapter->PrivateInfo.WEPCONTEXT, pRc4GTK,GTK,  TKIP_GTK_LENGTH);
            NdisMoveMemory(Packet.KeyDesc.Data,  pRc4GTK, TKIP_GTK_LENGTH);

            // make a frame for Countint MIC,
            MakeOutgoingFrame(OutBuffer,                    &FrameLen,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);

            hmac_md5(pEntry->PTK, LEN_EAP_MICK, OutBuffer, FrameLen, mic);
            NdisMoveMemory(Packet.KeyDesc.MIC, mic, LEN_KEY_DESC_MIC);
            // make  outgoing frame
            MakeOutgoingFrame(OutBuffer,                    &FrameLen,
                            LENGTH_802_3,                   &Header802_3,
                            Packet.Body_Len[1] + 4,  &Packet,
                            END_OF_ARGS);
        }
         
        DBGPRINT(RT_DEBUG_TRACE,"<== Hard Transmit : FrameLen = %d \n", FrameLen);
        RTMPToWirelessSta(pAdapter, OutBuffer, FrameLen);
    }while (FALSE);
    kfree(OutBuffer);
    
    return (NDIS_STATUS_SUCCESS);
}

VOID WpaEAPOLASFAlertAction(
    IN PRTMP_ADAPTER    pAd, 
    IN MLME_QUEUE_ELEM  *Elem) 
{   
}

VOID DisAssocAction(
    IN PRTMP_ADAPTER    pAd, 
    IN MAC_TABLE_ENTRY  *pEntry,
    IN USHORT           SubType,
    IN USHORT           Reason)
{
    CHAR            *OutBuffer = NULL;
    ULONG           FrameLen = 0;
    MACHDR          DisassocHdr;
    MACADDR         addr;
    NDIS_STATUS     NStatus;

    //  send out a DISASSOC request frame
    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
    if(OutBuffer == NULL)
        return;
    
    NdisMoveMemory(addr.Octet, pEntry->Addr.Octet, MAC_ADDR_LEN);
    DBGPRINT(RT_DEBUG_ERROR, "ASSOC - Send DISASSOC  Reason = %d frame  TO %x %x %x %x %x %x \n",Reason,pEntry->Addr.Octet[0],\
    pEntry->Addr.Octet[1],pEntry->Addr.Octet[2],pEntry->Addr.Octet[3],pEntry->Addr.Octet[4],pEntry->Addr.Octet[5]);
    MgtMacHeaderInit(pAd, &DisassocHdr, SubType, 0, &addr, &pAd->PortCfg.Bssid);
    MakeOutgoingFrame(OutBuffer, &FrameLen, sizeof(MACHDR), (CHAR *)&DisassocHdr, 2, (char*)&Reason, END_OF_ARGS);
    MiniportMMRequest(pAd, OutBuffer, FrameLen);
    if (pEntry)
    {  
        if (pEntry->RetryTimerRunning == TRUE)
        {
            DBGPRINT(RT_DEBUG_TRACE, " del_timer_sync in DisAssocAction !");
            del_timer_sync(&pEntry->RetryTimer);
            pEntry->RetryTimerRunning = FALSE;              
        }     
        //ApLogEvent(pAd, &pEntry->Addr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd,  &pEntry->Addr);
    }
}

/*
    ==========================================================================
    Description:
        Timer execution function for periodically updating group key.
    Return:
    ==========================================================================
*/  
VOID GREKEYPeriodicExec(
    IN  unsigned long data) 
{
    UINT            i;
    ULONG           temp_counter = 0;
    RTMP_ADAPTER    *pAdapter = (RTMP_ADAPTER *)data;
    MAC_TABLE_ENTRY *pEntry;
    BOOLEAN         bTimer = TRUE;
        
    DBGPRINT(RT_DEBUG_INFO, "GROUP REKEY PeriodicExec ==>>   \n");

    if ((pAdapter->PortCfg.AuthMode != Ndis802_11AuthModeWPA) && (pAdapter->PortCfg.AuthMode != Ndis802_11AuthModeWPAPSK))
    {
        bTimer = FALSE;
        pAdapter->PortCfg.REKEYTimerRunning = FALSE;
    }    
    else if (pAdapter->PortCfg.WPAREKEY.ReKeyInterval == 0)
    {
        bTimer = FALSE;
        pAdapter->PortCfg.REKEYTimerRunning = FALSE;
    }
    
    if ((pAdapter->PortCfg.WPAREKEY.ReKeyMethod == TIME_REKEY) && (pAdapter->PortCfg.REKEYCOUNTER < 0xffffffff))
        temp_counter = (++pAdapter->PortCfg.REKEYCOUNTER);
    // REKEYCOUNTER is incremented every TX_RING_SIZE packets transmitted, 
    // But the unit of Rekeyinterval is 1K packets
    else if (pAdapter->PortCfg.WPAREKEY.ReKeyMethod == PKT_REKEY )
        temp_counter = pAdapter->PortCfg.REKEYCOUNTER/TX_RING_SIZE;
    else
    {
        bTimer = FALSE;
        pAdapter->PortCfg.REKEYTimerRunning = FALSE;
    }
    
    if (temp_counter > (pAdapter->PortCfg.WPAREKEY.ReKeyInterval))
    {
        pAdapter->PortCfg.REKEYCOUNTER = 0;
        DBGPRINT(RT_DEBUG_TRACE, "Rekey Interval Excess, GKeyDoneStations= %d  \n",pAdapter->MacTab.Size);
        pAdapter->PortCfg.WpaGTKState = SETKEYS;
        // take turn updating different groupkey index, 
        if ((pAdapter->PortCfg.GKeyDoneStations = pAdapter->MacTab.Size) > 0)
        {
            pAdapter->PortCfg.WPAGKeyID ++;
            if (pAdapter->PortCfg.WPAGKeyID >=4 )
                pAdapter->PortCfg.WPAGKeyID = 1;
            GenRandom(pAdapter, pAdapter->PortCfg.GNonce);
            for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
            {
                if ((pAdapter->MacTab.Content[i].Valid == TRUE) && (pAdapter->MacTab.Content[i].WpaState == AS_PTKINITDONE))
                {
                    pAdapter->MacTab.Content[i].GTKState = REKRY_NEGOTIATING;
                    NdisZeroMemory(pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].TxTsc, sizeof(pAdapter->PortCfg.GroupKey[pAdapter->PortCfg.WPAGKeyID].TxTsc));
                    WPAHardTransmit(pAdapter, &pAdapter->MacTab.Content[i]);
                    DBGPRINT(RT_DEBUG_TRACE, "Rekey interval excess, Update Group Key for %x %x %x %x %x %x , WPAGKeyID= %x \n",\
                        pAdapter->MacTab.Content[i].Addr.Octet[0],pAdapter->MacTab.Content[i].Addr.Octet[1],\
                        pAdapter->MacTab.Content[i].Addr.Octet[2],pAdapter->MacTab.Content[i].Addr.Octet[3],\
                        pAdapter->MacTab.Content[i].Addr.Octet[4],pAdapter->MacTab.Content[i].Addr.Octet[5],\
                        pAdapter->PortCfg.WPAGKeyID);
                }
            }
        }
    }
        
    if(bTimer)
    {
        pAdapter->PortCfg.REKEYTimer.expires = jiffies + GUPDATE_EXEC_INTV;
        add_timer(&pAdapter->PortCfg.REKEYTimer);
        pAdapter->PortCfg.REKEYTimerRunning = TRUE;
    }
}

VOID WPAMake8023Hdr(
    IN PRTMP_ADAPTER    pAd, 
    IN PCHAR            pDAddr, 
    IN OUT PCHAR        pHdr)
{    
     // Addr1: DA, Addr2: BSSID, Addr3: SA
    NdisMoveMemory(pHdr, pDAddr, MAC_ADDR_LEN);
    NdisMoveMemory(&pHdr[MAC_ADDR_LEN], pAd->CurrentAddress, MAC_ADDR_LEN);
    NdisMoveMemory(&pHdr[2*MAC_ADDR_LEN], EAPOL, LENGTH_802_3_TYPE);
}

/*
    ==========================================================================
    Description:
        Length of output is in octets rather than bits.  Since length is always a multiple of 8, output array is organized
        so first N octets starting from 0 contains PRF output.
        INPUT : supported input are 16, 32, 48, 64
        OUTPUT : output array should be 80 octets to allow for sha1 overflow
    Return:
    ==========================================================================
*/
VOID PRF(
    IN UCHAR    *Key,
    IN INT      key_len,
    IN UCHAR    *Prefix,
    IN INT      Prefix_len,
    IN UCHAR    *Data,
    IN INT      Data_len,
    OUT UCHAR   *Output,
    IN INT      Output_len)
{
    INT     i;
    UCHAR   input[1024];
    INT     currentindex=0;
    INT     total_len;
    
    NdisMoveMemory(input, Prefix, Prefix_len);
    input[Prefix_len] = 0;
    NdisMoveMemory(&input[Prefix_len+1], Data, Data_len);
    total_len = Prefix_len+1+Data_len;
    input[total_len] = 0;
    total_len++;
    for (i = 0; i < (Output_len+19)/20; i++)
    {
        HMAC_SHA1(input, total_len, Key, key_len, &Output[currentindex]);
        currentindex += 20;
        input[total_len-1]++;
    }
}

VOID CountPTK(
    IN UCHAR    *PMK,
    IN UCHAR    *ANonce,
    IN UCHAR    *AA,
    IN UCHAR    *SNonce,
    IN UCHAR    *SA,
    OUT UCHAR   *output,
    IN UINT     len)
{   
    UCHAR   concatenation[76];
    UINT    CurrPos=0;
    UCHAR   temp[32];
    INT     i;
    UCHAR   Prefix[] = {'P', 'a', 'i', 'r', 'w', 'i', 's', 'e', ' ', 'k', 'e', 'y', ' ', 
                        'e', 'x', 'p', 'a', 'n', 's', 'i', 'o', 'n'};

    NdisZeroMemory(temp, sizeof(temp));

    GetSmall(SA, AA, temp, 6);
    NdisMoveMemory(concatenation, temp, 6);
    CurrPos += 6;

    GetLarge(SA, AA, temp, 6);
    NdisMoveMemory(&concatenation[CurrPos], temp, 6);
    CurrPos += 6;

    GetSmall(ANonce, SNonce, temp, 32);
    NdisMoveMemory(&concatenation[CurrPos], temp, 32);
    CurrPos += 32;

    GetLarge(ANonce, SNonce, temp, 32);
    NdisMoveMemory(&concatenation[CurrPos], temp, 32);
    CurrPos += 32;
    
    PRF(PMK, PMK_LEN, Prefix, 22, concatenation, 76 , output, len);
}

VOID CountGTK(
    IN UCHAR    *GMK,
    IN UCHAR    *GNonce,
    IN UCHAR    *AA,
    OUT UCHAR   *output,
    IN UINT     len)
{
    UCHAR   concatenation[76];
    PUCHAR  pCurr;
    UINT    CurrPos=0;
    INT     i;
    UCHAR   temp[80];   
    UCHAR   Prefix[] = {'G', 'r', 'o', 'u', 'p', ' ',  'k', 'e', 'y', ' ', 
                        'e', 'x', 'p', 'a', 'n', 's', 'i', 'o', 'n'};

    NdisMoveMemory(&concatenation[CurrPos], AA, 6);
    CurrPos += 6;

    NdisMoveMemory(&concatenation[CurrPos], GNonce , 32);
    CurrPos += 32;

    PRF(GMK, PMK_LEN, Prefix, 19, concatenation, 38 , temp, len);
    NdisMoveMemory(output, temp, len);
}

VOID GetSmall(
    IN  PVOID   pSrc1,
    IN  PVOID   pSrc2,
    OUT PUCHAR  pOut,
    IN  ULONG   Length)
{
    PUCHAR  pMem1;
    PUCHAR  pMem2;
    ULONG   Index = 0;
    pMem1 = (PUCHAR) pSrc1;
    pMem2 = (PUCHAR) pSrc2;

    for (Index = 0; Index < Length; Index++)
    {
        if (pMem1[Index] != pMem2[Index])
        {
            if (pMem1[Index] > pMem2[Index])        
                NdisMoveMemory(pOut, pSrc2, Length);
            else
                NdisMoveMemory(pOut, pSrc1, Length);             

            break;
        }
    }
}

VOID GetLarge(
    IN  PVOID   pSrc1,
    IN  PVOID   pSrc2,
    OUT PUCHAR  pOut,
    IN  ULONG   Length)
{
    PUCHAR  pMem1;
    PUCHAR  pMem2;
    ULONG   Index = 0;
    pMem1 = (PUCHAR) pSrc1;
    pMem2 = (PUCHAR) pSrc2;

    for (Index = 0; Index < Length; Index++)
    {
        if (pMem1[Index] != pMem2[Index])
        {
            if (pMem1[Index] > pMem2[Index])        
                NdisMoveMemory(pOut, pSrc1, Length);
            else
                NdisMoveMemory(pOut, pSrc2, Length);             

            break;
        }
    }
}

// 802.1i  Annex F.9
VOID GenRandom(
    IN PRTMP_ADAPTER    pAd, 
    OUT UCHAR           *random)
{   
    INT             i, curr,cc;
    UCHAR           local[80];
    UCHAR           result[80];
    PUCHAR          pLook;
    LARGE_INTEGER   CurrentTime;
    UCHAR           prefix[] = {'I', 'n', 'i', 't', ' ', 'C', 'o', 'u', 'n', 't', 'e', 'r'};

    NdisZeroMemory(result, 80);
    NdisZeroMemory(local, 80);
    NdisMoveMemory(local, pAd->CurrentAddress, ETH_LENGTH_OF_ADDRESS);
    
    for (i = 0; i < 32; i++)
    {       
        curr =  ETH_LENGTH_OF_ADDRESS;
        CurrentTime = jiffies;
        NdisMoveMemory(local,  pAd->CurrentAddress, ETH_LENGTH_OF_ADDRESS);
        curr += ETH_LENGTH_OF_ADDRESS;
        NdisMoveMemory(&local[curr],  &CurrentTime, sizeof(CurrentTime));
        curr += sizeof(CurrentTime);
        NdisMoveMemory(&local[curr],  result, 32);
        curr += 32;
        NdisMoveMemory(&local[curr],  &i,  2);      
        curr += 2;
        PRF(pAd->PortCfg.Key_Counter, 32, prefix,12, local, curr, result, 32); 
    }

    for (i = 32; i > 0; i--)
    {   
        if (pAd->PortCfg.Key_Counter[i-1] == 0xff)
        {
            pAd->PortCfg.Key_Counter[i-1] = 0;
        }
        else
        {
            pAd->PortCfg.Key_Counter[i-1]++;
            break;
        }
    }
    NdisMoveMemory(random, result,  32);
}

/*
    ==========================================================================
    Description:
        ENCRYPT AES GTK before sending in EAPOL frame.
        AES GTK length = 128 bit,  so fix blocks for aes-key-wrap as 2 in this function.
        This function references to RFC 3394 for aes key wrap algorithm.
    Return:
    ==========================================================================
*/  
VOID AES_GTK_KEY_WRAP( 
    IN UCHAR    *key,
    IN UCHAR    *plaintext,
    OUT UCHAR   *ciphertext)
{
    UCHAR           A[8], BIN[16], BOUT[16];
    UCHAR           R1[8],R2[8];
    INT             num_blocks = 2;
    INT             i, j;
    aes_context     aesctx;
    UCHAR           xor;

    // init
    for (i = 0; i < 8; i++)
        A[i] = 0xa6;
    NdisMoveMemory(R1, plaintext, 8);
    NdisMoveMemory(R2, &plaintext[8], 8);
    aes_set_key(&aesctx, key, 128);

    for (j = 0; j < 6; j++)
    {
        NdisMoveMemory(BIN, A, 8);
        NdisMoveMemory(&BIN[8], R1, 8);
        aes_encrypt(&aesctx, BIN, BOUT );
        xor=num_blocks*j +1;
        NdisMoveMemory(A, &BOUT[0], 8);
        A[7] = BOUT[7] ^ xor;
        NdisMoveMemory(R1, &BOUT[8], 8);
        xor=num_blocks*j+2;

        NdisMoveMemory(BIN, A, 8);
        NdisMoveMemory(&BIN[8], R2, 8);
        aes_encrypt(&aesctx, BIN, BOUT );

        NdisMoveMemory(A, &BOUT[0], 8);
        A[7] = BOUT[7] ^ xor;
        NdisMoveMemory(R2, &BOUT[8], 8);            
    }

    // OUTPUT
    NdisMoveMemory(ciphertext, A, 8);
    NdisMoveMemory(&ciphertext[8], R1, 8);
    NdisMoveMemory(&ciphertext[16], R2, 8);
}

/*
    ========================================================================
    Routine Description:
       Send all EAP frames to wireless station.
       These frames don't come from normal SendPackets routine, but are EAPPacket, EAPOL, 
        
    Arguments:
        pRxD        Pointer to the Rx descriptor
        
    Return Value:
None
========================================================================
*/
VOID RTMPToWirelessSta(
    IN  PRTMP_ADAPTER   pAdapter,
    IN  PUCHAR          pFrame,
    IN  UINT            FrameLen)
{
    SST             Sst;
    USHORT          Aid;
    UCHAR           PsMode, Rate;
    MAC_TABLE_ENTRY *pEntry;
    BOOLEAN         result = FALSE;
    struct sk_buff *skb;

    DBGPRINT(RT_DEBUG_INFO, "RTMPToWirelessSta ====>>   \n");

    // decide the return value. TRUE if no need to indicate to LLC, FALSE otherwise
    pEntry = SsPsInquiry(pAdapter, (MACADDR *)pFrame, &Sst, &Aid, &PsMode, &Rate);
    if ((pAdapter->PortCfg.IsolateInterStaTraffic == 0) && (pEntry && (Sst == SST_ASSOC)))
    {
        NDIS_STATUS     Status;
        UCHAR           *pVirtualAddress;        

        do {
            // 1. build a NDIS packet and call RTMPSendPacket();
            //    be careful about how/when to release this internal allocated NDIS PACKET buffer
#ifdef RTMP_EMBEDDED
            if ((skb = __dev_alloc_skb(FrameLen + 2, GFP_DMA|GFP_ATOMIC)) != NULL)
#else
            if ((skb = dev_alloc_skb(FrameLen + 2)) != NULL)
#endif
            {
                skb->len = FrameLen;
                memcpy((skb->data), pFrame, FrameLen);
            }
            else
            {
                break;
            }

            // 2. send out the packet
            Status = RTMPSendPacket(pAdapter, skb);
            if (Status == NDIS_STATUS_SUCCESS)
            {
                // Dequeue one frame from TxSwQueue0..3 queue and process it
                // There are three place calling dequeue for TX ring.
                // 1. Here, right after queueing the frame.
                // 2. At the end of TxRingTxDone service routine.
                // 3. Upon NDIS call RTMPSendPackets
                if ((!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS)) && (!RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS)))
                {
                    RTMPDeQueuePacket(pAdapter);
                }   
            }
            else // free this packet space
            {
                RTMPFreeSkbBuffer(skb);
            }
        } while (FALSE);
    }
}

/*
    ========================================================================

    Routine Description:
        Sending EAP Req. frame to station in authenticating state.
        These frames come from Authenticator deamon.

    Arguments:
        pAdapter        Pointer to our adapter
        pPacket     Pointer to outgoing EAP frame body + 8023 Header
        Len             length of pPacket
        
    Return Value:
        None
    ========================================================================
*/
VOID WpaSend(
    IN  PRTMP_ADAPTER   pAdapter,
    IN  PUCHAR          pPacket,
    IN  ULONG           Len)
{
    UCHAR           FrameGap;
    PUCHAR          pDest;
    PUCHAR          pSrc;
    ULONG           Iv16;
    ULONG           Iv32;
    PWPA_KEY        pWpaKey;
    PEAP_HDR        pEapHdr;
    UCHAR           RetryMode = SHORT_RETRY;
    MACADDR         Addr;
    MAC_TABLE_ENTRY *pEntry;
    
    DBGPRINT(RT_DEBUG_TRACE, "WpaHardEncrypt ====>> Len=%d  \n",Len);

    RTMPToWirelessSta(pAdapter, pPacket, Len);
    NdisMoveMemory(&(Addr.Octet), pPacket, 6);
    pEapHdr = (EAP_HDR*)(pPacket + LENGTH_802_3  + LENGTH_8021X_HDR);
    if ((pEntry = MacTableLookup(pAdapter, &Addr)) == NULL)
        return;    

    if (RTMPEqualMemory((pPacket+12), EAPOL, 2) )
    {
        switch (pEapHdr->code)
        {
            case EAP_CODE_SUCCESS:
                DBGPRINT(RT_DEBUG_TRACE,"Send EAP_CODE_SUCCESS \n");
                if (pEntry && (pEntry->Sst == SST_ASSOC))
                {
                    pEntry->WpaState = AS_INITPMK;
                    WPAStart4WayHS(pAdapter, pEntry);
                    if ((pEntry->RetryTimerRunning == FALSE) )
                    {
                        // retry up to 3 times at interval 1 second
                        DBGPRINT(RT_DEBUG_TRACE, "add_timer  Timer   ====>>>   \n");
                        pEntry->RetryTimerRunning = TRUE;
                        init_timer(&pEntry->RetryTimer);
                        pEntry->RetryTimer.expires = jiffies + WPA_RETRY_EXEC_INTV;
                        pEntry->RetryTimer.data = (unsigned long)pAdapter;
                        pEntry->RetryTimer.function = &WPARetryExec;
                        add_timer(&pEntry->RetryTimer);
                        pEntry->ReTryCounter = PAIR_HS_RETRY_TIMER_CTR;
                    }                
                }
                break;

            case EAP_CODE_FAILURE:
                break;

            default:
                break;    
        }
    }
    else     
    {
        DBGPRINT(RT_DEBUG_TRACE, "Send Deauth , Reason : REASON_NO_LONGER_VALID  \n");
        DisAssocAction(pAdapter, pEntry, SUBTYPE_DEAUTH, REASON_NO_LONGER_VALID);
    }
}

VOID RTMPMakeRSNIE(
    IN  PRTMP_ADAPTER   pAdapter,
    IN  UINT            AuthMode,
    IN  UINT            WepStatus)
{
    RSNIE       *pRsnie;
    RSNIE_AUTH  *pRsnie_auth;
    UCHAR       oui01[4] = {0x00, 0x50, 0xf2, 0x01};
    UCHAR       oui02[4] = {0x00, 0x50, 0xf2, 0x02};
    UCHAR       oui04[4] = {0x00, 0x50, 0xf2, 0x04};


    pRsnie = (RSNIE*)pAdapter->PortCfg.RSN_IE;    
    NdisMoveMemory(pRsnie->oui, oui01, 4);
    pRsnie->version = 1;

    switch (WepStatus)
    {
        case Ndis802_11Encryption2Enabled:
            NdisMoveMemory(pRsnie->mcast, oui02, 4);
            // currently support one unicast cipher. one auth  
            pRsnie->ucount = 1;
            NdisMoveMemory(pRsnie->ucast[0].oui, oui02, 4);
            break;

        case Ndis802_11Encryption3Enabled:
            NdisMoveMemory(pRsnie->mcast, oui04, 4);
            // currently support one unicast cipher.
            pRsnie->ucount = 1;
            NdisMoveMemory(pRsnie->ucast[0].oui, oui04, 4);
            break;
    }

    pRsnie_auth = (RSNIE_AUTH*)(pAdapter->PortCfg.RSN_IE + sizeof(RSNIE));

    switch (AuthMode)
    {
        case Ndis802_11AuthModeWPA:
            pRsnie_auth->acount = 1;
            NdisMoveMemory(pRsnie_auth->auth[0].oui, oui01, 4);
            DBGPRINT(RT_DEBUG_INFO,"auth = %x %x %x %x %x %x  \n",pAdapter->PortCfg.RSN_IE[16],pAdapter->PortCfg.RSN_IE[17],
                pAdapter->PortCfg.RSN_IE[18],pAdapter->PortCfg.RSN_IE[19],pAdapter->PortCfg.RSN_IE[20],pAdapter->PortCfg.RSN_IE[21]);
            break;

        case Ndis802_11AuthModeWPAPSK:
            pRsnie_auth->acount = 1;
            NdisMoveMemory(pRsnie_auth->auth[0].oui, oui02, 4);
            DBGPRINT(RT_DEBUG_INFO,"auth = %x %x %x %x %x %x  \n",pAdapter->PortCfg.RSN_IE[16],pAdapter->PortCfg.RSN_IE[17],
                pAdapter->PortCfg.RSN_IE[18],pAdapter->PortCfg.RSN_IE[19],pAdapter->PortCfg.RSN_IE[20],pAdapter->PortCfg.RSN_IE[21]);
            break;
    }
#ifdef BIG_ENDIAN
    pRsnie->version = SWAP16(pRsnie->version);
    pRsnie->ucount = SWAP16(pRsnie->ucount);
    pRsnie_auth->acount = SWAP16(pRsnie_auth->acount);
#endif
    
    pAdapter->PortCfg.RSNIE_Len = sizeof(RSNIE) + sizeof(RSNIE_AUTH);

    DBGPRINT(RT_DEBUG_TRACE,"RTMPMakeRSNIE : RSNIE_Len = %d\n",pAdapter->PortCfg.RSNIE_Len);
}

