/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is SHA 180-1 Reference Implementation (Compact version)
 * 
 * The Initial Developer of the Original Code is Paul Kocher of
 * Cryptography Research.  Portions created by Paul Kocher are 
 * Copyright (C) 1995-9 by Cryptography Research, Inc.  All
 * Rights Reserved.
 * 
 * Contributor(s):
 *  (Endian OK) tt_lin_big
 */

#include "rt_config.h"


#define SHA_ROTL(X,n) ((((X) << (n)) | ((X) >> (32-(n)))) & 0xffffffffL)

void A_SHAInit(A_SHA_CTX *ctx) 
{
	int i;

	ctx->lenW = 0;
	ctx->sizeHi = ctx->sizeLo = 0;

	/* Initialize H with the magic constants (see FIPS180 for constants)
	 */
	ctx->H[0] = 0x67452301L;
	ctx->H[1] = 0xefcdab89L;
	ctx->H[2] = 0x98badcfeL;
	ctx->H[3] = 0x10325476L;
	ctx->H[4] = 0xc3d2e1f0L;

	for (i = 0; i < 80; i++)
		ctx->W[i] = 0;
}


void A_SHAUpdate(A_SHA_CTX *ctx, unsigned char *dataIn, int len)
{
	int i;

	/* Read the data into W and process blocks as they get full */
	for (i = 0; i < len; i++) 
	{
		ctx->W[ctx->lenW / 4] <<= 8;
		ctx->W[ctx->lenW / 4] |= (unsigned long)dataIn[i];
		if ((++ctx->lenW) % 64 == 0) 
		{
			shaHashBlock(ctx);
			ctx->lenW = 0;
		}
		ctx->sizeLo += 8;
		ctx->sizeHi += (ctx->sizeLo < 8);
	}
}


void A_SHAFinal(A_SHA_CTX *ctx, unsigned char hashout[20])
{
	unsigned char pad0x80 = 0x80;
	unsigned char pad0x00 = 0x00;
	unsigned char padlen[8];
	int i;

	/* Pad with a binary 1 (e.g. 0x80), then zeroes, then length */
	padlen[0] = (unsigned char)((ctx->sizeHi >> 24) & 255);
	padlen[1] = (unsigned char)((ctx->sizeHi >> 16) & 255);
	padlen[2] = (unsigned char)((ctx->sizeHi >> 8) & 255);
	padlen[3] = (unsigned char)((ctx->sizeHi >> 0) & 255);
	padlen[4] = (unsigned char)((ctx->sizeLo >> 24) & 255);
	padlen[5] = (unsigned char)((ctx->sizeLo >> 16) & 255);
	padlen[6] = (unsigned char)((ctx->sizeLo >> 8) & 255);
	padlen[7] = (unsigned char)((ctx->sizeLo >> 0) & 255);
	A_SHAUpdate(ctx, &pad0x80, 1);
	while (ctx->lenW != 56)
		A_SHAUpdate(ctx, &pad0x00, 1);
	A_SHAUpdate(ctx, padlen, 8);

	/* Output hash
	 */
	for (i = 0; i < 20; i++) 
	{
		hashout[i] = (unsigned char)(ctx->H[i / 4] >> 24);
		ctx->H[i / 4] <<= 8;
	}

	/*
	 *  Re-initialize the context (also zeroizes contents)
	 */
	A_SHAInit(ctx); 
}

void shaHashBlock(A_SHA_CTX *ctx)
{
	int t;
	unsigned long A,B,C,D,E,TEMP;

	for (t = 16; t <= 79; t++)
		ctx->W[t] = SHA_ROTL(ctx->W[t-3] ^ ctx->W[t-8] ^ ctx->W[t-14] ^ ctx->W[t-16], 1);

	A = ctx->H[0];
	B = ctx->H[1];
	C = ctx->H[2];
	D = ctx->H[3];
	E = ctx->H[4];

	for (t = 0; t <= 19; t++) 
	{
		TEMP = (SHA_ROTL(A,5) + (((C^D)&B)^D)     + E + ctx->W[t] + 0x5a827999L) & 0xffffffffL;
		E = D; D = C; C = SHA_ROTL(B, 30); B = A; A = TEMP;
	}
	for (t = 20; t <= 39; t++) 
	{
		TEMP = (SHA_ROTL(A,5) + (B^C^D)           + E + ctx->W[t] + 0x6ed9eba1L) & 0xffffffffL;
		E = D; D = C; C = SHA_ROTL(B, 30); B = A; A = TEMP;
	}
	for (t = 40; t <= 59; t++) 
	{
		TEMP = (SHA_ROTL(A,5) + ((B&C)|(D&(B|C))) + E + ctx->W[t] + 0x8f1bbcdcL) & 0xffffffffL;
		E = D; D = C; C = SHA_ROTL(B, 30); B = A; A = TEMP;
	}
	for (t = 60; t <= 79; t++) 
	{
		TEMP = (SHA_ROTL(A,5) + (B^C^D)           + E + ctx->W[t] + 0xca62c1d6L) & 0xffffffffL;
		E = D; D = C; C = SHA_ROTL(B, 30); B = A; A = TEMP;
	}

	ctx->H[0] += A;
	ctx->H[1] += B;
	ctx->H[2] += C;
	ctx->H[3] += D;
	ctx->H[4] += E;
}

static void hmac_sha1(unsigned char *text, int text_len, unsigned char *key, int key_len, unsigned char *digest) 
{ 
	A_SHA_CTX context; 
	unsigned char k_ipad[65]; /* inner padding - key XORd with ipad */ 
	unsigned char k_opad[65]; /* outer padding - key XORd with opad */ 
	int i; 

	/* if key is longer than 64 bytes reset it to key=SHA1(key) */ 
	if (key_len > 64) 
	{ 
		A_SHA_CTX tctx; 

		A_SHAInit(&tctx); 
		A_SHAUpdate(&tctx, key, key_len); 
		A_SHAFinal(&tctx, key); 

		key_len = 20; 
	} 

	/* 
	* the HMAC_SHA1 transform looks like: 
	* 
	* SHA1(K XOR opad, SHA1(K XOR ipad, text)) 
	* 
	* where K is an n byte key 
	* ipad is the byte 0x36 repeated 64 times 
	* opad is the byte 0x5c repeated 64 times 
	* and text is the data being protected 
	*/ 

	/* start out by storing key in pads */ 
	memset(k_ipad, 0, sizeof k_ipad); 
	memset(k_opad, 0, sizeof k_opad); 
	memcpy(k_ipad, key, key_len); 
	memcpy(k_opad, key, key_len); 

	/* XOR key with ipad and opad values */ 
	for (i = 0; i < 64; i++) 
	{ 
		k_ipad[i] ^= 0x36; 
		k_opad[i] ^= 0x5c; 
	} 

	/* perform inner SHA1*/ 
	A_SHAInit(&context); /* init context for 1st pass */ 
	A_SHAUpdate(&context, k_ipad, 64); /* start with inner pad */ 
	A_SHAUpdate(&context, text, text_len); /* then text of datagram */ 
	A_SHAFinal(&context, digest); /* finish up 1st pass */ 

	/* perform outer SHA1 */ 
	A_SHAInit(&context); /* init context for 2nd pass */ 
	A_SHAUpdate(&context, k_opad, 64); /* start with outer pad */ 
	A_SHAUpdate(&context, digest, 20); /* then results of 1st hash */ 
	A_SHAFinal(&context, digest); /* finish up 2nd pass */ 
} 

/*
* F(P, S, c, i) = U1 xor U2 xor ... Uc 
* U1 = PRF(P, S || Int(i)) 
* U2 = PRF(P, U1) 
* Uc = PRF(P, Uc-1) 
*/ 

static void F(char *password, unsigned char *ssid, int ssidlength, int iterations, int count, unsigned char *output) 
{ 
	unsigned char digest[36], digest1[A_SHA_DIGEST_LEN]; 
	int i, j; 

	/* U1 = PRF(P, S || int(i)) */ 
	memcpy(digest, ssid, ssidlength); 
	digest[ssidlength] = (unsigned char)((count>>24) & 0xff); 
	digest[ssidlength+1] = (unsigned char)((count>>16) & 0xff); 
	digest[ssidlength+2] = (unsigned char)((count>>8) & 0xff); 
	digest[ssidlength+3] = (unsigned char)(count & 0xff); 
	hmac_sha1(digest, ssidlength+4, (unsigned char*) password, (int) strlen(password), digest1); // for WPA update

	/* output = U1 */ 
	memcpy(output, digest1, A_SHA_DIGEST_LEN); 

	for (i = 1; i < iterations; i++) 
	{ 
		/* Un = PRF(P, Un-1) */ 
		hmac_sha1(digest1, A_SHA_DIGEST_LEN, (unsigned char*) password, (int) strlen(password), digest); // for WPA update
		memcpy(digest1, digest, A_SHA_DIGEST_LEN); 

		/* output = output xor Un */ 
		for (j = 0; j < A_SHA_DIGEST_LEN; j++) 
		{ 
			output[j] ^= digest[j]; 
		} 
	} 
} 
/* 
* password - ascii string up to 63 characters in length 
* ssid - octet string up to 32 octets 
* ssidlength - length of ssid in octets 
* output must be 40 octets in length and outputs 256 bits of key 
*/ 
int PasswordHash(char *password, unsigned char *ssid, int ssidlength, unsigned char *output) 
{ 
	if ((strlen(password) > 63) || (ssidlength > 32)) 
		return 0; 

	F(password, ssid, ssidlength, 4096, 1, output); 
	F(password, ssid, ssidlength, 4096, 2, &output[A_SHA_DIGEST_LEN]); 
	return 1; 
}


