/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************
 
    Module Name:
    connect.c
 
    Abstract:
    Routines to deal Link UP/DOWN and build/update BEACON frame contents
 
    Revision History:
    Who         When          What
    --------    ----------    ----------------------------------------------
    John Chang  08-04-2003    created for 11g soft-AP
 */

#include "rt_config.h"

static const ULONG Bit[32] = {
    0x00000001, 0x00000002, 0x00000004, 0x00000008, 0x00000010, 0x00000020, 0x00000040, 0x00000080,
    0x00000100, 0x00000200, 0x00000400, 0x00000800, 0x00001000, 0x00002000, 0x00004000, 0x00008000,
    0x00010000, 0x00020000, 0x00040000, 0x00080000, 0x00100000, 0x00200000, 0x00400000, 0x00800000,
    0x01000000, 0x02000000, 0x04000000, 0x08000000, 0x10000000, 0x20000000, 0x40000000, 0x80000000 };

/*
    ==========================================================================
    Description:
        Pre-build a BEACON frame in the shared memory
    ==========================================================================
*/
VOID MakeBssBeacon(
    IN PRTMP_ADAPTER pAd) 
{
    UCHAR         SsidIe = IE_SSID, DsIe = IE_DS_PARM, SuppIe = IE_SUPP_RATES, 
                  DsLen = 1, TimIe = IE_TIM, TimLen = 4,
                  BitmapControl = 0, VirtualBitmap = 0, SsidLen;
    UCHAR         RSNIe = IE_WPA_RSN_IE;
    MACHDR        BcnHdr;
    LARGE_INTEGER FakeTimestamp;
    ULONG         FrameLen;
    UCHAR         SupportedRatesLen;
    PTXD_STRUC	  pTxD = (PTXD_STRUC)pAd->BeaconRing.va_addr;
    CHAR         *pBeaconFrame = (CHAR *)pAd->BeaconRing.va_data_addr;

    if ((pAd->PortCfg.PhyMode == PHY_11BG_MIXED) || (pAd->PortCfg.PhyMode == PHY_11G))
        SupportedRatesLen = 8;
    else
        SupportedRatesLen = pAd->PortCfg.SupportedRatesLen;

    if (pAd->PortCfg.HideSsid)
        SsidLen = 0;
    else
        SsidLen = pAd->PortCfg.SsidLen;
    
    MgtMacHeaderInit(pAd, &BcnHdr, SUBTYPE_BEACON, 0, &pAd->PortCfg.Broadcast, &pAd->PortCfg.Bssid);
    
    // Append RSN_IE when  WPA OR WPAPSK, 
    if (pAd->PortCfg.AuthMode >= Ndis802_11AuthModeWPA)
    {
        MakeOutgoingFrame(pBeaconFrame,                  &FrameLen,
                        MAC_HDR_LEN,                     &BcnHdr, 
                        TIMESTAMP_LEN,                   &FakeTimestamp,
                        2,                               &pAd->PortCfg.BeaconPeriod,
                        2,                               &pAd->PortCfg.CapabilityInfo,
                        1,                               &SsidIe, 
                        1,                               &SsidLen, 
                        SsidLen,                         pAd->PortCfg.Ssid,
                        1,                               &SuppIe, 
                        1,                               &SupportedRatesLen,
                        SupportedRatesLen,               pAd->PortCfg.SupportedRates, 
                        1,                               &DsIe, 
                        1,                               &DsLen, 
                        1,                               &pAd->PortCfg.Channel,
                        1,                               &RSNIe,
                        1,                               &pAd->PortCfg.RSNIE_Len,
                        pAd->PortCfg.RSNIE_Len,          pAd->PortCfg.RSN_IE,
                        END_OF_ARGS);
    }
    else
    {
        MakeOutgoingFrame(pBeaconFrame,                  &FrameLen,
                        MAC_HDR_LEN,                     &BcnHdr, 
                        TIMESTAMP_LEN,                   &FakeTimestamp,
                        2,                               &pAd->PortCfg.BeaconPeriod,
                        2,                               &pAd->PortCfg.CapabilityInfo,
                        1,                               &SsidIe, 
                        1,                               &SsidLen, 
                        SsidLen,                         pAd->PortCfg.Ssid,
                        1,                               &SuppIe, 
                        1,                               &SupportedRatesLen,
                        SupportedRatesLen,               pAd->PortCfg.SupportedRates, 
                        1,                               &DsIe, 
                        1,                               &DsLen, 
                        1,                               &pAd->PortCfg.Channel,
                        END_OF_ARGS);
    }
#ifdef BIG_ENDIAN
    RTMPFrameEndianChange(pAd, pBeaconFrame, DIR_WRITE, FALSE);
#endif
    RTMPWriteTxDescriptor(pTxD, FALSE, CIPHER_NONE, FALSE, FALSE, TRUE, SHORT_RETRY, IFS_BACKOFF, 
	pAd->PortCfg.MlmeRate, 4, FrameLen, pAd->PortCfg.TxPreamble, 0);
	pAd->Mlme.TimIELocationInBeacon = (UCHAR)FrameLen; 
    pAd->Mlme.CapabilityInfoLocationInBeacon = MAC_HDR_LEN + TIMESTAMP_LEN + 2;

}

/*
    ==========================================================================
    Description:
        Update the BEACON frame in the shared memory. Because TIM IE is variable
        length. other IEs after TIM has to shift and total frame length may change
        for each BEACON period.
    Output:
        pAd->PortCfg.CapabilityInfo
        pAd->Mlme.ErpIeContent
    ==========================================================================
*/
VOID UpdateBeaconFrame(
    IN PRTMP_ADAPTER pAd) 
{
    PTXD_STRUC pTxD = (PTXD_STRUC)pAd->BeaconRing.va_addr;
    MACHDR *pBcnHdr = (MACHDR *)pAd->BeaconRing.va_data_addr;
    UCHAR  *ptr;
    ULONG FrameLen = pAd->Mlme.TimIELocationInBeacon;
    UCHAR byte0 = (UCHAR)(pAd->Mlme.TimBitmap & 0x000000fe);  // skip AID#0
    UCHAR byte1 = (UCHAR)((pAd->Mlme.TimBitmap & 0x0000ff00) >> 8);
    UCHAR byte2 = (UCHAR)((pAd->Mlme.TimBitmap & 0x00ff0000) >> 16);
    UCHAR byte3 = (UCHAR)((pAd->Mlme.TimBitmap & 0xff000000) >> 24);

    //
    // step 1 - update BEACON frame's sequence number
    // 
    pAd->Sequence = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
    pBcnHdr->Seq = pAd->Sequence;
#ifdef BIG_ENDIAN
    *(USHORT*)((UCHAR*)pBcnHdr + 22) = SWAP16(*(USHORT*)((UCHAR*)pBcnHdr + 22) & 0xFFF0);
#endif

    // step 1.1 - update BEACON's Capability
    ptr = (UCHAR *)pBcnHdr + pAd->Mlme.CapabilityInfoLocationInBeacon;
    *ptr = (UCHAR)(pAd->PortCfg.CapabilityInfo & 0x00ff);
    *(ptr+1) = (UCHAR)((pAd->PortCfg.CapabilityInfo & 0xff00) >> 8);
    
    //
    // step 2 - update TIM IE
    // 
    if (pAd->Mlme.DtimCount == 0)
        pAd->Mlme.DtimCount = pAd->Mlme.DtimPeriod - 1;
    else
        pAd->Mlme.DtimCount -= 1;
    ptr = (UCHAR *)pBcnHdr + pAd->Mlme.TimIELocationInBeacon;
    *ptr = IE_TIM;
    *(ptr + 2) = pAd->Mlme.DtimCount;
    *(ptr + 3) = pAd->Mlme.DtimPeriod;

    if (byte0 || byte1) // there's some backlogged frame for AID 1-15
    {
        *(ptr + 4) = 0;      // Virtual TIM bitmap stars from AID #0
        *(ptr + 5) = byte0;
        *(ptr + 6) = byte1;
        *(ptr + 7) = byte2;
        *(ptr + 8) = byte3;
        if (byte3)      *(ptr + 1) = 7; // IE length
        else if (byte2) *(ptr + 1) = 6; // IE length
        else if (byte1) *(ptr + 1) = 5; // IE length
        else            *(ptr + 1) = 4; // IE length
    }
    else if (byte2 || byte3) // there's some backlogged frame for AID 16-31
    {
        *(ptr + 4) = 2;      // Virtual TIM bitmap starts from AID #16
        *(ptr + 5) = byte2;
        *(ptr + 6) = byte3;
        if (byte3) *(ptr + 1) = 5; // IE length
        else       *(ptr + 1) = 4; // IE length
    }
    else // no backlogged frames for AID 1-31
    {
        *(ptr + 1) = 4; // IE length
        *(ptr + 4) = 0;
        *(ptr + 5) = 0;
    }
    
    // bit0 means backlogged mcast/bcast
    *(ptr + 4) |= (pAd->Mlme.TimBitmap & 0x01); 
    
    // adjust BEACON length according to the new TIM
    FrameLen += (2 + *(ptr+1)); 

    if ((pAd->PortCfg.PhyMode == PHY_11BG_MIXED) || (pAd->PortCfg.PhyMode == PHY_11G))
    {
        UCHAR ExtendedRatesLen = pAd->PortCfg.SupportedRatesLen - 8;
        //
        // step 3 - fill ERP IE
        // 
        ptr = (UCHAR *)pBcnHdr + FrameLen; // pTxD->DataByteCnt;
        *ptr = IE_ERP;
        *(ptr + 1) = 1;
        *(ptr + 2) = pAd->Mlme.ErpIeContent;

        //
        // step 4 - fill up EXTENDED RATE IE. The first 4 rates (1/2/5.5/11) are already in
        //          SUPPORTED RATES IE.
        // 
        *(ptr + 3) = IE_EXT_SUPP_RATES;
        *(ptr + 4) = ExtendedRatesLen;
        NdisMoveMemory(ptr+5, &pAd->PortCfg.SupportedRates[8], *(ptr+4));
        ptr      += (3 + 2 + ExtendedRatesLen);
        FrameLen += (3 + 2 + ExtendedRatesLen);
    }

    //
    // step 5 - fill up SSN/WPA IE
    //

    // Since FrameLen may change, update TXD
    RTMPWriteTxDescriptor(pTxD, FALSE, CIPHER_NONE, FALSE, FALSE, TRUE, SHORT_RETRY, IFS_BACKOFF, 
        pAd->PortCfg.MlmeRate, 4, FrameLen, pAd->PortCfg.TxPreamble, 0);

    //
    // step 6 - if DTIM, then move backlogged bcast/mcast frames from PSQ to TXQ whenever DtimCount==0
    // NOTE: This updated BEACON frame will be sent at "next" TBTT instead of at cureent TBTT. The reason is
    //       because ASIC already fecth the BEACON content down to TX FIFO before driver can make any
    //       modification. To compenstate this effect, the actual time to deilver PSQ frames will be
    //       at the time that we wrapping around DtimCount from 0 to DtimPeriod-1
    if ((pAd->Mlme.DtimCount + 1) == pAd->Mlme.DtimPeriod)
    {
        PQUEUE_ENTRY	pEntry;
        
        NdisAcquireSpinLock(&pAd->MacTabLock);
        NdisAcquireSpinLock(&pAd->TxSwQueueLock);
        while (pAd->MacTab.McastPsQueue.Head)
        {
            pEntry = RemoveHeadQueue(&pAd->MacTab.McastPsQueue);
            InsertTailQueue(&pAd->TxSwQueue0, pEntry);
            DBGPRINT(RT_DEBUG_TRACE, "DTIM=%d/%d, tx mcast/bcast out...\n",pAd->Mlme.DtimCount,pAd->Mlme.DtimPeriod);
        }
        NdisReleaseSpinLock(&pAd->TxSwQueueLock);
        NdisReleaseSpinLock(&pAd->MacTabLock);
        pAd->Mlme.TimBitmap &= ~Bit[0];
        pAd->MacTab.PsQIdleCount = 0;
        
        // Dequeue one frame from TxSwQueue0..3 queue and process it
        RTMPDeQueuePacket(pAd);
    }
}
