/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************
 
	Module Name:
	soft_ap.c
 
	Abstract:
	Access Point specific routines and MAC table maintenance routines
 
	Revision History:
	Who         When          What
	--------    ----------    ----------------------------------------------
	John Chang  08-04-2003    created for 11g soft-AP

 */

#include "rt_config.h"

#define MAX_NUM_OF_FREE_NDIS_PACKET     32

char const *pEventText[EVENT_MAX_EVENT_TYPE] = {
	"restart access point",
	"successfully associated",
	"has disassociated",
	"has been aged-out and disassociated" ,    
	"active countermeasures",
	"has disassociated with invalid PSK password"};


/*
	==========================================================================
	Description:
		Initialize AP specific data especially the NDIS packet pool that's
		used for wireless client bridging.
	==========================================================================
 */
NDIS_STATUS ApInitialize(
	IN	PRTMP_ADAPTER	pAd)
{
	NDIS_STATUS     Status = NDIS_STATUS_SUCCESS;
	UCHAR   GTK[TKIP_GTK_LENGTH];

	DBGPRINT(RT_DEBUG_TRACE, "---> ApInitialize\n");

	// initialize MAC table and allocate spin lock
	NdisZeroMemory(&pAd->MacTab, sizeof(MAC_TABLE));
	NdisAllocateSpinLock(&pAd->MacTabLock);

	// Init TKIP Group-Key-related variables
	GenRandom(pAd, pAd->PortCfg.GMK);
	GenRandom(pAd, pAd->PortCfg.GNonce);
	CountGTK(pAd->PortCfg.GMK,  (UCHAR*)pAd->PortCfg.GNonce, 
			pAd->CurrentAddress, pAd->PortCfg.GroupKey[0].Key, 
			TKIP_GTK_LENGTH);
	NdisMoveMemory(pAd->PortCfg.GroupKey[pAd->PortCfg.WPAGKeyID].Key, GTK, 16);
	NdisMoveMemory(pAd->PortCfg.GroupKey[pAd->PortCfg.WPAGKeyID].TxMic, &GTK[16], 8);
	NdisMoveMemory(pAd->PortCfg.GroupKey[pAd->PortCfg.WPAGKeyID].RxMic, &GTK[24], 8);            
	// Init Group key update timer, and countermeasures timer
	init_timer(&pAd->PortCfg.REKEYTimer);
	pAd->PortCfg.REKEYTimer.data = (unsigned long)pAd;
	pAd->PortCfg.REKEYTimer.function = &GREKEYPeriodicExec;

	init_timer(&pAd->PortCfg.CounterMeasureTimer);
	pAd->PortCfg.CounterMeasureTimer.data = (unsigned long)pAd;
	pAd->PortCfg.CounterMeasureTimer.function = &CMTimerExec;

	DBGPRINT(RT_DEBUG_TRACE, "<--- ApInitialize\n");
	return Status;
}//end of ApInitialize()

/*
	==========================================================================
	Description:
		Shutdown AP and free AP specific resources
	==========================================================================
 */
VOID ApShutdown(
	IN PRTMP_ADAPTER pAd)
{
	DBGPRINT(RT_DEBUG_TRACE, "---> ApShutdown\n");
	ApStop(pAd);
	DBGPRINT(RT_DEBUG_TRACE, "<--- ApShutdown\n");
}//end of ApShutdown()

/*
	==========================================================================
	Description:
		Start AP service. If any vital AP parameter is changed, a STOP-START
		sequence is required to disassociate all STAs.
	Note:
	==========================================================================
 */
VOID ApStartUp(
	IN PRTMP_ADAPTER pAd) 
{

	if ((pAd->PortCfg.SsidLen <= 0) || (pAd->PortCfg.SsidLen >= MAX_LEN_OF_SSID))
		return;
	
	DBGPRINT(RT_DEBUG_TRACE, "!!! ApStartUp !!!\n");

	pAd->PortCfg.CapabilityInfo    = CAP_GENERATE((pAd->PortCfg.WepStatus != Ndis802_11EncryptionDisabled), 1, pAd->PortCfg.UseShortSlotTime);
		
	AsicSetBssid(pAd, (MACADDR *)pAd->CurrentAddress); 
	AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
	AsicLockChannel(pAd, pAd->PortCfg.Channel);
	MlmeSetTxPreamble(pAd, pAd->PortCfg.TxPreamble);
	MlmeUpdateTxRates(pAd);

	// start sending BEACON out        
	MakeBssBeacon(pAd);
	UpdateBeaconFrame(pAd);
	ApUpdateCapabilityAndErpIe(pAd);
	AsicEnableBssSync(pAd);
	ASIC_LED_ACT_ON(pAd);

	DBGPRINT(RT_DEBUG_TRACE, " ApStartUp :  Group rekey method= %d , interval = 0x%x\n",pAd->PortCfg.WPAREKEY.ReKeyMethod,pAd->PortCfg.WPAREKEY.ReKeyInterval);
	// Group rekey related
	if ((pAd->PortCfg.WPAREKEY.ReKeyInterval != 0) && ((pAd->PortCfg.WPAREKEY.ReKeyMethod == TIME_REKEY) || (pAd->PortCfg.WPAREKEY.ReKeyMethod == PKT_REKEY))) 
	{
		// Regularly check the timer
		if (pAd->PortCfg.REKEYTimerRunning == FALSE)
		{
			pAd->PortCfg.REKEYTimer.expires = jiffies + GUPDATE_EXEC_INTV;
			add_timer(&pAd->PortCfg.REKEYTimer);
			pAd->PortCfg.REKEYTimerRunning = TRUE;
			pAd->PortCfg.REKEYCOUNTER = 0;
		}
	}
	else
		pAd->PortCfg.REKEYTimerRunning = FALSE;

	ApLogEvent(pAd, (PMACADDR)pAd->CurrentAddress, EVENT_RESET_ACCESS_POINT);
	//v1.2.0
	pAd->Mlme.PeriodicRound = 0;
	{ 
		// send SIGUSR1 to rt2500apd
		if (pAd->PortCfg.AuthMode >= Ndis802_11AuthModeWPA)
		{
			PUCHAR					src;
			struct file				*srcf;
			INT 					retval, orgfsuid, orgfsgid;
			mm_segment_t			orgfs;
			CHAR					buffer[MAX_INI_BUFFER_SIZE];
			CHAR					tmpbuf[600];
			src = PROFILE_PATH;
	
			// Save uid and gid used for filesystem access.
			// Set user and group to 0 (root)	
			orgfsuid = current->fsuid;
			orgfsgid = current->fsgid;
			current->fsuid=current->fsgid = 0;
			orgfs = get_fs();
			set_fs(KERNEL_DS);

			if (src && *src) 
			{
				srcf = filp_open(src, O_RDONLY, 0);
				if (IS_ERR(srcf)) 
				{
					DBGPRINT(RT_DEBUG_TRACE, "--> Error %ld opening %s\n", -PTR_ERR(srcf),src);    
				}
				else 
				{
					/* The object must have a read method */
					if (srcf->f_op && srcf->f_op->read) 
					{
						memset(buffer, 0x00, MAX_INI_BUFFER_SIZE);
						retval=srcf->f_op->read(srcf, buffer, MAX_INI_BUFFER_SIZE, &srcf->f_pos);
						if (retval < 0)
						{
							DBGPRINT(RT_DEBUG_TRACE, "--> Read %s error %d\n", src, -retval);
						}
						else
						{
							// set file parameter to portcfg
							//CountryRegion
							if(RTMPGetKeyParameter("Default", "pid", tmpbuf, 255, buffer))
							{
								pAd->PortCfg.ApdPid= (ULONG) simple_strtol(tmpbuf, 0, 10);
								DBGPRINT(RT_DEBUG_TRACE, "ApdPid=%d\n", pAd->PortCfg.ApdPid);
								kill_proc(pAd->PortCfg.ApdPid, SIGUSR1, 0);
							}
						}
					}
					else
					{
						DBGPRINT(RT_DEBUG_TRACE, "--> %s does not have a write method\n", src);
					}
			
					retval=filp_close(srcf,NULL);
			
					if (retval)
					{
						DBGPRINT(RT_DEBUG_TRACE, "--> Error %d closing %s\n", -retval, src);
					}
				}
			}
			set_fs(orgfs);
			current->fsuid = orgfsuid;
			current->fsgid = orgfsgid;
		}
	}//v1.2.0	

}//end of ApStartUp()

/*
	==========================================================================
	Description:
		disassociate all STAs and stop AP service.
	Note:
	==========================================================================
 */
VOID ApStop(
	IN PRTMP_ADAPTER pAd) 
{
	DBGPRINT(RT_DEBUG_TRACE, "!!! ApStop !!!\n");
	//NICDisableInterrupt(pAd);
	AsicDisableSync(pAd);

	if(pAd->PortCfg.REKEYTimerRunning==TRUE)
	{
		del_timer_sync(&pAd->PortCfg.REKEYTimer);
		pAd->PortCfg.REKEYTimerRunning=FALSE;
	}

	MacTableReset(pAd);
	ASIC_LED_ACT_OFF(pAd);
}//end of ApStop()

/*
	==========================================================================
	Description:
		This routine is used to clean up a specified power-saving queue. It's
		used whenever a wireless client is deleted.
	==========================================================================
 */
VOID CleanupPsQueue(
	IN  PRTMP_ADAPTER   pAd,
	IN  PQUEUE_HEADER   pQueue)
{
	struct sk_buff	*skb;
	
	while (pQueue->Head)
	{
		skb = (struct sk_buff *)RemoveHeadQueue(pQueue);
		if(skb)		RTMPFreeSkbBuffer(skb);
	}
}//end of CleanupPsQueue()

/*
	==========================================================================
	Description:
		This routine reset the entire MAC table. All packets pending in
		the power-saving queues are freed here.
	==========================================================================
 */
VOID MacTableReset(
	IN  PRTMP_ADAPTER  pAd)
{
	int i;

	NdisAcquireSpinLock(&pAd->MacTabLock);
	CleanupPsQueue(pAd, &pAd->MacTab.McastPsQueue);
	for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
	{
		if (pAd->MacTab.Content[i].Valid)
	   {
			if (pAd->MacTab.Content[i].RetryTimerRunning == TRUE)
			{
				DBGPRINT(DEBUG_TEMP, " pAd->MacTab.Content[%d]  Cancel Retry Timer ! \n",i);
				del_timer_sync(&pAd->MacTab.Content[i].RetryTimer);
				pAd->MacTab.Content[i].RetryTimerRunning = FALSE;             
			}
			CleanupPsQueue(pAd, &pAd->MacTab.Content[i].PsQueue);
		}
	}
	NdisZeroMemory(&pAd->MacTab, sizeof(MAC_TABLE));
	InitializeQueueHeader(&pAd->MacTab.McastPsQueue);
	NdisReleaseSpinLock(&pAd->MacTabLock);
}//end of MacTableReset()

/*
	==========================================================================
	Description:
		Add and new entry into MAC table
	==========================================================================
 */
MAC_TABLE_ENTRY *MacTableInsertEntry(
	IN	PRTMP_ADAPTER	pAd, 
	IN  PMACADDR pAddr) 
{
	UCHAR /*Cidx,*/ HashIdx;
	int i;
	MAC_TABLE_ENTRY *pEntry = NULL, *pCurrEntry;

	// if FULL, return
	if (pAd->MacTab.Size >= MAX_LEN_OF_MAC_TABLE) 
		return NULL;

	// allocate one MAC entry
	NdisAcquireSpinLock(&pAd->MacTabLock);
	for (i = 0; i< MAX_LEN_OF_MAC_TABLE; i++)
	{
		// pick up the first available vacancy
		if (pAd->MacTab.Content[i].Valid == FALSE)
		{
			pEntry = &pAd->MacTab.Content[i];
			NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
			pEntry->Valid = TRUE;

			pEntry->RetryTimerRunning = FALSE;
			pEntry->CMTimerRunning = FALSE;
			pEntry->RSNIE_Len = 0;
			NdisMoveMemory(pEntry->R_Counter, pAd->PortCfg.R_Counter, sizeof(pEntry->R_Counter));
			pEntry->ReTryCounter = 0;
			pEntry->AuthMode = pAd->PortCfg.AuthMode;
			pEntry->WepStatus = pAd->PortCfg.WepStatus;

			if (pEntry->AuthMode < Ndis802_11AuthModeWPA)
				pEntry->WpaState = AS_NOTUSE;
			else
				pEntry->WpaState = AS_INITIALIZE;
			pEntry->GTKState = REKRY_NEGOTIATING;
			pEntry->PrivacyFilter = Ndis802_11PrivFilterAcceptAll;        
			pEntry->PairwiseKey[0].KeyLen = 0;
			pEntry->PortSecured = WPA_802_1X_PORT_NOT_SECURED;
			pEntry->Flag = 0;
			pEntry->Addr = *pAddr;
			pEntry->Sst = SST_NOT_AUTH;
			pEntry->AuthState = AS_NOT_AUTH;
			pEntry->Aid = 0;
			pEntry->CapabilityInfo = 0;
			pEntry->PsMode = PWR_ACTIVE;
			pEntry->MaxSupportedRate = RATE_11;
			pEntry->CurrTxRate = RATE_11;
			pEntry->PsQIdleCount = 0;
			pEntry->NoDataIdleCount = 0;
			InitializeQueueHeader(&pEntry->PsQueue);
			pAd->MacTab.Size ++;

			DBGPRINT(RT_DEBUG_TRACE, "MacTableInsertEntry - allocate entry #%d, Total= %d\n",i, pAd->MacTab.Size);
			break;
		}
	}

	// add this MAC entry into HASH table
	if (pEntry)
	{
		HashIdx = MAC_ADDR_HASH_INDEX(*pAddr);
		if (pAd->MacTab.Hash[HashIdx] == NULL)
		{
			pAd->MacTab.Hash[HashIdx] = pEntry;
		}
		else
		{
			pCurrEntry = pAd->MacTab.Hash[HashIdx];
			while (pCurrEntry->pNext != NULL)
				pCurrEntry = pCurrEntry->pNext;
			pCurrEntry->pNext = pEntry;
		}
	}
	
	NdisReleaseSpinLock(&pAd->MacTabLock);
	return pEntry;
}//end of MacTableInsertEntry()
	
/*
	==========================================================================
	Description:
		Delete a specified client from MAC table
	==========================================================================
 */
BOOLEAN MacTableDeleteEntry(
	IN PRTMP_ADAPTER pAd, 
	IN PMACADDR pAddr) 
{
	USHORT HashIdx;
	MAC_TABLE_ENTRY *pEntry, *pPrevEntry;

	NdisAcquireSpinLock(&pAd->MacTabLock);
	HashIdx = MAC_ADDR_HASH_INDEX(*pAddr);
	pEntry = pAd->MacTab.Hash[HashIdx];
	if (pEntry)
	{
		if(pEntry->RetryTimerRunning==TRUE)
		{
			DBGPRINT(DEBUG_TEMP, " MacTableDeleteEntry :   Cancel Retry Timer ! \n");
			del_timer_sync(&pEntry->RetryTimer);
			pEntry->RetryTimerRunning=FALSE;                
		}

		if (MAC_ADDR_EQUAL(&pEntry->Addr, pAddr))
		{
			pAd->MacTab.Hash[HashIdx] = pEntry->pNext;
			CleanupPsQueue(pAd, &pEntry->PsQueue); // return all NDIS packet in PSQ
			NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
			pAd->MacTab.Size --;
			DBGPRINT(RT_DEBUG_TRACE, "MacTableDeleteEntry - Total= %d\n", pAd->MacTab.Size);
		}
		else
		{
			while (pEntry->pNext)
			{
				pPrevEntry = pEntry;
				pEntry = pEntry->pNext;
				if (MAC_ADDR_EQUAL(&pEntry->Addr, pAddr))
				{
					pPrevEntry->pNext = pEntry->pNext;
					CleanupPsQueue(pAd, &pEntry->PsQueue); // return all NDIS packet in PSQ
					NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
					pAd->MacTab.Size --;
					DBGPRINT(RT_DEBUG_TRACE, "MacTableDeleteEntry - Total= %d\n", pAd->MacTab.Size);
					break;
				}
			}
		}
	}

	NdisReleaseSpinLock(&pAd->MacTabLock);
	ApUpdateCapabilityAndErpIe(pAd);
	
	return TRUE;
}//end of MacTableDeleteEntry()

/*
	==========================================================================
	Description:
		Look up the MAC address in the MAC table. Return NULL if not found.
	Return:
		pEntry - pointer to the MAC entry; NULL is not found
	==========================================================================
*/
MAC_TABLE_ENTRY *MacTableLookup(
	IN PRTMP_ADAPTER pAd, 
	MACADDR *pAddr) 
{
	ULONG HashIdx/*, Pidx*/;
	MAC_TABLE_ENTRY *pEntry;

	HashIdx = MAC_ADDR_HASH_INDEX(*pAddr);
	pEntry = pAd->MacTab.Hash[HashIdx];

	while (pEntry && pEntry->Valid)
	{
		if (MAC_ADDR_EQUAL(&pEntry->Addr, pAddr)) 
		{
			break;
		}
		else
			pEntry = pEntry->pNext;
	}
#if 0
	if(  pEntry == NULL )
		DBGPRINT(RT_DEBUG_TRACE, "Mac address--%02x:%02x:%02x:%02x:%02x:%02x not in Mac table\n",
			pAddr->Octet[0],pAddr->Octet[1],pAddr->Octet[2],pAddr->Octet[3],pAddr->Octet[4],pAddr->Octet[5]);
#endif    

	return pEntry;
}//end of MacTableLookup()

/*
	==========================================================================
	Description:
		This routine is called by MlmePeriodicExec() every second to check if
		1. any associated client in PSM. If yes, then TX MCAST/BCAST should be
		   out in DTIM only
		2. any client being idle for too long and should be aged-out from MAC table
		3. garbage collect PSQ
	==========================================================================
*/
VOID MacTableMaintenance(
	IN PRTMP_ADAPTER pAd)
{
	int i;
	
	pAd->MacTab.fAnyStationInPsm = FALSE;
	for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++) 
	{
		MAC_TABLE_ENTRY *pEntry = &pAd->MacTab.Content[i];

		if (pEntry->Valid == FALSE)
			continue;
		
		pEntry->NoDataIdleCount ++;  

		// 0. STA failed to complete association should be removed to save MAC table space.
		if ((pEntry->Sst != SST_ASSOC) && (pEntry->NoDataIdleCount >= MAC_TABLE_ASSOC_TIMEOUT))
		{
			DBGPRINT(RT_DEBUG_TRACE, "%02x:%02x:%02x:%02x:%02x:%02x fail to complete ASSOC in %d sec\n",
					pEntry->Addr.Octet[0],pEntry->Addr.Octet[1],pEntry->Addr.Octet[2],pEntry->Addr.Octet[3],
					pEntry->Addr.Octet[4],pEntry->Addr.Octet[5],MAC_TABLE_ASSOC_TIMEOUT);
			MacTableDeleteEntry(pAd, &pEntry->Addr);
			continue;
		}
		
		// 1. check if there's any associated STA in power-save mode. this affects outgoing
		//    MCAST/BCAST frames should be stored in PSQ till DtimCount=0
		if (pEntry->PsMode == PWR_SAVE)
			pAd->MacTab.fAnyStationInPsm = TRUE;

		// 2. delete those MAC entry that has been idle for a long time
		if (pEntry->NoDataIdleCount >= MAC_TABLE_AGEOUT_TIME)
		{
			DBGPRINT(RT_DEBUG_TRACE, "ageout %02x:%02x:%02x:%02x:%02x:%02x after %d-sec silence\n",
					pEntry->Addr.Octet[0],pEntry->Addr.Octet[1],pEntry->Addr.Octet[2],pEntry->Addr.Octet[3],
					pEntry->Addr.Octet[4],pEntry->Addr.Octet[5],MAC_TABLE_AGEOUT_TIME);
			ApLogEvent(pAd, &pEntry->Addr, EVENT_AGED_OUT);
			MacTableDeleteEntry(pAd, &pEntry->Addr);
			continue;
		}
		
		// 3. garbage collect the PsQueue if the STA has being idle for a while
		if (pEntry->PsQueue.Head)
		{
			pEntry->PsQIdleCount ++;  
			if (pEntry->PsQIdleCount > 2) 
			{
				NdisAcquireSpinLock(&pAd->MacTabLock);
				CleanupPsQueue(pAd, &pEntry->PsQueue);
				NdisReleaseSpinLock(&pAd->MacTabLock);
				pEntry->PsQIdleCount = 0;
			}
		}
		else
			pEntry->PsQIdleCount = 0;
	}
	
	// 4. garbage collect pAd->MacTab.McastPsQueue if backlogged MCAST/BCAST frames
	//    stale in queue. Since MCAST/BCAST frames always been sent out whenever 
	//    DtimCount==0, the only case to let them stale is surprise removal of the NIC,
	//    so that ASIC-based Tbcn interrupt stops and DtimCount dead.
	if (pAd->MacTab.McastPsQueue.Head)
	{
		pAd->MacTab.PsQIdleCount ++;
		if (pAd->MacTab.PsQIdleCount > 1)
		{
			NdisAcquireSpinLock(&pAd->MacTabLock);
			CleanupPsQueue(pAd, &pAd->MacTab.McastPsQueue);
			NdisReleaseSpinLock(&pAd->MacTabLock);
			pAd->MacTab.PsQIdleCount = 0;
		}
	}
	else
		pAd->MacTab.PsQIdleCount = 0;
}//end of MacTableMaintenance()

/*
	==========================================================================
	Description:
		Look up a STA MAC table. Return its Sst to decide if an incoming
		frame from this STA or an outgoing frame to this STA is permitted.
	Return:
	==========================================================================
*/

MAC_TABLE_ENTRY *SsPsInquiry(
	IN	PRTMP_ADAPTER	pAd, 
	IN MACADDR *pAddr, 
	OUT SST   *Sst, 
	OUT USHORT *Aid,
	OUT UCHAR *PsMode,
	OUT UCHAR *Rate) 
{
	MAC_TABLE_ENTRY *pEntry = NULL;

	if (MAC_ADDR_IS_GROUP(*pAddr)) // mcast & broadcast address
	{
		*Sst        = SST_ASSOC;
		*Aid        = 0;
		*PsMode     = PWR_ACTIVE;
		*Rate       = pAd->PortCfg.MlmeRate; //pAd->PortCfg.MaxBasicRate;
	} 
	else // unicast address
	{
		pEntry = MacTableLookup(pAd, pAddr);
		if (pEntry) 
		{
			*Sst        = pEntry->Sst;
			*Aid        = pEntry->Aid;
			*PsMode     = pEntry->PsMode;
			*Rate       = pEntry->CurrTxRate;
		} 
		else 
		{
			*Sst        = SST_NOT_AUTH;
			*Aid        = 0;
			*PsMode     = PWR_ACTIVE;
			*Rate       = pAd->PortCfg.MlmeRate; //pAd->PortCfg.MaxBasicRate;
		}
	}
	return pEntry;
}//end of SsPsInquiry()

/*
	==========================================================================
	Description:
		Update the station current power save mode. Calling this routine also
		prove the specified client is still alive. Otherwise AP will age-out
		this client once IdleCount exceeds a threshold.
	==========================================================================
 */
BOOLEAN PsIndicate(
	IN PRTMP_ADAPTER pAd, 
	IN PMACADDR pAddr, 
	IN UCHAR Rssi,
	IN UCHAR Psm) 
{
	MAC_TABLE_ENTRY *pEntry;

	pEntry = MacTableLookup(pAd, pAddr);
	if (pEntry) 
	{
		if ((pEntry->PsMode == PWR_SAVE) && (Psm == PWR_ACTIVE))
		{
			DBGPRINT(RT_DEBUG_TRACE, "PsIndicate - %02x:%02x:%02x:%02x:%02x:%02x wakes up, act like rx PS-POLL\n", pAddr->Octet[0],pAddr->Octet[1],pAddr->Octet[2],pAddr->Octet[3],pAddr->Octet[4],pAddr->Octet[5]);
			// sleep station awakes, move all pending frames from PSQ to TXQ if any
			RTMPHandleRxPsPoll(pAd, pAddr, pEntry->Aid);
		}
		else if ((pEntry->PsMode != PWR_SAVE) && (Psm == PWR_SAVE))
		{
			DBGPRINT(RT_DEBUG_TRACE, "PsIndicate - %02x:%02x:%02x:%02x:%02x:%02x sleeps\n", pAddr->Octet[0],pAddr->Octet[1],pAddr->Octet[2],pAddr->Octet[3],pAddr->Octet[4],pAddr->Octet[5]);
		}
		
		pEntry->NoDataIdleCount = 0;
		pEntry->PsMode = Psm;
		pEntry->LastRssi = Rssi;
	} 
	else 
	{
		// not in table, try to learn it ???? why bother?
	}
	return TRUE;
}//end of PsIndicate();

extern unsigned long get_cmos_time(void);
/*
	==========================================================================
	Description:
		This routine is called to log a specific event into the event table.
		The table is a QUERY-n-CLEAR array that stop at full.
	==========================================================================
 */
VOID ApLogEvent(
	IN PRTMP_ADAPTER pAd,
	IN PMACADDR pAddr,
	IN USHORT   Event)
{
	if (pAd->EventTab.Num < MAX_NUM_OF_EVENT)
	{
		RT_802_11_EVENT_LOG *pLog = &pAd->EventTab.Log[pAd->EventTab.Num];
		pLog->SystemTime = jiffies;//get_cmos_time(); //tt_lin
		COPY_MAC_ADDR(&pLog->Addr, pAddr);
		pLog->Event = Event;
		DBGPRINT(RT_DEBUG_TRACE,"LOG#%d %02x:%02x:%02x:%02x:%02x:%02x %s\n",
			pAd->EventTab.Num, pAddr->Octet[0], pAddr->Octet[1], pAddr->Octet[2], 
			pAddr->Octet[3], pAddr->Octet[4], pAddr->Octet[5], pEventText[Event]);
		pAd->EventTab.Num += 1;
	}
}//end of ApLogEvent()

/*
	==========================================================================
	Description:
		Update ERP IE and CapabilityInfo based on STA association status.
		The result will be auto updated into the next outgoing BEACON in next
		TBTT interrupt service routine
	==========================================================================
 */
VOID ApUpdateCapabilityAndErpIe(
	IN PRTMP_ADAPTER pAd)
{
	UCHAR  i, ErpIeContent = 0;
	BOOLEAN ShortSlotCapable = (BOOLEAN)pAd->PortCfg.UseShortSlotTime;

	if (pAd->PortCfg.PhyMode != PHY_11BG_MIXED)
	{
			if (pAd->PortCfg.PhyMode == PHY_11G)
		{
			pAd->Mlme.ErpIeContent = ErpIeContent;
			AsicSetSlotTime(pAd, ShortSlotCapable);
		}
		return;
	}    
	for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
	{
		PMAC_TABLE_ENTRY pEntry = &pAd->MacTab.Content[i];
		if ((pEntry->Valid != TRUE) || (pEntry->Sst != SST_ASSOC))
			continue;
		
		// at least one 11b client associated, turn on ERP.NonERPPresent bit
		// almost all 11b client won't support "Short Slot" time, turn off for maximum compatibility
		if (pEntry->MaxSupportedRate < RATE_FIRST_OFDM_RATE)
		{
			ShortSlotCapable = FALSE;
			ErpIeContent |= 0x01;
		}

		// at least one client can't support short slot
		if ((pEntry->CapabilityInfo & 0x0400) == 0)
			ShortSlotCapable = FALSE;
	}

	// decide ErpIR.UseProtection bit, depending on pAd->PortCfg.UseBGProtection
	//    AUTO (0): UseProtection = 1 if any 11b STA associated
	//    ON (1): always USE protection
	//    OFF (2): always NOT USE protection
	if (pAd->PortCfg.UseBGProtection == 0)
	{
		ErpIeContent = (ErpIeContent) ? 0x03 : 0x00;
		if ((pAd->Mlme.LastOLBCDetectTime + (10000 * HZ)/1000) > pAd->Mlme.Now32) // legacy BSS exist within 10 sec
		{
			DBGPRINT(RT_DEBUG_TRACE, "ApUpdateCapabilityAndErpIe - Legacy 802.11b BSS overlaped\n");
			ErpIeContent |= 0x02;                                     // set Use_Protection bit
		}
	}
	else if (pAd->PortCfg.UseBGProtection == 1)   
		ErpIeContent |= 0x02;
	else
	{
		;
	}

	pAd->Mlme.ErpIeContent = ErpIeContent;

	//
	// deicide CapabilityInfo.ShortSlotTime bit
	//
	if (ShortSlotCapable)
		pAd->PortCfg.CapabilityInfo |= 0x0400;
	else
		pAd->PortCfg.CapabilityInfo &= 0xfbff;

	DBGPRINT(RT_DEBUG_TRACE, "ApUpdateCapabilityAndErpIe - Capability= 0x%04x, ERP is 0x%02x\n", 
		pAd->PortCfg.CapabilityInfo, ErpIeContent);

	AsicSetSlotTime(pAd, ShortSlotCapable);
	
}//end of ApUpdateCapabilityAndErpIe()

/*
	==========================================================================
	Description:
		Check if the specified STA pass the Access Control List checking.
		If fails to pass the checking, then no authentication nor association 
		is allowed
	Return:
		MLME_SUCCESS - this STA passes ACL checking
		
	==========================================================================
*/
BOOLEAN ApCheckAccessControlList(
	IN PRTMP_ADAPTER pAd,
	IN PMACADDR      pAddr) 
{
	BOOLEAN Result = TRUE;

	if (pAd->PortCfg.AccessControlList.Policy == 0)       // ACL is disabled
		Result = TRUE;
	else
	{
		ULONG i;
		if (pAd->PortCfg.AccessControlList.Policy == 1)   // ACL is a positive list
			Result = FALSE;
		else                                              // ACL is a negative list
			Result = TRUE;
		for (i=0; i<pAd->PortCfg.AccessControlList.Num; i++)
		{
			if (MAC_ADDR_EQUAL(pAddr, pAd->PortCfg.AccessControlList.Entry[i].Addr))
			{
				Result = !Result;
				break;
			}
		}
	}

	if (Result == FALSE)
	{
		DBGPRINT(RT_DEBUG_TRACE, "%02x:%02x:%02x:%02x:%02x:%02x failed ACL checking\n",
		pAddr->Octet[0],pAddr->Octet[1],pAddr->Octet[2],pAddr->Octet[3],pAddr->Octet[4],pAddr->Octet[5]);
	}

	return Result;
}//end of ApCheckAccessControlList()

/*
	==========================================================================
	Description:
		This routine update the current MAC table based on the current ACL.
		If ACL change causing an associated STA become un-authorized. This STA
		wil be kicked out immediately.
	==========================================================================
*/
VOID ApUpdateAccessControlList(
	IN PRTMP_ADAPTER pAd)
{
	USHORT   AclIdx, MacIdx;
	BOOLEAN  Matched;

	// ACL is disabled. do nothing about the MAC table
	if (pAd->PortCfg.AccessControlList.Policy == 0)
		return;
	
	for (MacIdx=0; MacIdx < MAX_LEN_OF_MAC_TABLE; MacIdx++)
	{
		if (! pAd->MacTab.Content[MacIdx].Valid) 
			continue;

		Matched = FALSE;
		for (AclIdx = 0; AclIdx < pAd->PortCfg.AccessControlList.Num; AclIdx++)
		{
			if (MAC_ADDR_EQUAL(&pAd->MacTab.Content[MacIdx].Addr, pAd->PortCfg.AccessControlList.Entry[AclIdx].Addr))
			{
				Matched = TRUE;
				break;
			}
		}
			
		if ((Matched == FALSE) && (pAd->PortCfg.AccessControlList.Policy == 1))
		{
			DBGPRINT(RT_DEBUG_TRACE, "STA not on positive ACL. remove it...\n");
			MacTableDeleteEntry(pAd, &pAd->MacTab.Content[MacIdx].Addr);
		}
		else if ((Matched == TRUE) && (pAd->PortCfg.AccessControlList.Policy == 2))
		{
			DBGPRINT(RT_DEBUG_TRACE, "STA on negative ACL. remove it...\n");
			MacTableDeleteEntry(pAd, &pAd->MacTab.Content[MacIdx].Addr);
		}
	}
}//end of ApUpdateAccessControlList()

/* 
	==========================================================================
	Description:
		Send out a NULL frame to a specified STA at a higher TX rate. The 
		purpose is to ensure the designated client is okay to received at this
		rate.
	==========================================================================
 */
VOID ApEnqueueNullFrame(
	IN PRTMP_ADAPTER pAd,
	IN PMACADDR      pAddr,
	IN UCHAR         TxRate,
	IN int			ToWhichWds) 
{
//    NDIS_STATUS    NState;
	PMACHDR         pNullFr;
	MACHDR			NullFr;

	DBGPRINT(RT_DEBUG_WDS, "ApEnqueueNullFrame->\n");

	// since TxRate may change, we have to change Duration each time
	pNullFr = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
	if (pNullFr != NULL) 
	{
		if(ToWhichWds == -1)
		{
			MgtMacHeaderInit( pAd, pNullFr, SUBTYPE_NULL_FUNC, 0, pAddr, &pAd->PortCfg.Bssid );
			//pNullFr->Type, pNullFr->SubType, pNullFr->Tods are Little-endian (s0) 
			// So changed to Big-endian 
			*(PUSHORT)&NullFr = RTMP_LE16_TO_CPU( *(PUSHORT)pNullFr ); //tt_lin_big (frame-control)
			NullFr.Type = BTYPE_DATA;//s0
			NullFr.Frds = 1; //s0
			*(PUSHORT)pNullFr = RTMP_CPU_TO_LE16( *(PUSHORT)&NullFr );//tt_lin_big

			//pNullFr.Duration = RTMPCalcDuration(pAd, TxRate, 14);//s1
			pNullFr->Duration = RTMP_CPU_TO_LE16( RTMPCalcDuration(pAd, TxRate, 14) ); //tt_lin_big
			
			ApSendNullFrame(pAd, (VOID *)pNullFr, sizeof(MACHDR), TxRate);
		}
#ifdef	WDS
		else
		{
			NdisZeroMemory(pNullFr, LENGTH_802_11_WITH_ADDR4);
			//init to big-endian
			pNullFr->Type = BTYPE_DATA; //s0
			pNullFr->SubType = SUBTYPE_NULL_FUNC; //s0
			pNullFr->Frds = 1; //s0
			pNullFr->Tods = 1; //s0
			*(PUSHORT)pNullFr = RTMP_CPU_TO_LE16( *(PUSHORT)pNullFr ); //tt_lin_big (Frame Control)
			
			
			COPY_MAC_ADDR(&pNullFr->Addr1, &pAd->WdsTab.WdsEntry[ToWhichWds].WdsAddr);
			COPY_MAC_ADDR(&pNullFr->Addr2, &pAd->CurrentAddress);
			COPY_MAC_ADDR(&pNullFr->Addr3, &pAd->WdsTab.WdsEntry[ToWhichWds].WdsAddr);
			COPY_MAC_ADDR(&pNullFr->Addr3 + MAC_ADDR_LEN, pAd->CurrentAddress);
			
			//pNullFr->Duration = RTMPCalcDuration(pAd, TxRate, 14); //s1
			pNullFr->Duration = RTMP_CPU_TO_LE16( RTMPCalcDuration(pAd, TxRate, 14) );

			ApSendNullFrame(pAd, (VOID *)pNullFr, LENGTH_802_11_WITH_ADDR4, TxRate);
		}
#endif
	}
}//end of ApEnqueueNullFrame()

//////////////////////////////////////////////////////////////////////////
//
VOID	ApSendNullFrame(
	IN	PRTMP_ADAPTER	pAdapter,
	IN	PVOID			pBuffer, /*MACHDR-(little-endian)*/
	IN	ULONG			Length,
	IN  UCHAR           TxRate)
{
	PUCHAR			pDest;
	PTXD_STRUC		pTxD;
	TXD_STRUC		TxD; //tt_lin_big

#if 0	
	if (pBuffer == NULL)
	{
		return;
	}
#endif
	if (RTMP_TEST_FLAG(pAdapter, fRTMP_ADAPTER_RESET_IN_PROGRESS))
	{
		MlmeConfirm((PVOID)pAdapter, pBuffer, 0);		
		return;
	}

#if 0	//tt_lin_wpa
	// WPA 802.1x secured port control
	if (((pAdapter->PortCfg.AuthMode == Ndis802_11AuthModeWPA) || 
		(pAdapter->PortCfg.AuthMode == Ndis802_11AuthModeWPAPSK)) &&
		(pAdapter->PortCfg.PortSecured == WPA_802_1X_PORT_NOT_SECURED))
	{
		MlmeConfirm((PVOID)pAdapter, pBuffer, 0);		
		return;
	}		

	if ((pAdapter->TxSwQueue0.Number != 0) || (pAdapter->TxSwQueue1.Number != 0) ||
		(pAdapter->TxSwQueue2.Number != 0) || (pAdapter->TxSwQueue3.Number != 0))
	{
		DBGPRINT(RT_DEBUG_TRACE,("Drop Null frame due to Tx queue not empty!\n"));
	}
	else
#endif     
	{
		// Make sure Tx ring resource won't be used by other threads
		NdisAcquireSpinLock(&pAdapter->TxRingLock);
	
		// Get the Tx Ring descriptor & Dma Buffer address
		pDest = (PUCHAR) pAdapter->TxRing[pAdapter->CurEncryptIndex].va_data_addr;              
		pTxD  = (PTXD_STRUC) pAdapter->TxRing[pAdapter->CurEncryptIndex].va_addr;
		*(PULONG)&TxD =  RTMP_LE32_TO_CPU( *(PULONG)pTxD ); //tt_lin_big

		if ((TxD.Owner == DESC_OWN_HOST) && //w0 
			(TxD.CipherOwn == DESC_OWN_HOST) &&  //w0
			(TxD.Valid == FALSE)) //w0
		{
			DBGPRINT(RT_DEBUG_WDS/*TRACE*/, "send NULL Frame @%d Mbps...\n", RateIdToMbps[TxRate]);
			NdisMoveMemory(pDest, pBuffer, Length); //output to header (802.11) 
			pAdapter->TxRing[pAdapter->CurEncryptIndex].FrameType = BTYPE_DATA;

			RTMPWriteTxDescriptor(pTxD, 
						TRUE, 
						CIPHER_NONE, 
						TRUE, 
						FALSE, 
						FALSE, 
						SHORT_RETRY, 
						IFS_BACKOFF, 
						TxRate, 
						4, 
						Length, 
						pAdapter->PortCfg.TxPreamble, 
						0);

			// Increase & maintain Tx Ring Index
			pAdapter->CurEncryptIndex++;
			if (pAdapter->CurEncryptIndex >= TX_RING_SIZE)
			{
				pAdapter->CurEncryptIndex = 0;
			}
			
			pAdapter->RalinkCounters.EncryptCount++;

			// Kick Encrypt Control Register at the end of all ring buffer preparation
			RTMP_IO_WRITE32(pAdapter, SECCSR1, 0x1);
			
		}
		NdisReleaseSpinLock(&pAdapter->TxRingLock);
	}
	MlmeConfirm((PVOID)pAdapter, pBuffer, 0);	//release memory pBuffer
}//end of ApSendNullFrame()

//v1.2.0
/* 
	==========================================================================
	Description:
		This routine is called at initialization. It returns a channel number
		that complies to regulation domain and less interference with current
		enviornment.
	Return:
		ch -  channel number that
	NOTE:
		the retrun channel number is guaranteed to comply to current regulation
		domain that recorded in pAd->PortCfg.CountryRegion
	==========================================================================
 */
UCHAR ApAutoSelectChannel(
	IN PRTMP_ADAPTER pAd)
{
	UCHAR ch;
	UCHAR dirtyness[MAX_LEN_OF_CHANNELS+1];
	ULONG FalseCca, FcsError;

	// passive scan channel 1-14. collect statistics
	NdisZeroMemory(dirtyness, MAX_LEN_OF_CHANNELS+1);
	for (ch=14; ch>=1; ch--)
	{
		AsicSwitchChannel(pAd, ch);
		pAd->Counters.GoodReceives = 0;
		pAd->Counters.RxErrors = 0;
		mdelay(500); // 0.5 sec at each channel
		RTMP_IO_READ32(pAd, CNT3, &FalseCca);
		FalseCca &= 0x0000ffff;
		RTMP_IO_READ32(pAd, CNT0, &FcsError);
		
		if (pAd->Counters.GoodReceives)
		{
			dirtyness[ch] += 10;
			if (ch > 1) dirtyness[ch-1] += 1;
			if (ch > 2) dirtyness[ch-2] += 1;
			if (ch > 3) dirtyness[ch-3] += 1;
			if (ch > 4) dirtyness[ch-4] += 1;
			if (ch < 14) dirtyness[ch+1] += 1;
			if (ch < 13) dirtyness[ch+2] += 1;
			if (ch < 12) dirtyness[ch+3] += 1;
			if (ch < 11) dirtyness[ch+4] += 1;
		}
		DBGPRINT(RT_DEBUG_TRACE,"Msleep at ch#%d to collect RX=%d, CRC error =%d, False CCA =%d\n", 
			ch, pAd->Counters.GoodReceives, FcsError, FalseCca);
	}

	DBGPRINT(RT_DEBUG_TRACE,"Dirtyness = %d %d %d %d - %d %d %d %d - %d %d %d %d - %d %d\n", 
		dirtyness[1], dirtyness[2], dirtyness[3], dirtyness[4], dirtyness[5], dirtyness[6], dirtyness[7],
		dirtyness[8], dirtyness[9], dirtyness[10], dirtyness[11], dirtyness[12], dirtyness[13], dirtyness[14]);

	// pick up a good channel that no one used
	for (ch = FirstChannel(pAd); ch !=0; ch = NextChannel(pAd, ch))
	{
		if (dirtyness[ch] == 0) break;
	}

	// if not available, then co-use a channel that's no interference
	if (ch == 0)
	{
		for (ch = FirstChannel(pAd); ch != 0; ch = NextChannel(pAd, ch))
		{
			if (dirtyness[ch] == 10) break;
		}
	}

	// if not available, then co-use a channel that has minimum interference
	if (ch == 0)
	{
		for (ch = FirstChannel(pAd); ch != 0; ch = NextChannel(pAd, ch))
		{
			if (dirtyness[ch] == 11) break;
		}
	}

	// still not available, pick up the first channel
	if (ch == 0)
		ch = FirstChannel(pAd);
	
	DBGPRINT(RT_DEBUG_TRACE,"ApAutoSelectChannel pick up ch#%d\n",ch);
	
	return ch;
}

UCHAR FirstChannel(
	IN PRTMP_ADAPTER pAd)
{
	UCHAR ch = 0;
	switch (pAd->PortCfg.CountryRegion)
	{
		case REGION_FCC:    // 1 - 11
			ch = 1;
			break;
		case REGION_IC:     // 1 -11
			ch = 1;
			break;
		case REGION_ISRAEL:  // 3 - 9
			ch = 3;
			break;
		case REGION_ETSI:   // 1 - 13
			ch = 1;
			break;
		case REGION_SPAIN:  // 10 - 11
			ch = 10;
			break;
		case REGION_FRANCE: // 10 -13
			ch = 10;
			break;
		case REGION_MKK:    // 14
			ch = 14;
			break;
		case REGION_MKK1:   // 1 - 14
			ch = 1;
			break;
		default:            // Error. should never happen
			break;
	}   
	return ch;
}

UCHAR NextChannel(
	IN PRTMP_ADAPTER pAd,
	IN UCHAR         ch)
{
	UCHAR next = ch + 1;
	switch (pAd->PortCfg.CountryRegion)
	{
		case REGION_FCC:    // 1 - 11
			if (next > 11) next = 0;
			break;
		case REGION_IC:     // 1 -11
			if (next > 11) next = 0;
			break;
		case REGION_ISRAEL: // 3 - 9
			if (next > 9)  next = 0;
			break;
		case REGION_ETSI:   // 1 - 13
			if (next > 13) next = 0;
			break;
		case REGION_SPAIN:  // 10 - 11
			if (next > 11) next = 0;
			break;
		case REGION_FRANCE: // 10 -13
			if (next > 13) next = 0;
			break;
		case REGION_MKK:    // 14
			if (next > 14) next = 0;
			break;
		case REGION_MKK1:   // 1 - 14
			if (next > 14) next = 0;
			break;
		default:            // Error. should never happen
			next = 0;
			break;
	}   
	return next;
}
//v1.2.0
