/******************************************************************************

    File:               isl38xx_pci.c
    Author:             W.Termorshuizen
    Version:            0.3.0.5
    Created:            June 19th, 2002
    Updated:            November 20th, 2002
    Company:            Intersil Americas Inc.
    Description:        This file contains the linux driver for the Intersil
                        Wireless LAN adapters.
    History:	    	Maintained in file isl38xx.h
    		
******************************************************************************/
#define __KERNEL_SYSCALLS__


#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/wait.h>

#include <asm/processor.h>      // Processor type for cache alignment.
#include <asm/bitops.h>
#include <asm/io.h>

#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

//#include <pcmcia/driver_ops.h>
#include <asm/uaccess.h>
#include <linux/unistd.h>
#include <linux/wireless.h>

#include "bloboidv2.h"      // additional types and defs for isl38xx fw
#include "isl38xx.h"


#define DEBUG( f, args... ) 	K_DEBUG( f, pc_debug, args )

/******************************************************************************
        Global variable definition section
******************************************************************************/
MODULE_AUTHOR("Wim Termorshuizen <wim.termorshuizen@intersil.com>");
MODULE_DESCRIPTION("Intersil ISL38xx 802.11 Wireless LAN adapter");
MODULE_LICENSE("GPL");

static const char *version =
    "version 0.3.0.5 20/11/2002 W.Termorshuizen, Intersil Americas Inc.\n";

int pc_debug = VERBOSE;
int init_mode = CARD_DEFAULT_MODE;
int init_channel = CARD_DEFAULT_CHANNEL;
int init_bsstype = CARD_DEFAULT_BSSTYPE;
int init_wep = CARD_DEFAULT_WEP;
int init_filter = CARD_DEFAULT_FILTER;
int init_wds = CARD_DEFAULT_WDS;
int init_authen = CARD_DEFAULT_AUTHEN;
int init_dot1x = CARD_DEFAULT_DOT1X;
char *init_ssid = NULL;

MODULE_PARM( pc_debug, "i" );
MODULE_PARM( init_mode, "i" );
MODULE_PARM( init_channel, "i" );
MODULE_PARM( init_bsstype, "i" );
MODULE_PARM( init_ssid, "s" );
MODULE_PARM( init_wep, "i" );
MODULE_PARM( init_filter, "i" );
MODULE_PARM( init_wds, "i" );
MODULE_PARM( init_authen, "i" );
MODULE_PARM( init_dot1x, "i" );

//struct net_device *root_isl38xx_device = NULL;

#ifdef	CONFIG_ISL3877
#include "isl3877fw.h"
#endif	/* CONFIG_ISL3877 */

#ifdef	CONFIG_ISL3890
#include "isl3890fw.h"
#endif	/* CONFIG_ISL3890 */
// temporary variables
static int errno;



#if TARGET_SYSTEM == CONFIG_PCMCIA

struct driver_operations isl38xx_ops =
{
    DRIVER_NAME, isl38xx_attach, isl38xx_suspend, isl38xx_resume,
        isl38xx_detach
};

#else


#define PCI_TYPE        (PCI_USES_MEM | PCI_ADDR0 | PCI_NO_ACPI_WAKE)

// PCI Class & Sub-Class code, Network-'Other controller'
#define PCI_CLASS_NETWORK_OTHERS 0x280

static struct pci_device_id pci_id_table[] =
{
#ifdef	CONFIG_ISL3877
    {
        PCIVENDOR_INTERSIL, PCIDEVICE_ISL3877,
        PCI_ANY_ID, PCI_ANY_ID,
        0, 0,
        // Driver data, we just put the name here
        (unsigned long) "Intersil PRISM Indigo Wireless LAN adapter"
    },
#endif	/* CONFIG_ISL3877 */
#ifdef	CONFIG_ISL3890
    {
        PCIVENDOR_INTERSIL, PCIDEVICE_ISL3890,
        PCI_ANY_ID, PCI_ANY_ID,
        0, 0,
        // Driver data, we just put the name here
        (unsigned long) "Intersil PRISM Duette Wireless LAN adapter"
    },
#endif	/* CONFIG_ISL3890 */
    {
        0, 0, 0, 0, 0, 0, 0
    }
};

//#if TARGET_SYSTEM == CONFIG_HOTPLUG

// register the device with the Hotplug facilities of the kernel
MODULE_DEVICE_TABLE( pci, pci_id_table );

//#endif	// CONFIG_HOTPLUG

static void	isl38xx_pci_remove_one(struct pci_dev *pdev) __devexit;
static int	isl38xx_probe_pci(struct pci_dev *,
			const struct pci_device_id *) __devinit;

struct pci_driver isl38xx_pci_drv_id =
{
    {},
    name:DRIVER_NAME,                // Driver name
    id_table:pci_id_table,               // id table
    probe:isl38xx_probe_pci,          // probe function
    remove:__devexit_p(isl38xx_pci_remove_one),   // remove function
    NULL,                       // save_state function
    NULL,                       // suspend function
    NULL,                       // resume function
    NULL,                       // enable_wake function
};

#endif


/******************************************************************************
    Device Control functions
******************************************************************************/
#if 0
void enable_update_interrupt(void *device)
{
    u32 reg;

    // set the update queue status bit in the Interrupt Enable Register
    reg = readl(device + ISL38XX_INT_EN_REG);
    reg |= ISL38XX_INT_IDENT_UPDATE;
    writel(reg, device + ISL38XX_INT_EN_REG);
}

void disable_update_interrupt(void *device)
{
    u32 reg;

    // clear the update queue status bit in the Interrupt Enable Register
    reg = readl(device + ISL38XX_INT_EN_REG);
    reg &= ~ISL38XX_INT_IDENT_UPDATE;
    writel(reg, device + ISL38XX_INT_EN_REG);
}
#endif	/* 0 */


/******************************************************************************
    Device Interface functions
******************************************************************************/
void isl38xx_interrupt(int irq, void *config, struct pt_regs *regs)
{
    u32 reg;
    isl38xx_private *private_config = config;
    void *device = private_config->device_id;

    // received an interrupt request on a shared IRQ line
    // check whether there is any source of interrupt on the device
    reg = readl(device + ISL38XX_INT_IDENT_REG);

    // also check the contents of the Interrupt Enable Register, because this
    // will filter out interrupt sources from other devices on the same irq !
    reg &= readl(device + ISL38XX_INT_EN_REG);

#ifdef CONFIG_BRECIS_PCI
    MSP_memory_sync();
#endif	/* CONFIG_BRECIS_PCI */
    if (reg &= ISL38XX_INT_SOURCES, reg != 0)
    {
        // a bit is set identifying an interrupt request from our device
        // reset the request bits in the Identification register
        writel(reg, device + ISL38XX_INT_ACK_REG);

#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_FUNCTION_CALLS,
	        "IRQ: Identification register 0x%p 0x%x \n", device, reg);
#endif

        // check for each bit in the register separately
        if (reg & ISL38XX_INT_IDENT_UPDATE)
        {
#if VERBOSE > SHOW_ERROR_MESSAGES 
            // Queue has been updated
            DEBUG(SHOW_TRACING, "IRQ: Update flag \n");

            DEBUG(SHOW_QUEUE_INDEXES,
                "CB drv Qs: [%i][%i][%i][%i][%i][%i]\n",
                le32_to_cpu(private_config->control_block->driver_curr_frag[0]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[1]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[2]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[3]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[4]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[5])
                );

            DEBUG(SHOW_QUEUE_INDEXES,
                "CB dev Qs: [%i][%i][%i][%i][%i][%i]\n",
                le32_to_cpu(private_config->control_block->device_curr_frag[0]),
                le32_to_cpu(private_config->control_block->device_curr_frag[1]),
                le32_to_cpu(private_config->control_block->device_curr_frag[2]),
                le32_to_cpu(private_config->control_block->device_curr_frag[3]),
                le32_to_cpu(private_config->control_block->device_curr_frag[4]),
                le32_to_cpu(private_config->control_block->device_curr_frag[5])
                );
#endif

            // check all three queues in priority order
            // call the PIMFOR receive function until the queue is empty
            while (isl38xx_in_queue(private_config->control_block,
                ISL38XX_CB_RX_MGMTQ) != 0)
            {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                DEBUG(SHOW_TRACING, "Received frame in Management Queue\n");
#endif
                isl38xx_receive_pimfor(private_config);

                // call the pimfor transmit function for processing the next
                // management frame in the shadow queue
                isl38xx_transmit_pimfor( private_config );
            }

            while (isl38xx_in_queue(private_config->control_block,
                ISL38XX_CB_RX_DATA_LQ) != 0)
            {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                DEBUG(SHOW_TRACING, "Received frame in Data Low Queue\n");
#endif
                isl38xx_receive(private_config);
            }

			// check whether the data transmit queues were full
            if (private_config->data_low_tx_full)
            {
                // check whether the transmit is not full anymore
                if (isl38xx_in_queue(private_config->control_block,
                    ISL38XX_CB_TX_DATA_LQ) != ISL38XX_CB_TX_QSIZE)
                {
                    // nope, the driver is ready for more network frames
                    netif_wake_queue(private_config->my_module);

                    // reset the full flag
                    private_config->data_low_tx_full = 0;
                }
            }
        }

        if (reg & ISL38XX_INT_IDENT_INIT)
        {
            // Device has been initialized
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "IRQ: Init flag, device initialized\n");
#endif

            // perform card initialization by sending objects
            isl38xx_initialize( private_config);
        }

        if (reg & ISL38XX_INT_IDENT_WAKEUP)
        {
            // Device has been woken up to active state
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "IRQ: Wakeup flag\n");
#endif
        }

        if (reg & ISL38XX_INT_IDENT_SLEEP)
        {
            // Device intends to move to powersave state
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "IRQ: Sleep flag\n");
#endif
        }
    }

    return;
}


/******************************************************************************
    Network Interface Control & Statistical functions
******************************************************************************/
int isl38xx_open(struct net_device *nwdev)
{
//    isl38xx_private *private_config = nwdev->priv;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_open\n");
#endif

    MOD_INC_USE_COUNT;

    netif_start_queue(nwdev);
//      netif_mark_up( nwdev );


    return 0;
}

int isl38xx_close(struct net_device *nwdev)
{
//      isl38xx_private *private_config = nwdev->priv;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_close\n");
#endif

    netif_stop_queue(nwdev);
//      netif_mark_down( nwdev );

    MOD_DEC_USE_COUNT;
    return 0;
}

void isl38xx_set_multicast_list(struct net_device *dev)
{
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_set_mc_list\n");
#endif
}

struct net_device_stats *isl38xx_statistics(struct net_device *nwdev)
{
    isl38xx_private *private_config = nwdev->priv;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_statistics\n");
#endif

    return &private_config->statistics;
}

int isl38xx_reset(isl38xx_private * private_config)
{

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_reset\n");
#endif

    // reset the device interface first ????????????
    isl38xx_interface_reset(private_config);

    // restore the configuration in the device


    // ........

    return 0;
}

void isl38xx_interface_reset(isl38xx_private * private_config)
{
    u32 reg;
    void *device = private_config->remapped_device_base;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_interface_reset\n");
#endif

    // load the address of the control block in the device
    writel(private_config->device_host_address,
           device + ISL38XX_CTRL_BLK_BASE_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // set the reset bit in the Device Interrupt Register
    reg = readl(device + ISL38XX_DEV_INT_REG);
    reg |= ISL38XX_DEV_INT_RESET;
    writel(reg, device + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // reset the reset bit in the Device Interrupt Register ?
    reg &= ~ISL38XX_DEV_INT_RESET;
    writel(reg, device + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // enable the interrupt for detecting initialization
    reg = ISL38XX_INT_IDENT_INIT | ISL38XX_INT_IDENT_UPDATE |
        ISL38XX_INT_IDENT_SLEEP | ISL38XX_INT_IDENT_WAKEUP;
    writel(reg, device + ISL38XX_INT_EN_REG);
    udelay(ISL38XX_WRITEIO_DELAY);
}

/******************************************************************************
    Device configuration functions
******************************************************************************/
//static int __devinit	isl38xx_upload_firmware(struct pci_dev *pci_device,
//			char *,
//#if defined(CONFIG_ISL3877) || defined(CONFIG_3890)
//			int size,
//#endif	/* CONFIG_ISL3877 || CONFIG_ISL3890 */
//			isl38xx_private *);

static int __devinit	isl38xx_upload_firmware(struct pci_dev *pci_device,
			char *filename,
#if	defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890)
			int size,
#endif	/* CONFIG_ISL3877 || CONFIG_ISL3890 */
			isl38xx_private *private_config)
{
    u32 reg;
    long bcount, length;
#if	! (defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890))
    int ifp;
    mm_segment_t fs;
    static char __devinitdata buffer[ISL38XX_MEMORY_WINDOW_SIZE];
#endif	/* ! (CONFIG_ISL3877 || CONFIG_ISL3890) */
    void *device;

    // firmware uploading starts with resetting the device
    device = private_config->remapped_device_base;

    // clear the RAMBoot and the Reset bit
    reg = readl(device + ISL38XX_CTRL_STAT_REG);
    reg &= ~ISL38XX_CTRL_STAT_RESET;
    reg &= ~ISL38XX_CTRL_STAT_RAMBOOT;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // set the Reset bit without reading the register !
    reg |= ISL38XX_CTRL_STAT_RESET;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // clear the Reset bit
    reg &= ~ISL38XX_CTRL_STAT_RESET;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);

    // wait a while for the device to reboot
    mdelay(50);

    // prepare the Direct Memory Base register
    reg = ISL38XX_DEV_FIRMWARE_ADDRES;

#if	defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890)
    length = 0;
#else
    // for file opening temporarily tell the kernel I am not a user for
    // memory management segment access
    fs = get_fs();
    set_fs(KERNEL_DS);

    // open the file with the firmware for uploading
    ifp = open(filename, O_RDONLY, 0);
    if (ifp < 0)
    {
        // error opening the file
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: File open failed, returning %d "
        	"errno=%d\n", ifp, errno);
        set_fs(fs);
        return -1;
    }
#endif	/* ! (CONFIG_ISL3877 || CONFIG_ISL3890) */
    // enter a loop which reads data blocks from the file and writes them
    // to the Direct Memory Windows
    do
    {
        // set the cards base address for writing the data
        writel(reg, device + ISL38XX_DIR_MEM_BASE_REG);

#if	defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890)
	bcount = ISL38XX_MEMORY_WINDOW_SIZE;
	if (bcount + length > size)
		bcount = size - length;
        // write the data to the Direct Memory Window
        if (bcount)
        	memcpy_toio(device + ISL38XX_DIRECT_MEM_WIN,
        		filename + length, bcount);

	length += bcount;
#else
        // read a block of data until buffer is full or end of file
        length = 0;
        do
        {
            bcount = read(ifp, &buffer[length], sizeof(buffer) - length);
            if (bcount < 0) {
                DEBUG(SHOW_ERROR_MESSAGES, "read returned %ld\n", bcount);
                close(ifp);
                set_fs(fs);
                return -1;
            }
            length += bcount;
        }
        while ((length != ISL38XX_MEMORY_WINDOW_SIZE) && (bcount != 0));

        // write the data to the Direct Memory Window
        memcpy_toio(device + ISL38XX_DIRECT_MEM_WIN, buffer, length);
#endif	/* CONFIG_ISL3877 || CONFIG_ISL3890 */

        // increment the write address
        reg += ISL38XX_MEMORY_WINDOW_SIZE;
    }
    while (bcount != 0);

#if ! (defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890))
    // close the file
    close(ifp);

    // switch back the segment setting
    set_fs(fs);
#endif	/* ! (CONFIG_ISL3877 || CONFIG_ISL3890) */

    // now reset the device
    // clear the Reset & ClkRun bit, set the RAMBoot bit
    reg = readl(device + ISL38XX_CTRL_STAT_REG);
	reg &= ~ISL38XX_CTRL_STAT_CLKRUN;
    reg &= ~ISL38XX_CTRL_STAT_RESET;
    reg |= ISL38XX_CTRL_STAT_RAMBOOT;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // set the reset bit latches the host override and RAMBoot bits
    // into the device for operation when the reset bit is reset
    reg |= ISL38XX_CTRL_STAT_RESET;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // clear the reset bit should start the whole circus
    reg &= ~ISL38XX_CTRL_STAT_RESET;
    writel(reg, device + ISL38XX_CTRL_STAT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // now the last step is to reset the interface
    isl38xx_interface_reset(private_config);

    return 0;
}

struct net_device *isl38xx_probe(struct net_device *nwdev,
                                       long ioaddr, int irq)
{
    // initially setup an ethernet device
    nwdev = init_etherdev(nwdev, 0);

    // setup the structure members
    nwdev->base_addr = ioaddr;
    nwdev->irq = irq;

    // initialize the function pointers
    nwdev->open = &isl38xx_open;
    nwdev->stop = &isl38xx_close;
    nwdev->get_stats = &isl38xx_statistics;
    nwdev->do_ioctl = &isl38xx_ioctl;
    nwdev->hard_start_xmit = &isl38xx_transmit;

//      nwdev->set_multicast_list = &set_rx_mode;

#ifdef HAVE_TX_TIMEOUT
    nwdev->watchdog_timeo = ISL38XX_TX_TIMEOUT;
    nwdev->tx_timeout = &isl38xx_tx_timeout;
#endif

    return nwdev;
}

int isl38xx_config(struct pci_dev *pci_device,
                         struct net_device *nw_device)
{
    void *mapped_address;
    void *driver_address;
    void *queue_base;
    dma_addr_t device_address;
    long counter;
    long offset;
    queue_entry *pointerq;
    isl38xx_private *private_config;
    isl38xx_control_block *control_block;
    struct sk_buff *skb;

    // remap the PCI device base address to accessable
    if (mapped_address = ioremap(nw_device->base_addr, ISL38XX_PCI_MEM_SIZE),
        mapped_address == NULL)
    {
        // error in remapping the PCI device memory address range
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: PCI memory remapping failed\n");
        return -1;
    }

    // save the start and end address of the PCI memory area
    nw_device->mem_start = (unsigned long) mapped_address;
    nw_device->mem_end = (unsigned long) mapped_address + ISL38XX_PCI_MEM_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "PCI Memory remapped to 0x%p\n", mapped_address);
#endif

    // allocate a block of memory on the host for the control block
    if (driver_address = pci_alloc_consistent(pci_device,
        ISL38XX_HOST_MEM_BLOCK, &device_address), driver_address == NULL)
    {
        // error allocating the block of PCI memory
        DEBUG(SHOW_ERROR_MESSAGES,
                "ERROR: Could not allocate DMA accessable " "memory \n");
        iounmap(mapped_address);
        return -1;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Device memory block at 0x%x \n", device_address);
    DEBUG(SHOW_TRACING, "Driver memory block at 0x%p \n", driver_address);
    DEBUG(SHOW_TRACING, "Memory size 0x%x bytes\n", ISL38XX_HOST_MEM_BLOCK);
#endif

    // add a private device structure to the network device for private
    // definitions needed by the driver
    if (private_config = kmalloc(sizeof(isl38xx_private), GFP_KERNEL |
        GFP_DMA), private_config == NULL)
    {
        // error allocating the DMA accessable memory area
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Could not allocate additional "
        "memory \n");
        pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK, driver_address,
            device_address);
        iounmap(mapped_address);
        return -1;
    }

    memset(private_config, 0, sizeof(isl38xx_private));
    nw_device->priv = private_config;
//    private_config->next_module = root_isl38xx_device;
//    root_isl38xx_device = nw_device;
    private_config->my_module = nw_device;

    // assign the Control Block pointer to the allocated area
    control_block = (isl38xx_control_block *) driver_address;
    private_config->control_block = control_block;

    // clear the indexes in the frame pointer
    for (counter = 0; counter < ISL38XX_CB_QCOUNT; counter++)
    {
        control_block->driver_curr_frag[counter] = cpu_to_le32(0);
        control_block->device_curr_frag[counter] = cpu_to_le32(0);
    }

    // save the device configurations in the device private configuration
    private_config->remapped_device_base = mapped_address;
    private_config->device_host_address = device_address;
    private_config->device_psm_buffer = device_address + CONTROL_BLOCK_SIZE;
    private_config->driver_mem_address = driver_address;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "PSM Buffer at 0x%p size %i\n",
        (void *) private_config->device_psm_buffer, PSM_BUFFER_SIZE );
#endif

    // configure the allocated area for PSM buffer and queueing mechanism
    // with the following layout
    // offset 0x0000:       Control block,  size = 1024 bytes
    //      ,  ,  0x0400:   PSM Buffer, size = entries * 0x0600
    //      ,  ,  0x????:   Queue memory

    // set the queue base address behind the control block
    queue_base = driver_address + CONTROL_BLOCK_SIZE + PSM_BUFFER_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Queue memory base at 0x%p \n", queue_base);
#endif

    // initialize all the queues in the private configuration structure
    init_queue(&private_config->mgmt_tx_freeq);
    init_queue(&private_config->mgmt_rx_freeq);

    init_queue(&private_config->mgmt_tx_shadowq);
    init_queue(&private_config->mgmt_rx_shadowq);
    init_queue(&private_config->ioctl_rx_queue);
    init_queue(&private_config->trap_rx_queue);
    init_queue(&private_config->pimfor_rx_queue);

    // determine the offset for each queue entry configuration
    // allign on 16 byte boundary
    offset = sizeof(queue_entry) / 16;
    offset *= 16;
    offset += ((sizeof(queue_entry) % 16) != 0 ? 16 : 0);

    // configure all management queue(s) entries and add them to the freeqs
    for (counter = 0; counter < MGMT_TX_FRAME_COUNT; counter++)
    {
        // configure the queue entry at the beginning of the memory block
        pointerq = (queue_entry *) queue_base;
        pointerq->host_address = (u32) queue_base + offset;
        pointerq->size = MGMT_FRAME_SIZE - offset;
        pointerq->fragments = 1;

        // also translate the host address for the device
        pointerq->dev_address = (u32) virt_to_bus((void *)
                pointerq->host_address);

        // add the configured entry to the management tx root queue
        put_queue(mapped_address, &private_config->mgmt_tx_freeq, pointerq);

        // increment the queue base address pointer to the next entry
        queue_base += MGMT_FRAME_SIZE;

        // perform a check on the destination address in PCI memory
        if (queue_base > driver_address + ISL38XX_HOST_MEM_BLOCK)
        {
            // Oops, end address of fragment will be out of bound
            DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Queue memory out of bound, "
                "addres 0x%p, enlarge host memory \n", queue_base);
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return -1;
        }
    }

    for (counter = 0; counter < MGMT_RX_FRAME_COUNT; counter++)
    {
        // configure the queue entry at the beginning of the memory block
        pointerq = (queue_entry *) queue_base;
        pointerq->host_address = (u32) queue_base + offset;
        pointerq->size = MGMT_FRAME_SIZE - offset;
        pointerq->fragments = 1;

        // also translate the host address for the device
        pointerq->dev_address = (u32) virt_to_bus((void *)
            pointerq->host_address);

        // add the configured entry to the management rx root queue
        put_queue(mapped_address, &private_config->mgmt_rx_freeq, pointerq);

        // increment the queue base address pointer to the next entry
        queue_base += MGMT_FRAME_SIZE;

        // perform a check on the destination address in PCI memory
        if (queue_base > driver_address + ISL38XX_HOST_MEM_BLOCK)
        {
            // Oops, end address of fragment will be out of bound
            DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Queue memory out of bound, "
                "addres 0x%p, enlarge host memory \n", queue_base);
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return -1;
        }
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Queue memory end at 0x%p \n", queue_base);
#endif

    // Joho, ready setting up the queueing stuff, now fill the shadow rx queues
    // of both the management and data queues
    for (counter = 0; counter < ISL38XX_CB_MGMT_QSIZE; counter++)
    {
        // get an entry from the freeq and put it in the shadow queue
        if (get_queue(mapped_address, &private_config->mgmt_rx_freeq,
            &pointerq))
        {
            // Oops, no more entries in the freeq ?
            DEBUG(SHOW_ERROR_MESSAGES, "Error: No more entries in freeq\n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return -1;
        }
        else
        {
            // use the entry for the device interface management rx queue
            // these should always be inline !
            control_block->rx_data_mgmt[counter].address =
                cpu_to_le32(pointerq->dev_address);

            // put the entry got from the freeq in the shadow queue
            put_queue(mapped_address, &private_config->mgmt_rx_shadowq,
                pointerq);
        }
    }

    for (counter = 0; counter < ISL38XX_CB_RX_QSIZE; counter++)
    {
        // allocate an sk_buff for received data frames storage
		// each frame on receive size consists of 1 fragment
	    // include any required allignment operations
        if (skb = dev_alloc_skb(MAX_FRAGMENT_SIZE+2), skb == NULL)
        {
            // error allocating an sk_buff structure elements
            DEBUG(SHOW_ERROR_MESSAGES, "Error allocating skb \n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return -1;
        }

		// add the new allocated sk_buff to the buffer array
		private_config->data_low_rx[counter] = skb;

		// map the allocated skb data area to pci
        private_config->pci_map_rx_address[counter] = (u32) pci_map_single(
        	private_config->pci_device, (void *) skb->data, skb->len,
            PCI_DMA_FROMDEVICE );
        if( private_config->pci_map_rx_address[counter] == (u32) NULL )
        {
            // error mapping the buffer to device accessable memory address
            DEBUG(SHOW_ERROR_MESSAGES, "Error mapping DMA address\n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return -1;
        }

		// update the fragment address values in the contro block rx queue
        control_block->rx_data_low[counter].address = cpu_to_le32( 
		 	private_config->pci_map_rx_address[counter] );
    }

    // since the receive queues are filled with empty fragments, now we can
    // set the corresponding indexes in the Control Block
    control_block->driver_curr_frag[ISL38XX_CB_RX_DATA_LQ] =
        cpu_to_le32(ISL38XX_CB_RX_QSIZE);
    control_block->driver_curr_frag[ISL38XX_CB_RX_MGMTQ] =
        cpu_to_le32(ISL38XX_CB_MGMT_QSIZE);

    // reset the real index registers and full flags
    private_config->index_mgmt_rx = 0;
	private_config->index_mgmt_tx = 0;
    private_config->free_data_rx = 0;
    private_config->free_data_tx = 0;
    private_config->data_low_tx_full = 0;

    // reset the queue read locks, process wait counter
    private_config->ioctl_queue_lock = 0;
    private_config->processes_in_wait = 0;
    private_config->data_fragments_in_wait = 0;

    return 0;
}

/******************************************************************************
    Module initialization functions
******************************************************************************/
#if TARGET_SYSTEM == CONFIG_PCMCIA

dev_node_t *isl38xx_attach(dev_locator_t * loc)
{
    u32 io;
    u16 dev_id;
    u8 bus, devfn, irq;
    struct pci_dev *pci_device;
    struct net_device *nw_device;
    dev_node_t *node;
    isl38xx_private *private_config;
    int rvalue;
    int dma_mask = 0xffffffff;
    char firmware[256];

    // perform some initial setting checks
    if (loc->bus != LOC_PCI)
        return NULL;
    bus = loc->b.pci.bus;
    devfn = loc->b.pci.devfn;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_attach(bus %d, function %d)\n",
        bus, devfn);
#endif

    // get some pci settings for verification
    pcibios_read_config_dword(bus, devfn, PCI_BASE_ADDRESS_0, &io);
    pcibios_read_config_byte(bus, devfn, PCI_INTERRUPT_LINE, &irq);
    pcibios_read_config_word(bus, devfn, PCI_DEVICE_ID, &dev_id);
    if (io &= ~3, io == 0 || irq == 0)
    {
        DEBUG(SHOW_ERROR_MESSAGES, "The ISL38XX Ethernet interface was not "
        "assigned an %s.\n" KERN_ERR "  It will not be activated.\n",
        io == 0 ? "I/O address" : "IRQ");
        return NULL;
    }
    // get pci device information by loading the pci_dev structure
    if (pci_device = pci_find_slot(bus, devfn), pci_device == NULL)
    {
        // error reading the pci device structure
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not get PCI device "
            "information \n");
        return NULL;
    }

    // determine what the supported DMA memory region is
    while( pci_set_dma_mask( pci_device, dma_mask ) != 0 )
    {
        // range not supported, shift the mask and check again
        if( dma_mask >>=  1, dma_mask == 0 )
        {
            // mask is zero, DMA memory not supported by PCI
            DEBUG(SHOW_ERROR_MESSAGES, "DMA Memory not supported\n" );
            return NULL;
        }
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "DMA Memory support mask is 0x%x \n", dma_mask );
#endif

    // setup the network device interface and its structure
    if (nw_device = isl38xx_probe(NULL, (long) io, (int) irq),
        nw_device == NULL)
    {
        // error configuring the driver as a network device
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not configure "
        "network device \n");
        return NULL;
    }

    if (isl38xx_config(pci_device, nw_device) == -1)
    {
        // error configuring the PCI bus
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not configure "
        "PCI device \n");
        return NULL;
    }
    // save the interrupt request line and use the remapped device base address
    // as the device identification both for uniqueness and parameter passing
    // to the interrupt handler
    private_config = nw_device->priv;
    private_config->pci_irq = irq;
    private_config->pci_dev_id = dev_id;
    private_config->device_id = private_config->remapped_device_base;
    private_config->pci_device = pci_device;

    // request for the interrupt before uploading the firmware
    if (rvalue = request_irq(irq, &isl38xx_interrupt, SA_INTERRUPT | SA_SHIRQ,
        DRIVER_NAME, private_config), rvalue != 0)
    {
        // error, could not hook the handler to the irq
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not install "
            "IRQ-handler \n");
        return NULL;
    }

    // select the firmware file depending on the device id, take for default
    // the 3877 firmware file
    if( dev_id == PCIDEVICE_ISL3890 )
    	strcpy( firmware, ISL3890_IMAGE_FILE );
    else
    	strcpy( firmware, ISL3877_IMAGE_FILE );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Firmware file: %s \n", firmware );
#endif
	
    if (isl38xx_upload_firmware( pci_device, firmware, private_config ) == -1)
    {
        // error uploading the firmware
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not upload the "
        "firmware \n");
        return NULL;
    }

    // finally setup the node structure with the device information
    node = kmalloc(sizeof(dev_node_t), GFP_KERNEL);
    strcpy(node->dev_name, nw_device->name);
    node->major = 0;
    node->minor = 0;
    node->next = NULL;
    MOD_INC_USE_COUNT;
    return node;
}

void isl38xx_suspend(dev_node_t * node)
{
    struct net_device **devp, **next;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_suspend(%s)\n", node->dev_name);
#endif

    for (devp = &root_isl38xx_device; *devp; devp = next)
    {
        next = &((isl38xx_private *) (*devp)->priv)->next_module;
        if (strcmp((*devp)->name, node->dev_name) == 0)
        break;
    }

    if (*devp)
    {
//        long ioaddr = (*devp)->base_addr;
//        isl38xx_pause(*devp);
    }
}

void isl38xx_resume(dev_node_t * node)
{
    struct net_device **devp, **next;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_resume(%s)\n", node->dev_name);
#endif

    for (devp = &root_isl38xx_device; *devp; devp = next)
    {
        next = &((isl38xx_private *) (*devp)->priv)->next_module;
        if (strcmp((*devp)->name, node->dev_name) == 0)
        break;
    }

    if (*devp)
    {
//        isl38xx_restart(*devp);
    }
}

void isl38xx_detach(dev_node_t * node)
{
    struct net_device **devp, **next;
    isl38xx_private *private_config;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_detach(%s)\n", node->dev_name);
#endif

    for (devp = &root_isl38xx_device; *devp; devp = next)
    {
        next = &((isl38xx_private *) (*devp)->priv)->next_module;
        if (strcmp((*devp)->name, node->dev_name) == 0)
        break;
    }

    if (*devp)
    {
        // free the interrupt request
        private_config = (*devp)->priv;
        writel(0, private_config->remapped_device_base + ISL38XX_INT_EN_REG);
        free_irq(private_config->pci_irq, private_config);

        // free the node
        kfree(node);
        MOD_DEC_USE_COUNT;
    }
}

int init_module(void)
{
    DEBUG(SHOW_ANYTHING, "Init module: %s", version);

    register_driver(&isl38xx_ops);

    return 0;
}

void cleanup_module(void)
{
    struct net_device *next_dev;
    isl38xx_private *private_config;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "Cleanup module \n");
#endif

    unregister_driver(&isl38xx_ops);

    /* No need to check MOD_IN_USE, as sys_delete_module() checks. */
    while (root_isl38xx_device)
    {
        private_config = (isl38xx_private *) root_isl38xx_device->priv;
        next_dev = private_config->next_module;

#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "Cleanup netdevice \n");
#endif

        // unregister the network device
        unregister_netdev(root_isl38xx_device);

        // free the PCI memory and unmap the remapped page
        pci_free_consistent(private_config->pci_device, ISL38XX_HOST_MEM_BLOCK,
            private_config->driver_mem_address,
            private_config->device_host_address);
        iounmap(private_config->remapped_device_base);

        // free the separately allocated areas
        kfree(private_config);
        kfree(root_isl38xx_device);

        // change the root device pointer to the next device for clearing
        root_isl38xx_device = next_dev;
    }
}

#else	/* TARGET_SYSTEM == CONFIG_PCMCIA */

static int __devinit isl38xx_probe_pci(struct pci_dev *pci_device,
    const struct pci_device_id *id)
{
    void *phymem;
    u16 irq;
    u8 bus, devfn;
    struct net_device *nw_device;
    isl38xx_private *private_config;
    int rvalue, dev_id;
    int dma_mask = 0xffffffff;
    char *firmware;
#if defined(CONFIG_ISL3877) || defined(CONFIG_ISL3890)
    int size;
#endif	/* CONFIG_ISL3877 || CONFIG_ISL3890 */

    // Enable the pci device
    if (pci_enable_device(pci_device))
    {
        DEBUG(SHOW_ERROR_MESSAGES, "%s: pci_enable_device() failed.\n",
            DRIVER_NAME );
        return -EIO;
    }
    // Figure out our resources
    irq = pci_device->irq;
    bus = pci_device->bus->number;
    devfn = pci_device->devfn;
    dev_id = pci_device->device;

    // determine what the supported DMA memory region is
    while( pci_set_dma_mask( pci_device, dma_mask ) != 0 )
    {
        // range not supported, shift the mask and check again
        if( dma_mask >>=  1, dma_mask == 0 )
        {
            // mask is zero, DMA memory not supported by PCI
            DEBUG(SHOW_ERROR_MESSAGES, "DMA Memory not supported\n" );
            return -EIO;
        }
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "DMA Memory support mask is 0x%x \n", dma_mask );
#endif

    // call for the physical address to address the PCI device
    phymem = (void *) pci_resource_start(pci_device, 0);

    // request the pci device for regions ???
    if (rvalue = pci_request_regions(pci_device, DRIVER_NAME ), rvalue)
    {
        DEBUG(SHOW_ERROR_MESSAGES, "pci_request_regions failure, rvalue %i\n",
            rvalue );
        return -EIO;
    }

    // enable PCI bus-mastering
    pci_set_master(pci_device);

    // Log the device resources information
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "isl38xx device: phymem:0x%lx, irq:%d \n",
        (long) phymem, irq);
#endif

    // setup the network device interface and its structure
    if (nw_device = isl38xx_probe(NULL, (long) phymem, (int) irq),
        nw_device == NULL)
    {
        // error configuring the driver as a network device
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not configure "
            "network device \n");
        return -EIO;
    }

    // configure the device memory resources
    if (isl38xx_config(pci_device, nw_device) == -1)
    {
        // error configuring the PCI bus
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not configure "
            "PCI device \n");
        return -EIO;
    }

    // save the interrupt request line and use the remapped device base address
    // as the device identification both for uniqueness and parameter passing
    // to the interrupt handler
    private_config = nw_device->priv;
    private_config->pci_irq = irq;
    private_config->pci_dev_id = dev_id;
    private_config->device_id = private_config->remapped_device_base;
    private_config->pci_device = pci_device;

    // request for the interrupt before uploading the firmware
    if (rvalue = request_irq(irq, &isl38xx_interrupt, SA_INTERRUPT | SA_SHIRQ,
        DRIVER_NAME, private_config), rvalue != 0)
    {
        // error, could not hook the handler to the irq
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not install "
            "IRQ-handler \n");
        return -EIO;
    }

#if	defined(CONFIG_ISL3877) && defined(CONFIG_ISL3890)
#define	FIRMWARE	firmware, size
    if (dev_id == PCIDEVICE_ISL3890)
	firmware = isl3890fw, size = sizeof isl3890fw;
    else
	firmware = isl3877fw, size = sizeof isl3877fw;
#elif	defined(CONFIG_ISL3877)
#define	FIRMWARE	firmware, size
    firmware = isl3877fw, size = sizeof isl3877fw;
#elif	defined(CONFIG_ISL3890)
#define	FIRMWARE	firmware, size
    firmware = isl3890fw, size = sizeof isl3890fw;
#else
    // select the firmware file depending on the device id, take for default
    // the 3877 firmware file
#define	FIRMWARE	firmware
    if( dev_id == PCIDEVICE_ISL3890 )
    	firmware = ISL3890_IMAGE_FILE;
    else
    	firmware = ISL3877_IMAGE_FILE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Firmware file: %s \n", firmware );
#endif
#endif	/* CONFIG_ISL3877 || CONFIG_ISL3890 */
	
    if (isl38xx_upload_firmware( pci_device, FIRMWARE, private_config ) == -1)
    {
        // error uploading the firmware
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: isl38xx could not upload the "
            "firmware\n");
        return -EIO;
    }

    MOD_INC_USE_COUNT;

    return 0;
}

static int	init_module(void)
{
    DEBUG(SHOW_ANYTHING, "%s", version);
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Loaded %s\n", DRIVER_NAME );
#endif

    if (pci_register_driver(&isl38xx_pci_drv_id) <= 0)
    {
        DEBUG(SHOW_ERROR_MESSAGES,
            "isl38xx: No devices found, driver not installed.\n");
        pci_unregister_driver(&isl38xx_pci_drv_id);
        return -ENODEV;
    }

    return 0;
}

#if 0
static void cleanup_module(void)
{
    struct net_device *next_dev;
    isl38xx_private *private_config;

    MOD_DEC_USE_COUNT;

    // No need to check MOD_IN_USE, as sys_delete_module() checks
    while (root_isl38xx_device)
    {
        private_config = (isl38xx_private *) root_isl38xx_device->priv;
        next_dev = private_config->next_module;

        writel(0,private_config->remapped_device_base + ISL38XX_INT_EN_REG);
        free_irq(private_config->pci_irq, private_config);

        // unregister the network device
        unregister_netdev(root_isl38xx_device);

        // free the PCI memory and unmap the remapped page
        pci_free_consistent(private_config->pci_device, ISL38XX_HOST_MEM_BLOCK,
            private_config->driver_mem_address,
            private_config->device_host_address);
        iounmap(private_config->remapped_device_base);
        pci_release_regions(private_config->pci_device);

        // free the separately allocated areas
        kfree(private_config);
        kfree(root_isl38xx_device);

        // change the root device pointer to the next device for clearing
        root_isl38xx_device = next_dev;
    }

    pci_unregister_driver(&isl38xx_pci_drv_id);

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Unloaded %s\n", DRIVER_NAME );
#endif

    return;
}
#endif	/* 0 */


static void	__devexit isl38xx_pci_remove_one(struct pci_dev *pdev)
{
	struct net_device	*dev = pci_get_drvdata(pdev);
	isl38xx_private	*priv = dev->priv;

#if VERBOSE > SHOW_FUNCTION_CALLS 
	DEBUG(SHOW_TRACING, DRIVER_NAME ": remove_one\n");
#endif

	if (dev == 0)
		BUG();

#if 0
	if (root_isl38xx_device) {
		this_dev = root_isl38xx_device;
		last_dev = 0;
		private_config = (isl38xx_private *) this_dev->priv;
		while (private_config->pci_device != pdev) {
			last_dev = this_dev;
			this_dev = this_dev->next_module;
			if (this_dev == 0)
				break;
			private_config = (isl38xx_private *) this_dev->priv;
		}
		if (root_isl38xx_device->next_module) {
			root_isl38xx_device->next_module =
				pdev->next_module;
		}
	}
	private_config = (isl38xx_private *) root_isl38xx_device->priv;
	next_dev = private_config->next_module;
#endif	/* 0 */

	writel(0,priv->remapped_device_base + ISL38XX_INT_EN_REG);
        free_irq(priv->pci_irq, priv);

        // unregister the network device
        unregister_netdev(dev);

        // free the PCI memory and unmap the remapped page
        pci_free_consistent(priv->pci_device, ISL38XX_HOST_MEM_BLOCK,
            priv->driver_mem_address,
            priv->device_host_address);
        iounmap(priv->remapped_device_base);
        pci_release_regions(priv->pci_device);

        // free the separately allocated areas
        kfree(priv);
        kfree(dev);

        // change the root device pointer to the next device for clearing
//        root_isl38xx_device = next_dev;
}


static int __init	isl38xx_pci_init(void)
{
#ifdef	MODULE
	printk(version);
#endif
	return init_module();
}


static void __exit	isl38xx_pci_exit(void)
{
	pci_unregister_driver(&isl38xx_pci_drv_id);
//	cleanup_module();
}


module_init(isl38xx_pci_init);
module_exit(isl38xx_pci_exit);

#endif      // TARGET_SYSTEM
