/*  $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/net/wireless/intersil/islpci_dev.c,v 1.3 2003/03/21 20:12:29 mrustad Exp $
 *  
 *  Copyright (C) 2002 Intersil Americas Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/config.h>
#include <linux/version.h>
#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/netdevice.h>
#include <linux/pci.h>
#include <linux/etherdevice.h>

#include <linux/init.h>
#include <asm/io.h>

#include "isl_gen.h"
#include "isl_38xx.h"

#ifdef WIRELESS_IOCTLS
#include "isl_ioctl.h"
#endif

#include "islpci_dev.h"
#include "islpci_mgt.h"
#include "islpci_eth.h"

#ifdef WDS_LINKS
#include <isl_mgt.h>
#include <isl_wds.h>
#endif

/******************************************************************************
        Global variable definition section
******************************************************************************/
extern int pc_debug;

//struct net_device *root_islpci_device = NULL;


/******************************************************************************
    Device Interrupt Handler
******************************************************************************/
void islpci_interrupt(int irq, void *config, struct pt_regs *regs)
{
    u32 reg;
    islpci_private *private_config = config;
    void *device = private_config->device_id;

    // received an interrupt request on a shared IRQ line
    // first check whether the device is in sleep mode
    if( reg = readl( device + ISL38XX_CTRL_STAT_REG),
        reg & ISL38XX_CTRL_STAT_SLEEPMODE )
        // device is in sleep mode, IRQ was generated by someone else
        return;
   
    // lock the interrupt handler
    spin_lock( &private_config->slock );
     
    // check whether there is any source of interrupt on the device
    reg = readl(device + ISL38XX_INT_IDENT_REG);

    // also check the contents of the Interrupt Enable Register, because this
    // will filter out interrupt sources from other devices on the same irq !
    reg &= readl(device + ISL38XX_INT_EN_REG);

    if (reg &= ISL38XX_INT_SOURCES, reg != 0)
    {
        // reset the request bits in the Identification register
        writel(reg, device + ISL38XX_INT_ACK_REG);

#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_FUNCTION_CALLS,
	        "IRQ: Identification register 0x%p 0x%x \n", device, reg);
#endif

        // check for each bit in the register separately
        if (reg & ISL38XX_INT_IDENT_UPDATE)
        {
#if VERBOSE > SHOW_ERROR_MESSAGES 
            // Queue has been updated
            DEBUG(SHOW_TRACING, "IRQ: Update flag \n");

            DEBUG(SHOW_QUEUE_INDEXES,
                "CB drv Qs: [%i][%i][%i][%i][%i][%i]\n",
                le32_to_cpu(private_config->control_block->driver_curr_frag[0]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[1]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[2]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[3]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[4]),
                le32_to_cpu(private_config->control_block->driver_curr_frag[5])
                );

            DEBUG(SHOW_QUEUE_INDEXES,
                "CB dev Qs: [%i][%i][%i][%i][%i][%i]\n",
                le32_to_cpu(private_config->control_block->device_curr_frag[0]),
                le32_to_cpu(private_config->control_block->device_curr_frag[1]),
                le32_to_cpu(private_config->control_block->device_curr_frag[2]),
                le32_to_cpu(private_config->control_block->device_curr_frag[3]),
                le32_to_cpu(private_config->control_block->device_curr_frag[4]),
                le32_to_cpu(private_config->control_block->device_curr_frag[5])
                );
#endif

            // device is in active state, update the powerstate flag
            private_config->powerstate = ISL38XX_PSM_ACTIVE_STATE;

            // check all three queues in priority order
            // call the PIMFOR receive function until the queue is empty
            while (isl38xx_in_queue(private_config->control_block,
                ISL38XX_CB_RX_MGMTQ) != 0)
            {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                DEBUG(SHOW_TRACING, "Received frame in Management Queue\n");
#endif
                islpci_mgt_receive(private_config);

                // call the pimfor transmit function for processing the next
                // management frame in the shadow queue
                islpci_mgt_transmit( private_config );
            }

            while (isl38xx_in_queue(private_config->control_block,
                ISL38XX_CB_RX_DATA_LQ) != 0)
            {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                DEBUG(SHOW_TRACING, "Received frame in Data Low Queue \n");
#endif
                islpci_eth_receive(private_config);
            }

			// check whether the data transmit queues were full
            if (private_config->data_low_tx_full)
            {
                // check whether the transmit is not full anymore
                if( ISL38XX_CB_TX_QSIZE - isl38xx_in_queue( private_config->control_block, 
		            ISL38XX_CB_TX_DATA_LQ ) >= ISL38XX_MIN_QTHRESHOLD )
                {
                    // nope, the driver is ready for more network frames
                    netif_wake_queue(private_config->my_module);

                    // reset the full flag
                    private_config->data_low_tx_full = 0;
                }
            }
        }

        if (reg & ISL38XX_INT_IDENT_INIT)
        {
            // Device has been initialized
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "IRQ: Init flag, device initialized \n");
#endif

            // perform card initlialization by sending objects
            islpci_mgt_initialize( private_config );
        }

        if (reg & ISL38XX_INT_IDENT_SLEEP)
        {
            // Device intends to move to powersave state
//#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_ANYTHING, "IRQ: Sleep flag \n");
//#endif
            isl38xx_handle_sleep_request( private_config->control_block,
                &private_config->powerstate,
                private_config->remapped_device_base );
        }

        if (reg & ISL38XX_INT_IDENT_WAKEUP)
        {
            // Device has been woken up to active state
//#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_ANYTHING, "IRQ: Wakeup flag \n");
//#endif
        
            isl38xx_handle_wakeup( private_config->control_block,
                &private_config->powerstate,
                private_config->remapped_device_base );
        }
    }

    // unlock the interrupt handler
    spin_unlock( &private_config->slock );

    return;
}


/******************************************************************************
    Network Interface Control & Statistical functions
******************************************************************************/
int islpci_open(struct net_device *nwdev)
{
#ifdef WDS_LINKS
  islpci_private *private_config = nwdev->priv;
  struct wds_priv *wdsp = private_config->wdsp;
#endif
//	unsigned long flags;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "islpci_open \n");
#endif

    // lock the driver code
    //driver_lock( &private_config->slock, &flags );
    
    MOD_INC_USE_COUNT;

#ifdef WDS_LINKS
    open_wds_links(wdsp);
#endif
    
    netif_start_queue(nwdev);
//      netif_mark_up( nwdev );

    // unlock the driver code
    //driver_unlock( &private_config->slock, &flags );

    return 0;
}

int islpci_close(struct net_device *nwdev)
{
//  islpci_private *private_config = nwdev->priv;
//	unsigned long flags;
#ifdef WDS_LINKS
  islpci_private *private_config = nwdev->priv;
  struct wds_priv *wdsp = private_config->wdsp;
#endif

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "islpci_close \n");
#endif

    // lock the driver code
    //driver_lock( &private_config->slock, &flags );

#ifdef WDS_LINKS
    close_wds_links(wdsp);
#endif
    netif_stop_queue(nwdev);
//      netif_mark_down( nwdev );

    MOD_DEC_USE_COUNT;

    // unlock the driver code
    //driver_unlock( &private_config->slock, &flags );

    return 0;
}

#if 0
int islpci_reset(islpci_private * private_config, char *firmware )
{
    isl38xx_control_block *control_block = private_config->control_block;
    queue_entry *entry;
//	unsigned long flags;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_reset \n");
#endif

    // lock the driver code
    //driver_lock( &private_config->slock, &flags );

    // stop the transmission of network frames to the driver
    netif_stop_queue(private_config->my_module);

    // disable all device interrupts
    writel(0, private_config->remapped_device_base + ISL38XX_INT_EN_REG);

    // flush all management queues
    while( isl38xx_in_queue(control_block, ISL38XX_CB_TX_MGMTQ) != 0 )
    {
        islpci_get_queue(private_config->remapped_device_base,
                &private_config->mgmt_tx_shadowq, &entry);
        islpci_put_queue(private_config->remapped_device_base,
                &private_config->mgmt_tx_freeq, entry);

        // decrement the real management index
        private_config->index_mgmt_tx--;
    }

//    while( isl38xx_in_queue(control_block, ISL38XX_CB_RX_MGMTQ) != 0 )
//    {
//        islpci_get_queue(private_config->remapped_device_base,
//                &private_config->mgmt_rx_shadowq, &entry);
//        islpci_put_queue(private_config->remapped_device_base,
//                &private_config->mgmt_rx_freeq, entry);

        // decrement the real management index
//        private_config->index_mgmt_rx--;
//    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Firmware file: %s \n", firmware );
#endif
	
    if (isl38xx_upload_firmware( firmware, private_config->remapped_device_base,
        private_config->device_host_address ) == -1)
    {
        // error uploading the firmware
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: could not upload the firmware \n" );

        // unlock the driver code
        //driver_unlock( &private_config->slock, &flags );
        return -1;
    }
  
    // unlock the driver code
    //driver_unlock( &private_config->slock, &flags );

    return 0;
}
#endif	/* 0 */

void islpci_set_multicast_list(struct net_device *dev)
{
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "islpci_set_multicast_list \n");
#endif
}

struct net_device_stats *islpci_statistics(struct net_device *nwdev)
{
    islpci_private *private_config = nwdev->priv;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "islpci_statistics \n");
#endif

    return &private_config->statistics;
}


/******************************************************************************
    Network device configuration functions
******************************************************************************/
struct net_device * islpci_probe(struct net_device *nwdev, struct pci_dev *pci_device, long ioaddr, int irq)
{
    void *mapped_address;
    void *driver_address;
    void *queue_base;
    dma_addr_t device_address;
    long counter;
    long offset;
    queue_entry *pointerq;
    islpci_private *private_config;
    isl38xx_control_block *control_block;
    struct sk_buff *skb;
    
    // initially setup an ethernet device
    nwdev = init_etherdev(nwdev, 0);

    // setup the structure members
    nwdev->base_addr = ioaddr;
    nwdev->irq = irq;

    // initialize the function pointers
    nwdev->open = &islpci_open;
    nwdev->stop = &islpci_close;
    nwdev->get_stats = &islpci_statistics;
#ifdef WIRELESS_IOCTLS
    nwdev->do_ioctl = &isl_ioctl;
#endif

    nwdev->hard_start_xmit = &islpci_eth_transmit;

//    nwdev->set_multicast_list = &set_rx_mode;

#ifdef HAVE_TX_TIMEOUT
    nwdev->watchdog_timeo = ISLPCI_TX_TIMEOUT;
    nwdev->tx_timeout = &islpci_eth_tx_timeout;
#endif

    // remap the PCI device base address to accessable
    if (mapped_address = ioremap(nwdev->base_addr, ISL38XX_PCI_MEM_SIZE),
        mapped_address == NULL)
    {
        // error in remapping the PCI device memory address range
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: PCI memory remapping failed \n");
        return NULL;
    }

    // save the start and end address of the PCI memory area
    nwdev->mem_start = (unsigned long) mapped_address;
    nwdev->mem_end = (unsigned long) mapped_address + ISL38XX_PCI_MEM_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "PCI Memory remapped to 0x%p \n", mapped_address);
#endif

    // allocate a block of memory on the host for the control block
    if (driver_address = pci_alloc_consistent(pci_device,
        ISL38XX_HOST_MEM_BLOCK, &device_address), driver_address == NULL)
    {
        // error allocating the bblock of PCI memory
        DEBUG(SHOW_ERROR_MESSAGES,
                "ERROR: Could not allocate DMA accessable memory \n");
        iounmap(mapped_address);
        return NULL;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Device memory block at 0x%x \n", device_address);
    DEBUG(SHOW_TRACING, "Driver memory block at 0x%p \n", driver_address);
    DEBUG(SHOW_TRACING, "Memory size 0x%x bytes\n", ISL38XX_HOST_MEM_BLOCK);
#endif

    // add a private device structure to the network device for private
    // definitions needed by the driver
    if ( ( private_config = kmalloc(sizeof(islpci_private), GFP_KERNEL | GFP_DMA) ) == NULL )
    {
        // error allocating the DMA accessable memory area
        DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Could not allocate additional "
            "memory \n");
        pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK, driver_address,
            device_address);
        iounmap(mapped_address);
        return NULL;
    }

    memset(private_config, 0, sizeof(islpci_private));
    nwdev->priv = private_config;
//    private_config->next_module = root_islpci_device;
//    root_islpci_device = nwdev;
    private_config->my_module = nwdev;
    
    // assign the Control Block pointer to the allocated area
    control_block = (isl38xx_control_block *) driver_address;
    private_config->control_block = control_block;

    // clear the indexes in the frame pointer
    for (counter = 0; counter < ISL38XX_CB_QCOUNT; counter++)
    {
        control_block->driver_curr_frag[counter] = cpu_to_le32(0);
        control_block->device_curr_frag[counter] = cpu_to_le32(0);
    }

    // save the device configurations in the device private configuration
    private_config->remapped_device_base = mapped_address;
    private_config->device_host_address = device_address;
    private_config->device_psm_buffer = device_address + CONTROL_BLOCK_SIZE;
    private_config->driver_mem_address = driver_address;

#if WDS_LINKS
    private_config->wdsp = kmalloc(sizeof(struct wds_priv), GFP_KERNEL );
    memset(private_config->wdsp, 0, sizeof(struct wds_priv));
#endif    
    
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "PSM Buffer at 0x%p size %i\n",
        (void *) private_config->device_psm_buffer, PSM_BUFFER_SIZE );
#endif

    // configure the allocated area for PSM buffer and queueing mechanism
    // with the following layout
    // offset 0x0000:       Control block,  size = 1024 bytes
    //      ,  ,  0x0400:   PSM Buffer, size = entries * 0x0600
    //      ,  ,  0x????:   Queue memory

    // set the queue base address behind the control block
    queue_base = driver_address + CONTROL_BLOCK_SIZE + PSM_BUFFER_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Queue memory base at 0x%p \n", queue_base);
#endif

    // initialize all the queues in the private configuration structure
    islpci_init_queue(&private_config->mgmt_tx_freeq);
    islpci_init_queue(&private_config->mgmt_rx_freeq);

    islpci_init_queue(&private_config->mgmt_tx_shadowq);
    islpci_init_queue(&private_config->mgmt_rx_shadowq);
    islpci_init_queue(&private_config->ioctl_rx_queue);
    islpci_init_queue(&private_config->trap_rx_queue);
    islpci_init_queue(&private_config->pimfor_rx_queue);

    // determine the offset for each queue entry configuration
    // allign on 16 byte boundary
    offset = sizeof(queue_entry) / 16;
    offset *= 16;
    offset += ((sizeof(queue_entry) % 16) != 0 ? 16 : 0);

    // configure all management queue(s) entries and add them to the freeqs
    for (counter = 0; counter < MGMT_TX_FRAME_COUNT; counter++)
    {
        // configure the queue entry at the beginning of the memory block
        pointerq = (queue_entry *) queue_base;
        pointerq->host_address = (u32) queue_base + offset;
        pointerq->size = MGMT_FRAME_SIZE - offset;
        pointerq->fragments = 1;

        // also translate the host address for the device
        pointerq->dev_address = (u32) virt_to_bus((void *)
                pointerq->host_address);

        // add the configured entry to the management tx root queue
        islpci_put_queue(mapped_address, &private_config->mgmt_tx_freeq, pointerq);

        // increment the queue base address pointer to the next entry
        queue_base += MGMT_FRAME_SIZE;

        // perform a check on the destination address in PCI memory
        if (queue_base > driver_address + ISL38XX_HOST_MEM_BLOCK)
        {
            // Oops, end address of fragment will be out of bound
            DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Queue memory out of bound, "
                "addres 0x%p, enlarge host memory \n", queue_base);
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return NULL;
        }
    }

    for (counter = 0; counter < MGMT_RX_FRAME_COUNT; counter++)
    {
        // configure the queue entry at the beginning of the memory block
        pointerq = (queue_entry *) queue_base;
        pointerq->host_address = (u32) queue_base + offset;
        pointerq->size = MGMT_FRAME_SIZE - offset;
        pointerq->fragments = 1;

        // also translate the host address for the device
        pointerq->dev_address = (u32) virt_to_bus((void *)
            pointerq->host_address);

        // add the configured entry to the management rx root queue
        islpci_put_queue(mapped_address, &private_config->mgmt_rx_freeq, pointerq);

        // increment the queue base address pointer to the next entry
        queue_base += MGMT_FRAME_SIZE;

        // perform a check on the destination address in PCI memory
        if (queue_base > driver_address + ISL38XX_HOST_MEM_BLOCK)
        {
            // Oops, end address of fragment will be out of bound
            DEBUG(SHOW_ERROR_MESSAGES, "ERROR: Queue memory out of bound, "
                "addres 0x%p, enlarge host memory \n", queue_base);
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return NULL;
        }
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Queue memory end at 0x%p \n", queue_base);
#endif

    // Joho, ready setting up the queueing stuff, now fill the shadow rx queues
    // of both the management and data queues
    for (counter = 0; counter < ISL38XX_CB_MGMT_QSIZE; counter++)
    {
        // get an entry from the freeq and put it in the shadow queue
        if (islpci_get_queue(mapped_address, &private_config->mgmt_rx_freeq,
            &pointerq))
        {
            // Oops, no more entries in the freeq ?
            DEBUG(SHOW_ERROR_MESSAGES, "Error: No more entries in freeq\n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return NULL;
        }
        else
        {
            // use the entry for the device interface management rx queue
            // these should always be inline !
            control_block->rx_data_mgmt[counter].address =
                cpu_to_le32(pointerq->dev_address);

            // put the entry got from the freeq in the shadow queue
            islpci_put_queue(mapped_address, &private_config->mgmt_rx_shadowq,
                pointerq);
        }
    }

    for (counter = 0; counter < ISL38XX_CB_RX_QSIZE; counter++)
    {
        // allocate an sk_buff for received data frames storage
		// each frame on receive size consists of 1 fragment
	    // include any required allignment operations
        if (skb = dev_alloc_skb(MAX_FRAGMENT_SIZE+2), skb == NULL)
        {
            // error allocating an sk_buff structure elements
            DEBUG(SHOW_ERROR_MESSAGES, "Error allocating skb \n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return NULL;
        }

		// add the new allocated sk_buff to the buffer array
		private_config->data_low_rx[counter] = skb;

		// map the allocated skb data area to pci
        private_config->pci_map_rx_address[counter] = pci_map_single(
        	private_config->pci_device, (void *) skb->data, skb->len,
            PCI_DMA_FROMDEVICE );
        if( private_config->pci_map_rx_address[counter] == (dma_addr_t) NULL )
        {
            // error mapping the buffer to device accessable memory address
            DEBUG(SHOW_ERROR_MESSAGES, "Error mapping DMA address\n");
            pci_free_consistent(pci_device, ISL38XX_HOST_MEM_BLOCK,
                driver_address, device_address);
            iounmap(mapped_address);
            return NULL;
        }

		// update the fragment address values in the contro block rx queue
        control_block->rx_data_low[counter].address = cpu_to_le32( (u32) 
		 	private_config->pci_map_rx_address[counter] );
    }

    // since the receive queues are filled with empty fragments, now we can
    // set the corresponding indexes in the Control Block
    control_block->driver_curr_frag[ISL38XX_CB_RX_DATA_LQ] =
        cpu_to_le32(ISL38XX_CB_RX_QSIZE);
    control_block->driver_curr_frag[ISL38XX_CB_RX_MGMTQ] =
        cpu_to_le32(ISL38XX_CB_MGMT_QSIZE);

    // reset the real index registers and full flags
    private_config->index_mgmt_rx = 0;
	private_config->index_mgmt_tx = 0;
    private_config->free_data_rx = 0;
    private_config->free_data_tx = 0;
    private_config->data_low_tx_full = 0;

    // reset the queue read locks, process wait counter
    private_config->ioctl_queue_lock = 0;
    private_config->processes_in_wait = 0;

    // set the power save state
    private_config->powerstate = ISL38XX_PSM_ACTIVE_STATE;
    private_config->resetdevice = 0;

    return nwdev;
}


