/*
 * $Id: brecis_flashmap.c,v 1.6 2003/02/06 22:30:09 swahl Exp $
 *
 * adapted from cstm_mips_ixx.c
 *
 * Mapping of a custom board with both AMD CFI and JEDEC flash in partitions.
 * Config with both CFI and JEDEC device support.  So far, only 4M CFI parts
 * tested.
 *
 * Basically physmap.c with the addition of partitions and 
 * an array of mapping info to accomodate more than one flash type per board.
 *
 * Copyright 2000 MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>
#include <asm/brecis/BrecisSysRegs.h>

#define  TYPE_SYSTEM_EQ_FLASHBOOT   //add by hook 2003/03/13

static __u8 b_flash_read8(struct map_info *map, unsigned long ofs)
{
	return __raw_readb(map->map_priv_1 + ofs);
}

static __u16 b_flash_read16(struct map_info *map, unsigned long ofs)
{
	return __raw_readw(map->map_priv_1 + ofs);
}

static __u32 b_flash_read32(struct map_info *map, unsigned long ofs)
{
	return __raw_readl(map->map_priv_1 + ofs);
}

static void b_flash_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy_fromio(to, map->map_priv_1 + from, len);
}

static void b_flash_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	__raw_writeb(d, map->map_priv_1 + adr);
	mb();
}

static void b_flash_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	__raw_writew(d, map->map_priv_1 + adr);
	mb();
}

static void b_flash_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	__raw_writel(d, map->map_priv_1 + adr);
	mb();
}

static void b_flash_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy_toio(map->map_priv_1 + to, from, len);
}

static const struct map_info basic_b_flash_map = {
	name: "Brecis flash mapper",
	size: 0,
	buswidth: 0,
	read8: b_flash_read8,
	read16: b_flash_read16,
	read32: b_flash_read32,
	copy_from: b_flash_copy_from,
	write8: b_flash_write8,
	write16: b_flash_write16,
	write32: b_flash_write32,
	copy_to: b_flash_copy_to,
};

/* board and partition description */

#define MAX_PHYSMAP_PARTITIONS    8
struct b_flash_info {
	char *name;
	unsigned long window_addr;
	unsigned long window_size;
	int buswidth;
	int num_partitions;
};

#define PHYSMAP_NUMBER  1  // number of board desc structs needed,
				// one per contiguous flash type 

static const struct b_flash_info b_flash_board_desc[PHYSMAP_NUMBER] = 
{
	{   // Brecis EVM configuration
		"flash",         // name
		0xbf800000,      // window_addr
		0x00400000,      // window_size
		1,               // buswidth
#ifndef    TYPE_SYSTEM_EQ_FLASHBOOT
		4,               // num_partitions
#else  /* !TYPE_SYSTEM_EQ_FLASHBOOT */
/*		5,               // num_partitions */
		1,               // num_partitions
#endif /* !TYPE_SYSTEM_EQ_FLASHBOOT */
	}

};

#ifndef    TYPE_SYSTEM_EQ_FLASHBOOT
static struct mtd_partition b_flash_partitions[PHYSMAP_NUMBER][MAX_PHYSMAP_PARTITIONS] = {
	{   // Brecis EVM configuration
		{
			name: "main partition",
			size: 0x370000, 
			offset: 0x80000,
		},
		{
			name: "pmon storage",
			size: 0x70000, 
			offset: 0,
		},
		{
			name: "pmon start script",
			size: 0x10000, 
			offset: 0x70000,
		},
		{
			name: "region with copyprotected blocks",
			size: 0x10000,
			offset: 0x3f0000,
		},
	},
};
#else  /* !TYPE_SYSTEM_EQ_FLASHBOOT */
static struct mtd_partition b_flash_partitions[PHYSMAP_NUMBER][MAX_PHYSMAP_PARTITIONS] = {
	{   // Brecis EVM configuration
/* -----start of add by hook 2003/03/21 */
	    {
			 name: "main partition",
 		     size: 0x400000,
		     offset: 0x0,
		   },
/*-----start of add by hook 2003/03/21    */
		
/* -----start of mark by hook 2003/03/21			 
		{
			name: "main partition",
			size: 0x0f0000, 
			offset: 0x300000,
		},
		{
			name: "bbload storage",
			size: 0x10000, 
			offset: 0,
		},
		{
			name: "bbload start script",
			size: 0x10000, 
			offset: 0x10000,
		},
		{
			name: "load image",
			size: 0x2e0000, 
			offset: 0x20000,
		},
		{
			name: "region with copyprotected blocks",
			size: 0x10000,
			offset: 0x3f0000,
		},
		-------- end of mark by hook 2003/03/21     */
	},
};
#endif /* !TYPE_SYSTEM_EQ_FLASHBOOT */

static struct map_info b_flash_map[PHYSMAP_NUMBER];

int __init init_b_flash(void)
{
	int i;
	int jedec;
	struct mtd_info *mymtd;
	struct mtd_partition *parts;
	slmRegs *sreg = (slmRegs *) SREG_BASE ;
	unsigned int id ;

	/* If ELB is disabled by "ful-mux" mode, we can't get at flash */

	id = sreg->dev_id ;
	if ( (id &DEV_ID_SINGLE_PC)
	     && (sreg->single_pc_enable & SINGLE_PCCARD) )
	{
		printk(KERN_NOTICE "Single PC Card mode: no flash access\n") ;
		return -ENXIO ;
	}

	/* Initialize mapping */
	for (i=0;i<PHYSMAP_NUMBER;i++) {
		printk(KERN_NOTICE
		       "b_flash flash device: %lx at %lx\n",
		       b_flash_board_desc[i].window_size,
		       b_flash_board_desc[i].window_addr);
                memcpy((char *)&b_flash_map[i],
		       (char *)&basic_b_flash_map,
		       sizeof(struct map_info));
		b_flash_map[i].map_priv_1 = (unsigned long)
			ioremap(b_flash_board_desc[i].window_addr,
				b_flash_board_desc[i].window_size);
		if (!b_flash_map[i].map_priv_1) {
			printk(KERN_WARNING "Failed to ioremap\n");
			return -EIO;
	        }
		b_flash_map[i].name =
			b_flash_board_desc[i].name;
		b_flash_map[i].size =
			b_flash_board_desc[i].window_size;
		b_flash_map[i].buswidth =
			b_flash_board_desc[i].buswidth;
		//printk(KERN_NOTICE "b_flash: ioremap is %x\n",(unsigned int)(b_flash_map[i].map_priv_1));
	}

	for (i=0;i<PHYSMAP_NUMBER;i++) {
                parts = &b_flash_partitions[i][0];
		jedec = 0;
		mymtd = (struct mtd_info *)
			do_map_probe("cfi_probe", &b_flash_map[i]);
		//printk(KERN_NOTICE "phymap %d cfi_probe: mymtd is %x\n",i,(unsigned int)mymtd);
#if 0
		if (!mymtd) {
			jedec = 1;
			mymtd = (struct mtd_info *)
				do_map_probe("jedec", &b_flash_map[i]);
		        printk(KERN_NOTICE
			       "b_flash %d jedec: mymtd is %x\n",
			       i,(unsigned int)mymtd);
		}
#endif
		if (mymtd) {
			mymtd->module = THIS_MODULE;

	                b_flash_map[i].map_priv_2 =
				(unsigned long)mymtd;
		        add_mtd_partitions(mymtd, parts,
					   b_flash_board_desc[i].num_partitions);
		}
		else
	           return -ENXIO;
	}
	return 0;
}

static void __exit cleanup_b_flash(void)
{
	int i;
        struct mtd_info *mymtd;

	for (i=0;i<PHYSMAP_NUMBER;i++) {
	        mymtd = (struct mtd_info *)b_flash_map[i].map_priv_2;
		if (mymtd) {
			del_mtd_partitions(mymtd);
			map_destroy(mymtd);
		}
		if (b_flash_map[i].map_priv_1) {
			iounmap((void *)b_flash_map[i].map_priv_1);
			b_flash_map[i].map_priv_1 = 0;
		}
	}
}

module_init(init_b_flash);
module_exit(cleanup_b_flash);


MODULE_LICENSE("GPL");
MODULE_AUTHOR("BRECIS");
MODULE_DESCRIPTION("MTD map driver for BRECIS boards");
