/*
  $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/brecis/lg79063/SLAClg79063Drv.c,v 1.29 2003/03/07 06:48:43 awarner Exp $
*/
/****************************************************************************
 *
 *  Driver for Legerity QSLAC Am 79Q063/061 and SLIC Am 79R79 on BRECIS MSP EVM boards
 *
 *****************************************************************************/

/******************************************************************/
/* Copyright (c) 2001 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/telephony.h>
#include <linux/phonedev.h>
#include "mspCommon.h"
#include <asm/brecis/BrecisSysRegs.h>
#include "msp.h"
#include "SLAClg79063Drv.h"
#include "slacpri.h"
#include <brecis/NanoDelay.h>

int loopback=0;	/* module parameter to turn on interface loopback (ILB) */
int tsoffset=0; /* module parameter to offset time slots from 0 */
int clock=-1;	/* module parameter to set non-default clock frequency */

MODULE_PARM(loopback, "i");
MODULE_PARM(tsoffset, "i");
MODULE_PARM(clock, "i");

#ifdef DEBUG
#define DBG_SLAC(a1, a2...)	printk("SLAC " a1, ##a2)
#else
#define DBG_SLAC(a...)
#endif

#define QSLAC_IRQ 2 /*QSLAC is connected to External interrupt 2*/

static sSLACInfo   *pSLACInfo= NULL;
static int         slac_max_module_num =0;
static slac_companding_law slac_companding_type = SLAC_MU_LAW;
static int slac_hw_init_done =0;
static int debounce_timer_running =0;
static struct timer_list debounce_timer;

/**********************************************************************
 ** Default SLAC coef. table
 ***********************************************************************/
static slac_coeff_type Am79063DefCoeff = {
	/*AISN: */0x0E,
	{/*GX[2]:*/0x2A, 0x40},
	{/*GR[2]:*/0xA4, 0x21},
	{/*Z[15]:*/0x3C, 0xA2, 0x43, 0xB7, 0x27, 0xAB, 0x32, 0x4C, 0xBB, 0x24, 0xA2, 0x63, 0x97, 0xA8, 0xF0},
	{/*B[14]:*/0xBA, 0xC3, 0xA9, 0xBB, 0x9A, 0xAB, 0x2A, 0xE3, 0xA7, 0xC3, 0x77, 0x97, 0xC9, 0x70},
	{/*X[12]:*/0x01, 0x11, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90},
	{/*R[14]:*/0xDC, 0x01, 0x01, 0x11, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90, 0x01, 0x90},
	{/*B2[2]:*/0x2D, 0x01},
	/*AX:   */0x00,
	/*AR:   */0x00
};

static U8 SlicCtrlByte[SLAC_TYPE_MAX][SLIC_STATE_MAX] =	{
	{
		SLIC_OPEN_CIRCUIT_MODE | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_OFF,
		SLIC_RINGING_MODE      | SLIC_RING_ON  | SLIC_E1_LOOP_START | SLIC_B2EN_OFF,
		SLIC_ACTIVE_MODE       | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_ON,
		SLIC_OHT_MODE          | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_OFF,
#ifndef   ENABLE_79Q061_NOT_79Q063
		SLIC_TIP_OPEN_MODE     | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_OFF,
		SLIC_STANDBY_MODE      | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_ON,
		SLIC_APR_MODE          | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_ON,
		SLIC_OHT_PR_MODE       | SLIC_RING_OFF | SLIC_E1_LOOP_START | SLIC_B2EN_ON,
#endif /* ENABLE_79Q061_NOT_79Q063 */
	},
#ifndef   ENABLE_79Q061_NOT_79Q063
	{
		SLIC_OPEN_CIRCUIT_MODE | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_OFF,
		SLIC_RINGING_MODE      | SLIC_RING_ON  | SLIC_E1_GROUND_START | SLIC_B2EN_OFF,
		SLIC_ACTIVE_MODE       | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_ON,
		SLIC_OHT_MODE          | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_OFF,
		SLIC_TIP_OPEN_MODE     | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_OFF,
		SLIC_STANDBY_MODE      | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_ON,
		SLIC_APR_MODE          | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_ON,
		SLIC_OHT_PR_MODE       | SLIC_RING_OFF | SLIC_E1_GROUND_START | SLIC_B2EN_ON,
	},
#endif /* ENABLE_79Q061_NOT_79Q063 */
};

/*serialize access to the SPI MPI bus*/
static spinlock_t spi_mpi_lock = SPIN_LOCK_UNLOCKED;

static void slac_reset_peri_register(void);
static U32 sys_set_pcm_highway_clk(U32 clkToSet,U32 clkRate);
static void slac_cfg_ringer_clk( U32 clkFreq);

static int slac_init(slac_info *pSlacConfigInfo,int NumOfSlac);
static int slac_fasync(int fd, struct file *file_p, int mode);
static void slac_fxs_init(U8 ChipNo,U8 *pTxTSMap,U8 *pRxTSMap);
#ifndef   CONFIG_BRECIS_FPGA
static U32 sys_get_clk_div(int Out);
#endif /* CONFIG_BRECIS_FPGA */
#ifndef   CONFIG_BRECIS_FPGA
#ifndef   DEBUG_REMOVE_ISR
static void slac_isr(int irq,void* dev_id,struct pt_regs *regs );
#endif /* DEBUG_REMOVE_ISR */
#endif /* CONFIG_BRECIS_FPGA */
static void slac_timeout(unsigned long ptr);
static void slac_add_timer(void);

static U8 QSLACRdCMD_1(U8 ChipNo, U8 Command);


#ifdef    CONFIG_BRECIS_FPGA
//#define QSLAC_CS_BIT  ( 1<<8) /* GPIO 8 -- may be different for real POLO EVM */
#define QSLAC_CS_BIT  ( 1<<1) /* GPIO 1 -- may be different for real POLO EVM */
#else  /* CONFIG_BRECIS_FPGA */
/* NOTE: MPI CS is negative true! */
#define MPI_CS_ALL_DEASSERTED (SEL_MPI | SPI_MPI_CS0 | SPI_MPI_CS1 | \
	SPI_MPI_CS2 | SPI_MPI_CS3 | SPI_MPI_CS4 | SPI_MPI_CS5 |      \
	SPI_MPI_CS6 | SPI_MPI_CS7 | SPI_MPI_CS8 | SPI_MPI_CS9)
#endif /* CONFIG_BRECIS_FPGA */

static inline void slac_assert_chipselect( unsigned int chip)
{
#ifdef    CONFIG_BRECIS_FPGA
	*SPI_MPI_CS_REG &= (~SEL_SPI);	/* make sure we are in MPI mode */
	*GPIO_DATA_REG &= ~QSLAC_CS_BIT; /* chip select is negative true */
#else  /* CONFIG_BRECIS_FPGA */
	/*
	 * *assume* individual chip selects are in order, starting
	 *  with bit 0, and negative true (xor to turn bit OFF)
	 */
	*SPI_MPI_CS_REG = MPI_CS_ALL_DEASSERTED ^ (1 << chip);
#endif /* CONFIG_BRECIS_FPGA */
}

static inline void slac_deassert_chipselect(void)
{
#ifdef CONFIG_BRECIS_FPGA
	*GPIO_DATA_REG |= QSLAC_CS_BIT; /* chip select is negative true */
#else  /* CONFIG_BRECIS_FPGA */
	*SPI_MPI_CS_REG = MPI_CS_ALL_DEASSERTED;
#endif /* CONFIG_BRECIS_FPGA */
}

static inline int slac_locate_circuit(U32 cookie,U32 *chipNo,U32 *channel)
{
	*chipNo = (cookie & 0x7F00) >> 8;
	*channel = (cookie & 0x00FF);
	if (( *channel < CHANNELS_PER_QSLAC) && (*chipNo < SLAC_MAX_NUM_OF_CHIP))
	{
		return 0;
	}
	*channel = CHANNELS_PER_QSLAC;
	*chipNo = SLAC_MAX_NUM_OF_CHIP;
	return -ENODEV;		
}

static inline U32 slac_make_cookie(U32 chip,U32 channel)
{
	if (( channel >= CHANNELS_PER_QSLAC) && (chip >= SLAC_MAX_NUM_OF_CHIP))
		DBG_SLAC(KERN_WARNING "slac_make_cookie got invalid chipNo %d, channelNo%d\n",chip,channel);
	
	return( (chip <<8) | 0x8000 | channel);
}


static int slac_hw_init(void)
{
	int rc=test_and_set_bit(0, &slac_hw_init_done);
	if( rc )
	{
		DBG_SLAC(KERN_WARNING "slac_hw_init:Already initialized 0x%X\n",rc);
		return(-EALREADY);
	}
	
	/*Take peripheral block out of reset */
	slac_reset_peri_register();
#ifdef  CONFIG_BRECIS_FPGA
#if 0
	/*
	 * GPIO 8 is used as chip select for the SLAC
	 *  configure it as output
	 * Don't depend on any default values
	 */
	*GPIO_CFG2_REG &= (~(0xf)); /* turn off all GPIO 8 config bits */
	*GPIO_CFG2_REG |= (1 << 3); /* make direction be output, used data reg 8 */
	*GPIO_OD_CFG_REG &= (~(1 << 8)); /* active output drive */
#else
	/*
	 * GPIO 1 is used as chip select for the SLAC
	 *  configure it as output
	 * Don't depend on any default values
	 */
	*GPIO_CFG1_REG &= (~(0xf << 4)); /* turn off all GPIO 1 config bits */
	*GPIO_CFG1_REG |= ((1 << 3) << 4); /* make direction be output, used data reg 8 */
	*GPIO_OD_CFG_REG &= (~(1 << 1)); /* active output drive */
#endif
	/*
	 * The GPIO pins must be specifically configured for MPI
	 */
	/* turn off all the associated config bits GPIO 10 and 11 */
	*GPIO_CFG2_REG &= (~((1 << 15) | (1 << 12) | (1 << 11) | (1 << 8)));
	/* turn on the required config bits
	 * (1 << 15) MPI clock (out)
	 * (1 << 12) select MPI clock (as opposed to GPIO)
	 * (1 << 11) MPI data (purportedly a don't care, but make an input)
	 * (1 <<  8) select MPI data (as opposed to GPIO)
	*/
	*GPIO_CFG2_REG |= ((1 << 15) | (1 << 12) | (0 << 11) | (1 << 8));
#endif
	
	/*configure TDM clocks */
	{
		U32 clk_rate =   ( (clock >0)? (clock):(1544) ) * 1000 ;
		sys_set_pcm_highway_clk(MSP_PCM_HIGHWAY_ZERO, clk_rate );
#ifndef   CONFIG_BRECIS_FPGA
		sys_set_pcm_highway_clk(MSP_PCM_HIGHWAY_ONE,  clk_rate );
#endif /* CONFIG_BRECIS_FPGA */
		printk(KERN_INFO "sys_set_pcm_highway_clk %d\n", clk_rate);
	}

	/* Configure PIT to generate 20 Hz ringer freq on GPIO */
	slac_cfg_ringer_clk(SLIC_RINGER_FREQ);	

	/*Initialize the SLAC data structures*/
	rc = slac_init(NULL,1);
	if(rc)
	{
		return (rc);
	}
#ifndef   CONFIG_BRECIS_FPGA
#ifndef   DEBUG_REMOVE_ISR
	/*Install handler for SLAC interrupt (external interrupt 2)
	 pass the ChipNo as device id*/
	rc =  request_irq(QSLAC_IRQ,slac_isr,0,"QSLAC",(void *)0);
	if(rc)
	{
		DBG_SLAC(KERN_WARNING "slac_hw_init:couldn't get IRQ %d, Error: %d\n", QSLAC_IRQ, rc);
	}
	else
	{
		DBG_SLAC(KERN_WARNING "slac_hw_init:Successfully completed SLAC hardware initialization\n");
	}
#endif /* DEBUG_REMOVE_ISR */
#endif /* CONFIG_BRECIS_FPGA */
	return(rc);
}

/***************************************************************
SLAC Module Init Function
 - Initializes hardware needed by QSLAC
 - Installs ISR for SLAC interrupt
 - registers QSLAC's as phone devices
***************************************************************/
static int __init slac_module_init( void)
{
	if(slac_hw_init())
	{
		DBG_SLAC(KERN_WARNING "slac_hw_init:Already initialized\n");
	}
	printk(KERN_INFO "SLAC: "
		"using IRQ %d (" __DATE__ " " __TIME__ " %s)\n",
		QSLAC_IRQ, ""
#ifdef    ENABLE_79Q061_NOT_79Q063
		" 79Q061"
#endif /* ENABLE_79Q061_NOT_79Q063 */
#ifdef    DONT_REQUIRE_TELEPHONY_API
		" NO.TELE.API"
#endif /* DONT_REQUIRE_TELEPHONY_API */
#ifdef    CONFIG_BRECIS_FPGA
		" FPGA"
#endif /* CONFIG_BRECIS_FPGA */
		);
	return (0);
}

static void __exit slac_module_exit( void)
{
	int i,j;

	free_irq(QSLAC_IRQ,NULL );

	for (i=0; i<slac_max_module_num; i++)
	{
		for (j=0; j<pSLACInfo[i].ChansInModule; j++)
		{
#ifndef    DONT_REQUIRE_TELEPHONY_API
			phone_unregister_device(&pSLACInfo[i].PhoneDevice[j]);
#endif  /* DONT_REQUIRE_TELEPHONY_API */
		}
	}
	if(pSLACInfo)
	{
		kfree(pSLACInfo);
	}

	del_timer(&debounce_timer);
	pSLACInfo = NULL;

}

void
slic_xopen(U8 chipNo, U8 chanNo)
{
	/*
	 * Don't call me with bad values !
	 */
	pSLACInfo[chipNo].SLICEnable[chanNo] = TRUE;
	pSLACInfo[chipNo].SLICCurrentStatus[chanNo] = ((pSLACInfo[chipNo].SLACLiveStatus) >> 2*chanNo) & 0x01;
}

/**********************************************************************
 **File Operations
 ***********************************************************************/
static int slac_open(struct phone_device *PhoneDev, struct file *file_p)
{
	int chipNo, chanNo;

	file_p->private_data = (void *)PhoneDev->board;

	if (slac_locate_circuit(PhoneDev->board, &chipNo, &chanNo))
	{
		return (-ENODEV);
	}

	pSLACInfo[chipNo].SLICEnable[chanNo] = TRUE;
	pSLACInfo[chipNo].SLICCurrentStatus[chanNo] = ((pSLACInfo[chipNo].SLACLiveStatus) >> 2*chanNo) & 0x01;

	/*Maually increment the use count as kernel does not know
	  about module usage since we are registered with phonedev*/
	MOD_INC_USE_COUNT;	

	return (0);
}


static int slac_release(struct inode *inode, struct file *file_p)
{
	int chipNo, chanNo;
	
	if (slac_locate_circuit((int)file_p->private_data, &chipNo, &chanNo))
	{
		return (-ENODEV);
	}
	/*Set the port to active state*/
	set_slic_state( chipNo,  chanNo, SLIC_ACTIVE);
	
	pSLACInfo[chipNo].SLICEnable[chanNo] = FALSE;
	pSLACInfo[chipNo].SLICCurrentStatus[chanNo] = 0x01;
	
	/*Remove from the list of async notification */
	slac_fasync(-1,file_p,0);	
	
	file_p->private_data = (void *)0;
	
	/*Maually decrement the use count as kernel does not know
	  about module usage since we are registered with phonedev*/
	MOD_DEC_USE_COUNT;
	
	return(0);
}


static int slac_ioctl(struct inode *inode, struct file *file_p, U32 cmd, unsigned long arg)
{
	int returnval=0;
	int chipNo, chanNo;
	
	if (slac_locate_circuit((int) file_p->private_data, &chipNo, &chanNo))
	{
		return (-ENODEV);
	}

	switch (cmd)
	{
	case PHONE_RING_START:
		/*This passes CID value but we don't do
		  that yet so just turn on the ring for now*/
	case PHONE_RING:
		set_slic_state(chipNo,chanNo,SLIC_RINGING);		
		break;
	case PHONE_RING_STOP:
		set_slic_state(chipNo,chanNo,SLIC_ACTIVE);
		break;
	case PHONE_HOOKSTATE:
		returnval = ( (pSLACInfo[chipNo].SLICCurrentStatus[chanNo] ^ 0x1) << 1 );
		break;
	default:
		DBG_SLAC(KERN_WARNING "SLAC: Received unsupported IOCTL %d on SLAC %d Channel: %d\n",cmd,chipNo,chanNo);
		returnval = -EOPNOTSUPP;
		break;
	}
	
	return (returnval);
}

static int slac_fasync(int fd, struct file *file_p, int mode)
{
	int chipNo, chanNo;
	
	if (slac_locate_circuit( (int)file_p->private_data, &chipNo, &chanNo))
	{
		return (-ENODEV);
	}
	return fasync_helper(fd, file_p, mode,	&pSLACInfo[chipNo].async_queue[chanNo]);
}

static unsigned int slac_poll(struct file *file_p, poll_table * wait)
{
	int chipNo, chanNo;
	unsigned int rc=0;
	
	if (slac_locate_circuit( (int)file_p->private_data, &chipNo, &chanNo))
	{
		return (-ENODEV);
	}
	poll_wait(file_p, &(pSLACInfo[chipNo].poll_q[chanNo]), wait);
	if(pSLACInfo[chipNo].SLICExceptionFlag[chanNo])
	{
		pSLACInfo[chipNo].SLICExceptionFlag[chanNo]=0;
		rc |= POLLPRI;	
	}

	return rc;	
}

static struct file_operations slac_fops =
{
	owner:		THIS_MODULE,
	poll:       slac_poll,
	ioctl:      slac_ioctl,
	release:    slac_release,
	fasync:    	slac_fasync
};

/**********************************************************************
 ** SlacInit - Initializes SLAC related data structures
 ***********************************************************************/
static int slac_init(slac_info *pSlacConfigInfo,int NumOfSlac)
{
	int i, j;
	
	if ((NumOfSlac == 0) || (NumOfSlac >= SLAC_MAX_NUM_OF_CHIP))
	{
		DBG_SLAC( KERN_WARNING "slac_init(), Invalid NumOfSlac (%d ) !\n", NumOfSlac);
		return (-ENODEV);
	}

	pSLACInfo = (sSLACInfo *)kmalloc(NumOfSlac * sizeof(sSLACInfo),GFP_KERNEL);
	if (NULL == pSLACInfo)
	{
		DBG_SLAC(KERN_WARNING "slac_init(), Couldn't allocate memory ref!\n" );
		return (-ENOMEM);
	}

	memset(pSLACInfo, 0, NumOfSlac * sizeof(sSLACInfo));

	slac_max_module_num = NumOfSlac;

	for (i=0; i<slac_max_module_num; i++)
	{

		pSLACInfo[i].ChansInModule = CHANNELS_PER_QSLAC; /*4 channels */

		pSLACInfo[i].SLACLiveStatus = 0xFF;    /*init onhook state !*/

		pSLACInfo[i].OpStatus = FALSE;

		for (j=0; j<pSLACInfo[i].ChansInModule; j++)
		{
			pSLACInfo[i].SLICEnable[j] = FALSE;
			pSLACInfo[i].SLICCurrentStatus[j] = 0x01;
			if (pSlacConfigInfo)	/* check for NULL config */
			{
				/* configuration info supplied */
				pSLACInfo[i].CircuitType[j] = pSlacConfigInfo[i].SlacChannelInfo[j].CircuitType;
				pSLACInfo[i].SlacChannel[j].source = pSlacConfigInfo[i].SlacChannelInfo[j].SlacChannel.source;
				pSLACInfo[i].SlacChannel[j].port = pSlacConfigInfo[i].SlacChannelInfo[j].SlacChannel.port;
				pSLACInfo[i].TimeSlotMap[j] = pSlacConfigInfo[i].SlacChannelInfo[j].SlacChannel.port;
			}
			else
			{
				/* default configuration info */
				pSLACInfo[i].CircuitType[j] = SLAC_LOOP_START;
				pSLACInfo[i].SlacChannel[j].source = 0;
				pSLACInfo[i].SlacChannel[j].port = j;
				pSLACInfo[i].TimeSlotMap[j] = (j+tsoffset) & 0x7f;
			}
			pSLACInfo[i].NoRingDBCount[j]    = SLAC_DEFAULT_NO_RING_DEBOUNCE_COUNT;
			pSLACInfo[i].RingingDBCount[j]   = SLAC_DEFAULT_RINGING_DEBOUNCE_COUNT;
			pSLACInfo[i].B2TransitDBCount[j] = SLAC_DEFAULT_B2_TRANSIT_DEBOUNCE_COUNT;
			pSLACInfo[i].SLICEnable[j]        = TRUE;
			pSLACInfo[i].SLICCurrentStatus[j] = 0x01;

			pSLACInfo[i].PhoneDevice[j].board = slac_make_cookie(i,j);
			pSLACInfo[i].PhoneDevice[j].f_op = &slac_fops;
			pSLACInfo[i].PhoneDevice[j].open = slac_open;
			init_waitqueue_head(&(pSLACInfo[i].poll_q[j]) );
#ifndef    DONT_REQUIRE_TELEPHONY_API
			{
				int returnval;

				returnval = phone_register_device(&pSLACInfo[i].PhoneDevice[j], PHONE_UNIT_ANY);

				if (returnval)
				{
					{
						/*Unregister what we already registered*/
						int SLACNum, Port, MaxPorts;
						for (SLACNum=0; SLACNum <=i;SLACNum++ )
						{
							MaxPorts = (SLACNum == i)? j :pSLACInfo[SLACNum].ChansInModule;
							for (Port=0; Port < MaxPorts ; Port++ )
							{
								phone_unregister_device(&pSLACInfo[SLACNum].PhoneDevice[Port]);
							}
						}
					}

					DBG_SLAC(KERN_WARNING "SLAC: Unable to register phone device SLAC %d, PORT%d\n",i,j);
					kfree(pSLACInfo);
					pSLACInfo = NULL;
					return(returnval);
				}
			}
#endif  /* DONT_REQUIRE_TELEPHONY_API */
		}
		
		/*Initialize SLAC and set ports to active state */
		slac_fxs_init(i, pSLACInfo[i].TimeSlotMap, pSLACInfo[i].TimeSlotMap);
	}
	/*Initialize the debouce timer */
	init_timer(&debounce_timer);
	debounce_timer.function = slac_timeout;
	debounce_timer_running=0;
	slac_add_timer();
	return (0);
}


static void slac_add_timer()
{
	debounce_timer.expires = jiffies + (SLAC_DEBOUNCE_TIMER_INTERVAL * HZ/1000 );
	add_timer(&debounce_timer);
	debounce_timer_running=~0;
}
#ifndef   CONFIG_BRECIS_FPGA
#ifndef   DEBUG_REMOVE_ISR
/**********************************************************************
 ** slac_isr
 **
 ** This is the ISR for QSLAC HW interrupts.
 **
 ***********************************************************************/

static void slac_isr(int irq,void* dev_id,struct pt_regs *regs )
{
	U8  slacstatus,j;
	U8 chipNo = (U8) (U32) dev_id;
	
	/*Read the data register to clear the interrupt */
	slacstatus =  QSLACRdCMD_1(chipNo, RD_17_RT_DATA_REG_CL) & SLIC_DETECT_MASK ;
	DBG_SLAC(KERN_WARNING "Got QSLAC interrupt on ChipNo: %d, Data Reg 0x%X \n",chipNo,slacstatus);
	if (pSLACInfo[chipNo].SLACLiveStatus != slacstatus)
	{
		/* update the current status */
		pSLACInfo[chipNo].SLACLiveStatus = slacstatus;
		for (j=0; j<pSLACInfo[chipNo].ChansInModule; j++)
		{
			if (pSLACInfo[chipNo].SLICCurrentStatus[j] != (slacstatus & 0x01))
			{
				/* The status changed: remember current (old) state and set off the debounce*/
				if (pSLACInfo[chipNo].SLICTimerCounter[j] == 0)
				{
					pSLACInfo[chipNo].SLICCheckStatus[j] = pSLACInfo[chipNo].SLICCurrentStatus[j];
					if (pSLACInfo[chipNo].SLICState[j] == SLIC_RINGING)
					{
						pSLACInfo[chipNo].SLICTimerCounter[j] = pSLACInfo[chipNo].RingingDBCount[j];
					}
					else
					{
						pSLACInfo[chipNo].SLICTimerCounter[j] = pSLACInfo[chipNo].NoRingDBCount[j];
					}
				}
				/* update the current status for this channel */
				pSLACInfo[chipNo].SLICCurrentStatus[j] = slacstatus & 0x01;
			}
			slacstatus = slacstatus >> 2;
		}
	}
} /* slac_isr */
#endif /* DEBUG_REMOVE_ISR */
#endif /* CONFIG_BRECIS_FPGA */

static void slac_reset_peri_register(void)
{
	int count=1000;
	/*Take the peripheral block out of reset*/
	*RST_CLR_REG = PER_RST;
	while(count--)
	{
		if (1 == count){printk(KERN_ALERT
			"Peripheral block not out of reset after 1 S\n");}
		if (!(*SYS_RST_REG & PER_RST));break;
		NanoDelay(1000000);	/* delay 1 ms = 1000000 ns */
	}

	/*Configure the SPI core control register */
	/*
	*  (1 << 7) Interrupt enabled (hardwired 1)
	 * (00 << 5) clk divide = 8
	 * (01 << 5) clk divide = 16
	 * (10 << 5) clk divide = 32
	 * (11 << 5) clk divide = 64
	 *  (1 << 4) Neg edge flop in sin path (see manual)
	 *  (0 << 4) Neg edge flop in sout path
	 *  (1 << 3) SCK (clock) idles high
	 *  (0 << 3) SCK (clock) idles low
	 *  (1 << 2) Wire-OR mode enabled
	 *  (0 << 2) Wire-OR mode disabled
	 *  (1 << 1) SPI master mode (hardwired 1)
	 * Bit 0 is reserved
	 */
	/* div 8, neg edge flop in sin, clk idles high, no wire-OR */
	*SPI_MPI_CORE_CNTL_REG |= 0x18;
}


static void QSLACWrCMD(int ChipNo, U8 Command, U8 *pData, U8 Len)
{
	U8  index;
	unsigned long intFlags;

	if (ChipNo >=SLAC_MAX_NUM_OF_CHIP )
		return;

	spin_lock_irqsave(&spi_mpi_lock,intFlags);

	slac_deassert_chipselect();

	*SPI_MPI_CNTL_REG = SPI_MPI_FLUSH_Q;

	if (((*PER_STAT_BIT_REG) & SPI_MPI_FIFO_EMPTY) == SPI_MPI_FIFO_EMPTY)
	{
		*SPI_MPI_TX_SIZE_REG = 1; /* Transmit MPI QSLAC command only byte-by-byte */
		/* send the READ command */
		*SPI_MPI_DATA_FIFO_REG_U8 = Command ;
		
		slac_assert_chipselect(ChipNo);

		*SPI_MPI_CNTL_REG = SPI_MPI_TX_START ;
		while (((*PER_STAT_BIT_REG) & SPI_MPI_TX_BUSY) == SPI_MPI_TX_BUSY){/*wait*/}
		
		slac_deassert_chipselect();

		/* queue the FIFO buffer */
		for(index = 0; index < Len; index++)  /* size excluded the command */
		{
            /*
			** Poll the receive busy signal but data is written back in FIFO reg
            */
            if (((*PER_STAT_BIT_REG) & SPI_MPI_FIFO_FULL) != SPI_MPI_FIFO_FULL)
            {
				*SPI_MPI_DATA_FIFO_REG_U8 = pData[index];
				slac_assert_chipselect(ChipNo);
			
				*SPI_MPI_CNTL_REG = SPI_MPI_TX_START ;
				while (((*PER_STAT_BIT_REG) & SPI_MPI_TX_BUSY) == SPI_MPI_TX_BUSY){/*wait*/}
				
				slac_deassert_chipselect();
			}
            else
			{
				DBG_SLAC(KERN_WARNING "SLACWrCMD(), FIFO full\n");
				goto wr_release_lock_return;
            }
		}
	}
	else
	{
		DBG_SLAC(KERN_WARNING "SLACWrCMD(), FIFO not empty\n");
		goto wr_release_lock_return;
	}

	*SPI_MPI_CNTL_REG = SPI_MPI_FLUSH_Q;  /* flush Q for next operation */
wr_release_lock_return:
	spin_unlock_irqrestore(&spi_mpi_lock,intFlags);
	return;

} /* QSLACWrCMD */


/**********************************************************************
 ** QSLACRdCMD
 ** NOTE: delay between DeAssertCS and AssertCS must be > 2.5u sec.
 ***********************************************************************/
static void QSLACRdCMD(U8 ChipNo, U8 Command, U8 *pData, U8 Len)
{
	U8  data=0;
	U8  index;
	unsigned long intFlags;

	if (ChipNo >= 8)
	{
		DBG_SLAC(KERN_WARNING "QSLACRdCMD invalid ChipNo  %d,\n",ChipNo);
		return;
	}

	spin_lock_irqsave(&spi_mpi_lock,intFlags);

	slac_deassert_chipselect();
	
	*SPI_MPI_TX_SIZE_REG = 1; /* Transmit MPI QSLAC Read command */
	*SPI_MPI_RX_SIZE_REG = 1;/*size - 1 ; Recvd size from QSLAC exclude the command */
	*SPI_MPI_DATA_FIFO_REG_U8 = Command ;
	
	slac_assert_chipselect(ChipNo);
	
	*SPI_MPI_CNTL_REG = SPI_MPI_TX_START ;
	while (((*PER_STAT_BIT_REG) & SPI_MPI_TX_BUSY) == SPI_MPI_TX_BUSY){/*wait*/}
	
	slac_deassert_chipselect();
	
	/* dequeue the FIFO buffer */
	for(index = 0; index < Len; index++)
	{
		/*
		** Poll the receive busy signal but data is written back in FIFO reg
		*/
		if (((*PER_STAT_BIT_REG) & SPI_MPI_FIFO_FULL) != SPI_MPI_FIFO_FULL)
		{
			slac_assert_chipselect(ChipNo);
			
			*SPI_MPI_CNTL_REG = SPI_MPI_RX_START ;
			while (((*PER_STAT_BIT_REG) & SPI_MPI_RX_BUSY) == SPI_MPI_RX_BUSY){/*wait*/}
			slac_deassert_chipselect();
			
			data = *SPI_MPI_DATA_FIFO_REG_U8 ;
			pData[index] = data;					
		}
		else
		{
			DBG_SLAC(KERN_WARNING "SLACRdCMD(), FIFO full\n");
			goto rd_release_lock_return;
		}
	}

	*SPI_MPI_CNTL_REG = SPI_MPI_FLUSH_Q;  /* flush Q for next operation */
rd_release_lock_return:
	spin_unlock_irqrestore(&spi_mpi_lock,intFlags);
	return;
} /* QSLACRdCMD */


/**********************************************************************
 ** QSLACWrCMD_0
 **
 **
 **
 ***********************************************************************/
static void QSLACWrCMD_0(U8 ChipNo, U8 Command)
{
	U8     data;

	QSLACWrCMD(ChipNo, Command, &data, 0);
} /* QSLACWrCMD_0 */

/**********************************************************************
 ** QSLACWrCMD_1
 **
 **
 **
 ***********************************************************************/
static void QSLACWrCMD_1(U8 ChipNo, U8 Command, U8 Data)
{
	U8     data;

	data = Data;
	QSLACWrCMD(ChipNo, Command, &data, 1);
} /* QSLACWrCMD_1 */

/**********************************************************************
 ** QSLACRdCMD_1
 ***********************************************************************/
static U8 QSLACRdCMD_1(U8 ChipNo, U8 Command)
{
	U8     data;

	QSLACRdCMD(ChipNo, Command, &data, 1);
	return(data);
} /* QSLACRdCMD_1 */


/**********************************************************************
 ** SelectChannel
 **
 **
 **
 ***********************************************************************/
static void SelectChannel(U8 ChipNo, U8 Channel)
{
	/*
	 * robbed bit signaling disabled
	 * Vout = Vref
	 * low power mode OFF
	 */
	QSLACWrCMD_1(ChipNo, WR_14_ENABLE_CH_REG, 1 << Channel);
} /* SelectChannel */

/**********************************************************************
 ** SelectAllChannels
 **
 **
 **
 ***********************************************************************/
static void SelectAllChannels(U8 ChipNo)
{
	/*
	 * robbed bit signaling disabled
	 * Vout = Vref
	 * low power mode OFF
	 */
	QSLACWrCMD_1(ChipNo, WR_14_ENABLE_CH_REG, ENABLE_ALL_CHANNELS);
} /* SelectAllChannels */


/**********************************************************************
 ** SlacdrvFxSQSLACFilterSetup
 **********************************************************************/
void SlacdrvFxSSLACFilterSetup(U8 ChipNo, slac_companding_law compandingLaw, slac_coeff_type *pSlacCoeff)
{

	SelectAllChannels(ChipNo);

	QSLACWrCMD_1(ChipNo, WR_18_AISN_ANA_GAIN, pSlacCoeff->AISN);

	QSLACWrCMD(ChipNo, WR_31_GX_FILTER, &pSlacCoeff->GX[0], 2);

	QSLACWrCMD(ChipNo, WR_33_GR_FILTER, &pSlacCoeff->GR[0], 2);

	QSLACWrCMD(ChipNo, WR_35_Z_FILTER, &pSlacCoeff->Z[0], 15);

	QSLACWrCMD(ChipNo, WR_37_B_FILTER, &pSlacCoeff->B[0], 14);

	QSLACWrCMD(ChipNo, WR_39_X_FILTER, &pSlacCoeff->X[0], 12);

	QSLACWrCMD(ChipNo, WR_41_R_FILTER, &pSlacCoeff->R[0], 14);

	QSLACWrCMD(ChipNo, WR_43_B2_FILTER, &pSlacCoeff->B2[0], 2);

	if (compandingLaw == SLAC_A_LAW)
	{
		QSLACWrCMD_1(ChipNo, WR_24_OP_REG,
			(COMPRESSED | ALAW_BIT | ENABLE_PROG_FILTERS));
	}
	else
	{
		QSLACWrCMD_1(ChipNo, WR_24_OP_REG,
			(COMPRESSED | MULAW_BIT | ENABLE_PROG_FILTERS));
	}

} /* SlacdrvFxSSLACFilterSetup */


/**********************************************************************
 ** slac_fxs_init
 ***********************************************************************/

static inline U8 QSLACConfSetup(void)
{
	/*
	 * turn on the otherwise unused chopper clock bit (6) to provide
	 *  a more unique value for config for detecting readback errors
	 */
	U8 ConfSetup=((QSLAC_CONF_SETUP | (1 << 6)) & 0xf0);
	U8 ClockConstant;

	switch (clock)
	{
		case 1536:
			ClockConstant = 0x0;
			break;
		case 1544:
			ClockConstant = 0x1;
			break;
		case 2048:
			ClockConstant = 0x2;
			break;
		case 3072:
//			ClockConstant = 0x4;
			ClockConstant = 0x0;
			break;
		case 3088:
//			ClockConstant = 0x5;
			ClockConstant = 0x1;
			break;
		case 4096:
//			ClockConstant = 0x6;
			ClockConstant = 0x2;
			break;
		case 6144:
			ClockConstant = 0x8;
			break;
		case 6176:
			ClockConstant = 0x9;
			break;
		case 8192:
			ClockConstant = 0xa;
			break;
		default:
			ClockConstant = 0x1;
			break;
	}
	return (ConfSetup | ClockConstant);
}

void slac_fxs_init(U8 ChipNo,U8 *pTxTSMap, U8 *pRxTSMap)
{
	volatile U8  ReadBackVal;
	U32 j;

	QSLACWrCMD_0(ChipNo, WR_3_HW_RES_REG); /* put HW in known state */
	NanoDelay(1000000); /* wait 1 ms */
	/*
	 * INTM   (7) = TTL compatible
	 * CHP    (6) = chopper clock 256 KHz
	 * PCMSIG (5) = No signaling on PCM Hwy
	 * CLKSRC (4) = MCLK, no E1
	 * CLK (3..0) = 0001 = 1.544 MHz (implicitly sets # of time slots on PCM
	 */
	QSLACWrCMD_1(ChipNo, WR_12_CONFIG_REG, QSLACConfSetup());

	ReadBackVal = QSLACRdCMD_1(ChipNo, RD_13_CONFIG_REG);
	if (ReadBackVal != QSLACConfSetup())
	{
		pSLACInfo[ChipNo].OpStatus = FALSE;
		printk(KERN_ALERT "=====> slac_fxs_init(), RD_CONFIG_REG failed:"
			" ChipNo %d, rdVal=0x%02x != 0x%02x!!\n",
			ChipNo, ReadBackVal, QSLACConfSetup());
		return;
	}
	printk(KERN_INFO "QSLAC CONFIG REG: 0x%02x\n", ReadBackVal);


	QSLACWrCMD_1(ChipNo, WR_22_IO_DIR_REG, SLIC_IO_SETUP);

	/*
	 * TAB = 0 = Transmit on selected PCM Hwy
	 * XE = 1 = Transmit on + edge
	 * RCS = 0 = No RX clock skew
	 * TCS = 0 = No TX clock skew
	 */
	QSLACWrCMD_1(ChipNo, WR_10_CLOCK_EDGE_REG, QSLAC_CLK_EDGE_SETUP);

	for (j=0;j<pSLACInfo[ChipNo].ChansInModule;j++)
	{
		SelectChannel(ChipNo, j);
		QSLACWrCMD_1(ChipNo, WR_6_TX_TS_SEL_REG, pTxTSMap[j]);
		QSLACWrCMD_1(ChipNo, WR_8_RX_TS_SEL_REG, pRxTSMap[j]);
		printk(KERN_INFO "QSLAC[%d] TX TS: %d  RX TS: %d\n",
			j, pTxTSMap[j], pRxTSMap[j]);
	}

	if ((clock != 3072) && (clock != 3088) && (clock != 4096))
	{
		/* no double PCLK mode, chopper off */
		QSLACWrCMD_1(ChipNo, WR_45_DEBOUNCE_TIME_REG, (0xF<<2)); /*15 ms Debounce */
	}
	else
	{
		/* double PCLK mode, chopper off */
		QSLACWrCMD_1(ChipNo, WR_45_DEBOUNCE_TIME_REG, ((0xF<<2)|(1 << 1))); /*15 ms Debounce */
		printk(KERN_INFO "Double-clock mode enabled %d [0x%02x]\n",
			clock, QSLACRdCMD_1(ChipNo, (WR_45_DEBOUNCE_TIME_REG+1)));
	}

	SelectAllChannels(ChipNo);

	SlacdrvFxSSLACFilterSetup(ChipNo, slac_companding_type, &Am79063DefCoeff );

	QSLACWrCMD_1(ChipNo, WR_28_OP_COND_REG, 0); /* set default */
	if (loopback)
	{
		QSLACWrCMD_1(ChipNo, WR_28_OP_COND_REG, (1 << 2)); /* I/F Loop Back */
		printk(KERN_INFO "QSLAC ILB ON!\n");
	}

	/* activate chip to turn on PCM Hwy operations (after 2nd frame sync) */
	QSLACWrCMD_0(ChipNo, WR_5_ACTIVATE_REG);

	pSLACInfo[ChipNo].OpStatus = TRUE;

	/* all "A" channels unmasked, all "B" channels masked */
	QSLACWrCMD_1(ChipNo, WR_26_INT_MASK_REG, ~SLIC_INT_MASK);

	for (j=0;j<pSLACInfo[ChipNo].ChansInModule;j++)
	{
		set_slic_state(ChipNo, j,SLIC_ACTIVE);
	}

} /* slac_fxs_init */


/**********************************************************************
 **  set_slic_state
 ***********************************************************************/
void set_slic_state(U8 ChipNo, U8 Channel, slac_slic_state NewState)
{
	slac_circuit_type    cirType;
	slac_slic_state      CurrentState;
	U8        CurrentCtrlByte, NewCtrlByte;

	CurrentState = pSLACInfo[ChipNo].SLICState[Channel];

	if (CurrentState == NewState)
	{
		return;
	}
	DBG_SLAC(KERN_WARNING "SetSLICState: Received State change request 0x%02X  on SLAC %d Channel: %d\n",
			 NewState,ChipNo,Channel);
	cirType = pSLACInfo[ChipNo].CircuitType[Channel];

	SelectChannel(ChipNo, Channel);
	/* write output bits to set SLIC state */
	QSLACWrCMD_1(ChipNo, WR_20_SLIC_IO_REG, SlicCtrlByte[cirType][NewState]);

	pSLACInfo[ChipNo].SLICState[Channel] = NewState;
	CurrentCtrlByte =	SlicCtrlByte[cirType][CurrentState];
	NewCtrlByte = 	SlicCtrlByte[cirType][NewState];

	if ((CurrentCtrlByte & SLIC_B2EN_MASK) != (NewCtrlByte & SLIC_B2EN_MASK))
	{
		/* change in B2EN, need debouncing here -- if timercounter already active: overwrite it ! */
		pSLACInfo[ChipNo].SLICCheckStatus[Channel] = pSLACInfo[ChipNo].SLICCurrentStatus[Channel];
		pSLACInfo[ChipNo].SLICTimerCounter[Channel] = pSLACInfo[ChipNo].B2TransitDBCount[Channel];

	}
}


static void slac_timeout(unsigned long ptr)
{
	U8  i, j;
	for (i=0; i<slac_max_module_num; i++)
	{
		if (pSLACInfo[i].OpStatus == TRUE)
		{
			for (j=0;j<pSLACInfo[i].ChansInModule;j++)
			{
				if (pSLACInfo[i].SLICTimerCounter[j] > 0)
				{
					pSLACInfo[i].SLICTimerCounter[j]--;
					if (pSLACInfo[i].SLICTimerCounter[j] == 0)
					{
						/* let's check if there is real status change after debouncing */
						if (pSLACInfo[i].SLICCheckStatus[j] != pSLACInfo[i].SLICCurrentStatus[j])
						{
							/* yes, status changed -- Current Status is the new status. */
#if 0
							if (pSLACInfo[i].SLICCurrentStatus[j])
							{
								DBG_SLAC(KERN_WARNING "QSLAC: LOOP Open: %d:%d\n",i,j);
							}
							else
							{
								DBG_SLAC(KERN_WARNING "QSLAC: LOOP Closed: %d:%d\n",i,j);
							}
#endif
							pSLACInfo[i].SLICExceptionFlag[j] = 0xFF;
							/*Post a signal to the application */
							kill_fasync(&pSLACInfo[i].async_queue[j],SIGIO,POLL_IN);
							/*Wake up any blocked selects */
							wake_up_interruptible(&(pSLACInfo[i].poll_q[j]) );
						}
					}
				}
			}
		}
	}
	slac_add_timer();
}


/*************************************************************************
void SlacCfgRingerClk( U32 clkFreq)	
configures Programmable Interval Timer2
to generate a square wave  of (clkFreq) Hz and connects the output of the timer2
to GPIO pin 2.
*************************************************************************/
static void slac_cfg_ringer_clk( U32 clkFreq)	
{
#ifndef   CONFIG_BRECIS_FPGA
#ifndef   DEBUG_REMOVE_ISR
	/*Configure GPIO2 as output
	  for timer counter 2*/
	*GPIO_CFG1_REG |=0x00000B00;
	*PER_PIT2_MODE_REG = TIMER_NO_INVERSION | TIMER_MODE_3;
	*PER_PIT2_CNT_REG  =  PER_CLOCK_RATE/clkFreq;
	*PER_PIT2_GATE_REG = TIMER_GATE_CONTROL_ON;
#endif /* DEBUG_REMOVE_ISR */
#endif /* CONFIG_BRECIS_FPGA */
}


/***************************************************************************
 *
 * sysGetPllDivider - get the PLL divider value
 *
 * This routine calculates the PLL divider vlaue based on the
 * formula provided in the Hardware Users Manual
 *
 * RETURNS: PLL divider value
 *
 */
#ifndef   CONFIG_BRECIS_FPGA
static U32 sysGetPllDivider(void)
{
	U32  pllDivider;

	pllDivider = (*PLL_DIV_REG & 0x1F);

	if((pllDivider > 20) || (pllDivider < 15))
	{
		pllDivider = PLL_MIN_VAL;

		/* write it back to make someone happy */
		*PLL_DIV_REG = (*PLL_DIV_REG & ~(0x1F)) | pllDivider;
	}

	return pllDivider;
}

static int is5000Family(void)
{
// ***** check for POLO *****
	U8 devID=((*DEV_ID_REG >> 8) & 0xFF);

	if (devID == 0x50)	/* MSP 5000 */
	{
		return (~0);
	}
	if (devID == 0x00)	/* FPGA */
	{
		if ((*DEV_ID_REG & 0xFF) < 0x90)
		{
			return (~0);
		}
	}
	return 0;
}
#endif /* CONFIG_BRECIS_FPGA */

/******************************************************************************
 *
 * sysSetPCMHighwayClkRate - set the PCM highway clock rate
 *
 * This routine takes as input the clock and dersired clock rate
 * in cycles per second and sets it.
 *
 * clkToSet: one of
 *          MSP_PCM_HIGHWAY_ZERO
 *          MSP_PCM_HIGHWAY_ONE
 *          MSP_WAN_SERIAL_INTERFACE_ZERO_TX
 *          MSP_WAN_SERIAL_INTERFACE_ONE_TX
 *          MSP_WAN_SERIAL_INTERFACE_ZERO_RX
 *          MSP_WAN_SERIAL_INTERFACE_ONE_RX
 *
 * RETURNS: OK or ERROR
 *
 */

static U32 sys_set_pcm_highway_clk(U32 clkToSet,U32 clkRate)
{
#ifndef   CONFIG_BRECIS_FPGA
	switch (clkToSet)
	{
	case MSP_PCM_HIGHWAY_ZERO:
		/* Select PCM0 */
		if (is5000Family())
		{
			*SER_STA_REG = (*SER_STA_REG & ~0x00000002) | 0x00000002;
		}
		else
		{
			*SER_STA_REG = (*SER_STA_REG & ~0x00000202) | 0x00000202;
		}
		*PCM0_CLK_REG =  sys_get_clk_div(clkRate);
		return 0;

	case MSP_PCM_HIGHWAY_ONE:
		/* Select PCM1 */
		if (is5000Family())
		{
			*SER_STA_REG = (*SER_STA_REG & ~0x00000008) | 0x00000008;
		}
		else
		{
			*SER_STA_REG = (*SER_STA_REG & ~0x00000808) | 0x00000808;
		}
		*PCM1_CLK_REG =  sys_get_clk_div(clkRate);
		return 0;

	case MSP_WAN_SERIAL_INTERFACE_ZERO_TX:
	case MSP_WAN_SERIAL_INTERFACE_ONE_TX:
		/*
		 * WAN serial interface internal clocks
		 * these WAN clocks are shared by both serial WAN interfaces
		 * eg the clocks for serial WAN interface 0 are the SAME
		 * as the clocks for serial WAN interface 1.
		 *
		 * these clock registers have nothing to do with
		 * the clock registers for the PCM Highway
		 *
		 * PCM2_CLK is the tx clock for the serial interface
		 * PCM3_CLK is the rx clock for the serial interface
		 */
		*PCM2_CLK_REG = sys_get_clk_div(clkRate);
		return 0;

	case MSP_WAN_SERIAL_INTERFACE_ZERO_RX:
	case MSP_WAN_SERIAL_INTERFACE_ONE_RX:
		*PCM3_CLK_REG = sys_get_clk_div(clkRate);
		return 0;

	default:
		return ~0 ;
	}
#else  /* CONFIG_BRECIS_FPGA */
	return 0;
#endif /* CONFIG_BRECIS_FPGA */
}

/***************************************************************************
*
* sys_get_clk_div - calculate clock divider
*
* This routine will calculate clock divider needed given the output
* frequency.
*
* RETURNS: divider value
*
*/
#ifndef   CONFIG_BRECIS_FPGA
#define FIXED_MULTIPLIER (1LL << 33)    /* includes extra factor of 2 */

static U32 sys_get_clk_div(int Out)
{
	long long temp=FIXED_MULTIPLIER;

	if (!is5000Family()){temp >>= 1;}
	temp *= (long long)Out;
	temp /= ((long long)sysGetPllDivider() * (long long)EXTERNAL_XTL);
	return (U32)temp;
}
#endif /* CONFIG_BRECIS_FPGA */

module_init(slac_module_init);
module_exit(slac_module_exit);
MODULE_LICENSE("GPL");

EXPORT_SYMBOL(set_slic_state) ;
EXPORT_SYMBOL(slic_xopen) ;
