/*
** $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/brecis/brecis_sec_v2.c,v 1.11 2003/03/31 23:11:00 swahl Exp $
*/

/*
** uClinux driver for Brecis security engine (second hw version).
**
** Copyright 2002 Brecis Communictaions
**
*/

#include <linux/config.h>
#include <linux/version.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <linux/proc_fs.h>
#include <asm/addrspace.h>
#include "brecis/msp_sec.h"
#include "brecis/msp_sec_kern.h"
#include <asm/brecis/prom.h>

#ifdef CONFIG_BRECIS_SEC_V2

#include "brecis/msp_secv2.h"
#include "brecis/msp_secv2_hw.h"
#include <asm/brecis/brecisint.h>

/* "SWITCHES" --------------------------------------------------*/

//#define DEBUG
//#define DEBUG_VERBOSE
//#define DUMP_WQ_ENTRIES
//#define DUMP_CQ_ENTRIES

//#define VERIFICATION_TESTS

//#define SIMULATE_V2_HW
//#define DEBUG_SIM
//#define DEBUG_SIM_VERBOSE

#ifndef SIMULATE_V2_HW
//#define USE_SCRATCHPAD

#endif

/* support for "SWITCHES" ----------------------------------------*/

#ifdef SIMULATE_V2_HW
void do_simulate_v2_hardware(void) ;
#endif
#ifdef VERIFICATION_TESTS
void verification_tests(void) ;
#endif

/* #define BAD_HARDWARE_KLUDGES */

/* Prototypes --------------------------------------------------*/

static int v2_hw_present ;

extern void NanoDelay(int) ;

static void msp_secv2_interrupt(int irq, void *dev_id, struct pt_regs *regs) ;
static void msp_sec2_poll_work_queue_space(void) ;

/*************************************************************
** CACHE and other memory stuff.
*/

#define SYNC() __asm__ volatile("sync");

/* define for a Cache Invalidate */
#define msp_hit_inval_dcache(addr, ofst)    \
({ \
    __asm__ volatile (	"	.set push\n" \
			"	.set mips3\n" \
			"	cache 0x11,%1(%0)\n" \
			"	.set pop" \
		: : "r" (addr), "I" (ofst) );   \
})

/************************************************************************
 * Function Name:	invalidate_buffer
 * Stolen From:         triad.c (ethernet driver
 *
 * Purpose:	Invalidate buffer cache
 *
 *         SRW I think the following are of concern if you have
 *         a writeback cache, but we have a write-through, so
 *         invalidating more than just your memory is no problem:
 *
 * 		- must start on 16 byte boundary
 * 		- must "own" to next higher 16 byte boundary from end of buffer
/ ***********************************************************************/
static void
invalidate_buffer( void * bufaddr, unsigned int length)
{
        unsigned long end = (unsigned long)bufaddr + length ;
        bufaddr = (void *) ((unsigned int) bufaddr & ~(L1_CACHE_BYTES - 1));

        while ( (unsigned long) bufaddr < (end - 8 * L1_CACHE_BYTES) )
        {
                msp_hit_inval_dcache(  bufaddr , (0 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (1 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (2 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (3 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (4 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (5 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (6 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (7 * L1_CACHE_BYTES) );
                bufaddr += 8 * L1_CACHE_BYTES;
        }

        while ( (unsigned long) bufaddr < end )
        {
                msp_hit_inval_dcache(  bufaddr, 0 );
                bufaddr += L1_CACHE_BYTES;
        }
}

/*
** Managing the work queues and completion queues of the polo style
** security engine.
*/

/*
** The following constants are chosen by software.  Must be powers of two.
*/

#define SEC_WORK_Q_SIZE		256
#define SEC_COMP_Q_SIZE		512

/*
** This is software's picture of a work queue
*/
struct workq {
	struct sec2_q_regs *wq_regs ; /* registers for this queue */
	unsigned int *hw_ptr ;	/* Hardware writes its ptr here */
	unsigned char *base ;	/* base of queue */
	int mask ;		/* size - 1 */
	int in_ptr ;		/* copy of in_ptr as we wrote to hw */
	int out_ptr ;		/* local copy of out pointer */

	/* temporaries used in building work queue entry */
	
	int in_ptr_tmp ;	/* temporary in ptr while building */
	int building_desc ;	/* non-zero when building a descriptor */
	int first_gather ;	/* non-zero if gather hasn't been seen yet */
	int last_gather ;	/* offset of last gather entry */
	int prev_gather ;	/* offset of 2nd to last gather entry */
	int first_scatter ;	/* non-zero if scatter not seen yet */
	int last_scatter ;	/* offset of last scatter entry */
	MSP_SEC2_SA *sa ;	/* current security association */
	int cbk_special ;	/* for handling "special" callback values */
	unsigned int old_mask ;	/* old value for int mask */
#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
	unsigned int g_tot;	/* total gather space */
	unsigned int s_tot;	/* total scatter space */
	unsigned int g_len;	/* length of last gather */
#endif
	unsigned int sanity_checks_failed ;
	/* Other things related to the wait queue */
	wait_queue_head_t space_wait ;  /* tasks waiting for wq space */
	int low_water ;		/* when avail space reaches this, wake tasks */
	int mask_ints ;		/* !=0 if ints masked while buiding entry */

} ;

/* XXX This will change when the hw pointer is written to RAM */
#define WQ_AVAIL_SPACE(wq) \
	( ( (wq)->out_ptr - (wq)->in_ptr - 1 ) & (wq)->mask )

/*
** This is software's picture of a completion queue
*/

struct compq {
	struct sec2_q_regs *cq_regs ; /* registers for this queue */
	unsigned int *hw_ptr ;	/* HW writes pointer here */
	unsigned char *base ;	/* base of queue */
	int mask ;		/* size - 1 */
	int out_ptr ;		/* copy of out pter as we wrote to hw */
} ;

struct workq sec_work_queues[HW_NR_WORK_QUEUES] ;
struct compq sec_comp_queues[HW_NR_COMP_QUEUES] ;

#ifndef SIMULATE_V2_HW
struct sec2_regs *sec2_regs = (struct sec2_regs *) SEC2_BASE ;
#else
struct sec2_regs sregs ;
struct sec2_regs *sec2_regs = &sregs ;
static msp_sec_des_regs *Des = (msp_sec_des_regs *) (DES_NDA_REG) ;
static msp_sec_hsh_regs *Hsh = (msp_sec_hsh_regs *) (HSH_NDA_REG) ;
#endif

int irqvalid = 0 ;

/*------------------------------------------------------------*/
/* XXX The following should move to the scratchpad RAM */

/*
** The work queues and completion queues should eventually be in the
** scratchpad RAM when the method for divvying up that space is defined.
**
** The hardware guys didn't invest in adders for the work and
** completion queues.  In other words, they must be aligned on a
** boundary the size of the queue.  E.g. a 4K queue must be on a 4K
** boundary; an 8k queue on an 8k boundary, etc.
*/
#ifndef USE_SCRATCHPAD
static unsigned char work_queues[HW_NR_WORK_QUEUES] [SEC_WORK_Q_SIZE]
	__attribute__ ((aligned(SEC_WORK_Q_SIZE))) ;

static unsigned char comp_queues[HW_NR_COMP_QUEUES] [SEC_COMP_Q_SIZE]
	__attribute__ ((aligned(SEC_COMP_Q_SIZE))) ;

static unsigned int ptr_values_w[HW_NR_WORK_QUEUES] ;
static unsigned int ptr_values_c[HW_NR_COMP_QUEUES] ;
#else
static unsigned char *work_queues[2] = {
	(unsigned char *)0xB7F80000,
	(unsigned char *)0xB7F81000
} ;
static unsigned char *comp_queues[2] = {
	(unsigned char *)0xB7F82000,
	(unsigned char *)0xB7F82200
} ;
static unsigned int *ptr_values_w = (unsigned int*) 0xB7F82400 ;
static unsigned int *ptr_values_c = (unsigned int*) 0xB7F82408 ;
#endif

/* XXX End of stuff that should move to scratchpad RAM */
/*------------------------------------------------------------*/

#define WQ_INC_TMP(wq, sz) \
	do { \
		(wq)->in_ptr_tmp = ((wq)->in_ptr_tmp + sz) & (wq)->mask ; \
	} while (0)

#define WQ_TMP_PUT_INT(wq, val) \
	do { \
		* (unsigned int *) ((wq)->base + (wq)->in_ptr_tmp) \
 			= (unsigned int)val ; \
 		WQ_INC_TMP(wq, 4) ; \
	} while (0)

#define WQ_TMP_PUT_SHORT(wq, val) \
	do { \
		* (unsigned short *) ((wq)->base + (wq)->in_ptr_tmp) = \
 			(unsigned short) val ; \
 		WQ_INC_TMP(wq, 2) ; \
	} while (0)

#define WQ_TMP_PUT_CHAR(wq, val) \
	do { \
		* (unsigned char *) ((wq)->base + (wq)->in_ptr_tmp) = \
			(unsigned char) val ; \
 		WQ_INC_TMP(wq, 1) ; \
	} while (0)


/*
** I have discovered that I need extra information when it comes to
** completion time.  In addition to the callback function and its parameter,
** I need to have the addresses of the scatter buffers to perform cache
** invalidation.
**
** The cache invalidation cannot be done before the security operation
** is complete, because we cannot require that the buffers don't
** share cache lines with any other data; and if a cache line is shared,
** reading of the data that shares the line can put stale data in the cache.
**
** One could dynamically allocate a linked list to store the list of addresses
** and the callback parameter, then store a pointer to this list in the second
** software-use word (that gets coppied to the completion queue).
**
** However, I want to avoid (a) incuring the overhead of memory
** allocation routines, and (b) writing these addresses twice; they're
** already there in the work queue.
** 
** Instead, I'm going to use one of the software-use words to point back
** to the work queue entry, and read the scatter addresses and lengths
** from there for cache invalidation.
**
** However, I now need a place to store what was previously in that
** software-use word.  Without spending time on dynamic allocation, etc.
**
** The following arrays, then, parallel the work queues.  If tha SA for a
** work queue entery is at work_queues[0][0x40], the callback paramater
** will be placed at work_queues_extra[0][0x40].  Simple math, and no
** storage allocation necessary.
**
** One final problem:  We need to ensure we don't overwrite the work
** queue entries before we have used them to invalidate the cache.
** So we keep a private copy of the out pointer for the queue that
** we will only advance after receiving the completion reply for
** an entry.  Since replies may come back out of order (two completion
** queues), the work entries must be somehow marked when they are
** no longer needed, then the private out pointer moved forward until
** it encoutners a work entry that's not marked.  For now, the
** mark will be setting bit 0 of the SA field of the work entry (making
** it odd).
*/
static unsigned char work_queues_extra[HW_NR_WORK_QUEUES] [SEC_WORK_Q_SIZE] ;


/* DEBUGGING SUPPORT -------------------------------------------------- */

#ifdef DEBUG
#define DBG_SEC(a1, a2...)	printk("SECURITY " a1, ##a2)
#else
#define DBG_SEC(a...)
#endif

#ifdef DEBUG_SIM
#define DBG_SIM(a1, a2...)	printk("SEC_SIM " a1, ##a2)
#else
#define DBG_SIM(a...)
#endif

static void dump_sec_regs(void) ;
#if defined(DEBUG)
#define debug_dump_sec_regs dump_sec_regs
#else
#define debug_dump_sec_regs()
#endif

#ifdef DUMP_WQ_ENTRIES
static void dump_wq_entry(struct workq *wq) ;
#else
#define dump_wq_entry(wq)
#endif

#ifdef DUMP_CQ_ENTRIES
static void dump_cq_entry(struct compq *cq) ;
#else
#define dump_cq_entry(cq)
#endif

int sec_debug_counters[16] ;

/* END DEBUGGING SUPPORT ------------------------------------------------*/

int sec_type_counters[8] ;


/*
** How to sanity check an operation before passing it to the hardware
** (because the hardware can hang when given bad values).
**
** This code is somewhat complex to make it fast.  Hopefully, the
** verbose comments will make it a bit clearer.
*/

#define MAX_SDRAM_VALUE	0x08000000ul

static inline int
sanity_check(MSP_SEC2_SA *sa,
	     int gather_len,
	     int last_gather_len,
	     int scatter_len,
	     unsigned int we_ctrl)
{
	unsigned int saflg, tmp2 ;
	unsigned int hashlen, cryptmod, ivlen ;
	unsigned int size ;

	/*
	** Check the sa address.  The base address of the structure and
	** the final word must both be valid SDRAM addresses.  This
	** will be true if 0 < addr < (maxram - sizeof(sa struct)).
	**
	** The pointer could be user space, kernel space, or even
	** uncached, so we mask off those bits.
	**
	** if using scratchpad (USE_SCRATCHPAD here to be found in
	** later editor searches) then note that I'm assuming that the
	** SA is not in scratchpad here.
	*/

	if (((unsigned int)sa & 0x1fffffff) > (MAX_SDRAM_VALUE - sizeof(*sa)))
	{
		DBG_SEC("sanity check: Bad SA pointer %p\n", sa) ;
		return -1 ;
	}

	saflg = sa->flags ;

	/*
	** check crypt algorithm, crypt blocking mode, hash algorithm,
	** and engine mode fields of flags for invalid or reserved
	** values in one operation.
	**
	** This checks all fields for invalid values, regardles of
	** whether a particular field is used in a particular engine
	** mode, assuming that there is no need for software to set
	** these unused fields to some other value.  Calling software
	** must abide by this rule.
	**
	** Normally, the blocking mode could not be checked in this
	** fashion, because it's immediatle next to the crypt algorithm
	** field, and the test method would overflow into it.  However,
	** both binary '110' and binary '111' are bad values for
	** crypt algorithm, so we can ignore the lsb of that field
	** and use it for overflow purposes.
	**
	** The method is quite simple, if you break the value up into
	** bits and write it out on paper.  We isolate the bits we care
	** about, then add a number to each field.  If the field is out
	** of range, the add will cause a carry to the next bit to the
	** left of the field.  So if any of these bits are set, a value
	** is out of range.
	*/

	/* isolate high two bits of crypt algorithm, crypt blocking mode,
	** hash algorithm, and engine mode: */
	tmp2 = saflg & 0x01b8e007 ;

	/* add in the value to cause carry for illegal values */
	tmp2 += 0x00886002 ;

	/* now, only look at the carry bits we may have caused */
	tmp2 &= 0x02410008 ;

	/* if any bits are set, we have an invalid value */
	if (tmp2 != 0)
	{
		DBG_SEC("sanity_check: Bad sa flags %08x\n", saflg) ;
		return -2 ;	/* invalid value */
	}

	/*
	** Now, check for ((crypt Alg == DES) or (crypt Alg == 3DES)) and
	** ((crypt block mode == CTR) or (crypt block mode == OFB)).
	**
	** Crypt alg can be checked by ( (crypt alg & b'110') == b'000'),
	** but I don't see a way to check crypt block mode without two tests.
	*/

	/* isolate two bits of crypt alg, plus three bits of
	** blocking mode */
	tmp2 = saflg & 0x01b8000 ;

	/* compares with CTR and OFB mode */
	if ( (tmp2 == 0x00080000) || (tmp2 == 0x00300000) )
	{
		DBG_SEC("sanity_check: Bad crypt mode; sa flags %08x\n",
			saflg) ;
		return -3 ;
	}

	/* special early check for AES AKO */
	if (we_ctrl & SEC2_WE_CTRL_AKO)
	{
		/* must be an AES encryption */
		if ( (saflg & SAFLG_MODE_MASK) != SAFLG_MODE_CRYPT )
		{
			DBG_SEC("sanity_check: AKO not in crypt mode\n") ;
			return -4 ;
		}
		if ( (saflg & SAFLG_KEYS_MASK) != 0 )
		{
			DBG_SEC("sanity_check: AKO with bad keys\n") ;
			return -5 ;
		}
		if ( (saflg & SAFLG_BLK_MASK) != 0 )
		{
			DBG_SEC("sanity_check: AKO with bad blocking type\n") ;
			return -6 ;
		}
		if ( (gather_len % 16) != 0 )
		{
			DBG_SEC("sanity_check: AKO with bad gather len\n") ;
			return -7 ;
		}
		if ( scatter_len != 32 )
		{
			DBG_SEC("sanity_check: AKO with bad scatter len\n") ;
			return -8 ;
		}
		switch (saflg & SAFLG_CRYPT_TYPE_MASK)
		{
		case SAFLG_DES:
		case SAFLG_3DES:
			DBG_SEC("sanity_check: AKO bad crypt algorithm\n") ;
			return -9 ;
		default:
			return 0;
		}
	}

	/* checking the scatter/gather lengths.  kinda complicated */

	if ( (saflg & SAFLG_CRYPT_TYPE_MASK) == SAFLG_CRYPTNULL )
	{
		cryptmod = 1 ;
		ivlen = 0 ;
	}
	else if ( (saflg & 0x1800000) == 0 )
	{
		cryptmod = 8 ;		/* not AES */
		ivlen = 8 ;
	}
	else
	{
		cryptmod = 16 ;		/* AES */
		ivlen = 16 ;
	}
			
	switch (saflg & SAFLG_HASH_MASK)
	{
	case SAFLG_MD5_96:
	case SAFLG_SHA1_96:
		hashlen = 12 ;
		break ;
	case SAFLG_MD5:
		hashlen = 16 ;
		break ;
	case SAFLG_SHA1:
		hashlen = 20 ;
		break ;
	default:
		hashlen = 0 ;
	}

	switch (saflg & SAFLG_MODE_MASK)
	{
	case SAFLG_MODE_ESP_IN:
		/* must have at least 2 gather segments */
		if ( ( gather_len == last_gather_len )
		     && ((saflg & SAFLG_HASH_MASK) != SAFLG_HASHNULL) )
		{
			DBG_SEC("sanity_check: ESP_IN needs >1 gather buf\n") ;
			return -10 ;
		}

		/* gather segment must have ICV of correct length */
		if ( (saflg & SAFLG_HASH_MASK) != SAFLG_HASHNULL )
		{
			if (last_gather_len != hashlen)
			{
				DBG_SEC("sanity_check: "
					"ESP_IN ICV 0x%x s/b 0x%x\n",
					last_gather_len, hashlen) ;
				return -12 ;
			}
			size = gather_len - last_gather_len - 8 - ivlen ;
		}
		else
			size = gather_len - 8 - ivlen ;

		/* must have multiple of block length in encrypted section */
		if ( (size % cryptmod) != 0 )
		{
			DBG_SEC("sanity_check: ESP_IN 0x%x %% 0x%x != 0\n",
				size, cryptmod) ;
			return -13 ;
		}

		/* if CPI is not set, scatter must include ICV space */
		if ( (saflg & SAFLG_CPI) == 0 )
			size += hashlen ;

		if (size != scatter_len)
		{
			DBG_SEC("sanity_check: ESP_IN scatter len bad. "
				"is %x, s/b %x\n", scatter_len, size) ;
			return -14 ;
		}

		return 0 ;
		
		break ;
	case SAFLG_MODE_ESP_OUT:
		if ( (saflg & SAFLG_EM) == 0 )
		{
			/*
			** not manual mode
			**
			** Gather size + padding + trailer must be % cryptmod
			**
			** Scatter size must be
			**  spi(4) + seq(4) + IV(0/8/16)
			**   + gather + pad + trailer + icv
			**
			** hashlen is length of icv
			*/
			size = gather_len +
				((we_ctrl >> SEC2_WE_CTRL_PADLEN_SHF) & 0xff) +
				2 ; /* 2 bytes for trailer */

			if ( (size % cryptmod) != 0 )
			{
				DBG_SEC("sanity_check: ESP_OUT "
					"0x%x %% 0x%x != 0\n",
					size, cryptmod) ;
				return -15 ;
			}
			if ( 8 + ivlen + size + hashlen != scatter_len )
			{
				DBG_SEC("sanity_check: ESP_OUT "
					"bad scatter len.  %x s/b %x\n",
					scatter_len,
					8 + ivlen + size + hashlen) ;
				return -16 ;
			}
		}
		else
		{
			/*
			** Manual mode.  Incomming data includes
			** SPI (4 bytes), seq no (4), IV (cryptmod),
			** payload data, padding data, trailer.
			**
			** payload + pad data + trailer must be % cryptmod
			**    (where payload = gather - spi(4) - seq(4) - IV)
			**
			** scatter len must be
			**   gather + icv
			*/
			size = gather_len - 8 - ivlen ;
			if ( (size % cryptmod) != 0 )
			{
				DBG_SEC("sanity_check: ESP_OUT(man) "
					"0x%x %% 0x%x != 0\n",
					size, cryptmod) ;
				return -17 ;
			}
			if ( gather_len + hashlen != scatter_len)
			{
				DBG_SEC("sanity_check: ESP_OUT(man) "
					"bad scatter len.  %x s/b %x\n",
					scatter_len,
					gather_len + hashlen) ;
				return -18 ;
			}
		}
		return 0 ;

		break ;
	case SAFLG_MODE_HMAC:
	case SAFLG_MODE_HASH_PAD:
		if (scatter_len == hashlen)
			return 0 ;
		DBG_SEC("sanity_check: HMAC/HASH_PAD bad scatter len "
			"(%x s/b %x\n", scatter_len, hashlen) ;
		break ;
	case SAFLG_MODE_HASH:
		if ( (gather_len % 64) == 0
		     && scatter_len == hashlen )
			return 0 ;
		DBG_SEC("sanity_check: HASH param bad, "
			"gather %x scatter %x, hashlen %x\n",
			gather_len, scatter_len, hashlen) ;
		break ;
	case SAFLG_MODE_CRYPT:
		if ( ( (gather_len % cryptmod) == 0 )
		     && (gather_len == scatter_len) )
			return 0 ;
		DBG_SEC("sanity_check: CRYPT param bad, "
			"gather %x scatter %x, cryptmod %x\n",
			gather_len, scatter_len, cryptmod) ;
		break ;
	}
	return -19 ;
}
	
/*
** sanity check an individual scatter gather entry to make sure it points
** within sdram.  Returns 0 if no problem.
**
** note: this check is only to make sure the address is valid and won't
** make the hardware stop.  It does not make sure the address is "real".
*/

static inline int
sg_sanity_check(void *addr, unsigned int size)
{
	unsigned int tmp = (unsigned int) addr ;

	tmp &= 0x1fffffff ;	/* mask off upper bits */

	/*
	** The maximum length of a segment is 4k, so if the base is
	** within 0 .. (max - 4k), the whole segment is in memory
	** and you don't have to check the length.
	**
	** if the base is not within that range, there are two cases;
	** it is in the range (max - 4k) .. (max), or it is
	** outside of valid memory.  If it is within the valid range,
	** (base + len) will also be within this range if the entire buffer
	** is in valid memory.
	**
	** If base is outside of the range of valid memory, base + len
	** cannot be within the range of valid memory -- because of
	** the masking of bits above and the limit of 4k on the length.
	*/

	if ( (tmp < (MAX_SDRAM_VALUE - SEC2_WE_SG_SIZE))
	     || ((tmp + size) < MAX_SDRAM_VALUE)  )
		return 0 ;
	return -1 ;
}

static
void sec_init_queues(void)
{
	int i ;
	struct workq *wq ;
	struct compq *cq ;

	for (i = 0 ; i < HW_NR_COMP_QUEUES ; i++)
	{
		cq = &sec_comp_queues[i] ;

		cq->cq_regs = & sec2_regs->cq[i] ;
		cq->hw_ptr  = (unsigned int*)KSEG1ADDR(&ptr_values_c[i]) ;
		cq->base    = comp_queues[i] ;
		cq->mask    = SEC_COMP_Q_SIZE - 1 ;
		cq->out_ptr = 0 ;

		*cq->hw_ptr = 0 ;

		cq->cq_regs->avail_addr = cq->hw_ptr ;
		cq->cq_regs->base       = cq->base ;
		cq->cq_regs->size       = SEC_COMP_Q_SIZE ;
		cq->cq_regs->in         = 0 ;
		cq->cq_regs->out        = 0 ;
	}

	for (i = 0 ; i < HW_NR_WORK_QUEUES ; i++)
	{
		wq = &sec_work_queues[i] ;

		init_waitqueue_head(&wq->space_wait) ;

		wq->wq_regs       = &sec2_regs->wq[i] ;
		wq->hw_ptr        = (unsigned int*)KSEG1ADDR(&ptr_values_w[i]);
		wq->base          = work_queues[i] ;
		wq->mask          = SEC_WORK_Q_SIZE - 1 ;
		wq->in_ptr        = 0 ;
		wq->in_ptr_tmp    = 0 ;
		wq->out_ptr       = 0 ;
		wq->building_desc = 0 ;
		wq->low_water     = wq->mask >> 1 ; /* wake when half full */
		wq->mask_ints     = 1 ;

		*wq->hw_ptr       = 0 ;

		wq->wq_regs->avail_addr = wq->hw_ptr ;
		wq->wq_regs->base       = wq->base ;
		wq->wq_regs->size       = SEC_WORK_Q_SIZE ;
		wq->wq_regs->in         = 0 ;
		wq->wq_regs->out        = 0 ;
		
	}
	debug_dump_sec_regs() ;
}

int __init
msp_secv2_init(void)
{
	int irqval ;
	int rc = 0 ;

	switch (identify_sec())
	{
	case SEC_POLO:
		break ;
	default:
#ifndef CONFIG_BRECIS_SEC_V1
		/* only supports "POLO" hardware */
		return -ENXIO ;
#else
		return 0 ;
#endif
	}

	v2_hw_present = 1 ;

	sec2_regs->rst |= SEC2_RST_MASTER ;
	NanoDelay(500) ;

	/* start random number generator */
	*(unsigned int *)SEC_RNG_CNF = 0x00010000 ;
	*(unsigned int *)SEC_RNG_CNF = 0x00000101 ;

	DBG_SEC("========================= Loading Driver ================\n");
	sec_init_queues() ;

#ifndef SIMULATE_V2_HW
	irqval = request_irq( BRECISINT_SEC, msp_secv2_interrupt,
			      SA_SAMPLE_RANDOM, "brecis_sec",
			      (void *)SEC2_BASE ) ;

	if (irqval)
	{
		printk( KERN_WARNING "%s: unable to get IRQ %d (rc=%d).\n",
			"brecis_sec", BRECISINT_SEC, irqval) ;
	}
	else
		irqvalid = 1 ;
	irqval = request_irq( 7, msp_secv2_interrupt,
			      SA_SAMPLE_RANDOM, "brecis_sec",
			      (void *)SEC2_BASE ) ;

	if (irqval)
	{
		printk( KERN_WARNING "%s: unable to get IRQ %d (rc=%d).\n",
			"brecis_sec", BRECISINT_SEC, irqval) ;
	}
	else
		irqvalid = 1 ;
	sec2_regs->ier |= SEC2_INT_MBX_ENABLE ;

#else
	irqval = 0 ;
	if(irqval) msp_secv2_interrupt(12, 0, 0) ;

	Des->interface_control = DIFC_DES_RST ;
	Hsh->interface_control = HIFC_HSH_RST ;

#endif
#ifdef VERIFICATION_TESTS
	verification_tests() ;
#endif
	return rc ;
}

void
msp_secv2_exit(void)
{
	if (irqvalid)
	{
		free_irq( BRECISINT_SEC, (void *)SEC2_BASE) ;
		free_irq( 7, (void *)SEC2_BASE) ;
		irqvalid = 0 ;
	}
}

static void
msp_secv2_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	/* static int status0count ; */
	int tmp ;
	int rc ;
	/*
	** XXX Fixme - This clears all interrupts, and assumes
        ** what the cause was.
	*/
	unsigned int status ;
	status = sec2_regs->sis ;
	sec2_regs->sis = /* ~status */ 0 ; 

	/* clear the mailbox */
	tmp = *(int*)0xBC000078 ;

	DBG_SEC("interrupt irq %d status was %x mbox %x\n",
	       irq, status, tmp) ;

	rc = msp_sec2_poll_completion_queues(3, 0) ;
	if (rc == -1) {
		printk("*** interrupt with no completion queue entries,\n") ;
		printk("*** pointer(s) read were 0x%x 0x%x\n",
		       sec_debug_counters[2], sec_debug_counters[3]) ;
		dump_sec_regs() ;
	}

}

/*------------------------------------------------------------
** Support for "special" callback functions (CBK_NONE, CBK_POLL,
** and CBK_SLEEP).  We provide the callback functions to do the
** dirty work.  In the cases that wait for the operation to
** finish before returning from msp_sec2_end_request (that is,
** POLL and SLEEP), we need more information than the single
** int storage space provided by the caller.  In those cases,
** msp_sec2_end_request will create a struct sec_wait_info on
** the stack, and we'll use that in these routines.
*/

struct sec_wait_info {
	wait_queue_head_t wait_q ;
	int event_happened ;
	int *pstatus ;
} ;

#define INIT_SEC_WAIT_INFO(wi, pstatus) \
	do { \
		init_waitqueue_head(&(wi)->wait_q) ; \
		(wi)->event_happened = 0 ; \
		(wi)->pstatus = (pstatus) ; \
	} while (0) 

static void
callback_handle_none(void *parm, unsigned int status)
{
	unsigned int *ip = (unsigned int *)parm ;
	if (ip)
		*ip = status ;
}

static void
callback_handle_poll(void *parm, unsigned int status)
{
	struct sec_wait_info *wi = (struct sec_wait_info *)parm ;
	if (wi) {
		*wi->pstatus = status ;
		wi->event_happened = 1 ;
	}
}

static void
callback_handle_sleep(void *parm, unsigned int status)
{
	struct sec_wait_info *wi = (struct sec_wait_info *)parm ;
	if (wi) {
		*wi->pstatus = status ;
		wi->event_happened = 1 ;
		wake_up(&wi->wait_q) ;
	}
}

/*
** msp_sec2_set_q_options:  set options for a queue
**
** input:
**    work_q           Which queue to put this request on
**    opt              Which option to set
**    val              What value to set the option to
**
** returns:
**    0 -- success.
**    1 -- unknown option number 'opt'
**    2 -- Value 'val' is out of range for option 'opt'
*/

int
msp_sec2_set_q_options(		int work_q,
				int opt,
				int val )
{
	struct workq *wq = &sec_work_queues[work_q] ;

	switch (opt) {
	case WQ_OPT_MASKINT:
		if ( (unsigned int)val < 2)
			wq->mask_ints = val ;
		else
			return 2 ;
		break ;

	default:
		return 1 ;
	}
	return 0 ;
}

/*
** msp_sec2_new_request:  start adding a new request to a work queue
**
** input:
**    work_q           Which queue to put this request on
**    n_sg_entries     How many Scatter/Gather entries this request will use
**    sa               Security association to use with this request
**    control          Control word to use for this request
**                         (descriptor size is set by these functions, though)
**    callback_fn      function to call when operation completes
**    cbk_prm          parameter passed to callback function
**    block            BLK_NONE, BLK_POLL, or BLK_SLEEP -- return, poll,
**                         or sleep when no space available
**
** returns:
**    0 -- success.
**    other - failure: not enough space or other failure
**
** note:
**    This is the only time space is checked for (with
**    n_sg_entries).  You must not call msp_sec2_add_sg() more times
**    than you indicated when you call this function.  You may,
**    however, call it fewer than n_sg_entries; the descriptor will
**    be adjusted.
*/

int
msp_sec2_new_request(	int work_q, 
			int n_sg_entries ,
			MSP_SEC2_SA *sa, 
			int control,
			void (*callback_fn)(void *,unsigned int),
			void *cbk_prm,
			int block) 

{
	struct workq *wq = &sec_work_queues[work_q] ;
	int entry_offset ;
	unsigned int flags = 0 ;
#ifdef DEBUG
	int spacetry = 0 ;
#endif

	if (! v2_hw_present)
		return -ENXIO ;

	if (wq->mask_ints)
	{
		save_flags(flags) ;
		cli() ;
	}

	if (wq->building_desc)
	{
#ifdef DEBUG
		printk("SECURITY2 - msp_sec2_new_request() while already active\n") ;
#endif
		if (wq->mask_ints)
			restore_flags(flags) ;
		return -EBUSY ;	/* failure */
	}
	wq->building_desc = 1 ;

	if (wq->mask_ints)
		wq->old_mask = flags ;

	if (callback_fn == CBK_NONE )
	{
		callback_fn = callback_handle_none ;
		wq->cbk_special = 0 ;
	}
	else if (callback_fn == CBK_POLL)
	{
		callback_fn = callback_handle_poll ;
		wq->cbk_special = (int) CBK_POLL ;
	}
	else if (callback_fn == CBK_SLEEP)
	{
		callback_fn = callback_handle_sleep ;
		wq->cbk_special = (int) CBK_SLEEP ;

		/* if sleeping, *must* use interrupt to wake up */
		control |= SEC2_WE_CTRL_GI ;
	}
	else
		wq->cbk_special = 0 ;
	
	/*
	** work queue header is 4 * 32 bits long.  I know this is
	** hard-coded, but so is the creation of those 4 32 bit
	** pieces below.
	*/

	while ( WQ_AVAIL_SPACE(wq) <= ( (4 * 4) + n_sg_entries * 8) )
	{
		debug_dump_sec_regs() ;
		if (block == BLK_SLEEP)
		{
			DBG_SEC("No space in queue; sleeping for space\n") ;

			sleep_on(&wq->space_wait) ;
		}
		else if (block == BLK_POLL)
		{
#ifdef DEBUG
			if (spacetry++ % 128 == 0)
				DBG_SEC("No queue space; polling "
					"(in %x out %x)\n",
					wq->in_ptr, wq->out_ptr) ;
#endif
			/* Must poll both completion queues, because */
			/* we don't know which one might be holding us */
			msp_sec2_poll_completion_queues(3, 1) ;
		}
		else
		{
			DBG_SEC("No space in queue, returning\n") ;
			if (wq->mask_ints)
				restore_flags(wq->old_mask) ;
			wq->building_desc = 0 ;
			return -ENOSPC ;	/* no space */
		}
	}

	wq->first_gather = 1 ;
	wq->last_gather = -1 ;
	wq->prev_gather = -1 ;
	wq->first_scatter = 1 ;
	wq->last_scatter = -1 ;
	wq->sa = sa ;

#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
	wq->g_tot = 0 ;
	wq->s_tot = 0 ;
	wq->g_len = 0 ;
#endif
	wq->sanity_checks_failed = 0 ;

	wq->in_ptr_tmp = wq->in_ptr ;

	entry_offset = wq->in_ptr_tmp ;

	/* first word of descriptor header: SA address */
	WQ_TMP_PUT_INT(wq, sa) ;

	/*
	** 2nd word of descriptor: 16 bits ESP trailer,
	** 8 bits mode, 8 bits descriptor size
	*/
	WQ_TMP_PUT_INT(wq, control) ;

	/*
	** 3rd word of descriptor: software use -- callback function
	*/
	WQ_TMP_PUT_INT(wq, (unsigned int)callback_fn) ;

	/*
	** 4th word of descriptor: software use -- offset of workq entry,
	** with work q number in high 16 bits
	*/
	WQ_TMP_PUT_INT(wq, work_q << 16 | entry_offset) ;

	/*
	** and put the callback parameter some place where we can
	** find it
	*/
	*(void**)&work_queues_extra[work_q][entry_offset] = cbk_prm ;

	DBG_SEC("Setting Callback parm at extra[%x][%x] = %p\n",
			       work_q, entry_offset, cbk_prm) ;

	return 0 ;		/* success */
}

void
msp_sec2_add_sg(	int work_q, 
			int scatter,
			void * address,
			int size )
{
	struct workq *wq = &sec_work_queues[work_q] ;
	unsigned int i ;

	/* first word, address */
	WQ_TMP_PUT_INT(wq, address) ;

	/* second word, flags and size */

	i = size ;

	if (i & ~SEC2_WE_SG_SIZE)
	{
		DBG_SEC("scatter gather with bad size\n") ;
		wq->sanity_checks_failed = 1 ;
		return ;
	}
#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
	if (sg_sanity_check(address, size) != 0)
	{
		DBG_SEC("sg sanity check failed\n") ;
		wq->sanity_checks_failed = 1 ;
		return ;
	}
#endif


	if (scatter)
	{
#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
		wq->s_tot += size ;
#endif
		
		i |= SEC2_WE_SG_SCATTER ;

		wq->last_scatter = wq->in_ptr_tmp ;

		if (wq->first_scatter)
		{
			wq->first_scatter = 0 ;
			i |= SEC2_WE_SG_SOP ;
		}
	}
	else
	{
#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
		wq->g_tot += size ;
		wq->g_len = size ;
#endif
		wq->prev_gather = wq->last_gather ;
		wq->last_gather = wq->in_ptr_tmp ;
		
		if (wq->first_gather)
		{
			wq->first_gather = 0 ;
			i |= SEC2_WE_SG_SOP ;
		}
	}

	WQ_TMP_PUT_INT(wq, i) ;
}

void
msp_sec2_end_request( int work_q )
{
	struct workq *wq = &sec_work_queues[work_q] ;
	int cbk_special ;
	struct sec_wait_info wi ;
	int control ;
	int insane = 0 ;

	/*
	** "Polo Work Element rev 1.0 8/22/02"
	**    says descriptor size is "(#words - 1)"
	*/
	int size = (wq->in_ptr_tmp - wq->in_ptr) & wq->mask ;

	int ofst = (wq->in_ptr + 4) & wq->mask ;

	control = *(int *) (wq->base + ofst) |= (size/4) - 1 ;

#ifdef CONFIG_BRECIS_SEC_V2_CKPARM
	insane = sanity_check(wq->sa, wq->g_tot, wq->g_len,
				    wq->s_tot, control) ;
#endif
	if (insane || wq->sanity_checks_failed)
	{
		msp_sec2_abort_request(work_q) ;
		if(insane)
			printk(KERN_WARNING
			       "*** Security parameter check failed %d\n",
			       insane) ;
		else
			printk(KERN_WARNING "*** Security sg check failed\n") ;
		return ;
	}

	
	/* Add "EOP" and "EOD" to last scatter entry */
	if (wq->last_scatter != -1)
		*(int *) (wq->base + wq->last_scatter)
			|= SEC2_WE_SG_EOP | SEC2_WE_SG_EOD ;

	/* Add "EOP" to last gather entry */
	if (wq->last_gather != -1)
		*(int *) (wq->base + wq->last_gather) |= SEC2_WE_SG_EOP ;

	/* Add "EOD" to last gather entry, for not ESP incoming */
	if ( ( ((wq->sa->flags & SAFLG_MODE_MASK) != SAFLG_MODE_ESP_IN)
	       || ((wq->sa->flags & SAFLG_HASH_MASK) == SAFLG_HASHNULL) )
	     && (wq->last_gather != -1) )
		*(int *) (wq->base + wq->last_gather) |= SEC2_WE_SG_EOD ;
		
	/* Add "EOD" to previous gather entry, for ESP incoming */
	else if ( ((wq->sa->flags & SAFLG_MODE_MASK) == SAFLG_MODE_ESP_IN)
		  && (wq->prev_gather != -1) )
		*(int *) (wq->base + wq->prev_gather) |= SEC2_WE_SG_EOD ;

	/*
	** XXX perhaps we want to put in a sort for the sg list, if
	** it's more than so many elements long?
	*/

	DBG_SEC("Giving workq entry to engine at %d - %d\n",
		wq->in_ptr, wq->in_ptr_tmp) ;
	dump_wq_entry(wq) ;
#ifdef DEBUG_VERBOSE
	if (wq->in_ptr_tmp < wq->in_ptr || 1)
	{
		dump_sec_regs() ;
	}
#endif

	/*
	** Hold onto a copy of cbk_special past the release of interrupts
	** below
	*/
	cbk_special = wq->cbk_special ;

	/*
	** If the sleep or poll "special" callback function is in use,
	** we need to substitute a pointer to the waitinfo struct
	** for the callback parameter.
	*/
	if (cbk_special)
	{
		int *pstatus ;
		
		int entry_offset = wq->in_ptr ;

		pstatus = *(int **)&work_queues_extra[work_q][entry_offset] ;

		*(void **) &work_queues_extra[work_q][entry_offset]
			= (void *) &wi ;

		DBG_SEC("Resetting Callback parm at extra[%x][%x] to %p\n",
			work_q, entry_offset, &wi) ;

		INIT_SEC_WAIT_INFO(&wi, pstatus) ;
	}

	*MEM_CNFG1_REG ; /* ensure processor completes write b4 hw read */
	/* hand off to hardware */
	wq->wq_regs->in = wq->in_ptr = wq->in_ptr_tmp ;

	sec_type_counters[wq->sa->flags & SAFLG_MODE_MASK] ++ ;

	wq->building_desc = 0 ;

	if(wq->mask_ints)
	{
		restore_flags(wq->old_mask) ;
	}

	sec_debug_counters[0]++ ;
#ifdef SIMULATE_V2_HW
	do_simulate_v2_hardware() ;	
#endif

	/* now, if requested, do a poll or sleep */
	if (cbk_special == 0)
		return ;
	else if (cbk_special == (int) CBK_POLL)
	{
		/* polling */
		int i = 0 ;
		int rc ;
		int cq = (control & SEC2_WE_CTRL_CQ) ? 2 : 1 ;
		while (wi.event_happened == 0)
		{
			rc = msp_sec2_poll_completion_queues(cq, 0) ;
			if (rc == -1 && i++ > 10000000)
			{
				printk(KERN_ERR "******** SECURITY OPERATION TIMED OUT ******\n") ;
				dump_sec_regs() ;
				break ;
			}
		}
	}
	else /* only option left: (cbk_special == (int) CBK_SLEEP) */
	{
		/* sleeping */
		wait_event(wi.wait_q, wi.event_happened == 1) ;
	}
}

void
msp_sec2_abort_request( int work_q )
{
	struct workq *wq = &sec_work_queues[work_q] ;

	if(wq->mask_ints)
	{
		restore_flags(wq->old_mask) ;
	}

	wq->building_desc = 0 ;
}

int
msp_sec2_set_hmac_key(		MSP_SEC2_SA *sa,
				unsigned char *key,
				int keylen,
				int workq,
				int compq,
				int sleep)
{
	int hashlen ;
	unsigned char hmac_key[HSH_BLK] ;
	MSP_SEC2_SA tmp_sa ;
	int rc ;
	int control ;
	void (*cbk_fn)(void*, unsigned int) ;
	int status ;
	int i ;

	if ( (unsigned int) workq > 1 )
		return -1 ;
	if ( (unsigned int) compq > 1 )
		return -1 ;

	memset(&tmp_sa, 0, sizeof(tmp_sa)) ;

	control = 0 ;
	if(sleep) control |= SEC2_WE_CTRL_GI ; /* generate interrupt */
	if(compq) control |= SEC2_WE_CTRL_CQ ; /* completion queue 1 */

	cbk_fn = CBK_POLL ;
	if(sleep)
		cbk_fn = CBK_SLEEP ;

	switch(sa->flags & SAFLG_HASH_MASK)
	{
	case SAFLG_MD5_96:
		tmp_sa.flags = SAFLG_MODE_HASH | SAFLG_MD5 ;
		hashlen = HSH_LEN_MD5 ;
		break ;
	case SAFLG_MD5:
		tmp_sa.flags = SAFLG_MODE_HASH | SAFLG_MD5 ;
		hashlen = HSH_LEN_MD5 ;
		break ;
	case SAFLG_SHA1_96:
		tmp_sa.flags = SAFLG_MODE_HASH | SAFLG_SHA1 ;
		hashlen = HSH_LEN_SHA1 ;
		break ;
	case SAFLG_SHA1:
		tmp_sa.flags = SAFLG_MODE_HASH | SAFLG_SHA1 ;
		hashlen = HSH_LEN_SHA1 ;
		break ;
	default:
		return -1 ;
	}

	/*
	** For reference, the computation is:
	**    H(K XOR opad, H(K XOR ipad, text))
	** where H is the hash fn;
	**       K is key if it is short enough,
	**           else H(key) if key is > hash block length
	**       ipad is 0x36 repeated
	**       opad is 0x5c repeated
	*/

	if (keylen <= HSH_BLK)
	{
		for (i = 0 ; i < keylen; i++)
			hmac_key[i] = key[i] ^ 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;
	}
	else
	{
		/* hash down to smaller value */
		tmp_sa.flags -- ; /* change mode to SAFLG_MODE_HASH_PAD */

		rc = msp_sec2_new_request(workq,
					  2, /* n_sg_entries */
					  &tmp_sa,
					  control,
					  cbk_fn,
					  &status, /* cbk_prm */
					  /* do or don't block */
					  sleep ? BLK_SLEEP : BLK_POLL 
					  ) ;
			
		
		if (rc != 0)
			return -1 ;
		msp_sec2_add_sg(workq,
				SG_GATHER,
				key, /* address */
				keylen /* size */
				) ;

		msp_sec2_add_sg(workq,
				SG_SCATTER,
				hmac_key, /* address */
				hashlen	/* size */
				) ;

		msp_sec2_end_request(workq) ;

		if (status)
		{
			DBG_SEC("status 0x%x from hash in hmac preprocess\n",
				status) ;
			return -1 ;
		}

		keylen = hashlen ;

		for (i = 0 ; i < keylen; i++)
			hmac_key[i] ^= 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;

		tmp_sa.flags ++ ; /* return to SAFLG_MODE_HASH */

	}

	/*
	** hmac_key now has (K XOR ipad).
	** Compute H(K XOR ipad).
	*/
	
	rc = msp_sec2_new_request(workq,
				  2, /* n_sg_entries */
				  &tmp_sa,
				  control,
				  cbk_fn,
				  &status, /* cbk_prm */
				  /* do or don't block */
				  sleep ? BLK_SLEEP : BLK_POLL 
				  ) ;
	if (rc != 0)
		return -1 ;

	msp_sec2_add_sg(workq,
			SG_GATHER,
			hmac_key, /* address */
			HSH_BLK /* size */
			) ;

	msp_sec2_add_sg(workq,
			SG_SCATTER,
			sa->hash_chain_a, /* address */
			hashlen	/* size */
			) ;

	msp_sec2_end_request(workq) ;

	if (status)
	{
		DBG_SEC("status 0x%x from hash in hmac preprocess(2)\n",
			status) ;
		return -1 ;
	}

	/*
	** Create (K XOR opad) from ( (K XOR ipad) XOR (ipad XOR opad) )
	*/
	for (i = 0 ; i < (HSH_BLK/4) ; i++)
		((unsigned long *)hmac_key)[i] ^= 0x6A6A6A6AUL ;

	/*
	** hmac_key  has (K XOR opad)
	** Compute H(hmac_key)
	*/
	
	rc = msp_sec2_new_request(workq,
				  2, /* n_sg_entries */
				  &tmp_sa,
				  control,
				  cbk_fn,
				  &status, /* cbk_prm */
				  /* do or don't block */
				  sleep ? BLK_SLEEP : BLK_POLL 
				  ) ;
	if (rc != 0)
		return -1 ;

	msp_sec2_add_sg(workq,
			SG_GATHER,
			hmac_key, /* address */
			HSH_BLK /* size */
			) ;

	msp_sec2_add_sg(workq,
			SG_SCATTER,
			sa->hash_chain_b, /* address */
			hashlen	/* size */
			) ;

	msp_sec2_end_request(workq) ;

	if (status)
	{
		DBG_SEC("status 0x%x from hash in hmac preprocess(2)\n",
			status) ;
		return -1 ;
	}

	sa->hash_init_len[0] = 0x00000000 ;
	sa->hash_init_len[1] = 0x00000200 ;

	return 0 ;
}

int
msp_sec2_set_aes_decrypt_key(	MSP_SEC2_SA *sa,
				int workq,
				int compq,
				int sleep)
{
	MSP_SEC2_SA tmp_sa ;
	static char junk_buf[16] ;
	int rc ;
	int control ;
	void (*cbk_fn)(void*, unsigned int) ;
	int status ;

	if ( (unsigned int) workq > 1 )
		return -1 ;
	if ( (unsigned int) compq > 1 )
		return -1 ;

	memset(&tmp_sa, 0, sizeof(tmp_sa)) ;

	control = SEC2_WE_CTRL_AKO ; /* AES Key Out */
	if(sleep) control |= SEC2_WE_CTRL_GI ; /* generate interrupt */
	if(compq) control |= SEC2_WE_CTRL_CQ ; /* completion queue 1 */

	cbk_fn = CBK_POLL ;
	if(sleep)
		cbk_fn = CBK_SLEEP ;

	tmp_sa.flags = sa->flags & SAFLG_CRYPT_TYPE_MASK ;

	/* MUST be AES type */
	if (tmp_sa.flags < SAFLG_AES_128)
		return -1 ;
	
	tmp_sa.flags |=  SAFLG_MODE_CRYPT | SAFLG_ECB ;

	memcpy(tmp_sa.crypt_keys, sa->crypt_keys, sizeof(sa->crypt_keys)) ;

	rc = msp_sec2_new_request(workq,
				  2, /* n_sg_entries */
				  &tmp_sa,
				  control,
				  cbk_fn,
				  &status, /* cbk_prm */
				  /* do or don't block */
				  sleep ? BLK_SLEEP : BLK_POLL 
				  ) ;
	
	if (rc != 0)
		return -1 ;

	msp_sec2_add_sg(workq,
			SG_GATHER,
			junk_buf, /* address */
			16 /* size */
			) ;

	msp_sec2_add_sg(workq,
			SG_SCATTER,
			sa->crypt_keys, /* address */
			32	/* size -- ALWAYS 32 for SEC2_WE_CTRL_AKO */
			) ;

	msp_sec2_end_request(workq) ;

	if (status)
	{
		DBG_SEC("status 0x%x from hash in hmac preprocess(2)\n",
			status) ;
		return -1 ;
	}

	return 0 ;

}



void
invalidate_cache_and_free_we(int work_q, int we_ofst)
{
	struct workq *wq = &sec_work_queues[work_q] ;
	unsigned int count ;
	void *address ;
	unsigned int flags ;
	int orig_ofst ;
	int our_out ;
	int hw_out ;

	/* save a copy of we_ofst for later */
	orig_ofst = we_ofst ;

	/*
	** set bit 0 of SA field, so we can tell we're done
	** with this work entry
	*/
	*(int *)(wq->base + we_ofst) |= 1 ;

	/* get count of entries */
	count = (*(unsigned int *)(wq->base + we_ofst + 4)) & 0xff ;
	count -= 3 ;

	/* skip software use words, so we go to s/g entries */
	we_ofst = (we_ofst + 16) & wq->mask ;

	while (count)
	{
		count -= 2 ;

		address = *(void **)(wq->base + we_ofst) ;
		flags   = *(unsigned int*)(wq->base + we_ofst + 4) ;

		if (flags & SEC2_WE_SG_SCATTER)
		{
			flags &= 0x1FFF ;	/* size of entry */
			invalidate_buffer(address, flags) ;
		}

		we_ofst = (we_ofst + 8) & wq->mask ;
	}

	/*
	** Now, move our local copy of the work queue out pointer
	** forward until we reach either the hardware's out pointer
	** or an entry that's not marked as completed
	*/
	
	/* XXX This will change when the hw pointer is written to RAM */
	hw_out = *wq->hw_ptr ;
	our_out = wq->out_ptr ;
	
	/*
	** save some time.  If this was the first entry
	** in the queue, we've already know it's free, have
	** read the size, and know where to move forward to (we_ofst
	** has been moved forward to the next descriptor)
	*/
	if (our_out == orig_ofst)
	{
		our_out = we_ofst ;
	}

	while (our_out != hw_out)
	{
		if ( ((*(int *)(wq->base + our_out)) & 1) == 0 )
			break ;
		count = (*(int *)(wq->base + our_out + 4)) & 0xff ;
		count = (count + 1) * 4 ;
		our_out = (our_out + count) & wq->mask ;
	}

	wq->out_ptr = our_out ;
}

int
msp_sec2_poll_completion_queues(int queue_mask, int max_req)
{
	int q ;
	struct compq *cq ;
	int work_ct = 0 ;
	int in ;
	int flags ;
	
	/*
	** 0 means no max value ; MAXINT is close enough for
	** all practical purposes.
	*/
	   
	if (max_req == 0)
		max_req-- ;

	cq = sec_comp_queues ;
	for (q = 0 ; q < HW_NR_COMP_QUEUES; q++, cq++)
	{
		if ( ! (queue_mask & 1 << q) )
			continue ;
		sec_debug_counters[1]++ ;


		save_flags(flags) ;
		cli() ;

		*MEM_CNFG1_REG ; /* ensure hardware completes write b4 read */
		while ( (( (in = *cq->hw_ptr) - cq->out_ptr)&cq->mask)
			>= 4*4)
		{
			unsigned int ofst ;
			unsigned int we_ofst ;
			unsigned int work_q ;
			unsigned int status ;
			void (*cbk_fn)(void *, unsigned int) ;
			void *cbk_parm ;
			
			sec_debug_counters[2]++ ;

			/*
			** If we've already exhausted max_req,
			** indicate we're returning with work
			** still to do.
			*/
			if (max_req == 0) {
				msp_sec2_poll_work_queue_space() ;
				restore_flags(flags) ;
				return 1 ;
			}

			DBG_SEC("Getting compq entry from engine at %d\n", cq->out_ptr) ;
			dump_cq_entry(cq) ;
#ifdef DEBUG_VERBOSE
			dump_sec_regs() ;
#endif

			ofst = cq->out_ptr ;

			invalidate_buffer(cq->base+ofst, 16) ;

			cbk_fn = * ( (void (**)(void*,unsigned int))
				     (cq->base + ofst)) ;
			ofst = (ofst + 4) & cq->mask ;
			
			we_ofst = * (int *) (cq->base + ofst) ;
			work_q = we_ofst >> 16 ;
			we_ofst &= 0xffff ;

			cbk_parm = *(void **)
				(&work_queues_extra[work_q][we_ofst]) ;

			DBG_SEC("Callback parm at extra[%x][%x] = %p\n",
			       work_q, we_ofst, cbk_parm) ;

			ofst = (ofst + 4) & cq->mask ;
			
			status = *(unsigned int *) (cq->base + ofst) ;
			/* skip status and reserved word */
			ofst = (ofst + 8) & cq->mask ;

			/* store new offset to hardware and to software copy */
			cq->cq_regs->out = cq->out_ptr = ofst ;

			status &=
				(SEC2_INT_BAD_ADDR |
				 SEC2_INT_HASH_NON_64 |
				 SEC2_INT_DES_NON_8 |
				 SEC2_INT_AES_NON_16 |
				 SEC2_INT_BAD_GATHER |
				 SEC2_INT_ICV_COMP_ERR |
				 SEC2_INT_OFFSET_ERR |
				 SEC2_INT_GS_BALANCE_ERR) ;
			/* XXX FIXME  ICV_COMP_ERR showing up erroneously */
			status &= ~SEC2_INT_ICV_COMP_ERR ;

			DBG_SEC("Invalidating cache\n") ;
			invalidate_cache_and_free_we(work_q, we_ofst) ;

			/* call completion callback routine */
			DBG_SEC("calling callback fn\n") ;
			cbk_fn(cbk_parm, status) ;

			DBG_SEC("callback fn returns\n") ;
			max_req-- ;
			work_ct++ ;
			
		}
		restore_flags(flags) ;
		sec_debug_counters[2+q] = in ;
	}
	if (work_ct) {
		msp_sec2_poll_work_queue_space() ;
		return 0 ;	/* work done, now empty */
	}

	return -1 ;		/* there was nothing to do */
}

void
msp_sec2_wait_for_space( int work_q )
{
	struct workq *wq = &sec_work_queues[work_q] ;

	sleep_on(&wq->space_wait) ;
}

static void
msp_sec2_poll_work_queue_space(void)
{
	int q ;
	struct workq *wq ;

	wq = sec_work_queues ;
	for (q = 0 ; q < HW_NR_WORK_QUEUES; q++, wq++)
	{
		if ( waitqueue_active(&wq->space_wait)
		     && (WQ_AVAIL_SPACE(wq) >= wq->low_water) )
			wake_up(&wq->space_wait) ;
	}
}

int
msp_secv2_read_proc(char *page, char **start, off_t off,
		    int count, int *eof, void *data)
{
	int len ;

	len = 0 ;

	len += sprintf(page+len, "Security operations, by type:\n") ;

	len += sprintf(page+len, "ESP in  %08X  ESP out %08X  HMAC    %08X\n",
		       sec_type_counters[0], sec_type_counters[1],
		       sec_type_counters[2]) ;
	len += sprintf(page+len, "HSH+pad %08X  Hash    %08X  Crypt   %08X\n",
		       sec_type_counters[3], sec_type_counters[4],
		       sec_type_counters[5]) ;
	return len ;
}

static inline int
do_user_wqe(MSP_SEC2_WQE * we)
{
	struct work_queue_sg *sg ;
	int i, rc ;
	int g_len, s_tot, g_tot ; /* last gather len & s/g totals */

	g_len = s_tot = g_tot = 0 ;
				
	DBG_SEC("do_user_wqe with %d sg entries\n", we->next_sge) ;

	rc = msp_sec2_new_request(we->wqn,
				  we->next_sge,
				  we->sa,
				  we->control | SEC2_WE_CTRL_GI,
				  CBK_SLEEP,
				  &we->status,
				  BLK_SLEEP) ;
	if (rc != 0)
		return -EINVAL ;

	for (i = 0, sg = we->sge ; i < we->next_sge; i++, sg++)
	{
		int sz = sg->size_flags & SEC2_WE_SG_SIZE;
					
		if (sg_sanity_check(sg->addr, sz) != 0)
		{
			DBG_SEC("do_user_wqe sg sanity check failed %d\n", rc);
			dump_wq_entry(&sec_work_queues[we->wqn]) ;
			msp_sec2_abort_request(we->wqn) ;
			return -EINVAL ;
		}

		if(sg->size_flags & SEC2_WE_SG_SCATTER)
		{
			s_tot += sz ;

			msp_sec2_add_sg(we->wqn,
					SG_SCATTER,
					sg->addr,
					sz) ;

		}
		else
		{
			g_tot += sz ;
			g_len = sz ;

			msp_sec2_add_sg(we->wqn,
					SG_GATHER,
					sg->addr,
					sz) ;
		}
	}

	if ( (rc = sanity_check(we->sa, g_tot, g_len, s_tot, we->control)) )
	{
		DBG_SEC("do_user_wqe sanity check failed %d\n", rc) ;
		dump_wq_entry(&sec_work_queues[we->wqn]) ;
		msp_sec2_abort_request(we->wqn) ;
		we->status = rc ;
		return -EINVAL ;
	}

	msp_sec2_end_request(we->wqn) ;

	return 0 ;
}
		
static inline int
do_user_set_hmac(MSP_SEC2_SET_HMAC_ARGS *args)
{
	if (msp_sec2_set_hmac_key(args->sa, args->key, args->keylen,
			      args->workq, args->compq, 1))
		return -EINVAL ;
	return 0 ;
}

static inline int
do_user_set_aes(MSP_SEC2_SET_AES_ARGS *args)
{
	if (msp_sec2_set_aes_decrypt_key(args->sa, args->workq,
					 args->compq, 1))
		return -EINVAL ;
	return 0 ;
}

int
msp_secv2_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{

	DBG_SEC("driver2 ioctl %x\n", cmd) ;
	
	if (v2_hw_present)
	{
		switch(cmd)
		{
		case MSP_SEC_CTLV2:
			return (do_user_wqe((MSP_SEC2_WQE *)arg)) ;
		case MSP_SEC_SET_HMAC:
			return do_user_set_hmac((MSP_SEC2_SET_HMAC_ARGS*)arg);
		case MSP_SEC_SET_AES:
			return do_user_set_aes((MSP_SEC2_SET_AES_ARGS*)arg);
		}
	}
	return -EINVAL ;
}


EXPORT_SYMBOL(sec_debug_counters) ;
EXPORT_SYMBOL(msp_sec2_new_request) ;
EXPORT_SYMBOL(msp_sec2_add_sg) ;
EXPORT_SYMBOL(msp_sec2_end_request) ;
EXPORT_SYMBOL(msp_sec2_abort_request) ;
EXPORT_SYMBOL(msp_sec2_poll_completion_queues) ;
EXPORT_SYMBOL(msp_sec2_set_hmac_key) ;
EXPORT_SYMBOL(msp_sec2_set_aes_decrypt_key) ;

static void
dump_sec_regs(void)
{
	int i ;
	
	printk("SECURITY " "%08x   sis (interupt status)\n", sec2_regs->sis) ;
	printk("SECURITY " "%08x   ier (interrupt enable)\n", sec2_regs->ier) ;
	printk("SECURITY " "%08x   esr (engine status)\n", sec2_regs->esr) ;
	for(i = 0 ; i < HW_NR_WORK_QUEUES; i++)
	{
		printk("SECURITY " "----------\n") ;
		printk("SECURITY " " %08x   wq%d hwptr_addr (%08x)\n",
			(int)sec2_regs->wq[i].avail_addr, i,
			*(int*)(0x80000000|((int)sec2_regs->wq[i].avail_addr))) ;
		printk("SECURITY " " %08x   wq%d avail\n",
			sec2_regs->wq[i].avail, i) ;
		printk("SECURITY " " %08x   wq%d base\n",
			(int)sec2_regs->wq[i].base, i) ;
		printk("SECURITY " " %08x   wq%d size\n",
			sec2_regs->wq[i].size, i) ;
		printk("SECURITY " " %08x   wq%d in\n",
			sec2_regs->wq[i].in, i) ;
		printk("SECURITY " " %08x   wq%d out\n",
			sec2_regs->wq[i].out, i) ;
	}

	for(i = 0 ; i < HW_NR_COMP_QUEUES; i++)
	{
		printk("SECURITY " "----------\n") ;
		printk("SECURITY " " %08x   cq%d hwptr_addr (%08x)\n",
			(int)sec2_regs->cq[i].avail_addr, i,
			*(int*)(0x80000000|((int)sec2_regs->cq[i].avail_addr))) ;
		printk("SECURITY " " %08x   cq%d avail\n",
			sec2_regs->cq[i].avail, i) ;
		printk("SECURITY " " %08x   cq%d base\n",
			(int)sec2_regs->cq[i].base, i) ;
		printk("SECURITY " " %08x   cq%d size\n",
			sec2_regs->cq[i].size, i) ;
		printk("SECURITY " " %08x   cq%d in\n",
			sec2_regs->cq[i].in, i) ;
		printk("SECURITY " " %08x   cq%d out\n",
			sec2_regs->cq[i].out, i) ;
	}
	printk("SECURITY " "----------\n") ;

}

#ifdef DUMP_WQ_ENTRIES
static void
dump_wq_entry(struct workq *wq)
{
	int idx, i ;
	unsigned int val ;
	unsigned int eng_mode ;
	MSP_SEC2_SA *sa ;

	idx = wq->in_ptr ;

	printk("Work descriptor:\n") ;

	printk("SA        ESP/MODE  (size)\n") ;

	val = *(unsigned int *)(wq->base + idx) ;
	idx = (idx + 4) & wq->mask ;
	sa = (MSP_SEC2_SA *) val ;

	printk("%08x  ", val) ;
	
	val = *(unsigned int *)(wq->base + idx) ;
	idx = (idx + 4) & wq->mask ;

	printk("%08x  ", val) ;
	printk("(%d)\n", val & 0xff) ;

	printk("Cbk-fn    cbk-prm\n") ;

	val = *(unsigned int *)(wq->base + idx) ;
	idx = (idx + 4) & wq->mask ;

	printk("%08x  ", val) ;

	val = *(unsigned int *)(wq->base + idx) ;
	idx = (idx + 4) & wq->mask ;

	printk("%08x\n", val) ;

	printk("SG entries:") ;

	i = 0 ;
	while(idx != wq->in_ptr_tmp)
	{
		if ( (i++ % 4) == 0 )
		{
			printk("\n") ;
		}
		printk("%08x  ", *(unsigned int *)(wq->base + idx)) ;
		idx = (idx + 4) & wq->mask ;
	}
	printk("\n") ;

	/* sometimes the sa is written to by the hardware */
	invalidate_buffer(sa, sizeof(*sa)) ;

	printk("SA struct:\n") ;
	printk(" flags     esp_spi   esp_seq\n") ;
	printk(" %08x  %08x  %08x\n",
	       sa->flags, sa->esp_spi, sa->esp_sequence) ;
	switch( (eng_mode = sa->flags & SAFLG_MODE_MASK) )
	{
	case SAFLG_MODE_ESP_IN:
		printk("   ESP_IN  ") ;
		break ;
	case SAFLG_MODE_ESP_OUT:
		printk("   ESP_OUT ") ;
		break ;
	case SAFLG_MODE_HMAC:
		printk("   HMAC    ") ;
		break ;
	case SAFLG_MODE_HASH_PAD:
		printk("  HASH+PAD ") ;
		break;
	case SAFLG_MODE_HASH:
		printk("  HASH     ") ;
		break;
	case SAFLG_MODE_CRYPT:
		printk("  CRYPT    ") ;
		break ;
	default:
		printk("*BAD*ENG*MODE*") ;
		break ;
	}

	if (eng_mode == SAFLG_MODE_ESP_OUT)
	{
		printk((sa->flags & SAFLG_SI)? " SI " : " NO_SI ") ;
		printk((sa->flags & SAFLG_CRI)? " CRI " : " NO_CRI ") ;
		printk((sa->flags & SAFLG_EM )? " EM " : "") ;
	}

	if (eng_mode == SAFLG_MODE_ESP_IN) 
		printk((sa->flags & SAFLG_CPI)? " CPI " : " NO_CPI ") ;
	
	if (eng_mode != SAFLG_MODE_CRYPT)
	{
		printk((sa->flags & SAFLG_CV )? " CV " : " NO_CV ") ;

		switch(sa->flags & SAFLG_HASH_MASK)
		{
		case SAFLG_MD5_96:
			printk(" MD5-96  ") ;
			break ;
		case SAFLG_MD5:
			printk(" MD5 ") ;
			break ;
		case SAFLG_SHA1_96:
			printk(" SHA1-96 ") ;
			break ;
		case SAFLG_SHA1:
			printk(" SHA1 ") ;
			break ;
		case SAFLG_HASHNULL:
			printk(" HSH_NULL ") ;
			break ;
		default:
			printk(" *BAD*HASH* ") ;
			break ;
		}
	}

	if ( (eng_mode <= SAFLG_MODE_ESP_OUT)
	     || (eng_mode == SAFLG_MODE_CRYPT) )
	{
		switch(sa->flags & SAFLG_CRYPT_TYPE_MASK)
		{
		case SAFLG_DES:
			printk(" DES-") ;
			break ;
		case SAFLG_3DES:
			printk(" 3DES-") ;
			break ;
		case SAFLG_AES_128:
			printk(" AES128-") ;
			break ;
		case SAFLG_AES_192:
			printk(" AES192-") ;
			break ;
		case SAFLG_AES_256:
			printk(" AES256-") ;
			break ;
		case SAFLG_CRYPTNULL:
			printk(" CRYPTNULL-") ;
			break ;
		default:
			printk(" BADCRYPT-") ;
			break ;
		}
		printk((sa->flags & SAFLG_DES_K1_DECRYPT)? "D" : "E" ) ;
		if( (sa->flags & SAFLG_CRYPT_TYPE_MASK) == SAFLG_3DES )
		{
			printk((sa->flags & SAFLG_DES_K2_DECRYPT)? "D" : "E" ) ;
			printk((sa->flags & SAFLG_DES_K3_DECRYPT)? "D" : "E" ) ;
		}

		switch(sa->flags & SAFLG_BLK_MASK)
		{
		case SAFLG_ECB:
			printk(" ECB\n") ;
			break ;
		case SAFLG_CTR:
			printk(" CTR\n") ;
			break ;
		case SAFLG_CBC_ENCRYPT:
			printk(" CBC-ENCRYPT\n") ;
			break ;
		case SAFLG_CBC_DECRYPT:
			printk(" CBC-DECRYPT\n") ;
			break ;
		case SAFLG_CFB_ENCRYPT:
			printk(" CFB-ENCRYPT\n") ;
			break ;
		case SAFLG_CFB_DECRYPT:
			printk(" CFB-DECRYPT\n") ;
			break ;
		case SAFLG_OFB:
			printk(" OFB\n") ;
			break ;
		default:
			printk(" BAD*BLOCK*MODE\n") ;
			break ;
		}
	}
	else
		printk("\n") ;

	printk(" hash_chain_a:") ;
	for(i = 0 ; i < 0x5 ; i++)
	{
		if ((i % 6) == 0)
		{
			printk("\n   %04x  ", i * 4) ;
		}
		printk("%08x  ", sa->hash_chain_a[i]) ;
	}
	printk("\n") ;

	printk(" hash_chain_b:") ;
	for(i = 0 ; i < 0x5 ; i++)
	{
		if ((i % 6) == 0)
		{
			printk("\n   %04x  ", i * 4) ;
		}
		printk("%08x  ", sa->hash_chain_b[i]) ;
	}
	printk("\n") ;

	printk(" encryption keys:") ;
	for(i = 0 ; i < 0x8 ; i++)
	{
		if ((i % 4) == 0)
		{
			printk("\n   %04x  ", i * 4) ;
		}
		printk("%08x ", sa->crypt_keys[i]) ;
	}
	printk("\n") ;

	printk(" Hash Initial Length:  %08x %08x\n",
	       sa->hash_init_len[0], sa->hash_init_len[1]) ;
	

	printk(" IV:") ;
	for(i = 0 ; i < 0x4 ; i++)
	{
		if ((i % 4) == 0)
		{
			printk("\n   %04x  ", i * 4) ;
		}
		printk("%08x ", sa->crypt_iv[i]) ;
	}
	printk("\n") ;

#if 0
	printk("proto_ip (UNUSED): ") ;

	for(i = 0 ; i < 0x5 ; i++)
	{
		if ((i % 4) == 0)
		{
			printk("\n %04x  ", i * 4) ;
		}
		printk("%08x  ", (sa->proto_ip[i]) ;
	}
	printk("\n") ;

#endif
}
#endif

#ifdef DUMP_CQ_ENTRIES
static void
dump_cq_entry(struct compq *cq)
{
	unsigned int *ip;

	ip = (unsigned int *)(cq->base + cq->out_ptr) ;

	printk(" -----> Completion entry  %08x %08x %08x %08x\n",
		ip[0], ip[1], ip[2], ip[3]) ;
}
#endif



#ifdef CONFIG_BRECIS_SEC_V1ON2_COMPAT

/*************************************************************
** Routines to provide compatibility with version 1 hardware
** api.
*/

/*
** These functions replace the msp_sec_do_*_operation functions from
** the original driver.
**
** The driver can check which security engine is present
** at runtime, and use these functions instead.
*/

#ifdef DEBUG_VERBOSE
static void
dump_descriptor(sec_desc *desc)
{
	while (desc)
	{
		printk("DESCR at %x\n", (int)desc) ;
		printk("DESCR:    nda %08x   len %08x  mode %08x\n",
		       desc->ioctl.nda, desc->ioctl.buflen, desc->ioctl.mode);
		printk("DESCR:    src %8p   dst %8p    id %08x\n",
		       desc->ioctl.srcaddr, desc->ioctl.dstaddr, desc->ioctl.id);
		printk("DESCR:    ivl %08x   ivh %08x  type %08x\n",
		       desc->ioctl.ivhigh, desc->ioctl.ivlow, desc->ioctl.type);
		printk("DESCR:    ch0 %08x   ch1 %08x   ch2 %08x\n",
		       desc->ioctl.chainvar[0],
		       desc->ioctl.chainvar[1],
		       desc->ioctl.chainvar[2]) ;

		printk("DESCR:    ch3 %08x   ch4 %08x   ch5 %08x\n",
		       desc->ioctl.chainvar[3],
		       desc->ioctl.chainvar[4],
		       desc->ioctl.chainvar[5]) ;

		printk("DESCR:    opt %08x  stat %08x   nxt %8p\n",
		       desc->ioctl.options, desc->ioctl.status, desc->next) ;

#ifdef DEBUG_VERBOSE
		invalidate_buffer(desc->ioctl.srcaddr, desc->ioctl.buflen) ;

		printk("DESCR: source data:\n") ;

		{
			int i ;
			char *DestBuf = (char *)desc->ioctl.srcaddr ;

			for(i = 0 ; i < desc->ioctl.buflen; i++)
			{
				switch(i % 16)
				{
				case 0:
					printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
					break;
				case 7:
					printk("%02x - ", DestBuf[i] & 0xff) ;
					break ;
				case 15:
					printk("%02x\n", DestBuf[i] & 0xff) ;
					break ;
				default:
					printk("%02x ", DestBuf[i] & 0xff) ;
					break ;
				}
			}
		}
	
#endif
		
		desc = desc->next ;
	}
}
#else
#define dump_descriptor(x)
#endif

void
msp_sec_do_des_operation_v2(sec_desc *desc)
{
	MSP_SEC2_SA sa ;	/* security association struct on stack */
	int mode ;
	int tmp ;
	int rc ;
	int count ;
	sec_desc *d1 ;

	mode = desc->ioctl.mode ;

	tmp = SAFLG_MODE_CRYPT ;

	if ( (mode & DMC_MOD) == DMC_MOD_PT)
	{
		tmp |= SAFLG_CRYPTNULL ;
	}
	else
	{
		if (mode & DMC_K1_DEC)
			tmp |= SAFLG_DES_K1_DECRYPT ;
		
		if (mode & DMC_3DES)
		{
			tmp |= SAFLG_3DES ;

			if (mode & DMC_K2_DEC)
				tmp |= SAFLG_DES_K2_DECRYPT ;
			if (mode & DMC_K3_DEC)
				tmp |= SAFLG_DES_K3_DECRYPT ;
		} else {
			/* not 3des, K1 mode must match CBC/CFB mode */
			if (mode & DMC_MOD_DEC)
			{
				tmp |= SAFLG_DES_K1_DECRYPT ;
			}
		}

		if ( (mode & DMC_MOD) == DMC_MOD_CBC )
		{
			tmp |= (mode & DMC_MOD_DEC) ?
				SAFLG_CBC_DECRYPT : SAFLG_CBC_ENCRYPT ;
		}
		else if ( (mode & DMC_MOD) == DMC_MOD_CFB)
		{
			tmp |= (mode & DMC_MOD_DEC) ?
				SAFLG_CFB_DECRYPT : SAFLG_CFB_ENCRYPT ;
		}
	}

	sa.flags = tmp ;

	memcpy((void *)sa.crypt_iv, &desc->ioctl.ivhigh, 8) ;

	memcpy((void *)sa.crypt_keys, &desc->ioctl.desc_key1high, 24) ;

	/*
	** figure out how many S/G entries we will have
	** Each entry in the old system counts as two
	** entries in the new system (one gather and one scatter)
	*/
	count = 1 ;
	d1 = desc ;
	while (d1->next)
	{
		count++ ;
		d1 = d1->next ;
	}

	rc = msp_sec2_new_request(0, /* work_q */
				  count * 2, /* n_sg_entries */
				  &sa, /* sa */
				  0, /* control */
				  CBK_POLL, /* callback_fn */
				  (void *)&desc->ioctl.status, /* cbk_prm */
				  BLK_POLL /* poll for space */
				  ) ;

	if (rc == 0)
	{
		d1 = desc ;
		while (d1 != NULL)
		{
			/* add a gather entry */
			msp_sec2_add_sg(0,  /* work queue */
					SG_GATHER,
					d1->ioctl.srcaddr, /* address */
					d1->ioctl.buflen /* size */
					) ;

			/* add scatter entry */
			msp_sec2_add_sg(0, /* work queue */
					SG_SCATTER,
					d1->ioctl.dstaddr, /* address */
					d1->ioctl.buflen /* size */
					) ;

			d1 = d1->next ;
		}

		msp_sec2_end_request(0) ;
	}
}

void
msp_sec_do_hsh_operation_v2(sec_desc *desc)
{
	int hashlen ;
	MSP_SEC2_SA sa ;
	int mode ;
	int tmp ;
	int rc ;
	int count ;
	sec_desc *d1 ;

	DBG_SEC("---------------------------------------\n") ;
	DBG_SEC("do_hsh_operation given this descriptor:\n") ;
	dump_descriptor(desc) ;
	DBG_SEC("---------------------------------------\n") ;

	mode = desc->ioctl.mode ;

	if (desc->ioctl.options & MSP_SEC_DO_PADDING)
		tmp = SAFLG_MODE_HASH_PAD ;
	else
		tmp = SAFLG_MODE_HASH ;

	if (mode & HMC_SHA1)
		tmp |= SAFLG_SHA1 ;
	else
		tmp |= SAFLG_MD5 ;

	if (mode & HMC_CRV)
	{
		tmp |= SAFLG_CV ;
		memcpy((void *)sa.hash_chain_a, desc->ioctl.chainvar, 20) ;
	}

	sa.flags = tmp ;
	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = 0 ;

	/*
	** figure out how many S/G entries we will have
	** Each entry in the old system counts as one
	** entries in the new system (gather entries),
	** plus one scatter entry for writing the results.
	*/
	count = 1 ;
	d1 = desc ;
	while (d1->next)
	{
		count++ ;
		d1 = d1->next ;
	}

	if (desc->ioctl.mode & HMC_SHA1)
		hashlen = HSH_LEN_SHA1 ;
	else
		hashlen = HSH_LEN_MD5 ;

	rc = msp_sec2_new_request(0, /* work_q */
				  count + 1, /* n_sg_entries */
				  &sa, /* sa */
				  0, /* control */
				  CBK_POLL, /* callback_fn */
				  (void *)&desc->ioctl.status, /* cbk_prm */
				  BLK_POLL /* poll for space */
				  ) ;

	if (rc == 0)
	{
		d1 = desc ;
		while (d1 != NULL)
		{
			/* add a gather entry */
			msp_sec2_add_sg(0,  /* work queue */
					SG_GATHER,
					d1->ioctl.srcaddr, /* address */
					d1->ioctl.buflen /* size */
					) ;
			d1 = d1->next ;
		}

		/* add scatter entry */
		msp_sec2_add_sg(0, /* work queue */
				SG_SCATTER,
				desc->ioctl.dstaddr, /* address */
				hashlen /* size */
				) ;

		msp_sec2_end_request(0) ;
	}
}

void
msp_sec_do_hmac_operation_v2(sec_desc *desc)
{
	int hashlen ;
	int keylen ;
	unsigned char hmac_key[HSH_BLK] ;
	sec_desc tmp_desc, *d1 ;
	MSP_SEC2_SA sa ;
	int count ;
	int i ;
	int mode ;
	int tmp ;
	int rc ;
	
	DBG_SEC("=======================================\n") ;
	DBG_SEC("do_HMAC_operation given this descriptor:\n") ;
	dump_descriptor(desc) ;
	DBG_SEC("=======================================\n") ;

	/*
	** For reference, the computation is:
	**    H(K XOR opad, H(K XOR ipad, text))
	** where H is the hash fn;
	**       K is key if it is short enough,
	**           else H(key) if key is > hash block length
	**       ipad is 0x36 repeated
	**       opad is 0x5c repeated
	*/

	if (desc->ioctl.mode & HMC_SHA1)
		hashlen = HSH_LEN_SHA1 ;
	else
		hashlen = HSH_LEN_MD5 ;

	/*
	** key is pointed to by ioctl struct;
	** text is in continue buffers
	*/

	/* for hmac, if key is longer than blocksize, hash it down to size */

	keylen = desc->ioctl.buflen ;

	if (keylen <= HSH_BLK)
	{
		for (i = 0 ; i < keylen; i++)
			hmac_key[i] = desc->ioctl.srcaddr[i] ^ 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;
	}
	else
	{
		
		memcpy(&tmp_desc, desc, sizeof(sec_desc)) ;
		tmp_desc.ioctl.dstaddr   = hmac_key ;
		tmp_desc.ioctl.options   = MSP_SEC_DO_PADDING ;
		tmp_desc.ioctl.status    = 0 ;
		tmp_desc.next            = 0 ;
		
		msp_sec_do_hsh_operation_v2(&tmp_desc) ;

		if (tmp_desc.ioctl.status)
		{
			desc->ioctl.status = tmp_desc.ioctl.status ;
			return ;
		}

		invalidate_buffer(hmac_key, hashlen) ;
		keylen = hashlen ;

		for (i = 0 ; i < keylen; i++)
			hmac_key[i] ^= 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;
	}

	/*
	** hmac_key now has (K XOR ipad).
	** Compute H(K XOR ipad).
	*/
	
	tmp_desc.ioctl.srcaddr   = hmac_key ;
	tmp_desc.ioctl.buflen    = HSH_BLK ;
	tmp_desc.ioctl.dstaddr   = (void *)sa.hash_chain_a ;
	tmp_desc.ioctl.options   = 0 ;
	tmp_desc.ioctl.status    = 0 ;
	tmp_desc.ioctl.mode      = desc->ioctl.mode ;
	tmp_desc.next            = 0 ;

	msp_sec_do_hsh_operation_v2(&tmp_desc) ;
	
	if (tmp_desc.ioctl.status)
	{
		desc->ioctl.status = tmp_desc.ioctl.status ;
		return ;
	}

	/*
	** Create (K XOR opad) from ( (K XOR ipad) XOR (ipad XOR opad) )
	*/
	for (i = 0 ; i < (HSH_BLK/4) ; i++)
		((unsigned long *)hmac_key)[i] ^= 0x6A6A6A6AUL ;

	/*
	** hmac_key  has (K XOR opad)
	** Compute H(hmac_key)
	*/
	
	tmp_desc.ioctl.srcaddr   = hmac_key ;
	tmp_desc.ioctl.buflen    = HSH_BLK ;
	tmp_desc.ioctl.dstaddr   = (void *)sa.hash_chain_b ;
	tmp_desc.ioctl.options   = 0 ;
	tmp_desc.ioctl.mode      = desc->ioctl.mode ;
	tmp_desc.ioctl.status    = 0 ;
	tmp_desc.next            = 0 ;

	msp_sec_do_hsh_operation_v2(&tmp_desc) ;

	if (tmp_desc.ioctl.status)
	{
		desc->ioctl.status = tmp_desc.ioctl.status ;
		return ;
	}

	mode = desc->ioctl.mode ;

	tmp = SAFLG_MODE_HMAC ;

	tmp |= SAFLG_CV ;
	
	sa.hash_init_len[0] = 0x00000000 ;
	sa.hash_init_len[1] = 0x00000200 ;

	if (mode & HMC_SHA1)
		tmp |= SAFLG_SHA1 ;
	else
		tmp |= SAFLG_MD5 ;

	sa.flags = tmp ;

	/*
	** figure out how many S/G entries we will have
	** Each entry in the old system counts as one
	** entries in the new system (gather entries),
	** plus one scatter entry for writing the results.
	*/
	count = 0 ;
	d1 = desc ;
	while (d1->next)
	{
		count++ ;
		d1 = d1->next ;
	}

	rc = msp_sec2_new_request(0, /* work_q */
				  count + 1, /* n_sg_entries */
				  &sa, /* sa */
				  0, /* control */
				  CBK_POLL, /* callback_fn */
				  (void *)&desc->ioctl.status, /* cbk_prm */
				  BLK_POLL /* poll for space */
				  ) ;

	if (rc == 0)
	{
		d1 = desc->next ;
		while (d1 != NULL)
		{
			/* add a gather entry */
			msp_sec2_add_sg(0,  /* work queue */
					SG_GATHER,
					d1->ioctl.srcaddr, /* address */
					d1->ioctl.buflen /* size */
					) ;
			d1 = d1->next ;
		}
		/* add scatter entry */
		msp_sec2_add_sg(0, /* work queue */
				SG_SCATTER,
				desc->ioctl.dstaddr, /* address */
				hashlen /* size */
				) ;

		msp_sec2_end_request(0) ;
	}
}

void
msp_sec_do_hmac2_operation_v2(sec_desc *desc)
{
	int hashlen ;
	sec_desc *d1 ;
	MSP_SEC2_SA sa ;
	int count ;
	int mode ;
	int tmp ;
	int rc ;
	
	DBG_SEC("=======================================\n") ;
	DBG_SEC("do_HMAC2_operation given this descriptor:\n") ;
	dump_descriptor(desc) ;
	DBG_SEC("=======================================\n") ;

	/* OK, this matches the version 2 hardware much better */

	/* must have second descriptor for hmac2 operation */
	if (desc->next == 0)
	{
		desc->ioctl.status = -EINVAL ;
		return ;
	}

	mode = desc->ioctl.mode ;

	tmp = SAFLG_MODE_HMAC | SAFLG_CV ;

	if (mode & HMC_SHA1)
	{
		tmp |= SAFLG_SHA1 ;
		hashlen = HSH_LEN_SHA1 ;
	}
	else
	{
		tmp |= SAFLG_MD5 ;
		hashlen = HSH_LEN_MD5 ;
	}

	/* partial hashes pointed to by desc's srcaddr and dstaddr */

	memcpy((void *)sa.hash_chain_a, desc->ioctl.srcaddr, hashlen) ;
	memcpy((void *)sa.hash_chain_b, desc->ioctl.dstaddr, hashlen) ;

	sa.flags = tmp ;
	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = HSH_BLK * 8 ; /* length in bits */

	count = 1 ;
	d1 = desc->next ;
	while (d1->next)
	{
		count++ ;
		d1 = d1->next ;
	}

	rc = msp_sec2_new_request(0, /* work_q */
				  count + 1, /* n_sg_entries */
				  &sa, /* sa */
				  0, /* control */
				  CBK_POLL, /* callback_fn */
				  (void *)&desc->ioctl.status, /* cbk_prm */
				  BLK_POLL /* poll for space */
				  ) ;

	if (rc == 0)
	{
		d1 = desc->next ;
		while (d1 != NULL)
		{
			/* add a gather entry */
			msp_sec2_add_sg(0,  /* work queue */
					SG_GATHER,
					d1->ioctl.srcaddr, /* address */
					d1->ioctl.buflen /* size */
					) ;
			d1 = d1->next ;
		}

		/* add scatter entry */
		msp_sec2_add_sg(0, /* work queue */
				SG_SCATTER,
				desc->next->ioctl.dstaddr, /* address */
				hashlen /* size */
				) ;

		msp_sec2_end_request(0) ;
	}
	
}

#endif /* CONFIG_BRECIS_SEC_V1ON2_COMPAT */


#ifdef SIMULATE_V2_HW
/*
** For use in testing new code on old hardware.  Not at all complete,
** and built with the idea of verification, not speed, in mind
*/

/*************************************************************
** hash function padding tools.
*************************************************************/
static unsigned char hash_pad_buf[69] = { 0x80 } ; /* all other bytes zero */

typedef struct pad_struct {
	/* two descriptors, one for pad space, one for length buffer */
	sec_ioctl pad_desc[2] ;
	/* buffer for length */
	union {
		unsigned long len[2] ;
		unsigned char clen[8] ;
	} u ;
} pad_struct ;

static void show_des_regs (void)
{
	printk("DES Registers:\n");
	printk("nextDescAddress:  %x\n", 
					(unsigned int)Des->next_desc_address);
	printk("buff_Size:          %lx  (%ld)\n", 
					Des->byte_count, 
					Des->byte_count);
	printk("source_Address:    %x\n", (unsigned int) Des->source_address);
	printk("dest_Address:      %x\n", (unsigned int) Des->dest_address);
	printk("Mode_Control:      %lx\n", Des->mode_control);
	printk("Descriptor_ID:     %lx\n", Des->descriptor_id);
	printk("Init_Vector_High:   %lx\n", Des->init_vector_high);
	printk("Init_Vector_Low:    %lx\n", Des->init_vector_low);
	printk("Key1High:         %lx\n", Des->key1_high);
	printk("Key1Low:          %lx\n", Des->key1_low);
	printk("Key2High:         %lx\n", Des->key2_high);
	printk("Key2Low:          %lx\n", Des->key2_low);
	printk("Key3High:         %lx\n", Des->key3_high);
	printk("Key3Low:          %lx\n", Des->key3_low);
	printk("interface_Control: %lx\n", Des->interface_control);
	printk("interface_Status:  %lx\n", Des->interface_status);
	printk("Input_Data_High:	  %lx\n", Des->input_data_high);
	printk("Input_Data_Low:	  %lx\n", Des->input_data_low);
	printk("Output_Data_High:	  %lx\n", Des->output_data_high);
	printk("Output_Data_Low:	  %lx\n", Des->output_data_low);
	printk("CDA_Register:	  %x\n", *DES_CDA_REG);
	printk("\n");
}

#define NXT_NDA(ptr)	((sec_desc *)((ptr)->ioctl.nda & ~3))
static void
dump_descriptor_nda(sec_desc *desc)
{
	while (desc)
	{
		printk("DESCR at %x\n", (int)desc) ;
		printk("DESCR:    nda %08x   len %08x  mode %08x\n",
		       desc->ioctl.nda, desc->ioctl.buflen, desc->ioctl.mode);
		printk("DESCR:    src %8p   dst %8p    id %08x\n",
		       desc->ioctl.srcaddr, desc->ioctl.dstaddr, desc->ioctl.id);
		printk("DESCR:    ivl %08x   ivh %08x  type %08x\n",
		       desc->ioctl.ivhigh, desc->ioctl.ivlow, desc->ioctl.type);
		printk("DESCR:    ch0 %08x   ch1 %08x   ch2 %08x\n",
		       desc->ioctl.chainvar[0],
		       desc->ioctl.chainvar[1],
		       desc->ioctl.chainvar[2]) ;

		printk("DESCR:    ch3 %08x   ch4 %08x   ch5 %08x\n",
		       desc->ioctl.chainvar[3],
		       desc->ioctl.chainvar[4],
		       desc->ioctl.chainvar[5]) ;

		printk("DESCR:    opt %08x  stat %08x   nxt %8p\n",
		       desc->ioctl.options, desc->ioctl.status, desc->next) ;

#ifdef DEBUG_SIM_VERBOSE
		invalidate_buffer(desc->ioctl.srcaddr, desc->ioctl.buflen) ;

		printk("DESCR: source data:\n") ;

		{
			int i ;
			char *DestBuf = (char *)desc->ioctl.srcaddr ;

			for(i = 0 ; i < desc->ioctl.buflen; i++)
			{
				switch(i % 16)
				{
				case 0:
					printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
					break;
				case 7:
					printk("%02x - ", DestBuf[i] & 0xff) ;
					break ;
				case 15:
					printk("%02x\n", DestBuf[i] & 0xff) ;
					break ;
				default:
					printk("%02x ", DestBuf[i] & 0xff) ;
					break ;
				}
			}
		}
	
#endif
		

		desc = NXT_NDA(desc) ;
	}
}

void
make_cq_entry(struct sec2_q_regs *cqregs,
	      int sw1, int sw2,
	      int status)
{
	*(int *)(cqregs->base + cqregs->in + 0x0) = sw1 ;
	*(int *)(cqregs->base + cqregs->in + 0x4) = sw2 ;
	*(int *)(cqregs->base + cqregs->in + 0x8) = status ;
	*(int *)(cqregs->base + cqregs->in + 0xc) = 0 ;

	cqregs->in = (cqregs->in + 16) & (cqregs->size - 1) ;
}
	

#define NDA_NXT(x) ( (sec_ioctl *)( (x->nda) & ~3 ) )
int
do_hash_with_pad(sec_ioctl *desc, int init_len, void *dstaddr)
{
	pad_struct pad_area ;
	int pad_len ;
	union
	{
		unsigned long l ;
		unsigned char c[4] ;
	} u ;
	int i = 0 ;
	int len ;
	int ncbuf = desc->cbufcount ;
	sec_ioctl my_desc, *d1, *d2, *next_desc, pdesc ;
	int remain ;
	int hash_mode = desc->mode & ~HMC_CRV ;
	unsigned long flags ;


	DBG_SIM("Begin do_hash_with_pad\n") ;

	/* figure out padding */
	len = init_len ;
	len += desc->buflen ;

	desc->dstaddr = (void *)&my_desc.chainvar[0] ;

	if (ncbuf)
	{
		desc->nda = (int) (&desc->cbuflist[0])
			| DNDA_CNT | DNDA_VLD ;
		
		for(i = 0 ; i < (ncbuf - 1); i++)
		{
			desc->cbuflist[i].nda =
				(int) (&desc->cbuflist[i+1])
				| DNDA_CNT | DNDA_VLD ;
			desc->cbuflist[i].dstaddr =
				(void*)&my_desc.chainvar[0] ;
			len += desc->cbuflist[i].buflen ;
		}

		desc->cbuflist[i].nda = 0 ;
		desc->cbuflist[i].dstaddr =
			(void*)&my_desc.chainvar[0] ;
		len += desc->cbuflist[i].buflen ;
	}
	else
	{
		desc->nda = 0 ;
	}

	DBG_SIM("do_hash_with_pad, init_len %x, tot len %x, ncbuf %x\n",
	       init_len, len, ncbuf) ;

	if (hash_mode == HMC_SHA1)
	{
		pad_area.u.len[0] = 0 ;
		pad_area.u.len[1] = len * 8 ;
	}
	else
	{
		/* 64 bits of little endian int for MD5 */
		u.l = len * 8 ;
		pad_area.u.clen[0] = u.c[3] ;
		pad_area.u.clen[1] = u.c[2] ;
		pad_area.u.clen[2] = u.c[1] ;
		pad_area.u.clen[3] = u.c[0] ;
		pad_area.u.len[1] = 0 ;
	}
	pad_len = len & 0x3f ;	/* modulo 64 */
	if (pad_len <= 55)
		pad_len = (64 - 8) - pad_len ;
	else
		pad_len = (128 - 8) - pad_len ;

	pad_area.pad_desc[0].buflen  = pad_len ;
	pad_area.pad_desc[0].srcaddr = hash_pad_buf ;
	pad_area.pad_desc[0].dstaddr =
		(void*)&my_desc.chainvar[0] ;
	pad_area.pad_desc[0].mode    = desc->mode ;

	pad_area.pad_desc[0].nda = 
		( ((int)&pad_area.pad_desc[1]) | (DNDA_VLD | DNDA_CNT) ) ;

	pad_area.pad_desc[1].buflen  = 8 ;
	pad_area.pad_desc[1].srcaddr = (void *)&pad_area.u ;
	pad_area.pad_desc[1].dstaddr =
		(void *)&my_desc.chainvar[0] ;
	pad_area.pad_desc[1].mode    = desc->mode ;

	pad_area.pad_desc[1].nda = 0 ;

	if(ncbuf)
	{
		desc->cbuflist[i].nda = ((int) &pad_area.pad_desc[0])
			| DNDA_VLD | DNDA_CNT ;
	}
	else
	{
		desc->nda = ((int) &pad_area.pad_desc[0])
			| DNDA_VLD | DNDA_CNT ;
	}

	memcpy(&my_desc, desc, sizeof(my_desc)) ;
	d1 = &my_desc ;

#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
	DBG_SIM("Hash descriptor before breaking\n") ;
	dump_descriptor_nda((sec_desc *)&my_desc) ;
#endif

	while(d1)
	{
		next_desc = NULL ;
		
		d2 = d1 ;

		while (d2)
		{
			remain = d2->buflen & 0x3f ;

			if (remain != 0)
			{
				while (NDA_NXT(d2))
				{
					remain += NDA_NXT(d2)->buflen ;
					if (remain >= 0x40)
						break ;
					d2 = NDA_NXT(d2) ;
				}

				/* remain is >= 0x40 at this point */
				remain -= 0x40 ;

				if (remain != 0)
				{
					/*
					** split this descriptor to
					** get to a 64 byte boundary
					*/

					next_desc = NDA_NXT(d2) ;

					len = NDA_NXT(d2)->buflen
						- remain ;

					pdesc.nda = 0 ;
					pdesc.buflen = len ;
					pdesc.srcaddr =
						NDA_NXT(d2)->srcaddr ;
					pdesc.dstaddr = (void *)
						&my_desc.chainvar[0] ;
					d2->nda = (((int)&pdesc)
							 |DNDA_VLD|DNDA_CNT) ;

					next_desc->srcaddr += len ;
					next_desc->buflen -= len ;

				}
				else if (NDA_NXT(NDA_NXT(d2)))
				{
					next_desc = NDA_NXT(NDA_NXT(d2)) ;
					NDA_NXT(d2)->nda = 0 ;
				}
				else
				{
					next_desc = 0 ;
				}
					
				break ;
			}
			else
			{
				d2 = NDA_NXT(d2) ;
			}
		}

		memcpy(&my_desc, d1, 16) ;

		my_desc.status = 0 ;

		DBG_SIM("hash doing one subchain\n") ;
#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
		dump_descriptor_nda((sec_desc *)&my_desc) ;
#endif

		save_flags(flags) ;
		cli() ;
		Hsh->next_desc_address = (void *)
			(((unsigned long) &my_desc) | DNDA_VLD) ;

		Hsh->interface_control = HIFC_DMA_START ;
		i = 0 ;
		while (((Hsh->interface_status & (HIFS_DON|HIFS_BSY)) != 0 )
		       || ( (long)Hsh->next_desc_address & DNDA_VLD))
		{
			NanoDelay(500) ;
			if (i++ > 20000)
			{
				printk(KERN_WARNING "HSH taking too long\n") ;
				my_desc.status = 1 ;

				dump_descriptor_nda((sec_desc *)&my_desc) ;

				break ;
			}
		}

		Hsh->interface_control = HIFC_HSH_RST ;

		restore_flags(flags) ;

		d1 = next_desc ;

		my_desc.mode |= HMC_CRV ;

		if (my_desc.status)
			break ;
		
	}

	invalidate_buffer(my_desc.chainvar, 20) ;
	memcpy(dstaddr, my_desc.chainvar, 20) ;

	DBG_SIM("End do_hash_with_pad\n") ;

	return my_desc.status ;

}

int
do_straight_hash(sec_ioctl *desc, void *dstaddr)
{
	int i = 0 ;
	int len ;
	int ncbuf = desc->cbufcount ;
	sec_ioctl my_desc, *d1, *d2, *next_desc, pdesc ;
	int remain ;
	unsigned long flags ;


	DBG_SIM("Begin do_straight_hash\n") ;

	desc->dstaddr = (void *)&my_desc.chainvar[0] ;

	if (ncbuf)
	{
		desc->nda = (int) (&desc->cbuflist[0])
			| DNDA_CNT | DNDA_VLD ;
		
		for(i = 0 ; i < (ncbuf - 1); i++)
		{
			desc->cbuflist[i].nda =
				(int) (&desc->cbuflist[i+1])
				| DNDA_CNT | DNDA_VLD ;
			desc->cbuflist[i].dstaddr =
				(void*)&my_desc.chainvar[0] ;
			len += desc->cbuflist[i].buflen ;
		}

		desc->cbuflist[i].nda = 0 ;
		desc->cbuflist[i].dstaddr =
			(void*)&my_desc.chainvar[0] ;
		len += desc->cbuflist[i].buflen ;
	}
	else
	{
		desc->nda = 0 ;
	}

	memcpy(&my_desc, desc, sizeof(my_desc)) ;
	d1 = &my_desc ;

#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
	DBG_SIM("Hash descriptor before breaking\n") ;
	dump_descriptor_nda((sec_desc *)&my_desc) ;
#endif

	while(d1)
	{
		next_desc = NULL ;
		
		d2 = d1 ;

		while (d2)
		{
			remain = d2->buflen & 0x3f ;

			if (remain != 0)
			{
				while (NDA_NXT(d2))
				{
					remain += NDA_NXT(d2)->buflen ;
					if (remain >= 0x40)
						break ;
					d2 = NDA_NXT(d2) ;
				}

				/* remain is >= 0x40 at this point */
				remain -= 0x40 ;

				if (remain != 0)
				{
					/*
					** split this descriptor to
					** get to a 64 byte boundary
					*/

					next_desc = NDA_NXT(d2) ;

					len = NDA_NXT(d2)->buflen
						- remain ;

					pdesc.nda = 0 ;
					pdesc.buflen = len ;
					pdesc.srcaddr =
						NDA_NXT(d2)->srcaddr ;
					pdesc.dstaddr = (void *)
						&my_desc.chainvar[0] ;
					d2->nda = (((int)&pdesc)
							 |DNDA_VLD|DNDA_CNT) ;

					next_desc->srcaddr += len ;
					next_desc->buflen -= len ;

				}
				else if (NDA_NXT(NDA_NXT(d2)))
				{
					next_desc = NDA_NXT(NDA_NXT(d2)) ;
					NDA_NXT(d2)->nda = 0 ;
				}
				else
				{
					next_desc = 0 ;
				}
					
				break ;
			}
			else
			{
				d2 = NDA_NXT(d2) ;
			}
		}

		memcpy(&my_desc, d1, 16) ;

		my_desc.status = 0 ;

		DBG_SIM("hash doing one subchain\n") ;
#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
		dump_descriptor_nda((sec_desc *)&my_desc) ;
#endif

		save_flags(flags) ;
		cli() ;
		Hsh->next_desc_address = (void *)
			(((unsigned long) &my_desc) | DNDA_VLD) ;

		Hsh->interface_control = HIFC_DMA_START ;
		i = 0 ;
		while (((Hsh->interface_status & (HIFS_DON|HIFS_BSY)) != 0 )
		       || ( (long)Hsh->next_desc_address & DNDA_VLD))
		{
			NanoDelay(500) ;
			if (i++ > 20000)
			{
				printk(KERN_WARNING "HSH taking too long\n") ;
				my_desc.status = 1 ;

				dump_descriptor_nda((sec_desc *)&my_desc) ;

				break ;
			}
		}

		Hsh->interface_control = HIFC_HSH_RST ;

		restore_flags(flags) ;

		d1 = next_desc ;

		my_desc.mode |= HMC_CRV ;

		if (my_desc.status)
			break ;
		
	}

	invalidate_buffer(my_desc.chainvar, 20) ;
	memcpy(dstaddr, my_desc.chainvar, 20) ;

	DBG_SIM("End do_straight_hash\n") ;

	return my_desc.status ;

}

typedef struct scatter_gather_state {
	struct sec2_q_regs *wqregs ;
	int	mask ;	
	int	total_bytes ;
	int	remain_bytes ;
	int	ofst ;		/* -1 = out of space */
	int	end_offset ;	/* offset past end of work entry */
	void	*addr ;		/* next address to write to */
	int	len ;		/* length left in current segment */
	int	sg_val ;	/* 0 for gthr, SEC2_WE_SG_SCATTER for scat */
	int	eod_stop ;	/* stop on eod? */
	int	eod_seen ;	/* eod seen? */
} sgstate ;

void
sgst_init(sgstate *sgst, int scatter,
	  struct sec2_q_regs *wqregs, int wqe_word1)
{
	int sg ;
	int size ;
	int len ;
	void *addr ;

	if (scatter)
		sgst->sg_val = SEC2_WE_SG_SCATTER ;
	else
		sgst->sg_val = 0 ;

	sgst->eod_stop = 0 ;
	sgst->eod_seen = 0 ;

	DBG_SIM("begin sgst_init\n") ;

	sgst->wqregs = wqregs ;
	sgst->mask = wqregs->size - 1 ;

	sgst->ofst = -1 ;

	sg = wqregs->out ;
	
	size = (wqe_word1 & 0xff) - 3 ;
	sgst->end_offset = (sg + size * 4) & sgst->mask ;

	sgst->total_bytes = 0 ;
	while ( sg != sgst->end_offset )
	{
		DBG_SIM("sgst init while sg %x end offset %x\n",
		       sg, sgst->end_offset) ;

		addr = *(void **)(wqregs->base+sg) ;
		len  = *(int *)(wqregs->base+sg+4) ;

		if ( ( len & SEC2_WE_SG_SCATTER ) != sgst->sg_val)
		{
			sg = (sg + 8) & sgst->mask ;
			continue ;
		}

		if (len & SEC2_WE_SG_EOD)
			sgst->eod_seen = 1 ;

		len &= 0x1fff ;
		
		if (sgst->ofst == -1)
		{
			sgst->ofst = sg ;
			sgst->addr = addr ;
			sgst->len = len ;
		}

		sgst->total_bytes += len ;

		sg = (sg + 8) & sgst->mask ;
	}
	sgst->remain_bytes = sgst->total_bytes ;
	DBG_SIM("end sgst_init\n") ;
}


int
sgst_write(sgstate *sgst, void *buf, int size)
{
	int written ;
	int plen ;

	DBG_SIM("Begin sgst_write\n") ;

	written = 0 ;

	while ( size != 0 && sgst->ofst != -1 )
	{
		plen = size ;
		if (plen > sgst->len)
			plen = sgst->len ;
		memcpy(sgst->addr, buf, plen) ;

		sgst->addr += plen ;
		sgst->len -= plen ;
		sgst->remain_bytes -= plen ;

		if (sgst->len == 0)
		{
			int sg = sgst->ofst ;
			int len ;
			void *addr ;

			if (sgst->eod_stop && sgst->eod_seen)
				break ;

			sg = (sg + 8) & sgst->mask ;
			while (sg != sgst->end_offset)
			{
				addr = *(void **)(sgst->wqregs->base+sg) ;
				len  = *(int *)(sgst->wqregs->base+sg+4) ;
				if ( (len&SEC2_WE_SG_SCATTER) != sgst->sg_val)
				{
					sg = (sg + 8) & sgst->mask ;
					continue ;
				}
				if (len & SEC2_WE_SG_EOD)
					sgst->eod_seen = 1 ;

				len &= 0x1fff ;
				
				sgst->ofst = sg ;
				sgst->addr = addr ;
				sgst->len = len ;
				break ;
			}
			if (sg == sgst->end_offset)
			{
				sgst->ofst = -1 ;
			}
		}

		buf += plen ;
		size -= plen ;
		written += plen ;
	}
	DBG_SIM("End sgst_write\n") ;
	return written ;
}

/*
** sgst_getbuffer: Consume exactly <size> or fewer bytes from
** the scatter list, in a contiguous buffer.  Stores a pointer
** to the buffer in *ptr, and returns the available size.
*/

int
sgst_getbuffer(sgstate *sgst, void **ptr, int size)
{
	int sg ;
	int len ;
	void *addr ;
	int my_size ;
	
	DBG_SIM("begin sgst_getbuffer\n") ;

 again:
	my_size = size ;
	if (sgst->ofst == -1)
	{
		DBG_SIM("0 end sgst_getbuffer\n") ;
		return 0 ;
	}

	if (my_size < sgst->len)
	{
		*ptr = sgst->addr ;

		sgst->addr += my_size ;
		sgst->len -= my_size ;
		sgst->remain_bytes -= my_size ;

		DBG_SIM("1 %x end sgst_getbuffer\n", my_size) ;

		return my_size ;
	}

	/* he's consuming the entire available buffer */
	my_size = sgst->len ;
	*ptr = sgst->addr ;

	sgst->len = 0 ;
	sgst->remain_bytes -= my_size ;

	if (sgst->eod_stop && sgst->eod_seen)
	{
		DBG_SIM("3 %x end sgst_getbuffer\n", my_size) ;
		return my_size ;
	}

	sg = (sgst->ofst + 8) & sgst->mask ;
	while (sg != sgst->end_offset)
	{
		DBG_SIM("sgst getbuffer while sg %x end offset %x\n",
		       sg, sgst->end_offset) ;
		addr = *(void **)(sgst->wqregs->base + sg) ;
		len  = *(int *)(sgst->wqregs->base + sg + 4) ;
		if ( ( len & SEC2_WE_SG_SCATTER ) != sgst->sg_val)
		{
			sg = (sg + 8) & sgst->mask ;
			continue ;
		}
		if (len & SEC2_WE_SG_EOD)
			sgst->eod_seen = 1 ;

		len &= 0x1fff ;

		sgst->ofst = sg ;
		sgst->addr = addr ;
		sgst->len = len ;
		
		break ;
	}
	if (sg == sgst->end_offset)
	{
		sgst->ofst = -1 ;
	}

	if (my_size == 0)
		goto again ;

	DBG_SIM("3 %x end sgst_getbuffer\n", size) ;
	return my_size ;
}	

int
sgst_read(sgstate *sgst, char *buf, int size)
{
	int read = 0 ;
	int plen ;

	DBG_SIM("begin sgst_read\n") ;

	while ( size != 0 && sgst->ofst != -1 )
	{
		plen = size ;
		if (plen > sgst->len)
			plen = sgst->len ;
		memcpy(buf, sgst->addr, plen) ;

		sgst->addr += plen ;
		sgst->len -= plen ;
		sgst->remain_bytes -= plen ;

		if (sgst->len == 0)
		{
			int sg = sgst->ofst ;
			int len ;
			void *addr ;

			if (sgst->eod_stop && sgst->eod_seen)
				break ;

			sg = (sg + 8) & sgst->mask ;
			while (sg != sgst->end_offset)
			{
				addr = *(void **)(sgst->wqregs->base + sg) ;
				len  = *(int *)(sgst->wqregs->base + sg + 4) ;
				if ( (len&SEC2_WE_SG_SCATTER) != sgst->sg_val)
				{
					sg = (sg + 8) & sgst->mask ;
					continue ;
				}
				
				if (len & SEC2_WE_SG_EOD)
					sgst->eod_seen = 1 ;

				len &= 0x1fff ;

				sgst->ofst = sg ;
				sgst->addr = addr ;
				sgst->len = len ;
				break ;
			}
			if (sg == sgst->end_offset)
				sgst->ofst = -1 ;
		}

		buf += plen ;
		size -= plen ;
		read += plen ;
	}
	DBG_SIM("end sgst_read\n") ;
	return read ;
}

int
convert_sg_to_cbuf_espout_crypt(sec_ioctl *desc,
				sec_cbuf *cbuf,
				sgstate *sst,
				sgstate *gst,
				int size
				)
{
	void *srcaddr, *dstaddr ;
	int srclen, dstlen ;
	int len ;
	int part ;
	int cbuf_idx = -1 ;

	DBG_SIM("enter convert_sg_to_cbuf_espout_crypt\n") ;
	while (size) {

		part = 0xfff ;
		if (size < part)
			part = size ;

		srclen = sgst_getbuffer(gst, &srcaddr, part) ;
		len = srclen ;
		while(len) {
			dstlen = sgst_getbuffer(sst, &dstaddr, len) ;

			if (dstlen == 0)
			{
				DBG_SIM("bad leave convert_sg_to_cbuf_espout_crypt\n") ;
				return -1 ;
			}

			if (cbuf_idx == -1)
			{
				desc->srcaddr = srcaddr ;
				desc->dstaddr = dstaddr ;
				desc->buflen = dstlen ;
			}
			else
			{
				cbuf[cbuf_idx].srcaddr = srcaddr ;
				cbuf[cbuf_idx].dstaddr = dstaddr ;
				cbuf[cbuf_idx].buflen = dstlen ;
			}
			cbuf_idx++ ;

			srcaddr += dstlen ;
			len -= dstlen ;
		}
		size -= srclen ;
	}
	DBG_SIM("leave convert_sg_to_cbuf_espout_crypt\n") ;
	return cbuf_idx ;
}

int
do_espout_hmac_function(MSP_SEC2_SA *sa,
			sec_ioctl *desc,
			int wqe_word1,
			char *hash_rslt,
			int *rslt_len)
{
	int hash_mode ;
	int hash_len ;
	int hash_96 = 0 ;
	int rc ;

	DBG_SIM("Begin do_espout_hmac\n") ;

	switch(sa->flags & 0xE000)
	{
	case SAFLG_MD5_96:
		hash_mode = HMC_MD5 ;
		hash_96 = 1 ;
		hash_len = 16 ;
		break ;
	case SAFLG_MD5:
		hash_mode = HMC_MD5 ;
		hash_96 = 0 ;
		hash_len = 16 ;
		break ;
	case SAFLG_SHA1_96:
		hash_mode = HMC_SHA1 ;
		hash_96 = 1 ;
		hash_len = 20 ;
		break ;
	case SAFLG_SHA1:
		hash_mode = HMC_SHA1 ;
		hash_96 = 0 ;
		hash_len = 20 ;
		break ;
	default:
		return -1 ;
	}

	if ( ! (sa->flags & SAFLG_CV) )
	{
		printk("ERROR:  Must use chaining var on HMAC\n") ;
		return -1 ;
	}

	memcpy(desc->chainvar, sa->hash_chain_a, 20) ;

	desc->mode = hash_mode | HMC_CRV ;

	DBG_SIM("Calling do_hadh_with_pad A (out)\n") ;

	rc = do_hash_with_pad(desc, sa->hash_init_len[1] / 8, hash_rslt) ;
	if (rc)
	{
		printk("HASH error") ;
		return -1 ;
	}

	memcpy(desc->chainvar, sa->hash_chain_b, 20) ;
	desc->mode = hash_mode | HMC_CRV ;
	desc->buflen = hash_len ;
	desc->srcaddr = hash_rslt ;
	desc->cbufcount = 0 ;

	DBG_SIM("Calling do_hash_with_pad B (out)\n") ;

	rc = do_hash_with_pad(desc, sa->hash_init_len[1] / 8, hash_rslt) ;
	if (rc)
	{
		printk("HASH error") ;
		return -1 ;
	}

	if (hash_96)
		hash_len = 12 ;

	*rslt_len = hash_len ;

	return 0 ;

	DBG_SIM("end espout_hmac_fn\n") ;

}

void
do_espout_function(struct sec2_q_regs *wqregs,
		   struct sec2_q_regs *cqregs,
		   MSP_SEC2_SA *sa,
		   int wqe_word1,
		   int wqe_word2,
		   int wqe_word3)
{
	sgstate sst ;
	sgstate gst ;
	int rc ;
	sec_ioctl desc;
	sec_cbuf cbuf[32], *cbuf_crypt ;
	int ncbuf ;
	int pad_len ;
	static char padding[] = "\01\02\03\04\05\06\07\010\011\012\013\014\015\016\017\020" ;
	static char prdm[] = { 0x5a, 0x5a, 0x5a, 0x5a,
			       0x5a, 0x5a, 0x5a, 0x5a } ;
	char esp_trailer[2] ;
	void *dstaddr ;
	static char esp_hdr_iv[32] ;
	int esp_hdr_iv_len ;
	char hash_rslt[20] ;
	int hash_len ;
	int size ;
	int i ;
	int alg ;
	int ivlen ;
	int blkmode ;

	DBG_SIM("Begin do_espout\n") ;

	sgst_init(&sst, 1, wqregs, wqe_word1) ;
	sgst_init(&gst, 0, wqregs, wqe_word1) ;

	/*
	** ESP out is made of the following parts:
	**
	** ESP_HDR  -- from SA
	** IV       -- from SA or RNG
	**  == Following encrypted ==
	**    Payload data (from input)
	**    Padding (created by security engine, size in wq)
	**    ESP trailer (from work queue)
	**  == end encrypted section ==
	** ESP authentication (ICV) generated by sec engine
	**
	** ICV covers data from ESP_HDR through the end of the encrypted
	** section.
	*/

	/* make ESP_HDR */
	memcpy(&esp_hdr_iv[0], &sa->esp_spi, 4) ;
	memcpy(&esp_hdr_iv[4], &sa->esp_sequence, 4) ;

	if (sa->flags & SAFLG_SI)
		sa->esp_sequence++ ;

	if (sa->flags & SAFLG_CRI)
	{
		/* supposed to be random number */
		/* this is somewhat random :-) */
		memcpy(&esp_hdr_iv[8], prdm, 8) ;
	}
	else
		memcpy(&esp_hdr_iv[8], &sa->crypt_iv, 8) ;

	esp_hdr_iv_len = 16 ;
	rc = sgst_write(&sst, esp_hdr_iv, esp_hdr_iv_len) ;
	if (rc != esp_hdr_iv_len)
	{
		DBG_SIM("FAILURE: write of iv\n") ;
		goto error ;
	}

	/* leave room for one extra continuation buffer
	** at begining for hmac later
	*/
	cbuf_crypt = &cbuf[1] ;

	/* create list for DES encryption */
	ncbuf = convert_sg_to_cbuf_espout_crypt(&desc,
						cbuf_crypt,
						&sst,
						&gst,
						gst.total_bytes
						) ;
	if (ncbuf == -1)
	{
		DBG_SIM("FAILURE: conversion to cbufs\n") ;
		goto error ;
	}

	pad_len = (wqe_word1 >> 24) & 0xff ;
	padding[pad_len] = pad_len ;
	padding[pad_len + 1] = wqe_word1 >> 16 ;

	for (i = 0 ; i < pad_len + 2 ; )
	{
		rc = sgst_getbuffer(&sst, &dstaddr, pad_len + 2 - i) ;
		if (rc == 0)
		{
			DBG_SIM("FAILURE: out of buffer space (padding)\n") ;
			goto error ;
		}
		cbuf_crypt[ncbuf].srcaddr = padding + i ;
		cbuf_crypt[ncbuf].dstaddr = dstaddr ;
		cbuf_crypt[ncbuf].buflen = rc ;

		i += rc ;
		ncbuf++ ;
	}

	esp_trailer[0] = pad_len ;
	esp_trailer[1] = wqe_word1 >> 16 ;


	/* now, do the encryption */

	desc.mode = 0 ;

	/* which encryption algorithm */
	alg = sa->flags & 0x01C00000 ;
	switch (alg) {
	case SAFLG_DES:
		ivlen = 8 ; 
		break ;
	case SAFLG_3DES:
		ivlen = 8 ;
		desc.mode |= DMC_3DES ;
		break ;
	case SAFLG_CRYPTNULL:
		ivlen = 0 ;
		desc.mode |= DMC_MOD_PT ;
		break ;
	default:
		printk("ERROR on algorithm\n") ;
		goto error ;
	}

	/* which encryption block mode */
	blkmode = sa->flags & 0x380000 ;
	switch(blkmode) {
	case SAFLG_ECB:
		break ;

	case SAFLG_CBC_ENCRYPT:
		desc.mode |= DMC_MOD_CBC ;
		break ;

	case SAFLG_CBC_DECRYPT:
		desc.mode |= DMC_MOD_CBC | DMC_MOD_DEC ;
		break ;

	case SAFLG_CFB_ENCRYPT:
		desc.mode |= DMC_MOD_CFB ;
		break ;

	case SAFLG_CFB_DECRYPT:
		desc.mode |= DMC_MOD_CFB | DMC_MOD_DEC ;
		break ;

	default:
		printk("Error on block mode\n") ;
		goto error ;
	}

	if (sa->flags & SAFLG_DES_K1_DECRYPT)
		desc.mode |= DMC_K1_DEC ;

	if (sa->flags & SAFLG_DES_K2_DECRYPT)
		desc.mode |= DMC_K2_DEC ;

	if (sa->flags & SAFLG_DES_K3_DECRYPT)
		desc.mode |= DMC_K3_DEC ;

	memcpy(&desc.ivhigh, &esp_hdr_iv[8], 8) ;

	memcpy((void *)&desc.chainvar[0], sa->crypt_keys, 24) ;

	desc.nda = ((int)cbuf_crypt) | DNDA_VLD | DNDA_CNT ;
	for (i = 1 ; i < ncbuf; i++)
	{
		cbuf_crypt[i-1].nda =
			((int)&cbuf_crypt[i]) | DNDA_VLD | DNDA_CNT ;
	}
	cbuf_crypt[ncbuf - 1].nda = 0 ;

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "DES taking too long at entry\n") ;
			printk("\n=====================================\n\n") ;
			show_des_regs() ;
			Des->interface_control = DIFC_DES_RST ;

			break ;
		}
	}

	DBG_SIM("===== About to write DES hw %p\n", &desc) ;

#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
	dump_descriptor_nda((sec_desc *)&desc) ;
#endif

	Des->next_desc_address = (void *)(((unsigned long) &desc) | DNDA_VLD) ;

	Des->interface_control = DIFC_DMA_START ;

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "*** DES taking too long\n") ;
			printk("\n=====================================\n\n") ;

			dump_descriptor_nda((sec_desc *)&desc) ;
			show_des_regs() ;

			Des->interface_control = DIFC_DES_RST ;

			goto error ;
		}
	}

	padding[pad_len] = pad_len + 1 ;
	padding[pad_len + 1] = pad_len + 2 ;

	/*
	** Prepend esp_hdr_iv to the chain for hmac processing,
	** by placing desc's buffer into the first cbuf,
	** then putting esp_hdr_iv into desc.
	*/
	memcpy(&cbuf[0], &desc, 16) ;

	ncbuf ++ ;		/* one more now */

	desc.nda = ((int)&cbuf[0]) | DNDA_VLD | DNDA_CNT ;
	desc.buflen = esp_hdr_iv_len ;
	desc.srcaddr = esp_hdr_iv ;
	desc.dstaddr = hash_rslt ;

	desc.cbufcount = ncbuf ;
	desc.cbuflist = cbuf ;

	for (i = 0 ; i < ncbuf ; i++)
		cbuf[i].srcaddr = cbuf[i].dstaddr ;

	rc = do_espout_hmac_function(sa, &desc, wqe_word1,
				     hash_rslt, &hash_len) ;
	if (rc == -1)
	{
		DBG_SIM("FAILURE: hmac function bad return\n") ;
		goto error ;
	}

	rc = sgst_write(&sst, hash_rslt, hash_len) ;
	if (rc != hash_len)
	{
		DBG_SIM("FAILURE: write of hmac ICV; rc = 0x%x\n", rc) ;
		goto error ;
	}
	
	DBG_SIM("End do_espout\n") ;
	make_cq_entry(cqregs, wqe_word2, wqe_word3, 0) ;
	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
	return ;

 error:
	DBG_SIM("error End do_espout\n") ;
	make_cq_entry(cqregs, wqe_word2, wqe_word3, -1) ;
	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
	
}

int
create_hmac_cbuf_from_gathers(sgstate *gst,
			      sec_ioctl *desc,
			      sec_cbuf *cbuf,
			      void *dstaddr)
{
	int glen=0 ;
	char *gaddr=0 ;
	int cbuf_idx = -1 ;

	DBG_SIM("Begin create_hmac_from_gathers\n") ;

	gst->eod_stop = 1 ;

	glen = sgst_getbuffer(gst, (void *)&gaddr, 0xfff) ;

	while(glen)
	{
		if (cbuf_idx == -1)
		{
			desc->srcaddr = gaddr ;
			desc->dstaddr = dstaddr ;
			desc->buflen = glen ;
		} else {
			cbuf[cbuf_idx].srcaddr = gaddr ;
			cbuf[cbuf_idx].dstaddr = dstaddr ;
			cbuf[cbuf_idx].buflen = glen ;
		}
		cbuf_idx++ ;

		glen = sgst_getbuffer(gst, (void *)&gaddr, 0xfff) ;
	}

	return cbuf_idx ;

}

/* returns 0 on success, -1 on failiure */
int
do_espin_hmac_function(sgstate *gst,
		       MSP_SEC2_SA *sa,
		       int wqe_word1,
		       int wqe_word2,
		       int wqe_word3,
		       char *rslt,
		       int *rslt_len)
{
	int rc ;
	static sec_ioctl desc ;
	static sec_cbuf cbuf[32] ;/* assuming nothing larger - cheap
                                      muffler rule applies here */
	int hash_mode ;
	int hash_len ;
	int hash_96 = 0 ;
	char hash_rslt[20] ;
	int ncbuf ;

	int icv_ofst ;
	void *icv_addr ;
	int icv_len ;

	DBG_SIM("Begin do_espin_hmac\n") ;

	switch(sa->flags & 0xE000)
	{
	case SAFLG_MD5_96:
		hash_mode = HMC_MD5 ;
		hash_96 = 1 ;
		hash_len = 16 ;
		break ;
	case SAFLG_MD5:
		hash_mode = HMC_MD5 ;
		hash_96 = 0 ;
		hash_len = 16 ;
		break ;
	case SAFLG_SHA1_96:
		hash_mode = HMC_SHA1 ;
		hash_96 = 1 ;
		hash_len = 20 ;
		break ;
	case SAFLG_SHA1:
		hash_mode = HMC_SHA1 ;
		hash_96 = 0 ;
		hash_len = 20 ;
		break ;
	default:
		return -1 ;
	}

	if ( ! (sa->flags & SAFLG_CV) )
	{
		printk("ERROR:  Must use chaining var on HMAC\n") ;
		return -1 ;
	}

	ncbuf = create_hmac_cbuf_from_gathers(gst, &desc, cbuf, &icv_ofst) ;
	if (icv_ofst == -1)
	{
		printk("ERROR - no EOD bit...\n") ;
		return -1 ;
	}

	memcpy(desc.chainvar, sa->hash_chain_a, 20) ;

	desc.mode = hash_mode | HMC_CRV ;
	desc.cbufcount = ncbuf ;
	desc.cbuflist = cbuf ;

	DBG_SIM("Calling do_hash_with_pad A\n") ;

	rc = do_hash_with_pad(&desc, sa->hash_init_len[1] / 8, hash_rslt) ;
	if (rc)
	{
		printk("HASH error") ;
		return -1 ;
	}

	memcpy(desc.chainvar, sa->hash_chain_b, 20) ;
	desc.mode = hash_mode | HMC_CRV ;
	desc.buflen = hash_len ;
	desc.srcaddr = hash_rslt ;
	desc.cbufcount = 0 ;

	DBG_SIM("Calling do_hash_with_pad B\n") ;

	rc = do_hash_with_pad(&desc, sa->hash_init_len[1] / 8, hash_rslt) ;
	if (rc)
	{
		printk("HASH error") ;
		return -1 ;
	}

	memcpy(rslt, hash_rslt, hash_len) ;

	if (hash_96)
		hash_len = 12 ;

	if(sa->flags & SAFLG_CPI)
	{
		/* compare ICV */
		gst->eod_stop = 0 ;
		icv_len = sgst_getbuffer(gst, &icv_addr, hash_len) ;

		if (icv_len != hash_len)
		{
			printk("Hash compare: length wrong\n") ;
			return -1 ;
		}
		if (memcmp(hash_rslt, icv_addr, hash_len))
		{
			printk("Hash compare: data differs\n") ;
			return -1 ;
		}
	}
	*rslt_len = hash_len ;

	return 0 ;
}

int
convert_sg_to_cbuf_espin_crypt(sgstate *sst,
			       sgstate *gst,
			       sec_ioctl *desc,
			       sec_cbuf *cbuf)
{
	int glen=0, slen=0 ;
	char *gaddr=0, *saddr=0 ;
	int cbuf_idx = -1 ;

	while ((glen = sgst_getbuffer(gst, (void *)&gaddr, 0xfff)))
	{
		while (glen)
		{
			slen = sgst_getbuffer(sst, (void *)&saddr, glen) ;

			if (slen == 0)
			{
				DBG_SIM("bad leave convert_sg_to_cbuf_espin_crypt\n") ;
				return -1 ;
			}

			if (cbuf_idx == -1)
			{
				desc->srcaddr = gaddr ;
				desc->dstaddr = saddr ;
				desc->buflen = slen ;
			}
			else
			{
				cbuf[cbuf_idx].srcaddr = gaddr ;
				cbuf[cbuf_idx].dstaddr = saddr ;
				cbuf[cbuf_idx].buflen = slen ;
			}
			cbuf_idx++ ;
			gaddr += slen ;
			glen -= slen ;
		}
	}
	DBG_SIM("leave convert_sg_to_cbuf_espin_crypt\n") ;
	return cbuf_idx ;
}

int
do_espin_crypt_function(sgstate *sst,
			sgstate *gst,
			MSP_SEC2_SA *sa)
{
	static sec_ioctl desc ;
	static sec_cbuf cbuf[32] ; /* assuming nothing larger - cheap
                                      muffler rule applies here */
	int ncbuf ;
	int alg ;
	int blkmode ;
	int i ;
	int ivlen ;
	static char esp_hdr[8] ;
	int rc ;

	DBG_SIM("Begin do_espin_crypt_function\n") ;

	gst->eod_stop = 1 ;

	desc.mode = 0 ;

	rc = sgst_read(gst, esp_hdr, 8) ;

	if (rc != 8)
		return -1 ;

	/* which encryption algorithm */
	alg = sa->flags & 0x01C00000 ;
	switch (alg) {
	case SAFLG_DES:
		ivlen = 8 ; 
		break ;
	case SAFLG_3DES:
		ivlen = 8 ;
		desc.mode |= DMC_3DES ;
		break ;
	case SAFLG_CRYPTNULL:
		ivlen = 0 ;
		desc.mode |= DMC_MOD_PT ;
		break ;
	default:
		printk("ERROR on algorithm\n") ;
		goto error ;
	}

	if (ivlen)
	{
		rc = sgst_read(gst, (char *)&desc.ivhigh, ivlen) ;
		if (rc != ivlen)
			return -1 ;
	}

	ncbuf = convert_sg_to_cbuf_espin_crypt(sst, gst, &desc, cbuf) ;

	if (ncbuf == -1)
		goto error ;

	/* which encryption block mode */
	blkmode = sa->flags & 0x380000 ;
	switch(blkmode) {
	case SAFLG_ECB:
		break ;

	case SAFLG_CBC_ENCRYPT:
		desc.mode |= DMC_MOD_CBC ;
		break ;

	case SAFLG_CBC_DECRYPT:
		desc.mode |= DMC_MOD_CBC | DMC_MOD_DEC ;
		break ;

	case SAFLG_CFB_ENCRYPT:
		desc.mode |= DMC_MOD_CFB ;
		break ;

	case SAFLG_CFB_DECRYPT:
		desc.mode |= DMC_MOD_CFB | DMC_MOD_DEC ;
		break ;

	default:
		printk("Error on block mode\n") ;
		goto error ;
	}

	if (sa->flags & SAFLG_DES_K1_DECRYPT)
		desc.mode |= DMC_K1_DEC ;

	if (sa->flags & SAFLG_DES_K2_DECRYPT)
		desc.mode |= DMC_K2_DEC ;

	if (sa->flags & SAFLG_DES_K3_DECRYPT)
		desc.mode |= DMC_K3_DEC ;

	memcpy((void *)&desc.chainvar[0], sa->crypt_keys, 24) ;

	if (ncbuf == 0)
		desc.nda = 0 ;
	else
	{
		desc.nda = ((int)cbuf) | DNDA_VLD | DNDA_CNT ;
		for (i = 1 ; i < ncbuf; i++)
		{
			cbuf[i-1].nda = ((int)&cbuf[i]) | DNDA_VLD | DNDA_CNT ;
		}
		cbuf[ncbuf - 1].nda = 0 ;
	}

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "DES taking too long at entry\n") ;
			printk("\n=====================================\n\n") ;
			show_des_regs() ;
			Des->interface_control = DIFC_DES_RST ;

			break ;
		}
	}

	DBG_SIM("=====  About to write DES hw %p\n", &desc) ;

#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
	dump_descriptor_nda((sec_desc *)&desc) ;
#endif

	Des->next_desc_address = (void *)(((unsigned long) &desc) | DNDA_VLD) ;

	Des->interface_control = DIFC_DMA_START ;

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "*** DES taking too long\n") ;
			printk("\n=====================================\n\n") ;

			dump_descriptor_nda((sec_desc *)&desc) ;
			show_des_regs() ;

			Des->interface_control = DIFC_DES_RST ;

			goto error ;
		}
	}

	DBG_SIM("leaving espin crypt, good return\n") ;
	return 0 ;

 error:
	DBG_SIM("Espin crypt error return\n") ;
	return -1 ;
}
		

void
do_espin_function(struct sec2_q_regs *wqregs,
		  struct sec2_q_regs *cqregs,
		  MSP_SEC2_SA *sa,
		  int wqe_word1,
		  int wqe_word2,
		  int wqe_word3)
{
	int rc ;
	int size ;
	char hsh_rslt[20] ;
	int hsh_len ;
	sgstate sst ;
	sgstate gst ;

	DBG_SIM("Begin do_espin_function\n") ;

	sgst_init(&sst, 1, wqregs, wqe_word1) ;
	sgst_init(&gst, 0, wqregs, wqe_word1) ;
	
	rc = do_espin_hmac_function(&gst, sa,
				    wqe_word1, wqe_word2, wqe_word3,
				    hsh_rslt, &hsh_len) ;
	if (rc) goto error ;

	DBG_SIM("do_espin_calling crypt\n") ;
	
	sgst_init(&gst, 0, wqregs, wqe_word1) ;
	

	rc = do_espin_crypt_function(&sst, &gst, sa) ;

	if (rc)
		goto error ;

	/* if necessary, copy the ICV for comparisson to the scatter list */
	if ( (sa->flags & SAFLG_CPI) == 0 )
	{
		rc = sgst_write(&sst, hsh_rslt, hsh_len) ;
		if (rc != hsh_len)
			goto error ;
	}

	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;

	/* completed with no problems */
	make_cq_entry(cqregs, wqe_word2, wqe_word3, 0) ;
	DBG_SIM("leaving esp in, good return\n") ;
	return ;

 error:
	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
	DBG_SIM("Espin error return\n") ;
	make_cq_entry(cqregs,
		      wqe_word2, wqe_word3, 0xffffffff) ;
	return ;

}


void
do_hmac_function(struct sec2_q_regs *wqregs,
		 struct sec2_q_regs *cqregs,
		 MSP_SEC2_SA *sa,
		 int wqe_word1,
		 int wqe_word2,
		 int wqe_word3)
{
	char hash_rslt[20] ;
	int hash_len ;
	sgstate sst, gst ;
	void *addr ;
	int len, size ;
	sec_ioctl desc;
	sec_cbuf cbuf[32] ;
	int cbuf_idx = -1 ;
	int rc ;

	sgst_init(&sst, 1, wqregs, wqe_word1) ;
	sgst_init(&gst, 0, wqregs, wqe_word1) ;

	while ( (len = sgst_getbuffer(&gst, &addr, 0x800)) )
	{
		if (cbuf_idx == -1)
		{
			desc.srcaddr = addr ;
			desc.dstaddr = hash_rslt ;
			desc.buflen = len ;
		}
		else
		{
			cbuf[cbuf_idx].srcaddr = addr ;
			cbuf[cbuf_idx].dstaddr = hash_rslt ;
			cbuf[cbuf_idx].buflen = len ;
		}
		cbuf_idx++ ;
	}

	desc.cbufcount = cbuf_idx ;
	desc.cbuflist = cbuf ;

	do_espout_hmac_function(sa, &desc, wqe_word1,
				hash_rslt, &hash_len) ;

	rc = sgst_write(&sst, hash_rslt, hash_len) ;


	if (rc != hash_len)
		make_cq_entry(cqregs, wqe_word2, wqe_word3, -1) ;
	else
		make_cq_entry(cqregs, wqe_word2, wqe_word3, 0) ;

	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
		
}

void
do_hash_pad_function(struct sec2_q_regs *wqregs,
		     struct sec2_q_regs *cqregs,
		     MSP_SEC2_SA *sa,
		     int wqe_word1,
		     int wqe_word2,
		     int wqe_word3)
{
	char hash_rslt[20] ;
	int hash_mode ;
	int hash_len ;
	int hash_96 = 0 ;
	sgstate sst, gst ;
	void *addr ;
	int len, size ;
	sec_ioctl desc;
	sec_cbuf cbuf[32] ;
	int cbuf_idx = -1 ;
	int rc ;
	int retval = 0 ;

	switch(sa->flags & 0xE000)
	{
	case SAFLG_MD5_96:
		hash_mode = HMC_MD5 ;
		hash_96 = 1 ;
		hash_len = 16 ;
		break ;
	case SAFLG_MD5:
		hash_mode = HMC_MD5 ;
		hash_96 = 0 ;
		hash_len = 16 ;
		break ;
	case SAFLG_SHA1_96:
		hash_mode = HMC_SHA1 ;
		hash_96 = 1 ;
		hash_len = 20 ;
		break ;
	case SAFLG_SHA1:
		hash_mode = HMC_SHA1 ;
		hash_96 = 0 ;
		hash_len = 20 ;
		break ;
	default:
		retval = -1 ;
		goto exit ;
	}

	if (sa->flags & SAFLG_CV)
		hash_mode |= HMC_CRV ;

	desc.mode = hash_mode ;
	memcpy(&desc.chainvar, sa->hash_chain_a, 20) ;
	
	sgst_init(&sst, 1, wqregs, wqe_word1) ;
	sgst_init(&gst, 0, wqregs, wqe_word1) ;

	while ( (len = sgst_getbuffer(&gst, &addr, 0x800)) )
	{
		if (cbuf_idx == -1)
		{
			desc.srcaddr = addr ;
			desc.dstaddr = hash_rslt ;
			desc.buflen = len ;
		}
		else
		{
			cbuf[cbuf_idx].srcaddr = addr ;
			cbuf[cbuf_idx].dstaddr = hash_rslt ;
			cbuf[cbuf_idx].buflen = len ;
		}
		cbuf_idx++ ;
	}

	desc.cbufcount = cbuf_idx ;
	desc.cbuflist = cbuf ;

	rc = do_hash_with_pad(&desc, sa->hash_init_len[1]/8, hash_rslt) ;

	if (rc)
	{
		retval = -1 ;
		goto exit ;
	}

	if (hash_96)
		hash_len = 12 ;

	rc = sgst_write(&sst, hash_rslt, hash_len) ;


	if (rc != hash_len)
		retval = -1 ;

 exit:
	make_cq_entry(cqregs, wqe_word2, wqe_word3, retval) ;
	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
		
}


void
do_hash_function(struct sec2_q_regs *wqregs,
		 struct sec2_q_regs *cqregs,
		 MSP_SEC2_SA *sa,
		 int wqe_word1,
		 int wqe_word2,
		 int wqe_word3)
{
	char hash_rslt[20] ;
	int hash_mode ;
	int hash_len ;
	int hash_96 = 0 ;
	sgstate sst, gst ;
	void *addr ;
	int len, size ;
	sec_ioctl desc;
	sec_cbuf cbuf[32] ;
	int cbuf_idx = -1 ;
	int rc ;
	int retval = 0 ;
	int totlen ;

	switch(sa->flags & 0xE000)
	{
	case SAFLG_MD5_96:
		hash_mode = HMC_MD5 ;
		hash_96 = 1 ;
		hash_len = 16 ;
		break ;
	case SAFLG_MD5:
		hash_mode = HMC_MD5 ;
		hash_96 = 0 ;
		hash_len = 16 ;
		break ;
	case SAFLG_SHA1_96:
		hash_mode = HMC_SHA1 ;
		hash_96 = 1 ;
		hash_len = 20 ;
		break ;
	case SAFLG_SHA1:
		hash_mode = HMC_SHA1 ;
		hash_96 = 0 ;
		hash_len = 20 ;
		break ;
	default:
		retval = -1 ;
		goto exit ;
	}

	if (sa->flags & SAFLG_CV)
		hash_mode |= HMC_CRV ;

	desc.mode = hash_mode ;
	memcpy(&desc.chainvar, sa->hash_chain_a, 20) ;
	
	sgst_init(&sst, 1, wqregs, wqe_word1) ;
	sgst_init(&gst, 0, wqregs, wqe_word1) ;

	totlen = 0 ;
	while ( (len = sgst_getbuffer(&gst, &addr, 0x800)) )
	{
		totlen += len ;
		if (cbuf_idx == -1)
		{
			desc.srcaddr = addr ;
			desc.dstaddr = hash_rslt ;
			desc.buflen = len ;
		}
		else
		{
			cbuf[cbuf_idx].srcaddr = addr ;
			cbuf[cbuf_idx].dstaddr = hash_rslt ;
			cbuf[cbuf_idx].buflen = len ;
		}
		cbuf_idx++ ;
	}

	if (totlen & 0x3f)
	{
		retval = -1 ;
		goto exit ;
	}

	desc.cbufcount = cbuf_idx ;
	desc.cbuflist = cbuf ;

	rc = do_straight_hash(&desc, hash_rslt) ;

	if (rc)
	{
		retval = -1 ;
		goto exit ;
	}

	if (hash_96)
		hash_len = 12 ;

	rc = sgst_write(&sst, hash_rslt, hash_len) ;


	if (rc != hash_len)
		retval = -1 ;

 exit:
	make_cq_entry(cqregs, wqe_word2, wqe_word3, retval) ;
	size = ((wqe_word1 & 0xff) - 3) * 4 ;
	wqregs->out = (wqregs->out + size) & (wqregs->size - 1) ;
		
}



int
convert_sg_to_cbuf_crypt(struct sec2_q_regs *wqregs,
			 sec_ioctl *desc,
			 sec_cbuf *cbuf,
			 int nsg)
{
	int g, s ;
	int glen=0, slen=0 ;
	char *gaddr=0, *saddr=0 ;
	int end_offset ;
	int part_len ;
	int mask = wqregs->size - 1 ;
	int cbuf_idx = -1 ;

	DBG_SIM("Begin convert_sg_to_cbuf_crypt, out %x\n", wqregs->out) ;

	g = wqregs->out ;
	s = wqregs->out ;
	end_offset = (wqregs->out + nsg * 8) & mask ;

	while (g != end_offset)
	{
		gaddr = *(char **)(wqregs->base + g) ;
		glen  = *(int *)(wqregs->base + g + 4) ;

		DBG_SIM("G: %x addr %p len %x\n", g, gaddr, glen) ;

		if ( (glen & SEC2_WE_SG_SCATTER) == 0 )
		{
			DBG_SIM("Use g of %x\n", g) ;
			glen &= 0x1FFF ;
			break ;
		}
		else
		{
			g = (g + 8) & mask ;
			DBG_SIM("Move g forward to %x\n", g) ;
		}
	}

	while (s != end_offset)
	{
		saddr = *(char **)(wqregs->base + s) ;
		slen  = *(int *)(wqregs->base + s + 4) ;

		DBG_SIM("S: %x addr %p len %x\n", s, saddr, slen) ;

		if ( (slen & SEC2_WE_SG_SCATTER) != 0 )
		{
			DBG_SIM("Use s of %x\n", s) ;
			slen &= 0x1FFF ;
			break ;
		}
		else
		{
			s = (s + 8) & mask ;
			DBG_SIM("move s forward to %x\n", s) ;
		}
	}

	while ( (g != end_offset) && (s != end_offset) )
	{

		DBG_SIM("While loop, g %x s %x end %x\n", g, s, end_offset) ;
		DBG_SIM("glen %x gaddr %p slen %x saddr %p\n", glen, gaddr, slen, saddr) ;
		part_len = glen ;
		if (part_len > slen)
			part_len = slen ;

		if (cbuf_idx == -1)
		{
			desc->srcaddr = gaddr ;
			desc->dstaddr = saddr ;
			desc->buflen = part_len ;
		} else {
			cbuf[cbuf_idx].srcaddr = gaddr ;
			cbuf[cbuf_idx].dstaddr = saddr ;
			cbuf[cbuf_idx].buflen = part_len ;
		}
		cbuf_idx++ ;

		gaddr += part_len ;
		glen -= part_len ;
		if (glen == 0)
		{
			g = (g + 8) & mask ;
			DBG_SIM("Move g forward to %x\n", g) ;
			while (g != end_offset)
			{
				gaddr = *(char **)(wqregs->base + g) ;
				glen  = *(int *)(wqregs->base + g + 4) ;

				DBG_SIM("G: %x addr %p len %x\n", g, gaddr, glen) ;

				if ( (glen & SEC2_WE_SG_SCATTER) == 0 )
				{
					DBG_SIM("use g of %x\n", g) ;
					glen &= 0x1FFF ;
					break ;
				}
				else
				{
					g = (g + 8) & mask ;
					DBG_SIM("move g forward to %x\n", g) ;
				}
			}
		}

		saddr += part_len ;
		slen -= part_len ;
		if (slen == 0)
		{
			s = (s + 8) & mask ;
			DBG_SIM("Move s forward to %x\n", s) ;
			while (s != end_offset)
			{
				saddr = *(char **)(wqregs->base + s) ;
				slen  = *(int *)(wqregs->base + s + 4) ;

				DBG_SIM("S: %x addr %p len %x\n", s, saddr, slen) ;
		
				if ( (slen & SEC2_WE_SG_SCATTER) != 0 )
				{
					DBG_SIM("Use s of %x\n", s) ;
					slen &= 0x1FFF ;
					break ;
				}
				else
				{
					s = (s + 8) & mask ;
					DBG_SIM("move s forard to %x\n", s) ;
				}
			}

		}

	}
	/* move the out pointer forward */
	wqregs->out = end_offset ;
	
	DBG_SIM("(near)End convert_sg_to_cbuf_crypt\n") ;

	if (s != g)
	{
		printk("EMULATION WARNING: scatter and gather lengths don't match\n") ;
		return -1 ;
	}
	return cbuf_idx ;
}

void
do_crypt_function(struct sec2_q_regs *wqregs,
		  struct sec2_q_regs *cqregs,
		  MSP_SEC2_SA *sa,
		  int wqe_word1,
		  int wqe_word2,
		  int wqe_word3)
{
	static sec_ioctl desc ;
	static sec_cbuf cbuf[32] ; /* assuming nothing larger - cheap
                                      muffler rule applies here */
	int nsg ;		/* number of s/g entries */
	int ncbuf ;
	int alg ;
	int blkmode ;
	int i ;

	DBG_SIM("Begin do_crypt_function\n") ;

	nsg = ((wqe_word1 & 0xff) - 3) / 2 ;

	ncbuf = convert_sg_to_cbuf_crypt(wqregs, &desc, cbuf, nsg) ;

	if (ncbuf == -1)
		goto error ;

	desc.mode = 0 ;

	/* which encryption algorithm */
	alg = sa->flags & 0x01C00000 ;
	switch (alg) {
	case SAFLG_DES:
		break ;
	case SAFLG_3DES:
		desc.mode |= DMC_3DES ;
		break ;
	case SAFLG_CRYPTNULL:
		desc.mode |= DMC_MOD_PT ;
		break ;
	default:
		printk("ERROR on algorithm\n") ;
		goto error ;
	}

	/* which encryption block mode */
	blkmode = sa->flags & 0x380000 ;
	switch(blkmode) {
	case SAFLG_ECB:
		break ;

	case SAFLG_CBC_ENCRYPT:
		desc.mode |= DMC_MOD_CBC ;
		break ;

	case SAFLG_CBC_DECRYPT:
		desc.mode |= DMC_MOD_CBC | DMC_MOD_DEC ;
		break ;

	case SAFLG_CFB_ENCRYPT:
		desc.mode |= DMC_MOD_CFB ;
		break ;

	case SAFLG_CFB_DECRYPT:
		desc.mode |= DMC_MOD_CFB | DMC_MOD_DEC ;
		break ;

	default:
		printk("Error on block mode\n") ;
		goto error ;
	}

	if (sa->flags & SAFLG_DES_K1_DECRYPT)
		desc.mode |= DMC_K1_DEC ;

	if (sa->flags & SAFLG_DES_K2_DECRYPT)
		desc.mode |= DMC_K2_DEC ;

	if (sa->flags & SAFLG_DES_K3_DECRYPT)
		desc.mode |= DMC_K3_DEC ;

	memcpy((void *)&desc.ivhigh, sa->crypt_iv, 8) ;
	memcpy((void *)&desc.chainvar[0], sa->crypt_keys, 24) ;

	if (ncbuf == 0)
		desc.nda = 0 ;
	else
	{
		desc.nda = ((int)cbuf) | DNDA_VLD | DNDA_CNT ;
		for (i = 1 ; i < ncbuf; i++)
		{
			cbuf[i-1].nda = ((int)&cbuf[i]) | DNDA_VLD | DNDA_CNT ;
		}
		cbuf[ncbuf - 1].nda = 0 ;
	}

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "DES taking too long at entry\n") ;
			printk("\n=====================================\n\n") ;
			show_des_regs() ;
			Des->interface_control = DIFC_DES_RST ;

			break ;
		}
	}

	DBG_SIM("===== About to write DES hw %p\n", &desc) ;

#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
	dump_descriptor_nda((sec_desc *)&desc) ;
#endif

	Des->next_desc_address = (void *)(((unsigned long) &desc) | DNDA_VLD) ;

	Des->interface_control = DIFC_DMA_START ;

	i = 0 ;
	while (((Des->interface_status & (DIFS_DON|DIFS_BSY)) != 0 )
	       || ( (long)Des->next_desc_address & DNDA_VLD))
	{
		NanoDelay(500) ;
		if (i++ > 20000)
		{
			printk("\n=====================================\n\n") ;
			printk(KERN_WARNING "*** DES taking too long\n") ;
			printk("\n=====================================\n\n") ;

			dump_descriptor_nda((sec_desc *)&desc) ;
			show_des_regs() ;

			Des->interface_control = DIFC_DES_RST ;

			goto error ;
		}
	}

	/* completed with no problems */
	make_cq_entry(cqregs, wqe_word2, wqe_word3, 0) ;
	DBG_SIM("leaving crypt, good return\n") ;
	return ;

 error:
	DBG_SIM("Crypt error return\n") ;
	make_cq_entry(cqregs,
		      wqe_word2, wqe_word3, 0xffffffff) ;
	return ;
}

void
do_work_entry(struct sec2_q_regs *wqregs,
	      struct sec2_q_regs *cqregs,
	      MSP_SEC2_SA *sa,
	      int wqe_word1,
	      int wqe_word2,
	      int wqe_word3)
{
	DBG_SIM("Begin do_work_entry\n") ;
	switch (sa->flags & SAFLG_MODE_MASK)
	{
	case SAFLG_MODE_CRYPT:
		do_crypt_function(wqregs, cqregs, sa,
				  wqe_word1, wqe_word2, wqe_word3) ;
		break ;

	case SAFLG_MODE_ESP_IN:
		do_espin_function(wqregs, cqregs, sa,
				  wqe_word1, wqe_word2, wqe_word3) ;
		break;

	case SAFLG_MODE_ESP_OUT:
		do_espout_function(wqregs, cqregs, sa,
				   wqe_word1, wqe_word2, wqe_word3) ;
		break ;

	case SAFLG_MODE_HMAC:
		do_hmac_function(wqregs, cqregs, sa,
				 wqe_word1, wqe_word2, wqe_word3) ;
		break ;

	case SAFLG_MODE_HASH_PAD:
		do_hash_pad_function(wqregs, cqregs, sa,
				     wqe_word1, wqe_word2, wqe_word3) ;
		break ;
	case SAFLG_MODE_HASH:
		do_hash_function(wqregs, cqregs, sa,
				 wqe_word1, wqe_word2, wqe_word3) ;
		break ;
	default:
		{
			/*
			** not supported... best thing to do seems to be
			** to make a CQ entry with FFFFFFF status?
			*/
			int size ;
			/* swallow all scatter/gather elements */
			size = ((wqe_word1 & 0xff) - 3) * 4 ;
			wqregs->out = (wqregs->out + size) & (wqregs->size-1) ;
		
			make_cq_entry(cqregs,
				      wqe_word2, wqe_word3, 0xffffffff) ;
		}
	}
	DBG_SIM("End do_work_entry\n") ;
}
void
do_simulate_v2_hardware(void)
{
	int work_q ;
	struct sec2_q_regs *wqregs, *cqregs ;
	MSP_SEC2_SA *sa ;
	int wqe_word1 ;
	int wqe_word2 ;
	int wqe_word3 ;

	DBG_SIM("Entering v2 simulation\n") ;
	for (work_q = 0 ; work_q < HW_NR_WORK_QUEUES; work_q++)
	{
		wqregs = &sec2_regs->wq[work_q] ;
		while (wqregs->out != wqregs->in)
		{
#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
			debug_dump_sec_regs() ;
#endif

			sa = *(MSP_SEC2_SA **)(wqregs->base + wqregs->out) ;
			wqe_word1 = *(int *)(wqregs->base + wqregs->out + 4) ;

			wqregs->out = (wqregs->out + 8) & (wqregs->size - 1) ;

			wqe_word2 = *(int *)(wqregs->base + wqregs->out) ;
			wqe_word3 = *(int *)(wqregs->base + wqregs->out + 4) ;

			wqregs->out = (wqregs->out + 8) & (wqregs->size - 1) ;

			if (wqe_word1 & SEC2_WE_CTRL_CQ)
				cqregs = &sec2_regs->cq[1] ;
			else
				cqregs = &sec2_regs->cq[0] ;

			DBG_SIM("calling do work entry ( %p %x %x %x )\n",
			       sa, wqe_word1, wqe_word2, wqe_word3) ;
			do_work_entry(wqregs, cqregs,
				      sa, wqe_word1, wqe_word2, wqe_word3) ;
			DBG_SIM("do_work_entry returned\n") ;
#if defined(DEBUG_SIM_VERBOSE) && defined(DEBUG_SIM)
			debug_dump_sec_regs() ;
#endif

			*wqregs->avail_addr = wqregs->out ;
			*cqregs->avail_addr = cqregs->in ;
		}

	}
}

#endif

#ifdef VERIFICATION_TESTS

unsigned char espin_SrcData0 [] =
{
	0x12,0x34,0x56,0x78,0x00,0x00,0x00,0x01, // ESP header- 2words
	0x0c,0xc3,0x38,0x7d,0x73,0x2d,0x5a,0x34, // TCBC/CFB IV   - 8bytes
	0x16,0xed,0xdc,0x24,0x32,0x76,0x9d,0xac, // ESP TCBC encrption digest 48 bytes total
	0xc1,0xb5,0x89,0x11,0x88,0xef,0x7d,0x63, //
	0x9f,0x6c,0xda,0x54,0x42,0x79,0x2e,0x12,
	0x86,0x66,0x48,0xd8,0x6a,0x91,0xf7,0xe5,
	0x82,0xbe,0x9e,0xa4,0x29,0xeb,0xd5,0xef,
	0x6c,0x27,0x21,0x7b,0x36,0x8b,0x97,0x55
};
/* Next with ICV data bytes  */
unsigned char espin_SrcData0_a [] =
{           // ICV value for 2nd gather of ESP incoming bytes
	0x5C,0x27,0x6C,0xC8,0x44,0x2A,0x7B,0x24,
	0xBF,0xCF,0xC7,0x64,0xB3,0xC4,0xC2,0xEE,
	0x83,0x77,0x7A,0x94
};

/* IPAD- precalculated-0x064C662A,0xCC6ED1B0,0x6CFA9AB0,0x042F137F,0xCEA570EB
   OPAD- precalculated- 0xD62ACCCC,0x7E6A98B3,0xDF015B02,0xD885C3E0,0x9BD184CB
*/
unsigned int espin_SABlk0 [] =
{
	SAFLG_MODE_ESP_IN  |
	SAFLG_CV |
	SAFLG_SHA1 |
	SAFLG_CBC_DECRYPT |
	SAFLG_3DES |
	SAFLG_DES_K1_DECRYPT |
	SAFLG_DES_K3_DECRYPT , // increment sequence no,using HMAC CV
	0x12345678,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x064C662A,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0xCC6ED1B0,      // 10 Ipad Chain Var
	0x6CFA9AB0,      // 14 Ipad Chain Var
	0x042F137F,      // 18 Ipad Chain Var
	0xCEA570EB,      // 1c Ipad Chain Var
	0x57b3ea0b,      // 20 DES/TDES/AES128 Key-1 Hi
	0x3e1f83fe,      // 24 DES/TDES/AES128 Key-1 Lo
	0x57b3ea0b,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0x3e1f83fe,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x57b3ea0b,      // 30 DES/TDES Key-3 Hi
	0x3e1f83fe,      // 34 DES/TDES Key-3 Lo
	0x15151515,      // 38 AES256 Key Hi
	0x16161616,      // 3c AES256 Key Lo
	0xD62ACCCC,      // 40 Opad Chain Var
	0x7E6A98B3,      // 44 Opad Chain Var
	0xDF015B02,      // 48 Opad Chain Var
	0xD885C3E0,      // 4c Opad Chain Var
	0x9BD184CB,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0xdead387d,      // 5c CBC/CFB IV Hi
	0xdead5a34,      // 60 CBC/CFB IV Lo
	0x00000000,      // 64 AES CBC/CFB IV Hi
	0x00000000 } ;   // 68 AES CBC/CFB IV Lo

unsigned int hmac_SABlk0 [] =
{
	SAFLG_MODE_HMAC  |
	SAFLG_CV |
	SAFLG_SHA1,
	0x12345678,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x064C662A,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0xCC6ED1B0,      // 10 Ipad Chain Var
	0x6CFA9AB0,      // 14 Ipad Chain Var
	0x042F137F,      // 18 Ipad Chain Var
	0xCEA570EB,      // 1c Ipad Chain Var
	0x57b3ea0b,      // 20 DES/TDES/AES128 Key-1 Hi
	0x3e1f83fe,      // 24 DES/TDES/AES128 Key-1 Lo
	0x57b3ea0b,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0x3e1f83fe,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x57b3ea0b,      // 30 DES/TDES Key-3 Hi
	0x3e1f83fe,      // 34 DES/TDES Key-3 Lo
	0x15151515,      // 38 AES256 Key Hi
	0x16161616,      // 3c AES256 Key Lo
	0xD62ACCCC,      // 40 Opad Chain Var
	0x7E6A98B3,      // 44 Opad Chain Var
	0xDF015B02,      // 48 Opad Chain Var
	0xD885C3E0,      // 4c Opad Chain Var
	0x9BD184CB,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0xdead387d,      // 5c CBC/CFB IV Hi
	0xdead5a34,      // 60 CBC/CFB IV Lo
	0x00000000,      // 64 AES CBC/CFB IV Hi
	0x00000000 } ;   // 68 AES CBC/CFB IV Lo

unsigned char espin_exp_data [] =
{
	0x72, 0xac, 0x3c, 0x01, 0xc5, 0xf2, 0xf5, 0x7d,
	0xa0, 0xb1, 0x00, 0xbd, 0xf9, 0x9d, 0x96, 0x2a,
	0x63, 0x8b, 0x88, 0x62, 0xd5, 0x27, 0x23, 0x8b,
	0xad, 0xe5, 0x56, 0xe8, 0xf4, 0x3a, 0x17, 0xa9,
	0xe4, 0x0b, 0x02, 0x08, 0xa7, 0x66, 0x43, 0x91,
	0x62, 0x70, 0xb7, 0xc1, 0x72, 0x9e, 0x4e, 0xd3,
	0x5c, 0x27, 0x6c, 0xc8, 0x44, 0x2a, 0x7b, 0x24,
	0xbf, 0xcf, 0xc7, 0x64, 0xb3, 0xc4, 0xc2, 0xee,
	0x83, 0x77, 0x7a, 0x94
} ;
	

unsigned char espout_SrcData0 [] =
{
	0x72,0xac,0x3c,0x01,0xc5,0xf2,0xf5,0x7d, // original Payload 40 bytes
	0xa0,0xb1,0x00,0xbd,0xf9,0x9d,0x96,0x2a,
	0x63,0x8b,0x88,0x62,0xd5,0x27,0x23,0x8b,
	0xad,0xe5,0x56,0xe8,0xf4,0x3a,0x17,0xa9,
	0xe4,0x0b,0x02,0x08,0xa7,0x66,0x43,0x91,
	0x62,0x70,0xb7,0xc1,0x72,0x9e
};

/* IPAD- precalculated-0x064C662A,0xCC6ED1B0,0x6CFA9AB0,0x042F137F,0xCEA570EB
   OPAD- precalculated- 0xD62ACCCC,0x7E6A98B3,0xDF015B02,0xD885C3E0,0x9BD184CB
*/
unsigned int espout_SABlk0 [] =
{
	SAFLG_MODE_ESP_OUT |
	SAFLG_SI |
	SAFLG_CV |
	SAFLG_SHA1 |
	SAFLG_CBC_ENCRYPT |
	SAFLG_3DES |
	SAFLG_DES_K2_DECRYPT , // increment sequence no,using HMAC CV
	0x12345678,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x064C662A,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0xCC6ED1B0,      // 10 Ipad Chain Var
	0x6CFA9AB0,      // 14 Ipad Chain Var
	0x042F137F,      // 18 Ipad Chain Var
	0xCEA570EB,      // 1c Ipad Chain Var
	0x57b3ea0b,      // 20 DES/TDES/AES128 Key-1 Hi
	0x3e1f83fe,      // 24 DES/TDES/AES128 Key-1 Lo
	0x57b3ea0b,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0x3e1f83fe,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x57b3ea0b,      // 30 DES/TDES Key-3 Hi
	0x3e1f83fe,      // 34 DES/TDES Key-3 Lo
	0x15151515,      // 38 AES256 Key Hi
	0x16161616,      // 3c AES256 Key Lo
	0xD62ACCCC,      // 40 Opad Chain Var
	0x7E6A98B3,      // 44 Opad Chain Var
	0xDF015B02,      // 48 Opad Chain Var
	0xD885C3E0,      // 4c Opad Chain Var
	0x9BD184CB,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0x0cc3387d,      // 5c CBC/CFB IV Hi
	0x732d5a34,      // 60 CBC/CFB IV Lo
	0x00000000,      // 64 AES CBC/CFB IV Hi
	0x00000000 } ;   // 68 AES CBC/CFB IV Lo

/*
** simply used to decrypt the output of espout_SABlk0, to verify
** the padding values
*/
unsigned int decrypt_espout_SABlk0 [] =
{
	SAFLG_MODE_CRYPT |
	SAFLG_CBC_DECRYPT |
	SAFLG_3DES |
	SAFLG_DES_K1_DECRYPT |
	SAFLG_DES_K3_DECRYPT,
	0x12345678,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x064C662A,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0xCC6ED1B0,      // 10 Ipad Chain Var
	0x6CFA9AB0,      // 14 Ipad Chain Var
	0x042F137F,      // 18 Ipad Chain Var
	0xCEA570EB,      // 1c Ipad Chain Var
	0x57b3ea0b,      // 30 DES/TDES Key-3 Hi
	0x3e1f83fe,      // 34 DES/TDES Key-3 Lo
	0x57b3ea0b,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0x3e1f83fe,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x57b3ea0b,      // 20 DES/TDES/AES128 Key-1 Hi
	0x3e1f83fe,      // 24 DES/TDES/AES128 Key-1 Lo
	0x15151515,      // 38 AES256 Key Hi
	0x16161616,      // 3c AES256 Key Lo
	0xD62ACCCC,      // 40 Opad Chain Var
	0x7E6A98B3,      // 44 Opad Chain Var
	0xDF015B02,      // 48 Opad Chain Var
	0xD885C3E0,      // 4c Opad Chain Var
	0x9BD184CB,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0x0cc3387d,      // 5c CBC/CFB IV Hi
	0x732d5a34,      // 60 CBC/CFB IV Lo
	0x00000000,      // 64 AES CBC/CFB IV Hi
	0x00000000 } ;   // 68 AES CBC/CFB IV Lo



unsigned char espout_exp_data[] =
{
	0x12, 0x34, 0x56, 0x78, 0x00, 0x00, 0x00, 0x01, 
	0x0c, 0xc3, 0x38, 0x7d, 0x73, 0x2d, 0x5a, 0x34,
	0x16, 0xed, 0xdc, 0x24, 0x32, 0x76, 0x9d, 0xac, 
	0xc1, 0xb5, 0x89, 0x11, 0x88, 0xef, 0x7d, 0x63,
	0x9f, 0x6c, 0xda, 0x54, 0x42, 0x79, 0x2e, 0x12, 
	0x86, 0x66, 0x48, 0xd8, 0x6a, 0x91, 0xf7, 0xe5,
	0x82, 0xbe, 0x9e, 0xa4, 0x29, 0xeb, 0xd5, 0xef, 
	0x78, 0x76, 0x04, 0x24, 0x9e, 0x5b, 0xa0, 0x89,
	0x4f, 0x90, 0x05, 0x30, 0x1c, 0x83, 0x9e, 0x18, 
	0xb9, 0xf5, 0x5f, 0xa4, 0x68, 0x5d, 0xe7, 0xc7,
	0xb8, 0x00, 0x30, 0xee
} ;

unsigned char Sha1EncExpData0 [] =
{
                        0x16,0xed,0xdc,0x24,0x32,0x76,0x9d,0xac, // ESP TCBC encrption digest 48 bytes total
                        0xc1,0xb5,0x89,0x11,0x88,0xef,0x7d,0x63, //
                        0x9f,0x6c,0xda,0x54,0x42,0x79,0x2e,0x12,
                        0x86,0x66,0x48,0xd8,0x6a,0x91,0xf7,0xe5,
                        0x82,0xbe,0x9e,0xa4,0x29,0xeb,0xd5,0xef,
                        0x10,0x5c,0x77,0xcf,0x0e,0x7a,0xe1,0xa7, /* output with padding */
};

unsigned int decode_SA [] =
{
	SAFLG_MODE_CRYPT |
	SAFLG_SI |
	SAFLG_CV |
	SAFLG_SHA1 |
	SAFLG_CBC_DECRYPT |
	SAFLG_3DES |
	SAFLG_DES_K1_DECRYPT |
	SAFLG_DES_K3_DECRYPT, // increment sequence no,using HMAC CV
	0x12345678,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x064C662A,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0xCC6ED1B0,      // 10 Ipad Chain Var
	0x6CFA9AB0,      // 14 Ipad Chain Var
	0x042F137F,      // 18 Ipad Chain Var
	0xCEA570EB,      // 1c Ipad Chain Var
	0x57b3ea0b,      // 30 DES/TDES Key-3 Hi
	0x3e1f83fe,      // 34 DES/TDES Key-3 Lo
	0x57b3ea0b,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0x3e1f83fe,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x57b3ea0b,      // 20 DES/TDES/AES128 Key-1 Hi
	0x3e1f83fe,      // 24 DES/TDES/AES128 Key-1 Lo
	0x15151515,      // 38 AES256 Key Hi
	0x16161616,      // 3c AES256 Key Lo
	0xD62ACCCC,      // 40 Opad Chain Var
	0x7E6A98B3,      // 44 Opad Chain Var
	0xDF015B02,      // 48 Opad Chain Var
	0xD885C3E0,      // 4c Opad Chain Var
	0x9BD184CB,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0x0cc3387d,      // 5c CBC/CFB IV Hi
	0x732d5a34,      // 60 CBC/CFB IV Lo
	0x00000000,      // 64 AES CBC/CFB IV Hi
	0x00000000 } ;   // 68 AES CBC/CFB IV Lo

/* Case #5: Sample transport-mode ESP packet (ping 192.168.123.100) */

/*   Original packet:
**   IP header (20 bytes): 45000054 08f20000 4001f9fe c0a87b03 c0a87b64
**   Data (64 bytes):
**   08000ebd a70a0000 8e9c083d b95b0700 08090a0b 0c0d0e0f 10111213 14151617
**   18191a1b 1c1d1e1f 20212223 24252627 28292a2b 2c2d2e2f 30313233 34353637
*/
unsigned char aesesp_srcdata[] =
{
	/* 0x45, 0x00, 0x00, 0x54, 0x08, 0xf2, 0x00, 0x00,
	** 0x40, 0x01, 0xf9, 0xfe, 0xc0, 0xa8, 0x7b, 0x03,
	** 0xc0, 0xa8, 0x7b, 0x64,
	*/
	
	/*   Data (64 bytes): */
	0x08, 0x00, 0x0e, 0xbd, 0xa7, 0x0a, 0x00, 0x00,
	0x8e, 0x9c, 0x08, 0x3d, 0xb9, 0x5b, 0x07, 0x00,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
} ;

unsigned int aesesp_sa [] =
{
	SAFLG_MODE_ESP_OUT  |
	SAFLG_CV |
	SAFLG_AES_128 |
	SAFLG_HASHNULL |
	SAFLG_CBC_ENCRYPT,
	0x00004321,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x00000000,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0x00000000,      // 10 Ipad Chain Var
	0x00000000,      // 14 Ipad Chain Var
	0x00000000,      // 18 Ipad Chain Var
	0x00000000,      // 1c Ipad Chain Var
	0x90d382b4,      // 20 DES/TDES/AES128 Key-1 Hi
	0x10eeba7a,      // 24 DES/TDES/AES128 Key-1 Lo
	0xd938c46c,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0xec1a82bf,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x00000000,      // 30 DES/TDES Key-3 Hi
	0x00000000,      // 34 DES/TDES Key-3 Lo
	0x00000000,      // 38 AES256 Key Hi
	0x00000000,      // 3c AES256 Key Lo
	0x00000000,      // 40 Opad Chain Var
	0x00000000,      // 44 Opad Chain Var
	0x00000000,      // 48 Opad Chain Var
	0x00000000,      // 4c Opad Chain Var
	0x00000000,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0xe96e8c08,      // 5c CBC/CFB IV Hi
	0xab465763,      // 60 CBC/CFB IV Lo
	0xfd098d45,      // 64 AES CBC/CFB IV Hi
	0xdd3ff893 } ;   // 68 AES CBC/CFB IV Lo

unsigned int aesesp_pre_sa [] =
{
	SAFLG_MODE_CRYPT  |
	SAFLG_AES_128 |
	SAFLG_ECB,
	0x00004321,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x00000000,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0x00000000,      // 10 Ipad Chain Var
	0x00000000,      // 14 Ipad Chain Var
	0x00000000,      // 18 Ipad Chain Var
	0x00000000,      // 1c Ipad Chain Var
	0x90d382b4,      // 20 DES/TDES/AES128 Key-1 Hi
	0x10eeba7a,      // 24 DES/TDES/AES128 Key-1 Lo
	0xd938c46c,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0xec1a82bf,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x00000000,      // 30 DES/TDES Key-3 Hi
	0x00000000,      // 34 DES/TDES Key-3 Lo
	0x00000000,      // 38 AES256 Key Hi
	0x00000000,      // 3c AES256 Key Lo
	0x00000000,      // 40 Opad Chain Var
	0x00000000,      // 44 Opad Chain Var
	0x00000000,      // 48 Opad Chain Var
	0x00000000,      // 4c Opad Chain Var
	0x00000000,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0xe96e8c08,      // 5c CBC/CFB IV Hi
	0xab465763,      // 60 CBC/CFB IV Lo
	0xfd098d45,      // 64 AES CBC/CFB IV Hi
	0xdd3ff893 } ;   // 68 AES CBC/CFB IV Lo

unsigned int aesesp_in_sa [] =
{
	SAFLG_MODE_ESP_IN  |
	SAFLG_CV |
	SAFLG_AES_128 |
	SAFLG_AES_DECRYPT |
	SAFLG_HASHNULL |
	SAFLG_CBC_DECRYPT,
	0x00004321,      // 04 ESP hdr word 0
	0x00000001,      // 08 ESP hdr word1 Seq#
	0x00000000,      // 0c Ipad Chain Var, filled by WQ1 scatter output(Hash(Key xor 0x36))
	0x00000000,      // 10 Ipad Chain Var
	0x00000000,      // 14 Ipad Chain Var
	0x00000000,      // 18 Ipad Chain Var
	0x00000000,      // 1c Ipad Chain Var
	0x90d382b4,      // 20 DES/TDES/AES128 Key-1 Hi
	0x10eeba7a,      // 24 DES/TDES/AES128 Key-1 Lo
	0xd938c46c,      // 28 DES/TDES/AES128/AES192 Key-2 Hi
	0xec1a82bf,      // 2c DES/TDES/AES128/AES192 Key-2 Lo
	0x00000000,      // 30 DES/TDES Key-3 Hi
	0x00000000,      // 34 DES/TDES Key-3 Lo
	0x00000000,      // 38 AES256 Key Hi
	0x00000000,      // 3c AES256 Key Lo
	0x00000000,      // 40 Opad Chain Var
	0x00000000,      // 44 Opad Chain Var
	0x00000000,      // 48 Opad Chain Var
	0x00000000,      // 4c Opad Chain Var
	0x00000000,      // 50 Opad Chain Var
	0x00000000,      // 54 hash Initial length
	0x00000200,      // 58 hash Initial length
	0xe96e8c08,      // 5c CBC/CFB IV Hi
	0xab465763,      // 60 CBC/CFB IV Lo
	0xfd098d45,      // 64 AES CBC/CFB IV Hi
	0xdd3ff893 } ;   // 68 AES CBC/CFB IV Lo

/*   Key: 90d382b4 10eeba7a d938c46c ec1a82bf
**   SPI: 4321
**   Source address: 192.168.123.3
**   Destination address: 192.168.123.100
**   Sequence number: 1
**   IV: e96e8c08 ab465763 fd098d45 dd3ff893
**   
**   Augment data with:
**   Padding: 01020304 05060708 090a0b0c 0d0e
**   Pad length: 0e
**   Next header: 01 (ICMP)
**
**   Pre-encryption Data with padding, pad length and next header (80 bytes):
**   08000ebd a70a0000 8e9c083d b95b0700 08090a0b 0c0d0e0f 10111213 14151617
**   18191a1b 1c1d1e1f 20212223 24252627 28292a2b 2c2d2e2f 30313233 34353637
**   01020304 05060708 090a0b0c 0d0e0e01
**
**   Post-encryption packet with SPI, Sequence number, IV:
**   IP header: 4500007c 08f20000 4032f9a5 c0a87b03 c0a87b64
**   SPI/Seq #: 00004321 00000001
**   IV: e96e8c08 ab465763 fd098d45 dd3ff893
**   Encrypted Data (80 bytes):
**   f663c25d 325c18c6 a9453e19 4e120849 a4870b66 cc6b9965 330013b4 898dc856
**   a4699e52 3a55db08 0b59ec3a 8e4b7e52 775b07d1 db34ed9c 538ab50c 551b874a
**   a269add0 47ad2d59 13ac19b7 cfbad4a6
*/

unsigned char aesesp_expdata[] =
{
	/* SPI/Seq #: */
	0x00, 0x00, 0x43, 0x21, 0x00, 0x00, 0x00, 0x01,

	/* IV: */
	0xe9, 0x6e, 0x8c, 0x08, 0xab, 0x46, 0x57, 0x63,
	0xfd, 0x09, 0x8d, 0x45, 0xdd, 0x3f, 0xf8, 0x93,

	/* Encrypted Data (80 bytes): */
	0xf6, 0x63, 0xc2, 0x5d, 0x32, 0x5c, 0x18, 0xc6,
	0xa9, 0x45, 0x3e, 0x19, 0x4e, 0x12, 0x08, 0x49,
	0xa4, 0x87, 0x0b, 0x66, 0xcc, 0x6b, 0x99, 0x65,
	0x33, 0x00, 0x13, 0xb4, 0x89, 0x8d, 0xc8, 0x56,
	0xa4, 0x69, 0x9e, 0x52, 0x3a, 0x55, 0xdb, 0x08,
	0x0b, 0x59, 0xec, 0x3a, 0x8e, 0x4b, 0x7e, 0x52,
	0x77, 0x5b, 0x07, 0xd1, 0xdb, 0x34, 0xed, 0x9c,
	0x53, 0x8a, 0xb5, 0x0c, 0x55, 0x1b, 0x87, 0x4a,
	0xa2, 0x69, 0xad, 0xd0, 0x47, 0xad, 0x2d, 0x59,
	0x13, 0xac, 0x19, 0xb7, 0xcf, 0xba, 0xd4, 0xa6,
};

unsigned char aesesp_in_expdata[] =
{
	/*   Data (64 bytes): */
	0x08, 0x00, 0x0e, 0xbd, 0xa7, 0x0a, 0x00, 0x00,
	0x8e, 0x9c, 0x08, 0x3d, 0xb9, 0x5b, 0x07, 0x00,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	/* padding, trailer (16 bytes): */
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0e, 0x01,
} ;

void
verification_test_complete(void *parm, unsigned int status)
{
	/* parm points at array of two ints; setting first non-zero
	   means operation complete; second int gets status */
	int *p = (int *) parm ;
	p[1] = status ;
	p[0] = 1 ;
}

void
verification_tests(void)
{
	int comp[2] ;
	int i ;
	int padlen ;

	static char DestBuf[1024] ;

	/* esp in test */

	printk("----------------- Doing ESP IN test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     3,	/* n_sg_entries */
			     (void*)espin_SABlk0, /* sa */
			     0x00060000, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)espin_SrcData0, /* address */
			64	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)espin_SrcData0_a, /* address */
			20	/* size */
			) ;
	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			68	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	if (memcmp(DestBuf, espin_exp_data, sizeof(espin_exp_data)))
		printk("************ Test compare failed\n") ;
	else
		printk("=== P A S S E D ===\n") ;

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < 68; i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	/* esp out test */

	printk("----------------- Doing ESP OUT test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     3,	/* n_sg_entries */
			     (void*)espout_SABlk0, /* sa */
			     0x06060000, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)espout_SrcData0, /* address */
			40	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			40 + 8 + 8 + 6 + 2 + 20	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	if (memcmp(DestBuf, espout_exp_data, sizeof(espin_exp_data)))
		printk("************ Test compare failed\n") ;
	else
		printk("=== P A S S E D ===\n") ;

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < 84; i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	/* crypt decode */

	printk("----------------- Doing CRYPT test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     2,	/* n_sg_entries */
			     (void*)decode_SA, /* sa */
			     0x06060000, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)Sha1EncExpData0, /* address */
			48	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			48	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < 48; i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	printk("----------------- Doing HMAC test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     3,	/* n_sg_entries */
			     (void*)hmac_SABlk0, /* sa */
			     0x00060000, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)espin_SrcData0, /* address */
			64	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			20	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	if (memcmp(DestBuf, espin_SrcData0_a, sizeof(espin_SrcData0_a)))
		printk("************ Test compare failed\n") ;
	else
		printk("=== P A S S E D ===\n") ;

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < sizeof(espin_SrcData0_a); i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	
	printk("----------------- Doing AES ESPout test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     2,	/* n_sg_entries */
			     (void*)aesesp_sa, /* sa */
			     0x0e010000, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)aesesp_srcdata, /* address */
			64	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			104	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	if (memcmp(DestBuf, aesesp_expdata, sizeof(aesesp_expdata)))
		printk("************ Test compare failed\n") ;
	else
		printk("=== P A S S E D ===\n") ;

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < sizeof(aesesp_expdata); i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	


	printk("----------------- Doing AES ESPin test\n") ;

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     2,	/* n_sg_entries */
			     (void*)aesesp_pre_sa, /* sa */
			     SEC2_WE_CTRL_AKO, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)aesesp_srcdata, /* address */
			16	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			(void *)&aesesp_in_sa[8], /* address */
			32	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	/* key precalc'd, now do operation */

	comp[0] = 0 ;
	
	msp_sec2_new_request(0, /* work_q */
			     2,	/* n_sg_entries */
			     (void*)aesesp_in_sa, /* sa */
			     0, /* control */
			     verification_test_complete, /* cbk_fn */
			     comp, /* cbk_parm */
			     BLK_SLEEP	/* block */
			     ) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)aesesp_expdata, /* address */
			104	/* size */
			) ;

	/* dummy ICV for hashnull */
	msp_sec2_add_sg(0,	/* work_q */
			SG_GATHER,
			(void*)aesesp_expdata, /* address */
			12	/* size */
			) ;

	msp_sec2_add_sg(0,	/* work_q */
			SG_SCATTER,
			DestBuf, /* address */
			80	/* size */
			) ;
	msp_sec2_end_request(0) ;

	i = 0 ;
	while (comp[0] == 0)
	{
		if (i++ > 1000)
		{
			printk("waiting too long\n") ;
			dump_sec_regs() ;
			break ;
		}
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	if (memcmp(DestBuf, aesesp_in_expdata, sizeof(aesesp_in_expdata)))
		printk("************ Test compare failed\n") ;
	else
		printk("=== P A S S E D ===\n") ;

	printk("test complete... Here's the data\n") ;
	for(i = 0 ; i < sizeof(aesesp_in_expdata); i++)
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", DestBuf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", DestBuf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", DestBuf[i] & 0xff) ;
			break ;
		}
	}
	printk("\n") ;
	
	


	/* esp padding generation test */

	printk("----------------- Doing ESP padding generation test\n") ;

	for (padlen = 0 ; padlen < 16; padlen++)
	{

		comp[0] = 0 ;
	
		msp_sec2_new_request(0, /* work_q */
				     3,	/* n_sg_entries */
				     (void*)espout_SABlk0, /* sa */
				     0x00060000 | (padlen << 24), /* control */
				     verification_test_complete, /* cbk_fn */
				     comp, /* cbk_parm */
				     BLK_SLEEP	/* block */
				     ) ;

		msp_sec2_add_sg(0,	/* work_q */
				SG_GATHER,
				(void*)espout_SrcData0, /* address */
				38 - padlen	/* size */
				) ;

		msp_sec2_add_sg(0,	/* work_q */
				SG_SCATTER,
				DestBuf, /* address */
				38 + 8 + 8 + 2 + 20	/* size */
				) ;
		msp_sec2_end_request(0) ;

		/* now, decrypt it */

		msp_sec2_new_request(0, /* work_q */
				     3,	/* n_sg_entries */
				     (void*)decrypt_espout_SABlk0, /* sa */
				     0x00060000 | (padlen << 24), /* control */
				     verification_test_complete, /* cbk_fn */
				     comp, /* cbk_parm */
				     BLK_SLEEP	/* block */
				     ) ;

		msp_sec2_add_sg(0,	/* work_q */
				SG_GATHER,
				(void*)DestBuf + 16, /* address */
				38 + 2	/* size */
				) ;

		msp_sec2_add_sg(0,	/* work_q */
				SG_SCATTER,
				DestBuf + 16, /* address */
				38 + 2 /* size */
				) ;
		msp_sec2_end_request(0) ;

		i = 0 ;
		while (comp[0] == 0)
		{
			if (i++ > 1000)
			{
				printk("waiting too long\n") ;
				dump_sec_regs() ;
				break ;
			}
			msp_sec2_poll_completion_queues(3, 0) ;
		}

		printk("=== Here's the data for padlen %d\n", padlen) ;
		for(i = 0 ; i < 76 ; i++)
		{
			switch(i % 16)
			{
			case 0:
				printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
				break;
			case 7:
				printk("%02x - ", DestBuf[i] & 0xff) ;
				break ;
			case 15:
				printk("%02x\n", DestBuf[i] & 0xff) ;
				break ;
			default:
				printk("%02x ", DestBuf[i] & 0xff) ;
				break ;
			}
		}
		printk("\n") ;
	}	
}

#endif /* VERIFICATION_TESTS */

#endif /* CONFIG_BRECIS_SEC_V2 */
