/*
** $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/brecis/brecis_sec_v1.c,v 1.8 2003/05/20 19:08:10 swahl Exp $
*/

/*
** uClinux driver for Brecis security engine (first version).
**
** Copyright 2001,2002 Brecis Communictaions
**
*/


#include <linux/config.h>
#include <linux/version.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <linux/proc_fs.h>
#include "brecis/msp_sec.h"
#include "brecis/msp_sec_kern.h"
#include <asm/brecis/prom.h>

extern void NanoDelay(int) ;
#define NDTIME 250

/*------------------------------------------------------------
** Kernel interface debugging/testing
*/

/* #define KERNEL_INTERFACE_TEST */

#ifdef KERNEL_INTERFACE_TEST
	int sec_count[2] ;
#	define SECUP(X) sec_count[(X)]++
#	define SECDN(X) sec_count[(X)]--
	static int kernel_intface_test(unsigned long arg) ;
#else
#	define SECUP(X)
#	define SECDN(X)
#endif

typedef void (*op_fns)(sec_desc *desc) ;

#ifdef CONFIG_BRECIS_SEC_V1
static void msp_sec_do_des_operation(sec_desc *desc) ;
static void msp_sec_do_hsh_operation(sec_desc *desc) ;
static void msp_sec_do_hmac_operation(sec_desc *desc) ;
static void msp_sec_do_hmac2_operation(sec_desc *desc) ;
static op_fns v1_ops[] = {
	msp_sec_do_des_operation,
	msp_sec_do_hsh_operation,
	msp_sec_do_hmac_operation,
	msp_sec_do_hmac2_operation
} ;
#endif

#ifdef CONFIG_BRECIS_SEC_V2
int  msp_secv2_init(void) ;
void msp_secv2_exit(void) ;
int  msp_secv2_read_proc(char *page, char **start, off_t off,
			 int count, int *eof, void *data) ;
int  msp_secv2_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg) ;
#endif
#ifdef CONFIG_BRECIS_SEC_V1ON2_COMPAT
void msp_sec_do_des_operation_v2(sec_desc *desc) ;
void msp_sec_do_hsh_operation_v2(sec_desc *desc) ;
void msp_sec_do_hmac_operation_v2(sec_desc *desc) ;
void msp_sec_do_hmac2_operation_v2(sec_desc *desc) ;
static op_fns v2_ops[] = {
	msp_sec_do_des_operation_v2,
	msp_sec_do_hsh_operation_v2,
	msp_sec_do_hmac_operation_v2,
	msp_sec_do_hmac2_operation_v2
} ;
#endif

static op_fns *sec1_operations = 0 ;

static int info_hw ;
static int info_driver ;

/*------------------------------------------------------------
** Tuning delays.
**
** Approximate speed of des and hash engines, to figure delays
** based on packet length.
*/
#define DES_SPEED 400	/* in MBits/Sec */
#define DES_DLY(x) ( (8000 * (x)) / DES_SPEED )

#define HASH_SPEED 340		/* in MBits/Sec */
#define HASH_DLY(x) ( (8000 * (x)) / HASH_SPEED )

/*------------------------------------------------------------
** various debugging functions
*/

/* #define DBG_HSH_BUG */
/* #define DBG_HSH_DESC */
/* #define DEBUG */
/* #define DBG_DES_DESC */
/* #define DEBUG_FREELIST */
/* #define DEBUG_DESC_HIST */

#ifdef DEBUG
#define DBG_SEC(a1, a2...)	printk("SECURITY " a1, ##a2)
#else
#define DBG_SEC(a...)
#endif


static struct timer_list rng_timer ;
static int rng_timer_used = 0 ;

int init_ok = 1 ;

/*------------------------------------------------------------
** various macros
*/

#define ACCESS_OK(type,addr,size) ((addr) != 0 && access_ok(type,addr,size))

#define HASH_RSLT_LEN(mode) ( (mode & HMC_SHA1) ? HSH_LEN_SHA1 : HSH_LEN_MD5 )

#define NXT_NDA(ptr)	((sec_desc *)((ptr)->ioctl.nda & ~3))

#define DESC_POOL_SIZE 60	/* number of descriptors in the pool */

/* must not use one while the other is busy */
#	define DES_STATUS_MASK	( DIFS_DON | DIFS_BSY | HIFS_DON | HIFS_BSY )
#	define DES_STATUS_VAL	0
#	define HSH_STATUS_MASK	( DIFS_DON | DIFS_BSY | HIFS_DON | HIFS_BSY )
#	define HSH_STATUS_VAL	0

#ifdef CONFIG_BRECIS_SEC_V1
static msp_sec_des_regs *Des = (msp_sec_des_regs *) (DES_NDA_REG) ;
static msp_sec_hsh_regs *Hsh = (msp_sec_hsh_regs *) (HSH_NDA_REG) ;
#endif

/*
** New security hardware ("Polo") requires us to not read
** from the interface status register of one engine while
** the other is running.  This variable will point at the
** interface status register of the last running engine.
**
** This is where we will look to see if the hardware is idle.
** The new hardware doesn't shadow the bits between registers.
** But since only one can be active at a time, we will always
** be looking at the only register with active bits.
*/
volatile unsigned long *cur_interface_status =
	& ((msp_sec_des_regs *) (DES_NDA_REG))->interface_status ;

static struct proc_dir_entry *proc_entry ;
static int msp_sec_proc_register(void) ;
static void msp_sec_proc_unregister(void) ;
#ifdef CONFIG_BRECIS_SEC_V1
static void do_partial_hash(sec_desc *desc, int contflag);
static int do_hash_workaround(sec_desc *desc) ;
#endif
#ifdef DEBUG_DESC_HIST
static void show_des_regs (void) ;
static void show_hsh_regs (void) ;
#endif

/*------------------------------------------------------------
** Descriptor history, used for debugging.
**
** Maintains a list of the last 8 descriptor chains used by the
** DES and Hash engines.  Up to 8 continuation descriptors for
** each chain are kept, also.
*/


#ifndef DEBUG_DESC_HIST
#define RECORD_DES(desc)
#define RECORD_HSH(desc)
#else
#define RECORD_DES(desc) record_desc((desc), &des_desc_hist)
#define RECORD_HSH(desc) record_desc((desc), &hash_desc_hist)

struct desc_hist_ent {
	int ncdesc ;
	sec_desc init_desc ;
	sec_desc cont_desc[8] ;
};
static struct desc_hist {
	int current ;
	struct desc_hist_ent ent[8] ;
} hash_desc_hist, des_desc_hist ;


static void record_desc(sec_desc *desc, struct desc_hist *desc_hist )
{
	struct desc_hist_ent *dh ;
	dh = &desc_hist->ent[desc_hist->current] ;
	desc_hist->current = (desc_hist->current + 1) % 8 ;

	dh->init_desc = *desc ;
	dh->ncdesc = 0 ;
	while(NXT_NDA(desc))
	{
		desc = NXT_NDA(desc) ;
		if (dh->ncdesc < 8)
			dh->cont_desc[dh->ncdesc] = *desc ;
		dh->ncdesc ++ ;
	}
}

static void dump_desc_hist(int first, struct desc_hist *desc_hist )
{
	struct desc_hist_ent *dh ;
	int i, j, k ;
	j = (desc_hist->current + first) % 8 ;
	for (i = first; i < 8; i++)
	{
		dh = &desc_hist->ent[j] ;
		j = (j + 1) % 8 ;

		printk("INIT Desc %d\n", i) ;
		printk("DESCR:    nda %08x   len %08x  mode %08x\n",
		       dh->init_desc.ioctl.nda,
		       dh->init_desc.ioctl.buflen,
		       dh->init_desc.ioctl.mode);
		printk("DESCR:    src %8p   dst %8p    id %08x\n",
		       dh->init_desc.ioctl.srcaddr,
		       dh->init_desc.ioctl.dstaddr,
		       dh->init_desc.ioctl.id);
		printk("DESCR:    ivl %08x   ivh %08x  type %08x\n",
		       dh->init_desc.ioctl.ivhigh,
		       dh->init_desc.ioctl.ivlow,
		       dh->init_desc.ioctl.type);
		printk("DESCR:    ch0 %08x   ch1 %08x   ch2 %08x\n",
		       dh->init_desc.ioctl.chainvar[0],
		       dh->init_desc.ioctl.chainvar[1],
		       dh->init_desc.ioctl.chainvar[2]) ;

		printk("DESCR:    ch3 %08x   ch4 %08x   ch5 %08x\n",
		       dh->init_desc.ioctl.chainvar[3],
		       dh->init_desc.ioctl.chainvar[4],
		       dh->init_desc.ioctl.chainvar[5]) ;

		printk("DESCR:    opt %08x  stat %08x   nxt %8p\n",
		       dh->init_desc.ioctl.options,
		       dh->init_desc.ioctl.status,
		       dh->init_desc.next) ;

		printk("DESCR:   %d continue descriptors\n", dh->ncdesc) ;
		for (k = 0 ; k < 8 && k < dh->ncdesc; k++)
		{
		printk("CDESCR:            nda %08x   len %08x  mode %08x\n",
		       dh->cont_desc[k].ioctl.nda,
		       dh->cont_desc[k].ioctl.buflen,
		       dh->cont_desc[k].ioctl.mode);
		printk("CDESCR:            src %8p   dst %8p    id %08x\n",
		       dh->cont_desc[k].ioctl.srcaddr,
		       dh->cont_desc[k].ioctl.dstaddr,
		       dh->cont_desc[k].ioctl.id);
		printk("CDESCR:            opt %08x  stat %08x   nxt %8p\n\n",
		       dh->cont_desc[k].ioctl.options,
		       dh->cont_desc[k].ioctl.status,
		       dh->cont_desc[k].next) ;

		}
	}
}
#endif



#if defined(CONFIG_BRECIS_SEC_V1) || defined(CONFIG_BRECIS_SEC_V1ON2_COMPAT)

/*************************************************************
** CACHE and other memory stuff.
*/

#define SYNC() __asm__ volatile("sync");

/* define for a Cache Invalidate */
#define msp_hit_inval_dcache(addr, ofst)    \
({ \
    __asm__ volatile (	"	.set push\n" \
			"	.set mips3\n" \
			"	cache 0x11,%1(%0)\n" \
			"	.set pop" \
		: : "r" (addr), "I" (ofst) );   \
})

/************************************************************************
 * Function Name:	invalidate_buffer
 * Stolen From:         triad.c (ethernet driver
 *
 * Purpose:	Invalidate buffer cache
 *
 *         SRW I think the following are of concern if you have
 *         a writeback cache, but we have a write-through, so
 *         invalidating more than just your memory is no problem:
 *
 * 		- must start on 16 byte boundary
 * 		- must "own" to next higher 16 byte boundary from end of buffer
/ ***********************************************************************/
static void
invalidate_buffer( void * bufaddr, unsigned int length)
{
        unsigned long end = (unsigned long)bufaddr + length ;
        bufaddr = (void *) ((unsigned int) bufaddr & ~(L1_CACHE_BYTES - 1));

        while ( (unsigned long) bufaddr < (end - 8 * L1_CACHE_BYTES) )
        {
                msp_hit_inval_dcache(  bufaddr , (0 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (1 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (2 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (3 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (4 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (5 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (6 * L1_CACHE_BYTES) );
                msp_hit_inval_dcache(  bufaddr , (7 * L1_CACHE_BYTES) );
                bufaddr += 8 * L1_CACHE_BYTES;
        }

        while ( (unsigned long) bufaddr < end )
        {
                msp_hit_inval_dcache(  bufaddr, 0 );
                bufaddr += L1_CACHE_BYTES;
        }
}

#endif

/*************************************************************
** hash function padding tools.
*************************************************************/
unsigned char hash_pad_buf[69] = { 0x80 } ; /* all other bytes zero */

typedef struct pad_struct {
	/* two descriptors, one for pad space, one for length buffer */
	sec_desc pad_desc[2] ;
	/* buffer for length */
	union {
		unsigned long len[2] ;
		unsigned char clen[8] ;
	} u ;
} pad_struct ;

/*************************************************************
** list of DMA descriptors used by the driver.
*************************************************************/
desc_list free_desc_list ;

/*************************************************************
** Descriptor List Functions
*************************************************************/

void
desc_add_to_list(desc_list *list, sec_desc *desc)
{
	if (list->head == NULL)
		list->head = desc ;
	else
		list->tail->next = desc ;

	while (desc->next)
		desc = desc->next ;
	list->tail = desc ;
}

sec_desc *
desc_get_next(void)
{
	sec_desc *d ;

	d = free_desc_list.head ;

	if (d)
	{
		free_desc_list.head = d->next ;
		d->next = NULL ;
		if (free_desc_list.tail == d)
			free_desc_list.tail = NULL ;
	}

	return d ;
}

void
desc_free_chain(sec_desc *desc)
{
	desc_add_to_list(&free_desc_list, desc) ;
}

sec_desc *
desc_free_one(sec_desc *desc)
{
	sec_desc *d ;

	d = desc->next ;
	desc->next = NULL ;
	desc_add_to_list(&free_desc_list, desc) ;

	return d ;
}

#ifdef DEBUG_FREELIST

#define PRINT_FREELIST print_freelist()

static void
print_freelist(void)
{
	sec_desc *d ;
	int cnt = 0 ;

#if VERBOSE_DBG
	DBG_SEC("Free List:\n") ;
#endif
	d = free_desc_list.head ;
	while (d)
	{
#if VERBOSE_DBG
		DBG_SEC("    Desc 0x%x, nxt 0x%x head 0x%x\n",
			(unsigned int)d,
			(unsigned int)d->next,
			(unsigned int)d->head) ;
#endif
		cnt++ ;
		d = d->next ;
	}
#if VERBOSE_DBG
	DBG_SEC("end of list, tail is 0x%x\n",
		(unsigned int)free_desc_list.tail) ;
#endif
	DBG_SEC("freelist size %d\n", cnt) ;
}
#else

#define PRINT_FREELIST do {} while(0) 

#endif

int
init_descriptor_pool(void)
{
	int i ;
	sec_desc *d ;

	for (i = 0 ; i < DESC_POOL_SIZE ; i++)
	{
		d = kmalloc(sizeof(struct sec_desc), GFP_KERNEL) ;
		if (!d)
		{
			while ( (d = desc_get_next()) )
				kfree(d) ;
			return -ENOMEM ;
		}
		d->next = 0 ;
		desc_add_to_list(&free_desc_list, d) ;
	}

	PRINT_FREELIST ;

	return 0 ;
}

void
destroy_descriptor_pool(void)
{
	sec_desc *d ;

	while ( (d = desc_get_next()) )
		kfree(d) ;
}

#if defined(DBG_DES_DESC) || defined(DBG_HSH_DESC)
static void
dump_descriptor(sec_desc *desc)
{
	/*
	** For hash continuation, security engine writes
	** into the descriptor.  Make sure we're looking at fresh
	** data
	*/
	invalidate_buffer(desc->ioctl.chainvar, 6*sizeof(long)) ;

	while (desc)
	{
		printk("DESCR at %x\n", (int)desc) ;
		printk("DESCR:    nda %08x   len %08x  mode %08x\n",
		       desc->ioctl.nda, desc->ioctl.buflen, desc->ioctl.mode);
		printk("DESCR:    src %8p   dst %8p    id %08x\n",
		       desc->ioctl.srcaddr, desc->ioctl.dstaddr, desc->ioctl.id);
		printk("DESCR:    ivl %08x   ivh %08x  type %08x\n",
		       desc->ioctl.ivhigh, desc->ioctl.ivlow, desc->ioctl.type);
		printk("DESCR:    ch0 %08x   ch1 %08x   ch2 %08x\n",
		       desc->ioctl.chainvar[0],
		       desc->ioctl.chainvar[1],
		       desc->ioctl.chainvar[2]) ;

		printk("DESCR:    ch3 %08x   ch4 %08x   ch5 %08x\n",
		       desc->ioctl.chainvar[3],
		       desc->ioctl.chainvar[4],
		       desc->ioctl.chainvar[5]) ;

		printk("DESCR:    opt %08x  stat %08x   nxt %8p\n",
		       desc->ioctl.options, desc->ioctl.status, desc->next) ;

#ifdef DBG_DESC_VERY_VERBOSE
		invalidate_buffer(desc->ioctl.srcaddr, desc->ioctl.buflen) ;

		printk("DESCR: source data:\n") ;

		{
			int i ;
			char *DestBuf = (char *)desc->ioctl.srcaddr ;

			for(i = 0 ; i < desc->ioctl.buflen; i++)
			{
				switch(i % 16)
				{
				case 0:
					printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
					break;
				case 7:
					printk("%02x - ", DestBuf[i] & 0xff) ;
					break ;
				case 15:
					printk("%02x\n", DestBuf[i] & 0xff) ;
					break ;
				default:
					printk("%02x ", DestBuf[i] & 0xff) ;
					break ;
				}
			}
		}
#endif
	
		desc = desc->next ;
	}
}

#endif

#if defined(DBG_HSH_DESC) || defined(DBG_HSH_BUG) || defined (DEBUG_DESC_HIST)
static void
dump_descriptor_nda(sec_desc *desc)
{
	while (desc)
	{
		printk("DESCR at %x\n", (int)desc) ;
		printk("DESCR:    nda %08x   len %08x  mode %08x\n",
		       desc->ioctl.nda, desc->ioctl.buflen, desc->ioctl.mode);
		printk("DESCR:    src %8p   dst %8p    id %08x\n",
		       desc->ioctl.srcaddr, desc->ioctl.dstaddr, desc->ioctl.id);
		printk("DESCR:    ivl %08x   ivh %08x  type %08x\n",
		       desc->ioctl.ivhigh, desc->ioctl.ivlow, desc->ioctl.type);
		printk("DESCR:    ch0 %08x   ch1 %08x   ch2 %08x\n",
		       desc->ioctl.chainvar[0],
		       desc->ioctl.chainvar[1],
		       desc->ioctl.chainvar[2]) ;

		printk("DESCR:    ch3 %08x   ch4 %08x   ch5 %08x\n",
		       desc->ioctl.chainvar[3],
		       desc->ioctl.chainvar[4],
		       desc->ioctl.chainvar[5]) ;

		printk("DESCR:    opt %08x  stat %08x   nxt %8p\n",
		       desc->ioctl.options, desc->ioctl.status, desc->next) ;

#ifdef DBG_DESC_VERY_VERBOSE
		invalidate_buffer(desc->ioctl.srcaddr, desc->ioctl.buflen) ;

		printk("DESCR: source data:\n") ;

		{
			int i ;
			char *DestBuf = (char *)desc->ioctl.srcaddr ;

			for(i = 0 ; i < desc->ioctl.buflen; i++)
			{
				switch(i % 16)
				{
				case 0:
					printk("%04x  %02x ", i, DestBuf[i] & 0xff) ;
					break;
				case 7:
					printk("%02x - ", DestBuf[i] & 0xff) ;
					break ;
				case 15:
					printk("%02x\n", DestBuf[i] & 0xff) ;
					break ;
				default:
					printk("%02x ", DestBuf[i] & 0xff) ;
					break ;
				}
			}
		}
#endif
		desc = NXT_NDA(desc) ;
	}
}
#endif

#ifdef CONFIG_BRECIS_SEC_V1

/*************************************************************
** msp_sec_do_des_operation
**
** Given a chain of descriptors, run them through the encryption
** engine.
*************************************************************/
static void
msp_sec_do_des_operation(sec_desc *desc)
{
	sec_desc *d2 ;
	unsigned long flags ;
	static int *status_save ;
	int len ;
	int i ;

	DBG_SEC("Entering DES operation\n") ;

	/*
	** make the chain appear valid from the hardware point of view
	** and count up the number of bytes while we are here.
	*/

	d2 = desc ;
	len = desc->ioctl.buflen ;

	while (d2->next)
	{
		d2->ioctl.nda = (((int) d2->next) | (DNDA_VLD | DNDA_CNT)) ;

		d2 = d2->next ;

		len += d2->ioctl.buflen ;
	}
	d2->ioctl.nda = 0 ;

	/* if not 3des and decrypting, K1_DEC must also be set */
	if ( (desc->ioctl.mode & (DMC_MOD_DEC | DMC_3DES)) == DMC_MOD_DEC )
	{
		desc->ioctl.mode |= DMC_K1_DEC ;
	}

#if defined(DBG_DES_DESC)
	printk("SEC  running DES engine on this descriptor:\n") ;
	dump_descriptor(desc) ;
#endif

	/* XXX if (len & 0x7) error not multiple of 64 bits */

	/*
	** wait for engine to go quiet
	*/	

	DBG_SEC("Wait for sec engine idle\n") ;

	i = 0 ;
	while ( ( (*cur_interface_status & DES_STATUS_MASK) != DES_STATUS_VAL )
		|| ( (long)Des->next_desc_address & DNDA_VLD) )
	{
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "DES engine taking too long at entry status = (%08lX), NDA = %p\n", 
					*cur_interface_status, Des->next_desc_address) ;
			if (status_save)
				*status_save = 1 ;
			Des->interface_control = DIFC_DES_RST ;

			break ;
		}
	}

	RECORD_DES(desc) ;

	save_flags(flags) ;
	cli() ;
	
	SECUP(0) ;

	/*
	** mark where a failed status should be kept, should this
	** routine be re-entered by an interrupt and a failure be
	** detected there.
	*/
	desc->ioctl.status = 0 ;
	status_save = &desc->ioctl.status ;

	Des->next_desc_address = (void *)(((unsigned long) desc) | DNDA_VLD) ;

	SYNC() ;

	/* release RESET bit; set DMA start bit */
	Des->interface_control = DIFC_DMA_START ;
	cur_interface_status = &Des->interface_status ;

	restore_flags(flags) ;

	DBG_SEC("Engine started - wait for finish\n") ;

	/*
	** calculate how long it will take the security
	** engine to encrypt <len> bytes, and sleep almost
	** that long, to avoid slowing it down with polling
	** accesses to the interface status register.
	*/

	NanoDelay( DES_DLY(len) ) ;

	/* wait for engine to go quiet */
	i = 0 ;
	while ( ( (*cur_interface_status & DES_STATUS_MASK) != DES_STATUS_VAL)
		|| ( (long)Des->next_desc_address & DNDA_VLD) )
	{
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "DES engine taking too long: status = (%08lX), NDA = %p\n", 
					*cur_interface_status, Des->next_desc_address) ;
			desc->ioctl.status = 1 ;

			Des->interface_control = DIFC_DES_RST ;

			break ;
		}

	}

	SECDN(0) ;
	
	status_save = NULL ;

	/* At this point the operation is complete. */

}
#endif	/* CONFIG_BRECIS_SEC_V1 */

#ifdef CONFIG_BRECIS_SEC_V1
/*
** Padding for hash functions, to the multiple 64 byte size they are
** designed to work with.  For md5 it is defined as:
**    0x80 {zero fill} {orig_len in bits as 64 bit little endian int}
** For sha1 it is defined as
**    0x80 {zero fill} {orig_len in bits as 64 bit big endian int}
**
** Both add at least 9 bytes (the 0x80 and the 8-byte length), and must
** end on an 64 byte boundary.
*/

static inline void
hash_add_padding(sec_desc *desc,
		 sec_desc *tail_desc,
		 pad_struct *pad_area,
		 unsigned int orig_len)
{
	int pad_len ;
	union
	{
		unsigned long l ;
		unsigned char c[4] ;
	} u ; 

	if (desc->ioctl.mode & HMC_SHA1)
	{
		/* 64 bits of big endian int for SHA1 */
		pad_area->u.len[0] = 0 ;
		pad_area->u.len[1] = orig_len * 8;
	}
	else
	{
		/* 64 bits of little endian int for MD5 */
		u.l = orig_len * 8 ;
		pad_area->u.clen[0] = u.c[3] ;
		pad_area->u.clen[1] = u.c[2] ;
		pad_area->u.clen[2] = u.c[1] ;
		pad_area->u.clen[3] = u.c[0] ;
		pad_area->u.len[1] = 0 ;
	}

	pad_len = orig_len & 0x3F ;	/* modulo 64 */
	if (pad_len <= 55)
		pad_len = (64 - 8) - pad_len ;
	else
		pad_len = (128 - 8) - pad_len ;
	
	pad_area->pad_desc[0].ioctl.buflen  = pad_len ;
	pad_area->pad_desc[0].ioctl.srcaddr = hash_pad_buf ;
	pad_area->pad_desc[0].ioctl.dstaddr = desc->ioctl.dstaddr ;
	pad_area->pad_desc[0].ioctl.mode    = desc->ioctl.mode ;

	pad_area->pad_desc[0].next =&pad_area->pad_desc[1] ;
	pad_area->pad_desc[0].ioctl.nda = 
		( ((int)&pad_area->pad_desc[1]) | (DNDA_VLD | DNDA_CNT) ) ;

	pad_area->pad_desc[1].ioctl.buflen  = 8 ;
	pad_area->pad_desc[1].ioctl.srcaddr = (void *)&pad_area->u ;
	pad_area->pad_desc[1].ioctl.dstaddr = desc->ioctl.dstaddr ;
	pad_area->pad_desc[1].ioctl.mode    = desc->ioctl.mode ;

	pad_area->pad_desc[1].next      = 0 ;
	pad_area->pad_desc[1].ioctl.nda = 0 ;

	tail_desc->next = &pad_area->pad_desc[0] ;
	tail_desc->ioctl.nda = 
		( ((int)&pad_area->pad_desc[0]) | (DNDA_VLD | DNDA_CNT) ) ;

}
#endif 	/* CONFIG_BRECIS_SEC_V1 */

#ifdef CONFIG_BRECIS_SEC_V1

#ifndef HASH_MOD64_PROBLEM_FIXED 
/*------------------------------------------------------------
** The hash engine will give you fits if any of it's descriptors
** is not a multiple of 64 bytes.  To work around this, any time
** you hit a non-mod-64 descriptor, move ahead to the next 64 byte
** boundary in the chain, break the chain there, feed the first
** chain to the hash engine, wait for it to finish, use the results
** of that operation to continue on with the rest of the chain.
*/

static int
do_hash_workaround(sec_desc *desc)
{
	void *orig_dstaddr ;
	sec_desc *d1, *d2, *d3, *next_desc ;
	sec_desc pdesc ;	/* desc of partial buffer */
	int remain ;
	int contflag ;

	orig_dstaddr = desc->ioctl.dstaddr ;

#ifdef DBG_HSH_BUG
	printk("-----------WORKAROUND doing this descriptor\n") ;
	dump_descriptor_nda(desc) ;
#endif
	/*
	** The following code assumes that the list desc points to
	** has already been checked to be a multiple of 64 bytes in
	** length total.  This code only handles dma enging breaks
	** at non-64 byte intervals.
	*/

	d1 = desc ;
	contflag = 0 ;

	while(d1)
	{
		next_desc = NULL ;

		d2 = d1 ;

		while (d2)
		{

			remain = d2->ioctl.buflen & 0x3f ;

			if (remain != 0)
			{
				/* not a multiple of 64 in length */

				while (d2->next)
				{
					remain += d2->next->ioctl.buflen ;

					if (remain >= 0x40)
						break ;
					d2 = d2->next ;
				}
				/*
				** This is not supposed to happen
				** because of the mod64 enforcement
				** before this routine is called
				*/
				if (d2->next == NULL)
					return -1 ; /* failure */

				/* remain is >= 0x40 at this point */
				remain -= 0x40 ;
			
				if (remain != 0)
				{
					/*
					** Split this descriptor to
					** get to a 64 byte boundary
					*/

					int len ;

					len = d2->next->ioctl.buflen
						- remain ;

					pdesc.ioctl.nda = 0 ;
					pdesc.ioctl.buflen = len ;
					pdesc.ioctl.srcaddr =
						d2->next->ioctl.srcaddr ;
					pdesc.ioctl.dstaddr = (void *)
						d2->next->ioctl.chainvar ;
					d2->ioctl.nda =
						(((int)&pdesc)
						 | (DNDA_VLD|DNDA_CNT)) ;

					d2->next->ioctl.srcaddr += len ;
					d2->next->ioctl.buflen -= len ;
					
					next_desc = d2->next ;
					for(d3=d1 ; d3!=next_desc; d3=d3->next)
					{
						d3->ioctl.dstaddr = (void *)
							next_desc->ioctl.chainvar ;
					}
#ifdef DBG_HSH_BUG
					printk("---Results to be saved at %p\n",
					       next_desc->ioctl.chainvar) ;
#endif		

				}
				else if (d2->next->next)
				{
					next_desc = d2->next->next ;
					d2->next->ioctl.nda = 0 ;
					for(d3=d1 ; d3!=d2->next->next; d3=d3->next)
					{
						d3->ioctl.dstaddr = (void *)
							next_desc->ioctl.chainvar ;
					}
#ifdef DBG_HSH_BUG
					printk("---Results to be saved at %p\n",
					       next_desc->ioctl.chainvar) ;
#endif		
				}
				else
				{
#ifdef DBG_HSH_BUG
					printk("---Results straight to user\n") ;
#endif		
					next_desc = 0 ;
				}

					
				break ;
			}
			else
			{
				if (d2->next)
					d2->ioctl.nda =
						((int)d2->next)
						| (DNDA_VLD|DNDA_CNT) ;
				else
					d2->ioctl.nda = 0 ;
				d2 = d2->next ;
			}
		}

		d1->ioctl.mode = desc->ioctl.mode ;

		do_partial_hash(d1, contflag) ;

		d1 = next_desc ;

		contflag = HMC_CRV ; /* use chain vars next time through */

	}
	/*
	** restore original dest addr, so cache invalidate is
	** done to right address
	*/
	desc->ioctl.dstaddr = orig_dstaddr ;

	return 0 ;
}

static void
do_partial_hash(sec_desc *desc, int contflag)
{
	unsigned long flags ;
	int i ;
	int len ;
	sec_desc *d2 ;

	static int *status_save ;

#ifdef DEBUG_DESC_HIST

	static int nda_hist[32], nda_histp ;
	#define HISTRECORD(x) do { \
		int _y ; _y = (x) ; \
		if(nda_hist[nda_histp] != _y) { \
			nda_histp = (nda_histp + 1) & 31 ; \
			nda_hist[nda_histp] = _y ; \
		} \
	} while (0)

#else	
	#define HISTRECORD(x)

#endif

	desc->ioctl.mode |= contflag ;

#ifdef DBG_HSH_BUG
	printk("PARTIAL doing this:\n") ;
	dump_descriptor_nda(desc) ;
#endif
	RECORD_HSH(desc) ;
	
	/*
	** wait for engine to go quiet
	*/	

	DBG_SEC("Wait for sec engine idle\n") ;

	i = 0 ;
	while ( ( (*cur_interface_status & HSH_STATUS_MASK) != HSH_STATUS_VAL )
		|| ( (long)Hsh->next_desc_address & DNDA_VLD) )
	{
		HISTRECORD((long)Hsh->next_desc_address) ;
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "HASH engine taking too long at entry: status = (%08lX), NDA = %p\n", 
					*cur_interface_status, Hsh->next_desc_address) ;
#ifdef DEBUG_DESC_HIST
			printk(KERN_WARNING "status %x nda %x cda %x\n",
			       (int) *cur_interface_status,
			       (int) Hsh->next_desc_address,
			       *HSH_CDA_REG) ;
			printk("Previous NDA values:\n") ;
			for (i = 0; i < 32; i++)
				printk("%x ", nda_hist[(nda_histp+i+1)&31]);
			printk("\n") ;
			dump_desc_hist(6, &hash_desc_hist) ;
#endif
			if (status_save)
				*status_save = 1 ;

			Hsh->interface_control = HIFC_HSH_RST ;

			break ;
		}

	}
	HISTRECORD((long)Hsh->next_desc_address) ;

	Hsh->interface_control = HIFC_HSH_RST ;
	HISTRECORD(0xdeadbeef) ;

	desc->ioctl.status = 0 ;

	save_flags(flags) ;
	cli() ;

	SECUP(1) ;

	/*
	** mark where a failed status should be kept, should this
	** routine be re-entered by an interrupt and a failure be
	** detected there.
	*/
	desc->ioctl.status = 0 ;
	status_save = &desc->ioctl.status ;

	Hsh->next_desc_address = (void *)(((unsigned long) desc) | DNDA_VLD) ;

	SYNC() ;

	/* release RESET bit; set DMA start bit */
	Hsh->interface_control =  HIFC_DMA_START ;
	cur_interface_status = &Hsh->interface_status ;

	restore_flags(flags) ;

	DBG_SEC("Engine started - wait for finish\n") ;

	/*
	** calculate how long it will take the security
	** engine to hash <len> bytes, and sleep almost
	** that long, to avoid slowing it down with polling
	** accesses to the interface status register.
	*/

	d2 = desc ;
	len = desc->ioctl.buflen ;

	while (d2->ioctl.nda)
	{
		d2 = (sec_desc *)(d2->ioctl.nda & ~(DNDA_VLD | DNDA_CNT)) ;

		len += d2->ioctl.buflen ;
	}

	NanoDelay( HASH_DLY(len) ) ;

	/* wait for engine to go quiet */
	i = 0 ;
	while ( ( (*cur_interface_status & HSH_STATUS_MASK) != HSH_STATUS_VAL )
		|| ( (long)Hsh->next_desc_address & DNDA_VLD) )
	{
		HISTRECORD((long)Hsh->next_desc_address) ;
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "Hash engine taking too long: status = %08lX, NDA = %p\n",
					*cur_interface_status, Hsh->next_desc_address) ;
#ifdef DEBUG_DESC_HIST
			printk(KERN_WARNING "status %x nda %x cda %x\n",
			       (int) *cur_interface_status,
			       (int) Hsh->next_desc_address,
			       *HSH_CDA_REG) ;
			
	#ifdef KERNEL_INTERFACE_TEST
			printk(KERN_WARNING "sec_count 0: %x  1: %x\n",
			       sec_count[0], sec_count[1]) ;
	#endif
			dump_descriptor_nda(desc) ;
			printk("Previous NDA values:\n") ;
			for (i = 0; i < 32; i++)
				printk("%x ", nda_hist[(nda_histp+i+1)&31]);
			printk("\n") ;
			dump_desc_hist(6, &hash_desc_hist) ;
			show_des_regs() ;
			show_hsh_regs() ;
#endif
			desc->ioctl.status = 1 ;

			Hsh->interface_control = HIFC_HSH_RST ;

			break ;
		}
	}

	DBG_SEC("Engine finished\n") ;

	HISTRECORD((long)Hsh->next_desc_address) ;

	save_flags(flags) ;
	cli() ;

	SECDN(1) ;
	
	status_save = NULL ;

	Hsh->interface_control = HIFC_HSH_RST ;

	restore_flags(flags) ;


	/* At this point the operation is complete. */
}
#endif /* ! HASH_MOD64_PROBLEM_FIXED */
	
#endif /* CONFIG_BRECIS_SEC_V1 */

#ifdef CONFIG_BRECIS_SEC_V1

/*************************************************************
** msp_sec_do_hsh_operation
**
** Given a chain of descriptors, run them through the hash
** engine.
*************************************************************/
static void
msp_sec_do_hsh_operation(sec_desc *desc)
{
	sec_desc *d2 ;
	pad_struct pad_area ;
	sec_desc *pad_point ;
	int len ;

	pad_point = 0 ;

#ifdef HASH_MOD64_PROBLEM_FIXED 
	unsigned long flags ;
	int i ;
	static int *status_save ;
#endif

	DBG_SEC("Entering HASH operation\n") ;

	/*
	** make the chain appear valid from the hardware point of view
	** and count up the number of bytes while we are here.
	** Also make sure all the dest buffers point at the same
	** one.  I believe it is the last descriptor that is actually
	** used.
	*/

	d2 = desc ;
	len = desc->ioctl.buflen ;

	while (d2->next)
	{
		d2->ioctl.nda = ((int) d2->next) | (DNDA_VLD | DNDA_CNT) ;

		d2 = d2->next ;

		len += d2->ioctl.buflen ;
		d2->ioctl.dstaddr = desc->ioctl.dstaddr ;
	}
	d2->ioctl.nda = 0 ;

	if (desc->ioctl.options & MSP_SEC_DO_PADDING)
	{
		int extra_len = 0 ;

		pad_point = d2 ;

		if (desc->ioctl.mode & HMC_CRV)
			extra_len = desc->ioctl.chainvar[5] ;
		hash_add_padding(desc, pad_point, &pad_area,
				 len + extra_len) ;
	}
	else if (len & 0x3F)
	{
		DBG_SEC("Hash passed bad length\n") ;
		return ;
	}

#ifdef DBG_HSH_DESC
	printk("SEC  running HSH engine on this descriptor:\n") ;
	dump_descriptor_nda(desc) ;
#endif

#ifndef HASH_MOD64_PROBLEM_FIXED 
	do_hash_workaround(desc) ;
#else

	/*
	** wait for engine to go quiet
	*/	

	DBG_SEC("Wait for sec engine idle\n") ;

	i = 0 ;
	while ( ( (*cur_interface_status & HSH_STATUS_MASK) != HSH_STATUS_VAL )
		|| ( (long)Hsh->next_desc_address & DNDA_VLD) )
	{
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "Hash engine taking too long at entry status = %08lX, NDA = %p\n",
					*cur_interface_status, Hsh->next_desc_address) ;
			if (status_save)
				*status_save = 1 ;
			Hsh->interface_control = HIFC_HSH_RST ;

			break ;
		}
	}


	save_flags(flags) ;
	cli() ;
	
	/*
	** mark where a failed status should be kept, should this
	** routine be re-entered by an interrupt and a failure be
	** detected there.
	*/
	desc->ioctl.status = 0 ;
	status_save = &desc->ioctl.status ;

	Hsh->next_desc_address = (void *)(((unsigned long) desc) | DNDA_VLD) ;

	SYNC() ;

	/* release RESET bit; set DMA start bit */
	Hsh->interface_control =  HIFC_DMA_START ;
	cur_interface_status = &Hsh->interface_status ;

	restore_flags(flags) ;

	DBG_SEC("Engine started - wait for finish\n") ;

	/*
	** calculate how long it will take the security
	** engine to hash <len> bytes, and sleep almost
	** that long, to avoid slowing it down with polling
	** accesses to the interface status register.
	*/

	NanoDelay( HASH_DLY(len) ) ;

	/* wait for engine to go quiet */
	i = 0 ;
	while ( ( (*cur_interface_status & HSH_STATUS_MASK) != HSH_STATUS_VAL )
		|| ( (long)Hsh->next_desc_address & DNDA_VLD) )
	{
		NanoDelay(NDTIME) ;
		if (i++ > 20000)
		{
			printk(KERN_WARNING "Hash engine taking too long : status = %08lX, NDA = %p\n",
					*cur_interface_status, Hsh->next_desc_address) ;
			desc->ioctl.status = 1 ;

			Hsh->interface_control = HIFC_HSH_RST ;

			break ;
		}
	}

	status_save = NULL ;

	Hsh->interface_control = HIFC_HSH_RST ;

	/* At this point the operation is complete. */
#endif

	if(pad_point)
	{
		/* remove added padding descriptors from list */
		pad_point->next = 0 ;
	}
}

#endif /* CONFIG_BRECIS_SEC_V1 */
	

#ifdef CONFIG_BRECIS_SEC_V1

/*************************************************************
** msp_sec_do_hmac_operation
**
** Given a chain of descriptors, run them through the hash
** engine with the hmac algorithm.
**
** The key is given as the first descriptor,
** and text is in the following descriptors.
*************************************************************/
static void
msp_sec_do_hmac_operation(sec_desc *desc)
{
	sec_desc tmp_desc, part_desc ;
	int hashlen ;
	unsigned char hmac_key[HSH_BLK] ;
	unsigned char part_hash[HSH_LEN_MAX] ;
	int keylen ;
	int i ;

	if (desc->next == 0)
	{
		desc->ioctl.status = -EINVAL ;
		return ;
	}

	DBG_SEC("Entering HMAC operation\n") ;
	
#ifdef DBG_HSH_DESC
	printk("SEC  running HMAC operation on this descriptor:\n") ;
	dump_descriptor(desc) ;
#endif

	/*
	** For reference, the computation is:
	**    H(K XOR opad, H(K XOR ipad, text))
	** where H is the hash fn;
	**       K is key if it is short enough,
	**           else H(key) if key is > hash block length
	**       ipad is 0x36 repeated
	**       opad is 0x5c repeated
	*/

	hashlen = HASH_RSLT_LEN(desc->ioctl.mode) ;

	/*
	** key is pointed to by ioctl struct;
	** text is in continue buffers
	*/

	/* for hmac, if key is longer than blocksize, hash it down to size */

	keylen = desc->ioctl.buflen ;

	if (keylen <= HSH_BLK)
	{
		for (i = 0 ; i < keylen; i++)
			hmac_key[i] = desc->ioctl.srcaddr[i] ^ 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;
	}
	else
	{
		memcpy(&tmp_desc, desc, sizeof(sec_desc)) ;
		tmp_desc.ioctl.dstaddr   = hmac_key ;
		tmp_desc.ioctl.options   = MSP_SEC_DO_PADDING ;
		tmp_desc.ioctl.status    = 0 ;
		tmp_desc.next            = 0 ;
		
		msp_sec_do_hsh_operation(&tmp_desc) ;

		if (tmp_desc.ioctl.status)
		{
			desc->ioctl.status = tmp_desc.ioctl.status ;
			return ;
		}

		invalidate_buffer(hmac_key, hashlen) ;
		keylen = hashlen ;

		for (i = 0 ; i < keylen; i++)
			hmac_key[i] ^= 0x36 ;

		for (; i < HSH_BLK; i++)
			hmac_key[i] = 0x36 ;
	}

	/*
	** hmac_key now has (K XOR ipad).
	** Compute H(K XOR ipad, text).
	*/
	
	tmp_desc.ioctl.srcaddr   = hmac_key ;
	tmp_desc.ioctl.buflen    = HSH_BLK ;
	tmp_desc.ioctl.dstaddr   = part_hash ;
	tmp_desc.ioctl.options   = MSP_SEC_DO_PADDING ;
	tmp_desc.ioctl.status    = 0 ;
	tmp_desc.ioctl.mode      = desc->ioctl.mode ;
	tmp_desc.next            = desc->next ;

	msp_sec_do_hsh_operation(&tmp_desc) ;
	
	if (tmp_desc.ioctl.status)
	{
		desc->ioctl.status = tmp_desc.ioctl.status ;
		return ;
	}

	/*
	** Create (K XOR opad) from ( (K XOR ipad) XOR (ipad XOR opad) )
	*/
	for (i = 0 ; i < (HSH_BLK/4) ; i++)
		((unsigned long *)hmac_key)[i] ^= 0x6A6A6A6AUL ;

	/*
	** part_hash has H(K XOR ipad, text)
	** hmac_key  has (K XOR opad)
	** Compute H(hmac_key, part_hash)
	*/
	
	tmp_desc.ioctl.srcaddr   = hmac_key ;
	tmp_desc.ioctl.buflen    = HSH_BLK ;
	tmp_desc.ioctl.dstaddr   = desc->ioctl.dstaddr ;
	tmp_desc.ioctl.options   = MSP_SEC_DO_PADDING ;
	tmp_desc.ioctl.mode      = desc->ioctl.mode ;
	tmp_desc.ioctl.status    = 0 ;
	tmp_desc.next            = &part_desc ;

	part_desc.ioctl.srcaddr  = part_hash ;
	part_desc.ioctl.buflen   = hashlen ;
	part_desc.ioctl.dstaddr  = desc->ioctl.dstaddr ;
	part_desc.next           = 0 ;

	msp_sec_do_hsh_operation(&tmp_desc) ;

	desc->ioctl.status = tmp_desc.ioctl.status ;

	DBG_SEC("Exiting HMAC operation\n") ;
}

/*************************************************************
** msp_sec_do_hmac2_operation
**
** Given a chain of descriptors, run them through the hash
** engine with the hmac algorithm, using pre-computed ipad and
** opad keys.
**
** The pre-computed hash is given in the first descriptor,
** ipad as srcaddr, and opad as dstaddr.  Actual hash
** output goes into second descriptors destination address, 
** and text is in the following descriptors.
*************************************************************/
static void
msp_sec_do_hmac2_operation(sec_desc *desc)
{
	sec_desc tmp_desc;
	int hashlen ;
	unsigned char part_hash[HSH_LEN_MAX] ;

	if (desc->next == 0)
	{
		desc->ioctl.status = -EINVAL ;
		return ;
	}

	DBG_SEC("Entering HMAC2 operation\n") ;
	
#ifdef DBG_HSH_DESC
	printk("SEC  running HMAC2 operation on this descriptor:\n") ;
	dump_descriptor(desc) ;
#endif

	/*
	** For reference, the computation is:
	**    H(K XOR opad, H(K XOR ipad, text))
	** where H is the hash fn;
	**       K is key if it is short enough,
	**           else H(key) if key is > hash block length
	**       ipad is 0x36 repeated
	**       opad is 0x5c repeated
	**
	** We are given H(K XOR ipad) and H(K XOR opad)
	*/

	hashlen = HASH_RSLT_LEN(desc->ioctl.mode) ;

	/*
	** key is pointed to by ioctl struct;
	** text is in continue buffers
	*/

	/* for hmac, if key is longer than blocksize, hash it down to size */

	/*
	** Compute H(K XOR ipad, text).
	*/

	memcpy(tmp_desc.ioctl.chainvar, desc->ioctl.srcaddr, hashlen) ;
		
	tmp_desc.ioctl.srcaddr     = desc->next->ioctl.srcaddr ;
	tmp_desc.ioctl.buflen      = desc->next->ioctl.buflen ;
	tmp_desc.ioctl.dstaddr     = part_hash ;
	tmp_desc.ioctl.options     = MSP_SEC_DO_PADDING ;
	tmp_desc.ioctl.chainvar[5] = HSH_BLK ;
	tmp_desc.ioctl.status      = 0 ;
	tmp_desc.ioctl.mode        = desc->ioctl.mode | HMC_CRV ;
	tmp_desc.next              = desc->next->next ;

	msp_sec_do_hsh_operation(&tmp_desc) ;
	
	if (tmp_desc.ioctl.status)
	{
		desc->ioctl.status = tmp_desc.ioctl.status ;
		return ;
	}

	/*
	** part_hash has H(K XOR ipad, text)
	** desc->ioctl.dstaddr has partial hash H(K XOR opad)
	** Compute H(hmac_key, part_hash)
	*/
	
	memcpy(tmp_desc.ioctl.chainvar, desc->ioctl.dstaddr, hashlen) ;

	tmp_desc.ioctl.srcaddr     = part_hash ;
	tmp_desc.ioctl.buflen      = hashlen ;
	tmp_desc.ioctl.dstaddr     = desc->next->ioctl.dstaddr ;
	tmp_desc.ioctl.options     = MSP_SEC_DO_PADDING ;
	tmp_desc.ioctl.chainvar[5] = HSH_BLK ;
	tmp_desc.ioctl.mode        = desc->ioctl.mode | HMC_CRV ;
	tmp_desc.ioctl.status      = 0 ;
	tmp_desc.next              = 0 ;

	msp_sec_do_hsh_operation(&tmp_desc) ;

	desc->ioctl.status = tmp_desc.ioctl.status ;

	DBG_SEC("Exiting HMAC2 operation\n") ;
}

#endif /* CONFIG_BRECIS_SEC_V1 */
		

/*************************************************************
** msp_sec_open
**
** prepare to do operations with the security engine
*************************************************************/

static int msp_sec_open(struct inode * inode, struct file * file)
{
	DBG_SEC("driver open\n") ;
	return 0 ;
}

/*************************************************************
** msp_sec_close
**
** finish operations with the security engine
*************************************************************/

static int msp_sec_close(struct inode * inode, struct file * file)
{
	DBG_SEC("driver close\n") ;
	return 0 ;
}


/*************************************************************
** msp_sec_ioctl
**
** do something with the security engine
*************************************************************/

static int msp_sec_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	DBG_SEC("driver ioctl\n") ;

	if (!init_ok)
		return -ENXIO ;

#ifdef KERNEL_INTERFACE_TEST
	if (cmd == 42)
	{
		return kernel_intface_test(arg) ;
	}
#endif

	if (cmd == MSP_SEC_ENQ)
	{
		msp_sec_enquire((int *)arg) ;
		return 0 ;
	}

#if defined(CONFIG_BRECIS_SEC_V2)
	{
		int rc ;
		rc = msp_secv2_ioctl(inode, file, cmd, arg) ;

		/* -EINVAL if not hanlded by v2 ioctl */
		if (rc != -EINVAL)
			return rc ;
	}
#endif /* CONFIG_BRECIS_SEC_V2 */

	if (cmd != MSP_SEC_CTL)
	{
		DBG_SEC("bad ioctl\n") ;
		return -EINVAL ;
	}


#if defined(CONFIG_BRECIS_SEC_V1) || defined(CONFIG_BRECIS_SEC_V1ON2_COMPAT)
	sec_desc *d1, *d2 ;
	sec_cbuf *cbuf ;
	int cnt ;

	if (sec1_operations == 0)
	{
		DBG_SEC("ioctl not supported, or hw not present\n") ;
		return -EINVAL ;
	}
		
	/* some sort of support for the V1 API is present */

	d1 = desc_get_next() ;
	if (!d1)
	{
		DBG_SEC("no descriptors\n") ;
		return -ENOMEM ;
	}
	if ( (copy_from_user(d1, (void *)arg, sizeof(struct sec_ioctl)))
	     || !ACCESS_OK(VERIFY_READ, d1->ioctl.srcaddr, d1->ioctl.buflen) )
	{
		DBG_SEC("user fault\n") ;
		d1->next = NULL ; 
		desc_free_chain(d1) ;
		return -EFAULT ;
	}

	d1->next = NULL ; 

	d2 = d1 ;
	cbuf = d1->ioctl.cbuflist ;

	for (cnt = 0 ; cnt < d1->ioctl.cbufcount; cnt++)
	{
		d2->next = desc_get_next() ;
		if (!d2->next)
		{
			DBG_SEC("no descriptors while building chain %d\n",
				d1->ioctl.cbufcount) ;
			desc_free_chain(d1) ;
			return -ENOMEM ;
		}

		d2 = d2->next ;
		if ( (copy_from_user(d2, &cbuf[cnt], sizeof(*cbuf)))
		     || !ACCESS_OK(VERIFY_READ,
				   d2->ioctl.srcaddr,
				   d2->ioctl.buflen) )
		{
			DBG_SEC("user fault while building chain\n") ;
			d2->next = NULL ; 
			desc_free_chain(d1) ;
			return -EFAULT ;
		}

		d2->next = NULL ; 

	}

	/* write verifying is done differently for Hash vs Des */

	if (d1->ioctl.type == MSP_SEC_HMAC2)
	{
		/* HMAC2, with pre-calc'd keys, has destination
		** address in first continue buffer
		*/
		if ( ( d1->next == 0 )
		     || !ACCESS_OK(VERIFY_WRITE, d1->next->ioctl.dstaddr,
				HASH_RSLT_LEN(d1->next->ioctl.mode) ) )
		{
			DBG_SEC("user fault (write) in hash\n") ;
			desc_free_chain(d1) ;
			return -EFAULT ;
		}
	}
	else if (d1->ioctl.type != MSP_SEC_DES)
	{
		if ( !ACCESS_OK(VERIFY_WRITE, d1->ioctl.dstaddr,
				HASH_RSLT_LEN(d1->ioctl.mode) ) )
		{
			DBG_SEC("user fault (write) in hash\n") ;
			desc_free_chain(d1) ;
			return -EFAULT ;
		}
	}
	else
	{

		d2 = d1 ;
		while (d2)
		{
			if ( !ACCESS_OK(VERIFY_WRITE,
					d2->ioctl.dstaddr, d2->ioctl.buflen) )
			{
				DBG_SEC("user fault (write) in DES\n") ;
				desc_free_chain(d1) ;
				return -EFAULT ;
			}
			d2 = d2->next ;
		}
	}

	PRINT_FREELIST ;
	
	DBG_SEC("Calling security operation\n") ;

	d1->ioctl.status = 0 ;

	if ( (unsigned int)d1->ioctl.type <= MSP_SEC_HMAC2 )
	{
		sec1_operations[d1->ioctl.type](d1) ;
	}
	else
	{
		d1->ioctl.status = -EINVAL ;
		return -EINVAL ;
	}

	DBG_SEC("Security operation returned\n") ;

	/* invalidate cache where hardware wrote */
	
	if (d1->ioctl.type == MSP_SEC_HMAC2)
	{
		/* HMAC2, with pre-calc'd keys, has destination
		** address in first continue buffer
		*/
		invalidate_buffer(d1->next->ioctl.dstaddr,
				 HASH_RSLT_LEN(d1->ioctl.mode) ) ;
	}
	else if (d1->ioctl.type != MSP_SEC_DES)
	{
		invalidate_buffer(d1->ioctl.dstaddr,
				 HASH_RSLT_LEN(d1->ioctl.mode) ) ;
	}
	else
	{

		d2 = d1 ;
		while (d2)
		{
			invalidate_buffer(d2->ioctl.dstaddr, d2->ioctl.buflen) ;
			d2 = d2->next ;
		}
	}

	/* XXX get any failure info */
	copy_to_user((void *)(&( ((sec_ioctl *)arg)->status)),
		     &d1->ioctl.status,
		     sizeof(d1->ioctl.status)) ;
	
	DBG_SEC("returned %d from security operation\n", d1->ioctl.status) ;

	desc_free_chain(d1) ;
	
	PRINT_FREELIST ;

#endif  /* CONFIG_BRECIS_SEC_V1 is defined, or V1ONV2_COMPAT is defined */

	return 0 ;
}

/*************************************************************
** msp_sec_ioctl_kernel
**
** do something with the security engine from kernel context
**
** Arguments:
**
**    context   - msp_sec_context structure; holds variables
**                that would otherwise be globals to allow more
**                than one thread of context to enter the
**                security code simultaneously.
**
**    cmd       - Currently must be BRECIS_SEC_CTL, but is here
**                to allow for expansion of this API
**
**    arg       - A pointer to a list of sec_desc structures,
**                currently cast to void * for future api
**                expansion (could point to something different
**                when arg != BRECIS_SEC_CTL).  Each descriptor must
**                point to a piece of data <= 4088 bytes in length.
**
**    returns   - 0 for success, non-0 for failure.  return
**                value also reflected in first descriptor's
**                ioctl.status value.
**                
*************************************************************/

int msp_sec_ioctl_kernel(msp_sec_context *context,
			 unsigned int cmd,
			 void *arg)
{
	if (!init_ok)
		return -ENXIO ;

	DBG_SEC("kernel entry\n") ;

	if (cmd != MSP_SEC_CTL)
	{
		DBG_SEC("bad ioctl\n") ;
		return -EINVAL ;
	}

#if defined(CONFIG_BRECIS_SEC_V1) || defined(CONFIG_BRECIS_SEC_V1ON2_COMPAT)
	sec_desc *d1, *d2 ;

	if (sec1_operations == 0)
	{
		DBG_SEC("ioctl not supported, or hw not present\n") ;
		return -EINVAL ;
	}
		
	DBG_SEC("Calling security operation\n") ;

	d1 = (sec_desc *)arg ;

	d1->ioctl.status = 0 ;

	if ( (unsigned int)d1->ioctl.type <= MSP_SEC_HMAC2 )
	{
		sec1_operations[d1->ioctl.type](d1) ;
	}
	else
	{
		d1->ioctl.status = -EINVAL ;
		return -EINVAL ;
	}

	/* invalidate cache where hardware wrote */
	
	if (d1->ioctl.type == MSP_SEC_HMAC2)
	{
		/* HMAC2, with pre-calc'd keys, has destination
		** address in first continue buffer
		*/
		invalidate_buffer(d1->next->ioctl.dstaddr,
				 HASH_RSLT_LEN(d1->ioctl.mode) ) ;
	}
	else if (d1->ioctl.type != MSP_SEC_DES)
	{
		invalidate_buffer(d1->ioctl.dstaddr,
				 HASH_RSLT_LEN(d1->ioctl.mode) ) ;
	}
	else
	{

		d2 = d1 ;
		while (d2)
		{
			invalidate_buffer(d2->ioctl.dstaddr, d2->ioctl.buflen) ;
			d2 = d2->next ;
		}
	}

	
	DBG_SEC("returned from kernel security operation\n") ;

	return d1->ioctl.status ;
#else
	DBG_SEC("V1 API support not configured\n") ;
	return -EINVAL ;
#endif
	
}

/*
** initialize the random number hardare where it exists.
*/

void init_rng(void)
{
        static int i = 0;
        static const unsigned long icr_init[] =
                {       0x10101,
                        0x10102,
                        0x10103,
                        0x10201,
                        0x10202,
                        0x10203,
                        0x10301,
                        0x10302,
                        0x10303};


                i = i % ( sizeof(icr_init) / sizeof(icr_init[0]) );
                *(volatile long *) SEC_RNG_CNF = icr_init[i];
                i++;
}

/*
** get the next number from the rng.
*/

unsigned long get_rng(void)
{
        unsigned long random;
	int i = 0 ;

	if (info_hw == SECENQ_HW1)
	{
		/* can't access the rng while other security engine
		** pieces are operating
		*/
		while ( (*cur_interface_status & DES_STATUS_MASK)
			  != DES_STATUS_VAL )
		{
			NanoDelay(NDTIME) ;
			if (i++ > 20000)
				return 0 ;
		}
	}
		
        random = *(unsigned long *) SEC_RNG_VAL;
        init_rng();
        return random;
}



/*
** five times a second, add entropy from the rng hardware into
** linux's rng entropy pool.
*/
static void
do_rng_timer(unsigned long ignore)
{
	int flags ;
	static unsigned int last = 0 ;
	static int which = 0 ;
	unsigned int now ;

	extern void random_add_entropy(__u32 x, __u32 y) ;

	if (which == 0)
	{
		last = get_rng() ;
		which = 1 ;
	}
	else
	{
		now = get_rng() ;

		random_add_entropy(last, now) ;
		which = 0 ;
	}

	rng_timer.expires  = jiffies + 5 ;

	save_flags(flags) ;
	cli() ;
	add_timer(&rng_timer) ;
	restore_flags(flags) ;
}



/*************************************************************
** msp_sec_enquire
**
** return info about security hardware and driver on current
** system.
**
** Arguments:
**
**    info      - pointer to two ints, which are filled
**                in with information (see header file
**                msp_sec.h for values)
**
** Returns nothing.
**                
*************************************************************/

void msp_sec_enquire(int *info)
{
	info[0] = info_hw ;
	info[1] = info_driver ;
}


/*************************************************************
** various structures dealing with registering the driver with
** the kernel.
*************************************************************/

static struct file_operations msp_sec_fops = {
	owner:    THIS_MODULE,
	open:     msp_sec_open,
	release:  msp_sec_close,
	ioctl:    msp_sec_ioctl,
} ;

static struct miscdevice msp_sec_miscdev = {
	minor:        140,
	name:         "msp_sec",
	fops:         &msp_sec_fops,
} ;
	


/*************************************************************
** msp_sec_init
**
** Initialize the hardware, and install the driver.
**
*************************************************************/

static int __init msp_sec_init(void)
{
	int i ;
#ifdef CONFIG_BRECIS_SEC_V1
	int isduet = 0 ;
#endif
	char id ;

#if defined (CONFIG_BRECIS_SEC_V2) || defined (CONFIG_BRECIS_SEC_V2ON1_COMPAT)
	i = msp_secv2_init() ;
	if (i) return i ;
#endif

	id = identify_sec() ;
	switch (id)
	{
	case FEATURE_NOEXIST:
		printk(KERN_WARNING "Attempt to initialize security driver without propper hardware\n") ;
		init_ok = 0 ;
		return -ENXIO ;

	case SEC_POLO:
#ifdef CONFIG_BRECIS_SEC_V2
		info_hw = SECENQ_HW2 ;

#   ifdef CONFIG_BRECIS_SEC_V1ON2_COMPAT
		info_driver = SECENQ_API1 | SECENQ_API2 | SECENQ_API2_AES ;
		sec1_operations = v2_ops ;
		break ;
#   else
		info_driver = SECENQ_API2 | SECENQ_API2_AES ;
		sec1_operations = 0 ;
		break ;
#   endif
#else
		/*
		** in cases where ...SEC_V2 is not defined,
		** we can still support POLO hardware as if it was
		** DUET hardware, so no 'break' here, fall through...
		*/
#endif
	default:
		/* 
		** If it's not a polo, and yet it has a security
		** engine, it's DUET style.
		*/
#ifdef CONFIG_BRECIS_SEC_V1
		info_hw = SECENQ_HW1 ;
		info_driver = SECENQ_API1 ;
		sec1_operations = v1_ops ;
		isduet = 1 ;
		break ;
#else
		sec1_operations = 0 ;
#endif
	}

	if ( (id != SEC_TRIAD) && (id != FEATURE_NOEXIST) )
	{
		/*
		** Random number generator is available.
		**
		** use it to feed /dev/random.
		*/
		unsigned long flags ;
		
		rng_timer_used = 1 ;

		init_timer(&rng_timer) ;

		rng_timer.function = do_rng_timer ;
		rng_timer.expires = jiffies + HZ ;

		save_flags(flags) ;
		cli() ;
		add_timer(&rng_timer) ;
		restore_flags(flags) ;
	}

	#

#ifdef CONFIG_BRECIS_SEC_V1
	if (isduet)
	{
		/*
		** First make sure PIO and interrupt disabled,
		** then set the reset bits to begin DES and HASH reset
		*/
		Des->interface_control = DIFC_DES_RST ;
		Hsh->interface_control = HIFC_HSH_RST ;

		/*
		** Poll Interface Status until DES reset is complete
		*/
		i = 0 ;
		while ( Des->interface_status & (DIFS_DON | DIFS_BSY) )
		{
			mdelay(1) ;

			if (i++ >= 100)
			{
				printk(KERN_WARNING
				       "DES engine initialization failure\n") ;
				init_ok = 0 ;
				return -EBUSY ;    /* FAILURE */
			}
		}

		/*
		** Poll Interface Status until HSH reset is complete
		*/
		i = 0 ;
		while ( Hsh->interface_status & (HIFS_DON | HIFS_BSY) )
		{
			mdelay(1) ;

			if (i++ >= 100)
			{
				printk(KERN_WARNING
				       "Hash engine initialization failure\n") ;
				init_ok = 0 ;
				return -EBUSY ;    /* FAILURE */
			}
		}
	}
#endif

	if (init_descriptor_pool() != 0)
	{
		printk(KERN_WARNING "sec driver failed to init desc pool") ;
		init_ok = 0 ;
		return -EBUSY ;
	}

	misc_register(&msp_sec_miscdev) ;

	msp_sec_proc_register() ;

#ifdef KERNEL_INTERFACE_TEST
	kernel_intface_test(1) ;
#endif

	printk(KERN_INFO "Security engine driver installed\n") ;

	return 0 ;
}

static void __exit msp_sec_exit(void)
{
#if defined (CONFIG_BRECIS_SEC_V2) || defined (CONFIG_BRECIS_SEC_V2ON1_COMPAT)
	msp_secv2_exit() ;
#endif
	printk(KERN_INFO "Security engine driver removed\n") ;

	misc_deregister(&msp_sec_miscdev) ;

	destroy_descriptor_pool() ;

	msp_sec_proc_unregister() ;
}

static int
proc_calc_metrics(char *page, char **start, off_t off,
		  int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int msp_sec_read_proc(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{
	int len ;
	len = sprintf(page, "mknod /dev/secdev c %d %d\n",
		      MISC_MAJOR,
		      msp_sec_miscdev.minor) ;
#if defined (CONFIG_BRECIS_SEC_V2) || defined (CONFIG_BRECIS_SEC_V2ON1_COMPAT)
	len += msp_secv2_read_proc(page + len, start, off, count, eof, data) ;
#endif
	return proc_calc_metrics(page, start, off, count, eof, len) ;
}

static int msp_sec_proc_register(void)
{
	proc_entry = create_proc_read_entry("brecis_sec", 0, NULL,
					    msp_sec_read_proc, NULL) ;
	return ( proc_entry != 0 ) ;
}

static void msp_sec_proc_unregister(void)
{
	if (!proc_entry) return ;

	remove_proc_entry(proc_entry->name, proc_entry->parent) ;
	proc_entry = 0 ;
}

module_init(msp_sec_init) ;
module_exit(msp_sec_exit) ;

MODULE_LICENSE("GPL") ;
EXPORT_SYMBOL(msp_sec_ioctl_kernel);
EXPORT_SYMBOL(msp_sec_enquire) ;

#ifdef KERNEL_INTERFACE_TEST

/*************************************************************
**
** KERNEL INTERFACE TESTING
**
**   It's tricky to find a place to hook into the kernel such that
**   you're called freom within an interrupt routine.
*/

static struct timer_list tl ;
static unsigned char testbuf[512] ;

static int errcount ;
static int passcount ;
static int desnest, hshnest ;

static int testenabled ;

static void
do_des_test(void)
{
	sec_desc desc ;
	static msp_sec_context context ;
	int i, error ;

	error = 0 ;

	for(i = 0 ; i < 512; i++)
		testbuf[i] = i & 0xff ;

	desc.ioctl.type = MSP_SEC_DES ;
	desc.ioctl.options = 0 ;
	desc.ioctl.buflen = 512 ;
	desc.ioctl.srcaddr = testbuf ;
	desc.ioctl.dstaddr = testbuf ;

	desc.ioctl.mode = DMC_3DES | DMC_MOD_ENC | DMC_MOD_CBC ;
	desc.ioctl.mode |= DMC_K1_ENC | DMC_K2_DEC | DMC_K3_ENC ;
	
	desc.ioctl.ivhigh = 0x01020304 ;
	desc.ioctl.ivlow  = 0xFEFDFCFB ;

	desc.ioctl.desc_key1low  = 0xCCDDAABB ;
	desc.ioctl.desc_key1high = 0xEE11FF22 ;
	desc.ioctl.desc_key2low  = 0xBA98FEDC ;
	desc.ioctl.desc_key2high = 0x32107654 ;
	desc.ioctl.desc_key3low  = 0xCCDDAABB ;
	desc.ioctl.desc_key3high = 0xEE11FF22 ;

	desc.next = 0 ;

	msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

	for(i = 0 ; i < 512; i++)
	{
		int samecount = 0 ;
		if (testbuf[i] == (i & 0xff))
		{
			/* in case some bytes actually encode such that
			** plaintext == ciphertext
			*/
			if (++samecount >3)
			{
				error = 1 ;
				break ;
			}
		}
	}

	if (!error)
	{
		desc.ioctl.type = MSP_SEC_DES ;
		desc.ioctl.options = 0 ;
		desc.ioctl.buflen = 512 ;
		desc.ioctl.srcaddr = testbuf ;
		desc.ioctl.dstaddr = testbuf ;

		desc.ioctl.mode = DMC_3DES | DMC_MOD_DEC | DMC_MOD_CBC ;
		desc.ioctl.mode |= DMC_K1_DEC | DMC_K2_ENC | DMC_K3_DEC ;
	
		desc.ioctl.ivhigh = 0x01020304 ;
		desc.ioctl.ivlow  = 0xFEFDFCFB ;

		desc.ioctl.desc_key1low  = 0xCCDDAABB ;
		desc.ioctl.desc_key1high = 0xEE11FF22 ;
		desc.ioctl.desc_key2low  = 0xBA98FEDC ;
		desc.ioctl.desc_key2high = 0x32107654 ;
		desc.ioctl.desc_key3low  = 0xCCDDAABB ;
		desc.ioctl.desc_key3high = 0xEE11FF22 ;

		desc.next = 0 ;

		msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

		for(i = 0 ; i < 512; i++)
		{
			if (testbuf[i] != (i & 0xff))
			{
				error = 1 ;
				break ;
			}
		}
	}
	if (!error)
		passcount++ ;
	else
		errcount+=10000 ;
	if (sec_count[0])
		desnest++ ;

}

/*  SHA 1 Vectors out of FIPS - 180 */
static char sha1Vector1[] = "abc";
static unsigned char sha1Result1[] = {	0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A,
			0xBA, 0x3E, 0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C,
			0x9C, 0xD0, 0xD8, 0x9D };

static void
do_hsh_test(void)
{
	sec_desc desc ;
	static msp_sec_context context ;
	int i, error ;

	error = 0 ;

	for(i = 0 ; i < 20; i++) testbuf[i] = 0 ;

	desc.ioctl.type = MSP_SEC_HASH ;
	desc.ioctl.options = MSP_SEC_DO_PADDING ;
	desc.ioctl.buflen = 3 ;
	desc.ioctl.srcaddr = sha1Vector1 ;
	desc.ioctl.dstaddr = testbuf ;

	desc.ioctl.mode = HMC_SHA1 |HMC_CHAIN_DEFAULT ;
	
	desc.next = 0 ;

	msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

	for(i = 0 ; i < 20; i++)
	{
		if (testbuf[i] != sha1Result1[i])
		{

			printk("HASH mismatch %d: ", i) ;
			for (i = 0 ; i < 20 ; i++)
				printk("%2x ", testbuf[i]) ;
			printk("\n") ;
			printk(KERN_WARNING "sec_count 0: %x  1: %x\n",
			       sec_count[0], sec_count[1]) ;
#ifdef DEBUG_DESC_HIST
			printk("Here's what DES was up to:\n") ;
			dump_desc_hist(7, &des_desc_hist) ;
#endif
			error = 1 ;
			break ;

		}
	}

	if (!error)
		passcount++ ;
	else
	{
		errcount++ ;
	}
	if (sec_count[1])
		hshnest++ ;

}

static void
do_hw_timer(unsigned long ignore)
{
	static int i ;

	if (!testenabled)
		return ;
#ifndef HAVE_THIS_REMOVED
	do_hsh_test() ;
#else
	if (i == 0)
		do_des_test() ;
	else
		do_hsh_test() ;
#endif
	i ^= 1 ;
}
static void
do_my_timer(unsigned long ignore)
{
	int flags ;
	printk("KERNTEST: pass %d error %d nest %d %d\n", passcount, errcount,
	       desnest, hshnest) ;

	tl.expires  = jiffies + HZ ;

	save_flags(flags) ;
	cli() ;
	add_timer(&tl) ;
	restore_flags(flags) ;
}

void
msp_sec_timer(void)
{
	static int jif ;
	static int state ;

	jif ++ ;
	switch(state)
	{
	case 0:
		if (jif > (20*HZ))
			state = 1 ;
		break ;

	case 1:
		if (jif >HZ/50)
		{
			do_hw_timer(0) ;
			jif = 0 ;
		}
		break ;
	}
}

static int
kernel_intface_test(unsigned long arg)
{
	static int init ;
	static int timer_set ;
	unsigned long flags;

	if (init == 0)
	{
		init_timer(&tl) ;
		init = 1 ;
	}
	if (arg == 0)
	{
		if (timer_set)
		{
			timer_set = 0 ;
			del_timer(&tl) ;
		}
	} else {
		if (!timer_set)
		{
			timer_set = 1 ;

			tl.function = do_my_timer ;
			tl.expires  = jiffies + 20 * HZ ;

			save_flags(flags) ;
			cli() ;
			add_timer(&tl) ;
			restore_flags(flags) ;
		}
	}
	testenabled = arg ;
	return 0 ;
}

#endif

#ifdef DEBUG_DESC_HIST

/******************************************************************/
/*								  */
/* show_des_regs						  */
/*								  */
/* DESCRIPTION:							  */
/* This routine shows all of the Des register space		  */
/*								  */
/* PARAMETERS:	none						  */
/*								  */
/******************************************************************/
static void show_des_regs (void)
{
	printk("DES Registers:\n");
	printk("nextDescAddress:  %x\n", 
					(unsigned int)Des->next_desc_address);
	printk("buff_Size:          %lx  (%ld)\n", 
					Des->byte_count, 
					Des->byte_count);
	printk("source_Address:    %x\n", (unsigned int) Des->source_address);
	printk("dest_Address:      %x\n", (unsigned int) Des->dest_address);
	printk("Mode_Control:      %lx\n", Des->mode_control);
	printk("Descriptor_ID:     %lx\n", Des->descriptor_id);
	printk("Init_Vector_High:   %lx\n", Des->init_vector_high);
	printk("Init_Vector_Low:    %lx\n", Des->init_vector_low);
	printk("Key1High:         %lx\n", Des->key1_high);
	printk("Key1Low:          %lx\n", Des->key1_low);
	printk("Key2High:         %lx\n", Des->key2_high);
	printk("Key2Low:          %lx\n", Des->key2_low);
	printk("Key3High:         %lx\n", Des->key3_high);
	printk("Key3Low:          %lx\n", Des->key3_low);
	printk("interface_Control: %lx\n", Des->interface_control);
	printk("interface_Status:  %lx\n", Des->interface_status);
	printk("Input_Data_High:	  %lx\n", Des->input_data_high);
	printk("Input_Data_Low:	  %lx\n", Des->input_data_low);
	printk("Output_Data_High:	  %lx\n", Des->output_data_high);
	printk("Output_Data_Low:	  %lx\n", Des->output_data_low);
	printk("CDA_Register:	  %x\n", *DES_CDA_REG);
	printk("\n");
}

/******************************************************************/
/*								  */
/* show_hsh_regs						  */
/*								  */
/* DESCRIPTION:							  */
/* This routine shows all of the Hsh register space		  */
/*								  */
/* PARAMETERS:	none						  */
/*								  */
/******************************************************************/
static void show_hsh_regs (void)
{
	printk("HASH Registers:\n");
	printk("baseDesc.nextDescAddress:  %x\n", 
	       (unsigned int) Hsh->next_desc_address);
	printk("buff_Size:         %lx  (%lud)\n", 
	       Hsh->byte_count,
	       Hsh->byte_count);
	printk("Source_Address:	  %x\n", 
	       (unsigned int) Hsh->source_address);
	printk("dest_Address:      %x\n", 
	       (unsigned int) Hsh->dest_address);
	printk("Mode_Control:	  %lx\n", Hsh->mode_control);
	printk("Descriptor_ID:     %lx\n", Hsh->descriptor_id);
	printk("Error_Header1:     %lx\n", Hsh->error_header1);
	printk("Error_Header2:     %lx\n", Hsh->error_header2);
	printk("Chain_Var_A:        %lx\n", Hsh->chain_var_a);
	printk("Chain_Var_B:        %lx\n", Hsh->chain_var_b);
	printk("Chain_Var_C:        %lx\n", Hsh->chain_var_c);
	printk("Chain_Var_D:        %lx\n", Hsh->chain_var_d);
	printk("Chain_Var_E:        %lx\n", Hsh->chain_var_e);
	printk("Reserved1:        %lx\n", Hsh->reserved1);
	printk("interface_Control: %lx\n", Hsh->interface_control);
	printk("interface_Status:  %lx\n", Hsh->interface_status);
	printk("Input_Data:        %lx\n", Hsh->input_data);
	printk("CDA_Register:     %x\n", *HSH_CDA_REG);
	printk("\n");
}
#endif /* DEBUG_DESC_HIST */
