/* $Id: pci_ops.c,v 1.10 2003/04/10 22:43:34 mrustad Exp $ */
/*
 * Copyright 2001 MontaVista Software Inc.
 * Author: Jun Sun, jsun@mvista.com or jsun@junsun.net
 *
 * arch/mipsnommu/brecis/generic/pci_ops.c
 *     Define the pci_ops for BRECIS PCI.
 *
 * Much of the code is derived from the original DDB5074 port by 
 * Geert Uytterhoeven <geert@sonycom.com>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */

#define	PCI_COUNTERS	1

#include <linux/config.h>
#include <linux/types.h>
#include <linux/pci.h>
#if	defined(CONFIG_PROC_FS) && defined(PCI_COUNTERS)
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif	/* CONFIG_PROC_FS && PCI_COUNTERS */
#include <linux/kernel.h>
#include <linux/init.h>

#include <brecis/mspPCI.h>
#include <brecis/NanoDelay.h>
#include <asm/brecis/BrecisSysRegs.h>
#include <asm/byteorder.h>
#include <asm/pci_channel.h>


#define PCI_ACCESS_READ  0
#define PCI_ACCESS_WRITE 1

#undef	DEBUG
//#define	DEBUG	1

#ifdef DEBUG
#define DBG(x...) printk(x)
#define	IDBG(x...)	pkinit(x)
#else
#define DBG(x...) do {} while (0)
#define	IDBG(x...) do {} while (0)
#endif

#if	defined(CONFIG_PROC_FS) && defined(PCI_COUNTERS)
static char	proc_init;
extern struct proc_dir_entry	*proc_bus_pci_dir;
unsigned int	pci_int_count[32];

static int read_brecis_pci_counts(char *page, char **start, off_t off,
		int count, int *eof, void *data)
{
	int	i;
	int	len = 0;
	unsigned int	intcount, total = 0;

	for (i = 0; i < 32; ++i) {
		intcount = pci_int_count[i];
		if (intcount != 0) {
			len += sprintf(page + len, "[%d] = %u\n", i, intcount);
			total += intcount;
		}
	}
	len += sprintf(page + len, "total = %u\n", total);
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static void	pci_proc_init(void)
{
	create_proc_read_entry("brecispci", 0, NULL,
		read_brecis_pci_counts, NULL);
	return;
}
#endif	/* CONFIG_PROC_FS && PCI_COUNTERS */

spinlock_t	bpci_lock = SPIN_LOCK_UNLOCKED;

static struct resource pci_io_resource = {
	"pci IO space", 
	0x00000004,	//PCI_IOSPACE_BASE,
	0x00000FFF,	//PCI_IOSPACE_END,
	IORESOURCE_IO
};

static struct resource pci_mem_resource = {
	"pci memory space", 
	PCI_SPACE_BASE,
	PCI_SPACE_END,
	IORESOURCE_MEM
};


/*
 * bpci_interrupt - PCI status interrupt handler.
 */

static void	bpci_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	mspPciRegs	*preg = &msp_pci_regs;
	unsigned int	stat = preg->if_status;
#if	defined(CONFIG_PROC_FS) && defined(PCI_COUNTERS)
	int	i;
	for (i = 0; i < 32; ++i) {
		if ((1 << i) & stat)
			++pci_int_count[i];
	}
#endif	/* PROC_FS && PCI_COUNTERS */

	printk("PCI: Status=%08X\n", stat);
	preg->if_status = stat;
}


static int
msp_pcibios_config_access(unsigned char access_type, struct pci_dev *dev,
		unsigned char where, u32 *data)
{
	unsigned char	bus = dev->bus->number;
	unsigned char	dev_fn = dev->devfn;
	mspPciRegs	*preg = &msp_pci_regs;
	volatile unsigned long	*pcispace = &msp_pci_config_space;
	unsigned long	flags;
	unsigned long	intr;
	static	int	cardbus_detect = 0;
	static char	pciirqflag;

	if (cardbus_detect && PCI_SLOT(dev_fn) != 0) {
		*data = 0xFFFFFFFF;
		return -1;
	}

#if	defined(CONFIG_PROC_FS) && defined(PCI_COUNTERS)
	if (proc_init == 0) {
		pci_proc_init();
		proc_init = ~0;
	}
#endif	/* CONFIG_PROC_FS && PCI_COUNTERS */

	if (pciirqflag == 0) {
		request_irq(PCI_STAT_IRQ, &bpci_interrupt,
			0, "BRECIS PCI", preg);
		pciirqflag = ~0;
	}

	spin_lock_irqsave(&bpci_lock, flags);

	/* Clear cause register bits */
	preg->if_status = ~(BPCI_IFSTATUS_BC0F | BPCI_IFSTATUS_BC1F);

	/* Setup address */
	preg->config_addr = BPCI_CFGADDR_ENABLE |
		(bus	<< BPCI_CFGADDR_BUSNUM_SHF) |
		(dev_fn	<< BPCI_CFGADDR_FUNCTNUM_SHF) |
		(where & 0xFC);

	if (access_type == PCI_ACCESS_WRITE) {
		unsigned long	value = cpu_to_le32(*data);

		DBG("PCI: writing %08lX @ %02x:%02x.%d+%d", value,
			bus, PCI_SLOT(dev_fn), PCI_FUNC(dev_fn), where);
	        *pcispace = value;
	} else {
		unsigned long	value = le32_to_cpu(*pcispace);

		*data = value;
		DBG("PCI: data read=%08lX @ %02x:%02x.%d+%d",
			value, bus, PCI_SLOT(dev_fn), PCI_FUNC(dev_fn), where);
	}

	/* Check for master or target abort */
	intr = preg->if_status;
	DBG(", status=%08lx, ", intr);

	preg->config_addr = 0;	/* Clear config access */

	if (intr & ~(BPCI_IFSTATUS_BC0F | BPCI_IFSTATUS_BC1F))
	{
	        /* Error occurred */

	        /* Clear bits */
	        preg->if_status = ~(BPCI_IFSTATUS_BC0F | BPCI_IFSTATUS_BC1F);

		spin_unlock_irqrestore(&bpci_lock, flags);
	        DBG("fail\n");

		return -1;
	}

	spin_unlock_irqrestore(&bpci_lock, flags);

	if (cardbus_detect == 0 && PCI_SLOT(dev_fn) == 0) {
		cardbus_detect = 1;
		printk("PCI: cardbus detected\n");
	}

	DBG("success\n");

	return 0;
}


/*
 * We can't address 8 and 16 bit words directly. Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int
msp_pcibios_read_config_byte(struct pci_dev *dev, int where, u8 *val)
{
	u32	data = 0;

	if (msp_pcibios_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xFF;
		return -1;
	}

	*val = (data >> ((where & 3) << 3)) & 0xff;

	return PCIBIOS_SUCCESSFUL;
}


static int
msp_pcibios_read_config_word(struct pci_dev *dev, int where, u16 *val)
{
	u32	data = 0;

	if (where & 1) {
		*val = 0xFFFF;
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	if (msp_pcibios_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xFFFF;
		return -1;
	}

	*val = (data >> ((where & 3) << 3)) & 0xffff;

	return PCIBIOS_SUCCESSFUL;
}


static int
msp_pcibios_read_config_dword(struct pci_dev *dev, int where, u32 *val)
{
	u32	data = 0;

	if (where & 3) {
		*val = 0xFFFFFFFF;
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	if (msp_pcibios_config_access(PCI_ACCESS_READ, dev, where, &data)) {
		*val = 0xFFFFFFFF;
		return -1;
	}

	*val = data;

	return PCIBIOS_SUCCESSFUL;
}


static int
msp_pcibios_write_config_byte(struct pci_dev *dev, int where, u8 val)
{
	u32	data = 0;
       
	if (msp_pcibios_config_access(PCI_ACCESS_READ, dev, where, &data))
		return -1;

	data = (data & ~(0xff << ((where & 3) << 3))) |
	       (val << ((where & 3) << 3));

	if (msp_pcibios_config_access(PCI_ACCESS_WRITE, dev, where, &data))
		return -1;

	return PCIBIOS_SUCCESSFUL;
}


static int
msp_pcibios_write_config_word(struct pci_dev *dev, int where, u16 val)
{
        u32	data = 0;

	if (where & 1)
		return PCIBIOS_BAD_REGISTER_NUMBER;

        if (msp_pcibios_config_access(PCI_ACCESS_READ, dev, where, &data))
	       return -1;

	data = (data & ~(0xffff << ((where & 3) << 3))) | 
	       (val << ((where & 3) << 3));

	if (msp_pcibios_config_access(PCI_ACCESS_WRITE, dev, where, &data))
	       return -1;

	return PCIBIOS_SUCCESSFUL;
}


static int
msp_pcibios_write_config_dword(struct pci_dev *dev, int where, u32 val)
{
	if (where & 3)
		return PCIBIOS_BAD_REGISTER_NUMBER;

	if (msp_pcibios_config_access(PCI_ACCESS_WRITE, dev, where, &val))
	       return -1;

	return PCIBIOS_SUCCESSFUL;
}


struct pci_ops msp_pci_ops = {
	msp_pcibios_read_config_byte,
	msp_pcibios_read_config_word,
	msp_pcibios_read_config_dword,
	msp_pcibios_write_config_byte,
	msp_pcibios_write_config_word,
	msp_pcibios_write_config_dword
};

struct pci_channel mips_pci_channels[] = {
	{&msp_pci_ops, &pci_io_resource, &pci_mem_resource, 0, 0xFF},
	{(struct pci_ops *) NULL, (struct resource *) NULL,
	 (struct resource *) NULL, (int) NULL, (int) NULL}
};


/*
 * msp_pci_init - Initialize PCI hardware on msp.
 */

void __init	msp_pci_init(void)
{
	mspPciRegs	*preg = &msp_pci_regs;

	{
		slmRegs	*sreg = (slmRegs *) SREG_BASE;
		u32	id = sreg->dev_id;

		switch (id & 0xFF00) {
		case 0x2000:
			break;

#ifdef	CONFIG_BRECIS_FPGA
		case 0:
			pkinit("PCI: FPGA (Polo?), id = 0x%08X\n", id);
			break;
#endif	/* CONFIG_BRECIS_FPGA */

		default:
			pkinit("PCI: No PCI\n");
			goto no_pci;
		}

		if (!HasPCI(id)) {
			pkinit("PCI: PCI disabled\n");
			goto no_pci;
		}
		if (!IsPCIHost(id)) {
			pkinit("PCI: Not PCI Host\n");
			goto no_pci;
		}

		/* Check for single PC Card mode */
		if ((id & DEV_ID_SINGLE_PC) &&
			(sreg->single_pc_enable != (PCCARD_32|SINGLE_PCCARD))) {
			pkinit("PCI: No PCI - Single PC Card not Cardbus (%02x)\n",
				sreg->single_pc_enable);
			goto no_pci;
		}
	}

	/*
	preg->if_control = (1 << BPCI_IFCONTROL_CTO_SHF);
	*/

	*(unsigned long *)0xB7F40000 = 3;

	if (preg->reset_ctl & BPCI_RESETCTL_PE) {
#if 0
		int	count = 20;

		preg->reset_ctl = BPCI_RESETCTL_PR;	/* Set reset */
		NanoDelay(250000);	/* Wait 250 uS */
		preg->reset_ctl = 0;	/* Clear reset */

		while (--count >= 0) {
			if ((preg->reset_ctl & BPCI_RESETCTL_CT) != 0)
				break;
			NanoDelay(100000000);	/* Wait 100 ms */
		}

		if ((preg->reset_ctl & BPCI_RESETCTL_CT) == 0) {
			IDBG("PCI: PCI stuck in reset?, giving up\n");
			goto no_pci;
		}
#endif	/* 0 */
	} else {
		goto no_pci;
	}

	/* Clear cause register bits */
	preg->if_status = ~0;

	preg->config_addr = 0;	/* Clear config access */

	preg->oatran = 0xB8000000;

	preg->bar0 = 0xF0000000;	/* Move bar0 out of the way */
	preg->bar1 = 0x9;		/* Enable prefetch, memory */
	preg->bar2 = 0x9;		/* Enable prefetch, memory */
	preg->mask1 = 0x18000000;	/* 64 MB */
	preg->mask2 = 0x18000002;	/* Enable byte swizzling */

	/* Set gpio pins to pass through PCI INT_D to INT_A */
	*GPIO_CFG5_REG |= (1<<4) | (1<<8) |(1<<12) | (1<<16);

	preg->if_mask = ~0;	/* Enable all PCI status interrupts */

	return;

no_pci:
	/* Disable PCI channel */
	pkinit("PCI: no host PCI bus detected\n");
	mips_pci_channels[0] = mips_pci_channels[1];
	return;
}

