/* $Header: /proj/software/pub/CVSROOT/uClinux/brecis/vpm/vpm-tcp.c,v 1.4 2003/05/28 23:04:07 awarner Exp $ */
/****************************************************************************
 *
 * Program(s) to test brecis telephony driver
 *  
 *****************************************************************************/

/******************************************************************/
/* Copyright (c) 2003 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

/*
 * vpm-tcp [-p <pkt size>] -[t|r] <port> <device>
 *
 * Uses vanilla TCP, not RTP.
 */
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <linux/telephony.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>

char	buffer[1024] ;

int	pkt_size = 320 ;
int	daemonise = 0 ;

void
usage(void)
{
	fprintf(stderr,
		"Usage: vpm-tcp [-d] [-l <logfile>] [-p <pkt size>] -[t|r] port  <device>\n") ;
}

/*
 * strtosin() -	Convert a string of the form [<host>:]<port> to
 * 		a sockaddr_in structure. If the host component
 * 		is missing, default to localhost.
 * 		Hardwire address family to AF_INET.
 *
 * 		Return 0 if conversion is OK, !0 otherwise
 * 		(including syntax errors and possible hostname
 * 		lookup failure)
 */
int
strtosin(char *arg, struct sockaddr_in *s)
{
	const char *hostp ;
	const char *portp ;
	char *p ;
	int  tmp ;
	struct hostent *hp ;

	/*
	 * This looks backwards, but it isn't.
	 * Is there a "<host>:" component ?
	 */
	hostp = strtok(arg, ":") ;
	if ((portp = strtok(NULL, NULL)) == NULL) {
		/*
		 * Nope.
		 */
		s->sin_addr.s_addr = INADDR_ANY ;
		portp = arg ;
	} else {
		if ((hp = gethostbyname(hostp)) == NULL) {
			return(h_errno) ;
		}

		/*
		 * Note: we ignore multiple addresses, since this
		 * is a test fixture.
		 */
		s->sin_addr = *((struct in_addr *) *hp->h_addr_list) ;
	}

	s->sin_family = AF_INET ;

	tmp = (int)strtol(portp, &p, 10) ;

	/*
	 * Check that strtol was able to do his job.
	 */
	if ((*portp == '\0') || (p == portp) || (*p != '\0')) {
		return(-1) ;
	}

	s->sin_port = htonl(tmp) ;

	return(0) ;
}

/*
 * reader() -	open a connection to the specified port, open the
 * 		phone device start the rx codec, read data from the
 * 		phone device and write it to the network.
 *
 * 		Also write data to log_fd iff >= 0.
 */
void
reader(const char *phone_dev, struct sockaddr_in *s_in, int log_fd)
{
	int phone_fd, s, n ;

	/*
	 * Open the phone device.
	 */
	if ((phone_fd = open(phone_dev, O_RDONLY)) < 0) {
		perror("reader open") ;
		exit(10) ;
	}

	if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		perror("reader socket") ;
		close(phone_fd) ;
		exit(11) ;
	}

	if (connect(s, (struct sockaddr *)s_in, sizeof(*s_in)) != 0) {
		perror("reader connect") ;
		close(s) ;
		close(phone_fd) ;
		exit(12) ;
	}

	n = 1 ;
	if (setsockopt(s, SOL_TCP, TCP_NODELAY, &n, sizeof(n)) != 0) {
		perror("reader setsockopt") ;
		close(s) ;
		close(phone_fd) ;
		exit(16) ;
	}

	if (daemonise && (fork() != 0)) {
		exit(0) ;
	}

	ioctl(phone_fd, PHONE_REC_START, NULL) ;

	while ((n = read(phone_fd, buffer, pkt_size)) > 0) {
		if (write(s, buffer, n) != n) {
			perror("reader write") ;
			ioctl(phone_fd, PHONE_REC_STOP, NULL) ;
			close(s) ;
			close(phone_fd) ;
			if (log_fd >= 0) {
				close(log_fd) ;
			}
			exit(13) ;
		}
		if (log_fd >= 0) {
			write(log_fd, buffer, n) ;
		}
	}

	perror("reader read") ;
	close(s) ;
	close(phone_fd) ;
	if (log_fd >= 0) {
		close(log_fd) ;
	}
}

/*
 * writer() -	Open the phone device for writing, listen on the
 * 		specified port, accept an inbound connection, start
 * 		the tx codec, read data from the network and write
 * 		it to the device.
 *
 * 		Also write data to log_fd iff >= 0.
 */
void
writer(const char *phone_dev, struct sockaddr_in *s_in, int log_fd)
{
	int phone_fd, l, s, n ;

	/*
	 * Open the phone device.
	 */
	if ((phone_fd = open(phone_dev, O_WRONLY)) < 0) {
		perror("writer open") ;
		exit(20) ;
	}

	if ((l = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		perror("writer socket") ;
		close(phone_fd) ;
		exit(21) ;
	}

	n = 1 ;
	if (setsockopt(l, SOL_SOCKET, SO_REUSEADDR, &n, sizeof(n)) != 0) {
		perror("writer setsockopt") ;
		close(l) ;
		close(phone_fd) ;
		exit(26) ;
	}

	if (bind(l, (struct sockaddr *)s_in, sizeof(*s_in)) != 0) {
		perror("writer bind") ;
		close(l) ;
		close(phone_fd) ;
		exit(22) ;
	}

	if (listen(l, 1) != 0) {
		perror("writer listen") ;
		close(l) ;
		close(phone_fd) ;
		exit(23) ;
	}

	if (daemonise && (fork() != 0)) {
		exit(0) ;
	}

	if ((s = accept(l, NULL, 0)) < 0) {
		perror("writer accept") ;
		close(l) ;
		close(phone_fd) ;
		exit(24) ;
	}

	ioctl(phone_fd, PHONE_PLAY_START, NULL) ;

	while ((n = read(s, buffer, pkt_size)) > 0) {
		if (write(phone_fd, buffer, n) != n) {
			perror("writer write") ;
			close(l) ;
			close(s) ;
			close(phone_fd) ;
			if (log_fd >= 0) {
				close(log_fd) ;
			}
			exit(25) ;
		}
		if (log_fd >= 0) {
			write(log_fd, buffer, n) ;
		}
	}

	perror("writer read") ;
	ioctl(phone_fd, PHONE_PLAY_STOP, NULL) ;
	close(s) ;
	close(l) ;
	close(phone_fd) ;
	if (log_fd >= 0) {
		close(log_fd) ;
	}
}

int main(int argc,char *argv[])
{
	extern char *optarg ;
	extern int optind ;
	int c ;
	char *p ;
	char *phone_dev ;
	struct sockaddr_in s_in ;
	int tx = 0, rx = 0 ;
	int log_fd = -1 ;

	while ((c = getopt(argc, argv, "dl:p:t:r:")) >= 0) {
		switch(c) {
		    case 'd':
			daemonise++ ;
			break ;
		    case 'l':
			if ((log_fd = open(optarg, O_WRONLY|O_CREAT, 0644)) < 0) {
				perror("cannot open logfile") ;
				exit(5) ;
			}
			break ;
		    case 'p':
			pkt_size = (int)strtol(optarg, &p, 0) ;
			if ((*p != '\0') || (p == optarg)) {
				usage() ;
				exit(2) ;
			}
			break ;
		    case 't':
		    case 'r':
			if (strtosin(optarg, &s_in)) {
				usage() ;
				exit(2) ;
			}
			if (c == 't') {
				tx++ ;
			} else {
				rx++ ;
			}
			break ;
		    default:
			usage() ;
			exit(1) ;
		}
	}

	if ((phone_dev = argv[optind]) == NULL) {
		usage() ;
		exit(3) ;
	}

	if ((rx + tx) != 1) {
		usage() ;
		exit(4) ;
	}

	if (tx) {
		reader(phone_dev, &s_in, log_fd) ;
	} else {
		writer(phone_dev, &s_in, log_fd) ;
	}

	exit(0) ;
}
