/*****************************************************************************
* xpipemon.c	X25 Debugg Monitor.
*
* Authors:	Nenad Corbic <ncorbic@sangoma.com>	
*
* Copyright:	(c) 1999 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
*
* NOTE:  This program is still under construction.  Only critical features
*        are enabled.
* ----------------------------------------------------------------------------
* Jun 28, 1999	Nenad Corbic 	Initial version based on xpipemon for linux.
*****************************************************************************/
#include <linux/version.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <linux/if_ether.h>
#include <linux/wanpipe.h>

#if LINUX_VERSION_CODE >= 0x020100

#include <linux/if_packet.h>
#include <linux/sdla_x25.h>


#define TIMEOUT 1
#define MDATALEN 2024
#define TRUE 1
#define FALSE 0

#pragma pack(1)
typedef struct {
   unsigned char  signature[8];
   unsigned char  requestreply;
   unsigned char  id;
   unsigned char  reserved2[6];
   unsigned char  opp_flag;
   unsigned char  command;
   unsigned short buffer_length;
   unsigned char  return_code;
   unsigned char  reserved3;
   unsigned short lcn;
   unsigned char  bits;
   unsigned char  cause;
   unsigned char  diag;
   unsigned char  type;
   unsigned long  timestamp;
   unsigned char  data[MDATALEN];
} CBLOCK;
#pragma pack()

#pragma pack(1)
typedef struct {
	ip_pkt_t 	ip_pkt;
	udp_pkt_t	udp_pkt;
	CBLOCK		cb;
}udp_packet;
#pragma pack()

/******************************************************************************
 * Structure for tracing
 *
 */

/* Prototypes */
void ResetCB(CBLOCK *c1);
int MakeUdpConnection( void );
int MakeRawConnection( void );
int ObtainConfiguration( void );
unsigned char DoCommand( CBLOCK *cb );
void SaveCB( CBLOCK *dest, CBLOCK *src );
void init( int argv, char *argc[]);
void error( char return_code);
void read_hdlc_stat(void);
void flush_hdlc_stat(void);
void get_ip_addr(char *);
void banner (char *);

/* global for now */
int                sock;
int 		   raw_sock=0;
CBLOCK             cb, cbconfig;
char               codeversion[10];
unsigned int       frame_count;
unsigned char      station_config;
int                raw_data = FALSE;
unsigned char      par_port_A_byte, par_port_B_byte;
int                off_counter, green_counter, red_counter;
int                loop_counter, fail;
int		   ip_id_count=0;

/* defines for now */
char ipaddress[16];
char src_ip[16];
char i_name[16];
unsigned short udp_port = 9000;
unsigned short lcn_number = 0;
char cmd[5];


void ResetCB(CBLOCK *c1)
{
	memset((void *)&c1->opp_flag, 0, sizeof(CBLOCK)-16);
}; 

int MakeUdpConnection( void )
{
	struct sockaddr_in addr;

	printf("Starting UDP Connection\n");
	fflush(stdout);
	raw_sock=0;

   	sock = socket( PF_INET, SOCK_DGRAM, 0 );
   	if( sock < 0 ) {
      		printf("Error: Unable to open socket!\n");
		fflush(stdout);
      		return( FALSE );
   	} 
   	
	addr.sin_family = AF_INET;
   	addr.sin_addr.s_addr = inet_addr(ipaddress);
	if(addr.sin_addr.s_addr < 0){
		printf("Error: Invalid IP address!\n");
		fflush(stdout);
		return( FALSE );
	}
   	addr.sin_port = htons((u_short)udp_port);

	if( !connect( sock, (struct sockaddr *)&addr, sizeof(addr)) == 0 ) {
      		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
		fflush(stdout);
      		return( FALSE );
   	} 

   	if (!ObtainConfiguration()) {
      		printf("Error: Unable to obtain frame relay information.\n");
		printf("       Make sure the IP and UDP port are correct.\n");
      		close( sock );
		fflush(stdout);
      		return( FALSE );
   	}    
   	return( TRUE );
}; 

int MakeRawConnection( void ) 
{
	struct ifreq ifr;
	struct sockaddr_ll soin;

	printf("Starting RAW Connection\n");
	raw_sock=1;

   	sock = socket(PF_PACKET, SOCK_RAW, 0);
   	if (sock < 0){
      		printf("Error: Unable to open socket!\n");
		fflush(stdout);
      		return( FALSE );
   	} /* if */
   	
	soin.sll_family = AF_PACKET;
	soin.sll_protocol = htons(0x16);
	strcpy (ifr.ifr_name,i_name);
   	ioctl(sock,SIOCGIFINDEX,&ifr);
	soin.sll_ifindex = ifr.ifr_ifindex;


	if (bind(sock, (struct sockaddr *)&soin, sizeof(soin)) < 0) {
      		printf("Error: Cannot make connection!\nMake sure the IP address is correct\n");
      		return( FALSE );
   	} /* if */

   	if (!ObtainConfiguration()) {
      		printf("Error: Unable to obtain x25 information.\n");
                printf("Make sure the IP and UDP port are correct.\n");
      		close( sock );
		fflush(stdout);
      		return( FALSE );
   	} /* if */   

   	return( TRUE );

}; /* MakeConnection */

int ObtainConfiguration (void) 
{
   	unsigned char x;
   
   	x = 0;
   	cb.command = X25_READ_CONFIGURATION;
   	cb.buffer_length = 0;
   	cb.lcn = 0;
   	while (DoCommand(&cb) != 0 && ++x < 4) {
      		if (cb.return_code == 0xaa) {
	 		printf("Error: Command timeout occurred\n"); 
	 		return(FALSE);
      		}
      		if (cb.return_code == 0xCC ) return(FALSE);
   	}
   
	if (x >= 4) return(FALSE);
   	station_config = cb.data[0];
   
   	strcpy(codeversion, "?.??");
   
   	cb.command = X25_READ_CODE_VERSION;
  	cb.buffer_length = 0;
   	DoCommand(&cb);

   	if (cb.return_code == 0) {
      		cb.data[cb.buffer_length] = 0;
      		strcpy(codeversion, cb.data);
   	}
   	return(TRUE);
}; /* ObtainConfiguration */

unsigned char DoCommand( CBLOCK *cb ) 
{
   	static unsigned char id = 0;
   	fd_set ready;
   	struct timeval to;
   	int x,err=0;

	if (raw_sock){
		udp_packet udp_pkt;
		memset(&udp_pkt,0,sizeof(udp_pkt));

		udp_pkt.ip_pkt.ver_inet_hdr_length = 0x45;
		udp_pkt.ip_pkt.service_type = 0;
		udp_pkt.ip_pkt.total_length = sizeof(ip_pkt_t)+
					      sizeof(udp_pkt_t)+32+cb->buffer_length;
		udp_pkt.ip_pkt.identifier = 0;
      		udp_pkt.ip_pkt.flags_frag_offset = 0;
		udp_pkt.ip_pkt.ttl = 0x7F;
 		udp_pkt.ip_pkt.protocol = 0x11;
		udp_pkt.ip_pkt.hdr_checksum = 0x1232;
		udp_pkt.ip_pkt.ip_src_address = inet_addr("1.1.1.1");
		udp_pkt.ip_pkt.ip_dst_address = inet_addr("1.1.1.2"); 

		udp_pkt.udp_pkt.udp_src_port = htons((u_short)udp_port);
		udp_pkt.udp_pkt.udp_dst_port = htons((u_short)udp_port); 
		udp_pkt.udp_pkt.udp_length   = sizeof(udp_pkt_t)+32+cb->buffer_length;
		udp_pkt.udp_pkt.udp_checksum = 0x1234;
	
		memcpy(&udp_pkt.cb.signature, cb, 32); 
		if (cb->buffer_length){
			memcpy(&udp_pkt.cb.data, cb->data, cb->buffer_length); 
		}
		
		for( x = 0; x < 4; x += 1 ) {
			udp_pkt.cb.opp_flag = 0;
			udp_pkt.cb.requestreply = 0x01;
			udp_pkt.cb.id=id;
			udp_pkt.cb.return_code = 0xaa;
			err = send(sock, &udp_pkt, udp_pkt.ip_pkt.total_length,0);
			if (err < 0){
				perror("Send");
				continue;
			}
			FD_ZERO(&ready);
			FD_SET(sock,&ready);
			to.tv_sec = 10;
			to.tv_usec = 0;
			if(select(sock + 1,&ready, NULL, NULL, &to)) {
	 			err = recv(sock, &udp_pkt, 
				   (sizeof(ip_pkt_t)+sizeof(udp_pkt_t)+32+MDATALEN),0);

				if (err >= (32+udp_pkt.cb.buffer_length)){
					memcpy(cb,&udp_pkt.cb.signature,
				       		(32+udp_pkt.cb.buffer_length));
				}
				break;
			}else{
	 			printf("Error: No Data received from the RAW connected socket.\n"); 
      			}
		}

	}else{	

  	 	for( x = 0; x < 4; x += 1 ) {
			cb->opp_flag = 0;
			cb->requestreply = 0x01;
			cb->id = id;
			cb->return_code = 0xaa;	// 0xAA is our special return code indicating packet timeout
			send(sock, cb, 32+cb->buffer_length,0);
			FD_ZERO(&ready);
			FD_SET(sock,&ready);
			to.tv_sec = 10;
			to.tv_usec = 0;
			if(select(sock + 1,&ready, NULL, NULL, &to)) {
	 			recv(sock, cb, 32+MDATALEN,0);
				break;
     	 		} else {
		 		printf("Error: No Data received from the connected socket.\n"); 
      			}
		} 
   	} 

   	/* make sure the id is correct (returning results from a previous
   	   call may be disastrous if not recognized)
   	   also make sure it is a reply
	 */	
   	if (cb->id != id || cb->requestreply != 0x02) cb->return_code = 0xbb;
   	id++;
   	if (cb->return_code == 0xCC) {
      		printf("Error: Code is not running on the board!\n");
      		exit(1);
   	}
   	return(cb->return_code);
   
}; /* DoCommand */





void init( int argc, char *argv[]){

	int i, i_cnt=0, d_cnt=0, c_cnt=0, if_found=0;
	struct in_addr ip_str;
	cb.id = 0;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"-i")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid IP or Interface Name, Type xpipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(ipaddress,argv[i+1]);
			if (inet_aton(ipaddress,&ip_str) != 0 ){
			}else{
				strcpy(i_name,ipaddress);
				if_found=1;
			}
			i_cnt=1;
		}else if (!strcmp(argv[i],"-u")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid UDP PORT, Type xpipemon <cr> for help\n\n");
				exit(0);
			}

		 	if(isdigit(argv[i+1][0])){
				udp_port = atoi(argv[i+1]);
			}else{
				printf("ERROR: Invalid UDP Port, Type xpipemon <cr> for help\n\n");
				exit(0);
			}
		}else if (!strcmp(argv[i],"-c")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type xpipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(cmd,argv[i+1]);
			c_cnt=1;
		}else if (!strcmp(argv[i], "-d")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type xpipemon <cr> for help\n\n");
				exit(0);
			}
			if (isdigit(argv[i+1][0])){
				lcn_number = atoi(argv[i+1]);
			}else{
				printf("ERROR: DLCI must be an integer,Type xpipemon <cr> for help\n\n");
				exit(0);
			} 
			d_cnt=1;
		
		}
	}

	if (!i_cnt){
		printf("ERROR: No IP address or Interface Name, Type xpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!c_cnt){
		printf("ERROR: No Command, Type xpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!d_cnt){
		/* Default lcn Number to 0 */
		lcn_number = 0;
	}
	
	if (if_found){
		get_ip_addr(ipaddress);
	}

	strcpy( cb.signature, "XLINK8ND");
 
};

void get_ip_addr(char *if_name){

	int skfd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

    	if ((skfd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket");
		exit(1);
    	}

	strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));

	if (ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0){
		fprintf(stderr, "Error: Unknown interface: %s\n",if_name);
		exit(0);
	}

        if (ioctl(skfd, SIOCGIFDSTADDR , &ifr) < 0) {

		strcpy(ipaddress,"0.0.0.0");
	}else{
		sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_dstaddr;
		strcpy(ipaddress,inet_ntoa(sin->sin_addr));
	}

	if (ioctl(skfd, SIOCGIFADDR, &ifr) < 0) {
		
		strcpy(src_ip,"0.0.0.0");
	}else{
		sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_addr;
		strcpy(src_ip,inet_ntoa(sin->sin_addr));
	}	
 

	close (skfd);

	return;

}


void error( char return_code ) 
{
	switch( return_code ){
		case 0x04:
			printf("Error: An invalid DLCI was selected\n");
			break;
		case 0x10:
			printf("Error: A modem failure occurred - DCD and/or CTS were found to be unexpectedly low\n");
			break;
		case 0x11:
			printf("Error: The Channel moved from Operative to being Inoperative\n");
			break;
		case 0x12:
			printf("Error: The Channel moved from Inoperative to being Operative\n");
			break;
		case 0x13:
			printf("Error: The Access Node has reported a change in the status of a DLCI or a number of DLCIs\n");
			break;
		case 0x14:
			printf("Error: A Full Status Report included a DLCI or a number of DLCIis which were not included before\n"); 
			break;
		case 0x1F:
			printf("Error: The frame relay command is invalid\n");
			break;
		default:
			break;
   	}		
}; /* error */

/*===============================================
 * Link Status
 *
 *==============================================*/

void link_status( void ) 
{
	cb.command = X25_HDLC_LINK_STATUS;
	cb.buffer_length = 0;
	cb.lcn = 0;
	DoCommand(&cb);

        if(cb.return_code == 0 || cb.return_code == 1){

                banner("LINK STATUS");

                switch (cb.return_code){
		case 0:
              		printf("\t    Link is in a disconnected mode : Disconnected\n");
			break;
		case 1:
			printf("\t Asynchronously Balanced Mode (ABM): Connected\n");
			break;
		default:
			printf("\t Error: Command Failed, Please try again\n");
			return;
		}

        	printf("\tNumber of Inf. Frames queued for tx: %i\n", 
                     			cb.data[0]);
                printf("\tNumber of Inf. Frames queued for rx: %i\n",
                    			cb.data[1]);
		switch (cb.data[2]){
		case 1: 
			printf("\t                 Link Configured as: DTE\n");
			break;
		case 2:
			printf("\t                 Link Confiugred as: DCE\n");
			break;
		default:
  			printf("\t    Unknown Link Configuration: DTE or DCE ?\n");
		}
                
          	printf("\t            Supervisory frame count: %i\n",cb.data[4]); 
                 
	}else
             	error(cb.return_code);
      		
} /* link_status */


/*===============================================
 * Modem Status 
 *
 *==============================================*/

void modem_status( void ) 
{
   	cb.command = X25_READ_MODEM_STATUS;
   	cb.buffer_length = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {

		banner("MODEM STATUS");

                /* If bit 3 is set DCD is hight, otherwise low */
      		if (cb.data[0] & 0x08) 
	 		printf("DCD: HIGH\n");
      		else
	 		printf("DCD: LOW\n");
      	
                /* If bit 5 is set CTS is hight, otherwise low */
      		if( cb.data[0] & 0x20) 
	 		printf("CTS: HIGH\n");
      		else 
	 		printf("CTS: LOW\n");
      	} else {
      		error(cb.return_code);
   	} 

} 
 
/*===============================================
 * HDLC Statistics
 *
 *==============================================*/

void read_hdlc_stat( void)
{
	read_hdlc_stat_t *stat;
	ResetCB(&cb);        
  	cb.command = X25_HDLC_READ_STATS;
   	cb.buffer_length = 0;
   	DoCommand(&cb);

   	stat = (read_hdlc_stat_t *)&cb.data[0];

        if (!cb.return_code)
	{
		banner("X25:  HDLC STATISTICS");

		printf("       Inf. Frames received successfuly : %i\n",
					stat->inf_frames_rx_ok);
		printf("   Inf. Frames received out of sequence : %i\n",
					stat->inf_frames_rx_out_of_seq);
		printf("Inf. Frames received with no data filed : %i\n",
					stat->inf_frames_rx_no_data);
		printf("        Incomming Inf. Frames discarded : %i\n",
					stat->inf_frames_rx_dropped);
		printf("Incomming Inf. Frames with data to long : %i\n",
					stat->inf_frames_rx_data_too_long);
		printf(" Frames received with invalid HDLC addr : %i\n",
					stat->inf_frames_rx_invalid_addr);
		printf("   Inf. Frames tarnsmitted successfully : %i\n",
					stat->inf_frames_tx_ok);
		printf("              Inf. Frames retransmitted : %i\n",
					stat->inf_frames_tx_retransmit);
		printf("                  Number of T1 Timeouts : %i\n",
					stat->T1_timeouts);
		printf("         Number of SABM frames received : %i\n",
					stat->SABM_frames_rx);
		printf("         Number of DISC frames received : %i\n",
					stat->DISC_frames_rx);
		printf("           Number of DM frames received : %i\n",
					stat->DM_frames_rx);
		printf("         Number of FRMR frames received : %i\n",
					stat->FRMR_frames_rx);
		printf("      Number of SABM frames transmitted : %i\n",
					stat->SABM_frames_tx);
		printf("      Number of DISC frames transmitted : %i\n",
					stat->DISC_frames_tx);
		printf("        Number of DM frames transmitted : %i\n",
					stat->DM_frames_tx);
		printf("      Number of FRMR frames transmitted : %i\n",
					stat->FRMR_frames_tx);
	}else
		error(cb.return_code);

} /* read_hdlc_stat */

/*===============================================
 * Flush HDLC Statistics
 *
 *==============================================*/

void flush_hdlc_stat( void)
{
  	cb.command = X25_HDLC_FLUSH_STATS;
   	cb.buffer_length = 0;
   	DoCommand(&cb);

	if (!cb.return_code)
		printf("\n\tHDLC Statistics have been cleared \n");
	else
		error(cb.return_code);
	
} /* flush_hdlc_stat */


/*===============================================
 * Communication Error Statistics 
 *
 *==============================================*/

void comm_err() 
{
	read_comms_err_stats_t *stat; 
	ResetCB(&cb);
   	cb.command = X25_HDLC_READ_COMM_ERR;
   	cb.buffer_length = 0;
   	cb.lcn = 0;	// for supervisor display
   
	DoCommand(&cb);

	stat = (read_comms_err_stats_t *)&cb.data[0];
   
	if (cb.return_code == 0 && cb.buffer_length == 0x0A) {

		banner("X25: COMMUNICATION ERROR STATISTICS");

      		printf("               Number of receiver overrun errors: %d\n",
						stat->overrun_err_rx);
      		printf("                   Number of receiver CRC errors: %d\n",
						stat->CRC_err);
      		printf("                 Number of abort frames received: %d\n",
						stat->abort_frames_rx);
      		printf("Number of times receiver disabled (buffers full): %d\n",
						stat->frames_dropped_buf_full);
      		printf("              Number of abort frames transmitted: %d\n",
						stat->abort_frames_tx);
                printf("                    Number fo transmit underruns: %d\n",
						stat->transmit_underruns);
      		printf("   Number of transmit underrun interrupts missed: %d\n",
						stat->missed_tx_underruns_intr);
      		printf("        Number of times DCD dropped unexpectedly: %d\n",
						stat->DCD_drop);
      		printf("        Number of times CTS dropped unexpectedly: %d\n",
						stat->CTS_drop);
   	} else {
      		error(cb.return_code);
   	} 
}; /* comm_err(); */


/*===============================================
 * Flush Communication Error Statistics 
 *
 *==============================================*/


void flush_comm_err( void ) 
{
	cb.command = X25_HDLC_FLUSH_COMM_ERR;
	cb.buffer_length = 0;
	cb.lcn = 0;
	DoCommand(&cb);
	if (!cb.return_code) 
		printf("Comm Error Statistics have been Cleared\n");
	else
		error(cb.return_code);
}; /* flush_comm_err */


#ifdef NEX_XPIPE 

/* FIXME:  The following code is not used.  The xpipemon is half
           finished, and only supports critical commands */

void global_stats( void ) 
{
   	ResetCB(&cb);
   	cb.command = READ_DLC_STATISTICS;
   	cb.buffer_length = 0;
   	cb.dlci = 0;
   	DoCommand(&cb);
   
	if (cb.return_code == 0) {
      		if( station_config == 0 ) {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[12]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&cb.data[14]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&cb.data[16]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&cb.data[18]);
      		}
   	} else {
		error(cb.return_code);	  	
  	 } 
}; /* global_stats */



void flush_global_stats( void ) 
{
   	cb.command = FLUSH_DLC_STATISTICS;
   	cb.buffer_length = 1;
   	cb.dlci = 0;
	cb.data[0] = 0x01;
   	DoCommand(&cb);
   
	if (cb.return_code != 0) { 
		switch(cb.return_code){
			case 0x06:
				printf("Error: Global Statistics not flushed\n");
				break;
			default:
				error(cb.return_code);
				break;	  
		}
	}	
}; /* flush_global_stats */

#endif

void line_trace(int trace_mode) 
{
   	unsigned int j, count=0;
  	fd_set ready;
   	struct timeval to;
        unsigned char *trace_data;
        trace_data_t *trace_info;
  
    	ResetCB(&cb);
   	cb.command = XPIPE_DISABLE_TRACING;
   	cb.buffer_length = 2;
        cb.data[0] = 0x00;
        cb.data[1] = 0x00;
   	DoCommand(&cb);
 
        ResetCB(&cb);
   	cb.command = XPIPE_ENABLE_TRACING;
   	cb.buffer_length = 0x02;
	cb.data[0] |= TRACE_DEFAULT;

	if (trace_mode == TRACE_PROT){
		cb.data[0] |= TRACE_SUPERVISOR_FRMS | 
			      TRACE_ASYNC_FRMS;

	}else if (trace_mode == TRACE_DATA){
		cb.data[0] |= TRACE_DATA_FRMS;

	}else{
		cb.data[0] |= TRACE_ALL_HDLC_FRMS;	  
	}
        cb.data[1] = 0x00;
   	DoCommand(&cb);

   	if( cb.return_code == 0 ) { 
      		printf("Starting trace...(Press ENTER to exit)\n");
   	} else if (cb.return_code == 0x1F) {
      		printf("Line Tracing is possible only with S508 board.\n");
      		return;
   	} else if( cb.return_code == 0xCD ) {
      		printf("Cannot Enable Line Tracing from Underneath.\n");
      		return;
   	} else if( cb.return_code == 0x01 ) {
      		printf("Starting trace...(although it's already enabled!)\n");
      		printf("Press ENTER to exit.\n");
   	} else {
      		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", cb.return_code );
      		return;
   	}
  	fflush(stdout);
	 
	for(;;) {
      	    FD_ZERO(&ready);
     	    FD_SET(0,&ready);
     	    to.tv_sec = 1;
    	    to.tv_usec = 0;
     
	    if(select(1,&ready, NULL, NULL, &to)) {
	 	break;
     	    } /* if */
   
            ResetCB(&cb); 
	    cb.command = XPIPE_GET_TRACE_INFO;
      	    cb.buffer_length = 0;
      	    DoCommand(&cb);

            trace_info = (trace_data_t *)&cb.data[0];
            trace_data = (char *)&trace_info->data;
    
	    if (cb.return_code == 0 && cb.buffer_length) { 

		   /*  frame type */
//                   printf("\n\nLength     = %i",trace_info->length);  
//                   printf("\nTime Stamp = 0x%04X",trace_info->timestamp);
  		   if (trace_info->type == 0x00) {
       		   	printf("\n\nINCOMING\tLENGTH: %i\tTIME: %05i\n\t",
				trace_info->length,trace_info->timestamp);
    		   }else if (trace_info->type == 0x01) {
 		   	printf("\n\nOUTGOING\tLENGTH: %i\tTIME: %05i\n\t",
				trace_info->length,trace_info->timestamp);
    		   }else	
			printf("\n\nERROR\n\t");
                   

	    	   if (trace_info->length == 0) {
	       	   	printf("the frame data is not available" );
	    	   } else {

       			if (raw_data) {
			   /* show raw data */
                           count = 0;
	 		   for( j=0; j<trace_info->length; j++ ){ 
                                count ++;
                                if (count == 20){
                                	printf("\n\t");
					count = 0;
                                }
	     			printf("%02X ",(unsigned char)trace_data[j]);
				fflush(stdout);
                           }

	      	  	} else { 
			   /* show intrptd. data */
		 	}
	  	   }

      	   } 
//else if (cb.return_code != 0) 
//		error(cb.return_code);
      	
      	//   if (!(cb.data[0] & 0x02)) { 
	//	 sleep(TIMEOUT);
      	//   } 
  	}

	fflush(stdout);
        ResetCB(&cb);
   	cb.command = XPIPE_DISABLE_TRACING;
   	cb.buffer_length = 2;
        cb.data[0] = 0x00;
        cb.data[1] = 0x00;
   	DoCommand(&cb);
}; //line_trace


void x25_driver_stat_ifsend( void )
{
      if_send_stat_t *if_snd_stat;

      cb.command = XPIPE_DRIVER_STAT_IFSEND;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      if_snd_stat = (if_send_stat_t *) &cb.data[0];

      banner("X25:  DRIVER IF_SEND STATISTICS");       

      printf("                                    Total Number of If_Send entries:  %ld\n", 
                                     	if_snd_stat->if_send_entry);
      printf("                          Number of If_Send entries with SKB = NULL:  %ld\n", 
                                      	if_snd_stat->if_send_skb_null);
      printf("Number of If_Send entries with broadcast addressed packet discarded:  %ld\n", 
                    			if_snd_stat->if_send_broadcast);
      printf("Number of If_Send entries with multicast addressed packet discarded:  %ld\n", 
					if_snd_stat->if_send_multicast);
      printf("                Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n", 
					if_snd_stat->if_send_critical_ISR);
      printf("   Number of If_Send entries with Critical set and packet discarded:  %ld\n", 
					if_snd_stat->if_send_critical_non_ISR);
      printf("                     Number of If_Send entries with Device Busy set:  %ld\n", 
					if_snd_stat->if_send_tbusy);
      printf("                 Number of If_Send entries with Device Busy Timeout:  %ld\n", 
					if_snd_stat->if_send_tbusy_timeout);
      printf("               Number of If_Send entries with XPIPE MONITOR Request:  %ld\n", 
					if_snd_stat->if_send_PIPE_request);
      printf("                    Number of If_Send entries with WAN Disconnected:  %ld\n", 
					if_snd_stat->if_send_wan_disconnected);
      printf("                         Number of If_Send entries with Send failed:  %ld\n", 
					if_snd_stat->if_send_bfr_not_passed_to_adptr);
      printf("        If_send, number of packets passed to the board successfully:  %ld\n", 
					if_snd_stat->if_send_bfr_passed_to_adptr);
      
 
} /* x25_driver_stat_ifsend */

void x25_driver_stat_intr( void )
{
      global_stats_t *global_stat;
      rx_intr_stat_t *rx_int_stat;
      cb.command = XPIPE_DRIVER_STAT_INTR;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      global_stat = (global_stats_t *)&cb.data[0];
      rx_int_stat = (rx_intr_stat_t *)&cb.data[sizeof(global_stats_t)];
       
      banner("X25: DRIVER INTERRUPT STATISTICS");       

      printf("                                   Number of ISR entries:    %ld\n" , 
                              		global_stat->isr_entry);
      printf("                 Number of ISR entries with Critical Set:    %ld\n" , 
					global_stat->isr_already_critical);
      printf("                             Number of Receive Interrupt:    %ld\n" , 
					global_stat->isr_rx);
      printf("                            Number of Transmit Interrupt:    %ld\n" , 
					global_stat->isr_tx);
      printf("             Number of ISR entries for Interrupt Testing:    %ld\n" , 
					global_stat->isr_intr_test);
      printf("                            Number of Spurious Interrupt:    %ld\n" , 
					global_stat->isr_spurious);
      printf("      Number of Times Transmit Interrupts Enabled in ISR:    %ld\n" , 
					global_stat->isr_enable_tx_int);
      printf("        Number of Receive Interrupts with Corrupt Buffer:    %ld\n\n" , 
					global_stat->rx_intr_corrupt_rx_bfr);

      printf("                                  Number of Poll Entries:    %ld\n",
                 			global_stat->poll_entry);
      printf("                Number of Poll Entries with Critical set:    %ld\n", 
					global_stat->poll_already_critical);
      printf("                        Number of Poll Entries Processed:    %ld\n\n", 
					global_stat->poll_processed);


      printf("             Number of Receive Interrupts with No socket:    %ld\n" , 
					rx_int_stat->rx_intr_no_socket);
      printf("  Number of Receive Interrupts for XPIPE MONITOR Request:    %ld\n" , 
					rx_int_stat->rx_intr_PIPE_request);
      printf("Number of Receive Interrupts with Buffer Passed to Stack:    %ld\n" , 
					rx_int_stat->rx_intr_bfr_passed_to_stack);
     

} /* x25_driver_stat_intr */


void x25_driver_stat_gen( void )
{
     
      pipe_mgmt_stat_t *pipe_mgmt_stat;

      cb.command = XPIPE_DRIVER_STAT_GEN;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      pipe_mgmt_stat = (pipe_mgmt_stat_t *)&cb.data[0]; 

      banner("X25:  DRIVER GENERAL STATISTICS");       

      printf("            Number of XPIPE Monitor call with kmalloc error:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_kmalloc_err);
      printf("       Number of XPIPE Monitor call with Adapter Type error:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_adptr_type_err);
      printf("          Number of XPIPE Monitor call with Direction Error:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_direction_err);
      printf("  Number of XPIPE Monitor call with Adapter Command Timeout:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_adptr_cmnd_timeout);
      printf("       Number of XPIPE Monitor call with Adapter Command OK:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_adptr_cmnd_OK);
      printf("      Number of XPIPE Monitor call with Adapter Send passed:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_adptr_send_passed);
      printf("      Number of XPIPE Monitor call with Adapter Send failed:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_adptr_send_failed);
      printf("   Number of XPIPE Monitor call with packet passed to stack:  %ld\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_passed_to_stack);
      printf("  Number of XPIPE Monitor call with pkt NOT passed to stack:  %ld\n",
 					pipe_mgmt_stat->UDP_PIPE_mgmt_not_passed_to_stack);
      printf("                Number of XPIPE Monitor call with no socket:  %ld\n\n", 
					pipe_mgmt_stat->UDP_PIPE_mgmt_no_socket);

} /* x25_driver_stat_gen */


void flush_driver_stats( void )
{
      cb.command = XPIPE_FLUSH_DRIVER_STATS;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);

      printf("All Driver Statistics are Flushed.\n");

} /* flush_driver_stats */

void x25_router_up_time( void )
{
     unsigned long time;
     cb.command = XPIPE_ROUTER_UP_TIME;
     cb.buffer_length = 0;
     cb.data[0] = 0;
     DoCommand(&cb);
    
     banner("X25: ROUTER/API UP TIME");
 
     time = cb.data[0] + (cb.data[1]*256) + (cb.data[2]*65536) + 
					(cb.data[3]*16777216);
     if (time < 3600) {
	if (time<60) 
     		printf("    Router UP Time:  %ld seconds\n", time);
	else
     		printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     }else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
} /* x25_router_up_time */

void read_x25_statistics (void)
{
	TX25Stats *stats;

	ResetCB(&cb);
   	cb.command = X25_READ_STATISTICS;
   	cb.buffer_length = 0;
	DoCommand(&cb);	
	
	banner("X25: X25 STATISTICS");

	printf("Command %x\n",cb.command);

	if (cb.return_code != 0){
		printf("\t Error X25 Statistics Command Failed. Please Try Again\n");
		return;
	}

	stats = (TX25Stats *)&cb.data[0];
	
	printf ("             Restart Request Tx : %i\n", stats->txRestartRqst);
	printf ("             Restart Request Rx : %i\n", stats->rxRestartRqst);
	printf ("       Restart Confirmations Tx : %i\n", stats->txRestartConf);
	printf ("       Restart Confirmations Rx : %i\n", stats->rxRestartConf);
	printf ("               Reset Request Tx : %i\n", stats->txResetRqst);
	printf ("               Reset Request Rx : %i\n", stats->rxResetRqst);
	printf ("         Reset Confirmations Tx : %i\n", stats->txResetConf);
	printf ("         Reset Confirmations Rx : %i\n", stats->rxResetConf);
	printf ("                Call Request Tx : %i\n",stats->txCallRequest);	
	printf ("                Call Request Rx : %i\n",stats->rxCallRequest);
	printf ("               Call Accepted Tx : %i\n",stats->txCallAccept);
	printf ("               Call Accepted Rx : %i\n",stats->rxCallAccept); 
	printf ("               Clear Request Tx : %i\n",stats->txClearRqst);
	printf ("               Clear Request Rx : %i\n",stats->rxClearRqst);
	printf ("          Clear Confirmation Tx : %i\n",stats->txClearConf);
	printf ("          Clear Confirmation Rx : %i\n",stats->rxClearConf);
	printf ("         Diagnostics Packets Tx : %i\n",stats->txDiagnostic);
	printf ("         Diagnostics Packets Rx : %i\n",stats->rxDiagnostic);
	printf ("        Registration Request Tx : %i\n",stats->txRegRqst);
	printf ("        Registration Request Rx : %i\n",stats->rxRegRqst);
	printf ("   Registration Confirmation Tx : %i\n",stats->txRegConf);
	printf ("   Registration Confirmation Rx : %i\n",stats->rxRegConf);
	printf ("           Interrupt Packets Tx : %i\n",stats->txInterrupt);
	printf ("           Interrupt Packets Rx : %i\n",stats->rxInterrupt);
	printf ("     Interrupt Confirmations Tx : %i\n",stats->txIntrConf);
	printf ("     Interrupt Confirmations Rx : %i\n",stats->rxIntrConf);
	printf ("                Data Packets Tx : %i\n",stats->txData);
	printf ("                Data Packets Rx : %i\n",stats->rxData);
	printf ("                  RR Packets Tx : %i\n",stats->txRR);
	printf ("                  RR Packets Rx : %i\n",stats->rxRR);
	printf ("                 RNR Packets Tx : %i\n",stats->txRNR);
	printf ("                 RNR Packets Rx : %i\n",stats->rxRNR);
}


//----------------------- FT1 STATISTICS ------------------------------


void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
     	cb.command = XPIPE_FT1_STATUS_CTRL;
      	cb.buffer_length = 1;
      	cb.data[0] = status; 	
      	DoCommand(&cb);
      
	if( cb.return_code != 0 && status){
		fail = 1;
		if( cb.return_code == 0xCD )
                	printf("Error:  Cannot run this command from Underneath.\n");
		else 
			printf("Error:  This command is only possible with S508/FT1 board!");
      	}

} /* set_FT1_monitor_status */

void set_FT1_mode( void ){
 
	for(;;){
		cb.command = XPIPE_SET_FT1_MODE;
		cb.buffer_length = 0;
		DoCommand(&cb);
		if(cb.return_code == 0)
			break;
		else if( cb.return_code == 0xCD ){
                	printf("Error: Cannot run this command from Underneath.\n");
			exit(1);
		}

	}
} /* set_FT1_mode */

void read_FT1_status( void ){
     int i;
     long delay;
     struct timeval tv;

    
     i = gettimeofday(&tv,NULL);
     delay = tv.tv_usec;

     for(;;){
      i = gettimeofday(&tv,NULL);
      if((abs(tv.tv_usec - delay)) > 90000)
	break;  
     }

     cb.command = XPIPE_FT1_READ_STATUS;
     cb.buffer_length = 0;
     DoCommand(&cb); 
     if( cb.return_code == 0 ){
	par_port_A_byte = cb.data[0];
	par_port_B_byte = cb.data[1];
     }
} /* read_FT1_status */

void view_FT1_status( void ){
     
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     /* check for INS light */
     for(;;){
     	read_FT1_status();
     	if((par_port_B_byte & 0x30) == 0x30)
 		off_counter++;
     	if((par_port_B_byte & 0x10) == 0x00)
		red_counter++;
     	if((par_port_B_byte & 0x20) == 0x00)
		green_counter++;
	if(red_counter != 0 && off_counter != 0 && loop_counter == 30 ) {
		printf("Unit is not IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter == 0 && loop_counter == 30){
		printf("Unit is IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter != 0 && loop_counter == 30){
		printf("INS is flashing green\n");
		break;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("INS is off\n");
		break;
	}
	loop_counter++;
     }	
     /* check for ERR light */
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x0C) == 0x0C)
		off_counter++;
	if((par_port_B_byte & 0x08) == 0x00)
		red_counter++;
	if((par_port_B_byte & 0x04) == 0x00)
		green_counter++;
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("No Line Errors: Valid Line\n");
		break;
	}
	if(red_counter != 0 && off_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("Line Errors: Invalid Line\n");
		break;
	}

	loop_counter++;
     }

     /* check TXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x02) == 0x02)
		off_counter++;
	if((par_port_B_byte & 0x02) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Transmit data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Transmit data is present \n");
		break;
	}
	loop_counter++;
     }

     /* check RXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x01) == 0x01)
		off_counter++;
	if((par_port_B_byte & 0x01) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Receive data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Receive data is present\n");
		break;
	}
	loop_counter++;
     }

}/* view_FT1_status */


void FT1_operational_mode(void){
     
     printf("Operational Mode has been selected\n"); 
     printf("Putting S508/FT1 in operational mode....");
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
     	read_FT1_status();
     	/* ST light is OFF */
     	if((par_port_B_byte & 0xc0) == 0xc0 ){
		off_counter++;
     	}
     	/* ST light is GREEN */
     	if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
		red_counter = 0;
     	}
     	/* ST light is RED */
     	if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
		green_counter = 0;
   	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		break;
	}
	if((red_counter != 0 || green_counter != 0) && loop_counter == 20){
		set_FT1_mode();
		break;
	}

   	loop_counter++;
     } /* end of for */
       
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* DL light is OFF */
   	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;
	}
        /* DL light is RED */
   	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++; 
     } 
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* LL light is off */
   	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
   	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++;
     }
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
	/* RT light is OFF */
   	if((par_port_A_byte & 0x03) == 0x03){
		off_counter++;
	}
	/* RT light is RED */
   	if((par_port_A_byte & 0x01) == 0x00){
		red_counter++;
	}
	/* RT light is GREEN */
   	if((par_port_A_byte & 0x02) == 0x00){
   		green_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		printf("Done!\n");
		break;
	}
	if((red_counter != 0 || green_counter != 0) && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		printf("Done!\n");
		break;
	}
	if(off_counter == 0 && green_counter != 0 && loop_counter == 20){
		printf("Failed!\n");
		printf("Remote End is running Remote Test\n");
		printf("Exit Remote Test at remote end\n");
		break;
	}
	loop_counter++;
     }        	

} /* FT1_operational_mode */


void FT1_self_test(void){
     int started = 0; 
     int selftest = 0;
 
     set_FT1_mode();
     off_counter = 0;
     green_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Self Test has been selected\n");
     for(;;){
	  read_FT1_status();
	  printf("B Byte %x, OFF %i, GRN %i, RED %i, Loop Cnt %i\n",
	  par_port_B_byte,off_counter,green_counter,red_counter,loop_counter); 


	  /* ST light is OFF */
	  if((par_port_B_byte & 0xc0) == 0xc0){
		off_counter++;
	  }
	  /* ST light is GREEN */
	  if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
	  }
	  /* ST light is RED */
	  if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
	  }
          if(red_counter == 0 && loop_counter == 3){
		set_FT1_mode();    
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
     		printf("Selecting Self Test....\r");
		selftest++;
	  	if( selftest == 10){
		  printf("\t\t\tFailed!\n");
		  printf("Self Test will not operate while connected to a valid line\n");
		  FT1_operational_mode();
		  break;
	        }
	  }
	  if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
		printf("Performing Self Test....\r");
	        started = 1;
	  }
	  if(green_counter != 0 && off_counter != 0 && started){
	   	printf("\t\t\tDone!\n");
		break;
	  }
          loop_counter++;	  
     }/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void ){
     int dl_test=0;

     set_FT1_mode();
     off_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Bi-directional Digital Loop has been selected\n"); 
     for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;  
	}
	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Digital Loop Test....\r");
		dl_test++;
		if(dl_test==10){
	 		printf("\t\t\t\tFailed\n");
			printf("Remote End might be running Remote Test\n");
			break;
		}
	}
	if(red_counter != 0){
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = 0;
		printf("Performing Digital Loop Test....\r");
		for(;;){
			read_FT1_status();
			printf("Performing Digital Loop Test....\r");
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(red_counter != 0 && (par_port_B_byte & 0x08) == 0x00 ){
				printf("\t\t\t\tFailed!\n");
				printf("Bi-directional Digital Loop test has failed\n");
				printf("Either the unit is not connected or the line is Invalid\n");
				break;
			}
			if(green_counter != 0  && (par_port_B_byte & 0x0C) == 0x0C && loop_counter == 100 ){
				printf("\t\t\t\tDone!\n");
				printf("Bi-directional Digital Loop test has been successfully\n");
				printf("completed\n");
				break;
		
			}
			loop_counter++;
		} /* end of for */
		break;	
	} 
	loop_counter++;
     } /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void ){
    int ll_test = 0;

    set_FT1_mode();	
    off_counter = 0;
    red_counter = 0;
    loop_counter = 0;
    printf("Line Loop Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Line Loop Test....\r");
		ll_test++;
		if(ll_test == 10){
			printf("\t\t\t\tFailed!\n");
			printf("Line Loop Test will not operate while connected to a valid line\n");
			FT1_operational_mode();
		 	break;	
		}
	}
	if(red_counter != 0){	
		off_counter = 0;
		red_counter = 0;
		loop_counter = 0;
		for(;;){
			printf("Performing Line Loop Test....\r");
			read_FT1_status();	
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(green_counter != 0 && red_counter == 0){
				printf("\t\t\t\tDone!\n");
				printf("Line Loop Test has been successfully completed\n");
				break;
			}
			if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
				printf("\t\t\t\tFailed!\n");
				break;
			}
			loop_counter++;
		} /* end of for */
		break;
	} /* end of if */
	loop_counter++;
    } /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void ){
 
    off_counter = 0;
    red_counter = 0;
    green_counter = 0;	  	
    loop_counter = 0;
    printf("Remote Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x03) == 0x03)
		off_counter++;
	if((par_port_A_byte & 0x01) == 0x00)
		red_counter++;
	if((par_port_A_byte & 0x02) == 0x00)
		green_counter++;
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
		printf("Selecting Remote Test....\r");		
	}
	if(green_counter != 0 && loop_counter == 10){
	   loop_counter = 0;
	   off_counter = 0;
	   for(;;) {
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
	        if(off_counter == 0 && loop_counter == 20){
		   printf("Remote End is currently sending Remote Test data\n");
		   printf("Exit from the Remote Test mode at the remote end\n");
		   break;
		}
		if(off_counter != 0 && loop_counter == 20) {			
		   printf("This unit is currently in Remote Testing Mode\n");
		   break;
		}
	        loop_counter++;
	   }
	   break;
	}
	if(red_counter != 0){
		printf("Waiting for a valid response from remote end....\r");
		loop_counter = 0;
		for(;;){
			read_FT1_status();
			if((par_port_A_byte & 0x02) == 0x00){
				printf("\t\t\t\t\t\tDone!\n");
				printf("Valid response has been received from remote end\n");
				printf("Remote Test has been successfully completed\n");
				break;	
			}
			if((par_port_B_byte & 0x08) == 0x00){
				printf("\t\t\t\t\t\tFailed!\n");
				printf("The test can only be successful between two Sangoma\n");
				printf("S508/FT1 units configured the SAME WAY\n");

				break;
			}
		} /* end of for */
		break;
	}
    	loop_counter++;	
    } /* end of for */	
} /* FT1_rt_test */

//------------------------ END OF FT1 STATISTICS ------------------------






void usage( void ) {

	printf("xpipemon: Wanpipe X25 Debugging Utility\n\n");
	printf("Usage:\n");
	printf("-----\n\n");
	printf("xpipemon -i <ip-addr or interface> -u <port> -c <command> -d <dlci num>\n\n");
	printf("\tOption -i: \n");
	printf("\t\tWanpipe remote IP address or\n");
	printf("\t\tWanpipe network interface name (ex: wp1_fr16)\n");   
	printf("\tOption -u: (Optional, default 9000) \n");
	printf("\t\tWanpipe UDPPORT specified in /etc/wanpipe#.conf\n");
	printf("\tOption -c: \n");
	printf("\t\tFpipemon Command\n"); 
	printf("\t\t\tFirst letter is a command and the rest are options:\n"); 
	printf("\t\t\tex: xm = View Modem Status\n\n");
	printf("\tSupported Commands: x=status : s=statistics : t=trace \n");
	printf("\t                    c=config : T=FT1 stats  : f=flush\n\n");
	printf("\tCommand:  Options:   Description \n");	
	printf("\t-------------------------------- \n\n");    
	printf("\tCard Status\n");
	printf("\t   x         m       Modem Status\n");
	printf("\t             l       Link Status\n");
	printf("\t             ru      Display Router UP time\n");
	//printf("\tCard Configuration\n");
	//printf("\t   c         l       List Active DLCIs\n");
	printf("\tCard Statistics\n");
	//printf("\t   s         g       Global Statistics\n");
	printf("\t   s         c       Communication Error Statistics\n");
        printf("\t             x       X25 Statistics\n");
	printf("\t             h       HDLC Statistics\n");
	printf("\tTrace Data \n");
	//printf("\t   t         i       Trace and Interpret ALL frames\n");
	//printf("\t             ip      Trace and Interpret PROTOCOL frames only\n");
	//printf("\t             id      Trace and Interpret DATA frames only\n");
	printf("\t   t         r       Trace ALL frames, in RAW format\n");
	printf("\t             rp      Trace PROTOCOL frames only, in RAW format\n");
	printf("\t             rd      Trace DATA frames only, in RAW format\n");
	//printf("\tFT1 Configuration\n");
	//printf("\t   T         v       View Status \n");
	//printf("\t             s       Self Test\n");
	//printf("\t             l       Line Loop Test\n");
	//printf("\t             d       Digital Loop Test\n");
	//printf("\t             r       Remote Test\n");
	//printf("\t             o       Operational Mode\n");
	printf("\tDriver Statistics\n");
	printf("\t   d         s       Display If_Send Driver Statistics\n");
	printf("\t             i       Display Interrupt Driver Statistics\n");
	printf("\t             g       Display General Driver Statistics\n");	
	printf("\tFlush Statistics\n");
	//printf("\t   f         g       Flush Global Statistics\n");
	printf("\t   f         c       Flush Communication Error Statistics\n");
        printf("\t             h       Flush HDLC Statistics\n");
	//printf("\t             i       Flush DLCI Statistics\n");
	printf("\t             d       Flush Driver Statistics\n");
	printf("\tExamples:\n");
	printf("\t--------\n\n");
	printf("\tex: xpipemon -i wp1_svc1 -u 9000 -c xm   :View Modem Status \n");
	printf("\tex: xpipemon -i 201.1.1.2 -u 9000 -c ti  :Trace and Interpret ALL frames\n");
	//printf("\tex: xpipemon -i wp1_svc1 -u 9000 -c sd -l 1 :Statistics for LCN 1 \n\n");

}; //usage


void banner (char *title){
	
	int len,i;
	
	len = strlen(title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\t\t%s",title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\n");
	

}


int main(int argc, char* argv[])
{
	int proceed;
	char command;
	char *opt;

	printf("\n");
	if( argc > 2 ) {
		init( argc, argv);

		command = cmd[0];
		opt	= (char *) &cmd[1];

		if (!strcmp(ipaddress,"0.0.0.0")){
			proceed = MakeRawConnection();
		}else{
			proceed = MakeUdpConnection();
		}
		if(proceed == TRUE){
			switch(command) {
			case 'x':
				if (!strcmp(opt,"m")){
					modem_status();
				}else if (!strcmp(opt, "l")){
					link_status();
				}else if (!strcmp(opt, "ru")){
					x25_router_up_time();
				}else{
					printf("ERROR: Invalid Status Command 'x', Type xpipemon <cr> for help\n\n");
				}
				break;
			case 's':
				if (!strcmp(opt,"c")){
					comm_err();
	 			}else if (!strcmp(opt,"x")){
					read_x25_statistics();
			//	}else if (!strcmp(opt,"e")){
			//		error_stats();
				}else if (!strcmp(opt,"h")){
					read_hdlc_stat();
				}else {
					printf("ERROR: Invalid Statistics Command 's', Type xpipemon <cr> for help\n\n");
				}	
				break;
      			case 't':
				if(!strcmp(opt,"i" )){
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "ip")){
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "id")){
					line_trace(TRACE_DATA);
				}else if (!strcmp(opt, "r")){
					raw_data = TRUE;
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "rp")){
					raw_data = TRUE;
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "rd")){
					raw_data = TRUE;
					line_trace(TRACE_DATA);
				}else{
					printf("ERROR: Invalid Trace Command 't', Type xpipemon <cr> for help\n\n");
				}
	 			break;

      			case 'c':
				//if (!strcmp(opt, "l")){
	 			//	list_dlcis();
				//}else{
					printf("ERROR: Invalid Configuration Command 'c', Type xpipemon <cr> for help\n\n");

				//}
	 			break;
      			case 'd':
				  	/* Different signature for Driver Statistics */
	 			if(!strcmp(opt, "s")){
	 				x25_driver_stat_ifsend();
				}else if (!strcmp(opt, "i")){
					x25_driver_stat_intr();
	 			}else if (!strcmp(opt, "g")){
					x25_driver_stat_gen();
				}else{
					printf("ERROR: Invalid Driver Statistic 'd', Type xpipemon <cr> for help\n\n");
				}
	 			break;
      			case 'f':
	    			if (!strcmp(opt, "s")){
	       				flush_comm_err();
	       				comm_err();
				}else if (!strcmp(opt, "h")){
	       				flush_hdlc_stat();
	       				read_hdlc_stat();
			//	}else if (!strcmp(opt, "e")){
	      	 	//		flush_global_stats();
	       		//		error_stats();
			//	}else if (!strcmp(opt, "i")){
	       		//		flush_dlci_stats();
	       		//		read_dlci_stat();
				}else if (!strcmp(opt, "d")){
	       				flush_driver_stats();
				}else{
					printf("ERROR: Invalid Flush Command 'f', Type xpipemon <cr> for help\n\n");
				}
				break;
/*     			case 'T':

				if (!strcmp(opt, "v")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						view_FT1_status();
	     				}
	     				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "s")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_self_test();
	     				}
             				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "l")){
	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_ll_test();
	     				}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "d")){
   	     				set_FT1_monitor_status(0x01);
	     				if(!fail){
	     					FT1_dl_test();
	    				}
	     				set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "r")){
	     				set_FT1_monitor_status(0x01);
             				if(!fail){
	     					FT1_rt_test();
	     				}
					set_FT1_monitor_status(0x00);	
				}else if (!strcmp(opt, "o")){
	    				set_FT1_monitor_status(0x01);
	     				if(!fail){
						FT1_operational_mode();
	    	 			}
					set_FT1_monitor_status(0x00);
				}else{

					printf("ERROR: Invalid FT1 Command 'T', Type xpipemon <cr> for help\n\n");

				}
				break;	
*/
     			default:
				printf("ERROR: Invalid Command, Type xpipemon <cr> for help\n\n");

      			} //switch
     		}
     		close( sock );
   	} else {
      		usage();
   	} //if
   	printf("\n");
	return 0;
}; //main

#else

int main(int argc, char* argv[])
{
	printf("ERROR: This applicaton is not supported for kernels lower than 2.2.X\n");
	printf("       Upgrate to 2.2.X kernel for the X25 support\n");
	return 0;
}	

#endif

/*
 * EOF xpipemon.c
 */
