/*****************************************************************************
* sdla_edu.c	WANPIPE(tm) Multiprotocol WAN Link Driver. Educational kit module.
*
* Authors: 	David Rokhvarg <drokhvarg@sangoma.com>
*		 
*
* Copyright:	(c) 2001 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ============================================================================
* Sep 18, 2001  David Rokhvarg  Initial version.
* 			        
******************************************************************************/

#include <linux/version.h>
#include <linux/kernel.h>	/* printk(), and other useful stuff */
#include <linux/stddef.h>	/* offsetof(), etc. */
#include <linux/errno.h>	/* return codes */
#include <linux/string.h>	/* inline memset(), etc. */
#include <linux/slab.h>		/* kmalloc(), kfree() */
#include <linux/wanrouter.h>	/* WAN router definitions */
#include <linux/wanpipe.h>	/* WANPIPE common user API definitions */
#include <linux/if_arp.h>	/* ARPHRD_* defines */


#if defined(LINUX_2_4) || defined(LINUX_2_1)
  #include <linux/inetdevice.h>
  #include <asm/uaccess.h>

#else
  #include <net/route.h>          // Adding new route entries
  #include <asm/segment.h>
  #define test_and_set_bit set_bit
#endif


#include <linux/sdla_chdlc.h>		// CHDLC firmware API definitions
#include <linux/sdla_edu.h>		//educational package definitions

/****** Defines & Macros ****************************************************/
//#define DEBUG	//define this to get more output in kernel log
#define CODE_LENGTH	40000	//buffer length to store firmware module

 
/******Data Structures*****************************************************/



/******** Globals *********************************************************/
unsigned char data_buffer[MAX_DATA_LENGTH];
edu_exec_cmd_t ioctl_cmd;
unsigned char firmware_buffer[CODE_LENGTH];
int current_firmware_buffer_indx;


/****** Function Prototypes *************************************************/
/* WAN link driver entry points. These are called by the WAN router module. */
static int wpedu_exec (struct sdla *card, void *u_cmd, void *u_data);

static void DoIoctl(sdla_t *card);
static void read_block(sdla_t *card);
static void write_block(sdla_t * card);

/****** Public Functions ****************************************************/

/*============================================================================
 * Module initialization routine.
 *
 * This routine is called by the main WANPIPE module during setup.  At this
 * point adapter is completely initialized and CHDLC firmware is running.
 *  o read firmware version (to make sure it's alive)
 *  o configure adapter
 *  o initialize protocol-specific fields of the adapter data space.
 *
 * Return:	0	o.k.
 *		< 0	failure.
 */
int wpedu_init (sdla_t* card, wandev_conf_t* conf)
{
	unsigned char port_num;

	volatile CHDLC_MAILBOX_STRUCT* mb;
	CHDLC_MAILBOX_STRUCT* mb1;

#ifdef DEBUG	
	printk(KERN_INFO "%s:wpedu_init()\n", card->devname);
#endif
	/* Verify configuration ID */
	if (conf->config_id != WANCONFIG_EDUKIT) {
		printk(KERN_INFO "%s: invalid configuration ID %u!\n",
				  card->devname, conf->config_id);
		return -EINVAL;
	}

	/* Use primary port */
	card->u.c.comm_port = 0;
	
#ifdef DEBUG	
	/* Initialize protocol-specific fields */
	if(card->hw.type != SDLA_S514){
		card->mbox  = (void *) card->hw.dpmbase;
	}else{ 
		card->mbox = (void *) card->hw.dpmbase + PRI_BASE_ADDR_MB_STRUCT;
	}
#endif
	mb = mb1 = card->mbox;

	card->isr			= NULL;
	card->poll			= NULL;
	card->exec			= &wpedu_exec;
	card->wandev.update		= NULL;
 	card->wandev.new_if		= NULL;
	card->wandev.del_if		= NULL;
	card->wandev.state		= WAN_DUALPORT;
	card->wandev.udp_port   	= conf->udp_port;

	card->wandev.new_if_cnt = 0;

	/* This is for the ports link state */
	card->u.c.state = WAN_DISCONNECTED;
	
	/* reset the number of times the 'update()' proc has been called */
	card->u.c.update_call_count = 0;
	
	card->wandev.ttl = 0x7F;
	card->wandev.interface = 0; 

	card->wandev.clocking = 0;

	port_num = card->u.c.comm_port;

	/* Setup Port Bps */
       	card->wandev.bps = 0;

	card->wandev.mtu = MIN_LGTH_CHDLC_DATA_CFG;

	card->wandev.state = WAN_FT1_READY;
	printk(KERN_INFO "%s: Educational Kit Module Ready!\n",card->devname);

	return 0;
}


static int wpedu_exec(sdla_t *card, void *u_cmd, void *u_data)
{
		
#if defined(LINUX_2_1) || defined(LINUX_2_4)
	//copy command area
	if (copy_from_user((void*)&ioctl_cmd, u_cmd, sizeof(edu_exec_cmd_t))){
		printk(KERN_INFO "copy_from_user((void*)&ioctl_cmd...) Failed!\n");
		return -EFAULT;
	}

	//copy data
	if (copy_from_user((void*)data_buffer, u_data, MAX_DATA_LENGTH)){
		printk(KERN_INFO "wpedu_exec():copy_from_user((void*)data_buffer...) Failed!\n");
		return -EFAULT;
	}
	
	DoIoctl(card);

	// return result
	if( copy_to_user(u_cmd, (void*)&ioctl_cmd, sizeof(edu_exec_cmd_t))){
		printk(KERN_INFO "wpedu_exec():copy_to_user(u_cmd, ...) Failed!\n");
		return -EFAULT;
	}

	if( copy_to_user(u_data, (void*)data_buffer, MAX_DATA_LENGTH)){
		printk(KERN_INFO "wpedu_exec():copy_to_user(u_data, ...) Failed!\n");
		return -EFAULT;
	}
#else
	//copy command area 
        if (!u_cmd || verify_area(VERIFY_WRITE, u_cmd, sizeof(edu_exec_cmd_t))){
		printk(KERN_INFO "wpedu_exec():verify_area(VERIFY_WRITE, u_cmd...) Failed!\n");
                return -EFAULT;
	}
        memcpy_fromfs((void*)&ioctl_cmd, u_cmd, sizeof(edu_exec_cmd_t));

	//copy data
	if (!u_data || verify_area(VERIFY_WRITE, u_data,  MAX_DATA_LENGTH)){
		printk(KERN_INFO "wpedu_exec():verify_area(VERIFY_WRITE, u_data...) Failed!\n");
                return -EFAULT;
	}
        memcpy_fromfs((void*)data_buffer, u_data,  MAX_DATA_LENGTH);
	
	DoIoctl(card);
	
	// return result
	memcpy_tofs(u_cmd, (void*)&ioctl, sizeof(edu_exec_cmd_t));
        memcpy_tofs(u_data, (void*)data_buffer, MAX_DATA_LENGTH);

#endif
	return 0;
}

static void DoIoctl(sdla_t *card)
{
	int err;
	//char str[80];
	int timeout = jiffies;
		
	switch(ioctl_cmd.ioctl)
	{
	case IOCTL_READ_BLOCK:
#ifdef DEBUG		
		printk(KERN_INFO "IOCTL_READ_BLOCK\n");
#endif
		read_block(card);
		ioctl_cmd.return_code = 0;
		break;
		
	case IOCTL_WRITE_BLOCK:
#ifdef DEBUG
		printk(KERN_INFO "IOCTL_WRITE_BLOCK\n");
#endif
		write_block(card);
		ioctl_cmd.return_code = 0;
		break;

	case IOCTL_REGISTER_CARD:
#ifdef DEBUG
		printk(KERN_INFO "IOCTL_REGISTER_CARD\n");
#endif
		//This field will be used as a flag for determining if
		//it is safe to (re)load the card.
		//The flag is set to one before an application will load the card.
		//No application will be able to reload the card until the flag is reset by
		//IOCTL_DEREGISTER_CARD.
		if(card->wandev.interface == 0)
		{	card->wandev.interface = 1;
			ioctl_cmd.return_code = 0;			
		}else
		{	//indicate failure to the caller
			ioctl_cmd.return_code = 1;
		}
		break;

	case IOCTL_DEREGISTER_CARD:
#ifdef DEBUG
		printk(KERN_INFO "IOCTL_DEREGISTER_CARD\n");
#endif
		card->wandev.interface = 0;
		ioctl_cmd.return_code = 0;
		break;

	case IOCTL_CLEAN_UP:
	
		break;
	case IOCTL_LOAD:
#ifdef DEBUG
		printk(KERN_INFO "IOCTL_LOAD\n");
#endif
		/*
		  Decrement 'current_firmware_buffer_indx' by one to remove
		  the last garbage byte read after file end by 'fgetc(fw_file_ptr)'
		  in user mode.
		*/
		current_firmware_buffer_indx--;

		/*
		  Load firmware. If loader fails set ioctl->return_code to 1.
		  The firmware file is in firmware_buffer.
		*/
		err = sdla_setup (&card->hw, (void *)firmware_buffer, 
						current_firmware_buffer_indx);
		
		if (err){
			sdla_down(&card->hw);
			ioctl_cmd.return_code = 1;
		}

		/* Wait 1s for board to initialize */
		while (1){
			if ((jiffies - timeout) > 1*HZ)
				       	break;
		}
		
		break;
	case IOCTL_START_CODE_LOADING:
#ifdef DEBUG		
		printk(KERN_INFO "IOCTL_START_LOADING\n");
#endif
		//This ioctl will set everything up for firmware file
		//copying from user space into the kernel space buffer.
		current_firmware_buffer_indx = 0;
		break;
	case IOCTL_COPY_FIRMWARE:
#ifdef DEBUG
		printk(KERN_INFO "IOCTCL_COPY_FIRMWARE\n");
#endif
		memcpy(&firmware_buffer[current_firmware_buffer_indx],
		      data_buffer,
		      ioctl_cmd.buffer_length);

		current_firmware_buffer_indx += ioctl_cmd.buffer_length;
		break;
	default:
		printk(KERN_INFO "********* Unknown IOCTL : 0x%x *********\n", 
								ioctl_cmd.ioctl);	
		ioctl_cmd.return_code = 1;
		break;
	}
}


static void read_block(sdla_t *card)
{
	sdla_peek(&card->hw, ioctl_cmd.offset, data_buffer, ioctl_cmd.buffer_length);
}

static void write_block(sdla_t * card)
{
	sdla_poke(&card->hw, ioctl_cmd.offset, data_buffer, ioctl_cmd.buffer_length);
}


