/*****************************************************************************
* sdla_bitstrm.c WANPIPE(tm) Multiprotocol WAN Link Driver. Bit Stream module.
*
* Authors: 	Nenad Corbic <ncorbic@sangoma.com>
*		Gideon Hack  
*
* Copyright:	(c) 1995-2001 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ============================================================================
* Sep 20, 2001  Nenad Corbic    The min() function has changed for 2.4.9
*                               kernel. Thus using the wp_min() defined in
*                               wanpipe.h
* Aug 14, 2001	Nenad Corbic	Inital version, based on Chdlc module.
*      			        Using Gideons new bitstreaming firmware.
*****************************************************************************/

#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h>	/* printk(), and other useful stuff */
#include <linux/stddef.h>	/* offsetof(), etc. */
#include <linux/errno.h>	/* return codes */
#include <linux/string.h>	/* inline memset(), etc. */
#include <linux/slab.h>		/* kmalloc(), kfree() */
#include <linux/wanrouter.h>	/* WAN router definitions */
#include <linux/wanpipe.h>	/* WANPIPE common user API definitions */
#include <linux/if_arp.h>	/* ARPHRD_* defines */


#if defined(LINUX_2_1) || defined(LINUX_2_4)
 #include <asm/uaccess.h>
 #include <linux/inetdevice.h>
 #include <linux/netdevice.h>
#else 				
 #include <asm/segment.h>
 #include <net/route.h>          /* Adding new route entries : 2.0.X kernels */
#endif

#include <linux/in.h>		/* sockaddr_in */
#include <linux/inet.h>	
#include <linux/if.h>
#include <asm/byteorder.h>	/* htons(), etc. */
#include <linux/sdlapci.h>
#include <asm/io.h>

#include <linux/sdla_bitstrm.h>		/* BSTRM firmware API definitions */

#include <linux/if_wanpipe_common.h>    /* Socket Driver common area */
#include <linux/if_wanpipe.h>		

/****** Defines & Macros ****************************************************/

/* reasons for enabling the timer interrupt on the adapter */
#define TMR_INT_ENABLED_UDP   		0x01
#define TMR_INT_ENABLED_UPDATE		0x02
#define TMR_INT_ENABLED_CONFIG		0x10

#define MAX_IP_ERRORS	10

#define	BSTRM_DFLT_DATA_LEN	1500		/* default MTU */
#define BSTRM_HDR_LEN		1

#define BSTRM_API 0x01

#define PORT(x)   (x == 0 ? "PRIMARY" : "SECONDARY" )
#define MAX_BH_BUFF	10

//#define PRINT_DEBUG
#ifdef PRINT_DEBUG
#define dbg_printk(format, a...) printk(format, ## a)
#else
#define dbg_printk(format, a...)
#endif  

/******Data Structures*****************************************************/

/* This structure is placed in the private data area of the device structure.
 * The card structure used to occupy the private area but now the following 
 * structure will incorporate the card structure along with BSTRM specific data
 */

typedef struct bitstrm_private_area
{
	wanpipe_common_t common;
	sdla_t		*card;
	int 		TracingEnabled;		/* For enabling Tracing */
	unsigned long 	curr_trace_addr;	/* Used for Tracing */
	unsigned long 	start_trace_addr;
	unsigned long 	end_trace_addr;
	unsigned long 	base_addr_trace_buffer;
	unsigned long 	end_addr_trace_buffer;
	unsigned short 	number_trace_elements;
	unsigned  	available_buffer_space;
	unsigned long 	router_start_time;
	unsigned char 	route_status;
	unsigned char 	route_removed;
	unsigned long 	tick_counter;		/* For 5s timeout counter */
	unsigned long 	router_up_time;
        u32             IP_address;		/* IP addressing */
        u32             IP_netmask;
	u32		ip_local;
	u32		ip_remote;
	u32 		ip_local_tmp;
	u32		ip_remote_tmp;
	u8		ip_error;
	u8		config_bstrm;
	u8 		config_bstrm_timeout;
	unsigned char  mc;			/* Mulitcast support on/off */
	unsigned short udp_pkt_lgth;		/* udp packet processing */
	char udp_pkt_src;
	char udp_pkt_data[MAX_LGTH_UDP_MGNT_PKT];
	unsigned short timer_int_enabled;
	char update_comms_stats;		/* updating comms stats */

#if defined(LINUX_2_1) || defined(LINUX_2_4)
	bh_data_t *bh_head;	  	  /* Circular buffer for chdlc_bh */
	unsigned long  tq_working;
	volatile int  bh_write;
	volatile int  bh_read;
	atomic_t  bh_buff_used;
#endif
	
	unsigned char interface_down;

	/* Polling task queue. Each interface
         * has its own task queue, which is used
         * to defer events from the interrupt */
	struct tq_struct poll_task;
	struct timer_list poll_delay_timer;

	u8 gateway;
	u8 true_if_encoding;
	//FIXME: add driver stats as per frame relay!

} bitstrm_private_area_t;

/* Route Status options */
#define NO_ROUTE	0x00
#define ADD_ROUTE	0x01
#define ROUTE_ADDED	0x02
#define REMOVE_ROUTE	0x03


/* variable for tracking how many interfaces to open for WANPIPE on the
   two ports */

extern void disable_irq(unsigned int);
extern void enable_irq(unsigned int);


/****** Function Prototypes *************************************************/
/* WAN link driver entry points. These are called by the WAN router module. */
static int update (wan_device_t* wandev);
static int new_if (wan_device_t* wandev, netdevice_t* dev,
	wanif_conf_t* conf);

/* Network device interface */
static int if_init   (netdevice_t* dev);
static int if_open   (netdevice_t* dev);
static int if_close  (netdevice_t* dev);
static int if_header (struct sk_buff* skb, netdevice_t* dev,
	unsigned short type, void* daddr, void* saddr, unsigned len);

#if defined(LINUX_2_1) || defined(LINUX_2_4)
  static int if_rebuild_hdr (struct sk_buff *skb);
  static struct net_device_stats* if_stats (netdevice_t* dev);

#else
  static int if_rebuild_hdr (void* hdr, netdevice_t* dev, unsigned long raddr,
        struct sk_buff* skb);
  static struct enet_statistics* if_stats (netdevice_t* dev);
#endif
  
static int if_send (struct sk_buff* skb, netdevice_t* dev);

/* BSTRM Firmware interface functions */
static int bstrm_configure 	(sdla_t* card, void* data);
static int bstrm_comm_enable 	(sdla_t* card);
static int bstrm_read_version 	(sdla_t* card, char* str);
static int bstrm_set_intr_mode 	(sdla_t* card, unsigned mode);
static int bstrm_send (sdla_t* card, void* data, unsigned len);
static int bstrm_read_comm_err_stats (sdla_t* card);
static int bstrm_read_op_stats (sdla_t* card);
static int bstrm_error (sdla_t *card, int err, BSTRM_MAILBOX_STRUCT *mb);


static int bstrm_disable_comm_shutdown (sdla_t *card);
#ifdef LINUX_2_4
  static void if_tx_timeout (netdevice_t *dev);
#endif


/* Miscellaneous BSTRM Functions */
static int set_bstrm_config (sdla_t* card);
static void init_bstrm_tx_rx_buff( sdla_t* card);
static int process_bstrm_exception(sdla_t *card);
static int process_global_exception(sdla_t *card);
static int update_comms_stats(sdla_t* card,
        bitstrm_private_area_t* bstrm_priv_area);
static void port_set_state (sdla_t *card, int);
static int config_bstrm (sdla_t *card);
static void disable_comm (sdla_t *card);


/* Interrupt handlers */
static void wpbit_isr (sdla_t* card);
static void rx_intr (sdla_t* card);
static void timer_intr(sdla_t *);

#if defined(LINUX_2_1) || defined(LINUX_2_4)
  /* Bottom half handlers */
  static void bstrm_bh (netdevice_t *);
  static int bstrm_bh_cleanup (netdevice_t *);
  static int bh_enqueue (netdevice_t *, struct sk_buff *);
  static void wakeup_sk_bh (netdevice_t *dev);
#endif

/* Miscellaneous functions */
static int chk_bcast_mcast_addr(sdla_t* card, netdevice_t* dev,
				struct sk_buff *skb);
static int reply_udp( unsigned char *data, unsigned int mbox_len );
static int intr_test( sdla_t* card);
static int udp_pkt_type( struct sk_buff *skb , sdla_t* card);
static int store_udp_mgmt_pkt(char udp_pkt_src, sdla_t* card,
                                struct sk_buff *skb, netdevice_t* dev,
                                bitstrm_private_area_t* bstrm_priv_area);
static int process_udp_mgmt_pkt(sdla_t* card, netdevice_t* dev,  
				bitstrm_private_area_t* bstrm_priv_area);
static unsigned short calc_checksum (char *, int);
static void s508_lock (sdla_t *card, unsigned long *smp_flags);
static void s508_unlock (sdla_t *card, unsigned long *smp_flags);


static int  Intr_test_counter;

/* TE1 */
static WRITE_FRONT_END_REG_T write_front_end_reg;
static READ_FRONT_END_REG_T  read_front_end_reg;

/****** Public Functions ****************************************************/

/*============================================================================
 * Cisco HDLC protocol initialization routine.
 *
 * This routine is called by the main WANPIPE module during setup.  At this
 * point adapter is completely initialized and firmware is running.
 *  o read firmware version (to make sure it's alive)
 *  o configure adapter
 *  o initialize protocol-specific fields of the adapter data space.
 *
 * Return:	0	o.k.
 *		< 0	failure.
 */
int wpbit_init (sdla_t* card, wandev_conf_t* conf)
{
	unsigned char port_num;
	int err;
	unsigned long max_permitted_baud = 0;
	SHARED_MEMORY_INFO_STRUCT *flags;

	union
		{
		char str[80];
		} u;

	volatile BSTRM_MAILBOX_STRUCT* mb;
	BSTRM_MAILBOX_STRUCT* mb1;
	unsigned long timeout;

	/* Verify configuration ID */
	if (conf->config_id != WANCONFIG_BITSTRM) {
		printk(KERN_INFO "%s: invalid configuration ID %u!\n",
				  card->devname, conf->config_id);
		return -EINVAL;
	}

	/* Find out which Port to use */
	if ((conf->comm_port == WANOPT_PRI) || (conf->comm_port == WANOPT_SEC)){
		if (card->next){

			if (conf->comm_port != card->next->u.b.comm_port){
				card->u.b.comm_port = conf->comm_port;
			}else{
				printk(KERN_INFO "%s: ERROR - %s port used!\n",
        		        	card->wandev.name, PORT(conf->comm_port));
				return -EINVAL;
			}
		}else{
			card->u.b.comm_port = conf->comm_port;
		}
	}else{
		printk(KERN_INFO "%s: ERROR - Invalid Port Selected!\n",
                			card->wandev.name);
		return -EINVAL;
	}
	

	/* Initialize protocol-specific fields */
	if(card->hw.type != SDLA_S514){

		if (card->u.b.comm_port == WANOPT_PRI){	
			card->mbox  = (void *) card->hw.dpmbase;
		}else{
			card->mbox  = (void *) card->hw.dpmbase + 
				SEC_BASE_ADDR_MB_STRUCT - PRI_BASE_ADDR_MB_STRUCT;
		}	
	}else{ 
		/* for a S514 adapter, set a pointer to the actual mailbox in the */
		/* allocated virtual memory area */
		if (card->u.b.comm_port == WANOPT_PRI){
			card->mbox = (void *) card->hw.dpmbase + PRI_BASE_ADDR_MB_STRUCT;
		}else{
			card->mbox = (void *) card->hw.dpmbase + SEC_BASE_ADDR_MB_STRUCT;
		}	
	}

	mb = mb1 = card->mbox;

	if (!card->configured){

		/* The board will place an 'I' in the return code to indicate that it is
	   	ready to accept commands.  We expect this to be completed in less
           	than 1 second. */

		timeout = jiffies;
		while (mb->return_code != 'I')	/* Wait 1s for board to initialize */
			if ((jiffies - timeout) > 1*HZ) break;

		if (mb->return_code != 'I') {
			printk(KERN_INFO
				"%s: Initialization not completed by adapter\n",
				card->devname);
			printk(KERN_INFO "Please contact Sangoma representative.\n");
			return -EIO;
		}
	}

	/* TE1 Make special hardware initialization for T1/E1 board */
	if (IS_TE1(conf->te_cfg)) {
		wan_device_t* wandev = &card->wandev;
		memcpy(&wandev->te_cfg, &conf->te_cfg, sizeof(sdla_te_cfg_t));
		card->wandev.write_front_end_reg = write_front_end_reg;
		card->wandev.read_front_end_reg = read_front_end_reg;
		printk(KERN_INFO "%s: Configuring T1/E1 onboard CSU/DSU\n",
				card->devname);
		sdla_te_config(card);
		card->wandev.clocking = WANOPT_EXTERNAL;

	}else if (IS_56K(conf->te_cfg)) {
		wan_device_t* wandev = &card->wandev;
		memcpy(&wandev->te_cfg, &conf->te_cfg, sizeof(sdla_te_cfg_t));
		card->wandev.write_front_end_reg = write_front_end_reg;
		card->wandev.read_front_end_reg = read_front_end_reg;
		printk(KERN_INFO "%s: Configuring 56K onboard CSU/DSU\n",
				card->devname);
		sdla_56k_config(card);
		card->wandev.clocking = WANOPT_EXTERNAL;
	}

	/* Read firmware version.  Note that when adapter initializes, it
	 * clears the mailbox, so it may appear that the first command was
	 * executed successfully when in fact it was merely erased. To work
	 * around this, we execute the first command twice.
	 */

	if (bstrm_read_version(card, u.str))
		return -EIO;

	printk(KERN_INFO "%s: Running Bit Streaming firmware v%s\n",
		card->devname, u.str); 

	card->isr			= &wpbit_isr;
	card->poll			= NULL;
	card->exec			= NULL;
	card->wandev.update		= &update;
 	card->wandev.new_if		= &new_if;
	card->wandev.del_if		= NULL;
	card->wandev.udp_port   	= conf->udp_port;
	card->disable_comm		= &disable_comm;
	card->wandev.new_if_cnt = 0;

	/* Initialize Bit Stream Configuration */
	card->u.b.sync_options = conf->sync_options;
	card->u.b.rx_sync_char = conf->rx_sync_char;
	card->u.b.monosync_tx_time_fill_char = conf->monosync_tx_time_fill_char;
	
	card->u.b.max_length_tx_data_block = conf->max_length_tx_data_block;
	card->u.b.rx_complete_length = conf->rx_complete_length;
	card->u.b.rx_complete_timer = conf->rx_complete_timer;
	
	/* reset the number of times the 'update()' proc has been called */
	card->u.b.update_call_count = 0;
	
	card->wandev.ttl = conf->ttl;
	card->wandev.interface = conf->interface; 

	if ((card->u.b.comm_port == WANOPT_SEC && conf->interface == WANOPT_V35)&&
	    card->hw.type != SDLA_S514){
		printk(KERN_INFO "%s: ERROR - V35 Interface not supported on S508 %s port \n",
			card->devname, PORT(card->u.b.comm_port));
		return -EIO;
	}

	card->wandev.clocking = conf->clocking;

	port_num = card->u.b.comm_port;


	/* Setup Port Bps */

	if(card->wandev.clocking) {
		if(port_num == WANOPT_PRI) {
			/* For Primary Port 0 */
               		max_permitted_baud =
				(card->hw.type == SDLA_S514) ?
				PRI_MAX_BAUD_RATE_S514 : 
				PRI_MAX_BAUD_RATE_S508;

		}else if(port_num == WANOPT_SEC) {
			/* For Secondary Port 1 */
                        max_permitted_baud =
                               (card->hw.type == SDLA_S514) ?
                                SEC_MAX_BAUD_RATE_S514 :
                                SEC_MAX_BAUD_RATE_S508;
                        }
  
			if(conf->bps > max_permitted_baud) {
				conf->bps = max_permitted_baud;
				printk(KERN_INFO "%s: Baud too high!\n",
					card->wandev.name);
 				printk(KERN_INFO "%s: Baud rate set to %lu bps\n", 
					card->wandev.name, max_permitted_baud);
			}
			card->wandev.bps = conf->bps;
	}else{
        	card->wandev.bps = 0;
  	}

	/* Setup the Port MTU */
	if(port_num == WANOPT_PRI) {

		/* For Primary Port 0 */
		card->wandev.mtu =  wp_min(conf->mtu, PRI_MAX_LENGTH_TX_DATA_BLOCK);
	} else if(port_num == WANOPT_SEC) { 
		/* For Secondary Port 1 */
		card->wandev.mtu = wp_min(conf->mtu, SEC_MAX_LENGTH_TX_DATA_BLOCK);
	}

	/* Set up the interrupt status area */
	/* Read the BSTRM Configuration and obtain: 
	 *	Ptr to shared memory infor struct
         * Use this pointer to calculate the value of card->u.b.flags !
 	 */
	mb1->buffer_length = 0;
	mb1->command = READ_BSTRM_CONFIGURATION;
	err = sdla_exec(mb1) ? mb1->return_code : CMD_TIMEOUT;
	if(err != OK) {
                if(card->hw.type != SDLA_S514)
                	enable_irq(card->hw.irq);

		bstrm_error(card, err, mb1);
		return -EIO;
	}

	if(card->hw.type == SDLA_S514){
               	card->u.b.flags = (void *)(card->hw.dpmbase +
               		(((BSTRM_CONFIGURATION_STRUCT *)mb1->data)->
			ptr_shared_mem_info_struct));
        }else{
                card->u.b.flags = (void *)(card->hw.dpmbase +
                        (((BSTRM_CONFIGURATION_STRUCT *)mb1->data)->
			ptr_shared_mem_info_struct % SDLA_WINDOWSIZE));
	}

	flags = card->u.b.flags;
	
	/* This is for the ports link state */
	card->wandev.state = WAN_DUALPORT;
	card->u.b.state = WAN_DISCONNECTED;


	if (!card->wandev.piggyback){	
		int err;

		/* Perform interrupt testing */
		err = intr_test(card);

		if(err || (Intr_test_counter < MAX_INTR_TEST_COUNTER)) { 
			printk(KERN_INFO "%s: Interrupt test failed (%i)\n",
					card->devname, Intr_test_counter);
			printk(KERN_INFO "%s: Please choose another interrupt\n",
					card->devname);
			return -EIO;
		}
		
		printk(KERN_INFO "%s: Interrupt test passed (%i)\n", 
				card->devname, Intr_test_counter);
		card->configured = 1;
	}

	if (bstrm_set_intr_mode(card, APP_INT_ON_TIMER)){
		printk (KERN_INFO "%s: 
			Failed to set interrupt triggers!\n",
			card->devname);
		return -EIO;	
	}

	/* Mask the Timer interrupt */
	flags->interrupt_info_struct.interrupt_permission &= 
		~APP_INT_ON_TIMER;

	/* If we are using BSTRM in backup mode, this flag will
	 * indicate not to look for IP addresses in config_bstrm()*/
	card->u.b.backup = conf->backup;
	
	printk(KERN_INFO "\n");

	return 0;
}

/******* WAN Device Driver Entry Points *************************************/

/*============================================================================
 * Update device status & statistics
 * This procedure is called when updating the PROC file system and returns
 * various communications statistics. These statistics are accumulated from 3 
 * different locations:
 * 	1) The 'if_stats' recorded for the device.
 * 	2) Communication error statistics on the adapter.
 *      3) BSTRM operational statistics on the adapter.
 * The board level statistics are read during a timer interrupt. Note that we 
 * read the error and operational statistics during consecitive timer ticks so
 * as to minimize the time that we are inside the interrupt handler.
 *
 */
static int update (wan_device_t* wandev)
{
	sdla_t* card = wandev->private;
 	netdevice_t* dev;
        volatile bitstrm_private_area_t* bstrm_priv_area;
        SHARED_MEMORY_INFO_STRUCT *flags;
	unsigned long timeout;

	/* sanity checks */
	if((wandev == NULL) || (wandev->private == NULL))
		return -EFAULT;
	
	if(wandev->state == WAN_UNCONFIGURED)
		return -ENODEV;

	/* more sanity checks */
        if(!card->u.b.flags)
                return -ENODEV;

	if(test_bit(PERI_CRIT, (void*)&card->wandev.critical))
                return -EAGAIN;

	if((dev=card->wandev.dev) == NULL)
		return -ENODEV;

	if((bstrm_priv_area=dev->priv) == NULL)
		return -ENODEV;

      	flags = card->u.b.flags;
       	if(bstrm_priv_area->update_comms_stats){
		return -EAGAIN;
	}
			
	/* TE1	Change the update_comms_stats variable to 3,
	 * 	only for T1/E1 card, otherwise 2 for regular
	 *	S514/S508 card.
	 *	Each timer interrupt will update only one type
	 *	of statistics.
	 */
	bstrm_priv_area->update_comms_stats = 
		(IS_TE1(card->wandev.te_cfg) || IS_56K(card->wandev.te_cfg)) ? 3 : 2;
       	flags->interrupt_info_struct.interrupt_permission |= APP_INT_ON_TIMER;
	bstrm_priv_area->timer_int_enabled = TMR_INT_ENABLED_UPDATE;
  
	/* wait a maximum of 1 second for the statistics to be updated */ 
        timeout = jiffies;
        for(;;) {
		if(bstrm_priv_area->update_comms_stats == 0)
			break;
                if ((jiffies - timeout) > (1 * HZ)){
    			bstrm_priv_area->update_comms_stats = 0;
 			bstrm_priv_area->timer_int_enabled &=
				~TMR_INT_ENABLED_UPDATE; 
 			return -EAGAIN;
		}
        }

	return 0;
}


/*============================================================================
 * Create new logical channel.
 * This routine is called by the router when ROUTER_IFNEW IOCTL is being
 * handled.
 * o parse media- and hardware-specific configuration
 * o make sure that a new channel can be created
 * o allocate resources, if necessary
 * o prepare network device structure for registaration.
 *
 * Return:	0	o.k.
 *		< 0	failure (channel will not be created)
 */
static int new_if (wan_device_t* wandev, netdevice_t* dev, wanif_conf_t* conf)
{
	sdla_t* card = wandev->private;
	bitstrm_private_area_t* bstrm_priv_area;


	printk(KERN_INFO "%s: Configuring Interface: %s\n",
			card->devname, conf->name);
 
	if ((conf->name[0] == '\0') || (strlen(conf->name) > WAN_IFNAME_SZ)) {
		printk(KERN_INFO "%s: Invalid interface name!\n",
			card->devname);
		return -EINVAL;
	}
		
	/* allocate and initialize private data */
	bstrm_priv_area = kmalloc(sizeof(bitstrm_private_area_t), GFP_KERNEL);
	
	if(bstrm_priv_area == NULL) 
		return -ENOMEM;

	memset(bstrm_priv_area, 0, sizeof(bitstrm_private_area_t));

	bstrm_priv_area->card = card; 
	bstrm_priv_area->common.sk = NULL;
	bstrm_priv_area->common.func = NULL;	

	/* initialize data */
	strcpy(card->u.b.if_name, conf->name);

	if(card->wandev.new_if_cnt > 0) {
                kfree(bstrm_priv_area);
		return -EEXIST;
	}

	card->wandev.new_if_cnt++;

	bstrm_priv_area->TracingEnabled = 0;
	bstrm_priv_area->route_status = NO_ROUTE;
	bstrm_priv_area->route_removed = 0;

	/* Setup wanpipe as a router (WANPIPE) or as an API */
	if( strcmp(conf->usedby, "WANPIPE") == 0) {

		printk(KERN_INFO "%s: Bit Stream module doesn't support WANPIPE mode! API only!\n",
						wandev->name);
		kfree(bstrm_priv_area);
		return -EINVAL;

	} else if( strcmp(conf->usedby, "API") == 0) {
#if defined(LINUX_2_1) || defined(LINUX_2_4) 
		card->u.b.usedby = API;
		printk(KERN_INFO "%s: Running in API mode !\n",
			wandev->name);
#else
		printk(KERN_INFO "%s: API Mode is not supported for kernels lower than 2.2.X!\n",
			wandev->name);
		printk(KERN_INFO "%s: Please upgrade to a 2.2.X kernel fro the API support\n",
			wandev->name);
		kfree(bstrm_priv_area);
		return -EINVAL;
#endif
	}

	/* prepare network device data space for registration */
#ifdef LINUX_2_4
	strcpy(dev->name,card->u.b.if_name);
#else
	dev->name = (char *)kmalloc(strlen(card->u.b.if_name) + 2, GFP_KERNEL); 
	sprintf(dev->name, "%s", card->u.b.if_name);
#endif

	dev->init = &if_init;
	dev->priv = bstrm_priv_area;

	printk(KERN_INFO "\n");

	return 0;
}


/****** Network Device Interface ********************************************/

/*============================================================================
 * Initialize Linux network interface.
 *
 * This routine is called only once for each interface, during Linux network
 * interface registration.  Returning anything but zero will fail interface
 * registration.
 */
static int if_init (netdevice_t* dev)
	{
	bitstrm_private_area_t* bstrm_priv_area = dev->priv;
	sdla_t* card = bstrm_priv_area->card;
	wan_device_t* wandev = &card->wandev;
#ifdef LINUX_2_0
	int i;
#endif

	/* Initialize device driver entry points */
	dev->open		= &if_open;
	dev->stop		= &if_close;
	dev->hard_header	= &if_header;
	dev->rebuild_header	= &if_rebuild_hdr;
	dev->hard_start_xmit	= &if_send;
	dev->get_stats		= &if_stats;
#ifdef LINUX_2_4
	dev->tx_timeout		= &if_tx_timeout;
	dev->watchdog_timeo	= TX_TIMEOUT;
#endif

	
	/* Initialize media-specific parameters */
	dev->flags		|= IFF_POINTOPOINT;
	dev->flags		|= IFF_NOARP;

	/* Enable Mulitcasting if user selected */
	if (bstrm_priv_area->mc == WANOPT_YES){
		dev->flags 	|= IFF_MULTICAST;
	}
	
#ifdef LINUX_2_0
	dev->family		= AF_INET;
#endif	

	if (bstrm_priv_area->true_if_encoding){
#if defined(LINUX_2_1) || defined(LINUX_2_4)
		dev->type	= ARPHRD_HDLC; /* This breaks the tcpdump */
#else
		dev->type	= ARPHRD_PPP;
#endif
	}else{
		dev->type	= ARPHRD_PPP;
	}
	
	dev->mtu		= card->u.b.max_length_tx_data_block;
	/* for API usage, add the API header size to the requested MTU size */
	if(card->u.b.usedby == API) {
		dev->mtu += sizeof(api_tx_hdr_t);
	}
 
	dev->hard_header_len	= BSTRM_HDR_LEN;

	/* Initialize hardware parameters */
	dev->irq	= wandev->irq;
	dev->dma	= wandev->dma;
	dev->base_addr	= wandev->ioport;
	dev->mem_start	= wandev->maddr;
	dev->mem_end	= wandev->maddr + wandev->msize - 1;

	/* Set transmit buffer queue length 
	 * If too low packets will not be retransmitted 
         * by stack.
	 */
        dev->tx_queue_len = 100;
   
	/* Initialize socket buffers */
#if !defined(LINUX_2_1) && !defined(LINUX_2_4)
        for (i = 0; i < DEV_NUMBUFFS; ++i)
                skb_queue_head_init(&dev->buffs[i]);
#endif
	return 0;
}

/*============================================================================
 * Open network interface.
 * o enable communications and interrupts.
 * o prevent module from unloading by incrementing use count
 *
 * Return 0 if O.k. or errno.
 */
static int if_open (netdevice_t* dev)
{
	bitstrm_private_area_t* bstrm_priv_area = dev->priv;
	sdla_t* card = bstrm_priv_area->card;
	struct timeval tv;
	int err = 0;
	SHARED_MEMORY_INFO_STRUCT *flags = card->u.b.flags;
	


	/* Only one open per interface is allowed */

	if (is_dev_running(dev))
		return -EBUSY;

#if defined(LINUX_2_1) || defined(LINUX_2_4)
	/* Initialize the task queue */
	bstrm_priv_area->tq_working=0;

#ifndef LINUX_2_4
	bstrm_priv_area->common.wanpipe_task.next = NULL;
#endif
	bstrm_priv_area->common.wanpipe_task.sync = 0;
	bstrm_priv_area->common.wanpipe_task.routine = (void *)(void *)bstrm_bh;
	bstrm_priv_area->common.wanpipe_task.data = dev;

	/* Allocate and initialize BH circular buffer */
	/* Add 1 to MAX_BH_BUFF so we don't have test with (MAX_BH_BUFF-1) */
	bstrm_priv_area->bh_head = kmalloc((sizeof(bh_data_t)*(MAX_BH_BUFF+1)),GFP_ATOMIC);
	memset(bstrm_priv_area->bh_head,0,(sizeof(bh_data_t)*(MAX_BH_BUFF+1)));
	atomic_set(&bstrm_priv_area->bh_buff_used, 0);
#endif
 
	do_gettimeofday(&tv);
	bstrm_priv_area->router_start_time = tv.tv_sec;

#ifdef LINUX_2_4
	netif_start_queue(dev);
#else
	dev->interrupt = 0;
	dev->tbusy = 0;
	dev->start = 1;
#endif

	wanpipe_open(card);

	bstrm_priv_area->timer_int_enabled |= TMR_INT_ENABLED_CONFIG;
	flags->interrupt_info_struct.interrupt_permission |= 
			 APP_INT_ON_TIMER;
	return err;
}

/*============================================================================
 * Close network interface.
 * o if this is the last close, then disable communications and interrupts.
 * o reset flags.
 */
static int if_close (netdevice_t* dev)
{
	bitstrm_private_area_t* bstrm_priv_area = dev->priv;
	sdla_t* card = bstrm_priv_area->card;
	unsigned long smp_flags=0;
		
#if defined(LINUX_2_1) || defined(LINUX_2_4)
	lock_adapter_irq(&card->wandev.lock, &smp_flags);
	if (bstrm_priv_area->bh_head){
		int i;
		struct sk_buff *skb;
	
		for (i=0; i<(MAX_BH_BUFF+1); i++){
			skb = ((bh_data_t *)&bstrm_priv_area->bh_head[i])->skb;
			if (skb != NULL){
                		wan_dev_kfree_skb(skb, FREE_READ);
			}
		}
		kfree(bstrm_priv_area->bh_head);
		bstrm_priv_area->bh_head=NULL;
	}
	unlock_adapter_irq(&card->wandev.lock, &smp_flags);
#endif

	stop_net_queue(dev);
#ifndef LINUX_2_4
	dev->start=0;
#endif
	wanpipe_close(card);
	del_timer(&bstrm_priv_area->poll_delay_timer);
	return 0;
}

static void disable_comm (sdla_t *card)
{
	SHARED_MEMORY_INFO_STRUCT *flags = card->u.b.flags;
	
	if (card->u.b.comm_enabled){
		bstrm_disable_comm_shutdown (card);
	}else{
		flags->interrupt_info_struct.interrupt_permission = 0;	
	}

	return;
}


/*============================================================================
 * Build media header.
 *
 * The trick here is to put packet type (Ethertype) into 'protocol' field of
 * the socket buffer, so that we don't forget it.  If packet type is not
 * supported, set skb->protocol to 0 and discard packet later.
 *
 * Return:	media header length.
 */
static int if_header (struct sk_buff* skb, netdevice_t* dev,
	unsigned short type, void* daddr, void* saddr, unsigned len)
{
	skb->protocol = htons(type);

	return BSTRM_HDR_LEN;
}


#ifdef LINUX_2_4
/*============================================================================
 * Handle transmit timeout event from netif watchdog
 */
static void if_tx_timeout (netdevice_t *dev)
{
    	bitstrm_private_area_t* chan = dev->priv;
	sdla_t *card = chan->card;
	
	/* If our device stays busy for at least 5 seconds then we will
	 * kick start the device by making dev->tbusy = 0.  We expect
	 * that our device never stays busy more than 5 seconds. So this                 
	 * is only used as a last resort.
	 */

	++card->wandev.stats.collisions;

	printk (KERN_INFO "%s: Transmit timed out on %s\n", card->devname,dev->name);
	netif_wake_queue (dev);
}
#endif



/*============================================================================
 * Re-build media header.
 *
 * Return:	1	physical address resolved.
 *		0	physical address not resolved
 */
#if defined(LINUX_2_1) || defined(LINUX_2_4)
static int if_rebuild_hdr (struct sk_buff *skb)
{
	return 1;
}
#else
static int if_rebuild_hdr (void* hdr, netdevice_t* dev, unsigned long raddr,
                           struct sk_buff* skb)
{
        return 1;
}
#endif

/*============================================================================
 * Send a packet on a network interface.
 * o set tbusy flag (marks start of the transmission) to block a timer-based
 *   transmit from overlapping.
 * o check link state. If link is not up, then drop the packet.
 * o execute adapter send command.
 * o free socket buffer
 *
 * Return:	0	complete (socket buffer must be freed)
 *		non-0	packet may be re-transmitted (tbusy must be set)
 *
 * Notes:
 * 1. This routine is called either by the protocol stack or by the "net
 *    bottom half" (with interrupts enabled).
 * 2. Setting tbusy flag will inhibit further transmit requests from the
 *    protocol stack and can be used for flow control with protocol layer.
 */
static int if_send (struct sk_buff* skb, netdevice_t* dev)
{
	bitstrm_private_area_t *bstrm_priv_area = dev->priv;
	sdla_t *card = bstrm_priv_area->card;
	SHARED_MEMORY_INFO_STRUCT *flags = card->u.b.flags;
	INTERRUPT_INFORMATION_STRUCT *bstrm_int = &flags->interrupt_info_struct;
	int udp_type = 0;
	unsigned long smp_flags;
	int err=0;

#ifdef LINUX_2_4
	netif_stop_queue(dev);
#endif
	
	if (skb == NULL){
		/* If we get here, some higher layer thinks we've missed an
		 * tx-done interrupt.
		 */
		printk(KERN_INFO "%s: interface %s got kicked!\n",
			card->devname, dev->name);

		wake_net_dev(dev);
		return 0;
	}

#ifndef LINUX_2_4
	if (dev->tbusy){

		/* If our device stays busy for at least 5 seconds then we will
		 * kick start the device by making dev->tbusy = 0.  We expect 
		 * that our device never stays busy more than 5 seconds. So this
		 * is only used as a last resort. 
		 */
                ++card->wandev.stats.collisions;
		if((jiffies - bstrm_priv_area->tick_counter) < (5 * HZ)) {
			return 1;
		}

		printk (KERN_INFO "%s: Transmit timeout !\n",
			card->devname);

		/* unbusy the interface */
		clear_bit(0,&dev->tbusy);
	}
#endif

   	if (ntohs(skb->protocol) != htons(PVC_PROT)){

		/* check the udp packet type */
		
		udp_type = udp_pkt_type(skb, card);

		if (udp_type == UDP_CPIPE_TYPE){
                        if(store_udp_mgmt_pkt(UDP_PKT_FRM_STACK, card, skb, dev,
                                bstrm_priv_area)){
	                	bstrm_int->interrupt_permission |=
					APP_INT_ON_TIMER;
			}
			start_net_queue(dev);
			return 0;
		}

		/* check to see if the source IP address is a broadcast or */
		/* multicast IP address */
                if(chk_bcast_mcast_addr(card, dev, skb)){
			++card->wandev.stats.tx_dropped;
			wan_dev_kfree_skb(skb,FREE_WRITE);
			start_net_queue(dev);
			return 0;
		}
        }

	/* Lock the 508 Card: SMP is supported */
      	if(card->hw.type != SDLA_S514){
		s508_lock(card,&smp_flags);
	} 

    	if(test_and_set_bit(SEND_CRIT, (void*)&card->wandev.critical)) {
	
		printk(KERN_INFO "%s: Critical in if_send: %lx\n",
					card->wandev.name,card->wandev.critical);
                ++card->wandev.stats.tx_dropped;
		start_net_queue(dev);
		goto if_send_exit_crit;
	}

	if(card->u.b.state != WAN_CONNECTED){
       		++card->wandev.stats.tx_dropped;
		start_net_queue(dev);
		
	}else if(!skb->protocol){
        	++card->wandev.stats.tx_errors;
		start_net_queue(dev);
		
	}else {
		void* data = skb->data;
		unsigned len = skb->len;
		unsigned char attr;

		/* If it's an API packet pull off the API
		 * header. Also check that the packet size
		 * is larger than the API header
	         */
		if (card->u.b.usedby == API){
			api_tx_hdr_t* api_tx_hdr;

			/* discard the frame if we are configured for */
			/* 'receive only' mode or if there is no data */
			if (len <= sizeof(api_tx_hdr_t)) {
				
				++card->wandev.stats.tx_dropped;
				start_net_queue(dev);
				goto if_send_exit_crit;
			}
				
			api_tx_hdr = (api_tx_hdr_t *)data;
			attr = api_tx_hdr->attr;
			data += sizeof(api_tx_hdr_t);
			len -= sizeof(api_tx_hdr_t);
		}

		if(bstrm_send(card, data, len)) {
			stop_net_queue(dev);
		}else{
			++card->wandev.stats.tx_packets;
#if defined(LINUX_2_1) || defined(LINUX_2_4)
                        card->wandev.stats.tx_bytes += len;
#endif
			
			start_net_queue(dev);
			
#ifdef LINUX_2_4
		 	dev->trans_start = jiffies;
#endif
		}	
	}

if_send_exit_crit:
	
	if (!(err=is_queue_stopped(dev))) {
		wan_dev_kfree_skb(skb, FREE_WRITE);
	}else{
		bstrm_priv_area->tick_counter = jiffies;
		bstrm_int->interrupt_permission |= APP_INT_ON_TX_BLOCK;
	}

	clear_bit(SEND_CRIT, (void*)&card->wandev.critical);
	if(card->hw.type != SDLA_S514){
		s508_unlock(card,&smp_flags);
	}
	
	return err;
}


/*============================================================================
 * Check to see if the packet to be transmitted contains a broadcast or
 * multicast source IP address.
 */

static int chk_bcast_mcast_addr(sdla_t *card, netdevice_t* dev,
				struct sk_buff *skb)
{
	u32 src_ip_addr;
        u32 broadcast_ip_addr = 0;
#if defined(LINUX_2_1) || defined(LINUX_2_4)
        struct in_device *in_dev;
#endif
        /* read the IP source address from the outgoing packet */
        src_ip_addr = *(u32 *)(skb->data + 12);

	/* read the IP broadcast address for the device */
#if defined(LINUX_2_1) || defined(LINUX_2_4)
        in_dev = dev->ip_ptr;
        if(in_dev != NULL) {
                struct in_ifaddr *ifa= in_dev->ifa_list;
                if(ifa != NULL)
                        broadcast_ip_addr = ifa->ifa_broadcast;
                else
                        return 0;
        }
#else
        broadcast_ip_addr = dev->pa_brdaddr;
#endif
 
        /* check if the IP Source Address is a Broadcast address */
        if((dev->flags & IFF_BROADCAST) && (src_ip_addr == broadcast_ip_addr)) {
                printk(KERN_INFO "%s: Broadcast Source Address silently discarded\n",
				card->devname);
                return 1;
        } 

        /* check if the IP Source Address is a Multicast address */
        if((ntohl(src_ip_addr) >= 0xE0000001) &&
		(ntohl(src_ip_addr) <= 0xFFFFFFFE)) {
                printk(KERN_INFO "%s: Multicast Source Address silently discarded\n",
				card->devname);
                return 1;
        }

        return 0;
}


/*============================================================================
 * Reply to UDP Management system.
 * Return length of reply.
 */
static int reply_udp( unsigned char *data, unsigned int mbox_len )
{

	unsigned short len, udp_length, temp, ip_length;
	unsigned long ip_temp;
	int even_bound = 0;
  	bstrm_udp_pkt_t *c_udp_pkt = (bstrm_udp_pkt_t *)data;
	 
	/* Set length of packet */
	len = sizeof(ip_pkt_t)+ 
	      sizeof(udp_pkt_t)+
	      sizeof(wp_mgmt_t)+
	      sizeof(cblock_t)+
	      sizeof(trace_info_t)+ 
	      mbox_len;

	/* fill in UDP reply */
	c_udp_pkt->wp_mgmt.request_reply = UDPMGMT_REPLY;
   
	/* fill in UDP length */
	udp_length = sizeof(udp_pkt_t)+ 
		     sizeof(wp_mgmt_t)+
		     sizeof(cblock_t)+
	             sizeof(trace_info_t)+
		     mbox_len; 

 	/* put it on an even boundary */
	if ( udp_length & 0x0001 ) {
		udp_length += 1;
		len += 1;
		even_bound = 1;
	}  

	temp = (udp_length<<8)|(udp_length>>8);
	c_udp_pkt->udp_pkt.udp_length = temp;
		 
	/* swap UDP ports */
	temp = c_udp_pkt->udp_pkt.udp_src_port;
	c_udp_pkt->udp_pkt.udp_src_port = 
			c_udp_pkt->udp_pkt.udp_dst_port; 
	c_udp_pkt->udp_pkt.udp_dst_port = temp;

	/* add UDP pseudo header */
	temp = 0x1100;
	*((unsigned short *)(c_udp_pkt->data+mbox_len+even_bound)) = temp;	
	temp = (udp_length<<8)|(udp_length>>8);
	*((unsigned short *)(c_udp_pkt->data+mbox_len+even_bound+2)) = temp;

		 
	/* calculate UDP checksum */
	c_udp_pkt->udp_pkt.udp_checksum = 0;
	c_udp_pkt->udp_pkt.udp_checksum = calc_checksum(&data[UDP_OFFSET],udp_length+UDP_OFFSET);

	/* fill in IP length */
	ip_length = len;
	temp = (ip_length<<8)|(ip_length>>8);
	c_udp_pkt->ip_pkt.total_length = temp;
  
	/* swap IP addresses */
	ip_temp = c_udp_pkt->ip_pkt.ip_src_address;
	c_udp_pkt->ip_pkt.ip_src_address = c_udp_pkt->ip_pkt.ip_dst_address;
	c_udp_pkt->ip_pkt.ip_dst_address = ip_temp;

	/* fill in IP checksum */
	c_udp_pkt->ip_pkt.hdr_checksum = 0;
	c_udp_pkt->ip_pkt.hdr_checksum = calc_checksum(data,sizeof(ip_pkt_t));

	return len;

} /* reply_udp */

unsigned short calc_checksum (char *data, int len)
{
	unsigned short temp; 
	unsigned long sum=0;
	int i;

	for( i = 0; i <len; i+=2 ) {
		memcpy(&temp,&data[i],2);
		sum += (unsigned long)temp;
	}

	while (sum >> 16 ) {
		sum = (sum & 0xffffUL) + (sum >> 16);
	}

	temp = (unsigned short)sum;
	temp = ~temp;

	if( temp == 0 ) 
		temp = 0xffff;

	return temp;	
}


/*============================================================================
 * Get ethernet-style interface statistics.
 * Return a pointer to struct enet_statistics.
 */
#if defined(LINUX_2_1) || defined(LINUX_2_4)
static struct net_device_stats* if_stats (netdevice_t* dev)
{
	sdla_t *my_card;
	bitstrm_private_area_t* bstrm_priv_area;

	if ((bstrm_priv_area=dev->priv) == NULL)
		return NULL;

	my_card = bstrm_priv_area->card;
	return &my_card->wandev.stats; 
}
#else
static struct enet_statistics* if_stats (netdevice_t* dev)
{
        sdla_t *my_card;
        bitstrm_private_area_t* bstrm_priv_area = dev->priv;

	if ((bstrm_priv_area=dev->priv) == NULL)
		return NULL;

        my_card = bstrm_priv_area->card;
        return &my_card->wandev.stats;
}
#endif

/****** Cisco HDLC Firmware Interface Functions *******************************/

/*============================================================================
 * Read firmware code version.
 *	Put code version as ASCII string in str. 
 */
static int bstrm_read_version (sdla_t* card, char* str)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	int len;
	char err;
	mb->buffer_length = 0;
	mb->command = READ_BSTRM_CODE_VERSION;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;

	if(err != OK) {
		bstrm_error(card,err,mb);
	}
	else if (str) {  /* is not null */
		len = mb->buffer_length;
		memcpy(str, mb->data, len);
		str[len] = '\0';
	}
	return (err);
}

/*-----------------------------------------------------------------------------
 *  Configure BSTRM firmware.
 */
static int bstrm_configure (sdla_t* card, void* data)
{
	int err;
	BSTRM_MAILBOX_STRUCT *mailbox = card->mbox;
	int data_length = sizeof(BSTRM_CONFIGURATION_STRUCT);
	
	mailbox->buffer_length = data_length;  
	memcpy(mailbox->data, data, data_length);
	mailbox->command = SET_BSTRM_CONFIGURATION;
	err = sdla_exec(mailbox) ? mailbox->return_code : CMD_TIMEOUT;
	
	if (err != OK) bstrm_error (card, err, mailbox);
                           
	return err;
}


/*============================================================================
 * Set interrupt mode -- HDLC Version.
 */

static int bstrm_set_intr_mode (sdla_t* card, unsigned mode)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	BSTRM_INT_TRIGGERS_STRUCT* int_data =
		 (BSTRM_INT_TRIGGERS_STRUCT *)mb->data;
	int err;

	int_data->BSTRM_interrupt_triggers 	= mode;
	int_data->IRQ				= card->hw.irq;
	int_data->interrupt_timer               = 1;
   
	mb->buffer_length = sizeof(BSTRM_INT_TRIGGERS_STRUCT);
	mb->command = SET_BSTRM_INTERRUPT_TRIGGERS;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
	if (err != OK)
		bstrm_error (card, err, mb);
	return err;
}


/*===========================================================
 * bstrm_disable_comm_shutdown
 *
 * Shutdown() disables the communications. We must
 * have a sparate functions, because we must not
 * call bstrm_error() hander since the private
 * area has already been replaced */

static int bstrm_disable_comm_shutdown (sdla_t *card)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	BSTRM_INT_TRIGGERS_STRUCT* int_data =
		 (BSTRM_INT_TRIGGERS_STRUCT *)mb->data;
	int err;

	/* Disable Interrutps */
	int_data->BSTRM_interrupt_triggers 	= 0;
	int_data->IRQ				= card->hw.irq;
	int_data->interrupt_timer               = 1;
   
	mb->buffer_length = sizeof(BSTRM_INT_TRIGGERS_STRUCT);
	mb->command = SET_BSTRM_INTERRUPT_TRIGGERS;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;

	/* Disable Communications */

	mb->command = DISABLE_BSTRM_COMMUNICATIONS;
	
	mb->buffer_length = 0;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
	
	card->u.b.comm_enabled = 0;
	
	return 0;
}

/*============================================================================
 * Enable communications.
 */

static int bstrm_comm_enable (sdla_t* card)
{
	int err;
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;

	mb->buffer_length = 0;
	mb->command = ENABLE_BSTRM_COMMUNICATIONS;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
	if (err != OK)
		bstrm_error(card, err, mb);
	else
		card->u.b.comm_enabled = 1;
	
	return err;
}

/*============================================================================
 * Read communication error statistics.
 */
static int bstrm_read_comm_err_stats (sdla_t* card)
{
        int err;
        BSTRM_MAILBOX_STRUCT* mb = card->mbox;

        mb->buffer_length = 0;
        mb->command = READ_COMMS_ERROR_STATS;
        err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
        if (err != OK)
                bstrm_error(card,err,mb);
        return err;
}


/*============================================================================
 * Read BSTRM operational statistics.
 */
static int bstrm_read_op_stats (sdla_t* card)
{
        int err;
        BSTRM_MAILBOX_STRUCT* mb = card->mbox;

        mb->buffer_length = 0;
        mb->command = READ_BSTRM_OPERATIONAL_STATS;
        err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
        if (err != OK)
                bstrm_error(card,err,mb);
        return err;
}


/*============================================================================
 * Update communications error and general packet statistics.
 */
static int update_comms_stats(sdla_t* card,
	bitstrm_private_area_t* bstrm_priv_area)
{
        BSTRM_MAILBOX_STRUCT* mb = card->mbox;
  	COMMS_ERROR_STATS_STRUCT* err_stats;
        BSTRM_OPERATIONAL_STATS_STRUCT *op_stats;

	if(bstrm_priv_area->update_comms_stats == 3) {
		/* 1. On the first timer interrupt, update T1/E1 alarms 
		 * and PMON counters (only for T1/E1 card) (TE1) 
		 */
		/* TE1 Update T1/E1 alarms */
		if (IS_TE1(card->wandev.te_cfg)) {	
			card->wandev.cur_te_alarm = sdla_te_alarm(card, 0); 
			/* TE1 Update T1/E1 perfomance counters */
			sdla_te_pmon(card);
		}else if (IS_56K(card->wandev.te_cfg)) {
			/* 56K Update CSU/DSU alarms */
			card->wandev.k56_alarm = sdla_56k_alarm(card, 1);
		}
	 } else { 
		/* 2. On the second timer interrupt, read the comms error 
	 	 * statistics 
	 	 */
		if(bstrm_priv_area->update_comms_stats == 2) {
			if(bstrm_read_comm_err_stats(card))
				return 1;
			err_stats = (COMMS_ERROR_STATS_STRUCT *)mb->data;
			card->wandev.stats.rx_over_errors = 
				err_stats->Rx_overrun_err_count;
			card->wandev.stats.rx_fifo_errors = 
				err_stats->Rx_dis_pri_bfrs_full_count; 
			card->wandev.stats.rx_missed_errors =
				card->wandev.stats.rx_fifo_errors;
		} else {

        		/* on the third timer interrupt, read the operational 
			 * statistics 
		 	 */
        		if(bstrm_read_op_stats(card))
                		return 1;
			op_stats = (BSTRM_OPERATIONAL_STATS_STRUCT *)mb->data;
		}
	}

	return 0;
}

/*============================================================================
 * Send packet.
 *	Return:	0 - o.k.
 *		1 - no transmit buffers available
 */
static int bstrm_send (sdla_t* card, void* data, unsigned len)
{
	BSTRM_DATA_TX_STATUS_EL_STRUCT *txbuf = card->u.b.txbuf;

	if (txbuf->opp_flag)
		return 1;
	
	sdla_poke(&card->hw, txbuf->ptr_data_bfr, data, len);

	txbuf->block_length = len;
	txbuf->opp_flag = 1;		/* start transmission */
	
	/* Update transmit buffer control fields */
	card->u.b.txbuf = ++txbuf;
	
	if ((void*)txbuf > card->u.b.txbuf_last)
		card->u.b.txbuf = card->u.b.txbuf_base;
	
	return 0;
}

/*============================================================================
 * TE1
 * Read value from PMC register.
 */
static unsigned char read_front_end_reg (void* card1, unsigned short reg)
{
	sdla_t* card = (sdla_t*)card1;
        BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	char* data = mb->data;
        int err;

        mb->buffer_length = 1;
        mb->command = READ_PMC_REGISTER;
	data[0] = reg;
        err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
        if (err != OK)
                bstrm_error(card,err,mb);
        return data[0];
}

/*============================================================================
 * TE1 
 * Write value to PMC register.
 */
static unsigned char write_front_end_reg (void* card1, unsigned short reg, unsigned char value)
{
	sdla_t* card = (sdla_t*)card1;
        BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	char* data = mb->data;
        int err, i = 0;

        mb->buffer_length = 2;
        mb->command = SET_PMC_REGISTER;
	data[0] = reg;
	data[1] = value;
        err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
        if (err != OK)
                bstrm_error(card,err,mb);
	for(i=0;i<0xFFFF;i++);
	for(i=0;i<0xFFFF;i++);
	for(i=0;i<0xFFFF;i++);
	for(i=0;i<0xFFFF;i++);
        return err;
}
/****** Firmware Error Handler **********************************************/

/*============================================================================
 * Firmware error handler.
 *	This routine is called whenever firmware command returns non-zero
 *	return code.
 *
 * Return zero if previous command has to be cancelled.
 */
static int bstrm_error (sdla_t *card, int err, BSTRM_MAILBOX_STRUCT *mb)
{
	unsigned cmd = mb->command;

	switch (err) {

	case CMD_TIMEOUT:
		printk(KERN_INFO "%s: command 0x%02X timed out!\n",
			card->devname, cmd);
		break;

	case S514_BOTH_PORTS_SAME_CLK_MODE:
		if(cmd == SET_BSTRM_CONFIGURATION) {
			printk(KERN_INFO
			 "%s: Configure both ports for the same clock source\n",
				card->devname);
			break;
		}

	default:
		printk(KERN_INFO "%s: command 0x%02X returned 0x%02X!\n",
			card->devname, cmd, err);
	}

	return 0;
}

#if defined(LINUX_2_1) || defined(LINUX_2_4)
/********** Bottom Half Handlers ********************************************/

/* NOTE: There is no API, BH support for Kernels lower than 2.2.X.
 *       DO NOT INSERT ANY CODE HERE, NOTICE THE 
 *       PREPROCESSOR STATEMENT ABOVE, UNLESS YOU KNOW WHAT YOU ARE
 *       DOING */

static void bstrm_bh (netdevice_t * dev)
{
	bitstrm_private_area_t* chan = dev->priv;
	sdla_t *card = chan->card;
	struct sk_buff *skb;

	if (!chan->bh_head || atomic_read(&chan->bh_buff_used) == 0){
		clear_bit(0, &chan->tq_working);
		return;
	}

	while (atomic_read(&chan->bh_buff_used)){

		skb  = ((bh_data_t *)&chan->bh_head[chan->bh_read])->skb;

		if (skb != NULL){

			if (chan->common.sk == NULL || chan->common.func == NULL){
				++card->wandev.stats.rx_dropped;
				wan_dev_kfree_skb(skb, FREE_READ);
				bstrm_bh_cleanup(dev);
				continue;
			}

			if (chan->common.func(skb,dev,chan->common.sk) != 0){
				/* Sock full cannot send, queue us for another
                                 * try */
				atomic_set(&chan->common.receive_block,1);
				return;
			}else{
				bstrm_bh_cleanup(dev);
			}
		}else{
			bstrm_bh_cleanup(dev);
		}
	}	
	clear_bit(0, &chan->tq_working);

	return;
}

static int bstrm_bh_cleanup (netdevice_t *dev)
{
	bitstrm_private_area_t* chan = dev->priv;

	((bh_data_t *)&chan->bh_head[chan->bh_read])->skb = NULL;

	if (chan->bh_read == MAX_BH_BUFF){
		chan->bh_read=0;
	}else{
		++chan->bh_read;	
	}

	atomic_dec(&chan->bh_buff_used);
	return 0;
}



static int bh_enqueue (netdevice_t *dev, struct sk_buff *skb)
{
	/* Check for full */
	bitstrm_private_area_t* chan = dev->priv;
	sdla_t *card = chan->card;

	if (!chan->bh_head || atomic_read(&chan->bh_buff_used) == (MAX_BH_BUFF+1)){
		++card->wandev.stats.rx_dropped;
		wan_dev_kfree_skb(skb, FREE_READ);
		return 1; 
	}

	((bh_data_t *)&chan->bh_head[chan->bh_write])->skb = skb;

	if (chan->bh_write == MAX_BH_BUFF){
		chan->bh_write=0;
	}else{
		++chan->bh_write;
	}

	atomic_inc(&chan->bh_buff_used);

	return 0;
}

/* END OF API BH Support */


static void wakeup_sk_bh (netdevice_t *dev)
{

	bitstrm_private_area_t *chan = dev->priv;

	/* If the sock is in the process of unlinking the
	 * driver from the socket, we must get out. 
	 * This never happends but is a sanity check. */
	if (test_bit(0,&chan->common.common_critical)){
		return;
	}

	if (!chan->common.sk || !chan->common.func){
		return;
	}

	chan->common.func(NULL,dev,chan->common.sk);
	return;
}


#endif

/****** Interrupt Handlers **************************************************/

/*============================================================================
 * Cisco HDLC interrupt service routine.
 */
static void wpbit_isr (sdla_t* card)
{
	netdevice_t* dev;
	SHARED_MEMORY_INFO_STRUCT* flags = NULL;
	int i;
	sdla_t *my_card;


	/* Check for which port the interrupt has been generated
	 * Since Secondary Port is piggybacking on the Primary
         * the check must be done here. 
	 */

	flags = card->u.b.flags;
	if (!flags->interrupt_info_struct.interrupt_type){
		/* Check for a second port (piggybacking) */
		if ((my_card = card->next)){
			flags = my_card->u.b.flags;
			if (flags->interrupt_info_struct.interrupt_type){
				card = my_card;
				card->isr(card);
				return;
			}
		}
	}

	flags = card->u.b.flags;
	card->in_isr = 1;
	dev = card->wandev.dev;
	
	/* If we get an interrupt with no network device, stop the interrupts
	 * and issue an error */
	if (!dev && flags->interrupt_info_struct.interrupt_type != 
	    	COMMAND_COMPLETE_APP_INT_PEND){

		goto isr_done;
	}
	
	/* if critical due to peripheral operations
	 * ie. update() or getstats() then reset the interrupt and
	 * wait for the board to retrigger.
	 */
	if(test_bit(PERI_CRIT, (void*)&card->wandev.critical)) {
		printk(KERN_INFO "ISR CRIT TO PERI\n");
		goto isr_done;
	}

	/* On a 508 Card, if critical due to if_send 
         * Major Error !!! */
	if(card->hw.type != SDLA_S514) {
		if(test_bit(SEND_CRIT, (void*)&card->wandev.critical)) {
			printk(KERN_INFO "%s: Critical while in ISR: %lx\n",
				card->devname, card->wandev.critical);
			card->in_isr = 0;
			flags->interrupt_info_struct.interrupt_type = 0;
			return;
		}
	}

	switch(flags->interrupt_info_struct.interrupt_type) {

	case RX_APP_INT_PEND:	/* 0x01: receive interrupt */
		rx_intr(card);
		break;

	case TX_APP_INT_PEND:	/* 0x02: transmit interrupt */
		flags->interrupt_info_struct.interrupt_permission &=
			 ~APP_INT_ON_TX_BLOCK;

#if defined(LINUX_2_1) || defined(LINUX_2_4)

		if (dev && is_queue_stopped(dev)){
			if (card->u.b.usedby == API){
				start_net_queue(dev);
				wakeup_sk_bh(dev);
			}else{
				wake_net_dev(dev);
			}
		}
#else
		wake_net_dev(dev);
#endif
		break;

	case COMMAND_COMPLETE_APP_INT_PEND:/* 0x04: cmd cplt */
		++ Intr_test_counter;
		break;

	case BSTRM_EXCEP_COND_APP_INT_PEND:	/* 0x20 */
		process_bstrm_exception(card);
		break;

	case GLOBAL_EXCEP_COND_APP_INT_PEND:
		process_global_exception(card);
		break;

	case TIMER_APP_INT_PEND:
		timer_intr(card);
		break;

	default:
		printk(KERN_INFO "%s: spurious interrupt 0x%02X!\n", 
			card->devname,
			flags->interrupt_info_struct.interrupt_type);
		printk(KERN_INFO "Code name: ");
		for(i = 0; i < 4; i ++)
			printk(KERN_INFO "%c",
				flags->global_info_struct.code_name[i]); 
		printk(KERN_INFO "\n");
		printk(KERN_INFO "Code version: ");
	 	for(i = 0; i < 4; i ++)
			printk(KERN_INFO "%c", 
				flags->global_info_struct.code_version[i]); 
		printk(KERN_INFO "\n");	
		break;
	}

isr_done:

	card->in_isr = 0;
	flags->interrupt_info_struct.interrupt_type = 0;
	return;
}

/*============================================================================
 * Receive interrupt handler.
 */
static void rx_intr (sdla_t* card)
{
	netdevice_t *dev;
	bitstrm_private_area_t *bstrm_priv_area;
	SHARED_MEMORY_INFO_STRUCT *flags = card->u.b.flags;
	BSTRM_DATA_RX_STATUS_EL_STRUCT *rxbuf = card->u.b.rxmb;
	struct sk_buff *skb;
	unsigned len;
	unsigned addr = rxbuf->ptr_data_bfr;
	void *buf;
	int i,udp_type;

	if (rxbuf->opp_flag != 0x01) {
		printk(KERN_INFO 
			"%s: corrupted Rx buffer @ 0x%X, flag = 0x%02X!\n", 
			card->devname, (unsigned)rxbuf, rxbuf->opp_flag);
                printk(KERN_INFO "Code name: ");
                for(i = 0; i < 4; i ++)
                        printk(KERN_INFO "%c",
                                flags->global_info_struct.code_name[i]);
		printk(KERN_INFO "\n");
                printk(KERN_INFO "Code version: ");
                for(i = 0; i < 4; i ++)
                        printk(KERN_INFO "%c",
                                flags->global_info_struct.code_version[i]);
                printk(KERN_INFO "\n");


		/* Bug Fix: Mar 6 2000
                 * If we get a corrupted mailbox, it measn that driver 
                 * is out of sync with the firmware. There is no recovery.
                 * If we don't turn off all interrupts for this card
                 * the machine will crash. 
                 */
		printk(KERN_INFO "%s: Critical router failure ...!!!\n", card->devname);
		printk(KERN_INFO "Please contact Sangoma Technologies !\n");
		bstrm_set_intr_mode(card,0);	
		return;
	}

	len  = rxbuf->block_length;

	dev = card->wandev.dev;

	if (!dev){
		goto rx_exit;
	}

	if (!is_dev_running(dev))
		goto rx_exit;

	bstrm_priv_area = dev->priv;
	
	/* Allocate socket buffer */
	skb = dev_alloc_skb(len);

	if (skb == NULL) {
		printk(KERN_INFO "%s: no socket buffers available!\n",
					card->devname);
		++card->wandev.stats.rx_dropped;
		goto rx_exit;
	}

	buf = skb_put(skb, len);
	sdla_peek(&card->hw, addr, buf, len);

	skb->protocol = htons(ETH_P_IP);

	card->wandev.stats.rx_packets ++;
#if defined(LINUX_2_1) || defined(LINUX_2_4)
	card->wandev.stats.rx_bytes += skb->len;
#endif
	udp_type = udp_pkt_type( skb, card );

	if(udp_type == UDP_CPIPE_TYPE) {
		if(store_udp_mgmt_pkt(UDP_PKT_FRM_NETWORK,
   				      card, skb, dev, bstrm_priv_area)) {
     		        flags->interrupt_info_struct.
						interrupt_permission |= 
							APP_INT_ON_TIMER; 
		}
	}else{

		api_rx_hdr_t* api_rx_hdr;
		skb_push(skb, sizeof(api_rx_hdr_t));
		api_rx_hdr = (api_rx_hdr_t*)&skb->data[0x00];
		api_rx_hdr->error_flag = rxbuf->error_flag;
		api_rx_hdr->time_stamp = rxbuf->time_stamp;

		skb->protocol = htons(PVC_PROT);
		skb->mac.raw  = skb->data;
		skb->dev      = dev;
		skb->pkt_type = WAN_PACKET_DATA;

		bh_enqueue(dev, skb);

		if (!test_and_set_bit(0,&bstrm_priv_area->tq_working)){
			if (wanpipe_queue_tq(&bstrm_priv_area->common.wanpipe_task) != 0){
				clear_bit(0,&bstrm_priv_area->tq_working);
			}
			if (wanpipe_mark_bh() != 0){
				clear_bit(0,&bstrm_priv_area->tq_working);
			}
		}
	}
rx_exit:
	/* Release buffer element and calculate a pointer to the next one */
	rxbuf->opp_flag = 0x00;
	card->u.b.rxmb = ++ rxbuf;
	if((void*)rxbuf > card->u.b.rxbuf_last){
		card->u.b.rxmb = card->u.b.rxbuf_base;
	}
}

/*============================================================================
 * Timer interrupt handler.
 * The timer interrupt is used for two purposes:
 *    1) Processing udp calls from 'cpipemon'.
 *    2) Reading board-level statistics for updating the proc file system.
 */
void timer_intr(sdla_t *card)
{
        netdevice_t* dev;
        bitstrm_private_area_t* bstrm_priv_area = NULL;
        SHARED_MEMORY_INFO_STRUCT* flags = NULL;

        if ((dev = card->wandev.dev)==NULL){
		flags = card->u.b.flags;
                flags->interrupt_info_struct.interrupt_permission &=
                        ~APP_INT_ON_TIMER;
		return;
	}
	
        bstrm_priv_area = dev->priv;

	if (bstrm_priv_area->timer_int_enabled & TMR_INT_ENABLED_CONFIG) {
		config_bstrm(card);
		bstrm_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_CONFIG;
	}

	/* process a udp call if pending */
       	if(bstrm_priv_area->timer_int_enabled & TMR_INT_ENABLED_UDP) {
               	process_udp_mgmt_pkt(card, dev,
                       bstrm_priv_area);
		bstrm_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_UDP;
        }

	/* read the communications statistics if required */
	if(bstrm_priv_area->timer_int_enabled & TMR_INT_ENABLED_UPDATE) {
		update_comms_stats(card, bstrm_priv_area);
                if(!(-- bstrm_priv_area->update_comms_stats)) {
			bstrm_priv_area->timer_int_enabled &= 
				~TMR_INT_ENABLED_UPDATE;
		}
        }

	/* only disable the timer interrupt if there are no udp or statistic */
	/* updates pending */
        if(!bstrm_priv_area->timer_int_enabled) {
                flags = card->u.b.flags;
                flags->interrupt_info_struct.interrupt_permission &=
                        ~APP_INT_ON_TIMER;
        }
}

/*------------------------------------------------------------------------------
  Miscellaneous Functions
	- set_bstrm_config() used to set configuration options on the board
------------------------------------------------------------------------------*/

static int set_bstrm_config(sdla_t* card)
{
	BSTRM_CONFIGURATION_STRUCT cfg;

	memset(&cfg, 0, sizeof(BSTRM_CONFIGURATION_STRUCT));

	if(card->wandev.clocking){
		cfg.baud_rate = card->wandev.bps;
	}
	
	cfg.line_config_options = (card->wandev.interface == WANOPT_RS232) ?
		INTERFACE_LEVEL_RS232 : INTERFACE_LEVEL_V35;

	cfg.modem_config_options	= 0;
	cfg.modem_status_timer		= 500;

	cfg.API_options = 0;

	cfg.SYNC_options = card->u.b.sync_options;
	cfg.Rx_sync_char = card->u.b.rx_sync_char;
	cfg.monosync_Tx_time_fill_char = card->u.b.monosync_tx_time_fill_char;
	
	cfg.buffer_options = 0;
	
	cfg.max_length_Tx_data_block =  card->u.b.max_length_tx_data_block;
	cfg.Rx_complete_length = card->u.b.rx_complete_length;
	cfg.Rx_complete_timer = card->u.b.rx_complete_timer;

	cfg.statistics_options = TX_DATA_BYTE_COUNT_STAT | 
				 RX_DATA_BYTE_COUNT_STAT |
				 TX_THROUGHPUT_STAT |
				 RX_THROUGHPUT_STAT;
	
	return bstrm_configure(card, &cfg);
}


/*============================================================================
 * Process global exception condition
 */
static int process_global_exception(sdla_t *card)
{
	BSTRM_MAILBOX_STRUCT* mbox = card->mbox;
	int err;

	mbox->buffer_length = 0;
	mbox->command = READ_GLOBAL_EXCEPTION_CONDITION;
	err = sdla_exec(mbox) ? mbox->return_code : CMD_TIMEOUT;

	if(err != CMD_TIMEOUT ){
	
		switch(mbox->return_code) {
         
	      	case EXCEP_MODEM_STATUS_CHANGE:

			printk(KERN_INFO "%s: Modem status change\n",
				card->devname);

			switch(mbox->data[0] & (DCD_HIGH | CTS_HIGH)) {
				case (DCD_HIGH):
					printk(KERN_INFO "%s: DCD high, CTS low\n",card->devname);
					break;
				case (CTS_HIGH):
                                        printk(KERN_INFO "%s: DCD low, CTS high\n",card->devname); 
					break;
                                case ((DCD_HIGH | CTS_HIGH)):
                                        printk(KERN_INFO "%s: DCD high, CTS high\n",card->devname);
                                        break;
				default:
                                        printk(KERN_INFO "%s: DCD low, CTS low\n",card->devname);
                                        break;
			}
			break;

                default:
                        printk(KERN_INFO "%s: Global exception %x\n",
				card->devname, mbox->return_code);
                        break;
                }
	}
	return 0;
}


/*============================================================================
 * Process chdlc exception condition
 */
static int process_bstrm_exception(sdla_t *card)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	int err;

	mb->buffer_length = 0;
	mb->command = READ_BSTRM_EXCEPTION_CONDITION;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
	if(err != CMD_TIMEOUT) {
	
		switch (err) {

		case NO_BSTRM_EXCEP_COND_TO_REPORT:
			printk(KERN_INFO "%s: No exceptions reported.\n",
						card->devname);
			break;
		}

	}
	return 0;
}


/*=============================================================================
 * Store a UDP management packet for later processing.
 */

static int store_udp_mgmt_pkt(char udp_pkt_src, sdla_t* card,
                                struct sk_buff *skb, netdevice_t* dev,
                                bitstrm_private_area_t* bstrm_priv_area )
{
	int udp_pkt_stored = 0;

	if(!bstrm_priv_area->udp_pkt_lgth &&
	  (skb->len <= MAX_LGTH_UDP_MGNT_PKT)) {
        	bstrm_priv_area->udp_pkt_lgth = skb->len;
		bstrm_priv_area->udp_pkt_src = udp_pkt_src;
       		memcpy(bstrm_priv_area->udp_pkt_data, skb->data, skb->len);
		bstrm_priv_area->timer_int_enabled = TMR_INT_ENABLED_UDP;
		udp_pkt_stored = 1;
	}

	if(udp_pkt_src == UDP_PKT_FRM_STACK){
		wan_dev_kfree_skb(skb, FREE_WRITE);
	}else{
                wan_dev_kfree_skb(skb, FREE_READ);
	}
		
	return(udp_pkt_stored);
}


/*=============================================================================
 * Process UDP management packet.
 */

static int process_udp_mgmt_pkt(sdla_t* card, netdevice_t* dev,
				bitstrm_private_area_t* bstrm_priv_area ) 
{
	unsigned char *buf;
	unsigned int len;
	struct sk_buff *new_skb;
	int udp_mgmt_req_valid = 1;
	BSTRM_MAILBOX_STRUCT *mb = card->mbox;
	bstrm_udp_pkt_t *bstrm_udp_pkt;
	int err;

	bstrm_udp_pkt = (bstrm_udp_pkt_t *) bstrm_priv_area->udp_pkt_data;

	if(bstrm_priv_area->udp_pkt_src == UDP_PKT_FRM_NETWORK){

		/* Only these commands are support for remote debugging.
		 * All others are not */
		switch(bstrm_udp_pkt->cblock.command) {

			case READ_GLOBAL_STATISTICS:
			case READ_MODEM_STATUS:  
			case READ_BSTRM_STATUS:
			case READ_COMMS_ERROR_STATS:
			case READ_BSTRM_OPERATIONAL_STATS:

			/* These two commands are executed for
			 * each request */
			case READ_BSTRM_CONFIGURATION:
			case READ_BSTRM_CODE_VERSION:
				udp_mgmt_req_valid = 1;
				break;
			default:
				udp_mgmt_req_valid = 0;
				break;
		} 
	}
	
  	if(!udp_mgmt_req_valid) {

		/* set length to 0 */
		bstrm_udp_pkt->cblock.buffer_length = 0;

    		/* set return code */
		bstrm_udp_pkt->cblock.return_code = 0xCD;

		if (net_ratelimit()){	
			printk(KERN_INFO 
			"%s: Warning, Illegal UDP command attempted from network: %x\n",
			card->devname,bstrm_udp_pkt->cblock.command);
		}

   	} else {
			/* it's a board command */
			mb->command = bstrm_udp_pkt->cblock.command;
			mb->buffer_length = bstrm_udp_pkt->cblock.buffer_length;
			if (mb->buffer_length) {
				memcpy(&mb->data, (unsigned char *) bstrm_udp_pkt->
							data, mb->buffer_length);
	      		} 
			/* run the command on the board */
			err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
			if (err == OK) {
				/* copy the result back to our buffer */
				memcpy(&bstrm_udp_pkt->cblock, mb, sizeof(cblock_t)); 
				
				if (mb->buffer_length) {
					memcpy(&bstrm_udp_pkt->data, &mb->data, 
									mb->buffer_length); 
				}
			}

     	} /* end of else */

     	/* Fill UDP TTL */
	bstrm_udp_pkt->ip_pkt.ttl = card->wandev.ttl; 

     	len = reply_udp(bstrm_priv_area->udp_pkt_data, mb->buffer_length);
	

     	if(bstrm_priv_area->udp_pkt_src == UDP_PKT_FRM_NETWORK){

		/* Must check if we interrupted if_send() routine. The
		 * tx buffers might be used. If so drop the packet */
	   	if (!test_bit(SEND_CRIT,&card->wandev.critical)) {
		
			if(!bstrm_send(card, bstrm_priv_area->udp_pkt_data, len)) {
				++ card->wandev.stats.tx_packets;
#if defined(LINUX_2_1) || defined(LINUX_2_4)
				card->wandev.stats.tx_bytes += len;
#endif
			}
		}
	} else {	
	
		/* Pass it up the stack
    		   Allocate socket buffer */
		if ((new_skb = dev_alloc_skb(len)) != NULL) {
			/* copy data into new_skb */

 	    		buf = skb_put(new_skb, len);
  	    		memcpy(buf, bstrm_priv_area->udp_pkt_data, len);

            		/* Decapsulate pkt and pass it up the protocol stack */
	    		new_skb->protocol = htons(ETH_P_IP);
            		new_skb->dev = dev;
	    		new_skb->mac.raw  = new_skb->data;
	
			netif_rx(new_skb);
		} else {
	    	
			printk(KERN_INFO "%s: no socket buffers available!\n",
					card->devname);
  		}
    	}
 
	bstrm_priv_area->udp_pkt_lgth = 0;
 	
	return 0;
}

/*============================================================================
 * Initialize Receive and Transmit Buffers.
 */

static void init_bstrm_tx_rx_buff( sdla_t* card)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	BSTRM_TX_STATUS_EL_CFG_STRUCT *tx_config;
	BSTRM_RX_STATUS_EL_CFG_STRUCT *rx_config;
	char err;
	
	mb->buffer_length = 0;
	mb->command = READ_BSTRM_CONFIGURATION;
	err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;

	if(err != OK) {
		if (card->wandev.dev){
			bstrm_error(card,err,mb);
		}
		return;
	}

	if(card->hw.type == SDLA_S514) {
		tx_config = (BSTRM_TX_STATUS_EL_CFG_STRUCT *)(card->hw.dpmbase +
                (((BSTRM_CONFIGURATION_STRUCT *)mb->data)->
                            ptr_Tx_stat_el_cfg_struct));
        	rx_config = (BSTRM_RX_STATUS_EL_CFG_STRUCT *)(card->hw.dpmbase +
                (((BSTRM_CONFIGURATION_STRUCT *)mb->data)->
                            ptr_Rx_stat_el_cfg_struct));

       		/* Setup Head and Tails for buffers */
        	card->u.b.txbuf_base = (void *)(card->hw.dpmbase +
                tx_config->base_addr_Tx_status_elements);
        	card->u.b.txbuf_last = 
		(BSTRM_DATA_TX_STATUS_EL_STRUCT *)  
                card->u.b.txbuf_base +
		(tx_config->number_Tx_status_elements - 1);

        	card->u.b.rxbuf_base = (void *)(card->hw.dpmbase +
                rx_config->base_addr_Rx_status_elements);
        	card->u.b.rxbuf_last =
		(BSTRM_DATA_RX_STATUS_EL_STRUCT *)
                card->u.b.rxbuf_base +
		(rx_config->number_Rx_status_elements - 1);

 		/* Set up next pointer to be used */
        	card->u.b.txbuf = (void *)(card->hw.dpmbase +
                tx_config->next_Tx_status_element_to_use);
        	card->u.b.rxmb = (void *)(card->hw.dpmbase +
                rx_config->next_Rx_status_element_to_use);
	}
        else {
                tx_config = (BSTRM_TX_STATUS_EL_CFG_STRUCT *)(card->hw.dpmbase +
			(((BSTRM_CONFIGURATION_STRUCT *)mb->data)->
			ptr_Tx_stat_el_cfg_struct % SDLA_WINDOWSIZE));

                rx_config = (BSTRM_RX_STATUS_EL_CFG_STRUCT *)(card->hw.dpmbase +
			(((BSTRM_CONFIGURATION_STRUCT *)mb->data)->
			ptr_Rx_stat_el_cfg_struct % SDLA_WINDOWSIZE));

                /* Setup Head and Tails for buffers */
                card->u.b.txbuf_base = (void *)(card->hw.dpmbase +
		(tx_config->base_addr_Tx_status_elements % SDLA_WINDOWSIZE));
                card->u.b.txbuf_last =
		(BSTRM_DATA_TX_STATUS_EL_STRUCT *)card->u.b.txbuf_base
		+ (tx_config->number_Tx_status_elements - 1);
                card->u.b.rxbuf_base = (void *)(card->hw.dpmbase +
		(rx_config->base_addr_Rx_status_elements % SDLA_WINDOWSIZE));
                card->u.b.rxbuf_last = 
		(BSTRM_DATA_RX_STATUS_EL_STRUCT *)card->u.b.rxbuf_base
		+ (rx_config->number_Rx_status_elements - 1);

                 /* Set up next pointer to be used */
                card->u.b.txbuf = (void *)(card->hw.dpmbase +
		(tx_config->next_Tx_status_element_to_use % SDLA_WINDOWSIZE));
                card->u.b.rxmb = (void *)(card->hw.dpmbase +
		(rx_config->next_Rx_status_element_to_use % SDLA_WINDOWSIZE));
        }

        /* Setup Actual Buffer Start and end addresses */
        card->u.b.rx_base = rx_config->base_addr_Rx_status_elements;
}

/*=============================================================================
 * Perform Interrupt Test by running READ_BSTRM_CODE_VERSION command MAX_INTR
 * _TEST_COUNTER times.
 */
static int intr_test( sdla_t* card)
{
	BSTRM_MAILBOX_STRUCT* mb = card->mbox;
	int err,i;

	Intr_test_counter = 0;
	
	err = bstrm_set_intr_mode(card, APP_INT_ON_COMMAND_COMPLETE);

	if (err == CMD_OK) { 
		for (i = 0; i < MAX_INTR_TEST_COUNTER; i ++) {	
			mb->buffer_length  = 0;
			mb->command = READ_BSTRM_CODE_VERSION;
			err = sdla_exec(mb) ? mb->return_code : CMD_TIMEOUT;
			if (err != CMD_OK) 
				bstrm_error(card, err, mb);
		}
	}
	else {
		return err;
	}

	err = bstrm_set_intr_mode(card, 0);

	if (err != CMD_OK)
		return err;

	return 0;
}

/*==============================================================================
 * Determine what type of UDP call it is. CPIPEAB ?
 */
static int udp_pkt_type(struct sk_buff *skb, sdla_t* card)
{
	 bstrm_udp_pkt_t *bstrm_udp_pkt = (bstrm_udp_pkt_t *)skb->data;

#ifdef _WAN_UDP_DEBUG
		printk(KERN_INFO "SIG %s = %s\n\
				  UPP %x = %x\n\
				  PRT %x = %x\n\
				  REQ %i = %i\n\
				  36 th = %x 37th = %x\n",
				  bstrm_udp_pkt->wp_mgmt.signature,
				  UDPMGMT_SIGNATURE,
				  bstrm_udp_pkt->udp_pkt.udp_dst_port,
				  ntohs(card->wandev.udp_port),
				  bstrm_udp_pkt->ip_pkt.protocol,
				  UDPMGMT_UDP_PROTOCOL,
				  bstrm_udp_pkt->wp_mgmt.request_reply,
				  UDPMGMT_REQUEST,
				  skb->data[36], skb->data[37]);
#endif	
		
	if (!strncmp(bstrm_udp_pkt->wp_mgmt.signature,UDPMGMT_SIGNATURE,8) &&
	   (bstrm_udp_pkt->udp_pkt.udp_dst_port == ntohs(card->wandev.udp_port)) &&
	   (bstrm_udp_pkt->ip_pkt.protocol == UDPMGMT_UDP_PROTOCOL) &&
	   (bstrm_udp_pkt->wp_mgmt.request_reply == UDPMGMT_REQUEST)) {

		return UDP_CPIPE_TYPE;

	}else{ 
		return UDP_INVALID_TYPE;
	}
}

/*============================================================================
 * Set PORT state.
 */
static void port_set_state (sdla_t *card, int state)
{
        if (card->u.b.state != state)
        {
                switch (state)
                {
                case WAN_CONNECTED:
                        printk (KERN_INFO "%s: Link connected!\n",
                                card->devname);
                      	break;

                case WAN_CONNECTING:
                        printk (KERN_INFO "%s: Link connecting...\n",
                                card->devname);
                        break;

                case WAN_DISCONNECTED:
                        printk (KERN_INFO "%s: Link disconnected!\n",
                                card->devname);
                        break;
                }

                card->wandev.state = card->u.b.state = state;
		if (card->wandev.dev){
			netdevice_t *dev = card->wandev.dev;
			bitstrm_private_area_t *bstrm_priv_area = dev->priv;
			bstrm_priv_area->common.state = state;
		}
        }
}

/*===========================================================================
 * config_bstrm
 *
 *	Configure the chdlc protocol and enable communications.		
 *
 *   	The if_open() function binds this function to the poll routine.
 *      Therefore, this function will run every time the chdlc interface
 *      is brought up. We cannot run this function from the if_open 
 *      because if_open does not have access to the remote IP address.
 *      
 *	If the communications are not enabled, proceed to configure
 *      the card and enable communications.
 *
 *      If the communications are enabled, it means that the interface
 *      was shutdown by ether the user or driver. In this case, we 
 *      have to check that the IP addresses have not changed.  If
 *      the IP addresses have changed, we have to reconfigure the firmware
 *      and update the changed IP addresses.  Otherwise, just exit.
 *
 */

static int config_bstrm (sdla_t *card)
{
	SHARED_MEMORY_INFO_STRUCT *flags = card->u.b.flags;

	if (card->u.b.comm_enabled){
		return 0;
	}

	/* Setup the Board for BSTRM */
	if (set_bstrm_config(card)) {
		printk (KERN_INFO "%s: Failed BSTRM configuration!\n",
			card->devname);
		return 0;
	}

	/* Set interrupt mode and mask */
        if (bstrm_set_intr_mode(card, APP_INT_ON_RX_BLOCK |
                		APP_INT_ON_GLOBAL_EXCEP_COND |
                		APP_INT_ON_TX_BLOCK |
                		APP_INT_ON_BSTRM_EXCEP_COND | APP_INT_ON_TIMER)){
		printk (KERN_INFO "%s: Failed to set interrupt triggers!\n",
				card->devname);
		return 0;	
        }
	

	/* Mask the Transmit and Timer interrupt */
	flags->interrupt_info_struct.interrupt_permission &= 
		~(APP_INT_ON_TX_BLOCK | APP_INT_ON_TIMER);


	/* Enable communications */
	if (bstrm_comm_enable(card) != 0) {
		printk(KERN_INFO "%s: Failed to enable chdlc communications!\n",
				card->devname);
		flags->interrupt_info_struct.interrupt_permission = 0;
		card->u.b.comm_enabled=0;
		bstrm_set_intr_mode(card,0);
		return 0;
	}

	/* Initialize Rx/Tx buffer control fields */
	init_bstrm_tx_rx_buff(card);
	port_set_state(card, WAN_CONNECTED);
	return 0; 
}


void s508_lock (sdla_t *card, unsigned long *smp_flags)
{
#if defined(CONFIG_SMP) || defined(LINUX_2_4)
	spin_lock_irqsave(&card->wandev.lock, *smp_flags);
        if (card->next){
        	spin_lock(&card->next->wandev.lock);
	}
#else
        disable_irq(card->hw.irq);
#endif                                                                     
}

void s508_unlock (sdla_t *card, unsigned long *smp_flags)
{
#if defined(CONFIG_SMP) || defined(LINUX_2_4)
        if (card->next){
        	spin_unlock(&card->next->wandev.lock);
        }
        spin_unlock_irqrestore(&card->wandev.lock, *smp_flags);
#else
        enable_irq(card->hw.irq);
#endif           
}



/****** End ****************************************************************/
