/******************************************************************************
 * sdla_56k.c	WANPIPE(tm) Multiprotocol WAN Link Driver. 
 *				56K board configuration.
 *
 * Author: 	Nenad Corbic  <ncorbic@sangoma.com>
 *
 * Copyright:	(c) 1995-2001 Sangoma Technologies Inc.
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 * ============================================================================
 * Jul 22, 2001	Nenad Corbic	Initial version.
 * Oct 01, 2001 Gideon Hack	Modifications for interrupt usage.
 ******************************************************************************
 */


/*
 ******************************************************************************
			   INCLUDE FILES
 ******************************************************************************
*/
#if (defined __FreeBSD__) | (defined __OpenBSD__)
# ifdef __FreeBSD__
#  include <stddef.h>
# endif
# include <sys/param.h>
# include <sys/systm.h>
# include <sys/syslog.h>
# include <sys/malloc.h>
# include <sys/errno.h>
# include <sys/sockio.h>
# include <sys/socket.h>
# include <sys/kernel.h>
# ifdef __OpenBSD__
#  include <sys/device.h>
#  include <sys/timeout.h>
#  include <machine/bus.h>
# endif
# include <net/wanpipe.h>
#elif (defined __WINDOWS__)
# include "csu_dsu.h"
#else
# include <linux/version.h>	/**/
# include <linux/config.h>	/* OS configuration options */
# include <linux/stddef.h>	/* offsetof, etc. */
# include <linux/errno.h>	/* returns codes */
# include <linux/string.h>	/* inline memset, etc */
# include <linux/kernel.h>	/* printk()m and other usefull stuff */
# include <linux/wanrouter.h>	/* WAN router definitions */
# include <linux/wanpipe.h>	/* WANPIPE common user API definitions */
#endif

/*
 ******************************************************************************
			  DEFINES AND MACROS
 ******************************************************************************
*/
#define WRITE_REG(reg, value)		card->wandev.write_front_end_reg(card, reg, (unsigned char)(value))
#define READ_REG(reg)			card->wandev.read_front_end_reg(card, reg)

/* Macro for enabling/disabling debugging comments */
#undef SDLA_56K_DEBUG
#if (defined __FreeBSD__) || (defined __OpenBSD__)
# define PRINT(card, format, msg...)	log(LOG_INFO, format, ## msg)
# ifdef SDLA_56K_DEBUG
#  define DEBUG_PRINT(format, msg...)	log(LOG_INFO, format, ## msg)
# else
#  define DEBUG_PRINT(format, msg...)
# endif
#elif (defined __WINDOWS__)
# define PRINT			OutputLogString
# ifdef SDLA_56K_DEBUG
#  define DEBUG_PRINT		DbgPrint
# else
#  define DEBUG_PRINT
# endif
#else
# define PRINT(card, format, msg...)	printk(KERN_INFO format, ## msg)
# ifdef SDLA_56K_DEBUG
#  define DEBUG_PRINT(format, msg...)	printk(KERN_INFO format, ## msg)
# else
#  define DEBUG_PRINT(format, msg...)
# endif
#endif


/*
 ******************************************************************************
			STRUCTURES AND TYPEDEFS
 ******************************************************************************
*/

static void display_Rx_code_condition(sdla_t* card);

/*
 ******************************************************************************
			   GLOBAL VARIABLES
 ******************************************************************************
*/

/*
 ******************************************************************************
			  FUNCTION PROTOTYPES
 ******************************************************************************
*/
/*
 ******************************************************************************
			  FUNCTION DEFINITIONS
 ******************************************************************************
*/
unsigned short sdla_56k_alarm(void* arg1, char manual_read)
{

	unsigned short status = 0x00;
	sdla_t* card = (sdla_t *)arg1;
	wan_device_t* wandev = &card->wandev; 

	FE_ASSERT1(wandev->write_front_end_reg == NULL);
	FE_ASSERT1(wandev->read_front_end_reg == NULL);
	/* set to manually read the front-end registers */
	if(manual_read) {
		wandev->RR8_reg_56k = 0;
		wandev->RR8_reg_56k = READ_REG(REG_INT_EN_STAT);
		wandev->RRA_reg_56k = READ_REG(REG_DEV_STAT);
		wandev->RRC_reg_56k = READ_REG(REG_RX_CODES);
		status |= wandev->RRA_reg_56k;
		status = (status << 8) & 0xFF00;
		status |= wandev->RRC_reg_56k;
	}
	
	DEBUG_PRINT("56K registers: 8-%02X,C-%02X,A-%02X\n", 
			wandev->RR8_reg_56k,
			wandev->RRC_reg_56k, 
			wandev->RRA_reg_56k);

	/* discard an invalid Rx code interrupt */
	if((wandev->RR8_reg_56k & BIT_INT_EN_STAT_RX_CODE) && 
		(!wandev->RRC_reg_56k)) {
		return(status);
	}

	/* record an RLOS (Receive Loss of Signal) condition */
	if(wandev->RRA_reg_56k & BIT_DEV_STAT_RLOS) {
		wandev->RLOS_56k = 1;
	}

	/* display Rx code conditions if necessary */
	if(!((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) & 
		wandev->delta_RRC_reg_56k) ||
		(wandev->front_end_status == FE_CONNECTED)) {
			display_Rx_code_condition(card);
	}
	
	/* record the changes that occurred in the Rx code conditions */
	wandev->delta_RRC_reg_56k |= 
		(wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k);

	
	if(manual_read || (wandev->RR8_reg_56k & BIT_INT_EN_STAT_ACTIVE)) {

		/* if Insertion Loss is less than 44.4 dB, then we are connected */
		if ((wandev->RRA_reg_56k & 0x0F) > BIT_DEV_STAT_IL_44_dB) {
			if((wandev->front_end_status == FE_DISCONNECTED) ||
			 (wandev->front_end_status == FE_UNITIALIZED)) {
				
				wandev->front_end_status = FE_CONNECTED;
				/* reset the Rx code condition changes */
				wandev->delta_RRC_reg_56k = 0;
				
				if(wandev->RLOS_56k == 1) {
					wandev->RLOS_56k = 0;
					PRINT(card, "%s: 56k Receive Signal Recovered\n", 
							card->devname);
				}
				PRINT(card, "%s: 56k Connected\n",
							card->devname);
			}
		}else{
			if((wandev->front_end_status == FE_CONNECTED) || 
			 (wandev->front_end_status == FE_UNITIALIZED)) {
				
				wandev->front_end_status = FE_DISCONNECTED;
				/* reset the Rx code condition changes */
				wandev->delta_RRC_reg_56k = 0;

				if(wandev->RLOS_56k == 1) {
					PRINT(card, "%s: 56k Receive Loss of Signal\n", 
							card->devname);
				}
				PRINT(card, "%s: 56k Disconnected (loopback)\n", 
						card->devname);
			}
		}
	}
	
	wandev->prev_RRC_reg_56k = wandev->RRC_reg_56k;
	return(status);
}



int sdla_56k_config(void* arg1)
{
	sdla_t* card = (sdla_t *)arg1;
	wan_device_t* wandev = &card->wandev; 

	FE_ASSERT1(wandev->write_front_end_reg == NULL);
	FE_ASSERT1(wandev->read_front_end_reg == NULL);
	/* The 56k CSU/DSU front end status has not been initialized  */
	wandev->front_end_status = FE_UNITIALIZED;

	/* Zero the previously read RRC register value */
	wandev->prev_RRC_reg_56k = 0;
	
	/* Zero the RRC register changes */ 
	wandev->delta_RRC_reg_56k = 0;
	
	if(WRITE_REG(REG_INT_EN_STAT, (BIT_INT_EN_STAT_IDEL | 
		BIT_INT_EN_STAT_RX_CODE | BIT_INT_EN_STAT_ACTIVE))) {
		return 1;
	} 

	if(WRITE_REG(REG_RX_CODES, 0xFF & ~(BIT_RX_CODES_ZSC))) {
		return 1;
	}

	if(WRITE_REG(REG_DEV_CTRL, (BIT_DEV_CTRL_XTALI_INT | 
		BIT_DEV_CTRL_SCT_E_OUT | BIT_DEV_CTRL_DDS_PRI))) {
		return 1;
	}
 
	if(WRITE_REG(REG_TX_CTRL, 0x00)) {
		return 1;
	}

	if(WRITE_REG(REG_RX_CTRL, 0x00)) {
		return 1;
	}

	if(WRITE_REG(REG_EIA_SEL, 0x00)) {
		return 1;
	}

	if(WRITE_REG(REG_EIA_TX_DATA, 0x00)) {
		return 1;
	}

	if(WRITE_REG(REG_EIA_CTRL, (BIT_EIA_CTRL_RTS_ACTIVE | 
		BIT_EIA_CTRL_DTR_ACTIVE | BIT_EIA_CTRL_DTE_ENABLE))) {
		return 1; 
	}

	return 0;
}


static void display_Rx_code_condition(sdla_t* card)
{
	wan_device_t* wandev = &card->wandev;

	/* check for a DLP (DSU Non-latching loopback) code condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_DLP) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_DLP) {
			PRINT(card, "%s: 56k receiving DSU Loopback code\n", 
					card->devname);
		} else {
			PRINT(card, "%s: 56k DSU Loopback code condition ceased\n", 
					card->devname);
		}
	}

	/* check for an OOF (Out of Frame) code condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_OOF) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_OOF) {
			PRINT(card, "%s: 56k receiving Out of Frame code\n", 
					card->devname);
		} else {
			PRINT(card, "%s: 56k Out of Frame code condition ceased\n", 
					card->devname);
		}
	}

	/* check for an OOS (Out of Service) code condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_OOS) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_OOS) {
			PRINT(card, "%s: 56k receiving Out of Service code\n", 
					card->devname);
		} else {
			PRINT(card, "%s: 56k Out of Service code condition ceased\n", 
					card->devname);
		}
	}

	/* check for a CMI (Control Mode Idle) condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_CMI) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_CMI) {
			PRINT(card, "%s: 56k receiving Control Mode Idle\n",
					card->devname);
		} else {
			PRINT(card, "%s: 56k Control Mode Idle condition ceased\n", 
					card->devname);
		}
	}

	/* check for a ZSC (Zero Suppression Code) condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_ZSC) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_ZSC) {
			PRINT(card, "%s: 56k receiving Zero Suppression code\n", 
					card->devname);
		} else {
			PRINT(card, "%s: 56k Zero Suppression code condition ceased\n", 
					card->devname);
		}
	}

	/* check for a DMI (Data Mode Idle) condition */
	if((wandev->RRC_reg_56k ^ wandev->prev_RRC_reg_56k) &
		BIT_RX_CODES_DMI) {
		if(wandev->RRC_reg_56k & BIT_RX_CODES_DMI) {
			PRINT(card, "%s: 56k receiving Data Mode Idle\n",
				card->devname);
		} else {
			PRINT(card, "%s: 56k Data Mode Idle condition ceased\n", 
				card->devname);
		}
	}
}
