/*****************************************************************************
* bitstrm.c	- Bit Streaming API: Demonstration Module.
* 								 Set Tab to 4 when viewing.
*
* Author(s):	David Rokhvarg <drokhvarg@sangoma.com>
*
* Copyright:	(c) 2001 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ============================================================================
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/if_wanpipe.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <signal.h>
#include <linux/wanpipe.h>
#include <linux/sdla_chdlc.h>


#define FALSE	0
#define TRUE	1

#define MAX_TX_DATA     			3000  
#define MAX_LGTH_FILE_PATHNAME 		50
#define OP_MODE_NONE				0x00
#define OP_MODE_LOOPBACK			0x01
#define OP_MODE_RX_TO_FILE			0x02
#define OP_MODE_TX_FROM_FILE		0x04

#define MAXIMUM_BLOCK_LENGTH		6000
#define NO							0x00
#define YES							0x01
#define SDLA_WINDOW_SIZE			0x2000UL
#define PORT_OFFSET_WINDOW 			0x02
#define SIZEOF_MB_HDR				16
#define NUMBER_ADAPTERS 			2
#define ADPTR_0_DEFAULT_IO_PORT		0x360
#define ADPTR_1_DEFAULT_IO_PORT		0x350
#define ADPTR_0_DEFAULT_8K_WIN_ADDR	0xD0000000UL
#define ADPTR_1_DEFAULT_8K_WIN_ADDR	0xD2000000UL
#define MAX_LGTH_FILE_PATHNAME 		50
#define OP_MODE_NONE				0x00
#define OP_MODE_LOOPBACK			0x01
#define OP_MODE_RX_TO_FILE			0x02
#define OP_MODE_TX_FROM_FILE		0x04

typedef struct {
	unsigned long Rx_block_count;
	unsigned long Rx_byte_count;
	unsigned long Tx_block_count;
	unsigned long Tx_byte_count;
	unsigned short first_lgth_Rx_data;
	unsigned short last_lgth_Rx_data;
	unsigned char Rx_bfr[MAXIMUM_BLOCK_LENGTH];
	unsigned char first_Rx_bfr[MAXIMUM_BLOCK_LENGTH];
	unsigned char Tx_bfr[MAXIMUM_BLOCK_LENGTH];
} ADAPTER_STRUCT;

ADAPTER_STRUCT adptr_str[NUMBER_ADAPTERS];

char dual_card_bridge;
int no_adaptrs_in_use;
char op_mode;
char Rx_file[MAX_LGTH_FILE_PATHNAME];
char Tx_file[MAX_LGTH_FILE_PATHNAME];
char Tx_complete = NO;
unsigned char tx_tmp_bfr[6000];

unsigned char HDLC_streaming = FALSE;
unsigned short Rx_lgth;

unsigned char Rx_data[16000];
unsigned char Tx_data[MAX_TX_DATA + sizeof(api_tx_hdr_t)];

FILE * rx_file = NULL;
FILE * tx_file = NULL;

/* Prototypes */
int MakeConnection(int adptr_no, char *r_name, char *i_name );
void do_Tx_Rx( void);
void send_file();
void receive_to_file();
void process_cmnd_line_args(char *s);
void display_cmnd_line_args(void);
void cleanup(void);
int transmit_data_block(int adptr_no);
void reset_statistics(void);
int display_op_mode(void);
int receive_data_block(int adptr_no);
int open_Tx_Rx_files(void);


int 	sock[NUMBER_ADAPTERS];
int    quit=0;
fd_set 	tx_write[NUMBER_ADAPTERS], rx_ready[NUMBER_ADAPTERS];
int file_sent = FALSE;
int sent_successfully = TRUE;

char op_mode = OP_MODE_NONE;
char Rx_file[MAX_LGTH_FILE_PATHNAME];
char Tx_file[MAX_LGTH_FILE_PATHNAME];
int is_connected[NUMBER_ADAPTERS];


//
//signal handler for catching "Cntrl+C" signal and exiting main loop
//
void sighandler (int sigio)
{	
	quit=1;
	return;
}


/***************************************************
* MakeConnection
*
*   o Create a Socket
*   o Bind a socket to a wanpipe network interface
*       (Interface name is supplied by the user)
*/         

int MakeConnection(int adptr_no, char *r_name, char *i_name ) 
{
	struct wan_sockaddr_ll 	sa;

	memset(&sa,0,sizeof(struct wan_sockaddr_ll));
	errno = 0;
   	sock[adptr_no] = socket(AF_WANPIPE, SOCK_RAW, 0);
   	if( sock[adptr_no] < 0 ) {
      		perror("Socket");
      		return( FALSE );
   	} /* if */
  
	printf("\nConnecting to router %s, interface %s\n", r_name, i_name);
	
	strcpy( sa.sll_device, i_name);
	strcpy( sa.sll_card, r_name);
	sa.sll_protocol = htons(PVC_PROT);
	sa.sll_family=AF_WANPIPE;

        if(bind(sock[adptr_no], (struct sockaddr *)&sa, sizeof(struct wan_sockaddr_ll)) < 0){
            perror("bind");
			printf("Failed to bind a socket to %s interface\n",i_name);
            exit(0);
        }
	printf("Socket bound to %s\n\n",i_name);
	is_connected[adptr_no] = NO;

	return( TRUE );
}

int open_Tx_Rx_files()
{
	if(op_mode & OP_MODE_RX_TO_FILE){
		
		printf("Opening %s file...\n", Rx_file);
		if((  rx_file = fopen(Rx_file, "wb")) == NULL){
			printf("Error opening %s file! Exiting.\n", Rx_file);
			return ( FALSE );
		}
		printf("Opened %s file.\n", Rx_file);
	}

	if(op_mode & OP_MODE_TX_FROM_FILE){
			
		printf("Opening %s file...\n", Tx_file);
		if((  tx_file = fopen(Tx_file, "rb")) == NULL){
			printf("Error opening %s file! Check the file exist. Exiting.\n", Tx_file);
 			return ( FALSE );
		}
		printf("Opened %s file.\n", Tx_file);
	}

	return ( TRUE );
}


/***************************************************
* RECV_SOCKET 
*
*   o Read a socket 
*   o Cast data received to api_rx_element_t data type 
*   o The received packet contains 16 bytes header 
*
*	------------------------------------------
*      |  16 bytes      |        X bytes        ...
*	------------------------------------------
* 	   Header              Data
*
*   o Data structures:
*
*	typedef struct {
*        	unsigned char   error_flag      PACKED;
*        	unsigned short  time_stamp      PACKED;
*        	unsigned char   reserved[13]    PACKED;
*	} api_rx_hdr_t; 
*
*	typedef struct {
*        	api_rx_hdr_t    api_rx_hdr      PACKED;
*        	void *          data            PACKED;
*	} api_rx_element_t;
*
*
*/

void do_Tx_Rx(void)
{
	printf("Press <Ctrl + C> to exit...\n");
	
	do{
		if(dual_card_bridge)
		{	// receive on adapter #0, transmit on adapter #1
			if((adptr_str[0].Rx_block_count < 2) || 
					(adptr_str[0].Rx_block_count == adptr_str[1].Tx_block_count)) {
				
				receive_data_block(0);
			}
			
			if(!adptr_str[1].Tx_block_count) {
				if(adptr_str[0].Rx_block_count > 1) {
				   transmit_data_block(1);
				}
			}else if((adptr_str[1].Tx_block_count != adptr_str[0].Rx_block_count)) {
			  	transmit_data_block(1);
			}
			
			// receive on adapter #1, transmit on adapter #0
			if((adptr_str[1].Rx_block_count < 2) || 
					(adptr_str[1].Rx_block_count == adptr_str[0].Tx_block_count)) {
			
				receive_data_block(1);
			}
			
			if(!adptr_str[0].Tx_block_count) {
				if(adptr_str[1].Rx_block_count > 1) {
		
					transmit_data_block(0);
				}
			}else if((adptr_str[0].Tx_block_count != adptr_str[1].Rx_block_count)) {
				transmit_data_block(0);
			}
		}else{
		
			if(op_mode & (OP_MODE_RX_TO_FILE | OP_MODE_LOOPBACK)) {
				
				if(op_mode & OP_MODE_LOOPBACK) {
					
					if((adptr_str[0].Rx_block_count < 2) ||
						(adptr_str[0].Rx_block_count == adptr_str[0].Tx_block_count))
				   		{
							receive_data_block(0);
						}
				}else{
			   		receive_data_block(0);
	   			}
     		}
			

			if((op_mode & OP_MODE_TX_FROM_FILE) && !Tx_complete) {
	   	   		transmit_data_block(0);
			}

			if(op_mode & OP_MODE_LOOPBACK) {
				if(!adptr_str[0].Tx_block_count) {
					if(adptr_str[0].Rx_block_count > 1){
						transmit_data_block(0);
					}
				}else if((adptr_str[0].Tx_block_count != adptr_str[0].Rx_block_count)) {
					transmit_data_block(0);
	   			}
     		}

			if(Tx_complete && !(op_mode & OP_MODE_RX_TO_FILE)) {
				break;
			}
		}		
			
	}while(!quit);
	
	cleanup();
} 


int receive_data_block(int adptr_no)
{
	int err;
	int lgth_Rx_data = 0;

	struct timeval tv;

	tv.tv_usec = 0; 
	tv.tv_sec = 5;
				
    FD_ZERO(&rx_ready[adptr_no]);
	FD_SET(sock[adptr_no],&rx_ready[adptr_no]);

	if(select(sock[adptr_no] + 1, &rx_ready[adptr_no], NULL , NULL, &tv))
	{	if (FD_ISSET(sock[adptr_no],&rx_ready[adptr_no]))
		{	err = recv(sock[adptr_no], adptr_str[adptr_no].Rx_bfr, MAXIMUM_BLOCK_LENGTH, 0);
																	
			// err indicates bytes received
			if(err > 0)
			{	// Check the packet length
				lgth_Rx_data = err - sizeof(api_rx_hdr_t);
				
				if(lgth_Rx_data <= 0) {
					printf("\nShort frame received (%d)\n", lgth_Rx_data);
				   	return(NO);
				}
			} else {
				printf("\nError receiving data\n");
				return(NO);
			}
		}else{
			return(NO);
		}
	} else {
		printf("\nError selecting Rx socket\n");
		return(NO);
	}
	

	if(op_mode & OP_MODE_RX_TO_FILE) {
		fwrite((void *)&adptr_str[adptr_no].Rx_bfr[sizeof(api_rx_hdr_t)], 
												sizeof(char), lgth_Rx_data, rx_file);
	}

	adptr_str[adptr_no].Rx_block_count ++;
	adptr_str[adptr_no].Rx_byte_count += lgth_Rx_data;
	
	printf("ADAPTER #%d RECEIVED %-10lu DATA BLOCKS\n", adptr_no, 
														adptr_str[adptr_no].Rx_block_count);
	printf("ADAPTER #%d RECEIVED %-10lu DATA BYTES\n", adptr_no, 
														adptr_str[adptr_no].Rx_byte_count);

	if(!(op_mode & OP_MODE_RX_TO_FILE))
   	{	if(adptr_str[adptr_no].Rx_block_count == 1)
	   	{	//move header and the data to the buffer
			memmove(adptr_str[adptr_no].first_Rx_bfr, adptr_str[adptr_no].Rx_bfr,
			   											lgth_Rx_data + sizeof(api_rx_hdr_t));
			adptr_str[adptr_no].first_lgth_Rx_data = lgth_Rx_data;
		}else
	   	{	adptr_str[adptr_no].last_lgth_Rx_data = lgth_Rx_data;
		}
	}
			
	return(YES);
}


void process_cmnd_line_args(char *s)
{
	switch(toupper( *(s++) ))
       	{
	case 'D':
		dual_card_bridge = YES;
		no_adaptrs_in_use = 2;
		break;
	case 'L':
		op_mode = OP_MODE_LOOPBACK;
		break;
	case 'R':
		printf("case 'R':\n");
		strncpy(Rx_file, s, MAX_LGTH_FILE_PATHNAME);
		op_mode |= OP_MODE_RX_TO_FILE;
		break;
	case 'T':
		printf("case 'T':\n");
		strncpy(Tx_file, s, MAX_LGTH_FILE_PATHNAME);
		op_mode |= OP_MODE_TX_FROM_FILE;
		break;
	default:
		display_cmnd_line_args();
		exit(1);
	}
}

void display_cmnd_line_args(void)
{
	printf("\nValid command line arguments are:\n");
	printf("To save rx data to a file :\n");
	printf("	bitstrm <router> <if name> -rRX_FILE\n");
	printf("e.g.:	bitstrm wanpipe1 wp1_bstrm -rRx_file\n\n");
	
	printf("To send data from a file :\n");
	printf("	bitstrm <router> <if name> -tTX_FILE\n");
	printf("e.g.:	bitstrm wanpipe1 wp1_bstrm -tTx_file\n\n");
	
	printf("To perform the loopback test:\n");
	printf("	bitstrm <router> <if name> -L\n");
	printf("e.g.:	bitstrm wanpipe1 wp1_bstrm -L\n\n");
	
	printf("To perform the dual-card bridge test:\n");
	printf("	bitstrm <router1> <if name on router1> <router2> <if name on router2> -D\n");
	printf("e.g.:	bitstrm wanpipe1 wp1_bstrm wanpipe2 wp2_bstrm -D\n\n");

	printf("Press <Ctrl + C> to exit the program.\n");
}

void cleanup()
{
	if(rx_file){
		printf("Closing '%s' file.\n", Rx_file);
		fclose(rx_file);
	}

	if(tx_file){
		printf("Closing '%s' file.\n", Tx_file);
	       	fclose(tx_file);
	}
		
	if(is_connected[0] == YES)
	{	while (ioctl(sock[0],SIOC_WANPIPE_CHECK_TX,0));
		close (sock[0]);
	}

	if(dual_card_bridge == YES)
	{	if(is_connected[1] == YES)
		{	while (ioctl(sock[1],SIOC_WANPIPE_CHECK_TX,0));
			close (sock[1]);	
		}
	}
}

int transmit_data_block(int adptr_no)
{
	unsigned short bytes_to_Tx;
	int Rx_adptr_no;
		
	int err, i;
	static int count_sent = 0;
	api_tx_element_t * api_tx_el;
	unsigned char *data;
	struct timeval tv;

	tv.tv_usec = 0; 
	tv.tv_sec = 5;
				
	api_tx_el = (api_tx_element_t*)&adptr_str[adptr_no].Tx_bfr[0];
	data = (unsigned char *)&api_tx_el->data;
	
	FD_ZERO(&tx_write[adptr_no]);
	FD_SET(sock[adptr_no],&tx_write[adptr_no]);

	if(select(sock[adptr_no] + 1, NULL, &tx_write[adptr_no], NULL, &tv))
	{	if (FD_ISSET(sock[adptr_no],&tx_write[adptr_no]))
		{	if(op_mode & OP_MODE_TX_FROM_FILE)
		   	{	//read the data from the tx file
				if(sent_successfully == TRUE)
				{	memset(&adptr_str[adptr_no].Tx_bfr,0,MAX_TX_DATA + sizeof(api_tx_hdr_t));
					count_sent = 0;
					
					//read the data from file
					for(i=0;i < MAX_TX_DATA; i++)
					{	if(!feof(tx_file))
						{	data[i] = (unsigned char)fgetc(tx_file);
							count_sent++;
						}else{
							//end of the tx file reached 
							break;
						}
					}
				}

				err = send(sock[adptr_no], adptr_str[adptr_no].Tx_bfr,
				 										count_sent + sizeof(api_tx_hdr_t), 0);

				if (err > 0)
				{	sent_successfully = TRUE;
					bytes_to_Tx = count_sent;
					
					if( feof(tx_file) )
					{	Tx_complete = YES;
					}
				}else
				{	//card is busy, must retry to send next time the function is called
					sent_successfully = FALSE;
					return (NO);
				}
			}else
   			{	if(dual_card_bridge)
			   	{	Rx_adptr_no = (adptr_no == 0) ? 1 : 0;
				}else
			   	{	Rx_adptr_no = 0;
				}

				//Cast to api_tx_element_t data type so data will be copied
				//to the right offset
				api_tx_el = (api_tx_element_t *)&tx_tmp_bfr[0];
				data = (unsigned char *)&api_tx_el->data;
				
				if(!adptr_str[adptr_no].Tx_block_count)
			   	{	//skip past rx header
					memcpy(data, &adptr_str[Rx_adptr_no].first_Rx_bfr[sizeof(api_rx_hdr_t)],
									adptr_str[Rx_adptr_no].first_lgth_Rx_data);
							
					err = send(sock[adptr_no], tx_tmp_bfr,
						adptr_str[Rx_adptr_no].first_lgth_Rx_data + sizeof(api_tx_hdr_t), 0);
					
					bytes_to_Tx = adptr_str[Rx_adptr_no].first_lgth_Rx_data;
					
				}else
			   	{	//skip past rx header
					memcpy(data, &adptr_str[Rx_adptr_no].Rx_bfr[sizeof(api_rx_hdr_t)],
						adptr_str[Rx_adptr_no].last_lgth_Rx_data + sizeof(api_tx_hdr_t));
															
					err = send(sock[adptr_no], tx_tmp_bfr, 
							adptr_str[Rx_adptr_no].last_lgth_Rx_data + sizeof(api_tx_hdr_t), 0);

					bytes_to_Tx = adptr_str[Rx_adptr_no].last_lgth_Rx_data;
				}
			}

		}else
		{	return(NO);
		}
	}else
	{	printf("\nError selecting Tx socket\n");
		//we cannot transmit
		if(dual_card_bridge || (op_mode & OP_MODE_LOOPBACK))
		{	printf("TRANSMISSION FAILURE ON CARD #%d. PLEASE CHECK CLOCKING AND CABLING!\n",
				   																	adptr_no);
			quit = 1;
		}
		return(NO);
	}

	adptr_str[adptr_no].Tx_block_count ++;
	adptr_str[adptr_no].Tx_byte_count += bytes_to_Tx;
	
	printf("ADAPTER #%d SENT %-10lu 	DATA BLOCKS\n", adptr_no, 
								adptr_str[adptr_no].Tx_block_count);
	
	printf("ADAPTER #%d SENT %-10lu 	DATA BYTES\n", adptr_no, 
								adptr_str[adptr_no].Tx_byte_count);

	if((op_mode & OP_MODE_TX_FROM_FILE) && (feof(tx_file)))
	{	printf("End of '%s'\n", Tx_file);
	}

	return(YES);
}

void set_default_args(void)
{
	dual_card_bridge = NO;
	no_adaptrs_in_use = 1;
	op_mode = OP_MODE_NONE;
}


void reset_statistics(void)
{
	int i;

	for(i = 0; i < NUMBER_ADAPTERS; i ++) {
		adptr_str[i].Rx_block_count = 0l;
		adptr_str[i].Rx_byte_count = 0l;
		adptr_str[i].Tx_block_count = 0l;
		adptr_str[i].Tx_byte_count = 0l;

		is_connected[i] = NO;
	}

		
}

int display_op_mode(void)
{
	printf("\nOperation mode: ");
	
	if(dual_card_bridge) {
		printf("Dual card bridge");
		return(YES);
	}

	switch(op_mode) {
	case OP_MODE_LOOPBACK:
		printf("Single card - bridge receiver and transmitter");
		break;
	case OP_MODE_RX_TO_FILE:
		printf("Single card - Rx to file '%s'", Rx_file);
		break;
	case OP_MODE_TX_FROM_FILE:
		printf("Single card - Tx from file '%s'", Tx_file);
		break;
	case (OP_MODE_RX_TO_FILE | OP_MODE_TX_FROM_FILE):
		printf("Single card - Rx to file '%s'", Rx_file);
		printf("Tx from file '%s'", Tx_file);
		break;
	default:
		printf("Invalid");
		return(NO);
	}
	return(YES);
}


/***************************************************************
 * Main:
 *
 *    o Make a socket connection to the driver.
 *    o Call do_Tx_Rx() to start the main read/write loop 
 *
 **************************************************************/


int main(int argc, char* argv[])
{
	int proceed, ind;

	if (argc < 4)
	{	display_cmnd_line_args();
		exit(0);
	}
	
	//signal handler for terminating the program
	signal(SIGINT, sighandler);
	
	set_default_args();
	reset_statistics();

	ind=0;
	while( ind < argc )
	{	if(*((argv)[ind]) == '-')
	   	{	process_cmnd_line_args(argv[ind] + 1);
		}
		ind++;
	}

	if(!display_op_mode())
   	{	exit(1);
	}
	
	//connect to the driver
	proceed = MakeConnection(0, argv[1], argv[2]);
	
	if(dual_card_bridge == YES)
		proceed = MakeConnection(1, argv[3], argv[4]);

	if( proceed == TRUE )
	{	if(op_mode & OP_MODE_RX_TO_FILE || op_mode & OP_MODE_TX_FROM_FILE)
		{	//open files for rx and/or tx data
			if(open_Tx_Rx_files() == FALSE)
			{	cleanup();
				return 1;
			}
		}
		//start sending / receiving
		do_Tx_Rx();
		return 0;
	}
	
	return 1;
}


