/*	$OpenBSD: gprof.c,v 1.11 2002/05/08 16:46:35 art Exp $	*/
/*	$NetBSD: gprof.c,v 1.8 1995/04/19 07:15:59 cgd Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "gprof.h"
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

/* ------------------------------------------------------------------------ */
int             valcmp(const void *, const void *);

/* static struct gmon_hist_hdr gmonhdr; */
extern char    *__progname;

unsigned int START_ADDRESS = 0;
/* ------------------------------------------------------------------------ */
int             main(argc, argv)
int             argc;
char          **argv;
{
  char          **sp;
  nltype        **timesortnlp;
  char          **defaultEs;

  --argc;
  argv++;
  debug = 0;
  bflag = TRUE;
  while (*argv != 0 && **argv == '-') {
    (*argv)++;
    switch (**argv) {
	case 'a':
	  aflag = TRUE;
	  break;
	case 'b':
	  bflag = FALSE;
	  break;
	case 'C':
	  Cflag = TRUE;
	  cyclethreshold = atoi(*++argv);
	  break;
	case 'c':
	  cflag = TRUE;
	  break;
	case 'd':
	  dflag = TRUE;
	  setlinebuf(stdout);
	  debug |= atoi(*++argv);
	  debug |= ANYDEBUG;
#ifdef DEBUG
	  printf("[main] debug = %d\n", debug);
#else					/* not DEBUG */
	  warnx("-d ignored");
#endif					/* DEBUG */
	  break;
	case 'E':
	  ++argv;
	  addlist(Elist, *argv);
	  Eflag = TRUE;
	  addlist(elist, *argv);
	  eflag = TRUE;
	  break;
	case 'e':
	  addlist(elist, *++argv);
	  eflag = TRUE;
	  break;
	case 'F':
	  ++argv;
	  addlist(Flist, *argv);
	  Fflag = TRUE;
	  addlist(flist, *argv);
	  fflag = TRUE;
	  break;
	case 'f':
	  addlist(flist, *++argv);
	  fflag = TRUE;
	  break;
	case 'k':
	  addlist(kfromlist, *++argv);
	  addlist(ktolist, *++argv);
	  kflag = TRUE;
	  break;
#ifdef NOTDONEYET
	case 's':
	  sflag = TRUE;
	  break;
#endif
	case 'z':
	  zflag = TRUE;
	  break;
	default:
	  fprintf(stderr, "unrecognized -argument %c\n", **argv);
	  exit(1);
    }
    argv++;
  }
  if (*argv != 0) {
    a_outname = *argv;
    argv++;
  } else {
    a_outname = A_OUTNAME;
  }
  if (*argv != 0) {
    gmonname = *argv;
    argv++;
  } else {
    gmonname = GMONNAME;
  }
/* get information about a.out file.  */
  if (getnfile(a_outname, &defaultEs) == -1)
    errx(1, "%s: bad format", a_outname);
/* sort symbol table.  */
  qsort(nl, nname, sizeof(nltype), valcmp);
/* turn off default functions */
  for (sp = &defaultEs[0]; *sp; sp++) {
    Eflag = TRUE;
    addlist(Elist, *sp);
    eflag = TRUE;
    addlist(elist, *sp);
  }
/* get information about mon.out file(s).  */
  do {
    getpfile(gmonname);
    if (*argv != 0) {
      gmonname = *argv;
    }
  } while (*argv++ != 0);
/* how many ticks per second?  if we can't tell, report time in ticks.  */
  if (hz == 0) {
    hz = 1;
    warnx("time is in ticks, not seconds");
  }
#ifdef NOTDONEYET
/* dump out a gmon.sum file if requested */
  if (sflag) {
    dumpsum(GMONSUM);
  }
#endif	/* NOTDONEYET */
/* assign samples to procedures */
  asgnsamples();
/* assemble the dynamic profile */
  timesortnlp = doarcs();
/* print the dynamic profile */
  printgprof(timesortnlp);
/* print the flat profile */
  printprof();
/* print the index */
  printindex();

  return (0);
}

/* ------------------------------------------------------------------------ */
void            readsamples(pfile)
FILE           *pfile;
{
  UNIT            sample;
  int             i;
  int lth;

  if (samples == 0) {
    samples = (UNIT *) malloc(sampbytes * sizeof(UNIT));
    if (samples == 0)
      errx(1, "No room for %d sample pc's", sampbytes / sizeof(UNIT));
  }
  for (i = 0; i < nsamples; i++) {
    lth = fread(&sample, sizeof(UNIT), 1, pfile);
    if (lth == 0) {
      if (feof(pfile))
	break;
    }
    samples[i] += ntohs(sample);
  }
  if (i != nsamples)
    errx(1, "unexpected EOF after reading %d/%d samples", i, nsamples);
}

/* ------------------------------------------------------------------------ */
void read_hist(FILE *pfile, char *filename)
{
  struct gmon_hist_hdr tmp;
  int             rate;
  unsigned int    i;
  int lth;

  lth = fread(&tmp, sizeof(struct gmon_hist_hdr), 1, pfile);
  if (lth == 0) {
    if (feof(pfile))
      return;
  }
  i = (int) (*(char **) (tmp.low_pc));
  s_lowpc = ntohl(i);
  if (s_lowpc < 0x80000000) {
    START_ADDRESS = s_lowpc & ~(4096-1);
  }
  s_lowpc = s_lowpc - START_ADDRESS;
  i = (int) (*(char **) (tmp.high_pc));
  s_highpc = ntohl(i) - START_ADDRESS;
  i = (int) (*(char **) (tmp.hist_size));
  nsamples = ntohl(i);
  i = (int) (*(char **) (tmp.prof_rate));
  rate = ntohl(i);
  if (hz == 0) {
    hz = rate;
  } else if (hz != rate)
    errx(1, "%s: profile clock rate (%d) incompatible with clock rate "
	 "(%ld) in first gmon file", filename, rate, hz);
  lowpc = (unsigned long) s_lowpc / sizeof(UNIT);
  highpc = (unsigned long) s_highpc / sizeof(UNIT);
  sampbytes = nsamples * sizeof(UNIT);
#ifdef DEBUG
  if (debug & SAMPLEDEBUG) {
    printf("[%s]   s_lowpc 0x%lx   s_highpc 0x%lx\n",
	   __FUNCTION__, s_lowpc, s_highpc);
    printf("[%s]     lowpc 0x%x     highpc 0x%x\n",
	   __FUNCTION__, lowpc, highpc);
    printf("[%s] sampbytes %d nsamples %d\n",
	   __FUNCTION__, sampbytes, nsamples);
    printf("[%s] sample rate %ld\n", __FUNCTION__, hz);
  }
#endif					/* DEBUG */
  readsamples(pfile);
}

/* ------------------------------------------------------------------------ */
FILE           * openpfile(filename)
char           *filename;
{
  struct gmon_hdr ghdr __attribute__ ((aligned (__alignof__ (int))));
  FILE           *pfile;
  int version;
  int lth;

  if ((pfile = fopen(filename, "r")) == NULL)
    err(1, "fopen: %s", filename);

/* Read and check header. */
  lth = fread(&ghdr, sizeof (struct gmon_hdr), 1, pfile);
  if (lth == 0) {
    if (feof(pfile))
       err(1, "fread: %s", filename);
  }
  if (strncmp(GMON_MAGIC, &ghdr.cookie[0], sizeof (ghdr.cookie)) != 0) {
    fprintf(stderr, "header(%s) != %s\n", &ghdr.cookie[0], GMON_MAGIC);
    exit(1);
  }
  version = ntohl(*(int32_t *)ghdr.version);
  if (version != GMON_VERSION) {
    fprintf(stderr, "version(%d) != %d\n", version, GMON_VERSION);
    exit(1);
  }
  return (pfile);
}

/* ------------------------------------------------------------------------ */
void            tally(rawp)
struct rawarc *rawp;
{
  nltype         *parentp;
  nltype         *childp;
  unsigned int i;

  parentp = nllookup(rawp->raw_frompc);
  childp = nllookup(rawp->raw_selfpc);
  if (parentp == 0 || childp == 0)
    return;
  if (kflag
      && onlist(kfromlist, parentp->name)
      && onlist(ktolist, childp->name)) {
    return;
  }
  i = rawp->raw_count;
  childp->ncall += i;
#ifdef DEBUG
  if (debug & TALLYDEBUG) {
    printf("[tally] arc from %s to %s traversed %d times\n",
	   parentp->name, childp->name, i);
  }
#endif					/* DEBUG */
  addarc(parentp, childp, i);
}

/* ------------------------------------------------------------------------ */
void read_call_graph(FILE *pfile, char *filename)
{
  ssize_t lth;
  struct gmon_cg_arc_record raw_arc __attribute__ ((aligned (__alignof__ (char*))));
  struct rawarc arc;

  lth = fread(&raw_arc, sizeof(raw_arc), 1, pfile);
  if (lth == 0) {
    if (feof(pfile))
      return;
    perror("read_call_graph, read raw_arc");
    exit(1);
  }
  (int) arc.raw_frompc = *(char **)(raw_arc.from_pc);
  arc.raw_frompc = ntohl(arc.raw_frompc) - START_ADDRESS;
  (int) arc.raw_selfpc = *(char **)(raw_arc.self_pc);
  arc.raw_selfpc = ntohl(arc.raw_selfpc) - START_ADDRESS;
  (int) arc.raw_count = *(char **)(raw_arc.count);
  arc.raw_count = ntohl(arc.raw_count);

  tally(&arc);		/* add this arc */
}

/* ------------------------------------------------------------------------ */
void read_bb_counts(FILE *pfile, char *filename)
{
  fprintf(stderr, "NOT DONE YET\n");
  exit(1);
}

/* ------------------------------------------------------------------------ */
 /*
  *	information from a gmon.out file is in two parts:
  *	an array of sampling hits within pc ranges,
  *	and the arcs.
  */
void            getpfile(filename)
char           *filename;
{
  FILE           *pfile;
  unsigned char gmon_type;
  int lth;

  pfile = openpfile(filename);
/* First entry is type. */
  while (1) {
    lth = fread(&gmon_type, sizeof(gmon_type), 1, pfile);
    if (lth == 0) {
      if (feof(pfile) == 0) {
	perror("read gmon_type");
	exit(1);
      }
      break;			/* end of file */
    } else if (lth != sizeof(gmon_type)) {
      fprintf(stderr, "read of gmon_type lth wrong, only %d, should be %d\n", lth, sizeof(gmon_type));
      exit(1);
    }
    if (gmon_type == GMON_TAG_TIME_HIST) {
      read_hist(pfile, filename);
    } else if (gmon_type == GMON_TAG_CG_ARC) {
      read_call_graph(pfile, filename);
    } else if (gmon_type == GMON_TAG_BB_COUNT) {
      read_bb_counts(pfile, filename);
    } else {
      fprintf(stderr, "gmon_type unrecognized (%d)\n", gmon_type);
      exit(1);
    }
  }
  fclose(pfile);
}

/* ------------------------------------------------------------------------ */
#ifdef NOTDONEYET
/*
 * dump out the gmon.sum file
 */

void            dumpsum(sumfile)
char           *sumfile;
{
  nltype         *nlp;
  arctype        *arcp;
  struct gmon_cg_arc_record arc;
  FILE           *sfile;
  int i;

  if ((sfile = fopen(sumfile, "w")) == NULL)
    err(1, "fopen: %s", sumfile);
 /*
  * dump the header; use the last header read in
  */
  if (fwrite(&gmonhdr, sizeof gmonhdr, 1, sfile) != 1)
    err(1, "fwrite: %s", sumfile);
 /*
  * dump the samples
  */
  if (fwrite(samples, sizeof(UNIT), nsamples, sfile) != nsamples)
    err(1, "fwrite: %s", sumfile);
 /*
  * dump the normalized raw arc information
  */
  for (nlp = nl; nlp < npe; nlp++) {
    for (arcp = nlp->children; arcp; arcp = arcp->arc_childlist) {
      *(char **)arc.from_pc = (char *)(arcp->arc_parentp->value);
      *(char **)arc.self_pc = (char *)(arcp->arc_childp->value);
      *(char **)arc.count = (char *)(arcp->arc_count);
      if (fwrite(&arc, sizeof arc, 1, sfile) != 1)
	err(1, "fwrite: %s", sumfile);
#ifdef DEBUG
      if (debug & SAMPLEDEBUG) {
	printf("[dumpsum] from_pc 0x%x self_pc 0x%x count %d\n",
	       arc.from_pc, arc.self_pc, arc.count);
      }
#endif					/* DEBUG */
    }
  }
  fclose(sfile);
}
#endif /* NOTDONEYET */

/* ------------------------------------------------------------------------ */
int             valcmp(const void *vp1, const void *vp2)
{
  const nltype   *p1 = vp1;
  const nltype   *p2 = vp2;

  if (p1->value < p2->value) {
    return LESSTHAN;
  }
  if (p1->value > p2->value) {
    return GREATERTHAN;
  }
  return EQUALTO;
}

/* ------------------------------------------------------------------------ */
/*
 *	Assign samples to the procedures to which they belong.
 *
 *	There are three cases as to where pcl and pch can be
 *	with respect to the routine entry addresses svalue0 and svalue1
 *	as shown in the following diagram.  overlap computes the
 *	distance between the arrows, the fraction of the sample
 *	that is to be credited to the routine which starts at svalue0.
 *
 *	    svalue0                                         svalue1
 *	       |                                               |
 *	       v                                               v
 *
 *	       +-----------------------------------------------+
 *	       |					       |
 *	  |  ->|    |<-		->|         |<-		->|    |<-  |
 *	  |         |		  |         |		  |         |
 *	  +---------+		  +---------+		  +---------+
 *
 *	  ^         ^		  ^         ^		  ^         ^
 *	  |         |		  |         |		  |         |
 *	 pcl       pch		 pcl       pch		 pcl       pch
 *
 *	For the vax we assert that samples will never fall in the first
 *	two bytes of any routine, since that is the entry mask,
 *	thus we give call alignentries() to adjust the entry points if
 *	the entry mask falls in one bucket but the code for the routine
 *	doesn't start until the next bucket.  In conjunction with the
 *	alignment of routine addresses, this should allow us to have
 *	only one sample for every four bytes of text space and never
 *	have any overlap (the two end cases, above).
 */
void            asgnsamples()
{
  int             j;
  UNIT            ccnt;
  double          a_time;
  unsigned long   pcl,
                  pch;
  int             i;
  unsigned long   overlap;
  unsigned long   svalue0,
                  svalue1;

 /* read samples and assign to namelist symbols */
  scale = highpc - lowpc;
  scale /= nsamples;
  alignentries();
  for (i = 0, j = 1; i < nsamples; i++) {
    ccnt = samples[i];
    if (ccnt == 0)
      continue;
    pcl = lowpc + scale * i;
    pch = lowpc + scale * (i + 1);
    a_time = ccnt;
#ifdef DEBUG
    if (debug & SAMPLEDEBUG) {
      printf("[asgnsamples] pcl 0x%lx pch 0x%lx ccnt %d\n",
	     pcl, pch, ccnt);
    }
#endif					/* DEBUG */
    totime += a_time;
    for (j = j - 1; j < nname; j++) {
      svalue0 = nl[j].svalue;
      svalue1 = nl[j + 1].svalue;
 /*
  *	if high end of tick is below entry address,
  *	go for next tick.
  */
      if (pch < svalue0)
	break;
 /*
  *	if low end of tick into next routine,
  *	go for next routine.
  */
      if (pcl >= svalue1)
	continue;
      overlap = min(pch, svalue1) - max(pcl, svalue0);
      if (overlap > 0) {
#ifdef DEBUG
	if (debug & SAMPLEDEBUG) {
	  printf("[asgnsamples] (0x%lx->0x%lx-0x%lx) %s gets %f ticks %ld overlap\n",
		 nl[j].value / sizeof(UNIT), svalue0, svalue1,
		 nl[j].name,
		 overlap * a_time / scale, overlap);
	}
#endif					/* DEBUG */
	nl[j].time += overlap * a_time / scale;
      }
    }
  }
#ifdef DEBUG
  if (debug & SAMPLEDEBUG) {
    printf("[asgnsamples] totime %f\n", totime);
  }
#endif					/* DEBUG */
}

/* ------------------------------------------------------------------------ */

unsigned long   min(a, b)
unsigned long   a,
                b;
{
  if (a < b)
    return (a);
  return (b);
}

/* ------------------------------------------------------------------------ */
unsigned long   max(a, b)
unsigned long   a,
                b;
{
  if (a > b)
    return (a);
  return (b);
}

/* ------------------------------------------------------------------------ */
 /*
  *	calculate scaled entry point addresses (to save time in asgnsamples),
  *	and possibly push the scaled entry points over the entry mask,
  *	if it turns out that the entry point is in one bucket and the code
  *	for a routine is in the next bucket.
  */
void            alignentries()
{
  struct nl      *nlp;
  unsigned long   bucket_of_entry;
  unsigned long   bucket_of_code;

  for (nlp = nl; nlp < npe; nlp++) {
    nlp->svalue = nlp->value / sizeof(UNIT);
    bucket_of_entry = (nlp->svalue - lowpc) / scale;
    bucket_of_code = (nlp->svalue + UNITS_TO_CODE - lowpc) / scale;
    if (bucket_of_entry < bucket_of_code) {
#ifdef DEBUG
      if (debug & SAMPLEDEBUG) {
	printf("[alignentries] pushing svalue 0x%lx to 0x%lx\n",
	       nlp->svalue, nlp->svalue + UNITS_TO_CODE);
      }
#endif					/* DEBUG */
      nlp->svalue += UNITS_TO_CODE;
    }
  }
}

/* ------------------------------------------------------------------------ */
