/*
** uClinux kernel api test module for Brecis [SP]olo chip security engine.
**
** Copyright 2002 Brecis Communictaions
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <linux/proc_fs.h>
#include "brecis/msp_sec.h"
#include "brecis/msp_sec_kern.h"
#include "brecis/msp_secv2.h"

static void do_tests(void) ;

/*------------------------------------------------------------
** Kernel interface debugging/testing
*/


static int kernel_intface_test(unsigned long arg) ;

#define DEBUG
//#define VERBOSE

#ifdef DEBUG
#define DBG_SEC(a1, a2...)	printk(KERN_INFO "SECKTEST " a1, ##a2)
#else
#define DBG_SEC(a...)
#endif

#ifdef VERBOSE
#define VPRINTK(a1, a2...)	printk(KERN_INFO "SECKTEST " a1, ##a2)
#else
#define VPRINTK(a...)
#endif

int err_count ;
int success_count ;
int init_count ;
int init_err ;

#ifdef VERBOSE
#define COUNT_ERROR(s, exp, res, sz) { \
	err_count++ ; \
	printk(KERN_INFO "\nSECKTEST Error in %s ; expected:\n", s) ; \
	do_dump(exp, sz) ; \
	printk(KERN_INFO "SECKTEST actual results:\n") ; \
	do_dump(res, sz) ; \
	printk(KERN_INFO "\n") ; \
}
#else
#define COUNT_ERROR(s, exp, res, sz) \
	err_count++ ; \
	printk(KERN_INFO "SECKTEST Error in %s\n", s) ;
#endif

#ifdef VERBOSE
#define COUNT_SUCCESS(s) { \
	success_count++ ; \
	printk(KERN_INFO "SECKTEST %s passed\n", s) ; \
} 
#else
#define COUNT_SUCCESS(s) success_count++ ;
#endif

#define RESULT_CMP(s, exp, res, sz) { \
	if (memcmp(exp, res, sz)) { COUNT_ERROR(s, exp, res, sz) ; } \
	else { COUNT_SUCCESS(s) } \
}

static int timer_set ;
static struct timer_list tl ;

static struct proc_dir_entry *proc_entry ;
static int sectest_proc_register(void) ;
static void sectest_proc_unregister(void) ;

/*************************************************************
** sectest_open
**
** prepare to do operations with the security engine tester
*************************************************************/

static int sectest_open(struct inode * inode, struct file * file)
{
	DBG_SEC("driver open\n") ;
	return 0 ;
}

/*************************************************************
** sectest_close
**
** finish operations with the security engine tester
*************************************************************/

static int sectest_close(struct inode * inode, struct file * file)
{
	DBG_SEC("driver close\n") ;
	return 0 ;
}


/*************************************************************
** sectest_ioctl
**
** do something with the security engine tester
*************************************************************/

static int sectest_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{

	DBG_SEC("driver ioctl %d\n", cmd) ;

	if (cmd == 42)
	{
		return kernel_intface_test(arg) ;
	}
	
	DBG_SEC("bad ioctl\n") ;
	return -EINVAL ;
}



/*************************************************************
** various structures dealing with registering the driver with
** the kernel.
*************************************************************/

static struct file_operations sectest_fops = {
	owner:    THIS_MODULE,
	open:     sectest_open,
	release:  sectest_close,
	ioctl:    sectest_ioctl,
} ;

static struct miscdevice sectest_miscdev = {
	minor:        MISC_DYNAMIC_MINOR,
	name:         "msp_sec_test",
	fops:         &sectest_fops,
} ;
	


/*************************************************************
** sectest_init
**
** Initialize the hardware, and install the driver.
**
*************************************************************/

static int __init sectest_init(void)
{
	int rc ;
	
	rc = misc_register(&sectest_miscdev) ;

	DBG_SEC("misc_register returns %d\n", rc) ;

	sectest_proc_register() ;

	do_tests() ;
#if 1
	kernel_intface_test(1) ;
#endif

	DBG_SEC("driver test installed\n") ;

	return 0 ;
}

static void __exit sectest_exit(void)
{
	DBG_SEC("driver test removed\n") ;

	if (timer_set)
		del_timer(&tl) ;
	timer_set = 0 ;

	misc_deregister(&sectest_miscdev) ;

	sectest_proc_unregister() ;
}

static int
proc_calc_metrics(char *page, char **start, off_t off,
		  int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int sectest_read_proc(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{
	int len ;
	len = sprintf(page, "mknod /var/sectest c %d %d\n",
		      MISC_MAJOR,
		      sectest_miscdev.minor) ;
	return proc_calc_metrics(page, start, off, count, eof, len) ;
}

static int sectest_proc_register(void)
{
	proc_entry = create_proc_read_entry("sectest", 0, NULL,
					    sectest_read_proc, NULL) ;
	return ( proc_entry != 0 ) ;
}

static void sectest_proc_unregister(void)
{
	if (!proc_entry) return ;

	remove_proc_entry(proc_entry->name, proc_entry->parent) ;
	proc_entry = 0 ;
}

module_init(sectest_init) ;
module_exit(sectest_exit) ;

MODULE_LICENSE("GPL") ;

/*************************************************************
**
** KERNEL INTERFACE TESTING
**
*/

static int testenabled ;

static void
do_dump(char *buf, int len)
{
	int i ;
	for ( i = 0 ; i < len ; i++ )
	{
		switch(i % 16)
		{
		case 0:
			printk("%04x  %02x ", i, buf[i] & 0xff) ;
			break;
		case 7:
			printk("%02x - ", buf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", buf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", buf[i] & 0xff) ;
			break ;
		}
	}
	if (i%16 != 0) printk("\n") ;
}

/*------------------------------------------------------------
**
** General testing structures...
**
*/

struct databuf {
	int sz ;
	char *buf ;
} ;

typedef struct general_test
{
	char *description ;
	MSP_SEC2_SA *sa ;
	int we_control ;
	int nsrc ;
	struct databuf *dbuf ;
	int ndest ;
	struct databuf *ebuf ;
} general_test ;

struct general_test_results
{
	general_test *test ;
	char *rbuf[0] ;
} ;

#define ASIZEOF(x)  (sizeof(x) / sizeof(x[0]))

#define DBENTRY(x) {sizeof(x), x}
/* DBENTRYz skips null for null terminated strings */
#define DBENTRYz(x) {sizeof(x)-1, x}

#define GEN_TEST(TST, DESCR, WE_CTRL) \
general_test TST##_test = { \
	DESCR, \
	&TST##_SA, \
	WE_CTRL, \
 	ASIZEOF(TST##_src), \
	TST##_src, \
	ASIZEOF(TST##_exp), \
	TST##_exp \
}

/*------------------------------------------------------------
**
** DES tests.
**
*/

static char Nist1Src[] =
	{ 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
static char Nist1Exp[] =
	{ 0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15 };
static MSP_SEC2_SA nist1_SA = {
	SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0x01234567, 0x89abcdef,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0x90abcdef, 0x12345678,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
struct databuf nist1_src[] = {
	DBENTRY(Nist1Src)
} ;
struct databuf nist1_exp[] = {
	DBENTRY(Nist1Exp)
} ;

GEN_TEST(nist1, "DES NIST test 1", 0) ;

static char Nist2Src[] =
	{ 0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15 };
static char Nist2Exp[] =
	{ 0x81, 0xce, 0xd6, 0xad, 0x96, 0x4d, 0xde, 0xce } ;
static MSP_SEC2_SA nist2_SA = {
	SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0,0, 
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0x90abcdef, 0x12345678,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
struct databuf nist2_src[] = {
	DBENTRY(Nist2Src)
} ;
struct databuf nist2_exp[] = {
	DBENTRY(Nist2Exp)
} ;

GEN_TEST(nist2, "DES NIST test 2", 0) ;

/* ECB test */
static char ECB1Src[] =
	{ 0x10, 0x01, 0x02, 0x03,  0x14, 0x05, 0x06, 0x07,
	  0x18, 0x09, 0x0a, 0x0b,  0x1c, 0x0d, 0x0e, 0x0f };
static char ECB1Exp[] =
	{ 0xe1, 0x4a, 0x6c, 0xf8, 0xc1, 0xe7, 0x65, 0x7d, 
	  0x10, 0x5d, 0x53, 0xad, 0x3e, 0xe0, 0xd3, 0x20 };
static MSP_SEC2_SA ecb1_SA = {
	SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0xecb10001, 0x124efffe, 
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
struct databuf ecb1_src[] = {
	DBENTRY(ECB1Src)
} ;
struct databuf ecb1_exp[] = {
	DBENTRY(ECB1Exp)
} ;

GEN_TEST(ecb1, "ECB test 1 (DES, 16bytes)", 0) ;


static char ecb2Src[] =
	{ 0x20, 0x01, 0x02, 0x03,  0x24, 0x05, 0x06, 0x07,
	  0x28, 0x09, 0x0a, 0x0b,  0x2c, 0x0d, 0x0e, 0x0f };
struct databuf ecb2_src[] = {
	DBENTRY(ecb2Src)
} ;
static char ecb2Exp[] =
	{ 0x0e, 0x4e, 0xcb, 0xe0, 0x22, 0x67, 0xc1, 0x2a, 
	  0x33, 0xf3, 0xf2, 0xbd, 0x51, 0x74, 0xd3, 0xf6 };
struct databuf ecb2_exp[] = {
	DBENTRY(ecb2Exp)
} ;
static MSP_SEC2_SA ecb2_SA = {
	SAFLG_MODE_CRYPT | SAFLG_DES_K1_DECRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0xecb10702, 0x134ef8fd,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(ecb2, "ECB test 2 (DES DECRYPT, 16 bytes)", 0) ;

static char ecb3Src1[] =
	{ 0x80, 0x01, 0x02, 0x03,   0x84, 0x05, 0x06, 0x07,
	  0x88, 0x09, 0x0a, 0x0b,   0x8c, 0x0d, 0x0e, 0x0f };
static char ecb3Src2[] =
	{ 0x80, 0x11, 0x12, 0x13,   0x84, 0x15, 0x16, 0x17,
	  0x88, 0x19, 0x1a, 0x1b,   0x7c, 0x1d, 0x1e, 0x1f };
static char ecb3Src3[] =
	{ 0x80, 0x21, 0x22, 0x23,   0x84, 0x25, 0x26, 0x27,
	  0x88, 0x29, 0x2a, 0x2b,   0x8c, 0x2d, 0x2e, 0x2f };

struct databuf ecb3_src[] = {
	DBENTRY(ecb3Src1),
	DBENTRY(ecb3Src2),
	DBENTRY(ecb3Src3)
} ;
static char ecb3Exp[] =
	{ 0x9e, 0xef, 0x3c, 0x62,   0xa0, 0xc3, 0x54, 0x73,
	  0x77, 0xf5, 0x02, 0x85,   0xa0, 0xdd, 0x32, 0xc4,
	  0x04, 0x1d, 0x71, 0xa6,   0x82, 0xd5, 0x3e, 0x92,
	  0xcb, 0xe9, 0x8e, 0x7d,   0xef, 0xf8, 0x6f, 0xa3,
	  0x8d, 0x28, 0x09, 0x27,   0x1a, 0x90, 0xe4, 0xac,
	  0x29, 0xfa, 0x8e, 0x35,   0x1d, 0x0b, 0xf6, 0x48 } ;
struct databuf ecb3_exp[] = {
	DBENTRY(ecb3Exp)
} ;
static MSP_SEC2_SA ecb3_SA = {
	SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0xecb10001, 0x134efffe,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(ecb3, "ECB test 3 (DES ENCRYPT, 48 bytes)", 0) ;



static char ecb4Src1[] =
	{ 0x70, 0x01, 0x02, 0x03,  0x74, 0x05, 0x06, 0x07,
	  0x78, 0x09, 0x0a, 0x0b,  0x7c, 0x0d, 0x0e, 0x0f };
static char ecb4Src2[] =
	{ 0x70, 0x11, 0x12, 0x13,  0x74, 0x15, 0x16, 0x17,
	  0x78, 0x19, 0x1a, 0x1b,  0x7c, 0x1d, 0x1e, 0x1f };
static char ecb4Src3[] =
	{ 0x70, 0x21, 0x22, 0x23,  0x74, 0x25, 0x26, 0x27,
	  0x78, 0x29, 0x2a, 0x2b,  0x7c, 0x2d, 0x2e, 0x2f };
struct databuf ecb4_src[] = {
	DBENTRY(ecb4Src1),
	DBENTRY(ecb4Src2),
	DBENTRY(ecb4Src3)
} ;
static char ecb4Exp[] =
	{ 0x8e, 0x65, 0x3a, 0x30,  0x04, 0x1f, 0xf9, 0x5b,
	  0x58, 0x2d, 0xdf, 0x02,  0x38, 0x43, 0xe2, 0x63,
	  0x04, 0x85, 0x86, 0x36,  0x9d, 0xfd, 0xf6, 0x92,
	  0xa7, 0xbc, 0xde, 0x8a,  0x08, 0xcb, 0xfe, 0x54,
	  0x1b, 0xfe, 0xa0, 0x88,  0x39, 0xe2, 0xc3, 0x8a,
	  0x46, 0x90, 0x0b, 0x0a,  0x29, 0x3b, 0x82, 0x9b } ;
struct databuf ecb4_exp[] = {
	DBENTRY(ecb4Exp)
} ;
static MSP_SEC2_SA ecb4_SA = {
	SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0x71717171, 0x72727272,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(ecb4, "ECB test 4 (DES ENCRYPT, 48 bytes)", 0) ;

static char md5_1Src[] =
	{ 0x55, 0x55, 0x55, 0x55, 0xaa, 0xaa, 0xaa, 0xaa,
	  0x33, 0x33, 0x33, 0x33, 0xcc, 0xcc, 0xcc, 0xcc,
	  0x66, 0x66, 0x66, 0x66, 0x99, 0x99, 0x99, 0x99,
	  0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	  0x55, 0x55, 0x55, 0x55, 0xaa, 0xaa, 0xaa, 0xaa,
	  0x33, 0x33, 0x33, 0x33, 0xcc, 0xcc, 0xcc, 0xcc,
	  0x66, 0x66, 0x66, 0x66, 0x80, 0x00, 0x00, 0x00,
	  0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
struct databuf md5_1_src[] = {
	DBENTRY(md5_1Src)
} ;
static char md5_1Exp[] =
	{ 0x3f, 0x56, 0x62, 0x34, 0x16, 0xfc, 0x63, 0x7d,
	  0x7c, 0x81, 0x2b, 0xca, 0xef, 0x8a, 0xb4, 0xae };
struct databuf md5_1_exp[] = {
	DBENTRY(md5_1Exp)
} ;
static MSP_SEC2_SA md5_1_SA = {
	SAFLG_MODE_HASH | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_1, "MD5 Test 1", 0) ;


static char md5_2Src1[] =
	{ 0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f };
static char md5_2Src2[] =
	{ 0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f };
static char md5_2Src3[] =
	{ 0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x00,
	  0x80, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x10, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00 };
struct databuf md5_2_src[] = {
	{ 191, &md5_2Src1[0] },
	{ 1,   &md5_2Src1[191] },
	{ 3,   &md5_2Src2[0] },
	{ 189, &md5_2Src2[3] },
	{ 192, &md5_2Src3[0] }
} ;
static char md5_2Exp[] =
	{ 0x4D, 0xB2, 0x88, 0x6A, 0xF5, 0x79, 0x83, 0xBB, 
	  0xFD, 0xFB, 0x77, 0xA3, 0xE6, 0xBD, 0x0C, 0xA5 };
struct databuf md5_2_exp[] = {
	DBENTRY(md5_2Exp)
} ;
static MSP_SEC2_SA md5_2_SA = {
	SAFLG_MODE_HASH | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_2, "MD5 Test 2", 0) ;


/* md5 test 3 uses some data from md5 test 2 above */
static char md5_3Src3[] =
	{ 0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x3f,
	  0x40, 0x01, 0x02, 0x03,   0x44, 0x05, 0x06, 0x07,
	  0x48, 0x09, 0x0a, 0x0b,   0x4c, 0x0d, 0x0e, 0x0f,
	  0x40, 0x11, 0x12, 0x13,   0x44, 0x15, 0x16, 0x17,
	  0x48, 0x19, 0x1a, 0x1b,   0x4c, 0x1d, 0x1e, 0x1f,
	  0x40, 0x21, 0x22, 0x23,   0x44, 0x25, 0x26, 0x27,
	  0x48, 0x29, 0x2a, 0x2b,   0x4c, 0x2d, 0x2e, 0x2f,
	  0x40, 0x31, 0x32, 0x33,   0x44, 0x35, 0x36, 0x37,
	  0x48, 0x39, 0x3a, 0x3b,   0x4c, 0x3d, 0x3e, 0x00,
	  0x80, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x00, 0x00, 
	  0x00, 0x00, 0x00, 0x00,   0x00, 0x00, 0x10, 0x00 };
struct databuf md5_3_src[] = {
	{ 12,  &md5_2Src1[0] },
	{ 180, &md5_2Src1[12] },
	{ 61,  &md5_2Src2[0] },
	{ 131, &md5_2Src2[61] },
	{ 19,  &md5_3Src3[0] },
	{ 63,  &md5_3Src3[19] },
	{ 110, &md5_3Src3[82] }
} ;
static char md5_3Exp[] =
	{ 0x99, 0xda, 0xe1, 0xef, 0x30, 0xc9, 0xe3, 0xa4, 0x0b, 0x8d, 
	  0x84, 0x07, 0x0b, 0x6f, 0xbe, 0xf0, 0xf0, 0x00, 0xed, 0x58 };
struct databuf md5_3_exp[] = {
	DBENTRY(md5_3Exp)
} ;
static MSP_SEC2_SA md5_3_SA = {
	SAFLG_MODE_HASH | SAFLG_SHA1, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_3, "MD5 Test 3 (truly, a SHA1 test)", 0) ;


static char md5_4Src[] = "a" ;
struct databuf md5_4_src[] = {
	DBENTRYz(md5_4Src)
} ;
static char md5_4Exp[] =
	{ 0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8, 
	  0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61};
struct databuf md5_4_exp[] = {
	DBENTRY(md5_4Exp)
} ;
static MSP_SEC2_SA md5_4_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_4, "MD5 Test 4", 0) ;


static char md5_5Src[] = "abc" ;
struct databuf md5_5_src[] = {
	DBENTRYz(md5_5Src)
} ;
static char md5_5Exp[] =
	{ 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0, 
	  0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72};
struct databuf md5_5_exp[] = {
	DBENTRY(md5_5Exp)
} ;
static MSP_SEC2_SA md5_5_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_5, "MD5 Test 5", 0) ;


static char md5_6Src[] = "message digest" ;
struct databuf md5_6_src[] = {
	DBENTRYz(md5_6Src)
} ;
static char md5_6Exp[] =
	{ 0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d, 
	  0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0};
struct databuf md5_6_exp[] = {
	DBENTRY(md5_6Exp)
} ;
static MSP_SEC2_SA md5_6_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_6, "MD5 Test 6", 0) ;


static char md5_7Src[] = "abcdefghijklmnopqrstuvwxyz" ;
struct databuf md5_7_src[] = {
	DBENTRYz(md5_7Src)
} ;
static char md5_7Exp[] =
	{ 0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00, 
	  0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b};
struct databuf md5_7_exp[] = {
	DBENTRY(md5_7Exp)
} ;
static MSP_SEC2_SA md5_7_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_7, "MD5 Test 7", 0) ;


static char md5_8Src[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"; 
struct databuf md5_8_src[] = {
	DBENTRYz(md5_8Src)
} ;
static char md5_8Exp[] =
	{ 0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5, 
	  0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f};
struct databuf md5_8_exp[] = {
	DBENTRY(md5_8Exp)
} ;
static MSP_SEC2_SA md5_8_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_8, "MD5 Test 8", 0) ;


static char md5_9Src[] = "12345678901234567890123456789012345678901234567890123456789012345678901234567890";
struct databuf md5_9_src[] = {
	DBENTRYz(md5_9Src)
} ;
static char md5_9Exp[] =
	{ 0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55, 0xac, 
	  0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a};
struct databuf md5_9_exp[] = {
	DBENTRY(md5_9Exp)
} ;
static MSP_SEC2_SA md5_9_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_9, "MD5 Test 9", 0) ;


static char md5_10Src[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789A"; 
struct databuf md5_10_src[] = {
	DBENTRYz(md5_10Src)
} ;
static char md5_10Exp[] =
	{ 0x42, 0x64, 0x0a, 0xf7, 0x8b, 0x2a, 0x7d, 0xeb, 
	  0x4b, 0x57, 0x15, 0xb0, 0x0e, 0xf3, 0x10, 0x73};
struct databuf md5_10_exp[] = {
	DBENTRY(md5_10Exp)
} ;
static MSP_SEC2_SA md5_10_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_MD5, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(md5_10, "MD5 Test 10", 0) ;


static char sha1_1Src[] =
	{ 0x61, 0x62, 0x63, 0x80,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x00,
	  0x00, 0x00, 0x00, 0x00,  0x00, 0x00, 0x00, 0x18 };
struct databuf sha1_1_src[] = {
	DBENTRY(sha1_1Src)
} ;
static char sha1_1Exp[] =
	{ 0xa9, 0x99, 0x3e, 0x36,  0x47, 0x06, 0x81, 0x6a,
	  0xba, 0x3e, 0x25, 0x71,  0x78, 0x50, 0xc2, 0x6c,
	  0x9c, 0xd0, 0xd8, 0x9d };
struct databuf sha1_1_exp[] = {
	DBENTRY(sha1_1Exp)
} ;
static MSP_SEC2_SA sha1_1_SA = {
	SAFLG_MODE_HASH | SAFLG_SHA1, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(sha1_1, "SHA-1 Test 1", 0) ;


static char sha1_2Src[] = "abc" ;
struct databuf sha1_2_src[] = {
	DBENTRYz(sha1_2Src)
} ;
static char sha1_2Exp[] =
	{ 0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A,
	  0xBA, 0x3E, 0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C,
	  0x9C, 0xD0, 0xD8, 0x9D };
struct databuf sha1_2_exp[] = {
	DBENTRY(sha1_2Exp)
} ;
static MSP_SEC2_SA sha1_2_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_SHA1, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(sha1_2, "SHA-1 Test 2", 0) ;


static char sha1_3Src[] = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
struct databuf sha1_3_src[] = {
	DBENTRYz(sha1_3Src)
} ;
static char sha1_3Exp[] =
	{ 0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E,
	  0xBA, 0xAE, 0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5,  
	  0xE5, 0x46, 0x70, 0xF1 };
struct databuf sha1_3_exp[] = {
	DBENTRY(sha1_3Exp)
} ;
static MSP_SEC2_SA sha1_3_SA = {
	SAFLG_MODE_HASH_PAD | SAFLG_SHA1, /* flags */
	0, 		/* esp_spi */
	0,		/* esp_sequence */
	{0,0,0,0,0},	/* hash_chain_a */
	{0, 0,
	 0,0,
	 0,0,
	 0,0 }, 	/* crypt_keys */
	{0,0,0,0,0},	/* hash_chain_b */
	{0, 0},		/* hash_init_len */
	{0, 0,
	 0, 0}, 	/* crypt_iv */
	{0,0,0,0,0}	/* proto_ip */
} ;
GEN_TEST(sha1_3, "SHA-1 Test 3", 0) ;



static char espout_1Src[] =
	{ 0x45, 0x00, 0x00, 0x54, 0x00, 0x00, 0x40, 0x00,
	  0x40, 0x01, 0x8d, 0x30, 0xc7, 0x56, 0x0f, 0x69,
	  0xc7, 0x56, 0x0f, 0x63, 0x08, 0x00, 0x15, 0x50,
	  0xd8, 0x33, 0x01, 0x00, 0x6f, 0xea, 0xc0, 0x3d,
	  0xe4, 0x50, 0x0a, 0x00, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37 } ;
struct databuf espout_1_src[] = {
	DBENTRY(espout_1Src)
} ;
static char espout_1Exp[] =
	{ 0x12, 0x34, 0xab, 0xcd, 0x55, 0xaa, 0x55, 0xaa,
	  0x11, 0x22, 0x33, 0x44, 0x44, 0x33, 0x22, 0x11,
	  0x39, 0xf8, 0x07, 0x7b, 0xa4, 0xd8, 0xd1, 0x48,
	  0xad, 0xa7, 0x72, 0x07, 0xef, 0x7c, 0xd4, 0x7b,
	  0xbd, 0xaf, 0x9e, 0x1f, 0xd0, 0x60, 0xde, 0xd3,
	  0xda, 0xb5, 0xd5, 0xb5, 0x0d, 0x75, 0x8b, 0x5d,
	  0xe0, 0xc5, 0xda, 0xaf, 0x85, 0xbb, 0x9b, 0x49,
	  0x57, 0x28, 0x0a, 0xf2, 0x0e, 0xad, 0xfe, 0x54,
	  0xcd, 0xaa, 0xeb, 0x24, 0xbc, 0x2d, 0xa6, 0x16,
	  0xb7, 0x06, 0x38, 0xed, 0x40, 0xc7, 0x75, 0x67,
	  0x35, 0x57, 0x9b, 0xa3, 0x94, 0xa1, 0xef, 0x91,
	  0x9d, 0x92, 0xa1, 0x26, 0x99, 0x73, 0x43, 0x3c,
	  0x93, 0xe6, 0xee, 0xd8, 0xd7, 0x8d, 0x9b, 0x31,
	  0xc6, 0xdb, 0xd8, 0x2d, 0xdb, 0x2f, 0xd7, 0xe7,
	  0xa9, 0x12, 0x56, 0xed } ;
struct databuf espout_1_exp[] = {
	DBENTRY(espout_1Exp)
} ;
static MSP_SEC2_SA espout_1_SA = {
	SAFLG_MODE_ESP_OUT
	| SAFLG_3DES
	| SAFLG_CBC_ENCRYPT
	| SAFLG_DES_K2_DECRYPT
	| SAFLG_SHA1_96
	| SAFLG_CV,		/* flags */
	0x1234abcd,		/* esp_spi */
	0x55aa55aa,		/* esp_sequence */
	{0x064C662A,
	0xCC6ED1B0,
	0x6CFA9AB0,
	0x042F137F,
	0xCEA570EB},		/* hash_chain_a */
	{0x11111111, 0x22222222,
	 0x33333333, 0x44444444,
	 0x55555555, 0x66666666,
	 0,0 },			/* crypt_keys */
	{0xD62ACCCC,
	0x7E6A98B3,
	0xDF015B02,
	0xD885C3E0,
	0x9BD184CB},		/* hash_chain_b */
	{0, 0x200},		/* hash_init_len */
	{0x11223344, 0x44332211,
	 0, 0},			/* crypt_iv */
	{0,0,0,0,0}		/* proto_ip */
} ;
GEN_TEST(espout_1, "Esp Outgpoing Test 1", 0x02040000) ;



static char espin_1Src[] =
	{ 0x12, 0x34, 0xab, 0xcd, 0x55, 0xaa, 0x55, 0xaa,
	  0x11, 0x22, 0x33, 0x44, 0x44, 0x33, 0x22, 0x11,
	  0x39, 0xf8, 0x07, 0x7b, 0xa4, 0xd8, 0xd1, 0x48,
	  0xad, 0xa7, 0x72, 0x07, 0xef, 0x7c, 0xd4, 0x7b,
	  0xbd, 0xaf, 0x9e, 0x1f, 0xd0, 0x60, 0xde, 0xd3,
	  0xda, 0xb5, 0xd5, 0xb5, 0x0d, 0x75, 0x8b, 0x5d,
	  0xe0, 0xc5, 0xda, 0xaf, 0x85, 0xbb, 0x9b, 0x49,
	  0x57, 0x28, 0x0a, 0xf2, 0x0e, 0xad, 0xfe, 0x54,
	  0xcd, 0xaa, 0xeb, 0x24, 0xbc, 0x2d, 0xa6, 0x16,
	  0xb7, 0x06, 0x38, 0xed, 0x40, 0xc7, 0x75, 0x67,
	  0x35, 0x57, 0x9b, 0xa3, 0x94, 0xa1, 0xef, 0x91,
	  0x9d, 0x92, 0xa1, 0x26, 0x99, 0x73, 0x43, 0x3c,
	  0x93, 0xe6, 0xee, 0xd8, 0xd7, 0x8d, 0x9b, 0x31,
	  0xc6, 0xdb, 0xd8, 0x2d, 0xdb, 0x2f, 0xd7, 0xe7,
	  0xa9, 0x12, 0x56, 0xed } ;
struct databuf espin_1_src[] = {
	{ sizeof(espin_1Src) - 12, &espin_1Src[0] },
	{ 12, &espin_1Src[sizeof(espin_1Src) - 12] }
} ;
static char espin_1Exp[] =
	{ 0x45, 0x00, 0x00, 0x54, 0x00, 0x00, 0x40, 0x00,
	  0x40, 0x01, 0x8d, 0x30, 0xc7, 0x56, 0x0f, 0x69,
	  0xc7, 0x56, 0x0f, 0x63, 0x08, 0x00, 0x15, 0x50,
	  0xd8, 0x33, 0x01, 0x00, 0x6f, 0xea, 0xc0, 0x3d,
	  0xe4, 0x50, 0x0a, 0x00, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x01, 0x02, 0x02, 0x04, 
	  0xc6, 0xdb, 0xd8, 0x2d, 0xdb, 0x2f, 0xd7, 0xe7,
	  0xa9, 0x12, 0x56, 0xed } ;
struct databuf espin_1_exp[] = {
	DBENTRY(espin_1Exp)
} ;
static MSP_SEC2_SA espin_1_SA = {
	SAFLG_MODE_ESP_IN
	| SAFLG_3DES
	| SAFLG_CBC_DECRYPT
	| SAFLG_DES_K1_DECRYPT
	| SAFLG_DES_K3_DECRYPT
	| SAFLG_SHA1_96
	| SAFLG_CV,		/* flags */
	0x1234abcd,		/* esp_spi */
	0x55aa55aa,		/* esp_sequence */
	{0x064C662A,
	0xCC6ED1B0,
	0x6CFA9AB0,
	0x042F137F,
	0xCEA570EB},		/* hash_chain_a */
	{0x55555555, 0x66666666,
	 0x33333333, 0x44444444,
	 0x11111111, 0x22222222,
	 0,0 },			/* crypt_keys */
	{0xD62ACCCC,
	0x7E6A98B3,
	0xDF015B02,
	0xD885C3E0,
	0x9BD184CB},		/* hash_chain_b */
	{0, 0x200},		/* hash_init_len */
	{0x11223344, 0x44332211,
	 0, 0},			/* crypt_iv */
	{0,0,0,0,0}		/* proto_ip */
} ;
GEN_TEST(espin_1, "Esp incomming Test 1", 0x02040000) ;

static char espout_2Src[] =
	{ 0x45, 0x00, 0x05, 0xbb, 0x00, 0x00, 0x40, 0x00,
	  0x40, 0x01, 0x87, 0xc9, 0xc7, 0x56, 0x0f, 0x69,
	  0xc7, 0x56, 0x0f, 0x63, 0x08, 0x00, 0xb7, 0xb6,
	  0x46, 0x37, 0x02, 0x00, 0xf9, 0xf5, 0xc0, 0x3d,
	  0xbd, 0x35, 0x03, 0x00, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	  0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
	  0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
	  0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	  0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
	  0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
	  0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	  0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
	  0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
	  0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
	  0xfc, 0xfd, 0xfe, 0xff, 0x00, 0x01, 0x02, 0x03,
	  0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	  0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
	  0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
	  0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	  0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
	  0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
	  0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	  0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
	  0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
	  0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
	  0xfc, 0xfd, 0xfe, 0xff, 0x00, 0x01, 0x02, 0x03,
	  0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	  0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
	  0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
	  0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	  0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
	  0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
	  0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	  0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
	  0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
	  0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
	  0xfc, 0xfd, 0xfe, 0xff, 0x00, 0x01, 0x02, 0x03,
	  0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	  0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
	  0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
	  0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	  0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
	  0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
	  0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	  0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
	  0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
	  0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
	  0xfc, 0xfd, 0xfe, 0xff, 0x00, 0x01, 0x02, 0x03,
	  0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	  0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	  0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
	  0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
	  0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	  0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
	  0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb,
	  0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	  0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
	  0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3,
	  0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
	  0xfc, 0xfd, 0xfe, 0xff, 0x00, 0x01, 0x02, 0x03,
	  0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	  0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
	  0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
	  0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	  0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
	  0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
	  0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
	  0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
	  0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	  0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
	  0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
	  0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	  0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	  0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73,
	  0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b,
	  0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	  0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	  0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93,
	  0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
	  0x9c, 0x9d, 0x9e } ;
struct databuf espout_2_src[] = {
	DBENTRY(espout_2Src)
} ;
static char espout_2Exp[] =
	{ 0x12, 0x34, 0xab, 0xcd, 0x55, 0xaa, 0x55, 0xaa,
	  0x11, 0x22, 0x33, 0x44, 0x44, 0x33, 0x22, 0x11,
	  0x40, 0x74, 0xf7, 0x31, 0xbd, 0xb4, 0x8d, 0x49,
	  0x7f, 0x12, 0x0c, 0x98, 0x51, 0x78, 0x29, 0xaa,
	  0x15, 0x32, 0xb8, 0x5e, 0x62, 0x2c, 0xaf, 0x0e,
	  0x2c, 0xf6, 0x31, 0xd0, 0x1e, 0xa9, 0x1f, 0x2d,
	  0x19, 0x3d, 0xcc, 0xcd, 0x6f, 0x6b, 0xfa, 0xe7,
	  0xd4, 0xc5, 0x3c, 0x14, 0x5d, 0x8f, 0x57, 0xd2,
	  0xd9, 0xf7, 0x4f, 0xe5, 0x10, 0xb6, 0x44, 0xe7,
	  0x5e, 0xdc, 0x5f, 0xd6, 0x4d, 0x94, 0x5c, 0xde,
	  0x67, 0x2b, 0x2e, 0xec, 0x18, 0xa4, 0x48, 0x01,
	  0x45, 0x79, 0x18, 0xcc, 0xb5, 0x5f, 0x79, 0x0c,
	  0xd5, 0x1d, 0x0f, 0x4e, 0xb4, 0x3e, 0xb2, 0x50,
	  0x24, 0x2f, 0xab, 0xba, 0xfc, 0x34, 0x48, 0x41,
	  0x8a, 0x58, 0xd0, 0xf7, 0x56, 0xaf, 0x19, 0xb0,
	  0xc4, 0x03, 0x03, 0x09, 0x54, 0x14, 0xca, 0x1e,
	  0x32, 0x2e, 0xb6, 0x82, 0xbd, 0xff, 0xc2, 0xe9,
	  0x49, 0x30, 0x64, 0xf4, 0x7c, 0x4d, 0x09, 0x39,
	  0xd2, 0xa8, 0x80, 0x24, 0xa3, 0x94, 0xa4, 0xe2,
	  0x4c, 0x23, 0x7e, 0x40, 0x5c, 0x34, 0xd7, 0x60,
	  0x9e, 0x06, 0xa0, 0xe5, 0x13, 0x93, 0x0c, 0x14,
	  0x92, 0xfc, 0xcc, 0x28, 0x37, 0x66, 0x39, 0x8c,
	  0xd9, 0xbb, 0x1f, 0x2b, 0x4b, 0x5e, 0x2a, 0xbd,
	  0x25, 0x90, 0x3b, 0x61, 0x65, 0x2e, 0xb3, 0x8a,
	  0xda, 0x4a, 0x3b, 0xd1, 0x14, 0x8f, 0xd5, 0x24,
	  0x69, 0xd8, 0xba, 0x64, 0x28, 0xa5, 0xe1, 0xdf,
	  0xce, 0xef, 0x03, 0x4f, 0xfe, 0x1b, 0xee, 0x61,
	  0x38, 0xe3, 0x3c, 0xb3, 0xff, 0x4e, 0x5e, 0xb9,
	  0x96, 0xaf, 0xb1, 0xa5, 0x1c, 0xa6, 0xcc, 0x83,
	  0xfc, 0x9a, 0x1c, 0xfd, 0x87, 0xb4, 0x46, 0x6f,
	  0x29, 0x3c, 0xcf, 0x1d, 0xd2, 0xa8, 0xa7, 0xa6,
	  0x92, 0x82, 0xf9, 0xee, 0xe4, 0x8f, 0xc8, 0x74,
	  0x34, 0xdc, 0x89, 0xd6, 0x96, 0x9d, 0xfb, 0xf2,
	  0x80, 0xb0, 0x2c, 0x96, 0x6a, 0x7b, 0x4b, 0x52,
	  0xb9, 0xa6, 0x46, 0xa4, 0x5e, 0x18, 0x4c, 0x36,
	  0xc7, 0x7e, 0x3a, 0x35, 0x9d, 0xb3, 0x0b, 0xc7,
	  0x3f, 0xfc, 0x21, 0xd7, 0x16, 0x74, 0xaf, 0x28,
	  0x3a, 0xb4, 0x13, 0x4b, 0x71, 0x4e, 0x21, 0x8e,
	  0x01, 0x08, 0xea, 0x7b, 0x2d, 0xe5, 0x5f, 0x90,
	  0xe3, 0xd2, 0x94, 0x27, 0x24, 0xa3, 0xa1, 0x63,
	  0xaa, 0x20, 0x5b, 0x90, 0x09, 0xad, 0xd6, 0x90,
	  0xe1, 0xa3, 0x5d, 0x56, 0x6d, 0x95, 0xdf, 0xc3,
	  0x16, 0x20, 0xd7, 0x3d, 0xf9, 0x69, 0x5f, 0x21,
	  0x2f, 0x2e, 0x05, 0x21, 0x1f, 0xcd, 0x93, 0x39,
	  0x57, 0x3e, 0xb7, 0xb5, 0x44, 0xac, 0x2e, 0x7c,
	  0x75, 0x61, 0x1f, 0x99, 0x0b, 0x35, 0x60, 0xf2,
	  0x30, 0x9c, 0x38, 0x6b, 0x62, 0x93, 0xb6, 0x4b,
	  0x6d, 0xc8, 0x76, 0xa2, 0x9e, 0x71, 0x20, 0x52,
	  0x48, 0x7c, 0x58, 0xf3, 0x6e, 0x6a, 0x6b, 0xf4,
	  0x8b, 0x3e, 0xad, 0x8b, 0x02, 0xd0, 0x6f, 0x0b,
	  0x5b, 0x3b, 0xf0, 0x1a, 0xea, 0x52, 0x60, 0xa9,
	  0x37, 0xa0, 0x95, 0x1c, 0x0a, 0x53, 0x4e, 0xdf,
	  0x47, 0x0f, 0x08, 0x1c, 0x4d, 0x01, 0x8b, 0x0e,
	  0xd8, 0x6e, 0x07, 0xa0, 0xba, 0x0f, 0x48, 0xed,
	  0x81, 0x9c, 0x29, 0xb9, 0xe9, 0x0c, 0xe0, 0x71,
	  0x5e, 0xa4, 0x10, 0x46, 0x91, 0xf0, 0x81, 0x84,
	  0x43, 0xd5, 0x78, 0x05, 0xad, 0xb1, 0x13, 0x14,
	  0x69, 0x2e, 0x48, 0xab, 0x65, 0xba, 0x4f, 0x0d,
	  0xc5, 0x85, 0x9d, 0x6d, 0x57, 0xd8, 0x20, 0x21,
	  0xae, 0x1b, 0x2b, 0xdd, 0x4c, 0xed, 0xf3, 0x62,
	  0x09, 0x57, 0xa5, 0x3e, 0x54, 0x92, 0xd6, 0x4d,
	  0xc6, 0x65, 0x88, 0xec, 0x93, 0xb5, 0x18, 0xbd,
	  0x2e, 0xc7, 0x4f, 0x5a, 0x1e, 0x24, 0xc6, 0xe4,
	  0x78, 0xf1, 0xf8, 0x1c, 0xe7, 0x92, 0xee, 0xf4,
	  0x99, 0x37, 0x6f, 0x4c, 0x0e, 0x65, 0x6f, 0x1d,
	  0x9b, 0x32, 0x4b, 0xc9, 0xc3, 0x91, 0x9a, 0x76,
	  0x8d, 0xdb, 0x1f, 0x4f, 0x82, 0x55, 0xec, 0xf4,
	  0xf0, 0x72, 0xa2, 0x32, 0x73, 0xe1, 0x8a, 0x56,
	  0x83, 0x57, 0x74, 0x78, 0x9d, 0x80, 0x50, 0xf5,
	  0xf8, 0x90, 0xa3, 0x84, 0x64, 0x39, 0xe4, 0xdb,
	  0x1a, 0x1e, 0x62, 0x32, 0x96, 0x0b, 0x9a, 0x22,
	  0x94, 0x11, 0x71, 0xc9, 0x7e, 0x3d, 0x65, 0x53,
	  0xf8, 0x98, 0xd7, 0x21, 0xba, 0x2f, 0x27, 0xf4,
	  0x3c, 0xc9, 0xe7, 0xfd, 0x48, 0x22, 0x66, 0x47,
	  0x6a, 0xa5, 0x98, 0xb2, 0x68, 0x50, 0x56, 0x99,
	  0x67, 0xcb, 0x19, 0x38, 0x41, 0xa9, 0x52, 0x91,
	  0x68, 0xcd, 0xf7, 0xf0, 0x4d, 0x4e, 0x5b, 0x15,
	  0xdf, 0x5a, 0xda, 0x73, 0x19, 0xd1, 0x73, 0xe1,
	  0x2b, 0xeb, 0x4f, 0xdc, 0x97, 0x8f, 0xf9, 0x10,
	  0x74, 0xba, 0x19, 0x63, 0x83, 0x58, 0xf5, 0x38,
	  0xa6, 0x15, 0x49, 0x5b, 0xa4, 0xfc, 0x5c, 0xc9,
	  0x29, 0x3b, 0x68, 0x1e, 0x46, 0xbf, 0x32, 0xd7,
	  0x86, 0x2e, 0x7e, 0x6b, 0xef, 0xdd, 0x29, 0xcd,
	  0x04, 0xf2, 0xbf, 0xcb, 0x90, 0x09, 0x5f, 0x8a,
	  0x85, 0x41, 0xac, 0x9d, 0x8c, 0x5a, 0x74, 0x90,
	  0x2c, 0x19, 0xff, 0x3e, 0x52, 0xa9, 0xb6, 0x54,
	  0xb6, 0x32, 0x73, 0xef, 0x72, 0x3c, 0xae, 0x17,
	  0xc5, 0x70, 0x33, 0xc7, 0xfc, 0x67, 0x4a, 0x11,
	  0x61, 0xb0, 0xca, 0x85, 0x5a, 0xf6, 0xcc, 0x3d,
	  0xf6, 0x8e, 0x09, 0xa1, 0xbf, 0x4c, 0x83, 0xe6,
	  0x72, 0x1f, 0x6a, 0x88, 0xaa, 0x04, 0xf7, 0x77,
	  0x5a, 0xd1, 0xa8, 0xa1, 0xea, 0x3e, 0x26, 0xe8,
	  0xab, 0xe8, 0xb6, 0x4e, 0x13, 0xd3, 0x23, 0x13,
	  0xc4, 0xaf, 0xea, 0xe5, 0x0b, 0x9a, 0xa1, 0x43,
	  0x6f, 0x68, 0x71, 0xa0, 0x29, 0x37, 0x2a, 0x4f,
	  0xe4, 0x8e, 0xf5, 0xa3, 0x20, 0x50, 0xf0, 0x32,
	  0x36, 0x58, 0x52, 0x52, 0x9c, 0x84, 0xcc, 0xd1,
	  0x03, 0x63, 0xec, 0xe8, 0xb7, 0xee, 0x4b, 0xa1,
	  0xd3, 0x3f, 0xcd, 0x3f, 0xcc, 0x40, 0xbd, 0xc5,
	  0xda, 0x2a, 0x5f, 0x5f, 0xc0, 0x46, 0xff, 0xb7,
	  0x3c, 0x52, 0x01, 0x89, 0xe6, 0x10, 0x88, 0x7c,
	  0x0a, 0xc4, 0x7e, 0xf3, 0xc2, 0x10, 0xb5, 0xd5,
	  0xc5, 0x21, 0x1a, 0xd5, 0x15, 0x56, 0x8c, 0xc5,
	  0x8d, 0xd7, 0xe5, 0x43, 0xd1, 0xfd, 0x53, 0x8c,
	  0xa8, 0x83, 0xa7, 0x86, 0xf6, 0xef, 0xcf, 0x2f,
	  0xa3, 0x4d, 0x3d, 0x2d, 0xdc, 0xfa, 0xe6, 0x9b,
	  0x00, 0x03, 0xd6, 0xb4, 0x1b, 0x66, 0x34, 0x0a,
	  0x77, 0x76, 0xb3, 0x36, 0x81, 0xd0, 0x32, 0x27,
	  0x89, 0xe7, 0x0d, 0x84, 0xeb, 0xf0, 0x78, 0x42,
	  0xac, 0x8f, 0x5c, 0x18, 0xe9, 0x2a, 0x81, 0x4a,
	  0x42, 0x8f, 0xde, 0xd8, 0xa0, 0x97, 0xb7, 0xd2,
	  0x33, 0x12, 0x03, 0x4c, 0x5e, 0x77, 0x52, 0x6a,
	  0x47, 0x1b, 0x9c, 0xd5, 0x39, 0x37, 0xd9, 0x9f,
	  0xc3, 0x52, 0x0a, 0x11, 0x61, 0xa4, 0x09, 0xee,
	  0xeb, 0x9a, 0x43, 0xa0, 0x50, 0x54, 0xad, 0xd7,
	  0x01, 0x51, 0x41, 0x71, 0xac, 0x2a, 0x3b, 0xc6,
	  0x1a, 0xd2, 0x73, 0x73, 0xc2, 0xd1, 0x67, 0xf2,
	  0x95, 0x97, 0x2a, 0xf1, 0xee, 0xa2, 0x45, 0xe2,
	  0x97, 0xae, 0x7c, 0x92, 0xaf, 0xde, 0xdf, 0xcf,
	  0x94, 0x8c, 0x73, 0x54, 0xb3, 0xe8, 0xdd, 0xdb,
	  0xba, 0x32, 0x71, 0x53, 0x61, 0xa6, 0x6d, 0xf3,
	  0xfa, 0x97, 0xd6, 0xb0, 0x7c, 0x02, 0x8a, 0x21,
	  0x14, 0x72, 0x00, 0xad, 0xc1, 0xfd, 0x82, 0xe0,
	  0xff, 0x11, 0x11, 0x80, 0xc1, 0xae, 0x1e, 0xe6,
	  0x40, 0x2b, 0x1c, 0xe1, 0xff, 0xdd, 0xac, 0xec,
	  0xbd, 0x30, 0xd8, 0x5e, 0xed, 0x36, 0x22, 0x94,
	  0x78, 0xd1, 0x2e, 0xfc, 0x33, 0x8d, 0xbb, 0x8a,
	  0x32, 0xe5, 0xb3, 0x37, 0x48, 0xab, 0x4d, 0x2c,
	  0xe7, 0xd7, 0xca, 0x3d, 0x99, 0x7b, 0xe6, 0xa7,
	  0xa8, 0x98, 0x34, 0xc4, 0x60, 0x35, 0x9f, 0x75,
	  0x72, 0x30, 0xae, 0x45, 0xfd, 0xdb, 0x03, 0x00,
	  0x84, 0xdc, 0x0a, 0x4d, 0x76, 0x8c, 0x38, 0xdd,
	  0xd3, 0x50, 0x5d, 0x13, 0x2a, 0x35, 0x97, 0x28,
	  0x16, 0x8f, 0x6b, 0xb7, 0x38, 0x29, 0x05, 0x7f,
	  0xb4, 0x0d, 0x3f, 0x40, 0x23, 0xcb, 0x3b, 0x4d,
	  0xfd, 0x3c, 0x8b, 0x96, 0x9a, 0x82, 0x66, 0x54,
	  0x6a, 0xe8, 0xd8, 0xf8, 0x9d, 0x51, 0x5c, 0xfb,
	  0x60, 0x89, 0xe1, 0x78, 0xf9, 0x3a, 0xe2, 0xdb,
	  0x6f, 0xfd, 0xa2, 0x7d, 0x6b, 0x69, 0xed, 0xce,
	  0xd6, 0x7e, 0x32, 0xf8, 0x06, 0xff, 0xd3, 0xa8,
	  0xcf, 0xac, 0xb6, 0xcc, 0xa3, 0x9c, 0x10, 0x9c,
	  0xc1, 0xa2, 0xc2, 0xf1, 0x30, 0x0e, 0xbd, 0x96,
	  0x7b, 0xd8, 0x25, 0xd6, 0x9d, 0xc0, 0x09, 0x0e,
	  0x84, 0x26, 0x2e, 0xf5, 0xa3, 0x3f, 0x8c, 0xbb,
	  0x47, 0x28, 0x00, 0x54, 0x60, 0xce, 0x22, 0x4d,
	  0xd4, 0xb1, 0xc1, 0xdb, 0x99, 0xdc, 0x21, 0x58,
	  0x25, 0x73, 0xd1, 0xf1, 0x68, 0xf1, 0x50, 0xfa,
	  0x30, 0xe6, 0x00, 0xa9, 0x98, 0x74, 0x32, 0x10,
	  0x75, 0xbe, 0x8e, 0x04, 0x7a, 0x8e, 0xda, 0xef,
	  0x99, 0xe2, 0x53, 0x83, 0x46, 0x51, 0x54, 0x5d,
	  0x40, 0x94, 0xe7, 0x8e, 0x01, 0x6e, 0x5b, 0x31,
	  0x58, 0xb1, 0xa7, 0x1c, 0xad, 0xa2, 0x84, 0x8d,
	  0x21, 0x7f, 0x4c, 0xaa, 0x15, 0x7c, 0xad, 0xad,
	  0x0a, 0x81, 0x05, 0xea, 0x2b, 0x15, 0xca, 0xf8,
	  0x8c, 0x45, 0xd2, 0xad, 0x12, 0x08, 0xd4, 0xc4,
	  0x87, 0x26, 0xc1, 0x29, 0x61, 0x70, 0x11, 0x4f,
	  0x51, 0x15, 0x00, 0x1f, 0x65, 0x8f, 0x44, 0xd8,
	  0xdc, 0xe8, 0x37, 0x2e, 0xa4, 0x92, 0x26, 0x3e,
	  0xac, 0x8b, 0x94, 0x4a, 0xb6, 0x21, 0xeb, 0x36,
	  0x92, 0xe1, 0x98, 0x0b, 0x9e, 0x44, 0x5d, 0x21,
	  0x48, 0xb0, 0x27, 0x32, 0xff, 0x3b, 0xd6, 0x87,
	  0x16, 0x30, 0x66, 0x8a, 0x2b, 0x98, 0x52, 0x21,
	  0x4c, 0xdb, 0x59, 0xdc, 0x56, 0x08, 0xcd, 0xd7,
	  0x39, 0x16, 0xf9, 0xe9, 0xbf, 0xfc, 0x38, 0x48,
	  0x2f, 0xec, 0x07, 0xb0, 0xad, 0x9f, 0x40, 0xaa,
	  0xa6, 0xd7, 0x9b, 0xf1, 0x1a, 0xdf, 0x76, 0x8b,
	  0x45, 0xf0, 0x55, 0x1d, 0x69, 0x59, 0xdb, 0xbc,
	  0xf7, 0x26, 0x46, 0xe8, 0xe5, 0x61, 0xe3, 0x30,
	  0x6a, 0xf7, 0x6e, 0x92, 0x92, 0x43, 0x50, 0xd0,
	  0xd3, 0x41, 0x79, 0x71, 0x58, 0x79, 0x77, 0x70,
	  0xdf, 0x57, 0xab, 0xda, 0x39, 0x12, 0x1c, 0x3e,
	  0xc3, 0xea, 0x68, 0xea, 0xed, 0xe2, 0xc9, 0x2f,
	  0xca, 0xe7, 0x58, 0xf0, 0xa1, 0xc4, 0x43, 0x01,
	  0x95, 0x10, 0xa5, 0x9e, 0x2c, 0xb0, 0xbb, 0x78,
	  0x72, 0xa6, 0x03, 0x61, 0x97, 0x91, 0xbd, 0xbd,
	  0x2c, 0xc4, 0x62, 0x4b, 0x8d, 0x92, 0x11, 0x56,
	  0x65, 0x53, 0xbe, 0x54, 0xf9, 0x6d, 0x07, 0x47,
	  0x00, 0xfd, 0xad, 0x6a, 0x3f, 0x16, 0x44, 0x39,
	  0x39, 0x1f, 0xd8, 0x14, 0x21, 0xe6, 0xde, 0x3e,
	  0xd6, 0x40, 0x49, 0x0a, 0x01, 0xd5, 0xcd, 0xbd,
	  0xd3, 0x25, 0xc6, 0x7d, 0x90, 0xf4, 0xc5, 0xa2,
	  0x6d, 0x5b, 0xf5, 0xd8, 0x12, 0xcf, 0x0e, 0xcf,
	  0x55, 0x30, 0x60, 0x49, 0xd7, 0x5c, 0x5e, 0x36,
	  0xa3, 0x11, 0xe9, 0x06, 0x1a, 0x05, 0xaf, 0x1f,
	  0x26, 0x75, 0x89, 0x70, 0xb7, 0x7a, 0x2e, 0xd6,
	  0x85, 0xd5, 0x2a, 0x5c, 0x62, 0xe3, 0x4a, 0x1e,
	  0x7d, 0x02, 0x60, 0x83, 0xac, 0xe8, 0x39, 0x22,
	  0xc7, 0x5c, 0x87, 0x64 } ;

struct databuf espout_2_exp[] = {
	DBENTRY(espout_2Exp)
} ;
static MSP_SEC2_SA espout_2_SA = {
	SAFLG_MODE_ESP_OUT
	| SAFLG_3DES
	| SAFLG_CBC_ENCRYPT
	| SAFLG_DES_K2_DECRYPT
	| SAFLG_SHA1_96
	| SAFLG_CV,		/* flags */
	0x1234abcd,		/* esp_spi */
	0x55aa55aa,		/* esp_sequence */
	{0x064C662A,
	0xCC6ED1B0,
	0x6CFA9AB0,
	0x042F137F,
	0xCEA570EB},		/* hash_chain_a */
	{0x11111111, 0x22222222,
	 0x33333333, 0x44444444,
	 0x55555555, 0x66666666,
	 0,0 },			/* crypt_keys */
	{0xD62ACCCC,
	0x7E6A98B3,
	0xDF015B02,
	0xD885C3E0,
	0x9BD184CB},		/* hash_chain_b */
	{0, 0x200},		/* hash_init_len */
	{0x11223344, 0x44332211,
	 0, 0},			/* crypt_iv */
	{0,0,0,0,0}		/* proto_ip */
} ;
GEN_TEST(espout_2, "Esp Outgpoing Test 2", 0x03040000) ;


general_test *gen_testlist[] = {
	&nist1_test,
	&nist2_test,
	&ecb1_test,
	&ecb2_test,
	&ecb3_test,
	&ecb4_test,
	&md5_1_test,
	&md5_2_test,
	&md5_3_test,
	&md5_4_test,
	&md5_5_test,
	&md5_6_test,
	&md5_7_test,
	&md5_8_test,
	&md5_9_test,
	&md5_10_test,
	&sha1_1_test,
	&sha1_2_test,
	&sha1_3_test,
	&espout_1_test,
	&espin_1_test,
	&espout_2_test
} ;

#define NGTESTS ASIZEOF(gen_testlist)

static void
general_test_complete(void *parm, unsigned int status)
{
	struct general_test_results *rslts ;
	general_test *test ;
	int i, fail ;

	rslts = (struct general_test_results *)parm ;

	test = rslts->test ;

	if (status)
	{
		VPRINTK("Bad status results from test %s: 0x%x\n",
			test->description, status) ;
		err_count++ ;
	}
	else
	{
		fail = 0 ;
		for (i = 0; i < test->ndest; i++)
		{
			if (memcmp(test->ebuf[i].buf, rslts->rbuf[i],
				   test->ebuf[i].sz))
			{
				fail++ ;
				COUNT_ERROR(test->description,
					    test->ebuf[i].buf,
					    rslts->rbuf[i],
					    test->ebuf[i].sz) ;
			}
		}
		if (! fail)
			COUNT_SUCCESS(test->description) ;
	}
	kfree(rslts) ;
}

static int
general_test_initiate(general_test *test, int work_q, int gi_bit, int block)
{
	struct general_test_results *rslts ;
	int rc ;
	int i, len ;
	char *cp ;


	/*
	** malloc the whole thing as one piece... I hope
	**
	** This will lead to some nastiness if anybody overindexes
	** anyting.  I believe this would be a good thing to know, though.
	*/
	len = sizeof(struct general_test_results) ;
	len += 4 * test->ndest ;
	for (i = 0 ; i < test->ndest; i++)
	{
		len += test->ebuf[i].sz ;
	}


	rslts = kmalloc(len, GFP_ATOMIC) ;
	if (!rslts)
	{
		VPRINTK("NO MEMORY FOR TEST\n") ;
		return -1 ;
	}

	memset(rslts, 0, len) ; 

	rslts->test = test ;
	cp = ( (char *)&rslts->rbuf[test->ndest] ) ;

	for(i = 0 ; i < test->ndest ; i++)
	{
		rslts->rbuf[i] = cp ;
		cp += test->ebuf[i].sz ;
	}
	
	VPRINTK("Initiating %s test: 1st src %p, 1st dest %p\n",
		test->description, test->dbuf[0].buf, rslts->rbuf[0]) ;

	rc = msp_sec2_new_request(work_q,
				  test->nsrc + test->ndest, /* n_sg_entries */
				  test->sa,
				  test->we_control | gi_bit, /* control */
				  general_test_complete, /* callback fn */
				  (void *)rslts, /* cbk_prm */
				  block /* do or don't block */
				  ) ;

	if (rc != 0)
	{
		kfree(rslts) ;
		return -2 ;
	}
	for (i = 0 ; i < test->nsrc; i++)
	{
		msp_sec2_add_sg(work_q,
				SG_GATHER,
				test->dbuf[i].buf, /* address */
				test->dbuf[i].sz /* size */
				) ;
	}
	
	for (i = 0 ; i < test->ndest; i++)
	{
		msp_sec2_add_sg(work_q,
				SG_SCATTER,
				rslts->rbuf[i], /* address */
				test->ebuf[i].sz /* size */
				) ;
	}

	msp_sec2_end_request(work_q) ;

	return 0 ;
}
	





static void
do_tests(void)
{
	int i, j, rc ;
	int work_q = 0 ;
	int gi_bit = 0 ;
	int block = BLK_POLL ;

	for (i = 0 ; i < 1000 ; i++)
	{
		/* vary the work queue */
		work_q = i & 0x1 ;
		
		for(j = 0 ; j < NGTESTS ; j++)
		{
			if (work_q)
				work_q = 0 ;
			else
				work_q = 1 ;
			while ( (rc = general_test_initiate(gen_testlist[j],
							    work_q,
							    gi_bit,
							    block) ) == -2 )
			{
				msp_sec2_poll_completion_queues(3, 0) ;
			}

				
			if ( rc == -1 )
			{
				/* out of memory? */
				printk(KERN_INFO "SECTEST Out of memory?\n") ;
				init_err++ ;
			}
			else if ( rc == -2 )
			{
				/* buffer full */
				printk(KERN_INFO "SECTEST buffer full?\n") ;
				init_err++ ;
			}
			else init_count++ ;

		}
		if ( (init_count % 10) == 0 )
			printk("init %d init_err %d success %d error %d %s\n",
			       init_count, init_err, success_count, err_count,
			       (init_err + err_count) ? "***" : "" ) ;
	}
	while (err_count + success_count < init_count)
	{
		msp_sec2_poll_completion_queues(3, 0) ;
	}

	printk("TESTS FINISHED:\n"
	       "init %d init_err %d success %d error %d %s\n",
	       init_count, init_err, success_count, err_count,
	       (init_err + err_count) ? "***" : "" ) ;
}

static void
do_hw_timer(unsigned long ignore)
{

	if (!testenabled)
		return ;

	do_tests() ;

}

static void
do_my_timer(unsigned long ignore)
{
	int flags ;
	printk(KERN_INFO "KERNTEST: pass %d error %d\n", success_count,
	       err_count) ;

	do_hw_timer(0) ;

	tl.expires  = jiffies + HZ / 2 ;

	save_flags(flags) ;
	cli() ;
	add_timer(&tl) ;
	restore_flags(flags) ;
}

static int
kernel_intface_test(unsigned long arg)
{
	static int init ;
	unsigned long flags;

	if (init == 0)
	{
		init_timer(&tl) ;
		init = 1 ;
	}
	if (arg == 0)
	{
		if (timer_set)
		{
			timer_set = 0 ;
			del_timer(&tl) ;
		}
	} else {
		if (!timer_set)
		{
			timer_set = 1 ;

			tl.function = do_my_timer ;
			tl.expires  = jiffies + 2 * HZ ;

			save_flags(flags) ;
			cli() ;
			add_timer(&tl) ;
			restore_flags(flags) ;
		}
	}
	testenabled = arg ;
	return 0 ;
}

