/*
** uClinux kernel api test module for Brecis Triad chip security engine.
**
** Copyright 2002 Brecis Communictaions
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <linux/proc_fs.h>
#include "brecis/msp_sec.h"
#include "brecis/msp_sec_kern.h"

/*------------------------------------------------------------
** Kernel interface debugging/testing
*/


	static int kernel_intface_test(unsigned long arg) ;

#ifdef DEBUG
#define DBG_SEC(a1, a2...)	printk(KERN_INFO "SECKTEST " a1, ##a2)
#else
#define DBG_SEC(a...)
#endif

static int timer_set ;
static struct timer_list tl ;

static struct proc_dir_entry *proc_entry ;
static int sectest_proc_register(void) ;
static void sectest_proc_unregister(void) ;

/*************************************************************
** sectest_open
**
** prepare to do operations with the security engine tester
*************************************************************/

static int sectest_open(struct inode * inode, struct file * file)
{
	DBG_SEC("driver open\n") ;
	return 0 ;
}

/*************************************************************
** sectest_close
**
** finish operations with the security engine tester
*************************************************************/

static int sectest_close(struct inode * inode, struct file * file)
{
	DBG_SEC("driver close\n") ;
	return 0 ;
}


/*************************************************************
** sectest_ioctl
**
** do something with the security engine tester
*************************************************************/

static int sectest_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{

	DBG_SEC("driver ioctl %d\n", cmd) ;

	if (cmd == 42)
	{
		return kernel_intface_test(arg) ;
	}
	
	DBG_SEC("bad ioctl\n") ;
	return -EINVAL ;
}



/*************************************************************
** various structures dealing with registering the driver with
** the kernel.
*************************************************************/

static struct file_operations sectest_fops = {
	owner:    THIS_MODULE,
	open:     sectest_open,
	release:  sectest_close,
	ioctl:    sectest_ioctl,
} ;

static struct miscdevice sectest_miscdev = {
	minor:        MISC_DYNAMIC_MINOR,
	name:         "msp_sec_test",
	fops:         &sectest_fops,
} ;
	


/*************************************************************
** sectest_init
**
** Initialize the hardware, and install the driver.
**
*************************************************************/

static int __init sectest_init(void)
{
	int rc ;
	
	rc = misc_register(&sectest_miscdev) ;

	DBG_SEC("misc_register returns %d\n", rc) ;

	sectest_proc_register() ;

#if 0
	kernel_intface_test(1) ;
#endif

	DBG_SEC("driver test installed\n") ;

	return 0 ;
}

static void __exit sectest_exit(void)
{
	DBG_SEC("driver test removed\n") ;

	if (timer_set)
		del_timer(&tl) ;
	timer_set = 0 ;

	misc_deregister(&sectest_miscdev) ;

	sectest_proc_unregister() ;
}

static int
proc_calc_metrics(char *page, char **start, off_t off,
		  int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int sectest_read_proc(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{
	int len ;
	len = sprintf(page, "mknod /var/sectest c %d %d\n",
		      MISC_MAJOR,
		      sectest_miscdev.minor) ;
	return proc_calc_metrics(page, start, off, count, eof, len) ;
}

static int sectest_proc_register(void)
{
	proc_entry = create_proc_read_entry("sectest", 0, NULL,
					    sectest_read_proc, NULL) ;
	return ( proc_entry != 0 ) ;
}

static void sectest_proc_unregister(void)
{
	if (!proc_entry) return ;

	remove_proc_entry(proc_entry->name, proc_entry->parent) ;
	proc_entry = 0 ;
}

module_init(sectest_init) ;
module_exit(sectest_exit) ;

MODULE_LICENSE("GPL") ;

/*************************************************************
**
** KERNEL INTERFACE TESTING
**
**   It's tricky to find a place to hook into the kernel such that
**   you're called freom within an interrupt routine.
*/

static unsigned char testbuf[512] ;

static int errcount ;
static int passcount ;

static int testenabled ;

static void
do_des_test(void)
{
	sec_desc desc ;
	static msp_sec_context context ;
	int i, error ;

	error = 0 ;

	for(i = 0 ; i < 512; i++)
		testbuf[i] = i & 0xff ;

	desc.ioctl.type = MSP_SEC_DES ;
	desc.ioctl.options = 0 ;
	desc.ioctl.buflen = 512 ;
	desc.ioctl.srcaddr = testbuf ;
	desc.ioctl.dstaddr = testbuf ;

	desc.ioctl.mode = DMC_3DES | DMC_MOD_ENC | DMC_MOD_CBC ;
	desc.ioctl.mode |= DMC_K1_ENC | DMC_K2_DEC | DMC_K3_ENC ;
	
	desc.ioctl.ivhigh = 0x01020304 ;
	desc.ioctl.ivlow  = 0xFEFDFCFB ;

	desc.ioctl.desc_key1low  = 0xCCDDAABB ;
	desc.ioctl.desc_key1high = 0xEE11FF22 ;
	desc.ioctl.desc_key2low  = 0xBA98FEDC ;
	desc.ioctl.desc_key2high = 0x32107654 ;
	desc.ioctl.desc_key3low  = 0xCCDDAABB ;
	desc.ioctl.desc_key3high = 0xEE11FF22 ;

	desc.next = 0 ;

	msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

	for(i = 0 ; i < 512; i++)
	{
		int samecount = 0 ;
		if (testbuf[i] == (i & 0xff))
		{
			/* in case some bytes actually encode such that
			** plaintext == ciphertext
			*/
			if (++samecount >3)
			{
				error = 1 ;
				break ;
			}
		}
	}

	if (!error)
	{
		desc.ioctl.type = MSP_SEC_DES ;
		desc.ioctl.options = 0 ;
		desc.ioctl.buflen = 512 ;
		desc.ioctl.srcaddr = testbuf ;
		desc.ioctl.dstaddr = testbuf ;

		desc.ioctl.mode = DMC_3DES | DMC_MOD_DEC | DMC_MOD_CBC ;
		desc.ioctl.mode |= DMC_K1_DEC | DMC_K2_ENC | DMC_K3_DEC ;
	
		desc.ioctl.ivhigh = 0x01020304 ;
		desc.ioctl.ivlow  = 0xFEFDFCFB ;

		desc.ioctl.desc_key1low  = 0xCCDDAABB ;
		desc.ioctl.desc_key1high = 0xEE11FF22 ;
		desc.ioctl.desc_key2low  = 0xBA98FEDC ;
		desc.ioctl.desc_key2high = 0x32107654 ;
		desc.ioctl.desc_key3low  = 0xCCDDAABB ;
		desc.ioctl.desc_key3high = 0xEE11FF22 ;

		desc.next = 0 ;

		msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

		for(i = 0 ; i < 512; i++)
		{
			if (testbuf[i] != (i & 0xff))
			{
				error = 1 ;
				break ;
			}
		}
	}
	if (!error)
		passcount++ ;
	else
		errcount+=10000 ;

}

/*  SHA 1 Vectors out of FIPS - 180 */
static char sha1Vector1[] = "abc";
static unsigned char sha1Result1[] = {	0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A,
			0xBA, 0x3E, 0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C,
			0x9C, 0xD0, 0xD8, 0x9D };

static void
do_hsh_test(void)
{
	sec_desc desc ;
	static msp_sec_context context ;
	int i, error ;

	error = 0 ;

	for(i = 0 ; i < 20; i++) testbuf[i] = 0 ;

	desc.ioctl.type = MSP_SEC_HASH ;
	desc.ioctl.options = MSP_SEC_DO_PADDING ;
	desc.ioctl.buflen = 3 ;
	desc.ioctl.srcaddr = sha1Vector1 ;
	desc.ioctl.dstaddr = testbuf ;

	desc.ioctl.mode = HMC_SHA1 |HMC_CHAIN_DEFAULT ;
	
	desc.next = 0 ;

	msp_sec_ioctl_kernel(&context, MSP_SEC_CTL, &desc) ;

	for(i = 0 ; i < 20; i++)
	{
		if (testbuf[i] != sha1Result1[i])
		{

			printk("HASH mismatch %d: ", i) ;
			for (i = 0 ; i < 20 ; i++)
				printk("%2x ", testbuf[i]) ;
			printk("\n") ;
			error = 1 ;
			break ;

		}
	}

	if (!error)
		passcount++ ;
	else
	{
		errcount++ ;
	}

}

static void
do_hw_timer(unsigned long ignore)
{
	static int i ;

	if (!testenabled)
		return ;

	if (i == 0)
		do_des_test() ;
	else
		do_hsh_test() ;
	i ^= 1 ;
}

static void
do_my_timer(unsigned long ignore)
{
	int flags ;
	printk(KERN_INFO "KERNTEST: pass %d error %d\n", passcount, errcount) ;

	do_hw_timer(0) ;

	tl.expires  = jiffies + HZ ;

	save_flags(flags) ;
	cli() ;
	add_timer(&tl) ;
	restore_flags(flags) ;
}

static int
kernel_intface_test(unsigned long arg)
{
	static int init ;
	unsigned long flags;

	if (init == 0)
	{
		init_timer(&tl) ;
		init = 1 ;
	}
	if (arg == 0)
	{
		if (timer_set)
		{
			timer_set = 0 ;
			del_timer(&tl) ;
		}
	} else {
		if (!timer_set)
		{
			timer_set = 1 ;

			tl.function = do_my_timer ;
			tl.expires  = jiffies + 20 * HZ ;

			save_flags(flags) ;
			cli() ;
			add_timer(&tl) ;
			restore_flags(flags) ;
		}
	}
	testenabled = arg ;
	return 0 ;
}

