/*
** uClinux kernel api examples for Brecis [SP]olo chip security engine.
**
** Copyright 2002-2003 Brecis Communictaions
**
** This file is made to compile as a kernel module.  Insmod the resulting
** .o file to run it.
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>
#include <linux/proc_fs.h>
#include "brecis/msp_sec.h"
#include "brecis/msp_sec_kern.h"
#include "brecis/msp_secv2.h"

static void dump(char *buf, int len) ;

/* EXAMPLES ------------------------------------------------------------ */

/*  Encryption (ECB)
**
** This example works for AES and single DES encryptions; DES is shown.
*/

static int
ex_des_encrypt_ecb(void)
{
	static char src[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	static char exp[] = { 0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15 };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_des_encrypt_ecb --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES ;
	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("DES encrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

	

/* Decryption (ECB)
**
** This example works for single DES decryptions.  AES decryption takes a
** bit more work, and is explained in the next example.
*/

static int
ex_des_decrypt_ecb(void)
{
	static char src[] = { 0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15 };
	static char exp[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_des_decrypt_ecb --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_DES
		| SAFLG_DES_K1_DECRYPT ;
	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("DES decrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/* AES Decryption (ECB)
**
** Here, the AES key must be pre-processed, because the engine is going
** to be used in AES decrypt mode.
*/

static int
ex_aes_decrypt_ecb(void)
{
	static char src[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};

	static char exp[] = {
		0x42, 0x30, 0xa6, 0x67, 0x43, 0x98, 0x6d, 0xa7,
		0x9f, 0x2a, 0xe1, 0xc8, 0x67, 0x41, 0x40, 0xb3,
		0x42, 0x30, 0xa6, 0x67, 0x43, 0x98, 0x6d, 0xa7,
		0x9f, 0x2a, 0xe1, 0xc8, 0x67, 0x41, 0x40, 0xb3
	};
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_aes_decrypt_ecb --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_AES_128
		 | SAFLG_AES_DECRYPT ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0xfedcba98 ;
	sa.crypt_keys[3] = 0x76543210 ;

	/* pre-process the key for decryption */
	msp_sec2_set_aes_decrypt_key(&sa, 0, 0, 0) ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("AES Decrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/*  3DES Encryption (EDE/ECB)
**     That's Encrypt/Decrypt/Encrypt (standard for 3DES),
**     Electronic Code Book mode
*/

static int
ex_3des_encrypt_ecb(void)
{
	static char src[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	static char exp[] = { 0x35, 0x66, 0x86, 0xc5, 0x85, 0x9b, 0xc0, 0x72 };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_3des_encrypt_ecb --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	/* doing EDE, so set "D" for key 2 */

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_3DES 
		| SAFLG_DES_K2_DECRYPT ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0x22222222 ;
	sa.crypt_keys[3] = 0x33333333 ;
	sa.crypt_keys[4] = 0x44444444 ; 
	sa.crypt_keys[5] = 0x55555555 ;


	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("3DES encrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1;
	}
}

/*  3DES Decryption (EDE/ECB)
**     Examine closely what happens to the keys here.
*/

static int
ex_3des_decrypt_ecb(void)
{
	static char src[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	static char exp[] = { 0xc2, 0x03, 0xbc, 0xc9, 0xb0, 0xf8, 0x56, 0x2a };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_3des_decrypt_ecb --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	/* doing DED (inverse of EDE), so set "D" for keys 1 & 3 */

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_ECB | SAFLG_3DES 
		| SAFLG_DES_K1_DECRYPT | SAFLG_DES_K3_DECRYPT ;

	/* for 3des decrypt, must use keys in opposite order */

	sa.crypt_keys[0] = 0x44444444 ; 
	sa.crypt_keys[1] = 0x55555555 ;
	sa.crypt_keys[2] = 0x22222222 ;
	sa.crypt_keys[3] = 0x33333333 ;
	sa.crypt_keys[4] = 0x01234567 ;
	sa.crypt_keys[5] = 0x89abcdef ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("3DES decrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}


/* 3DES EDE Encryption, CBC Mode
**
**
** CBC mode takes one more piece of information: the Initialization
** vector (IV).
*/

static int
ex_3des_encrypt_cbc(void)
{
	static char src[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	static char exp[] = { 0x74, 0x15, 0x95, 0x43, 0xac, 0xf4, 0x5b, 0x84 };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_3des_encrypt_cbc --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   and hash_init_len.
	*/

	/* doing EDE, so set "D" for key 2 ; note ECB changed to CBC */

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_ENCRYPT | SAFLG_3DES 
		| SAFLG_DES_K2_DECRYPT ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0x22222222 ;
	sa.crypt_keys[3] = 0x33333333 ;
	sa.crypt_keys[4] = 0x44444444 ; 
	sa.crypt_keys[5] = 0x55555555 ;

	/* normally, you'd get these from some random source. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("3DES CBC encrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/* 3DES EDE Decryption, CBC Mode
*/

static int
ex_3des_decrypt_cbc(void)
{
	static char src[] = { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 };
	static char exp[] = { 0xd3, 0x12, 0xad, 0xd8, 0x92, 0xda, 0x74, 0x08 };
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_3des_decrypt_cbc --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   and hash_init_len.
	*/

	/* doing DED (inverse of EDE), so set "D" for keys 1 & 3 */

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_DECRYPT | SAFLG_3DES 
		| SAFLG_DES_K1_DECRYPT | SAFLG_DES_K3_DECRYPT ;

	/* for 3des decrypt, must use keys in opposite order */

	sa.crypt_keys[0] = 0x44444444 ; 
	sa.crypt_keys[1] = 0x55555555 ;
	sa.crypt_keys[2] = 0x22222222 ;
	sa.crypt_keys[3] = 0x33333333 ;
	sa.crypt_keys[4] = 0x01234567 ;
	sa.crypt_keys[5] = 0x89abcdef ;

	/* normally, you'd get these from the other end somehow. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("3DES decrypt CBC example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}


/*  AES Encryption, CBC Mode
**
**      The block size for AES is 16 bytes, so the IV has to 16
**      bytes as well (as opposed to 8 for DES/3DES).
*/

static int
ex_aes_encrypt_cbc(void)
{
	static char src[] = {
			0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
			0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};

	static char exp[] = {
		0xf3, 0x0c, 0x0f, 0xb9, 0xa9, 0x88, 0xde, 0xad,
		0xaa, 0x4c, 0xe9, 0xb5, 0x45, 0x4b, 0x2d, 0x01
	};
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_aes_encrypt_cbc --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   and hash_init_len.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_ENCRYPT | SAFLG_AES_128 ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0xfedcba98 ;
	sa.crypt_keys[3] = 0x76543210 ;

	/* normally, you'd get these from some random source. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;
	sa.crypt_iv[2] = 0x33333333 ; /* aes has 16 bytes of IV! */
	sa.crypt_iv[3] = 0x44444444 ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("AES CBC Encrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/* AES Decryption, CBC Mode
**
** Here again, the AES key must be pre-processed, because the engine is
** going to be used in AES decrypt mode.
*/

static int
ex_aes_decrypt_cbc(void)
{
	static char src[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};

	static char exp[] = {
		0x53, 0x21, 0xb7, 0x76, 0x61, 0xba, 0x4f, 0x85,
		0xac, 0x19, 0xd2, 0xfb, 0x23, 0x05, 0x04, 0xf7
	};
	char buf[sizeof(src)] ;
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_aes_decrypt_cbc --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_DECRYPT | SAFLG_AES_128
		 | SAFLG_AES_DECRYPT ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0xfedcba98 ;
	sa.crypt_keys[3] = 0x76543210 ;

	/* normally, you'd get these from the other end somehow. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;
	sa.crypt_iv[2] = 0x33333333 ;
	sa.crypt_iv[3] = 0x44444444 ;

	/* pre-process the key for decryption */
	msp_sec2_set_aes_decrypt_key(&sa, 0, 0, 0) ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf, src, size) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("AES CBC Decrypt example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/* AES Encryption, multiple buffers, CBC mode
*/

static int
ex_aes_encrypt_cbc_mb(void)
{
	static char src0[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};
	static char src1[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};

	static char exp0[] = {
		0xf3, 0x0c, 0x0f, 0xb9, 0xa9, 0x88, 0xde, 0xad,
		0xaa, 0x4c, 0xe9, 0xb5, 0x45, 0x4b, 0x2d, 0x01
	};
	static char exp1[] = {
		0x63, 0xeb, 0x3d, 0x36, 0x04, 0xb1, 0xbb, 0x3a,
		0x02, 0xda, 0x53, 0x2f, 0x74, 0x77, 0x48, 0xec
	};
	char buf0[sizeof(src0)] ;
	char buf1[sizeof(src1)] ;
	int size0 = sizeof(src0) ;
	int size1 = sizeof(src1) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_aes_encrypt_cbc_mb --- \n") ;
	printk("Original buffers:\n") ;
	dump(src0, sizeof(src0)) ;
	dump(src1, sizeof(src1)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   and hash_init_len.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_ENCRYPT | SAFLG_AES_128 ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0xfedcba98 ;
	sa.crypt_keys[3] = 0x76543210 ;

	/* normally, you'd get these from some random source. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;
	sa.crypt_iv[2] = 0x33333333 ; 
	sa.crypt_iv[3] = 0x44444444 ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf0, src0, size0) ;	
	memcpy(buf1, src1, size1) ;	

	/* note: s/g entries is now 4! */

	rc = msp_sec2_new_request( 0,		/* work_q */
				   4,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf0,		/* buffer address */
				 size0 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf1,		/* buffer address */
				 size1 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf0,		/* buffer address */
				 size0 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf1,		/* buffer address */
				 size1 ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("AES CBC multi-buffer encrypt: status %x\n", status) ;
	}
	printk("Results buffers:\n") ;
	dump(buf0, sizeof(buf0)) ;
	dump(buf1, sizeof(buf1)) ;
	if (!memcmp(buf0, exp0, sizeof(exp0))
	    && !memcmp(buf1, exp1, sizeof(exp1)) )
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp0, sizeof(exp0)) ;
		dump(exp1, sizeof(exp1)) ;
		return 1 ;
	}
}

/*  AES Decryption, multiple buffers, CBC mode
*/

static int
ex_aes_decrypt_cbc_mb(void)
{
	static char src0[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};
	static char src1[] = {
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15,
		0x3f, 0xa4, 0x0e, 0x8a, 0x98, 0x4d, 0x48, 0x15
	};

	static char exp0[] = {
		0x53, 0x21, 0xb7, 0x76, 0x61, 0xba, 0x4f, 0x85,
		0xac, 0x19, 0xd2, 0xfb, 0x23, 0x05, 0x04, 0xf7
	};
	static char exp1[] = {
		0x7d, 0x94, 0xa8, 0xed, 0xdb, 0xd5, 0x25, 0xb2,
		0xa0, 0x8e, 0xef, 0x42, 0xff, 0x0c, 0x08, 0xa6
	};
	char buf0[sizeof(src0)] ;
	char buf1[sizeof(src1)] ;
	int size0 = sizeof(src0) ;
	int size1 = sizeof(src1) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_aes_decrypt_cbc_mb --- \n") ;
	printk("Original buffers:\n") ;
	dump(src0, sizeof(src0)) ;
	dump(src1, sizeof(src1)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   and hash_init_len.
	*/

	sa.flags = SAFLG_MODE_CRYPT | SAFLG_CBC_DECRYPT | SAFLG_AES_128
		| SAFLG_AES_DECRYPT ;

	sa.crypt_keys[0] = 0x01234567 ;
	sa.crypt_keys[1] = 0x89abcdef ;
	sa.crypt_keys[2] = 0xfedcba98 ;
	sa.crypt_keys[3] = 0x76543210 ;

	/* normally, you'd get these from some random source. */

	sa.crypt_iv[0] = 0x11111111 ;
	sa.crypt_iv[1] = 0x22222222 ;
	sa.crypt_iv[2] = 0x33333333 ; 
	sa.crypt_iv[3] = 0x44444444 ;

	/* pre-process the key for decryption */
	msp_sec2_set_aes_decrypt_key(&sa, 0, 0, 0) ;

	/* 
	** Many uses of encryption work better with in-place
	** encryption.  So, for this example, we will put the
	** source data in the destination buffer first...
	*/
	memcpy(buf0, src0, size0) ;	
	memcpy(buf1, src1, size1) ;	

	rc = msp_sec2_new_request( 0,		/* work_q */
				   4,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf0,		/* buffer address */
				 size0 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &buf1,		/* buffer address */
				 size1 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf0,		/* buffer address */
				 size0 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf1,		/* buffer address */
				 size1 ) ;	/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("AES CBC multi-buffer decrypt: status %x\n", status) ;
	}
	printk("Results buffers:\n") ;
	dump(buf0, sizeof(buf0)) ;
	dump(buf1, sizeof(buf1)) ;
	if (!memcmp(buf0, exp0, sizeof(exp0))
	    && !memcmp(buf1, exp1, sizeof(exp1)) )
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp0, sizeof(exp0)) ;
		dump(exp1, sizeof(exp1)) ;
		return 1 ;
	}
}

/*  MD5 Hash With Pad
**
**      Make sure to zero the hash_init_len field if not using
**      Chain Variables (CV).
*/

static int
ex_md5_pad(void)
{
	static char src[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" ;
	static char exp[] = { 
		0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5, 
		0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f
	} ;
	char buf[16] ;		/* MD5 has 16 byte result */
	int size = sizeof(src) - 1 ;	/* -1 to skip trailing NUL */
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_md5_pad --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, size) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, crypt_keys, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_HASH_PAD | SAFLG_MD5 ;
	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = 0 ; 

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 16 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("MD5 hash example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/*  SHA1 Hash With Pad
*/


static int
ex_sha1_pad(void)
{
	static char src[] = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq" ;
	static char exp[] = { 
		0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E,
		0xBA, 0xAE, 0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5,  
		0xE5, 0x46, 0x70, 0xF1
	} ;
	char buf[20] ;		/* SHA-1 has 20 byte result */
	int size = sizeof(src) - 1 ;	/* -1 to skip trailing NUL */
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_sha1_pad --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, crypt_keys, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_HASH_PAD | SAFLG_SHA1 ;
	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = 0 ; 

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 20 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("SHA1 hash example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}

/* MD5 Hash (no pad), plus final hash with pad
*/

static int
ex_md5_partial(void)
{
	static char src[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" ;
	static char exp[] = { 
		0xc0, 0xc1, 0x45, 0x98, 0x08, 0x87, 0x98, 0xdc,
		0xd1, 0x3b, 0x43, 0xaa, 0x4f, 0x97, 0xe2, 0x97
	} ;
	char buf[16] ;		/* MD5 has 16 byte result */
	int size = sizeof(src) - 1 ;	/* -1 to skip trailing NUL */
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_md5_partial --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, crypt_keys, and crypt_iv.
	*/

	/* First, a partial operation, on the first 64 bytes of the src */

	sa.flags = SAFLG_MODE_HASH | SAFLG_MD5 ;

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 64 ) ;		/* length of buffer */

		/*
		**  note that the destination address; we are putting
		**  the partial hash result back into the SA.
		*/

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &sa.hash_chain_a,	/* buffer address */
				 16 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("MD5 partial hash example: bad status %x\n", status) ;
		return 1 ;
	}

	/*
	**  Then, complete the operation with the rest of the array, using
	**  chain variables, and the CV bit, to indicate that this is a
	**  continuation.
	*/
	

	/* Note: HASH_PAD, not just HASH; plus setting CV */

	sa.flags = SAFLG_MODE_HASH_PAD | SAFLG_MD5 | SAFLG_CV ;

	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = 0x200 ;	/* that's in bits, 64 * 8 */

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 src + 64,	/* buffer address */
				 size - 64 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 16 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("MD5 completed hash example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}


/* SHA1 Hash (no pad), plus final hash with pad
*/
static int
ex_sha1_partial(void)
{
	static char src[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" ;
	static char exp[] = { 
		0xa7, 0xd2, 0xfd, 0x75, 0x69, 0xa5, 0xb9, 0xa6,
		0xc6, 0x67, 0x1d, 0x3f, 0xef, 0x1e, 0x30, 0xde,
		0xcd, 0xca, 0xf1, 0x6e 
	} ;
	char buf[20] ;		/* SHA-1 has 20 byte result */
	int size = sizeof(src) - 1 ;	/* -1 to skip trailing NUL */
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_sha1_partial --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, hash_chain_a, hash_chain_b,
	**   hash_init_len, crypt_keys, and crypt_iv.
	*/

	/* First, a partial operation, on the first 64 bytes of the src */

	/* Note: HASH, not HASH_PAD! */

	sa.flags = SAFLG_MODE_HASH | SAFLG_SHA1 ;

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 64 ) ;		/* length of buffer */

		/*
		**  note that the destination address; we are putting
		**  the partial hash result back into the SA.
		*/

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &sa.hash_chain_a,	/* buffer address */
				 20 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("SHA1 partial hash example: bad status %x\n", status) ;
		return 1 ;
	}

	/*
	**  Then, complete the operation with the rest of the array, using
	**  chain variables, and the CV bit, to indicate that this is a
	**  continuation.
	*/
	

	/* Note: HASH_PAD, not just HASH; plus setting CV */

	sa.flags = SAFLG_MODE_HASH_PAD | SAFLG_SHA1 | SAFLG_CV ;

	sa.hash_init_len[0] = 0 ;
	sa.hash_init_len[1] = 0x200 ;	/* that's in bits, 64 * 8 */

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 src + 64,	/* buffer address */
				 size - 64 ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 20 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("SHA1 completed hash example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}


/* MD5 HMAC
*/

static int
ex_md5_hmac(void)
{
	/* test case 3 from rfc 2202 */
	static char src[] = {
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd
	}  ;
	static char key[] = { 
		0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa
	} ;
	static char exp[] = {
		0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
		0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6
	} ;
	char buf[16] ;		/* MD5 has 16 byte result */
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_md5_hmac --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, crypt_keys, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_HMAC | SAFLG_CV | SAFLG_MD5 ;

	/* set hash_chain_a, hash_chain_b, and hash_init_len */

	msp_sec2_set_hmac_key( &sa, key, sizeof(key), 0, 0, 0 ) ;


	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 16 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("MD5 hmac example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}


/* SHA1 HMAC
*/

static int
ex_sha1_hmac(void)
{
	/* test case 3 from rfc 2202 */
	static char src[] = {
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		0xdd, 0xdd
	}  ;
	static char key[] = { 
		0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		0xaa, 0xaa, 0xaa, 0xaa
	} ;
	static char exp[] = {
		0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd,
		0x91, 0xa3, 0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f,
		0x63, 0xf1, 0x75, 0xd3
	} ;
	char buf[20] ;		/* SHA1 has 16 byte result */
	int size = sizeof(src) ;
	MSP_SEC2_SA sa ;
	int status ;
	int rc ;

	printk("---------------------- ex_sha1_hmac --- \n") ;
	printk("Original buffer:\n") ;
	dump(src, sizeof(src)) ;

	/*
	** for this operation, these fields of the SA are not used:
	**   esp_spi, esp_sequence, crypt_keys, and crypt_iv.
	*/

	sa.flags = SAFLG_MODE_HMAC | SAFLG_CV | SAFLG_SHA1 ;

	/* set hash_chain_a, hash_chain_b, and hash_init_len */

	msp_sec2_set_hmac_key( &sa, key, sizeof(key), 0, 0, 0 ) ;


	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,		/* # scatter/gather entries */
				   &sa,		/* SA structure */
				   0,		/* control field */
				   CBK_POLL,	/* callback fn (do polling) */
				   &status,	/* callback parm (status) */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc)
	{
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &src,		/* buffer address */
				 size ) ;	/* length of buffer */

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &buf,		/* buffer address */
				 20 ) ;		/* length of buffer */

		msp_sec2_end_request( 0 ) ;	/* work_q */
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("SHA1 hmac example: bad status %x\n", status) ;
	}
	printk("Results buffer:\n") ;
	dump(buf, sizeof(buf)) ;
	if (!memcmp(buf, exp, sizeof(exp)))
	{
		printk("==== Matches expeced data\n") ;
		return (status == 0) ? 0 : 1 ;
	}
	else
	{
		printk("**** Does not match expected data:\n") ;
		dump(exp, sizeof(exp)) ;
		return 1 ;
	}
}




/*
** ESP EXAMPLES:
**
**   These examples have both outgoing and incomming processing
**   in each example.  This is because the examples also use
**   the option that automatically fills in the IV with a
**   random number.  Each time you run these examples, they
**   give different results for the outgoing packet.
**
**   So, to have something to check for success, we run the
**   inverse ESP_IN processing over the result, to see if we
**   end up with the same data we started with.
*/


/* ESP transport out/in using DES CBC mode, no authentication,
*/
static int
ex_esp_des_cbc_transport(void)
{
	static char src[] = {
		/* ip header */
		0x45, 0x00, 0x00, 0x20,
		0x00, 0x00, 0x00, 0x00,
		0x3c, 0x11, /* protocol 0x11, UDP */
		0x6a, 0x5a, /* checksum */
		0x0A, 0x2a, 0x00, 0x1e,  /* source address */
		0x0A, 0x2a, 0x00, 0x02,  /* destination address */

		/* udp header */
		0x80, 0x27, /* src port */
		0x80, 0x27, /* dst port */
		0x00, 0x0c, /* length 12 */
		0x00, 0x00, /* no checksum */
		
		/* UDP Payload */
		0x00, 0x04, 0x00, 0x00
	} ;				/* 32 bytes total */

	/* resulting packet will be:
	**    Original length                       32
	**  + ESP HDR (4(SPI) + 4(seq) + 8(iv))     16
        **  + padding                                2
        **  + trailer (pad len + protocol)           2
	**   ------------------------------------------
	**  for a grand total of                    52
	*/
	unsigned char out_buf[52] ;
	unsigned char in_buf[52] ;

	int rc, status ;
	int control ;
	int errors = 0 ;

	/* in real use, these would be up ahead of time, not
	** at the time of processing the packet.
	*/
	MSP_SEC2_SA out_sa ;
	MSP_SEC2_SA in_sa ;

	int len, padlen ;
	int in_pkt_len ;

	int in_seq_no ;
	int i ;
	unsigned char *cp ;

	printk("\n----------------ex_esp_des_cbc_transport\n") ;

	/* --------------- Outbound Processing ----------------- */

	out_sa.flags = SAFLG_MODE_ESP_OUT
		| SAFLG_SI           /* increment sequence number */
		| SAFLG_CRI          /* create IV from random number */
		| SAFLG_HASHNULL     /* no authentication */
		| SAFLG_CBC_ENCRYPT
		| SAFLG_DES ;

	out_sa.esp_spi = 0x2000 ;           /* spi as negotiated w/reciever */
	out_sa.esp_sequence = 1 ;
        out_sa.crypt_keys[0] = 0x11112222 ; /* 8 bytes for single DES */
	out_sa.crypt_keys[1] = 0x33334444 ;
	
	/*
	** encrypt/decrypt in place is a more realistic example,
	** so initialize the out buffer with the packet contents.
	*/
	memcpy(out_buf, src, sizeof(src)) ;

	printk("Original Packet:\n") ;
	dump(out_buf, sizeof(src)) ;

	control = 0 ;
	/* copy protocol field from original IP header */
	control |= ((unsigned int) out_buf[9]) << SEC2_WE_CTRL_NXTHDR_SHF ;
	/* put in padding length */
	len = sizeof(src) - 20 ;
	padlen = 8 - ( (len + 2) % 8 ) ;
	control |= padlen << SEC2_WE_CTRL_PADLEN_SHF ;

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,           /* scatter/gather entries */
				   &out_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For transport mode, the gather area starts 
		** just past the ip header.  In this case that's
		** offset 20.
		*/
		len = sizeof(src) - 20 ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &out_buf[20],  /* gather address */
				 len ) ;	/* length */

		/* Destination size: src size + 4 (SPI) + 4 (SEQ No)
		** + 8 (DES IV) + padlen + 2 (esp trailer) + 0 (no auth data)
		*/

		len += 18 + padlen ;

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &out_buf[20],  /* scatter address */
				 len ) ;	/* length */

		msp_sec2_end_request( 0 ) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP Out DES NoAuth example: bad status %x\n", status) ;
		errors++ ;
	}

	/* it's now almost a complete IPSEC packet, except the protocol
	** in the ip header is wrong.
	*/
	out_buf[9] = 0x32 ;   /* indicates an ESP packet */


	printk("Outgoing ESP packet:\n") ;
	dump(out_buf, 64) ;

	/* --------------- Inbound Processing ----------------- */

	/* use the outgoing packet as a source for the incomming packet */

	in_pkt_len = sizeof(in_buf) ;
	memcpy(in_buf, out_buf, in_pkt_len) ;

	

	in_sa.flags = SAFLG_MODE_ESP_IN
		| SAFLG_HASHNULL
		| SAFLG_DES_K1_DECRYPT
		| SAFLG_CBC_DECRYPT
		| SAFLG_DES ;

	/*
	** In real life, software uses the SPI field of the SA
	** to compare with the SPI in the incomming packet (the 4 byte
	** integer starting at in_buf[20] in this case) to find the correct
	** sa structure to apply to this packet.
	*/

	memcpy(&in_seq_no, &in_buf[24], 4) ;
	/* if (seq_out_of_range(&in_sa, in_seq_no)) ... */

	/* software must check the sequence number for validitiy
	** here (acutal code left as an exercise).
	*/
	
	in_sa.esp_sequence = 1 ;
        in_sa.crypt_keys[0] = 0x11112222 ; /* 8 bytes for single DES */
	in_sa.crypt_keys[1] = 0x33334444 ;
	
	rc = msp_sec2_new_request( 0,		/* work_q */
				   3,           /* scatter/gather entries */
				   &in_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For transport mode, the gather area starts 
		** just past the ip header.  In this case that's
		** offset 20.
		*/
		len = in_pkt_len - 20 ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &in_buf[20],  /* gather address */
				 len ) ;	/* length */

#if 0
		/* Null authentication requires a "dummy" 12 byte buffer */
		/* we re-use in_buf here */
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 in_buf,        /* scatter address */
				 12 ) ;		/* length */
#endif


		/*
		** we're going to overwwrite the SPI, SEQ, and IV 
		** fields, moving the packet back into the correct position
		**
		** Destination size: src size - 4 (spi) - 4 (seq no)
		** - 8 (DES IV) + 0 (ICV not being used)
		*/

		len -= 16 ;

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &in_buf[20],  /* scatter address */
				 len ) ;	/* length */

		/*
		** Check length of decryption to make sure it's modulo
		** the DES blocksize.  Should do this before calling
		** msp_sec2_new_request(), but it's clearer for the
		** example to put it here.
		*/
		if ( (len % 8) != 0 )
		{
			msp_sec2_abort_request(0) ;
			status = -1 ;
		}
		else
			msp_sec2_end_request(0) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP In DES NoAuth example: bad status %x\n", status) ;
		errors++ ;
	}

	printk("Incomming packet after driver operation:\n") ;
	dump(in_buf, len + 20) ;

	/*
	** Decoded portion of packet data now runs from inbuf[20] through
	** inbuf[20 + len].  This includes the padding, pad length, and
	** next header bytes.  Next header is at inbuf[20 + len - 1] ;
	** pad length is at inbuf[20 + len - 2], and padding preceeds that.
	*/
	
	cp = &in_buf[20 + len - 1 ] ;

	/* restore protocol in original IP header */
	in_buf[9] = *cp-- ;
	/* check pad length */
	padlen = *cp-- ;

	if (padlen > len)
	{
		printk ("bad pad length on incomming packet\n") ;
		errors++ ;
	}
	else
	{
		for (i = padlen; i > 0 ; i--)
			if (*cp-- != i)
			{
				printk("Bad padding on incomming packet\n") ;
				errors++ ;
				break ;
			}
		len -= padlen + 2 ;
	}

	len += 20 ;  /* add in ip header length */

	/* resulting packet in in_buf[0] through in_buf[len]  */
	printk("Incomming packet after esp removal:\n") ;
	dump(in_buf, len) ;

	if ( (errors!=0) || (len != sizeof(src)) || memcmp(src, in_buf, len))
		return 1;
	return 0 ;

}
	

/* ESP tunnel out/in using Triple DES (EDE) in CBC mode, with
**  MD5 authentication
*/

static int
ex_esp_3des_cbc_md5_tunnel(void)
{
	static char src[] = {
		/* ip header */
		0x45, 0x00, 0x00, 0x20,
		0x00, 0x00, 0x00, 0x00,
		0x3c, 0x11, /* protocol 0x11, UDP */
		0x6a, 0x5a, /* checksum */
		0x0A, 0x2a, 0x00, 0x1e,  /* source address */
		0x0A, 0x2a, 0x00, 0x02,  /* destination address */

		/* udp header */
		0x80, 0x27, /* src port */
		0x80, 0x27, /* dst port */
		0x00, 0x0c, /* length 12 */
		0x00, 0x00, /* no checksum */
		
		/* UDP Payload */
		0x00, 0x04, 0x00, 0x00
	} ;				/* 32 bytes total */

	/*
	** IP header to be added for tunnel
	*/
	static char tiphdr[] = {
		0x45, 0x00,
		0x00, 0x58, 	/* total length */
		0x11, 0x11, 0x00, 0x00,	/* id, flags, frag offset */
		0x3c, 0x32,	/* ttl, protocol = ESP */
		0x59, 0x21,	/* header checksum */
		0x0a, 0x20, 0x00, 0x01,	/* src addr */
		0x0a, 0x20, 0x00, 0x02,	/* dst addr */
	} ;

	/* resulting packet will be:
	**    New ip header                         20
	**  + ESP HDR (4(SPI) + 4(seq) + 8(iv))     16
	**  + Original length                       32
        **  + padding                                6
        **  + trailer (pad len + protocol)           2
	**  + authentication                        12
	**   ------------------------------------------
	**  for a grand total of                    88
	*/
	unsigned char out_buf[88] ;
	unsigned char in_buf[88] ;

	int rc, status ;
	int control ;
	int errors = 0 ;

	/* in real use, these would be up ahead of time, not
	** at the time of processing the packet.
	*/
	MSP_SEC2_SA out_sa ;
	MSP_SEC2_SA in_sa ;

	int len, padlen ;
	int in_pkt_len ;

	int in_seq_no ;
	int i ;
	unsigned char *cp ;

	printk("\n----------------ex_esp_3des_cbc_md5_tunnel\n") ;

	/* --------------- Outbound Processing ----------------- */

	out_sa.flags = SAFLG_MODE_ESP_OUT
		| SAFLG_SI           /* increment sequence number */
		| SAFLG_CV 	     /* necessary for HMAC processing */
		| SAFLG_CRI          /* create IV from random number */
		| SAFLG_MD5_96       /* MD5 authentication */
		| SAFLG_CBC_ENCRYPT
		| SAFLG_3DES
		| SAFLG_DES_K2_DECRYPT;

	out_sa.esp_spi = 0x2000 ;           /* spi as negotiated w/reciever */
	out_sa.esp_sequence = 1 ;
        out_sa.crypt_keys[0] = 0x11112222 ; /* 24 bytes for triple DES */
	out_sa.crypt_keys[1] = 0x33334444 ;
	out_sa.crypt_keys[2] = 0x55556666 ;
	out_sa.crypt_keys[3] = 0x77778888 ;
	out_sa.crypt_keys[4] = 0x9999aaaa ;
	out_sa.crypt_keys[5] = 0xbbbbcccc ;

	msp_sec2_set_hmac_key( &out_sa, "test key", 8, 0, 0, 0) ;
	
	/*
	** encrypt/decrypt in place is a more realistic example,
	** so initialize the out buffer with the packet contents.
	*/
	memcpy(out_buf, src, sizeof(src)) ;

	printk("Original Packet:\n") ;
	dump(out_buf, sizeof(src)) ;

	/* for transport mode, next header protocol is 4 (IP) */
	control = 4 << SEC2_WE_CTRL_NXTHDR_SHF ;
	/* put in padding length */
	len = sizeof(src) ;
	padlen = 8 - ( (len + 2) % 8 ) ;
	control |= padlen << SEC2_WE_CTRL_PADLEN_SHF ;

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,           /* scatter/gather entries */
				   &out_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For tunnel mode, the gather area starts 
		** at the begining of the IP packet.
		*/
		len = sizeof(src) ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &out_buf[0],   /* gather address */
				 len ) ;	/* length */

		/* Destination size: src size + 4 (SPI) + 4 (SEQ No)
		** + 8 (DES IV) + padlen + 2 (esp trailer) + 12 (auth)
		*/

		len += 30 + padlen ;

		/* note that scatter address moves packet in buffer,
		** leaving room for new IP header.
		*/

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &out_buf[20],  /* scatter address */
				 len ) ;	/* length */

		msp_sec2_end_request( 0 ) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP Out 3DES MD5 example: bad status %x\n", status) ;
		errors++ ;
	}

	/* Now put on new tunnel IP header */
	memcpy(out_buf, tiphdr, sizeof(tiphdr)) ;

	printk("Outgoing ESP packet:\n") ;
	dump(out_buf, 88) ;

	/* --------------- Inbound Processing ----------------- */

	/* use the outgoing packet as a source for the incomming packet */

	in_pkt_len = sizeof(in_buf) ;
	memcpy(in_buf, out_buf, in_pkt_len) ;

	

	in_sa.flags = SAFLG_MODE_ESP_IN
		| SAFLG_CPI	     /* compare ICV -- hardware checks auth */
		| SAFLG_CV 	     /* necessary for HMAC processing */
		| SAFLG_MD5_96       /* MD5 authentication */
		| SAFLG_DES_K1_DECRYPT  /* EDE encryption (DED to decrypt) */
		| SAFLG_DES_K3_DECRYPT
		| SAFLG_CBC_DECRYPT
		| SAFLG_3DES ;

	/*
	** In real life, software uses the SPI field of the SA
	** to compare with the SPI in the incomming packet (the 4 byte
	** integer starting at in_buf[20] in this case) to find the correct
	** sa structure to apply to this packet.
	*/

	memcpy(&in_seq_no, &in_buf[24], 4) ;
	/* if (seq_out_of_range(&in_sa, in_seq_no)) ... */

	/* software must check the sequence number for validitiy
	** here (acutal code left as an exercise).
	*/
	
	in_sa.esp_sequence = 1 ;

	/* Note the key order reversal for decrpyting 3des... */

	in_sa.crypt_keys[0] = 0x9999aaaa ; /* 24 bytes for triple DES */
	in_sa.crypt_keys[1] = 0xbbbbcccc ;
	in_sa.crypt_keys[2] = 0x55556666 ;
	in_sa.crypt_keys[3] = 0x77778888 ;
        in_sa.crypt_keys[4] = 0x11112222 ;
	in_sa.crypt_keys[5] = 0x33334444 ;
	
	msp_sec2_set_hmac_key( &in_sa, "test key", 8, 0, 0, 0) ;
	
	rc = msp_sec2_new_request( 0,		/* work_q */
				   3,           /* scatter/gather entries */
				   &in_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For tunnel mode, the gather area starts 
		** just past the ip header.  In this case that's
		** offset 20.
		*/
		len = in_pkt_len - 20 ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &in_buf[20],  /* gather address */
				 len - 12 ) ;	/* length, without auth */

		/* Auth header must be in a separate s/g buffer */
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &in_buf[20 + len - 12], /* scatter address */
				 12 ) ;		/* length */


		/*
		** we're going to overwwrite the IP header, SPI, SEQ, and IV 
		** fields, moving the packet back into the correct position
		**
		** Destination size: src size - 4 (spi) - 4 (seq no)
		** - 8 (DES IV) - 12 (auth checked by hardware)
		*/

		len -= 28 ;

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &in_buf[0],  /* scatter address */
				 len ) ;	/* length */

		/*
		** Check length of decryption to make sure it's modulo
		** the DES blocksize.  Should do this before calling
		** msp_sec2_new_request(), but it's clearer for the
		** example to put it here.
		*/
		if ( (len % 8) != 0 )
		{
			msp_sec2_abort_request(0) ;
			status = -1 ;
		}
		else
			msp_sec2_end_request(0) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP In 3DES MD5 example: bad status %x\n", status) ;
		errors++ ;
	}

	printk("Incomming packet after driver operation:\n") ;
	dump(in_buf, len) ;

	/*
	** Decoded portion of packet data now runs from inbuf[0] through
	** inbuf[len].  This includes the padding, pad length, and
	** next header bytes.  Next header is at inbuf[len - 1] ;
	** pad length is at inbuf[len - 2], and padding preceeds that.
	*/
	
	cp = &in_buf[len - 1 ] ;

	/* Check next header */
	if (*cp != 0x04)
	{
		printk("Wrong next header type %x (s/b 4)\n", *cp) ;
		errors++ ;
	}
	cp-- ;
	/* check pad length */
	padlen = *cp-- ;

	if (padlen > len)
	{
		printk ("bad pad length on incomming packet\n") ;
		errors++ ;
	}
	else
	{
		for (i = padlen; i > 0 ; i--)
			if (*cp-- != i)
			{
				printk("Bad padding on incomming packet\n") ;
				errors++ ;
				break ;
			}
		len -= padlen + 2 ;
	}

	/* resulting packet in in_buf[0] through in_buf[len]  */
	printk("Incomming packet after esp removal:\n") ;
	dump(in_buf, len) ;

	if ( (errors!=0) || (len != sizeof(src)) || memcmp(src, in_buf, len))
		return 1;
	return 0 ;

}
	
/* ESP tunnel out/in using AES in CBC mode, with SHA1 authentication.
*/

static int
ex_esp_aes_cbc_sha1_tunnel(void)
{
	/* Note that the only changes from the DES/MD5 examples are
	** in the setup of the SA structure, and the fact that the block
	** size for aes is 16, so things must be % 16, not % 8
	*/

	static char src[] = {
		/* ip header */
		0x45, 0x00, 0x00, 0x20,
		0x00, 0x00, 0x00, 0x00,
		0x3c, 0x11, /* protocol 0x11, UDP */
		0x6a, 0x5a, /* checksum */
		0x0A, 0x2a, 0x00, 0x1e,  /* source address */
		0x0A, 0x2a, 0x00, 0x02,  /* destination address */

		/* udp header */
		0x80, 0x27, /* src port */
		0x80, 0x27, /* dst port */
		0x00, 0x0c, /* length 12 */
		0x00, 0x00, /* no checksum */
		
		/* UDP Payload */
		0x00, 0x04, 0x00, 0x00
	} ;				/* 32 bytes total */

	/*
	** IP header to be added for tunnel
	*/
	static char tiphdr[] = {
		0x45, 0x00,
		0x00, 0x68, 	/* total length */
		0x11, 0x11, 0x00, 0x00,	/* id, flags, frag offset */
		0x3c, 0x32,	/* ttl, protocol = ESP */
		0x59, 0x11,	/* header checksum */
		0x0a, 0x20, 0x00, 0x01,	/* src addr */
		0x0a, 0x20, 0x00, 0x02,	/* dst addr */
	} ;

	/* resulting packet will be:
	**    New ip header                         20
	**  + ESP HDR (4(SPI) + 4(seq) + 16(iv))    24
	**  + Original length                       32
        **  + padding                               14
        **  + trailer (pad len + protocol)           2
	**  + authentication                        12
	**   ------------------------------------------
	**  for a grand total of                   104
	*/
	unsigned char out_buf[104] ;
	unsigned char in_buf[104] ;

	int rc, status ;
	int control ;
	int errors = 0 ;

	/* in real use, these would be up ahead of time, not
	** at the time of processing the packet.
	*/
	MSP_SEC2_SA out_sa ;
	MSP_SEC2_SA in_sa ;

	int len, padlen ;
	int in_pkt_len ;

	int in_seq_no ;
	int i ;
	unsigned char *cp ;

	printk("\n----------------ex_esp_aes_cbc_sha1_tunnel\n") ;

	/* --------------- Outbound Processing ----------------- */

	out_sa.flags = SAFLG_MODE_ESP_OUT
		| SAFLG_SI           /* increment sequence number */
		| SAFLG_CV 	     /* necessary for HMAC processing */
		| SAFLG_CRI          /* create IV from random number */
		| SAFLG_SHA1_96      /* SHA1 authentication */
		| SAFLG_CBC_ENCRYPT
		| SAFLG_AES_128 ;

	out_sa.esp_spi = 0x2000 ;           /* spi as negotiated w/reciever */
	out_sa.esp_sequence = 1 ;
        out_sa.crypt_keys[0] = 0x11112222 ; /* 16 bytes for AES_128 */
	out_sa.crypt_keys[1] = 0x33334444 ;
	out_sa.crypt_keys[2] = 0x55556666 ;
	out_sa.crypt_keys[3] = 0x77778888 ;

	msp_sec2_set_hmac_key( &out_sa, "test key", 8, 0, 0, 0) ;
	
	/*
	** encrypt/decrypt in place is a more realistic example,
	** so initialize the out buffer with the packet contents.
	*/
	memcpy(out_buf, src, sizeof(src)) ;

	printk("Original Packet:\n") ;
	dump(out_buf, sizeof(src)) ;

	/* for transport mode, next header protocol is 4 (IP) */
	control = 4 << SEC2_WE_CTRL_NXTHDR_SHF ;
	/* put in padding length */
	len = sizeof(src) ;
	padlen = 16 - ( (len + 2) % 16 ) ;
	control |= padlen << SEC2_WE_CTRL_PADLEN_SHF ;

	rc = msp_sec2_new_request( 0,		/* work_q */
				   2,           /* scatter/gather entries */
				   &out_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For tunnel mode, the gather area starts 
		** at the begining of the IP packet.
		*/
		len = sizeof(src) ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &out_buf[0],   /* gather address */
				 len ) ;	/* length */

		/* Destination size: src size + 4 (SPI) + 4 (SEQ No)
		** + 16 (AES IV) + padlen + 2 (esp trailer) + 12 (auth)
		*/

		len += 38 + padlen ;

		/* note that scatter address moves packet in buffer,
		** leaving room for new IP header.
		*/

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &out_buf[20],  /* scatter address */
				 len ) ;	/* length */

		msp_sec2_end_request( 0 ) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP Out AES SHA1 example: bad status %x\n", status) ;
		errors++ ;
	}

	/* Now put on new tunnel IP header */
	memcpy(out_buf, tiphdr, sizeof(tiphdr)) ;

	printk("Outgoing ESP packet:\n") ;
	dump(out_buf, 104) ;

	/* --------------- Inbound Processing ----------------- */

	/* use the outgoing packet as a source for the incomming packet */

	in_pkt_len = sizeof(in_buf) ;
	memcpy(in_buf, out_buf, in_pkt_len) ;

	

	in_sa.flags = SAFLG_MODE_ESP_IN
		| SAFLG_CPI	     /* compare ICV -- hardware checks auth */
		| SAFLG_CV 	     /* necessary for HMAC processing */
		| SAFLG_SHA1_96      /* MD5 authentication */
		| SAFLG_AES_DECRYPT
		| SAFLG_CBC_DECRYPT
		| SAFLG_AES_128 ;

	/*
	** In real life, software uses the SPI field of the SA
	** to compare with the SPI in the incomming packet (the 4 byte
	** integer starting at in_buf[20] in this case) to find the correct
	** sa structure to apply to this packet.
	*/

	memcpy(&in_seq_no, &in_buf[24], 4) ;
	/* if (seq_out_of_range(&in_sa, in_seq_no)) ... */

	/* software must check the sequence number for validitiy
	** here (acutal code left as an exercise).
	*/
	
	in_sa.esp_sequence = 1 ;

        in_sa.crypt_keys[0] = 0x11112222 ; /* 16 bytes for AES_128 */
	in_sa.crypt_keys[1] = 0x33334444 ;
	in_sa.crypt_keys[2] = 0x55556666 ;
	in_sa.crypt_keys[3] = 0x77778888 ;
	
	msp_sec2_set_hmac_key( &in_sa, "test key", 8, 0, 0, 0) ;

	msp_sec2_set_aes_decrypt_key( &in_sa, 0, 0, 0) ;
	
	rc = msp_sec2_new_request( 0,		/* work_q */
				   3,           /* scatter/gather entries */
				   &in_sa,	/* sa structure */
				   control,
				   CBK_POLL,    /* callback fn (do polling) */
				   &status,	/* callback param (status)  */
				   1 ) ;	/* block -- sleep if no space*/

	if (!rc) 
	{
		/* For tunnel mode, the gather area starts 
		** just past the ip header.  In this case that's
		** offset 20.
		*/
		len = in_pkt_len - 20 ;
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &in_buf[20],  /* gather address */
				 len - 12 ) ;	/* length, without auth */

		/* Auth header must be in a separate s/g buffer */
		msp_sec2_add_sg( 0,		/* work_q */
				 SG_GATHER,	/* it's a gather (src) buf */
				 &in_buf[20 + len - 12], /* scatter address */
				 12 ) ;		/* length */


		/*
		** we're going to overwwrite the IP header, SPI, SEQ, and IV 
		** fields, moving the packet back into the correct position
		**
		** Destination size: src size - 4 (spi) - 4 (seq no)
		** - 16 (AES IV) - 12 (auth checked by hardware)
		*/

		len -= 36 ;

		msp_sec2_add_sg( 0,		/* work_q */
				 SG_SCATTER,	/* it's a scatter (dst) buf */
				 &in_buf[0],  /* scatter address */
				 len ) ;	/* length */

		/*
		** Check length of decryption to make sure it's modulo
		** the AES blocksize.  Should do this before calling
		** msp_sec2_new_request(), but it's clearer for the
		** example to put it here.
		*/
		if ( (len % 16) != 0 )
		{
			msp_sec2_abort_request(0) ;
			status = -1 ;
		}
		else
			msp_sec2_end_request(0) ;
	}
	else
	{
		status = -1 ;
	}

	if (status != 0)
	{
		printk("ESP In AES SHA1 example: bad status %x\n", status) ;
		errors++ ;
	}

	printk("Incomming packet after driver operation:\n") ;
	dump(in_buf, len) ;

	/*
	** Decoded portion of packet data now runs from inbuf[0] through
	** inbuf[len].  This includes the padding, pad length, and
	** next header bytes.  Next header is at inbuf[len - 1] ;
	** pad length is at inbuf[len - 2], and padding preceeds that.
	*/
	
	cp = &in_buf[len - 1 ] ;

	/* Check next header */
	if (*cp != 0x04)
	{
		printk("Wrong next header type %x (s/b 4)\n", *cp) ;
		errors++ ;
	}
	cp-- ;
	/* check pad length */
	padlen = *cp-- ;

	if (padlen > len)
	{
		printk ("bad pad length on incomming packet\n") ;
		errors++ ;
	}
	else
	{
		for (i = padlen; i > 0 ; i--)
			if (*cp-- != i)
			{
				printk("Bad padding on incomming packet\n") ;
				errors++ ;
				break ;
			}
		len -= padlen + 2 ;
	}

	/* resulting packet in in_buf[0] through in_buf[len]  */
	printk("Incomming packet after esp removal:\n") ;
	dump(in_buf, len) ;

	if ( (errors!=0) || (len != sizeof(src)) || memcmp(src, in_buf, len))
		return 1;
	return 0 ;

}
	
/* END OF EXAMPLES ----------------------------------------------------- */

/* The remainder of this file is utility routines, and the "glue logic"
** necessary for inserting the module into the kernel and running it.
*/

static void
do_examples(void)
{
	int err = 0 ;

	err += ex_des_encrypt_ecb() ;
	err += ex_des_decrypt_ecb() ;
	err += ex_aes_decrypt_ecb() ;
	err += ex_3des_encrypt_ecb() ;
	err += ex_3des_decrypt_ecb() ;
	err += ex_3des_encrypt_cbc() ;
	err += ex_3des_decrypt_cbc() ;
	err += ex_aes_encrypt_cbc() ;
	err += ex_aes_decrypt_cbc() ;
	err += ex_aes_encrypt_cbc_mb() ;
	err += ex_aes_decrypt_cbc_mb() ;
	err += ex_md5_pad() ;
	err += ex_sha1_pad() ;
	err += ex_md5_partial() ;
	err += ex_sha1_partial() ;
	err += ex_md5_hmac() ;
	err += ex_sha1_hmac() ;
	err += ex_esp_des_cbc_transport() ;
	err += ex_esp_3des_cbc_md5_tunnel() ;
	err += ex_esp_aes_cbc_sha1_tunnel() ;

	printk("\n\n - - - - - - - Examples:  %d total errors\n", err) ;

}

/*------------------------------------------------------------
** Kernel interface debugging/testing
*/


#define DEBUG
//#define VERBOSE

#ifdef DEBUG
#define DBG_SEC(a1, a2...)	printk(KERN_INFO "SECKTEST " a1, ##a2)
#else
#define DBG_SEC(a...)
#endif

#ifdef VERBOSE
#define VPRINTK(a1, a2...)	printk(KERN_INFO "SECKTEST " a1, ##a2)
#else
#define VPRINTK(a...)
#endif

int err_count ;
int success_count ;
int init_count ;
int init_err ;

#ifdef VERBOSE
#define COUNT_ERROR(s, exp, res, sz) { \
	err_count++ ; \
	printk(KERN_INFO "\nSECKTEST Error in %s ; expected:\n", s) ; \
	dump(exp, sz) ; \
	printk(KERN_INFO "SECKTEST actual results:\n") ; \
	dump(res, sz) ; \
	printk(KERN_INFO "\n") ; \
}
#else
#define COUNT_ERROR(s, exp, res, sz) \
	err_count++ ; \
	printk(KERN_INFO "SECKTEST Error in %s\n", s) ;
#endif

#ifdef VERBOSE
#define COUNT_SUCCESS(s) { \
	success_count++ ; \
	printk(KERN_INFO "SECKTEST %s passed\n", s) ; \
} 
#else
#define COUNT_SUCCESS(s) success_count++ ;
#endif

#define RESULT_CMP(s, exp, res, sz) { \
	if (memcmp(exp, res, sz)) { COUNT_ERROR(s, exp, res, sz) ; } \
	else { COUNT_SUCCESS(s) } \
}


/*************************************************************
** sectest_open
**
** prepare to do operations with the security engine tester
*************************************************************/

static int sectest_open(struct inode * inode, struct file * file)
{
	DBG_SEC("driver open\n") ;
	return 0 ;
}

/*************************************************************
** sectest_close
**
** finish operations with the security engine tester
*************************************************************/

static int sectest_close(struct inode * inode, struct file * file)
{
	DBG_SEC("driver close\n") ;
	return 0 ;
}


/*************************************************************
** sectest_ioctl
**
** do something with the security engine tester
*************************************************************/

static int sectest_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{

	DBG_SEC("driver ioctl %d\n", cmd) ;

	DBG_SEC("bad ioctl\n") ;
	return -EINVAL ;
}



/*************************************************************
** various structures dealing with registering the driver with
** the kernel.
*************************************************************/

static struct file_operations sectest_fops = {
	owner:    THIS_MODULE,
	open:     sectest_open,
	release:  sectest_close,
	ioctl:    sectest_ioctl,
} ;

static struct miscdevice sectest_miscdev = {
	minor:        MISC_DYNAMIC_MINOR,
	name:         "msp_sec_test",
	fops:         &sectest_fops,
} ;
	


/*************************************************************
** sectest_init
**
** Initialize the hardware, and install the driver.
**
*************************************************************/

static int __init sectest_init(void)
{
	
	do_examples() ;

	printk("\n---------------  Examples Completed\n");
	printk("following EXPECTED errors cause automatic removal of test code from kernel\n") ;
	printk("---------------\n\n") ;

	return -ENXIO ;
}

static void __exit sectest_exit(void)
{
	DBG_SEC("driver test removed\n") ;

	misc_deregister(&sectest_miscdev) ;

}

module_init(sectest_init) ;
module_exit(sectest_exit) ;

MODULE_LICENSE("GPL") ;

/*************************************************************
**
** KERNEL INTERFACE TESTING
**
*/

static void
dump(char *buf, int len)
{
	int i ;
	for ( i = 0 ; i < len ; i++ )
	{
		switch(i % 16)
		{
		case 0:
			printk("%08x  %02x ", (int)buf + i, buf[i] & 0xff) ;
			break;
		case 8:
			printk("- %02x ", buf[i] & 0xff) ;
			break ;
		case 15:
			printk("%02x\n", buf[i] & 0xff) ;
			break ;
		default:
			printk("%02x ", buf[i] & 0xff) ;
			break ;
		}
	}
	if (i%16 != 0) printk("\n") ;
}



	




