/* unzip.c -- decompress files in gzip or pkzip format.
 * Copyright (C) 1992-1993 Jean-loup Gailly
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, see the file COPYING.
 *
 * The code in this file is derived from the file funzip.c written
 * and put in the public domain by Mark Adler.
 */

/*
   This version can extract files in gzip or pkzip format.
   For the latter, only the first entry is extracted, and it has to be
   either deflated or stored.
 */

#ifdef RCSID
static char rcsid[] = "$Id: unzip.c,v 1.2 2002/06/19 20:20:06 rsewill Exp $";
#endif

#include "string.h"
#include "gzip.h"

/* PKZIP header definitions */
#define LOCSIG 0x04034b50L      /* four-byte lead-in (lsb first) */
#define LOCFLG 6                /* offset of bit flag */
#define  CRPFLG 1               /*  bit for encrypted entry */
#define  EXTFLG 8               /*  bit for extended local header */
#define LOCHOW 8                /* offset of compression method */
#define LOCTIM 10               /* file mod time (for decryption) */
#define LOCCRC 14               /* offset of crc */
#define LOCSIZ 18               /* offset of compressed size */
#define LOCLEN 22               /* offset of uncompressed length */
#define LOCFIL 26               /* offset of file name field length */
#define LOCEXT 28               /* offset of extra field length */
#define LOCHDR 30               /* size of local header, including sig */
#define EXTHDR 16               /* size of extended local header, inc sig */
#define RAND_HEAD_LEN  12       /* length of encryption random header */


/* Globals */

int decrypt;        /* flag to turn on decryption */
char *key;          /* not used--needed to link crypt.c */
int pkzip = 0;      /* set for a pkzip file */
int ext_header = 0; /* set if extended local header */

uch *inbuf;
uch *outbuf;
DECLARE(uch, window, 2L*WSIZE);

#ifndef BITS
#  define BITS 16
#endif

/* ===========================================================================
 * Unzip in to out.  This routine works on both gzip and pkzip files.
 *
 * IN assertions: the buffer inbuf contains already the beginning of
 *   the compressed data, from offsets inptr to insize-1 included.
 *   The magic header has already been checked. The output buffer is cleared.
 */
int unzip(in, out)
    uch *in, *out;   /* input and output memory locations */
{
	extern char *lastmem;
	int i;
	ulg orig_crc = 0;
	ulg orig_len = 0;
	uch buf[8];
	uch flags;

	updcrc(NULL, 0);           /* initialize crc */

    /* Decompress */

	inbuf = in;
	outbuf = out;

	/* skip gzip header */

	if (inbuf[0] != 0x1f || inbuf[1] != 0x8b)
	{
		printf("unzip - Unknown header at address %x, %0x, %0x\n",
			   (unsigned int) &inbuf[0], inbuf[0], inbuf[1]);
		return -1;
	}

	if (inbuf[2] != 8)
	{
		printf("unzip - Unknown compression method, should be 8, was %x\n",
			   inbuf[2]);
		return -2;
	}

	flags = inbuf[3];

	inbuf += 
		4 /* header+method+flags */ +
		4 /* file modification time */ +
		2 /* extra flags + operating system on which compression took place */;

	if (flags & 0x4) 
	{
		printf("unzip - extra field present, change code to handle this\n");
		return -3;
	}

	if (flags & 0x08)
	{
		while (*inbuf != 0)
			inbuf++;

		inbuf++;
	}

	printf("Inflating image at %x to %x\n", 
		   (unsigned int) in, (unsigned int) out);
	int res = inflate();

	if (res != 0)
	{
		printf("ERROR %x inflating image\n", res);
		return res;
	}

	for (i = 0; i < 8; i++)
	{
		buf[i] = get_byte();
	}

	orig_crc = LG(buf);
	orig_len = LG(buf+4);

	printf("original crc 0x%lx and length 0x%lx\n", orig_crc, orig_len);

	if (orig_crc != updcrc(outbuf, 0))
	{
		printf("invalid compressed data--got computed crc 0x%lx",
		       updcrc(outbuf, 0));
		return -4;
	}

	if (orig_len != (ulg)(outbuf - out))
	{
		printf("invalid compressed data--got computed length 0x%lx\n",
		       (ulg) (outbuf - out));
		return -5;
	}
	
	printf("last memory location used = %x, status = %x\n", 
		   (unsigned int) lastmem, res);

	return res;
}
