/*************************************************************
 * File: pmon/main.c
 * Purpose: main module for PMON
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970217	Moved stuff from cmdtable.c into this module. Deleted
 *		cmdtable.c.
 *	970303	trace_mode defined here.
 *	970303	Removed dup def for Bpt[]
 *	970307	EPI tools generates a bunch of #174 warnings - ignore
 *	970325	Changed def of "Pmon" to FTEXT, was FDATA.
 *	970401	Lots of stuff adding hwbpts
 *	970507	Added non-aligned word reads
 *	970527	Changed is_writeable to use bytes
 *	970910	Moved hostInit(6) to just before hostInit(4)
 *	970911	Fixed bug w sstep self branches. Added parens.
 *	971124	Changed DCACHEI in flush_cmd to DCACHE, needed for the 4011.
 *	971124	Don't print NVRAM: if no nvmsg.
 *	980320	Changed read_target and write_target CP0 C0_SR to use R_STATUS
 *	980320	write_target(XT_PC) fixed. v is arg3. Not arg2.
 *	980616	Switched from devinit to hostInit(8).
 *	980713	Added MIPSEB ifdef to read_target() (mips16 -EL disassembly).
 *	980831	Print "assumed" if CLKFREQ has been set explictly.
 */

#include <pmon.h>
#include <stdio.h>
#include <brecis.h>

#ifdef MIPSEB
char endian[] = "EB";
#else
char endian[] = "EL";
#endif

int vflag;
int cp1ok;
Ulong DBGREG[NREGS];	/* debugger's register value holder */
Ulong topClientMem;
Ulong topRealMem;
int pmlst,clilst;	/* list of files opened by: PMON & client */
extern int *curlst;	/* list of files opened by: current context */
unsigned long initial_sr;
unsigned long _filebase;
char *client_av[MAX_AC];
char **envp_av;
Ulong hostType;

int iflush_needed,dflush_needed;
int icache_size;
int dcache_size;
int cache_line_size;

int trace_mode;
int trace_count;
int trace_verbose;
int trace_invalid;
int trace_over;
int trace_bflag;
int trace_cflag;
int mode_64bit;

Func *c_exception_ptr, *asm_exception_ptr;

extern char *regs_sw[];

extern char _ftext[];
extern char _etext[];

BrkList brkList[MAX_BPT];

char *bptype_names[] = {"inv","pc","data","itmp","trace","nonrt"};

char pmonstr[] = "#!/bin/pmon";
char eof1str[] = "#!eof";

#ifndef PROMPT
#define PROMPT "PMON> "
#endif

#ifndef DLECHO
#define DLECHO "off"
#endif

#ifndef DLPROTO
#define DLPROTO "none"
#endif

#ifndef HOSTPORT
#define HOSTPORT "tty0"
#endif

#ifndef ETHERHWADDR
#define ETHERHWADDR "aa:bb:cc:00:00:00"
#endif

#ifndef ETHERIPADDR
#define ETHERIPADDR "71.0.0.211"
#endif

int do_diagsw();
EnvRec envlist[] = {
	{"dlecho",DLECHO,"off on lfeed"},
	{"dlproto",DLPROTO,"none XonXoff EtxAck"},
	{"hostport",HOSTPORT},
	{"prompt",PROMPT},
#ifdef ETHERNET
	{"etheraddr",ETHERHWADDR},
	{"ipaddr",ETHERIPADDR},
#endif
	{"heaptop"},
	{"diag","0","N[:dev]",do_diagsw},
	{"clkfreq"},
	{0}};

int sdump(),memtst(),call(),stty();
int erase();
int exec();
int flush_cmd();
int ld();

#if 0 /* not working yet */
int hdb_cmd();
extern Optdesc hdb_opts[];
#endif

extern Optdesc ld_opts[];
extern Optdesc load_opts[];
extern Optdesc setup_opts[];
extern Optdesc ping_opts[];
extern Optdesc stty_opts[];
extern Optdesc mt_opts[];
extern Optdesc call_opts[];
extern Optdesc sdump_opts[];
extern Optdesc erase_opts[];
extern Optdesc exec_opts[];
extern Optdesc flush_opts[];

CmdRec cmdlist[] = {
	{"stty",stty_opts,stty},

	{"load",load_opts,load},
	{"dump",sdump_opts,sdump},
	{"ping",ping_opts,ping},

	{"mt",mt_opts,memtst},
	{"call",call_opts,call},

	{"ld",ld_opts,ld},

	{"erase",erase_opts,erase},

	{"exec",exec_opts,exec},

	{"flush",flush_opts,flush_cmd},
	{"setup",setup_opts,setup},

#if 0 /* not working yet */
	{"hdb",hdb_opts,hdb_cmd},
#endif
	{0}};

Ulong specialRead();

/*************************************************************
*  pmoninit(adr)
*	Called from mips.s after caches have been flushed
*	Executed from cacheable space
*	o  Identifies CPU type (inits cp2 if necessary)
*	o  Sets initial SR value
*	o  Inits devices, command history mechanism, and environment vars
*	o  Checks for NVRAM present
*	o  Prints banner
*	o  Inits breakpoint variables
*	Returns control to mips.s	
*/
pmoninit(adr)
char *adr;
{
int i,c,n,nlcnt,slot;
Ulong pmoncodestart;
char ch,buf[80],nvmsg[40];
initial_sr = mfc0(C0_SR);
Status = initial_sr;

#ifdef MEMSIZE /* dynamic memory sizing may not work on some boards */
topRealMem = MEMSIZE;
#endif

/* 
 * pmon code load makes use of SDRAM memory wrap.
 * We need to find the largest "safe" address in our wrapped memory
 * that other code may use that won't overwrite pmon
 */
 pmoncodestart = (Ulong) &_ftext;
 while (pmoncodestart > topRealMem && pmoncodestart != 0)
 {
	 unsigned long bit = 0x80000000;

	 while ((pmoncodestart & bit) == 0)
		 bit = bit >> 1;

	 pmoncodestart &= ~bit;
 }

 topRealMem |= K0BASE;
 topClientMem = (pmoncodestart & 0xffffff00) | K0BASE;
 flush_cache(DCACHE);
 flush_cache(ICACHE);

NanoInit();

hostInit(8); /* install basic devices */

#ifdef AUTOCONSOLE	
/* 
 * A (so far) unsuccessful attempt to scan all devices to 
 * assign the console device. 
 */
for (;;) {
	nlcnt = 0;
	write(1,"B",1);
	ioctl(0,FIONREAD,&n);
	if (n) {
		for (i=0;i<1000000;i++) {
			ioctl(0,FIONREAD,&n);
			for (;n>0;n--) {
				read(0,&ch,1);
				if (ch == '\n') nlcnt++;
				}
			if (nlcnt >= 2) break;
			}
		}
	if (nlcnt >= 2) break;
	rotateCfgTbl();
	devinit();
	}
#endif

moninit();
addDynamicDefaultEnvps();

for (i=0;cmdlist[i].name;i++) addCmdRec(&cmdlist[i]);
for (i=0;envlist[i].name;i++) addEnvRec(&envlist[i]);
identify_features();
sprintf(buf,"set heaptop %08x",CLIENTPC);
do_cmd(buf);
sprintf(buf,"set memsize 0x%08x", (topClientMem & 0x0fffffff));
do_cmd(buf);
sprintf(buf,"set realmemsize 0x%08x", (topRealMem & 0x0fffffff));
do_cmd(buf);
sprintf(buf,"set prompt \"%s\"",PROMPT);
do_cmd(buf);
sprintf(buf,"sym Pmon %08x",_ftext);
do_cmd(buf);
addGpRegs();
hostInit(2);
if (cp1ok) addFpRegs();

hostInit(6); /* add other devices */

 if (IsFPGA())
 {
	 printf("\nFPGA -- Delay to let FPGA hardware initialize\n"); 
	 for (i = 10; i > 0; i--)
	 {
		 printf("%2d\r", i);
		 mdelay(50);
	 }
	 if (cpuType() == CPU_MSP_2000_FAMILY)
	 {
		 MAC_BASE *mac_base = (MAC_BASE *) MAC0_BASE;

		 printf("CPU version %08x, MAC version %08x, SEC version %08x\n",
				deviceID(), mac_base->MAC_ID_REG, *DES_IFS_REG);
	 }
 }

printf("\nPMON version %s [%s",vers,endian);
sprintf(buf,"set pmonversion %s\n", vers);
do_cmd(buf);

sprintf(full_vers_info,"pmon:%s mon:%s lib:%s", vers,monvers,libvers);

#ifdef FPEM
cp1ok = 1;
printf(",FP");
#endif

printf("], %s\n",date);

#if 0
printf("This is free software, and comes with ABSOLUTELY NO WARRANTY.\n"); 
printf("You are welcome to redistribute it without restriction.\n");
#endif

#ifdef NVRAM 
printf("\n"); 
hostInit(4); /* nvInfo */
 for (slot=0; slot<MAXFLASH; slot++) 
	 if (nvCheck(nvmsg,slot)) 
	   printf("Chip select %d, Flash chip %s, address 0x%x, size 0x%x\n",
			  slot, nvmsg, nvInfo[slot].start, nvInfo[slot].size);
	 else if (*nvmsg != '\0') printf("Chip Select %d: %s\n", slot, nvmsg);
	 else printf("flash not detected in chip select %d\n", slot);
#endif

#ifdef ETHERNET
/* if I use printf, I must reset prnbuf and do a new-line */
printf("%sEthernet MAC address %s, IP address %s\n",
	   strlen(prnbuf)?"\n":"\0", getMonEnv("etheraddr"), getMonEnv("ipaddr"));
*prnbuf = 0;		
#endif

/* 
 * pmon has a little trick where it calls printfb() instead of printf
 * and printfb uses prnbuf to decide when to print out new lines.
 * Beyond this point, any time one uses "printf", one must worry about prnbuf.
 */
*prnbuf = 0;
if (cpuType()) printfb("CPU type BRECIS %d.",cpuType());
else printfb("CPU type UNKNOWN.");

printfb("ErrorEPC (may be PC before reset): %x.", FormerPC) ;

/* 980831 Print "assumed" if CLKFREQ has been set explictly */

printfb("CPU clock frequency %s MHz Assumed.",getMonEnv("clkfreq"));

printfb("Avail RAM %d KBytes.",(topClientMem-CLIENTPC)/1024);
hostInit(3); /* extra memory */

#ifdef NVRAM
/* the global nvmsg was set by nvCheck earlier */
#ifndef BRECIS5000
if (*nvmsg) printfb("NVRAM: %s.",nvmsg);
#endif
#endif

#if 0
#if defined(CROSSVIEW) && defined(GDB_SUPPORT)
printfb("Debugger support: CROSSVIEW, DBX.");
#else
#if defined(CROSSVIEW)
printfb("Debugger support: CROSSVIEW.");
#else
#if defined(GDB_SUPPORT)
printfb("Debugger support: DBX.");
#endif
#endif
#endif
#endif

#if 0
printfb("Visit www.carmel.com for updates.");
#endif

printfb("Type 'h' for on-line help."); 
printfb("\n");
printf("\n\n");

clrbpt(-1);

#ifdef FPEM
#ifndef NEWFP
	c1dat = (struct c1state *)malloc(_fpstatesz());
	_fpinit(c1dat);
#endif
#endif
hostInit(5);
}

/*************************************************************
*  run_target(n,flags,count)
*	n=0=go   flags,cmdstr
*	n=1=cont flags
*	n=2=step flags,count
*/
run_target(n,flags,count)
int n,flags,count;
{
int i,ac;
char *cmdstr;
Ulong epc;

if (verbose) fprintf(dfp,"run_target(%d,%x,%d)\n",n,flags,count);

epc = getPc();
switch (n) {
	case 0 : /* go */
#if 0 /* debug aid */
		printf("go ac=%d ",ac);
		for (i=0;i<ac;i++) printf("\"%s\" ",client_av[i]);
		printf("\n");
#endif
#if 0
		clrhndlrs();
		closelst(2);
		Status = initial_sr;
		Fcr &= ~CSR_EMASK; /* clear any outstanding exceptions */
#endif
		setQFlush();
		cmdstr = (char *)count;
		ac = argvize(client_av,cmdstr);
		putGpr(4,ac);
		putGpr(5,client_av);
		if (envp_av != NULL)
		  free(envp_av);
		envp_av = (char **) argvizeEnv();
		putGpr(6,envp_av);
		putGpr(7,0);
		if (!(flags&1)) putGpr(29,clienttos());
		/* fall thru */
	case 1 : /* continue */
	case 3 :
		if (brkTypes(BPTYPE_NONRT) != 0) { 
			/* any nonrt bpt requires sstep */
			trace_mode = TRACE_SS;
			trace_count = trace_over = 0;
			if (!setTrcbp(epc,0)) return;
			brkInstall(2); /* trace only */
			}
		else if (is_bpt(epc)) {
			trace_mode = TRACE_SG;
			if (!setTrcbp(epc,0)) return;
			brkInstall(2); /* trace only */
			}
		else {
			trace_mode = TRACE_GB;
			brkInstall(1); /* regular */
			}
		break;
	case 2 : /* step */
	case 4 :
		trace_count = count;
		if (flags&T_O) trace_over = 1;
		else trace_over = 0;
		if (flags&T_V) trace_verbose = 1;
		else trace_verbose = 0;
		trace_mode = TRACE_SS;
		if (verbose) 
			fprintf(dfp,"trace epc=%08x count=%d\n",epc,count);
		if (!setTrcbp(epc,trace_over)) return;
		brkInstall(2); /* trace only */
		break;
	}
flush_cache(ICACHE);
_go();
}

char *trace_modes[] = {
	"TRACE_NO",
	"TRACE_TB",
	"TRACE_TG",
	"TRACE_GB",
	"TRACE_DC",
	"TRACE_DS",
	"TRACE_TN",
	"TRACE_SG",
	"TRACE_SS"};

/*************************************************************
*  exception()
*	An exception has been generated within the client.
*	Control is passed here from _exception in mips.s
*/
exception()
{
Ulong epc,cause,inst,exccode;
int flag,type,wasbda;

if (verbose) fprintf(dfp,"stopped trace_mode=%d(%s)\n",
		trace_mode,trace_modes[trace_mode]);

epc = getPc();
cause = read_target(XT_CP0,C0_CAUSE,0);
type = brkRemove(epc);
inst = read_target(XT_MEM,epc,4);
exccode = cause&CAUSE_EXCMASK;

if (verbose) fprintf(dfp,"Exception Epc=%08x Cause=%08x(%s) type=%d(%s)\n",
	epc,cause,getexcname(cause&CAUSE_EXCMASK),type,bptype_names[type]);

if (exccode == EXC_DBE) wasbda = 1; /* bda bpt */
else wasbda = 0;

if ((gdbmode || xvwmode) && (type == BPTYPE_PC || wasbda)) xstop(epc,wasbda);

if (type == BPTYPE_ITMP || getBpid() != -1 ||
   (trace_mode == TRACE_SG && is_bpt(epc))) {
	if (verbose) fprintf(dfp,"stopping... line %d\n",__LINE__);
	brkDelete(3); /* itmp */
	stop(0);
	}

if (trace_mode == TRACE_SS) {
	if (gdbmode) gdbstop(2);
	if (trace_count && --trace_count == 0) stop(0);
	flag=1;
	if (trace_bflag || trace_cflag) {
		if (is_branch(epc,inst) && trace_bflag) flag=1;
		else if (is_jal(epc,inst) && trace_cflag) flag=1;
		else flag=0;
		}
	if (flag) {
		addpchist(epc);
		if (trace_verbose) {
			disasm(prnbuf,epc,inst);
			printf("%s\n",prnbuf);
			}
		else dotik(256,1);
		}
	else dotik(256,1);
	if (!setTrcbp(epc,trace_over)) {
		printf("ERROR: unable to set trace bpt\n");
		stop(0);
		}
	brkInstall(2); /* trace */
	flush_cache(ICACHE);
	_go();
	}

/* make sure it really is ok to go again */
if (trace_mode != TRACE_SG && (cause&CAUSE_EXCMASK) == EXC_BP && !is_bpt(epc)) 
	xstop(epc,wasbda);

/* now we are ready to go again */
if (is_bpt(epc) || wasbda) {
	trace_mode = TRACE_SG;
	if (!setTrcbp(epc,0)) {
		printf("ERROR: unable to set trace bpt\n");
		xstop(epc,wasbda);
		}
	brkInstall(2); /* trace */
	}
else if (exccode != EXC_BP) {
	printf("Exception! EPC=%08x CAUSE=%08x(%s)\n",
		epc,cause,excodes[(cause&CAUSE_EXCMASK)>>2]);
	xstop(epc,wasbda);
	}
else if (trace_mode == TRACE_SG) {
	trace_mode = TRACE_GB;
	brkInstall(1); /* std */
	}
else if (!is_bpt(epc)) xstop(epc,wasbda);
else {
	trace_mode = TRACE_GB;
	brkInstall(1); /* std */
	}
flush_cache(ICACHE);
_go();
}

/*************************************************************
*/
xstop(epc,wasbda)
Ulong epc;
int wasbda;
{
brkDelete(3); /* itmp */
if (gdbmode) gdbstop((wasbda)?3:1);
if (xvwmode) {
/* This exact string is expected by xvw - do not edit! */
	printf("!503!undefined breakpoint at %08x\n",epc);
	}
stop(0);
}

#if 0
/*************************************************************
*  exception()
*	An exception has been generated within the client.
*	Control is passed here from _exception in mips.s
*/
exception()
{
RegVal epc,cause,inst;
int flag,type;

if (verbose) fprintf(dfp,"stopped\n");
epc = getPc();
cause = read_target(XT_CP0,C0_CAUSE);
type = brkRemove(epc);
inst = read_target(XT_MEM,epc,4);

if (verbose) printf("\nException Epc=%08x Cause=%08x(%s) type=%d(%s)\n",
	epc,cause,getexcname(cause&CAUSE_EXCMASK),type,bptype_names[type]);

switch (type) {
	case BPTYPE_TRACE :
		if (gdbmode) gdbstop(2);
		printf("trace_mode=%d\n",trace_mode);
		if (trace_mode == TRACE_TG) trace_mode = TRACE_GB;
		else if (trace_mode == TRACE_TB) stop(0);
		else if (trace_mode == TRACE_TN) {
			if (trace_count && --trace_count == 0) stop(0);
			flag=1;
			if (trace_bflag || trace_cflag) {
				if (is_branch(epc,inst) && trace_bflag) flag=1;
				else if (is_jal(epc,inst) && trace_cflag) flag=1;
				else flag=0;
				}
			if (flag) {
				addpchist(epc);
				if (trace_verbose) {
					disasm(prnbuf,epc,inst);
					printf("%s\n",prnbuf);
					}
				else dotik(256,1);
				}
			else dotik(256,1);
			}
		if (!setTrcbp(epc,trace_over)) stop(0);
		brkInstall(2); /* trace */
		_go();
		break;
	case BPTYPE_ITMP :
		brkDelete(3); /* itemp */
		stop(0);
		break;
	case BPTYPE_DATA : /* should never happen */
	case BPTYPE_PC :
		brkDelete(3); /* itemp */
		if (gdbmode) gdbstop(1);
		if (xvwmode) {
		/* This exact string is expected by xvw - do not edit! */
			printf("!503!undefined breakpoint at %08x\n",epc);
			stop(0);
			}
		if (getBpid() == -1) break;
		stop(0);
		break;
	default :
#ifdef CROSSVIEW
		/* break 0x97: a0 1=getchar 2=putchar 3=reboot */
		if ((cause&CAUSE_EXCMASK) == EXC_BP 
			&& (inst&0x03ffffc0)==0x970000) {
			switch (Gpr[4]) {
				case 1 : putGpr(2,getchar()); 
					putPc(epc+4); break;
				case 2 : putchar(getGpr(5)); 
					putPc(epc+4); break;
				case 3 : stop(0);
				}
			break;
			}
#endif
		if ((cause&CAUSE_EXCMASK) == EXC_DBE) {
			if (getBpid() == -1) {
				if (!setTrcbp(epc,trace_over)) stop(0);
				brkInstall(2); /* trace */
				trace_mode = TRACE_TG;
				_go();
				}
			}
		else printf("Exception! EPC=%08x CAUSE=%08x(%s)\n",
			epc,cause,excodes[(cause&CAUSE_EXCMASK)>>2]);
		brkDelete(3); /* itemp */
		stop(0);
	}

brkInstall(1); /* regular */
_go();
}
#endif

/*************************************************************
*  RegVal read_target(type,adr,sz)
*	Read from the target address space
*	All reads from target memory should come through here
*	adr	The address you want to read from
*	sz	The size of the read operation (1, 2, or 4 bytes)
*/
RegVal read_target(type,adr,sz)
Ulong adr;
int sz,type;
{
Ulong v,save,cur;
int reg;
Ushort h0,h1;

reg = adr;
switch (type) {
	case XT_MEM :
#if 0
		if (read_target_mem_ptr && (* read_target_mem_ptr)(adr,sz,&v))
			return(v);
		switch (sz) {
			case 1 : return (*(Uchar *)(adr));
			case 2 : return (*(Ushort *)(adr));
			case 4 : return (*(Ulong *)(adr));
			}
#endif
#ifdef LR64360
	if (adr < (Ulong)M_DMACR) {
		/* special code for reading the IRAM */
		switch (sz) {
		case 1 :
			cur = *((Ulong *)(adr&~3));
			return (Uchar)((cur >> (24 - (adr&3)*8 ) ) &0xff);
		case 2 :
			cur = *((Ulong *)(adr&~3));
			return (Ushort)((cur >> (16 - (adr&3)*8 ) ) &0xffff);
		case 4 :
			save = BIU;
			BIU = (save&~BIU_DS1) | BIU_IS1;
			v = read_Disolated(adr);
			BIU = save;
			return(v);
			}
		}
#endif
	switch (sz) {
		case 1 : return (*(Uchar *)(adr));
		case 2 : return (*(Ushort *)(adr));
		case 4 : 
			adr &= ~1; /* clear the LS bit */
			if (adr&2) {
				h0 = read_target(type,adr,2);
				h1 = read_target(type,adr+2,2);
#ifdef MIPSEB
				return((h0<<16) | h1);
#else
				return((h1<<16) | h0);
#endif
				}
			return (*(Ulong *)(adr));
		}
	case XT_GPR : return Gpr[reg];
	case XT_PC : return pcReg;
	case XT_CP0 : 
		if (reg == C0_CAUSE) return DBGREG[R_CAUSE];
		if (reg == C0_EPC) return DBGREG[R_PC];
		if (reg == C0_SR) return DBGREG[R_STATUS];
		return mfc0(reg);
	case XT_DBX : return mfdbx(reg);
	default : printf("%d: read_target: bad type\n",type);
	}
return(0);
}

/*************************************************************
*  write_target(type,adr,v,sz)
*	write to the target address space
*	All writes to target memory should come through here
*	adr	The address you want to write to
*	v	The value you want to write
*	sz	The size of the write operation (1, 2, or 4 bytes)
*/
write_target(type,adr,v,sz)
Ulong adr;
RegVal v;
int sz,type;
{
Ulong save,cur;
int reg;

reg = adr;
switch (type) {
	case XT_MEM :
#ifdef LR64360
	if (adr < (Ulong)M_DMACR) {
		/* special code for writing the IRAM */
		switch (sz) {
			case 1 : 
				cur = *((Ulong *)(adr&~3));
				cur &= ~((Ulong)0xff << (24 - ((adr&3)*8)) );
				cur |= (v << (24 - ((adr&3)*8)) );
				*((Ulong *)(adr&~3)) = cur;
				return;
			case 2 : 
				cur = *((Ulong *)(adr&~3));
				cur &= ~((Ulong)0xffff << (16 - ((adr&3)*8)) );
				cur |= (v << (16 - ((adr&3)*8)) );
				*((Ulong *)(adr&~3)) = cur;
				return;
			case 4 : 
				save = BIU;
				BIU = (save&~BIU_DS1) | BIU_IS1;
				write_Disolated(adr,v);
				BIU = save;
				return;
			}
		}
#endif
	switch (sz) {
		case 1 : *(Uchar *)adr = v; break;
		case 2 : *(Ushort *)adr = v; break;
		case 4 : 
			if (bptReq(adr,v)) break;
			*(Ulong *)adr = v; 
			break;
		}
		break;
	case XT_GPR : Gpr[reg] = v; break;
	case XT_PC : pcReg = v; break;
	case XT_CP0 : 
		if (reg == C0_SR) DBGREG[R_STATUS] = v;
		else mtc0(reg,v); 
		break;
	case XT_DBX : mtdbx(reg,v); break;
	default : printf("%d: write_target: bad type\n",type);
	}
}

/*************************************************************
*  swlst(lst)
*/
swlst(lst)
int lst;
{

switch (lst) {
	case 1 : curlst = &pmlst; break;
	case 2 : curlst = &clilst; break;
	}
}

/*************************************************************
*  setQFlush()
*/
setQFlush()
{
	char featurepci = FEATURE_NOEXIST;
	char featurespad = FEATURE_NOEXIST;
	char *features = getMonEnv(FEATURES);

	while (*features)
	{
		if (*features == SPAD_KEY)
			featurespad = features[1];
		else if (*features == PCI_KEY)
			featurepci = features[1];

		features += 2;
	}

	if (featurepci == FEATURE_NOEXIST && featurespad == FEATURE_NOEXIST)
		return;

	*QFLUSH_REG_1 &= ~(QFLUSH_PCI_BIT | QFLUSH_SCR_BIT);
	*QFLUSH_REG_1 |= ((featurepci != FEATURE_NOEXIST ? QFLUSH_PCI_BIT : 0) |
					  (featurespad != FEATURE_NOEXIST ? QFLUSH_SCR_BIT : 0));
}

/*************************************************************
*  pmexception(epc,cause)
*	An exception has been generated within PMON
*/
pmexception(epc,cause)
unsigned long epc,cause;
{
if (c_exception_ptr) (* c_exception_ptr)();

printf("\nException Epc=%08x Cause=%08x (%s)\n", epc,cause,
	getexcname(cause&CAUSE_EXCMASK));
monmain();
}

/*************************************************************
*  disp_Gprs(n)
*/
disp_Gprs(n)
int n;
{
int i;

if (n==1) {
	printf("    ");
	for (i=0;i<8;i++) printf(" %~8s",regs_sw[i]);
	printf("\n");
	}
printf(" $0-");
for(i = 0; i < 8; i++) printf(" %08x",getGpr(i));

if (n==1) {
	printf("\n    ");
	for (i=8;i<16;i++) printf(" %~8s",regs_sw[i]);
	}
printf("\n $8-");
for(i = 8; i < 16; i++) printf(" %08x",getGpr(i));

if (n==1) {
	printf("\n    ");
	for (i=16;i<24;i++) printf(" %~8s",regs_sw[i]);
	}
printf("\n$16-");
for(i = 16; i < 24; i++) printf(" %08x",getGpr(i));

if (n==1) {
	printf("\n    ");
	for (i=24;i<32;i++) printf(" %~8s",regs_sw[i]);
	}
printf("\n$24-");
for(i = 24; i < 32; i++) printf(" %08x",getGpr(i));
printf("\n");
}

/*************************************************************
*  erase(x)
*	erase flash
*/
Optdesc erase_opts[] = {
  {"[-s] starting-address [ending-address]", "erase flash"},
  {"-s","do sector erase as opposed to chip erase"},
	{0}};

erase(ac,av)
int ac;
Uchar *av[];
{
int i,j,count,flags,slot;
Ulong addr,endaddr,start,len;
char *p;

count = 0;
flags = 0;

for (i=1;i<ac;i++) {
  if (av[i][0] == '-') {
	for (j=1;av[i][j];j++) {
	  if (av[i][j] == 's') flags |= 1;
	  else printf("%c: unrecognized option\n", av[i][j]);
	}
  }
  else {
	if (count == 0) {
	  if (!get_rsa(&addr,av[i])) return(1);
	  endaddr = addr;
	  count++;
	}
	else if (count == 1) {
	  if (!get_rsa(&endaddr, av[i])) return(1);
	  count++;
	}
  }
}

if (count < 1) {
  printf("no address specified\n");
  return 1;
}

slot = findFlashSlot(addr);

if (slot == -1) {
  printf("address %lx is not recognized flash\n", addr);
  return 1;
}

slot = findFlashSlot(endaddr);

if (slot == -1) {
  printf("address %lx is not recognized flash\n", endaddr);
  return 1;
}

if (count != 1 && ! flags) {
  printf("error specifying address range when erasing entire chip\n");
  return 1;
}

if (addr > endaddr) {
  printf("error ending address %lx is less than beginning address %lx\n",
		 endaddr, addr);
  return 1;
}

/* erase the chip/sector */

while (1)
  {

	if (flags) findSector(slot, addr, &start, &len);
	else {
	  start = nvInfo[slot].start;
	  len = nvInfo[slot].size;
	}

	printf("erasing flash %s starting at %x\n",
		   flags ? "block" : "chip", start);

	if (flags) (* nvInfo[slot].driver)(slot, NV_SERASE, start);
	else (* nvInfo[slot].driver)(slot, NV_ERASE, start);
	
	p = (char *) start;
	
	for (i = 0; i < len; i++)
	  {
		if (p[i] != 0xff) break;		/* sector not successfully erased */
	  }
	
	if (i == len) {						/* sector successfully erased */
	  if (flags) {
		addr = start + len;
		if (addr <= endaddr) continue;	/* if more sectors to erase */
	  }
	  break;							/* we are done */	
	}

	printf("  Retry erasing flash %s which starts at %x\n", 
		   flags ? "block" : "chip", start);
	printf("        Type control-c to abort\n");
  }

}

/*************************************************************
*  exec()
*	execute script from ram/flash
*/
Optdesc exec_opts[] = {
  {"[-c] address", "execute script from address"},
  {"[-c]","#!/bin/pmon must be first line"},
	{0}};

exec(ac,av)
int ac;
Uchar *av[];
{
int i,j,count,flags;
Ulong addr;
char *p;
Uchar recbuf[MAXREC+1];

flags = 0;
count = 0;

for (i=1;i<ac;i++) {
  if (av[i][0] == '-') {
	for (j=1;av[i][j];j++) {
	  if (av[i][j] == 'c') flags |= 1;
	  else printf("%c: unrecognized option\n", av[i][j]);
	}
  }
  else {
	if (count == 0) {
	  if (!get_rsa(&addr,av[i])) return(1);
	  count++;
	}
  }
}

if (count != 1) {
  printf("no address specified\n");
  return 1;
}

p = (char *) addr;

if ((flags & 1) && strncmp(p, pmonstr, strlen(pmonstr)) != 0) {
  printf("No script at address %x beginning with <%s> found\n", p, pmonstr);
  return;
}
else 
  printf("<Executing script at address %x>\n", p);

recbuf[MAXREC] = '\0';

while (1) {
  for (i = 0; i < MAXREC; i++)
	{
	  if (*p == '\n' || *p == '\r' || *p == '\0')
		{
		  recbuf[i] = '\0';
		  p++;
		  break;
		}

	  if (*p != '\t' && !isprint(*p)) {
		printf("non-printing character %x detected at byte location %x\n",
			   *p, p);
		return;
	  }
	  recbuf[i] = *p++;
	}

  if (recbuf[0] == '\0') continue;				/* skip blank line */

  printf("# <%s>\n", recbuf);

  if (recbuf[0] == '#')
	{
	  if (strncmp(&recbuf[0], eof1str, strlen(eof1str)) == 0)
		{
		  printf("# end of script detected\n");
		  break;								/* if end of file marker */
		}
	}

  do_cmd(recbuf);
};

}

/*************************************************************
*  Note: For Processors that have copy-back Dcaches, it's not ok
*  to DCACHEI once the sw is running from kseg0, because the the
*  Dcache will contain info that *must* be copied back to ram for
*  the program to execute correctly after the flush.
*/
Optdesc flush_opts[] = {
	{"","flush caches"},
	{"-i","flush Icache only"},
	{"-d","flush Dcache only"},
	{0}};

flush_cmd(ac,av)
int ac;
char *av[];
{

if (ac == 1) {
	flush_target(ICACHEI);
	flush_target(DCACHE);
	}
else if (strequ(av[1],"-i")) flush_target(ICACHEI);
else if (strequ(av[1],"-d")) flush_target(DCACHE);
}

#if 0 /* not working yet */
/*************************************************************
* This an experimental command that would be used to allow the user to
* explictly set hw data breakpoints. At present this is only available
* via the 'when' command.
*/
Optdesc hdb_opts[] = {
	{"[-r|-w] addr[..addr] [value]","set hardware data breakpoint"},
	{"-r","only break on reads"},
	{"-w","only break on writes"},
	{0}};

hdb_cmd(ac,av)
int ac;
char *av[];
{
int type,cnt,n,i;
Ulong addr1,addr2,value;
char *p,tmp[80];
char *strstr();

if (ac == 1) {
usage:
        printf("usage: hdb [-r|-w] addr[..addr] [value]\n");
        return;
        }

type = 0x30;
for (cnt=0,i=1;i<ac;i++) {
        if (strequ(av[i],"-r")) type = 0x10;
        else if (strequ(av[i],"-w")) type = 0x20;
        else if (cnt==0) {
                if (p=strstr(av[i],"..")) {
                        strncpy(tmp,av[i],p-av[i]);
			if (!get_rsa(&addr1,tmp)) return;
			if (!get_rsa(&addr2,p+2)) return;
                        }
                else {
			if (!get_rsa(&addr1,av[i])) return;
			addr2 = 0;
                        }
                cnt++;
                }
        else if (cnt==1) {
		if (!get_rsa(&value,av[i])) return;
                cnt++;
                }
        else goto usage;
        }

if (cnt == 2) {
	printf("value not currently supported\n");
	return;
	}
setbp_target(-1,type|BPTYPE_DATA,addr1,addr2,value);
}
#endif

/*************************************************************
*/
clrbp_target(n)
int n;  
{

if (verbose) fprintf(dfp,"clrbp_target(%d)\n",n);
if (n < 0 || n >= MAX_BPT) {
        printf("%d: bad bpt number\n",n);
        return(0);
        }
if (brkList[n].type==0) {
        printf("%d: bpt is not set\n",n);
        return(0);
        }
brkList[n].type = 0;
return(1);
}

/*************************************************************
*/
int which_bpt(addr)
Ulong addr;
{
int i;

for (i=0;i<MAX_BPT;i++) {
        if (brkList[i].type==0) continue;
        if (brkList[i].addr == addr) return(i);
        }
return(-1);
}

/*************************************************************
*  int is_bpt(adr)
*/
int is_bpt(adr)
Ulong adr;
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (brkList[i].addr == adr) return(brkList[i].type);
	}
return(0);
}

/*************************************************************
*  brkTypes(n)
*/
brkTypes(n)
int n;
{
int i,j;

for (i=j=0;i<MAX_BPT;i++) {
	if (brkList[i].type == n) j++;
	}
return(j);
}

/*************************************************************
*  int brkDelete(type)
*	returns type: 0=none 1=bpc 2=bda 3=itemp 4=sstep
*/
int brkDelete(type)
Ulong type;
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (type == brkList[i].type) brkList[i].type = 0;
	}
}

#if 0
/*************************************************************
*  printBrkList()
*/
printBrkList()
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	printf("brk%d: adr=%08x type=%d\n",i,brkList[i].adr,brkList[i].type);
	}
}
#endif

/*************************************************************
*  load_byte(adr)
*/
load_byte(adr)
Ulong adr;
{
read_target(XT_MEM,adr,1);
}

/*************************************************************
*  store_byte(adr,v)
*/
store_byte(adr,v)
Ulong adr,v;
{
write_target(XT_MEM,adr,v,1);
}


/*************************************************************
*  setTrcbp(adr,stepover)
*	creates the breakpoints required to perform one step
*	returns 1 on success, 0 on failure.
*	A type4 breakpoint is 'trace'.
*/
setTrcbp(adr,stepover)
Ulong adr;
int stepover;
{
Ulong target,inst;
int sz,dsz;

if (verbose) fprintf(dfp,"setTrcbp adr=%08x stepover=%d\n",adr,stepover);
inst = read_target(XT_MEM,adr,4);
sz = instr_size(adr,inst);
if (is_branch(adr,inst)) {
	target = actual_branch_target_address(adr,inst);
	if (verbose) fprintf(dfp,"inst=%08x target=%08x\n",inst,target);
	if (target == adr) target = adr+((sz==2)?2:8); /* skip self branches */
	if (is_jal(adr,inst) && stepover) {
		dsz = instr_size(adr+4,read_target(XT_MEM,adr+4,4));
		if (setbp_target(-1,BPTYPE_TRACE,adr+4+dsz) < 0) return(0);
		}
#if 1 /* this was for calls to PMON entry-points */
	else if (is_jr(adr,inst) && !is_writeable_target(target)) {
		setbp_target(-1,BPTYPE_TRACE,getGpr(31));
		}
#endif
	else if (setbp_target(-1,BPTYPE_TRACE,target) < 0) return(0);
	}
else if (setbp_target(-1,BPTYPE_TRACE,adr+sz) < 0) return(0); /* trace */
return(1);
}

/*************************************************************
*  is_writeable_target(adr)
*/
is_writeable_target(adr)
unsigned long adr;
{
Uchar x,y;

adr |= K1BASE;
x = read_target(XT_MEM,adr,1);
y = ~x;
write_target(XT_MEM,adr,y,1);
if (read_target(XT_MEM,adr,1) != y) return(0);
write_target(XT_MEM,adr,x,1);
return(1);
}

/*************************************************************
*  bptReq(addr,v)
*       Changes a write_target into a setbp or clrbp
*/
bptReq(addr,v)
Ulong addr,v;
{
int n;

if (!(xvwmode || gdbmode)) return(0);

if ((xvwmode && v == XVWBREAKCODE) ||
    (gdbmode && (v == DBXBREAKCODE || v == GDBBREAKCODE || v == 0xa0d))
    ) setbp_target(-1,BPTYPE_PC,addr);
else if ((n=which_bpt(addr)) != -1) clrbp_target(n);
else return(0);
return(1);
}

Func *setbp_target_ptr;
Func *brkInstall_ptr;
Func *brkRemove_ptr;

/*************************************************************
*  int setbp_target(n,type,addr,addr2,value)
*/
int setbp_target(n,type,addr,addr2,value)
int n,type;
Ulong addr,addr2,value;
{
if (!setbp_target_ptr) return(-1);
if (verbose>1) fprintf(dfp,"setbp_target(%d,%d,%08x,%08x,%08x)\n",
			n,type,addr,addr2,value);
return (* setbp_target_ptr)(n,type,addr,addr2,value);
}

/*************************************************************
*/
int brkInstall(type)
int type;
{
if (!brkInstall_ptr) return(-1);
return (* brkInstall_ptr)(type);
}

/*************************************************************
*/
int brkRemove(epc)
Ulong epc;
{
if (!brkRemove_ptr) return(0);
return (* brkRemove_ptr)(epc);
}


/*************************************************************
*  stubs
*/
niceExit() { /* should never be called */ }

#define outw(a,v)	(*((volatile Ulong *)(a))=(v))
#define LED_BASE	0xbe000020

int ledval[] = {
	(1|2|4|8|0x10|0x20),		/* 0 */
	(2|4),				/* 1 */
	(1|2|0x40|8|0x10),		/* 2 */
	(1|2|4|8|0x40),			/* 3 */
	(2|4|0x40|0x20),		/* 4 */
	(1|4|8|0x40|0x20),		/* 5 */
	(1|4|8|0x10|0x20|0x40),		/* 6 */
	(1|2|4),			/* 7 */
	(1|2|4|8|0x10|0x20|0x40),	/* 8 */
	(1|2|4|8|0x20|0x40),		/* 9 */
	(1|2|4|0x10|0x20|0x40),		/* A */
	(4|8|0x10|0x20|0x40),		/* B */
	(1|8|0x10|0x20),		/* C */
	(2|4|8|0x10|0x40),		/* D */
	(1|8|0x10|0x20|0x40),		/* E */
	(1|0x10|0x20|0x40)		/* F */
	};

