/*************************************************************
 * File: mon/sym.c
 * Purpose: Part of core Monitor
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	980721	Added arg-count check to do_sym.
 */

#include <stdio.h>
#include <termio.h>
#include <string.h>
#include <mon.h>

#ifndef FTEXT
#define FTEXT "9fc00000"
#endif

Sym *symaroot,*symnroot;
Sym *nextsym(),*rmsym();
static void dopat();

/*************************************************************
*  void addsym(p)  
*	add a structure to the chain 
*/
void addsym(p)
Sym *p;
{
Sym *q;
char *n1,*n2;

p->alchild = 0;
p->archild = 0;
p->nlchild = 0;
p->nrchild = 0;

if (symaroot == 0) { /* list was empty */
	p->aparent = 0;
	p->nparent = 0;
	symnroot = p;
	symaroot = p;
	return;
	}

/* insert into numerically ordered list */
q = symnroot;
while (1) {
  if (q->value > p->value) {
	if (q->nlchild == 0) {
	  q->nlchild = p;
	  p->nparent = q;
	  break;
	} else q = q->nlchild;
  } else {
	if (q->nrchild == 0) {
	  q->nrchild = p;
	  p->nparent = q;
	  break;
	} else q = q->nrchild;
  }
}

/* insert into alphabetically ordered list */
if (p->lname) n1 = p->lname;
else n1 = p->name;
q = symaroot;
while (1) {
  if (q->lname) n2 = q->lname;
  else n2 = q->name;
  if (strcmp(n2,n1) > 0) {
	if (q->alchild == 0) {
	  q->alchild = p;
	  p->aparent = q;
	  break;
	} else q = q->alchild;
  } else {
	if (q->archild == 0) {
	  q->archild = p;
	  p->aparent = q;
	  break;
	} else q = q->archild;
  }
}
}

#if 0
#define FIX_ADR(a)	((IS_K0SEG(a))?((unsigned)(a))|K1BASE:(a))
#else
#define FIX_ADR(a)	(a)
#endif

/*************************************************************
*  Sym *findsymbyname(name,foundp) 
*	returns struct that is equal name OR near name
*	We always return a struct if ANY symbol exists	
*/
Sym *findsymbyname(name,foundp)
char *name;
int *foundp;
{
Sym *p;
char *nam;
int found;

p = symaroot;

if (p == 0) return(0);

while (1) {
	if (p->lname) nam = p->lname;
	else nam = p->name;
	
	found = strcmp(nam,name);
	
	if (found == 0) break;
	else if (found < 0) {
		if (p->archild != 0) p = p->archild;
		else break;
	} else if (found > 0) {
		if (p->alchild != 0) p = p->alchild;
		else break;
	}
}
*foundp = found;
return(p);
}

/*************************************************************
*  Sym *findsymbyvalue(value) 
*	returns struct that is <= value 
*/
Sym *findsymbyvalue(value)
unsigned int value;
{
Sym *p;

p = symnroot;
while (p != 0) {
  if (p->value == value) return (p);

  if (p->value > value) {
	if (p->nlchild == 0) {
	  while (p != 0) {
		if (p->value < value) return (p);
		p = p->nparent;
	  }
	} else p = p->nlchild;
  } else {
	if (p->nrchild == 0) return (p);
	else p = p->nrchild;
  }
}
return (0);
}

/*************************************************************
*  Sym *getsym(value) 
*	returns struct that matches value 
*/
Sym *getsym(value)
unsigned long value;
{
Sym *p;

p = findsymbyvalue(value);
if (p != 0 && p->value == value) return (p);
return(0);
}

#define LFLAG 0x01
#define VFLAG 0x02
#define NFLAG 0x04
#define TFLAG 0x08
#define DFLAG 0x10
#define BFLAG 0x20
#define AFLAG 0x40

Optdesc ls_opts[] = {
	{"[-ln sym*|-va adr]","list symbols"},
	{"-l","long listing"},
	{"-n","numeric order"},
	{"-v","compute value"},
	{"-a","lookup address"},
	{0}};

/*************************************************************
*  do_ls(ac,av)
*	the 'ls' command 
*/
do_ls(ac,av)
int ac;
char *av[];
{
int i,j,flags,pats;

flags = pats = 0;
for (i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		j = 1;
		while (av[i][j] != 0) {
			if (av[i][j] == 'l') flags |= LFLAG;
			else if (av[i][j] == 'v') flags |= VFLAG;
			else if (av[i][j] == 'n') flags |= NFLAG;
			else if (av[i][j] == 'a') flags |= AFLAG;
			/* t=text d=data b=bss
			else if (av[i][j] == 't') flags |= TFLAG;
			else if (av[i][j] == 'b') flags |= BFLAG;
			else if (av[i][j] == 'd') flags |= DFLAG;
			*/
			else printf("bad option [%c]\n",av[i][j]);
			j++;
			}
		}
	else pats++;
	}

ioctl_cbreak(0L);

if (flags&(VFLAG|AFLAG)) {
	if (pats == 0) printf("no values specified\n");
	else {
		for (i=1;i<ac;i++) {
			if (av[i][0] != '-') doval(av[i],flags);
			}
		}
	return(0);
	}

if (pats == 0) dopat("*",flags);
else {
	for (i=1;i<ac;i++) {
		if (av[i][0] != '-') dopat(av[i],flags);
		}
	}
}

/************************************************************* 
*  void doval(val,flags) 
*	prints name from value 
*/
void doval(val,flags)
char *val;
int flags;
{
unsigned int value;
Sym *p;
char *nam,tmp[LINESZ];

if (!get_rsa(&value,val)) return;

if (flags&VFLAG) printf("0x%08x = 0t%d = 0o%o\n",value,value,value);
else { /* must be AFLAG */
	if (!adr2symoff(tmp,value,0)) printf("%08x: not found\n",value);
	else printf("%08x = %s\n",value,tmp);
	}
}

/************************************************************* 
*  int adr2symoff(char *dst,Ulong value,int width)
*	convert addr to symbol+offset 
*/
int adr2symoff(char *dst,Ulong value,int width)
{
Sym *p;
char *nam,tmp[16];
long offset;

dst[0] = 0;
p = findsymbyvalue(value);
if (p == 0) return(0);
else {
	if (p->lname) nam = p->lname;
	else nam = p->name;

	if (width == 0) sprintf(dst,"%s",nam);
	else sprintf(dst,"%*.*s",width,width,nam);

	offset = value-(p->value);
	if (offset == 0) strcat(dst,"       ");
	else {
		sprintf(tmp,"+0x%-4x",offset);
		strcat(dst,tmp);
		}
	}
return(1);
}

/************************************************************* 
*  static void dopat(pat,flags) 
*	prints all syms that match pat 
*/
static void dopat(pat,flags)
char *pat;
int flags;
{
Sym *q;
char sym[LINESZ];
int l,siz,col,maxlen,len,width;

if (!atob(&siz,getMonEnv("moresz"),10)) {
	printf("moresz has bad value\n");
	return;
	}

if (flags&LFLAG) {
	l = siz;
	q = 0;
	for (;;) {
		q = nextsym(prnbuf,q,pat,flags);
		if (more(prnbuf,&l,siz)) break;
		if (q == 0) break;
		}
	return;
	}

/* isn't a long listing, so need to columnize */
q = 0;
maxlen = 0;
for (;;) {
	q = nextsym(sym,q,pat,flags);
	len = strlen(sym);
	if (len > maxlen) maxlen = len;
	if (q == 0) break;
	}
width = (maxlen+8)&~7;
l = siz;
q = 0;
for (;;) {
	prnbuf[0] = 0;
	col = 0;
	for (;;) {
		q = nextsym(sym,q,pat,flags);
		col += width;
		if (col+width <= 79) str_fmt(sym,width,FMT_LJUST);
		strcat(prnbuf,sym);
		if (col+width > 79 || q == 0) break;
		}
	if (more(prnbuf,&l,siz)) break;
	if (q == 0) break;
	}
}

Optdesc sym_opts[] = {
	{"name val","define symbol"},
	{"name","symbol to be defined"},
	{"val","value to be assigned"},
	{0}};

/************************************************************* 
*  do_sym(ac,av) 
*	sym command 
*/
do_sym(ac,av)
int ac;
char *av[];
{
Sym *p;
unsigned long v;

if (ac != 3) {
	printf("usage: sym <name> <value>\n");
	return(1);
	}
if (!get_rsa(&v,av[2])) return(1);
if (!newsym(av[1],v)) printf("out of memory\n");
}

/************************************************************* 
*  newsym(name,value)  
*	allocates struct, & adds it to list 
*/
newsym(name,value)
char *name;
unsigned long value;
{
Sym *p;
int len;

if (p = rmsym(name)) { /* name exists */
	p->value = value;
	addsym(p);
	return(1);
	}

p = (Sym *)malloc(sizeof(Sym));
if (p == 0) return(0);
p->lname = 0;
len = strlen(name);
if (len > MAXSYM-1) {
	p->lname = (char *) malloc(len+1);
	if (p->lname == 0) return(0);
	strcpy(p->lname,name);
	}
else strcpy(p->name,name); 
p->value = value;
addsym(p);
return(1);
}

/************************************************************* 
*  Sym *rmsym(name) 
*	remove matching symbol, return NULL is not found 
*/
Sym *rmsym(name)
char *name;
{
char *nam;
Sym *p,*q;
int found;

/* remove from alphbetically ordered list */

p = findsymbyname(name,&found);
if (p == 0 || found != 0) return(0);

/* found, now remove it */
if (p->archild != 0) {
	q = p->archild;

	if (q->alchild != 0) {
		while (q->alchild != 0) q = q->alchild;
		q->aparent->alchild = q->archild;
		if (q->archild != 0) q->archild->aparent = q->aparent;
		q->archild = p->archild;
		q->archild->aparent = q;
	}
	q->alchild = p->alchild;
	if (q->alchild != 0) q->alchild->aparent = q;

} else if (p->alchild != 0) {
	q = p->alchild;

	if (q->archild != 0) {
		while (q->archild != 0) q = q->archild;
		q->aparent->archild = q->alchild;
		if (q->alchild != 0) q->archild->aparent = q->aparent;
		q->alchild = p->alchild;
		q->alchild->aparent = q;
	}

	q->archild = p->archild;
	if (q->archild != 0) q->archild->aparent = q;

} else q = 0;

if (p->aparent == 0) symaroot = q;
else if (p->aparent->alchild == p) p->aparent->alchild = q;
else p->aparent->archild = q;

/* remove from numerically ordered list */
if (p->nrchild != 0) {
	q = p->nrchild;

	if (q->nlchild != 0) {
		while (q->nlchild != 0) q = q->nlchild;
		q->nparent->nlchild = q->nrchild;
		if (q->nrchild != 0) q->nrchild->nparent = q->nparent;
		q->nrchild = p->nrchild;
		q->nrchild->nparent = q;
	}
	q->nlchild = p->nlchild;
	if (q->nlchild != 0) q->nlchild->nparent = q;

} else if (p->nlchild != 0) {
	q = p->nlchild;

	if (q->nrchild != 0) {
		while (q->nrchild != 0) q = q->nrchild;
		q->nparent->nrchild = q->nlchild;
		if (q->nlchild != 0) q->nrchild->nparent = q->nparent;
		q->nlchild = p->nlchild;
		q->nlchild->nparent = q;
	}

	q->nrchild = p->nrchild;
	if (q->nrchild != 0) q->nrchild->nparent = q;

} else q = 0;

if (p->nparent == 0) symnroot = q;
else if (p->nparent->nlchild == p) p->nparent->nlchild = q;
else p->nparent->nrchild = q;

return(p);
}

/************************************************************* 
*  sym2adr(vp,name) 
*	finds value from name 
*/
sym2adr(vp,name)
unsigned long *vp;
char *name;
{
Sym *p;
char *nam;
int found;

*vp = 0;
p = findsymbyname(name,&found);
if (p == 0 || found != 0) return(0);
*vp = p->value;
return(1);
}

/************************************************************* 
*  adr2sym(name,v) 
*	finds name from value 
*/
char *adr2sym(name,v)
char *name;
unsigned long v;
{
Sym *p;

if (name) name[0] = 0;
p = getsym(v);
if (p == 0) return(0);
if (name) {
	if (p->lname) strcpy(name,p->lname);
	else strcpy(name,p->name);
	return(name);
	}
else {
	if (p->lname) return(p->lname);
	else return(p->name);
	}
}

/************************************************************* 
*  Sym *nextsym(p,q,pat,flags) 
*	gets next matching symbol 
*/
Sym *nextsym(p,q,pat,flags)
char *p,*pat;
Sym *q;
int flags;
{
char *nam;
unsigned int value;

*p = 0;
if (symaroot == 0) return 0;

if (q == 0) {
  if (flags&NFLAG) {
	q = symnroot;
	while (q->nlchild != 0) q = q->nlchild;
  } else {
	q = symaroot;
	while (q->alchild != 0) q = q->alchild;
  }
}

for (;;) {
	if (q == 0) return(q);
	if (q->lname) nam = q->lname;
	else nam = q->name;
	value = q->value;

	if (flags&NFLAG) {
	  if (q->nrchild != 0) {
		q = q->nrchild;
		while (q->nlchild != 0) q = q->nlchild;
	  } else {
		while (q->nparent != 0 && q->nparent->nrchild == q)
		  q = q->nparent;
		q = q->nparent;
	  } 
	} else {
	  if (q->archild != 0) {
		q = q->archild;
		while (q->alchild != 0) q = q->alchild;
	  } else {
		while (q->aparent != 0 && q->aparent->archild == q)
		  q = q->aparent;
		q = q->aparent;
	  }
	}

	if (strpat(nam,pat)) {
	  if (flags&LFLAG) sprintf(p,"%08x ",value);
	  strcat(p,nam);
	  return(q);
	}
}
}

/************************************************************* 
*  clrsyms() 
*	clear entire symbol table 
*/
clrsyms()
{
Sym *p,*q;

p = symaroot;
while (p != 0) {
  while (p->alchild != 0) p = p->alchild;
  if (p->archild == 0) {
	q = p->aparent;
	if (q != 0) {
	  if (q->alchild == p) q->alchild = 0;
	  else q->archild = 0;
	}
	if (p->lname) free(p->lname);
	free(p);
	p = q;
  } else p = p->archild;
}

symaroot = symnroot = 0;
/* newsym("Pmon",FTEXT); */
}
