/*************************************************************
 * File: mon/set.c
 * Purpose: Part of core Monitor
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970409	Switched over to using addEnvRec
 *	970827	Fixed prob w "set foo".
 *	970820	Added value = 0 to addEnvRec. Needed for IMON. Symptom
 *		was bad ether downld to target.
 *	980715	Modified set_nvram() to write p2 marker.
 */

#include <string.h>
#include <mon.h>

struct EnvRec monenvs[] = {
	{"brkcmd","l @pc 1"},
	{"datasz","-b","-b -h -w"},
	{"inalpha","hex","hex symbol"},
	{"inbase","16","auto 8 10 16"},
	{"moresz","10"},
	{"regstyle","sw","hw sw"},
	{"rptcmd","trace","off on trace"},
	{"trabort","^K"},
	{"uleof","%"},
	{"ulcr","off","off on"},
	{"validpc","_ftext etext"},
	{0}};

EnvRec *envChain;
FILE *dfp;
int verbose;

/*************************************************************
*  envinit()
*/
envinit()
{
int i;

for (i=0;monenvs[i].name;i++) addEnvRec(&monenvs[i]);
}

Optdesc set_opts[] = {
	{"[-i] [name [value]]","display/set variable"},
	{"-i","indicates name=*value where value is a variable name"},
	{"name","variable to be set"},
	{"value","value to be assigned"},
	{"nvram","store current settings in nvram"},
	{0}};

/*************************************************************
*  do_set(ac,av)
*/
do_set(ac,av)
int ac;
char *av[];
{
int i;
EnvRec *p;

switch (ac) {
	case 1 : /* display all variables */
		for (p=envChain;p;p=p->next) printvar(p);
		break;
	case 2 : /* display specific variable */
		for (p=envChain;p;p=p->next) 
			if (strequ(av[1],p->name)) break;
		if (p) printvar(p); /* 970827 */
		else printf("%s: not found\n",av[1]);
		break;
	case 3 : /* set specific variable */
		if (!setenv(av[1],av[2])) printf("%s: not found\n",av[1]);
		break;
	case 4 : /* set variable== value of othervariable */
		if ( (av[1][1] == 'i') && (av[1][0] == '-') )
		if (!setenv(av[2],getMonEnv(av[3])))
			printf("%s: not found\n",av[1]); 
		break;
	}
}

/*************************************************************
*  setenv(name,value)
*/
setenv(name,value)
char *name,*value;
{
int i;
EnvRec *p;

for (p=envChain;p;p=p->next) 
	if (strequ(name,p->name)) break;

if (!p) {
  p = (EnvRec *) malloc(sizeof(EnvRec));
  if (!p) return 0;

  memset(p, 0, sizeof(EnvRec));
  p->name = strdup(name);
  p->init = strdup(value);
  addEnvRec(p);
}

if (p->value) free(p->value);
p->value = strdup(value);
if (p->func) (* p->func)(name,value);
return(1);
}

/*************************************************************
*  printvar(i)
*/
printvar(p)
EnvRec *p;
{

if (p->value == 0) 
	printf("%10s = %-10s",p->name,"UNDEFINED");
else if (strchr(p->value,' ')) 
	printf("%10s = \"%s\"",p->name,p->value);
else 
	printf("%10s = %-10s",p->name,p->value);
if (p->values) printf("  [%s]",p->values);
printf("\n");
}

/*************************************************************
*  char *getMonEnv(name) 
*	returns pointer to current value of named variable 
*/
char *getMonEnv(name)
char *name;
{
int i;
EnvRec *p;

for (p=envChain;p;p=p->next) 
	if (strequ(name,p->name)) break;
if (p == 0) return(0);

return(p->value);
}

/*************************************************************
*  int setdMonEnv(name,value) 
*	set the value of decimal monitor env var.
*/
int setdMonEnv(name,value)
char *name;
int value;
{
char tmp[20];

sprintf(tmp,"%d",value);
return setenv(name,tmp);
}

/*************************************************************
*  int setxMonEnv(name,value) 
*	set the value of hexadecimal monitor env var.
*/
int setxMonEnv(name,value)
char *name;
unsigned long value;
{
char tmp[20];

sprintf(tmp,"%08lx",value);
return setenv(name,tmp);
}

/*************************************************************
*  matchenv(name) 
*	returns integer corresponding to current value 
*/
matchenv(name)
char *name;
{
int i,j;
char *t,wrd[20];
EnvRec *p;

for (p=envChain;p;p=p->next) 
	if (strequ(name,p->name)) break;
if (p == 0) return(-1);

if (p->value == 0) p->value = strdup("");

for (j=0,t=p->values;;j++) {
	t = getword(wrd,t);
	if (t == 0) return(-2);
	if (striequ(wrd,p->value)) return(j);
	}
}

/*************************************************************
*  addEnvRec(p)
*	add a EnvRec to the envChain
*/
addEnvRec(p)
EnvRec *p;
{
EnvRec *q;
char buf[LINESZ];

p->next = 0;
p->value = 0;
if (!envChain) { /* empty */
	envChain = p;
	}
else {
	for (q=envChain;q->next;q=q->next) ; /* find end of chain */
	q->next = p;
	}
sprintf(buf,"set %s '%s'",p->name,p->init);
do_cmd(buf);
}

/*************************************************************
*  argvizeEnv()
*	argvize environment variables
*	Create a  single, malloc'ed, array of the following format	
*		1) array of pointers terminated by null pointer to strings
*		2) null terminated strings
*/
char **argvizeEnv()
{
EnvRec *p;
char *cp;
char **envp;
int count,len;

/* go through environment variables once computing needed length to malloc */

for (count = 0, len = 0, p = envChain; p != NULL; p = p->next) {
  count++;							/* count number of array pointers */
  
  if (p->value != 0)
	len += sizeof(char *) +				/* size of array pointer + */
	  strlen(p->name) + sizeof(char) +	/* size of name plus '=' + */
	  strlen(p->value) + sizeof(char);	/* size of value plus NULL char */
  else		/* if no value */
	len += sizeof(char *) +				/* size of array pointer + */
	  strlen(p->name) + sizeof(char);		/* size of name plus NULL char */
}

len += sizeof(char *);						/* size of null pointer */
count++;									/* one more array pointer */

envp = (char **) malloc(len);
if (envp == NULL)
  return NULL;

cp = (char *) &envp[count];		/* beginning of null terminated strings */

/* create envp array */

for (count = 0, p = envChain; p != NULL; p = p->next) {
  envp[count++] = cp;
  strcpy(cp, p->name);
  cp += strlen(p->name);
  if (p->value != 0) {
	*cp++ = '=';
	strcpy(cp, p->value);
	cp += strlen(p->value);
  }
  *cp++ = '\0';
}

envp[count++] = NULL;			/* terminate array with null pointer */

return envp;
}

/*************************************************************
*  do_diagsw(name,value)
*	Called when the envvar "diag" is changed.
*	"diag 0" turns diag off
*	"diag 1" turns on diag level 1 with output to stdout
*	"diag 2" turns on diag level 2 with output to stdout
*	"diag 1:tty1" turns on diag level 1 with output to tty1
*	"diag 2:tty1" turns on diag level 2 with output to tty1
*/
do_diagsw(name,value)
char *name,*value;
{
char devname[20];
int n;

#if defined(VR4300) || defined(LR64388)
/* the sscanf crashes on these processors. I have no idea why. */
return;
#endif

if (dfp && dfp != stdout) fclose(dfp);
dfp = 0;
n = sscanf(value,"%d:%s",&verbose,devname);
if (verbose == 0) return(0);

switch (n) {
	case 1 : /* verbose level only */
		dfp = stdout;
		break;
	case 2 : /* verbose level + device name */
		dfp = fopen(devname,"w");
		if (dfp) break;
		printf("can't open %s\n",devname);
		break;
	default:
		printf("%s: bad value\n",value);
	}
return(0);
}

