/*************************************************************
 * File: mon/dump.c
 * Purpose: Part of core Monitor
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970901	Added special meaning for "d -R addr", dump host mem
 *	980917	Added help for -S.
 *	980917	Fixed -s option. Needed read_target not *adr.
 */

#include <mon.h>

Optdesc d_opts[] = {
	{"[-bhwsS] adr [cnt]|-rreg","display memory"},
	{"-b","display as bytes"},
	{"-h","display as half-words"},
	{"-w","display as words"},
        {"-fR","display as fp values (reversed word order)"},
	{"-s","display a null terminated string"},
	{"-S","cnt is count of words, bytes, or halfwords"},
	{"-r<reg>","display as register"},
	{0}};

#define D_RFLAG         0x0001

/*************************************************************
*  dump(ac,av)
*	the 'd' command 
*/
dump(ac,av)
int ac;
char *av[];
{
Ulong adr, moresz, siz, l,v;
char *reg;
int i,j,datasz,n,bysize,flags;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

if (!atob(&moresz,getMonEnv("moresz"),10)) {
	printf("%s: bad moresz value\n",getMonEnv("moresz"));
	return(1);
	}

bysize = 0;
i = matchenv("datasz");
switch (i) {
	case 0 : datasz = 1; break;
	case 1 : datasz = 2; break;
	case 2 : datasz = 4; break;
	default :
		printf("%s: bad datasz value\n",getMonEnv("datasz"));
		return(1);
	}

flags = 0;
n = 0;
for (i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		for (j=1;av[i][j] != 0;j++) {
			switch (av[i][j]) {
                                case 'R' : flags |= D_RFLAG; break;
                                case 'f' : datasz=8; break;
				case 'S' : bysize=1; break;
				case 'w' : datasz=4; break;
				case 'b' : datasz=1; break;
				case 'h' : datasz=2; break;
				case 's' : datasz=0; break;
				case 'r' : 
					reg = &av[i][j+1];
					datasz = -1; 
					break;
				default :
					printf("%c: unknown option\n",av[i][j]);
					return(1);
				}
			if (datasz == -1) break;
			}
		}
	else {
		switch (n) {
			case 0 :
				if(!get_rsa(&adr,av[i])) return(1);
				break;
			case 1 :
				if(!get_rsa(&siz,av[i])) return(1);
				break;
			default :
				printf("%s: unknown argument\n",av[i]);
				return(1);
			}
		n++;
		}
	}

if (n == 0) {
	printf("address not specified\n");
	return(1);
	}

if (datasz == -1) { /* -r<reg> print as register */
	if (!disp_as_reg(read_target(XT_MEM,adr,4),reg)) 
		printf("%s: bad register name\n",reg);
	return(0);
	}

if (datasz != 0) {
  if (adr&1) datasz = 1;
  else if (datasz != 1 && adr&2) datasz = 2;
  
  if (n < 2) siz = moresz;
} else {
  if (n < 2) siz = 0;
}

if (bysize) {
	switch (datasz) {
		case 1 : moresz = (siz+15)/16; siz = (siz>16)?16:siz; break;
		case 2 : moresz = (siz+7)/8 ; siz = (siz>8)?8:siz; break;
		case 4 : moresz = (siz+3)/4 ; siz = (siz>4)?4:siz; break;
		}
	} else moresz = siz;

if (datasz == 0) { /* -s, print string */
  i = 0;
  while(1) {
	v = read_target8(adr++);
	prnbuf[i] = v;
	if (v != 0 && !isprint(v)) prnbuf[i] = '.';
	if (v == 0 || i >= 70 || (siz != 0 && i >= siz)) {
	  prnbuf[i] = 0;
	  printf("%s\n",prnbuf);
	  if (i >= siz) return(0);
	  siz -= i;
	  i = 0;
	}
	i++;
  }
}

ioctl_cbreak(0L);
l = moresz;
while(1) {
	adr = dispmem(prnbuf,adr,(bysize && (n > 1))?siz:0,datasz,flags);
	if (more(prnbuf,&l,(n > 1)?0:moresz)) break;
	}
}

/*************************************************************
*  static Ulong read_host_mem(adr,sz)
*	Added to support the special "d -R" command.
*/
static Ulong read_host_mem(adr,sz)
Ulong adr;
int sz;
{
switch (sz) {
	case 1 : return (*(Uchar *)(adr));
	case 2 : return (*(Ushort *)(adr));
	case 4 : return (*(Ulong *)(adr));
	}
}

/*************************************************************
*  Ulong dispmem(p,adr,lim,siz,flags)
*	writes one line of output to p.
*/
Ulong dispmem(p,adr,lim,siz,flags)
char *p;
Ulong adr;
int lim,siz,flags;
{
int i;
char tmp[10],*q;
char buf[16];
unsigned short *ph;
Ulong *pw,v,a;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

#ifdef FLOATINGPT
if (siz == 8) return dispfp(p,adr,flags);
#endif

adr &= ~(siz-1); /* align */
a = adr;
q = buf;
for (i=0;i<(16/siz);i++) {
	if (lim != 0 && i >= lim)
		break;
	if (flags&D_RFLAG) v = read_host_mem(a,siz); /* 970901 */
	else v = read_target(XT_MEM,a,siz);
	switch (siz) {
		case 1 : *q++ = v; break;
		case 2 : *((unsigned short *)q) = v; q+=2; break;
		case 4 : *((Ulong *)q) = v; q+=4; break;
		}
	a += siz;
	}

sprintf(p,"%08x  ",adr);
for(i = 0; i < 16; i += siz){
	if (lim != 0 && i >= (lim*siz))
		break;
	if(i == 8) strccat(p,' ');
	switch (siz) {
		case 1 :
			sprintf(tmp,"%02x ",buf[i]&0xff);
			break;
		case 2 :
			ph = (unsigned short *)&buf[i];
			sprintf(tmp,"%04x ",(*ph)&0xffff);
			break;
		case 4 :
			pw = (unsigned long *)&buf[i];
			sprintf(tmp,"%08x ",*pw);
			break;
		}
	strcat(p,tmp);
	}
strcat(p,"  ");
for(i = 0; i < 16 ; i++){
	if (lim != 0 && i >= lim)
		break;
	v = buf[i];
	if (isprint(v)) strccat(p,(Uchar)v);
	else strccat(p,'.');
	}
return(adr+16);
}

#ifdef FLOATINGPT
/*************************************************************
*  Ulong dispfp(p,adr,flags)
*/
Ulong dispfp(p,adr,flags)
char *p;
Ulong adr;
int flags;
{
int v0,v1,temp[2];
double *dp;
float *sp0,*sp1;
 
sp0 = (float *)&v0;
sp1 = (float *)&v1;
dp = (double *)temp;
v0 = read_target(XT_MEM,adr,4);
v1 = read_target(XT_MEM,adr+4,4);
if (flags&D_RFLAG) {
        temp[0] = read_target(XT_MEM,adr+4,4);
        temp[1] = read_target(XT_MEM,adr,4);
        }
else {
        temp[0] = read_target(XT_MEM,adr,4);
        temp[1] = read_target(XT_MEM,adr+4,4);
        }
#undef sprintf
sprintf(p,"%08x = %08x %13e %08x %13e dp=%e",adr,read_target(XT_MEM,adr,4),*sp0,
                                read_target(XT_MEM,adr+4,4),*sp1,*dp);
#define sprintf xsprintf
return(adr+8);
}
#endif

