/*************************************************************
 * File: sairq.c
 * Author: Phil Bunce (pjb@carmel.com)
 * Purpose:
 * 	This function makes it very easy to install interrupt handlers
 *	that are written in C. This version is for standalone environments
 *	eg. non PMON.
 * Revision History:
 *	981016	Created from pmirq.c
 */

#include <mips.h>
#include <utypes.h>
#include <irqvec.h>

#ifndef SR_ISHIFT	/* just in case you don't have this yet */
#define SR_ISHIFT	8
#endif

/***********************************************************************
*  Func *saIRQInstall(int IRQNum, Func *ISRFun)
*
* Install and enable a new Interrupt Service Routine.
*
* int IRQNum - interrupt request number (0..23)
*        0 = sw0        lo priority
*        1 = sw1             |
*        2 = int0            |
*        3 = int1            |
*        4 = int2            |
*        5 = int3            |
*        6 = int4            v
*        7 = int5       hi priority
*
* Func *ISRFun - new interrupt handler
*	If ISRFun is zero, the specified vector will be uninstalled
*
* Return:
*     The old interrupt service routine address.
*
* Description:
*     The installed function gets passed a pointer to the register
*     save area. Offsets are 0 thru 31 where the offset equals the
*     register number except for the following cases:
*
*                0 = EPC
*               26 = HI
*               27 = LO
*               28 = CAUSE
*               29 = SR
*
*     The ISRFun *must* return non-zero if it handled the exception.
*     If it returns zero, control will be passed to the ICEKernel.
*/
Func *saIRQInstall(IRQNum, ISRFun)
int IRQNum;
Func *ISRFun;
{
        Func * OldISR;

        /* Sanity check */
        if (IRQNum < 0 || IRQNum >= MAXINTHNDLRS)
                return((Func *)0);

	/* only does this the first time */
	if (pri_table[1]==0) buildPriTable(pri_table);

        /* Get old ISR function */
        OldISR = IRQVector[IRQNum];

	if (!ISRFun) { /* uninstall */
		if (IRQNum < 8) 
			mtc0(C0_SR,mfc0(C0_SR)&~(1<<(SR_ISHIFT+IRQNum)));
		IRQVector[IRQNum] = 0;
		return(OldISR);
		}

        /* Replace with new ISR function */
        IRQVector[IRQNum] = ISRFun;

	if (IRQNum < 8) {
		/* Enable the interrupt */
		mtc0(C0_SR,mfc0(C0_SR)|(1<<(SR_ISHIFT+IRQNum))|SR_IEC);
		}

        return(OldISR);
}

