/*************************************************************
 * File: lib/r333cfl.s
 * Purpose: Part of C runtime library
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	980306	Added DCACHEI case.
 *	980409	Added 'dummy' to #else clause
 */

#include <defines.h>
#include <mips.h>
#ifdef LR33300

/*--------------------------------------------------------------------
These routines flush the LR33300 and LR33310 caches.

	sw to BIU
		INV|IS1|IS0  	i-cache
		INV|DS	 	d-cache
		can invalidate i&d caches at the same time.
	disable ints
	set SR:IsC bit

	sw	zero,(t0)	# for each line invalidates cache


It is possible to flush the I&D caches at the same time.
However, the structure of PMON expects them to be separate. 

This code has been modfied so that locked entries in the LR33310 IS0
are unaffected by normal flushes.

r3330_flush:
	ICACHEI: r333_iflush..r333_end
	ICACHE: r333_iflush..r333_end
	DCACHEI: r333_dflush..r333_end
	DCACHE: r333_dflush..r333_end
	IACACHE: r333_iaflush..r333_end

r3331_flush:
	ICACHEI: r333_iflush..r333_end
	ICACHE: r333_iflush0..r333_iflush..r333_end
	DCACHE: r333_dflush..r333_end
	DCACHE: r333_dflush..r333_end
	IACACHE: r333_iaflush0..r333_iaflush..r333_end
--------------------------------------------------------------------*/

/*************************************************************
*  r3330_flush(type,adr)
*	Flush the designated LR33300 cache
*		ICACHEI	-- initial flush of Icaches (clear everything)
*		ICACHE	-- flush Icaches leaving LCK'd entries untouched
*		DCACHEI	-- flush the Dcache (initial)
*		DCACHE	-- flush the Dcache
*		IADDR	-- flush a single line in the Icaches
*/
	.globl r3330_flush
	.ent r3330_flush
r3330_flush:
	la	t0,1f
	or	t0,K1BASE
	j	t0

   1:	# disable ints
	.set noreorder
	mfc0	a2,C0_SR
	nop
	and	t0,a2,~SR_IEC
	mtc0	t0,C0_SR
	.set reorder

	bne	a0,ICACHEI,1f
	li	a0,BIU_IS1
	j	r333_iflush

   1:	bne	a0,ICACHE,1f
	li	a0,BIU_IS1
	j	r333_iflush

   1:	bne	a0,DCACHE,1f
	j	r333_dflush

   1:	bne	a0,DCACHEI,1f
	j	r333_dflush

   1:	move	a0,a1
	j	r333_iaflush
	.end r3330_flush

/*************************************************************
*  r3331_flush(type,adr)
*	Flush the designated LR33310 cache
*		ICACHEI	-- initial flush of Icaches (clear everything)
*		ICACHE	-- flush Icaches leaving LCK'd entries untouched
*		DCACHE	-- flush the Dcache
*		IADDR	-- flush a single line in the Icaches
*/
	.globl r3331_flush
	.ent r3331_flush
r3331_flush:
	la	t0,1f
	or	t0,K1BASE
	j	t0

   1:	# disable ints
	.set noreorder
	mfc0	a2,C0_SR
	nop
	and	t0,a2,~SR_IEC
	mtc0	t0,C0_SR
	.set reorder

   	bne	a0,ICACHEI,1f
	li	a0,(BIU_IS1|BIU_IS0)
	j	r333_iflush

   1:	bne	a0,ICACHE,1f
	j	r333_iflush0

   1:	bne	a0,DCACHE,1f
	j	r333_dflush

   1:	bne	a0,DCACHEI,1f		# 980306
	j	r333_dflush

   1:	move	a0,a1
	j	r333_iaflush0
	.end r3331_flush

/*************************************************************
*  r333_iflush0()
*	Flush the non-locked entries in the LR33310 Icache0.
*/
	.globl r333_iflush0
	.ent r333_iflush0
r333_iflush0:
	# select the correct cache set(s)
	li	t1,M_BIU
	lw	t4,(t1)
	or	t0,t4,(BIU_TAG|BIU_IS0)
	and	t0,~(BIU_DS1|BIU_DS0)	# make sure Dcache is NOT enabled
	sw	t0,(t1)

	.set noreorder
	# set IsC bit
	mfc0	t3,C0_SR
	nop
	or	t0,t3,SR_ISC
	mtc0	t0,C0_SR
	.set reorder

	# 33310 has 256 lines of 4 words in the Icache
	li	t1,K0BASE
	addu	t0,t1,256*4*4
   2:	lw	t2,(t1)
	sll	t5,t2,31-5	# move LCK bit into MSB posn
	bltz	t5,1f		# bra if LCK is set
	and	t2,~0xf		# clear valid bits
	sw	t2,(t1)		# update the tag bits
   1:
	addu	t1,16*4
	bltu	t1,t0,2b

	# restore IsC bit
	.set noreorder
	mtc0	t3,C0_SR
	.set reorder

	# restore BIU
	li	t1,M_BIU
	sw	t4,(t1)

	li	a0,BIU_IS1
	j	r333_iflush
	.end r333_iflush0

/*************************************************************
*  r333_iflush()
*	initialize the LR33300/LR33310 Icache (clear the lock bits)
*	The set(s) to operate on are specified by a0.
*/
	.globl r333_iflush
	.ent r333_iflush
r333_iflush:
	li	t1,M_BIU
	lw	a3,(t1)
	or	t0,a3,BIU_INV
	and	t0,~(BIU_DS1|BIU_DS0|BIU_IS0|BIU_IS1)
	or	t0,a0	# a0 specifies which Isets to flush
	sw	t0,(t1)

	.set noreorder
	# set IsC bit
	mfc0	t3,C0_SR
	nop
	or	t0,t3,SR_ISC
	mtc0	t0,C0_SR
	.set reorder

	# 333x0 has 256 lines in the Icache
	li	t1,K0BASE
	addu	t0,t1,256*4*4
1:	sw	zero,0*4(t1)
	sw	zero,4*4(t1)
	sw	zero,8*4(t1)
	sw	zero,12*4(t1)
	addu	t1,16*4
	bltu	t1,t0,1b

	# restore IsC bit
	.set noreorder
	mtc0	t3,C0_SR
	.set reorder

	j	r333_end
	.end r333_iflush


/*************************************************************
*  r333_end
*	common end code
*	a2 = SR value
*	a3 = BIU value
*/
	.globl r333_end
	.ent r333_end
r333_end:
	.set noreorder
	# restore BIU
	li	t1,M_BIU
	sw	a3,(t1)
	nop	# these nops are necessary (maybe not all four)
	nop
	nop
	nop
	# restore IEC bit
	mtc0	a2,C0_SR
	nop
	nop
	.set reorder

	j	ra
	.end r333_end

/*************************************************************
*  r333_iaflush0(adr)
*	Flush one line of the LR33310 Icache0 unless locked.
*/
	.globl r333_iaflush0
	.ent r333_iaflush0
r333_iaflush0:
	# select the correct mode and cache set
	li	t1,M_BIU
	lw	t4,(t1)
	or	t0,t4,(BIU_TAG|BIU_IS0)
	and	t0,~(BIU_DS1|BIU_DS0)	# make sure Dcache is NOT enabled
	sw	t0,(t1)

	.set noreorder
	# set IsC bit
	mfc0	t3,C0_SR
	nop
	or	t0,t3,SR_ISC
	mtc0	t0,C0_SR
	.set reorder

	and	a0,~3		# word align the address
   	lw	t2,(a0)
	sll	t5,t2,31-5	# move LCK bit into MSB posn
	bltz	t5,1f		# bra if LCK is set
	and	t2,~0xf		# clear valid bits
	sw	t2,(a0)		# update the tag bits
   1:

	# restore IsC bit
	.set noreorder
	mtc0	t3,C0_SR
	.set reorder

	# restore BIU
	li	t1,M_BIU
	sw	t4,(t1)

	j	r333_iaflush
	.end r333_iaflush0

/*************************************************************
*  r333_iaflush(adr)
*	Flush one line of the LR33300/LR33310 Icache
*/
	.globl r333_iaflush
	.ent r333_iaflush
r333_iaflush:
	li	t1,M_BIU
	lw	a3,(t1)
	or	t0,a3,(BIU_INV|BIU_IS1)
	and	t0,~(BIU_DS1|BIU_DS0)	# make sure Dcache is NOT enabled
	sw	t0,(t1)

	.set noreorder
	# set IsC bit
	mfc0	t3,C0_SR
	nop
	or	t0,t3,SR_ISC
	mtc0	t0,C0_SR
	.set reorder

	# flush one line
	and	a0,~3		# make sure adr is word aligned
	sw	zero,(a0)

	# restore IsC bit
	.set noreorder
	mtc0	t3,C0_SR
	.set reorder

	j	r333_end
	.end r333_iaflush

/*************************************************************
*  r333_dflush()
*	Flush the LR33300/LR33310 Dcache
*/
	.globl r333_dflush
	.ent r333_dflush
r333_dflush:
	li	t1,M_BIU
	lw	a3,(t1)
	or	t0,a3,(BIU_INV|BIU_DS1|BIU_DS0)
	and	t0,~(BIU_IS1|BIU_IS0)	# make sure Icache is NOT enabled
	sw	t0,(t1)

	.set noreorder
	# set IsC bit
	mfc0	t3,C0_SR
	nop
	or	t0,t3,SR_ISC
	mtc0	t0,C0_SR
	.set reorder

	# 33300 has 128 lines, 33310 256 lines
	# use largest value
	li	t1,K0BASE
	addu	t0,t1,256*4*4
1:	sw	zero,0*4(t1)
	sw	zero,4*4(t1)
	sw	zero,8*4(t1)
	sw	zero,12*4(t1)
	addu	t1,16*4
	bltu	t1,t0,1b

	# restore IsC bit
	.set noreorder
	mtc0	t3,C0_SR
	.set reorder

	j	r333_end
	.end r333_dflush

#else /* some versions of BSO tools don't like empty modules */
	.globl r3330_flush_dummy
	.ent r3330_flush_dummy
r3330_flush_dummy:
	j	ra
	.end r3330_flush_dummy
#endif /* LR33300 */
