/*************************************************************
 * File: lib/nvram.c
 * Purpose: Part of C runtime library
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970311	Added flush_cache(DCACHE) for 4010 copy-back
 *	970605	Added writeFlash
 *	970620	Cygnus requires Uchar in flash29.
 *	970910	Added support for Am29F080.
 *	971124	Don't create msg for famcode=0000.
 *	971125	Added code to writeFlash for 4011 copy from boot eprom
 *	980114	Added delay to start of nvType for GHS+4011.
 *	980120	Replaced outb() macro with a function. Fixes prob w
 *		nvType check on 4011. Removed delay.
 *	980324	Fixed problem with outb. It wasn't doing the right thing.
 *	980508	Added needcpy.
 *	980508	Flush D before I in nvCheck.
 *	980509	Discovered that for needcpy=1 outb MUST be a macro.
 *	980509	The new outb macro works on 4011 but fails on 4101.
 *	980510	Reverted to old outb macro until real fix is found.
 *	980602	Delay shrc execution until after the powerup banner.
 *		Do this by setting shrc_needed.
 *	980715	Modified do_shrc and shrc to support 2 separate init
 *		blocks. Before p2 is executed before the banner. After
 *		p2 is executed after the banner. Only commands that
 *		were specified in the "set nvram" command are located
 *		after the p2.
 */

/************************************************************* 
 * Performance data:
 *	3 secs to erase one sector
 *	128 secs/MB for writes
 */

#include <stdio.h>
#include <mips.h>
#include <termio.h>
#include <mon.h>
#include <flashconfig.h>

extern unsigned char FLASH_CONFIG_ADDRESS;

#define delay(x)										\
{														\
int b1, b2;												\
/* skip first access in case of write completion */		\
inb(x) != inb(x);										\
do {													\
	b1 = inb(adrA);										\
	b2 = b1 ^ inb(adrA);								\
/* b2 non-zero if flash busy or error */				\
/* b2 & 0x40 if either flash timeout or flash busy */	\
/* when flash busy and we should wait, !(b1 & 0x20) */	\
} while (b2 != 0 && (b2 & 0x40) && !(b1 & 0x20));		\
}


#define inb(x)	(*(volatile Uchar *)(x))

#if 0
#define outb(a,v)   *((Uchar *)(a))=((Uchar)(v))
#else
/*************************************************************
*  outb(a,v)   
*	store byte with write buffer flush.
*	For 4011 you MUST use Kseg1 addresses for tmp1 and tmp2.
*	980324 Added & for wbflushtmpX and volatile for wbftX.
*/
#define outb(a,v)   					\
{							\
volatile int *wbft1,*wbft2;				\
							\
*((Uchar *)(a))=((Uchar)(v));				\
wbft1 = (int *)(((Ulong)&wbflushtmp1)|K1BASE);		\
wbft2 = (int *)(((Ulong)&wbflushtmp2)|K1BASE);		\
*wbft1 = 0;						\
*wbft2 = *wbft1;					\
}
#endif

int nvTypeCFI(),nvTypeCFI_end();
int nvTypeJEDEC(),nvTypeJEDEC_end();
#if 0
Ulong flash29(); void flash29end();
#endif
Ulong flash02(); void flash02end();

NvType nvTypesCFI[] = {
	{0x0002,"CFI generic 0002",1<<16,flash02,flash02end},
	{0x0100,"AMD generic 0002",1<<16,flash02,flash02end},
	{0x989a,"TC58FVT321",1<<16,flash02,flash02end},
	{0x989C,"TC58FVB321",1<<16,flash02,flash02end},
	{0x9893,"TC58FVT641",1<<16,flash02,flash02end},
	{0}};

NvEraseInfo x29F400BTEraseInfo = {
  4,
  {64*1024,7,
  32*1024,1,
  8*1024,2,
  16*1024,1}};

NvEraseInfo x29F400BBEraseInfo = {
  4,
  {16*1024,1,
  8*1024,2,
  32*1024,1,
  64*1024,7}};

NvEraseInfo x29F800BTEraseInfo = {
  4,
  {64*1024,15,
  32*1024,1,
  8*1024,2,
  16*1024,1}};

NvEraseInfo x29F800BBEraseInfo = {
  4,
  {16*1024,1,
  8*1024,2,
  32*1024,1,
  64*1024,15}};

NvType nvTypesJEDEC[] = {
  {0x0123,"AM29F400BT",512*1024,flash02,flash02end,&x29F400BTEraseInfo},
  {0x01ab,"AM29F400BB",512*1024,flash02,flash02end,&x29F400BBEraseInfo},
  {0xc223,"MX29F400BT",512*1024,flash02,flash02end,&x29F400BTEraseInfo},
  {0xc2ab,"MX29F400BB",512*1024,flash02,flash02end,&x29F400BBEraseInfo},
  {0xadd6,"HY29F800BT",1024*1024,flash02,flash02end,&x29F800BTEraseInfo},
  {0xad58,"HY29F800BB",1024*1024,flash02,flash02end,&x29F800BBEraseInfo},
  {0}};

NvInfo nvInfo[MAXFLASH];

NvEraseInfo eraseInfo[MAXFLASH];


#define SPC_BITMSK (3<<21)
#define SPC_8BIT   (1<<21)
#define SPC_16BIT  (2<<21)
#define SPC_32BIT  (3<<21)

int machtype;
volatile int wbflushtmp1;
volatile int wbflushtmp2;

#ifdef TEST

/*************************************************************
*  main()
*/
main(argc,argv)
int argc;
char *argv[];
{
int n,i,j,c;
char nvtype[40];

machtype = getmachtype();
if (!nvCheck(nvtype)) printf("nvCheck FAILED\n");;
printf("nvram: %s\n",nvtype);

printf("reading: ");
for (i=0;;i++) {
	c = nvRead(i);
	if (c == 0 || c == 0xff) break;
	putchar(c);
	}
putchar('\n');

if (argc == 1) return;

for (j=1,i=0;j<argc;j++) {
	if (argv[j][0] == '-') {
		if (argv[j][1] == 'c') nvClear(0);
		else if (argv[j][1] == 'v') {
			j++;
			for (i=0;j<argc;j++) {
				for (p=argv[j];*p;p++) nvWrite(i++,*p);
				nvWrite(i++,' ');
				}
			}
		else {
			printf("%s: bad option\n",argv[j][1]);
			printf("usage: [-c] [-v msg]\n");
			exit(1);
			}
		}
	else {
		printf("%s: bad argument\n",argv[j]);
		printf("usage: [-c] [-v msg]\n");
		exit(1);
		}
	}

printf("reading: ");
for (i=0;;i++) {
	c = nvRead(i);
	if (c == 0 || c == 0xff) break;
	putchar(c);
	}
putchar('\n');
}
#endif

/*************************************************************
*  static int nvCheckCommon(char *msg, int slot
*								Func *nvType, Func *nvTypeEnd, NvType *nvTypes)
*/
static int nvCheckCommon(msg,slot,nvType,nvTypeEnd,nvTypes)
char *msg;
int slot;
Func *nvType,nvTypeEnd;
NvType *nvTypes;
{
Ulong r;
Func *f;
int len,i,j,famcode;

#ifdef BRECISNOFLASH
return (0);
#endif

/* first, find out what nvType we have */
famcode = nvType(slot);

/* look up the famcode in the nvTypesJEDEC or nvTypesCFI table */
for (i=0;nvTypes[i].driver;i++) {
	if (nvTypes[i].famcode == famcode) break;
	}
if (!nvTypes[i].driver) { /* famcode not found */
  eraseInfo[slot].num_erase_blocks = 0;
  return(0);
  }
nvInfo[slot].name = nvTypes[i].name;

/* now, install the correct driver */
nvInfo[slot].driver = nvTypes[i].driver;

if (!nvInfo[slot].size) nvInfo[slot].size = nvTypes[i].size;
nvInfo[slot].type = &nvTypes[i];

/*
 * If we have an 8 meg flash chip and the start address was
 * 0xbf800000, we are done.
 * If we have a 4 meg flash chip and the start address was
 * 0xbf800000, we need to adjust the start address to be
 * 0xbfc00000 because the HARDWARE will cause the 0xbf800000
 * to appear to also be at 0xbfc00000 because the MIPS expects
 * the rom start address to be at 0xbfc00000.
 */
if (nvInfo[slot].start == 0xbf800000 && nvInfo[slot].size < 0x800000)
  {
	nvInfo[slot].start = 0xbfc00000;
  }

if (eraseInfo[slot].num_erase_blocks == 0 && 
	nvTypes[i].eraseInfo != 0) {
  eraseInfo[slot].num_erase_blocks = nvTypes[i].eraseInfo->num_erase_blocks;
  for (j = 0; j < nvTypes[i].eraseInfo->num_erase_blocks; j++) {
	eraseInfo[slot].erase_block[j].sector_size =
	  nvTypes[i].eraseInfo->erase_block[j].sector_size;
	eraseInfo[slot].erase_block[j].num_sectors =
	  nvTypes[i].eraseInfo->erase_block[j].num_sectors;
  }
}

/* return if this device doesn't support sector erase */
if (eraseInfo[slot].num_erase_blocks == 0) {
	if (msg) strcpy(msg,"Erase function not supported");
	return(0);
	}

if (msg) strcpy(msg,nvInfo[slot].name);
return(1);
}

/*************************************************************
*  static int nvCheckCFI(char *msg)
*/
static int nvCheckCFI(msg,slot)
char *msg;
int slot;
{
  return nvCheckCommon(msg,slot, nvTypeCFI, nvTypeCFI_end, nvTypesCFI);
}

/*************************************************************
*  static int nvCheckJEDEC(char *msg)
*/
static int nvCheckJEDEC(msg,slot)
char *msg;
int slot;
{
  return nvCheckCommon(msg,slot, nvTypeJEDEC, nvTypeJEDEC_end, nvTypesJEDEC);
}

/*************************************************************
*  int nvCheck(char *msg, int slot)
*/
nvCheck(msg,slot)
char *msg;
int slot;
{
int i,result;

/* I believe msp == 0 is used as a flag for nvram, msg != 0 means flash */

if (msg == 0 || slot > MAXFLASH) return 0;

*msg = 0;
if (nvInfo[slot].driver) return(1);

nvInfo[slot].size = 0;

if (nvCheckCFI(msg,slot) || nvCheckJEDEC(msg,slot)) result = 1;
else result = 0;

return result;
}

/*************************************************************
*  int findFlashSlot(adr)
*  return flash slot -or- -1 if slot not found
*/
int findFlashSlot(adr)
Ulong adr;
{
int slot;

for (slot=0; slot<MAXFLASH; slot++)
  {
	if (adr >= nvInfo[slot].start &&
		nvInfo[slot].size > (adr-nvInfo[slot].start))
	  {
		if (!nvInfo[slot].driver) continue;
		break;
	  }
  }

if (slot == MAXFLASH) return -1;

return slot;
}

/*************************************************************
*  nvTypeCFI(slot)
*	determine family code
*	what we do know at start:
*		start address of device
*		width
*/
nvTypeCFI(slot)
int slot;
{
Ulong adrA,adrB,adrC;
int i,sh,inc;
Ulong vendoralg,sector_size;
char qry[3];

for (sh = 0; sh < 3; sh++)
{
	adrA = nvInfo[slot].start;
	adrB = (nvInfo[slot].start | (0x055<<sh));

	inc = 1<<sh;
	
	outb(adrA, 0xf0);		/* reset the flash */

	delay(adrA);

	outb(adrB, 0x98);
	
	delay(adrA);

	for (i = 0; i < 3; i++)
		qry[i] = inb(adrA + (0x10 + i) * inc);

	if (qry[0] == 'Q' && qry[1] == 'R' && qry[2] == 'Y')
	{
		/* flash follows CFI standard */

		nvInfo[slot].width = inc;
		nvInfo[slot].size = 1<<inb(adrA + 0x27 * inc);

		vendoralg = (inb(adrA + 0x14 * inc)) | inb(adrA + 0x13 * inc);

		adrB = adrA + 0x2c * inc;
		eraseInfo[slot].num_erase_blocks = inb(adrB);

		adrB += inc;	/* point to first erase block */

		for (i = 0; i < eraseInfo[slot].num_erase_blocks; i++)
		{
			/* 
			 * an erase block structure consists of one less than
			 * the number of sectors in the chip followed by
			 * the sector size * 256 where a sector size of 0 means 128.
			 */
			eraseInfo[slot].erase_block[i].num_sectors =
				inb(adrB) + ((inb(adrB+inc))<<8);
			eraseInfo[slot].erase_block[i].num_sectors++;
			adrB += inc + inc;

			sector_size = inb(adrB) + ((inb(adrB+inc))<<8);
			adrB += inc + inc;
			eraseInfo[slot].erase_block[i].sector_size =
				sector_size ? (sector_size*256) : 128;

		}

		outb(adrA, 0xf0);		/* reset the flash */

		delay(adrA);

		/* determine manufacturing and device code */

		adrB = (nvInfo[slot].start | (0x555 * inc));
		adrC = (nvInfo[slot].start | (0x2AA * inc));
		outb(adrB, 0xaa);
		outb(adrC, 0x55);
		outb(adrB, 0x90);

		delay(adrA);

		if (inb(adrA) == 0x98 &&
			(inb(adrA + inc) == 0x9a || inb(adrA + inc) == 0x93))
		  {
			int num;
			int ss;
			int total;

			/* 
			 * if Toshiba manufacturer (0x98) and 
			 * TC58FVT321 (0x9a) or TC58FVT641 (0x93)
			 * the erase blocks are reversed...so reverse our eraseInfo
			 * data structure.
			 */
			vendoralg = 0x989a;

			total = eraseInfo[slot].num_erase_blocks;
			for (i = 0; i < (total/2); i++)
			  {
				ss = eraseInfo[slot].erase_block[i].sector_size;
				num = eraseInfo[slot].erase_block[i].num_sectors;

				eraseInfo[slot].erase_block[i].sector_size =
				  eraseInfo[slot].erase_block[total-i-1].sector_size;
				eraseInfo[slot].erase_block[i].num_sectors =
				  eraseInfo[slot].erase_block[total-i-1].num_sectors;

				eraseInfo[slot].erase_block[total-i-1].sector_size = ss;
				eraseInfo[slot].erase_block[total-i-1].num_sectors = num;
			  }
		  }
		else if (inb(adrA) == 0x98 && inb(adrA + inc) == 0x9c)
		{
			vendoralg = 0x989c;
		}
		else if (inb(adrA) == 0x01)
		{
			vendoralg = 0x0100;
		}

		outb(adrA, 0xf0);		/* reset the flash */
			
		delay(adrA);

		nvInfo[slot].standard = NV_CFI;

		return vendoralg;
	}
}

return 0;
}
nvTypeCFI_end() {}

/*************************************************************
*  flash02(slot,op,sector,adr,val,nbytes) 
*/
Ulong flash02(slot,op,sector,adr,val,nbytes) 
int slot,op;
Ulong sector,adr,val,nbytes;
{
Ulong adrA,adrB,adrC,result;
char *p;

adrA = sector;
adrB = (sector | (0x555 * nvInfo[slot].width));
adrC = (sector | (0x2AA * nvInfo[slot].width));
result = 0;

outb(adrA, 0xf0);						/* reset device to read mode */

delay(adrA);

again:

switch (op) {
case NV_WRITE:
	outb(adrB, 0xaa);		/* unlock 1 */
	outb(adrC, 0x55);		/* unlock 2 */

	outb(adrB, 0xa0);		/* program */
	outb(adr, val);
	break;
  
case NV_ERASE:
	outb(adrB, 0xaa);		/* unlock 1 */
	outb(adrC, 0x55);		/* unlock 2 */

	outb(adrB, 0x80);		/* erase */

	outb(adrB, 0xaa);
	outb(adrC, 0x55);
	outb(adrB, 0x10);
	break;
  
case NV_SERASE:
	outb(adrB, 0xaa);		/* unlock 1 */
	outb(adrC, 0x55);		/* unlock 2 */

	outb(adrB, 0x80);		/* erase */

	outb(adrB, 0xaa);
	outb(adrC, 0x55);
	outb(sector, 0x30);
	break;

case NV_BLOCKWRITE:
	p = (char *) adr;

	while (nbytes--)
	{
		outb(adrB, 0xaa);		/* unlock 1 */
		outb(adrC, 0x55);		/* unlock 2 */

		outb(adrB, 0xa0);		/* program */
		outb(val++, *p++);

		delay(adrA);
	}
	break;

case NV_SECTORSTATUS:
		outb(adrB, 0xaa);
		outb(adrC, 0x55);
		outb(adrB, 0x90);

		delay(adrA);

		result = ((inb(adrA)) << 24) |
				((inb(adrA + nvInfo[slot].width)) << 16) |
				((inb(adrA + 2*nvInfo[slot].width)) << 8) |
				((inb(adrA + 3*nvInfo[slot].width)));
default:
	break;
}

delay(adrA);

outb(adrA, 0xf0);						/* reset device to read mode */

delay(adrA);

return result;

}
void flash02end() {}



/*************************************************************
*  nvTypeJEDEC(slot)
*	determine family code
*	what we do know at start:
*		start address of device
*		width
*/
nvTypeJEDEC(slot)
int slot;
{
Ulong adrA,adrB,adr0,adr1;
Ulong msk;
int sh,b1,b2,h0,h1;

for (sh = 0; sh < 3; sh++)
{
  adrA = (nvInfo[slot].start|(0x555<<sh));
  adrB = (nvInfo[slot].start|(0x2aa<<sh));
  adr0 = (nvInfo[slot].start|(0x000<<sh));
  adr1 = (nvInfo[slot].start|(0x001<<sh));
  b1 = inb(adr0);
  b2 = inb(adr1);
  h0 =  (b1<<8)|b2; /* read the existing value */

  outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0x90); 
  b1 = inb(adr0);
  b2 = inb(adr1);
  outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0xf0); 
  h1 =  (b1<<8)|b2; /* read the famcode */

  if (h0 == h1) continue;		/* not found flash width yet */
  nvInfo[slot].standard = NV_JEDEC;
  nvInfo[slot].width = 1<<sh;	/* set flash width */
  return(h1);
}
return 0;
}
nvTypeJEDEC_end() {}

#if 0
/*************************************************************
*  flash29(slot,op,sector,adr,val,nbytes) 
*/
Ulong flash29(slot,op,sector,adr,val,nbytes) 
int slot,op;
Ulong sector,adr,val,nbytes;
{
Ulong adrA,adrB,adr0,adr1;
Uchar epd;
int sh,cnt,i;
char *p;

sh = nvInfo[slot].width/2;
adrA = (sector | (0x5555<<sh));
adrB = (sector | (0x2aaa<<sh));

switch (op) {
	case NV_WRITE : 
#if 0
	  printf("flash29w: adr=%08x val=%02x adrA=%08x adrB=%08x\n",
			 adr,val,adrA,adrB);
#endif
	  outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0xa0); 
	  outb(adr,val);
	  for (i=0;i<3000;i++) wbflushtmp1 = 0; /* min 100us delay */
	  for (cnt=0;;cnt++) {
		epd = inb(adr);
		if ((epd&0x80) == (val&0x80) && epd == val) break; 
		if (epd&(1<<5)) break;
	  }
#if 0
	  if (epd != val) printf("flash29w: ## TIMEOUT\n");
#endif
	  if (epd != val) return(0);
	  return(1);
	case NV_SERASE : 
#if 0
		printf("flash29se: adr=%08x adrA=%08x adrB=%08x\n",
			adr,adrA,adrB);
#endif
		/* unlock, setup */
		outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0x80); 
		/* unlock, sectEra */
		outb(adrA,0xaa); outb(adrB,0x55); outb(adr,0x30); 
		for (i=0;i<3000;i++) wbflushtmp1 = 0; /* min 100us delay */
		val = 0xff;
		for (cnt=0;cnt<5000000;cnt++) { /* cnt=1.3M is normal */
			epd = inb(adr);
			if ((epd&0x80) == (val&0x80) && epd == val) break; 
			if (epd&(1<<5)) break;
			}
#if 0
		if (epd != val) printf("flash29se: ## TIMEOUT\n");
		printf("flash29se: epd=%02x cnt=%d\n",epd,cnt);
#endif
		if (epd != val) return(0);
		return(1);
	case NV_ERASE :
#if 0
		printf("flash29se: adr=%08x adrA=%08x adrB=%08x\n",
			adr,adrA,adrB);
#endif
		/* unlock, setup */
		outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0x80); 
		/* unlock, sectEra */
		outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0x10); 
		for (i=0;i<3000;i++) wbflushtmp1 = 0; /* min 100us delay */
		val = 0xff;
		for (cnt=0;cnt<5000000;cnt++) { /* cnt=1.3M is normal */
			epd = inb(adr);
			if ((epd&0x80) == (val&0x80) && epd == val) break; 
			if (epd&(1<<5)) break;
			}
#if 0
		if (epd != val) printf("flash29se: ## TIMEOUT\n");
		printf("flash29se: epd=%02x cnt=%d\n",epd,cnt);
#endif
		if (epd != val) return(0);
		return(1);
	case NV_BLOCKWRITE:
	  p = (char *) adr;
#if 0
	  printf("flash29w: adr=%08x val=%02x adrA=%08x adrB=%08x,nbytes=%x\n",
			 adr,val,adrA,adrB,nbytes);
#endif
	  while (nbytes-- > 0) {
		outb(adrA,0xaa); outb(adrB,0x55); outb(adrA,0xa0); 
		outb(val,*p);
		for (i=0;i<3000;i++) wbflushtmp1 = 0; /* min 100us delay */
		for (cnt=0;;cnt++) {
		  epd = inb(val);
			if ((epd&0x80) == (*p&0x80) && epd == *p) break; 
			if (epd&(1<<5)) break;
		}
#if 0
		if (epd != *p) printf("flash29w: ## TIMEOUT\n");
#endif
		if (epd != *p++) return(0);
		val++;
	  }
	  return(1);
	default:
	    printf("flash29e: NOT supported by device\n");
	}
}
void flash29end() {}
#endif

/*************************************************************
*  findSector(slot, to, start, len)
*/
void findSector(slot, to, start, len)
int slot;
Ulong to,*start,*len;
{
int i, j;
Ulong sector_pos;

if (eraseInfo[slot].num_erase_blocks)
  {
	sector_pos = nvInfo[slot].start;

	for (i = 0; i < eraseInfo[slot].num_erase_blocks; i++)
	  {
		for (j = 0; j < eraseInfo[slot].erase_block[i].num_sectors; j++)
		  {
			if ((sector_pos + eraseInfo[slot].erase_block[i].sector_size) > to)
			  {
				*start = sector_pos;
				*len = eraseInfo[slot].erase_block[i].sector_size;
				return;
			  }
			sector_pos += eraseInfo[slot].erase_block[i].sector_size;
		  }
	  }
  }

*start = nvInfo[slot].start;
*len = nvInfo[slot].size;
}

/*************************************************************
*  writeSectorsFlash(from, to, n)
*/
int writeSectorsFlash(from,to,n, preservemanufflag)
Ulong from,to,n;
int preservemanufflag;
{
char *pre,*post,*p;
flashConfigPtr *flashConfig, *flashcfg;
char *manufdata;
Ulong i,start,tmpstart,len, manufbegaddr, manufendaddr, count;
Ulong preaddr, prelen, postaddr, postlen;
int slot,tmpslot,startedwriting;
Func *callpmon;
extern char _ftext[];
extern char _etext[];

startedwriting = 0;

manufdata = pre = post = NULL;

if (n == 0) return startedwriting;		/* if nothing to write */

/* check for protected sectors */

tmpstart = to;
while (tmpstart < to+n)
 {
		 slot = findFlashSlot(tmpstart);

		 if (slot < 0 || eraseInfo[slot].num_erase_blocks == 0) 
		 {
				 printf("Address %lx is not a flash address\n", tmpstart);
				 return startedwriting;
		 }

		 findSector(slot, tmpstart, &start, &len);

		 i = (*nvInfo[slot].driver)(slot, NV_SECTORSTATUS, start, 0, 0, 0);

		 if (i & NV_FLASHPROTECTED)
		 {
				 printf("Sector at address %lx is locked\n", start);
				 return startedwriting;
		 }

		 tmpstart = start + len;
 }

/* check beginning and ending address */

/* check if we need to do malloc's and if the malloc's would work */

/* check beginning address */

slot = findFlashSlot(to);

if (preservemanufflag && testManufWrite(slot, to, n))
  {
	flashConfig = (flashConfigPtr *) 
	  (nvInfo[slot].start + ((Ulong) &FLASH_CONFIG_ADDRESS - nvInfo[0].start));
	
	manufdata = malloc(sizeof(flashConfigPtr) + flashConfig->length);

	if (manufdata == NULL)
	  {
		printf("Cannot do write without destroying manufacturing record at %lx\n",
			   flashConfig);
		return startedwriting;
	  }
  }

findSector(slot, to, &start, &len);

if (to != start)
  {
	preaddr = start;
	prelen = to - start;

	pre = malloc(prelen);

	if (pre == NULL)
	  {
		printf("Cannot do write without destroying data from %lx to %lx\n",
			   preaddr, to - 1);

		if (manufdata != NULL)
		  free(manufdata);

		return startedwriting;
	  }
  }

/* check ending address */

tmpslot = findFlashSlot(to+n-1);

if (slot != tmpslot)
  {
	if (pre != NULL)
	  {
		free(pre);
		pre = NULL;
	  }

	if (manufdata != NULL)
	  {
		free(manufdata);
		manufdata = NULL;
	  }

	slot = tmpslot;

	if (preservemanufflag && testManufWrite(slot, to, n))
	  {
		flashConfig = (flashConfigPtr *) 
		  (nvInfo[slot].start + 
		   ((Ulong) &FLASH_CONFIG_ADDRESS - nvInfo[0].start));
	
		manufdata = malloc(sizeof(flashConfigPtr) + flashConfig->length);

		if (manufdata == NULL)
		  {
			printf("Cannot do write without destroying manufacturing record at %lx\n",
				   flashConfig);
			return startedwriting;
		  }
	  }
  }

findSector(slot, to+n-1, &tmpstart, &len);

if (start != tmpstart)
  {
	/* pre and post are in different sectors - can free pre */

	if (pre != NULL)
	  {
		free(pre);
		pre = NULL;
	  }

	start = tmpstart;
  }

if ((to + n) < (start + len))
  {
	postaddr = to + n;
	postlen = start + len - to - n;
	post = malloc(postlen);

	if (post)
	  {
		free(post);
	  }
	else
	  {
		printf("Cannot do write without destroying data from %lx to %lx\n",
			   postaddr, start + len - 1);

		if (manufdata != NULL)
		  free(manufdata);

		return startedwriting;
	  }
  }

if (pre != NULL)
  {
	free(pre);
	pre = NULL;
  }

if (manufdata != NULL)
  {
	free(manufdata);
	manufdata = NULL;
  }

while (n > 0)
  {
	slot = findFlashSlot(to);

	if (slot < 0) return startedwriting;

	/* Permit the flash that follows the CFI standard since there is a
	 * way to dynamically learn the different sizes of the sectors.  
	 * Permit the flash that follows the the JEDEC standard if we have
	 * a table for that particular flash chip describing the sectors. 
	 * We can tell if we know the sizes of the sectors because our
	 * eraseInfo table will contain this information.
	 */
	if (eraseInfo[slot].num_erase_blocks == 0) return startedwriting;

	/* we are committed to writing */
	startedwriting = 1;

	findSector(slot, to, &start, &len);
	
	pre = post = 0;

	/* save any sector data before the location we wish to write */

	if (to != start)
	  {
		preaddr = start;
		prelen = to-start;
		pre = malloc(prelen);

		/*
		 * if we couldn't allocate space, hope person meant to write
		 * entire sector as would be the case for putting a new kernel
		 * into flash
		 */
		if (pre)
		  {
			p = (char *) preaddr;

			for (i = 0; i < prelen; i++)
			  pre[i] = p[i];
		  }
	  }

	/* save any manufacturing sector if we are to preserve it */

	if (preservemanufflag && testManufWrite(slot, to, start + len - to))
	  {
		flashConfig = (flashConfigPtr *) 
		  (nvInfo[slot].start + 
		   ((Ulong) &FLASH_CONFIG_ADDRESS - nvInfo[0].start));
	
		manufdata = malloc(sizeof(flashConfigPtr) + flashConfig->length);

		p = (char *) flashConfig;

		for (i = 0; i < sizeof(flashConfigPtr); i++)
		  manufdata[i] = p[i];

		p = (char *) flashConfig + flashConfig->offset;

		if ((Ulong) flashConfig < (Ulong) p)
		  manufbegaddr = (Ulong) flashConfig;
		else
		  manufbegaddr = (Ulong) p;

		for (i = sizeof(flashConfigPtr); 
			 i < sizeof(flashConfigPtr) + flashConfig->length;
			 i++)
		  manufdata[i] = *p++;

		if ((Ulong) &flashConfig[1] > (Ulong) p)
		  manufendaddr = (Ulong) &flashConfig[1];
		else
		  manufendaddr = (Ulong) p;
	  }

	/* save any sector data after the location we wish to write */

	if ((to + n) < (start + len))
	  {
		postaddr = to + n;
		postlen = start + len - to - n;
		post = malloc(postlen);

		/*
		 * if we couldn't allocate space, hope person meant to write
		 * entire sector as would be the case for putting a new kernel
		 * into flash
		 */
		if (post)
		  {
			p = (char *) postaddr;

			for (i = 0; i < postlen; i++)
			  post[i] = p[i];
		  }
	  }

	printf("Updating flash block which starts at %x\n", start);

	/* erase the sector */

	while (1)
	  {
		(* nvInfo[slot].driver)(slot, NV_SERASE, start);

		p = (char *) start;

		for (i = 0; i < len; i++)
		  {
			if (p[i] != 0xff) break;	/* sector not successfully erased */
		  }

		if (i == len) break;			/* sector successfully erased */
		printf("  Retry erasing flash block which starts at %x\n", start);
		printf("        Type control-c to abort\n");
	  }

	/* write back any sector data before what we are writing */

	if (pre)
	  {
		(* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, 
								start, pre, preaddr, prelen);
		free(pre);
		
	  }

	/* write back any manufacturing data before what we are writing */

	if (manufdata)
	  {
		p = (char *) flashConfig;

		flashcfg = (flashConfigPtr *) manufdata;

		(* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, 
								start, 
								manufdata, 
								(Ulong) p, sizeof(flashConfigPtr));

		p += flashcfg->offset;

		(* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, 
								start, 
								manufdata + sizeof(flashConfigPtr), 
								(Ulong) p, flashcfg->length);
	  }

	/* compute how much data we can write this time */

	if ((to + n) < (start + len))
	  {
		i = n;					/* can write all remaining new data */
	  }
	else
	  {
		i = start + len - to;	/* write what fits in this sector */
	  }

	if (manufdata)
	  {
		/* write any new data before manufacturing record */

		if (to < manufbegaddr)
		  {
			count = manufbegaddr - to;

			(* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, start, from, 
									to, count);
		  }

		/* count number of bytes we wrote plus skipping manuf record */

		if (manufendaddr < to + i)
		  {
			count = manufendaddr - to;
		  }
		else
		  {
			/* do not advance past end of data to be written */
			count = i;
		  }

		/* advance from, to, adjust number of bytes of data this sector */

		i -= count;
		n -= count;
		to += count;
		from += count;

		free(manufdata);
		manufdata = NULL;
	  }

	/* write the new data to the sector */

	if (i)
	  (* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, start, from, to, i);

	/* write back any sector data after what we are writing */

	if (post)
	  {
		(* nvInfo[slot].driver)(slot, NV_BLOCKWRITE, 
								start, post, postaddr, postlen);

		free(post);
	  }

	n -= i;
	to += i;
	from += i;
  }

return startedwriting;
}

int testManufWrite(int slot, Ulong to, Ulong n)
{
  unsigned char *ptr;
  unsigned long crc;
  flashConfigPtr *flashConfig;

  /* compute manufacturing address in this flash slot as follows:
   * assume the manufacturing record in every flash is at the same address.
   * Subtract the start of the boot flash chip from the address of the
   * manufacturing record address and add to the start of the flash
   * chip under question
   */
  flashConfig = (flashConfigPtr *) 
	(nvInfo[slot].start + ((Ulong) &FLASH_CONFIG_ADDRESS - nvInfo[0].start));

  if (validateManufConfig(flashConfig) == NULL)
	return 0;	/* if no manufacturing config, not overwriting it */

  /* determine if we are overwriting the manufacting record */

  ptr = (unsigned char *) flashConfig;
  ptr += flashConfig->offset;

  /* check if start addresses are greater than manufacturing record */
  if (to >= (Ulong) &flashConfig[1] && 
	  to >= (Ulong) ptr + flashConfig->length)
	return 0;			/* we are not overwriting the manufacturing record */

  /* check if end addresses are less than manufacturing record */
  if (to + n <= (Ulong) flashConfig && to + n <= (Ulong) ptr)
	return 0;			/* we are not overwriting the manufacturing record */

  /* we are overwriting the manufacturing record in flash */

  return 1;
  
}

int validateManufWrite(Ulong from, Ulong to, Ulong n)
{
  int slotfirst;
  int slotlast;

  slotfirst = findFlashSlot(to);
  slotlast = findFlashSlot(to + n - 1);

  if (slotfirst >= 0 && testManufWrite(slotfirst, to, n))
	{
	  return 1;			/* we are overwriting the manufacturing record */
	}

  if (slotlast < 0)
	return 0;			/* not flash, not overwriting manufacturing record */
  
  if (slotlast != slotfirst && testManufWrite(slotlast, to, n))
	{
	  return 1;			/* we are overwriting the manufacturing record */
	}

  return 0;				/* we are not overwriting the manufacturing record */
}

#if 0
/*************************************************************
*/
printNvInfo()
{
int slot;

for (slot=0; slot<MAXFLASH; slot++)
  { 
	printf("name=%s start=%08x width=%d size=0x%x driver=%08x\n",
		   nvInfo[slot].name, nvInfo[slot].start, nvInfo[slot].width, 
		   nvInfo[slot].size, 
		   nvInfo[slot].driver);
	
	if (nvInfo[slot].type)
	  printf("famcode=%04x name=%s size=0x%x driver=%08x\n", 
			 nvInfo[slot].type->famcode, nvInfo[slot].type->name, 
			 nvInfo[slot].type->size,nvInfo[slot].type->driver);
  }
}
#endif
