/******************************************************************/
/* Copyright (c) 2001 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

/* mspeth.c*/

/************************************************************************
 * modification history
 * --------------------
 *
 *	01/22/01		Jey	K Surier		Original.
 *	11/04/01		Jey K Surier		Added the Structres for FDs, BDs.
 *
 *
 ************************************************************************/

#include <string.h>
#include <utypes.h>
#include <mips.h>
#include <ether.h>
#include <brecis.h>
#include <mon.h>

#ifdef ETHERNET	/* skip the entire file if ethernet is not enabled */

/* #define VERBOSE */
#ifdef VERBOSE
#include <mon.h>
#endif
#ifdef CHECKS_ON
#include <assert.h>
#else
#define assert(x)
#endif

/* #define DEBUGPRINTPOLO */
/* #define DEBUGPRINTFON */

#ifdef DEBUGPRINTFON
#define PRINTF(x)		printf x
#else
#define PRINTF(x)
#endif

#define	OK		0
#define	ERROR	1

/************************************************************************
 * variables selecting MAC and PHY to manage                            *
 ************************************************************************/

static Ulong mac_ifindex;
static Ulong ethnum;

static MAC_BASE *mac_base[MSP_ETH_MAX_UNITS] =
{ (MAC_BASE *) MAC0_BASE, (MAC_BASE *) MAC1_BASE, (MAC_BASE *) MAC2_BASE };
static Ulong mac_rst[MSP_ETH_MAX_UNITS] = 
{ MAC0_RST, MAC1_RST, MAC2_RST };
static PHY_BASE *phy_base[MSP_ETH_MAX_UNITS] =
{ (PHY_BASE *) MAC0_BASE, (PHY_BASE *) MAC1_BASE, (PHY_BASE *) MAC2_BASE };
static Ulong macPhyId[MSP_ETH_MAX_UNITS];

static Uchar mac_address[MSP_ETH_MAX_UNITS][8];

static int Duplex[MSP_ETH_MAX_UNITS];
static int Speed[MSP_ETH_MAX_UNITS];
static int ENETType[MSP_ETH_MAX_UNITS];
static int ENETTxD[MSP_ETH_MAX_UNITS];


/************************************************************************
 * Receive related Buffer and Frames Descriptors pointers.              *
 ************************************************************************/
static BLFDBD	*BLFrmPtr[MSP_ETH_MAX_UNITS];
static FDBD		*FDAPtr[MSP_ETH_MAX_UNITS];
static FDBD		*FDAPtrCur[MSP_ETH_MAX_UNITS];

static Ulong	currentBlBD[MSP_ETH_MAX_UNITS];

/************************************************************************
 * Transmit related Buffer and Frames Descriptors pointers.             *
 ************************************************************************/
static FDBD		*TxFrmPtr[MSP_ETH_MAX_UNITS];
static FDBD		*TxFrmPtrCur[MSP_ETH_MAX_UNITS];
#ifdef VERBOSE
static FDBD		*TxFrmPtrComplete[MSP_ETH_MAX_UNITS];
#endif

/************************************************************************
 * Internal Function Declarations.                                      *
 ************************************************************************/

/*******************************************


 +--------+-----------+---------+---------+
  Byte 3     Byte 2     Byte 1     Byte 0
 +--------+-----------+---------+---------+
            FRAME DESCRIPTOR

+------------------------------------------+
|               FDNext					   | 0x00
+------------------------------------------+
|              FDSystem				       | 0x04
+------------------------------------------+
|               FDStat					   | 0x08	
+------------------+-----------------------+
|   FDCtl          |          FDLength     | 0x0c
+------------------+-----------------------+




            BUFFER DESCRIPTOR
+------------------------------------------+
|    BuffData (pointer to the buffer)      | 0x00
+------------------------------------------+
|              BStatus                     | 0x04
+------------------------------------------+


**********************************************/

#define ReadMacRegister(index,reg)		(mac_base[index]->reg)

#define WriteMacRegister(index,reg,value) (mac_base[index]->reg=(value))

#define ReadPhyRegister(index,reg)		(phy_base[index]->reg)

#define WritePhyRegister(index,reg,value) (phy_base[index]->reg=(value))

static Ulong waitForPhyNotBusy(Ulong ifIndex, Ulong value)
{
    int i;

    WritePhyRegister(ifIndex, MD_CA, 
		(macPhyId[ifIndex] | MD_CA_BUSY_BIT | value));

    i = 0;
    do
    {
        udelay(MD_REGISTER_UDELAY);
        if(((ReadPhyRegister(ifIndex, MD_CA)) & MD_CA_BUSY_BIT) == 0)
            return 0;

        i += MD_REGISTER_UDELAY;
    } while (i < PHYG_UDELAY_FOR_RESPONSE);

    return 1;
}

static void *xalloc(int size)
{
	void *v;

	v = (void *) malloc(size + 15);
	if (v == 0)
	{
#ifdef VERBOSE
		if (vflag)
		{
			printf("xalloc-malloc returned zero\n");
		}
#endif
		return v;
	}
	v = (void *)(((Ulong) v + 15 ) & 0xfffffff0);
	return v;
}

/************************************************************************
 *
 * Function Name:		MSPEthRMIIEnable10Mbit
 *
 * Purpose:				Enable 10 Mbit for RMII
 *
 **************************************************************************/
static void MSPEthRMIIEnable10Mbit(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_BRCTRL_REG,
			 ((ReadMacRegister(ifIndex, MAC_BRCTRL_REG)) |
			  (RMII_10MBIT)));
}

/************************************************************************
 *
 * Function Name:		MSPEthRMIIEnable100Mbit
 *
 * Purpose:				Enable 100 Mbit for RMII
 *
 **************************************************************************/
static void MSPEthRMIIEnable100Mbit(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_BRCTRL_REG,
			 ((ReadMacRegister(ifIndex, MAC_BRCTRL_REG)) &
			  (~RMII_10MBIT)));
}

/************************************************************************
 *
 * Function Name:		MSPEthRMIITxDRisingEdge
 *
 * Purpose:				Set TxD Rising Edge
 *
 **************************************************************************/
static void MSPEthRMIITxDRisingEdge(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_BRCTRL_REG,
			 ((ReadMacRegister(ifIndex, MAC_BRCTRL_REG)) |
			  (TxDClockRising)));
}

/************************************************************************
 *
 * Function Name:		MSPEthRMIITxDFallingEdge
 *
 * Purpose:				Set TxD Falling Edge
 *
 **************************************************************************/
static void MSPEthRMIITxDFallingEdge(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_BRCTRL_REG,
			 ((ReadMacRegister(ifIndex, MAC_BRCTRL_REG)) &
			  (~TxDClockRising)));
}

/*************************************************************************
*
* MSPPHYGGetSpeedAndDuplex - Gets the Working speed from the PHY 
*
* RETURNS: MSP_SUCCESS or ERROR 
*  
**************************************************************************/
static Ulong MSPPHYGGetSpeedAndDuplex(Ulong ifIndex)
{
	Ulong reg4;
	Ulong reg5;
	Ulong data;
	
	PRINTF(("MSPPHYGGetSpeedAndDuplex: for MAC# %d\n", ethnum));

	/* Read Register 0 (PHYG_CONTROL) */
	if (waitForPhyNotBusy(ifIndex, PHYG_CONTROL)) 
	{
		printf("MSPPHYGGetSpeedAndDuplex: BUSY Cannot Read PHYG_CONTROL\n");
		return;
	}

	data = ReadPhyRegister(ifIndex, MD_DATA);
	if ((data & PHYGb_CONTROL_AUTONEG) == 0)
	{
		PRINTF(("PHY is in MANUAL CONFIGURATION MODE\n"));
		if (data & PHYGb_CONTROL_SELECT_SPEED)
		{
			PRINTF(("PHY is set for 100 Mbps\n"));
			Speed[ifIndex] = MSPEth_100MBS;
		}
		else
		{
			PRINTF(("PHY is set for 10 Mbps\n"));
			Speed[ifIndex] = MSPEth_10MBS;
		}

		if (data & PHYGb_CONTROL_DUPLEXMODE)
		{
			PRINTF(("PHY is set for Full Duplex\n"));
			Duplex[ifIndex] = MAC_DPX_MODE_FULL;
		}
		else
		{
			PRINTF(("PHY is set for Half Duplex\n"));
			Duplex[ifIndex] = MAC_DPX_MODE_HALF;
		}
		return;
	}

	PRINTF(("PHY is in AUTO NEGOTIATION MODE\n"));
	/* doing auto negotiation */
	/* Read Register 4 (PHYG_AUTO_NEG_ADVERTISING) */
	if (waitForPhyNotBusy(ifIndex, PHYG_AUTO_NEG_ADVERTISING)) 
	{
		printf("MSPPHYGGetSpeedAndDuplex: PHY BUSY Cannot Read PHYG_AUTO_NEG_ADVERTISING\n");
		return ERROR ;
	}

	reg4 = ReadPhyRegister(ifIndex, MD_DATA);

	/* Read Register 5 (PHYG_AUTO_NEG_LINK_PARTNER_ABILITY) */
	if (waitForPhyNotBusy(ifIndex, PHYG_AUTO_NEG_LINK_PARTNER_ABILITY)) 
	{
		printf("MSPPHYGGetSpeedAndDuplex PHY BUSY Cannot Read PHYG_AUTO_NEG_LINK_PARTNER_ABILITY\n");
		return ERROR;
	}

	reg5 = ReadPhyRegister(ifIndex, MD_DATA);
	data = reg4 & reg5;

	if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_T4)
	{
		PRINTF(("PHY is set for 100Mbps Half Duplex\n"));
		Duplex[ifIndex] = MAC_DPX_MODE_HALF;
		Speed[ifIndex] = MSPEth_100MBS;
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_TX_FULL)
	{
		PRINTF(("PHY is set for 100Mbps Full Duplex\n"));
		Duplex[ifIndex] = MAC_DPX_MODE_FULL;
		Speed[ifIndex] = MSPEth_100MBS;
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_TX)
	{
		PRINTF(("PHY is set for 100Mbps Half Duplex\n"));
		Duplex[ifIndex] = MAC_DPX_MODE_HALF;
		Speed[ifIndex] = MSPEth_100MBS;
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_10BASE_T_FULL)
	{
		PRINTF(("PHY is set for 10Mbps Full Duplex\n"));
		Duplex[ifIndex] = MAC_DPX_MODE_FULL;
		Speed[ifIndex] = MSPEth_10MBS;
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_10BASE_T)
	{
		PRINTF(("PHY is set for 10Mbps Half Duplex\n"));
		Duplex[ifIndex] = MAC_DPX_MODE_HALF;
		Speed[ifIndex] = MSPEth_10MBS;
	}
}


static Ulong MSPPHYGWaitForLink(Ulong ifIndex) 
{
	int i;
	Ulong	data;
	Ulong	doingAuto;

	/* Read Register 0 (PHYG_CONTROL) */
	if (waitForPhyNotBusy(ifIndex, PHYG_CONTROL)) 
	{
		printf("MSPPHYGWaitForLink: PHY BUSY PHYG_STATUS_REG_ONE\n");
		return ERROR;
	}

	doingAuto = ReadPhyRegister(ifIndex, MD_DATA) & PHYGb_CONTROL_AUTONEG;

	/* wait for link or auto negotiation to complete */
	i = 0;
	do
	{
		mdelay(1000);  /* wait one second */

		/* Read Register 1 (PHYG_STATUS_REG_ONE) */
		if (waitForPhyNotBusy(ifIndex, PHYG_STATUS_REG_ONE)) 
		{
			printf("MSPPHYGWaitForLink: PHY BUSY PHYG_STATUS_REG_ONE\n");
			return ERROR;
		}

		data = ReadPhyRegister(ifIndex, MD_DATA);
		if(doingAuto)
		{
			if(data & PHYGb_STATUS_REG_ONE_AUTO_NEG_DONE) 
			{
				PRINTF(("AUTO NEGOTIATION COMPLETE\n"));
				return MSPPHYGGetSpeedAndDuplex(ifIndex);
			}
		}
		else if(data & PHYGb_STATUS_REG_ONE_LINK_STATUS)
		{
			PRINTF(("LINK DETECTED\n"));
			return MSPPHYGGetSpeedAndDuplex(ifIndex);
		}
       	i++;
	} while (i < MAX_TIME_TO_WAIT_FOR_LINK);

	printf("**** ERROR *** NO LINK DETECTED\n");
	return ERROR;
}

/*************************************************************************
*
* MSPPHYGSetup - initialize and configure the PHY device.
* This routine initialize and configure the PHY device 
*
**************************************************************************/
static void determinePHYAddress(Ulong ifIndex)
{
	char *phyaddr = getMonEnv(PHYADDR);
	Ulong i;
	Ulong phyIndex;
	Ulong	reg1;	

	if (phyaddr != NULL)
	{
		if (sscanf(phyaddr, "%d:%d", &phyIndex, &i) == 2 &&
			i < MD_MAX_PHY || phyIndex < MSP_ETH_MAX_UNITS)
		{
			macPhyId[ifIndex] = i << MD_CA_PhyShift;
			phy_base[ifIndex] = mac_base[phyIndex];
		}
		else
		{
			printf("MSPPHYGSetup(): could not parse variable %s, value %s\n",
				   PHYADDR, phyaddr);
			macPhyId[ifIndex] = 0 << MD_CA_PhyShift;
		}
		return;
	}
/*
    if (identify_family() != FAMILY_POLO)
	    phyIndex = 0;
    else
	    phyIndex = ifIndex;
*/
    switch (identify_family())
    {
    	case FAMILY_POLO:
        case FAMILY_ZEUS:
		phyIndex = ifIndex;
	        break;

        default:
                phyIndex = 0;
                break;
    }


	/*
	 * determine PHY address
	 */
	for (i = 0; i < MD_MAX_PHY; i++)
	{
		macPhyId[ifIndex] = i << MD_CA_PhyShift;

		if (waitForPhyNotBusy(ifIndex, PHYG_STATUS_REG_ONE))
			continue;			/* No PHY responding */

		reg1 = ReadPhyRegister(ifIndex, MD_DATA);

		if (reg1 != 0 && reg1 != 0xffff && (reg1 & PHYGb_MEDIAMASK) != 0)
		{
			/* phyIndex is number of PHYs to skip before matching
			 * to this MAC interface in case PHYs were hooked to
			 * the same mdc/mdio pins
			 */
			if (phyIndex-- == 0)
				break;			/* PHY present */
		}
	}

	if (i == MD_MAX_PHY) 
	{
		printf("MSPPHYGSetup(): no PHY address found, using default\n");
		macPhyId[ifIndex] = 0 << MD_CA_PhyShift;
	}
}

static void MSPPHYGSetup (Ulong ifIndex, int ethMode) 
{
	Ulong i;
	Ulong	data;
	int	doingAutoNeg = 0;

	PRINTF(("Entering MSPPHYGSetup() for MAC#: %d\n", ethnum));

	/* 
	 * Wait for Phy to come out of reset.  
	 * Should already be there when we are here, but wait in case
	 */
	i = 0;
	while(((data = ReadPhyRegister(ifIndex, MD_CA))) & MD_CA_BUSY_BIT)
	{
		udelay(MD_REGISTER_UDELAY);
		i += MD_REGISTER_UDELAY;
		if(i >= PHYG_MAX_UDELAY_FOR_EXIT_RESET)
		{
			printf("1. PHY BUSY After Start Up\n");
			return;
		}
	}

	determinePHYAddress(ifIndex);

	printf("Using PHY address %x with base %x for MAC#: %d\n", 
			macPhyId[ifIndex], phy_base[ifIndex], ethnum);

   	/*
	 * To write to the PHY, MD_Data register at 0x58 must be setup with
	 * the data that is to be written to the PHY, and then configure
	 * the MD_CA register at 0x5C with the PHY Addr and write/read 
	 * command.
	 */
	udelay(PHYG_MAX_UDELAY_FOR_EXIT_RESET);
	WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_RESET);
   	if (waitForPhyNotBusy(ifIndex, (MD_CA_Wr | PHYG_CONTROL)) )
	{
		printf("MSPPHYGSetup():  PHY BUSY After Sending Reset CMD\n");
		return;
	}

	/*
	 * PHY has accepted the request for Reset
	 * Read the Reset bit to see if the reset is completed.
	 * Need to Wait at least 300 ms
	 */
	udelay(PHYG_MAX_UDELAY_FOR_EXIT_RESET);
	i = 0;
	do
	{
		if (waitForPhyNotBusy(ifIndex, PHYG_CONTROL) )
		{
			printf("MSPPHYGSetup(): PHY Busy after Read for Reset \n");
			return;
		}

		data = ReadPhyRegister(ifIndex, MD_DATA);
		if(data & PHYGb_CONTROL_RESET)
		{
			PRINTF(("1. PHY Reset not cleared - PHY = %08X\n", data));
			udelay(PHYG_MAX_UDELAY_FOR_EXIT_RESET);
			i++;
		}
		else
		{
			PRINTF(("1. PHY Out of Reset - PHY = %08X\n", data));
			break;
		}
	} while (i < 10);

	/*
	 * Loopback=disabled
	 * Speed = 100/10
	 * Auto-Negotiation = Disabled
	 * Power Down = 0 normal
	 * Isolate = 0 normal
	 * Restart Auto-Negotiation = 0
	 * Duplex = 1 Full / 0 Half
	 * Collision Test = 0 Disabled
	 *
	 * For Full Dup 10Mbps:  0x0100
	 * For Half Dup 10Mbps:  0x0000
	 * For Full Dup 100Mbps: 0x2100
	 * For Half Dup 100Mbps: 0x2000
	 *
	 * Auto Negotiation Enabled: 0x1000
	 */
	switch(ethMode)
	{
		case ETH_AUTO:
			PRINTF(("PHY is set for Auto-Negotiation \n"));
			WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_AUTONEG);	
			doingAutoNeg = 1;
			break;

		case ETH_10Mbps_HALF:
			PRINTF(("PHY is set for 10 Mbps Half Duplex \n"));
			WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_SPEED_10_HALF);
			Duplex[ifIndex] = MAC_DPX_MODE_HALF;
			Speed[ifIndex] = MSPEth_10MBS;
			break;

		case ETH_10Mbps_FULL:
			PRINTF(("PHY is set for 10 Mbps Full Duplex \n"));
			WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_SPEED_10_FULL);
			Duplex[ifIndex] = MAC_DPX_MODE_FULL;
			Speed[ifIndex] = MSPEth_10MBS;
			break;
		
		case ETH_100Mbps_HALF:
			PRINTF(("PHY is set for 100 Mbps Half Duplex \n"));
			WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_SPEED_100_HALF);
			Duplex[ifIndex] = MAC_DPX_MODE_HALF;
			Speed[ifIndex] = MSPEth_100MBS;
			break;
		
		case ETH_100Mbps_FULL:
			PRINTF(("PHY is set for 100 Mbps Full Duplex \n"));
			WritePhyRegister(ifIndex, MD_DATA, PHYGb_CONTROL_SPEED_100_FULL);
			Duplex[ifIndex] = MAC_DPX_MODE_FULL;
			Speed[ifIndex] = MSPEth_100MBS;
			break;	
	}

	/* send the data to the PHY */
   	if (waitForPhyNotBusy(ifIndex, (MD_CA_Wr | PHYG_CONTROL)) )
	{
		printf("MSPPHYGSetup():  PHY BUSY Setting Speed/Mode\n");
		return;
	}
	
	MSPPHYGWaitForLink(ifIndex) ;
}


/**********************************************************************
 * Function Name:	EnableTxRx
 *
 *   Purpose:       Starts transmission and reception.
 *   
 *   Parameters:	MAC Interface Index.
 *
 *   Return Value:	None    
 *
 ***********************************************************************/
static void EnableTxRx(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, TX_CTL, ((ReadMacRegister(ifIndex, TX_CTL)) | 0x00000001)); /* Enable Transmission */
	WriteMacRegister(ifIndex, RX_CTL, ((ReadMacRegister(ifIndex, RX_CTL)) | 0x00000001)); /* Enable Reception    */
}

static int getSpeedDuplex()
{
	int isswitch = 0;
	char *speedduplex = getMonEnv(SPEEDDUPLEX);

	if (speedduplex == NULL)
		return ETH_AUTO;

	if (strlen(speedduplex) > 0 && speedduplex[strlen(speedduplex) - 1] == 's')
		isswitch = 1;

	if (strncmp(speedduplex, STRING10HALFANY, strlen(STRING10HALFANY)) == 0)
	{
		return (isswitch == 1) ? -ETH_10Mbps_HALF : ETH_10Mbps_HALF;
	}
	if (strncmp(speedduplex, STRING10FULLANY, strlen(STRING10FULLANY)) == 0)
	{
		return (isswitch == 1) ? -ETH_10Mbps_FULL : ETH_10Mbps_FULL;
	}
	if (strncmp(speedduplex, STRING100HALFANY, strlen(STRING100HALFANY)) == 0)
	{
		return (isswitch == 1) ? -ETH_100Mbps_HALF : ETH_100Mbps_HALF;
	}
	if (strncmp(speedduplex, STRING100FULLANY, strlen(STRING100FULLANY)) == 0)
	{
		return (isswitch == 1) ? -ETH_100Mbps_FULL : ETH_100Mbps_FULL;
	}

	return ETH_AUTO;
}

/*************************************************************************
*
* PhyInit - initialize and configure the PHY device.
*
* This routine initialize and configure the PHY device 
*
* RETURNS: OK, or ERROR
*  
**************************************************************************/
static int PhyInit(Ulong ifIndex)
{
	int status=OK, speed=getSpeedDuplex();
	int i;
	
	if (ifIndex >= MSP_ETH_MAX_UNITS)
		return ERROR;

	if (speed < 0)
	{
		speed = -speed;

		switch(speed)
		{
		case ETH_10Mbps_HALF:
		case ETH_10Mbps_FULL:
			Speed[ifIndex] = MSPEth_10MBS;
			break;
		case ETH_100Mbps_HALF:
		case ETH_100Mbps_FULL:
			Speed[ifIndex] = MSPEth_100MBS;
			break;
		}

		switch(speed)
		{
		case ETH_10Mbps_HALF:
		case ETH_100Mbps_HALF:
			Duplex[ifIndex] = MAC_DPX_MODE_HALF;
			break;
		case ETH_10Mbps_FULL:
		case ETH_100Mbps_FULL:
			Duplex[ifIndex] = MAC_DPX_MODE_FULL;
			break;
		}
		printf("Switch is set for %d Mbps %s Duplex\n",
		       Speed[ifIndex] == MSPEth_10MBS ? 10 : 
		       (Speed[ifIndex] == MSPEth_100MBS ? 100 : 0),
		       Duplex[ifIndex] == MAC_DPX_MODE_HALF ?
		       "Half" : 
		       (Duplex[ifIndex] == MAC_DPX_MODE_FULL ? 
			"Full" : "UNKNOWN"));
	}
	else
	{
		MSPPHYGSetup(ifIndex, speed);
	}

	if (Duplex[ifIndex] == MAC_DPX_MODE_HALF)
	{
		WriteMacRegister(ifIndex, MAC_CTL, 
						 ((ReadMacRegister(ifIndex, MAC_CTL)) & 
						  (~MAC_CTL_FullDuplex)) );
	}
	else
	{
		WriteMacRegister(ifIndex, MAC_CTL, 
						 ((ReadMacRegister(ifIndex, MAC_CTL)) |
						  MAC_CTL_FullDuplex) );
	}

#ifdef DEBUGPRINTPOLO
	if (vflag)
	{
		printf("ENETTxD[%d] %c, ENETType[%d] %c\n",
			   ifIndex, ENETTxD[ifIndex], ifIndex, ENETType[ifIndex]);
		printf("eth %d MAC_BRCTRL_REG value before %lx\n",
			   ethnum, ReadMacRegister(ifIndex, MAC_BRCTRL_REG));
	}
#endif	

	switch (ENETTxD[ifIndex])
	{
	case ENETTXD_RISING:
		/* Set TxD rising edge - I was told this was best default */

		MSPEthRMIITxDRisingEdge(ifIndex);
		break;

	case ENETTXD_FALLING:
		/* Set TxD falling edge */

		MSPEthRMIITxDFallingEdge(ifIndex);
		break;

	default:
		/* do not touch the TxD bit */
		break;
	}

	if (ENETType[ifIndex] == ENET_RMII)
	{
		/* Set speed of RMII based on PHY setup/negotiation */

		if (Speed[ifIndex] == MSPEth_10MBS)
			MSPEthRMIIEnable10Mbit(ifIndex);
		else
			MSPEthRMIIEnable100Mbit(ifIndex);
	}

#ifdef DEBUGPRINTPOLO
	if (vflag)
	{
		printf("eth %d MAC_BRCTRL_REG value after %lx\n",
			   ethnum, ReadMacRegister(ifIndex, MAC_BRCTRL_REG));
	}
#endif	

	EnableTxRx(ifIndex);

	return (status);
} 


/*********************************************************************
 *	Function Name:  SetupTxBDs
 *
 *  Purpose:		To allocate memory for transmit frame and buffer descriptors
 *                  and set up initial values for these.
 *
 *  Inputs:			interface number: 0 or 1.
 *
 *   Return Value:
 *					OK
 *					ERROR
 *
 *********************************************************************/
static Ulong SetupTxBDs(short ifIndex)
{
	int i, j;
	int firsttime = 0;
	FDBD *tempFdBd;
  
	if (TxFrmPtr[ifIndex] == 0)
	{
		/* not yet set up - set up only first time */

		firsttime = 1;

		/*
		 * Set up the TXQ. 
		 * But to make sure 16 byte aligned for each FD.
		 * Allocate memory for transmit frame and  buffer descriptors.
		 */
		if(!(TxFrmPtr[ifIndex] = (FDBD *) xalloc(TX_FD_COUNT * sizeof(FDBD)))) 
		{ 
			return(ERROR);
		}
	}

	tempFdBd = TxFrmPtr[ifIndex];

	for(i = 0; i < TX_FD_COUNT; i++) 
	{  
		tempFdBd->FD.FDSystem		= 0x11111111;		/* A pattern for debugging ! */
		tempFdBd->FD.FDStat			= 0x00000000;
		tempFdBd->FD.FDCtl			= SW_OWNS_FD|BD_PER_TX_FRAME;
		tempFdBd->FD.FDLength		= 0x0000;
		
		for(j = 0; j < MAX_BD_PER_FRAME; j++)
		{
			if (firsttime)
				tempFdBd->BD[j].BuffData = (Uchar *)xalloc(MAX_BUFF_SIZE+15);
			tempFdBd->BD[j].BDCtl		= 0x00;
			tempFdBd->BD[j].BDStat		= 0x33;
			tempFdBd->BD[j].BuffLength	= 0x4444;
		}

		tempFdBd->FD.FDNext			= (FD *)((Ulong)tempFdBd+sizeof(FDBD));
		tempFdBd++;
	}	
		
	/* Set the FDNext in the last FD, to point to the first.*/
	tempFdBd--;
	tempFdBd->FD.FDNext	= (FD *)TxFrmPtr[ifIndex]; 


	/* Set the TxFrmPtrCur to the base of the TXQ. */
	TxFrmPtrCur[ifIndex]		= TxFrmPtr[ifIndex];

#ifdef VERBOSE
	if (vflag)
	{
		printf("SetupTxBDs-for %d, TxFrmPtr %x, TxFrmPtrCur %x\n",
			   ethnum, &TxFrmPtr[ifIndex], &TxFrmPtrCur[ifIndex]);
	}
#endif

	return(OK);
}

/****************************************************************************
 *	Function Name:	SetupRxBDs                              
 *                                                                          
 *  Purpose:        Prepare memory for receive buffers and frame descriptors
 *                  and buffer descriptors.                                
 *                                                                         
 *  Inputs:			
 *			ifIndex		 - Interface Index, 0 for MAC0 and 1 for MAC1
 *
 *	Return Value:
 *			OK
 *			ERROR
 *
 *****************************************************************************/
static Ulong SetupRxBDs(Ulong ifIndex)
{
	int i;
	int firsttime = 0;
	Ulong fdaSize;	
	FD		*tempFd;

	if (BLFrmPtr[ifIndex] == 0)
	{
		/* buffers not set up yet - set up buffers first time */

		firsttime = 1;

		/*
		 * Buffer List: (which conisits of single FD and multiple BDs)
		 * There will be RX_BD_COUNT of BDs, 
		 * and each BD points to one RecBuffSpace
		 * of size of MAX_BUFF_SIZE. (add 16 for memeory boundary adjustment).
		 *
		 */
		if(!(BLFrmPtr[ifIndex] = (BLFDBD *) xalloc(sizeof(BLFDBD)))) 
		{ 
			return(ERROR);
		}
	
		/*
		 * Allocate 16 byte memory for (FDA)descriptors for received buffers.
		 * For each frame received, one FD+BD will be written in the FDA if the
		 * rx buffer that is pointed by the BD could fit the entire packet.
		 */
		fdaSize = RX_FDA_COUNT * FDA_1BLK_SIZE; 
		if(!(FDAPtr[ifIndex] = (FDBD *) xalloc(fdaSize))) 
		{ 
			return(ERROR);
		}
	}

	/*
	 * Initialize the FDA Area. Give the ownership of each 16 byte block
	 * to the controller. (The patterns of 0xBBB etc is for debugging only!)
	 */
	tempFd = &(FDAPtr[ifIndex]->FD);
	for(i = 0; i < RX_FDA_COUNT; i++)
	{
		tempFd->FDSystem = 0xBBBBBBBB;
		tempFd->FDStat   = 0xCCCCCCCC;
		tempFd->FDCtl    = MAC_OWNS_FD;
		tempFd->FDLength = 0xEEEE;
		tempFd++;
	}

	/*
	 * Initialize the BL Area. We have only one FD with BDs here, 
	 * so point to self as FDNext
	 */
	BLFrmPtr[ifIndex]->FD.FDNext	 = (FD *)BLFrmPtr[ifIndex];
	BLFrmPtr[ifIndex]->FD.FDSystem	 = 0xAAAAAAAA;		/* A pattern for debugging ! */
	BLFrmPtr[ifIndex]->FD.FDStat	 = 0x00000000;
	/*
	 * Set the number of BDs in the FDLength field of BLFrmPtr according to the
	 * spec.
	 *
     * Controller owns the FD, BDCount left blank as the controller
     * uses FDLength to show the number of BDs available in the Buffer
	 */
	BLFrmPtr[ifIndex]->FD.FDCtl	    = MAC_OWNS_FD;
	BLFrmPtr[ifIndex]->FD.FDLength  = RX_BD_COUNT;	/* Total Number of BDs in the BL area */
	
	/*
	 * Each BD points to a large enough buffer so that for each incoming packet
	 * only one BD per packet will be used.
	 *
	 */
	for(i = 0; i < RX_BD_COUNT; i++) 
	{
		/* Get a buf and attach to the BD.*/

		if (firsttime)
			BLFrmPtr[ifIndex]->BD[i].BuffData = (Uchar *)xalloc(MAX_BUFF_SIZE);

		/* Set all of the BDs to Controller ownership.*/
		BLFrmPtr[ifIndex]->BD[i].BDCtl		= MAC_OWNS_BD;
		BLFrmPtr[ifIndex]->BD[i].BuffLength	= MAX_PKT_SIZE;
	}


	FDAPtrCur[ifIndex] = FDAPtr[ifIndex];
	currentBlBD[ifIndex] = 0;

#ifdef VERBOSE
	if (vflag)
	{
		printf("SetupRxBDs-for %d, BLFrmPtr %x, FDAPtr %x\n",
			   ethnum, &BLFrmPtr[ifIndex], &FDAPtr[ifIndex]);
		printf("          -for %d, FDAPtrCur %x, currentBlBD %x\n",
			   ethnum, &FDAPtrCur[ifIndex], &currentBlBD[ifIndex]);
	}
#endif

	return(OK);
}


/*************************************************************************
 * Function Name:	RxRelease
 *
 * Purpose:         Release packet ownership to chip
 *
 *************************************************************************/
static void RxRelease(Ulong ifIndex, FDBD *rr)
{
	int	bdCount;
	FDBD *tempfda;
	BLFDBD *tempblf;

	/* 
	 * Return ownership of frame and buffer descriptors in both FDA and BL.
	 * In case of Jabbers, there could be several BDs, so make sure that
	 * all the BDs that MAC used are cleared here.
	 */

	if (rr != FDAPtrCur[ifIndex])
	{
#ifdef VERBOSE
		if (vflag)
		{
			printf("RxRelease-Expected %x, was %x\n", FDAPtrCur[ifIndex], rr);
		}
#endif
		return;
	}

	tempblf = BLFrmPtr[ifIndex];

	for(bdCount=0; 
		bdCount < (rr->FD.FDCtl & FDCTL_BD_COUNT); 
		bdCount++)
	{
		tempblf->BD[currentBlBD[ifIndex]].BDCtl			= MAC_OWNS_BD;
		tempblf->BD[currentBlBD[ifIndex]].BuffLength	= MAX_PKT_SIZE;
		rr->BD[bdCount].BDCtl							= MAC_OWNS_BD;
		
		currentBlBD[ifIndex] = (currentBlBD[ifIndex] + 1) % RX_BD_COUNT;
	}
	
	tempfda = (FDBD *)rr->FD.FDNext;
	rr->FD.FDCtl = MAC_OWNS_FD;
	FDAPtrCur[ifIndex] = tempfda;
	Sync();
}

/**********************************************************************
 * Function Name:	DisableTxRx
 *
 *   Purpose:       Stops transmission and reception.
 *   
 *   Parameters:	MAC Interface Index.
 *
 *   Return Value:	None    
 *
 ***********************************************************************/
static void DisableTxRx(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, TX_CTL, ((ReadMacRegister(ifIndex, TX_CTL)) & 0xFFFFFFF2)); /* Cancel Transmission */
	WriteMacRegister(ifIndex, RX_CTL, ((ReadMacRegister(ifIndex, RX_CTL)) & 0xFFFFFFF2)); /* Cancel Reception    */
}

static void ResetInternal(int bit)
{
	*RST_SET_REG = bit;
	while (!(*SYS_RST_REG & bit))
		;
	*RST_CLR_REG = bit;
	while (*SYS_RST_REG & bit)
		;
}

/*************************************************************************
 *
 * Function Name:		Reset
 *
 * Purpose:				To reset the MAC.
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static short Reset(Ushort ifIndex)
{
	ResetInternal(mac_rst[ifIndex]);
	
	return (OK);
}


static void SetupRegistersTx(unsigned int ifIndex)
{
#if 1
		WriteMacRegister(ifIndex, DMA_CTL,  0x0008C040);
#else
	if (cpuType() == 5000)
	{	
		WriteMacRegister(ifIndex, DMA_CTL,  0x0008C040);
	}
	else
	{
		WriteMacRegister(ifIndex, DMA_CTL,  0x0008C080);
	}
#endif			
	WriteMacRegister(ifIndex, MAC_CTL,  0x00000000); 	/* auto Duplex 0x000000 */
	WriteMacRegister(ifIndex, TXTHRSH,  0x00000000); 
	WriteMacRegister(ifIndex, TXPOLLCTR,0x00000fff);
	WriteMacRegister(ifIndex, TX_CTL,   0x00000000);
}

static void SetupRegistersRx(unsigned int ifIndex)
{
	WriteMacRegister(ifIndex, RXFRAGSIZE, 0x00); 
	WriteMacRegister(ifIndex, FDA_BAS, (Ulong) FDAPtr[ifIndex]); 

	WriteMacRegister(ifIndex, FDA_LIM, ((RX_FDA_COUNT-1)<<4));  

	WriteMacRegister(ifIndex, ARC_CTL, 0x08);			/* setting to promiscuous mode*/
	WriteMacRegister(ifIndex, ARC_ADR, 0x4);			/* setting to promiscuous mode*/
	WriteMacRegister(ifIndex, ARC_DATA, 0x02040405);	/* setting to promiscuous mode*/
	WriteMacRegister(ifIndex, ARC_ENA, 0x00); 

	WriteMacRegister(ifIndex, RX_CTL, 0x00000000);		/*originally 0x205 enabling VLAN, or 0x4A11, 0x4211*/

	WriteMacRegister(ifIndex, BLFRMPTR, (unsigned int)BLFrmPtr[ifIndex]); 
}


static void SetUpTxFrmPtrRegister(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, TXFRMPTR, ((Ulong)TxFrmPtr[ifIndex] & 0x1fffffff));
}

/*************************************************************************
 *
 *	Function Name:		TxRxInit
 *
 *   Purpose:           Set up the Ethernet to receive and transmit data.
 *						Same as BootMspEthInit(), but includes the ifIndex as
 *						parameter.
 *
 *
 *   Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong TxRxInit(Ulong ifIndex)
{
	
	Reset(ifIndex);
    
	if(SetupRxBDs(ifIndex) == ERROR)
		return(ERROR);
    
	if(SetupTxBDs(ifIndex) == ERROR)
		return(ERROR);

	SetupRegistersTx(ifIndex);
	
	SetupRegistersRx(ifIndex);

	DisableTxRx(ifIndex);

	SetUpTxFrmPtrRegister(ifIndex);

    return (PhyInit(ifIndex));
}


static void SetUpARC(Ulong ifIndex, Uchar *addr)
{
	Ulong data;
	Ulong *datap = (Ulong *) addr;

	WriteMacRegister(ifIndex, ARC_ADR, 0x00);
	data = *datap++;
	WriteMacRegister(ifIndex, ARC_DATA, data);

	WriteMacRegister(ifIndex, ARC_ADR, 0x04);			
	data = *datap & 0xffff0000;
	WriteMacRegister(ifIndex, ARC_DATA, data); 

	WriteMacRegister(ifIndex, ARC_ENA, 0x00000001); /* Enable Address at 0x00 */
	WriteMacRegister(ifIndex, ARC_CTL, 0x00000014); /* Enable ARC Compare+BCAST */
}


/*************************************************************************
 *
 * Function Name:		EnableVLAN
 *
 * Purpose:				To enable the VLAN functionalities, so that MAC 
 *						could handle packets as long as 1522 bytes(instead of 1518 bytes).
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong EnableVLAN(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, RX_CTL, ((ReadMacRegister(ifIndex, RX_CTL)) | ENA_RX_VLAN) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		DisableVLAN
 *
 * Purpose:				To disble the VLAN functionalities of the MAC.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong DisableVLAN(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, RX_CTL, ((ReadMacRegister(ifIndex, RX_CTL)) & DISA_RX_VLAN) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		SetSpeedAuto
 *
 * Purpose:				To Set the MAC in Automatic mode.
 *						NOTE: The actual speed is determined by the PHY
 *						that is connected to this MAC.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong SetSpeedAuto(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_CTL, ((ReadMacRegister(ifIndex, MAC_CTL)) & SPEED_AUTO) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		SetSpeed10
 *
 * Purpose:				To Set the MAC in 10 MBPS mode.
 *						NOTE: The speed of the PHY should match this speed.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong SetSpeed10(Ulong ifIndex)
{
	WriteMacRegister(ifIndex, MAC_CTL, ((ReadMacRegister(ifIndex, MAC_CTL)) & SPEED_AUTO) | 0xFFFFFFBF );

	return(OK);
}

#ifdef VERBOSE
static void CheckTxComplete(Ulong ifIndex)
{
	  FDBD	*rr;

	  rr = TxFrmPtrComplete[ifIndex];

	  if (rr == TxFrmPtrCur[ifIndex])
		 return;

	  HitInvDCache(rr,sizeof(*rr));

	  if (!( rr->FD.FDCtl & MAC_OWNS_FD))
	  {
			printf("ETHER_TxComplete-FD %x, status %x\n",
					rr, rr->FD.FDStat);					  	
			TxFrmPtrComplete[ifIndex] = (FDBD *) rr->FD.FDNext;
	  }
}
#endif

static int getFeature(int feature, Ulong interfaceNum)
{
	  char *features = getMonEnv(FEATURES);
	  while (*features)
	  {
			if (*features++ == feature && interfaceNum-- == 0)
			{
				break;			/* value for ethernet interface found */
			}
			features++;
	  }

	  if (*features == '\0')
			return FEATURE_NOEXIST;	/* feature does not exist */

	  return *features;
}

static Ulong compute_mac_ifindex(Ulong ethnum)
{
	int	logicalethnum = 0;
	int ifindex;
	int noexist = MSP_ETH_MAX_UNITS;

	for (ifindex = 0; ifindex < MSP_ETH_MAX_UNITS; ifindex++)
	{
		if (getFeature(ENET_KEY, ifindex) != FEATURE_NOEXIST)
		{
			if (logicalethnum++ == ethnum)
				return ifindex;
		}
		else
			noexist = ifindex;
	}

	return noexist;
}

/*************************************************************
*  void *mspeth_driver(int op,void *vp1,void *vp2)
*	Main entry point for this driver. Everything else in this
*	module should be static.
*/
void *mspeth_driver(int op,void *vp1,void *vp2)
{
	FDBD *rr = vp1;
	int *pLen = vp2;
	Uchar *macAddr = vp1;
	FDBD *cr;
	int delay;	
	int feature;

	switch (op) {
	case ETHER_INIT :
	  /* int ether_driver(ETHER_INIT,Uchar *macAddr,ifindex) */
		ethnum = (Ulong) vp2;

		mac_ifindex = compute_mac_ifindex(ethnum);

	  feature = getFeature(ENET_KEY, mac_ifindex);

	  if (feature == FEATURE_NOEXIST || 
		  mac_ifindex >= MSP_ETH_MAX_UNITS)
			return 0;	/* feature does not exist */

	  ENETType[mac_ifindex] = feature;
	  ENETTxD[mac_ifindex] = getFeature(ENETTXD_KEY, mac_ifindex);

	  if (identify_family() == FAMILY_POLO)
	  {
		  /* if Polo family, all PHYs use the same mdc/mdio register */

		  phy_base[mac_ifindex] = mac_base[0];
		  if (mac_ifindex == 2)
		  {
			  /* if Polo, must set GPIO bit to enable MAC 2 */
			  *GPIO_CFG7_REG |= POLO_MAC2_ENABLE;
		  }
	  }

	  if (TxRxInit(mac_ifindex) != OK)
		{
#ifdef VERBOSE
		  if (vflag)
			{
			  printf("ETHER_INIT-failed\n");
			}
#endif
		  return(0);
		}
	  memcpy(mac_address[mac_ifindex], macAddr, 6);
	  SetUpARC(mac_ifindex, macAddr);
#ifdef VERBOSE
	  if (vflag)
		{
		  printf("ETHER_INIT-macaddr=%02x:%02x:%02x:%02x:%02x:%02x\n",
				 macAddr[0], macAddr[1], macAddr[2],
				 macAddr[3], macAddr[4], macAddr[5]);
		}

		TxFrmPtrComplete[mac_ifindex] = TxFrmPtr[mac_ifindex];
#endif
	  return((void *)1);
	  
	case ETHER_GETTBA :
	  /* char *ether_driver(ETHER_GETTBA,int *len,void) */

	  rr = TxFrmPtrCur[mac_ifindex];
	  HitInvDCache(rr,sizeof(*rr));
#ifdef VERBOSE
	  if (vflag)
		{
		  if (rr->FD.FDCtl & MAC_OWNS_FD)
			{
#if 0
			  printPkt(rr->BD[0].BuffData, rr->FD.FDLength);
#endif
			  printf("ETHER_GETTBA-MAC owns FD %x, length %d\n", 
					 rr, *pLen);
			}
		  else
			printf("ETHER_GETTBA-We own FD %x, length %d\n", rr, *pLen);
		}
#endif
	  rr->BD[0].BuffLength = *pLen;
	  return rr->BD[0].BuffData;
	  
	case ETHER_TBARDY :
	  /* int ether_driver(ETHER_TBRDY,void,void) */
	  rr = TxFrmPtrCur[mac_ifindex];
#ifdef VERBOSE
	  if (vflag)
		{
#if 0
		  printPkt(rr->BD[0].BuffData, rr->BD[0].BuffLength);
#endif
		  if (rr->FD.FDCtl & MAC_OWNS_FD)
			printf("ETHER_TBARDY-MAC owns FD %x, length %d\n", 
				   rr, rr->BD[0].BuffLength);
		  else
			printf("ETHER_TBARDY-we own FD %x, length %d\n", 
				   rr, rr->BD[0].BuffLength);
		}
#endif
	  rr->BD[0].BDCtl = MAC_OWNS_BD;
	  rr->FD.FDCtl = MAC_OWNS_FD | BD_PER_TX_FRAME;
	  TxFrmPtrCur[mac_ifindex] = (FDBD *) rr->FD.FDNext;
	  Sync();
	  WriteMacRegister(mac_ifindex, DMA_CTL, 
					((ReadMacRegister(mac_ifindex, DMA_CTL))| 0x00010000) );
	  return((void *)1);
	  
	case ETHER_GETRXREC :
	  /* RXrec *ether_driver(ETHER_GETRXREC,void,void) */
	  rr = FDAPtrCur[mac_ifindex];
	  
	  if (!(rr->FD.FDStat & RCVD_OK_PKT))
		{
#ifdef VERBOSE
		  if (vflag)
			{
			  printf("ETHER_GETRXREC-packet not okay %x\n", rr);
			}
#endif
		  RxRelease(mac_ifindex, rr);
		  return 0;
		}
#ifdef VERBOSE
	  if (vflag)
		{
		  printf("ETHER_GETRXREC-packet okay %x\n", rr);
		}
#endif
	  HitInvDCache(rr->BD[0].BuffData, rr->FD.FDLength);
	  return (rr);
	  
	case ETHER_GETRBA :
	  /* char *ether_driver(ETHER_GETRBA,RXrec *q,int *len) */
	  *pLen = rr->FD.FDLength-4;
#ifdef VERBOSE
	  if (vflag)
		{
#if 0
		  printPkt(rr->BD[0].BuffData, rr->FD.FDLength);
#endif
		  printf("ETHER_GETRBA-got packet %x, length %d\n", 
				 rr, rr->FD.FDLength);
		}
#endif		
	  return(rr->BD[0].BuffData);
	  
	case ETHER_RXDONE :
	  /* int ether_driver(ETHER_RXDONE,RXrec *q,void) */
#ifdef VERBOSE
	  if (vflag)
		{
		  printf("ETHER_RXDONE-packet %x\n", rr);
		}
#endif
	  RxRelease(mac_ifindex, rr);
	  return((void *)1);
	  
	case ETHER_RXRDY :
	  /* int ether_driver(ETHER_RXRDY,void,void) */
	  
#ifdef VERBOSE
		if (vflag)
		{
			CheckTxComplete(mac_ifindex);
		}
#endif
	  rr = FDAPtrCur[mac_ifindex];
	  HitInvDCache(rr,sizeof(*rr));
	  
	  if (rr->FD.FDCtl & MAC_OWNS_FD)
		{
		  if (ReadMacRegister(mac_ifindex, INT_SRC) & INTR_BLEx)
		    {
				/* receive buffer list exhausted, stop/start MAC */
				mspeth_driver(ETHER_CLOSE, 0, 0);
#ifdef VERBOSE				
				if (vflag)
				  {
					  printf("Stopping/Starting MAC #%d\n", ethnum);
				  }
#endif
				mspeth_driver(ETHER_INIT, 
							  mac_address[mac_ifindex], (void *) mac_ifindex);
			}
		  return 0;
		}
	  if (!(rr->FD.FDStat & RCVD_OK_PKT))
		{
#ifdef VERBOSE
		  if (vflag)
			{
			  printf("ETHER_RXRDY-packet not okay %x\n", rr);
			}
#endif
		  RxRelease(mac_ifindex, rr);
		  return 0;
		}
#ifdef VERBOSE
	  if (vflag)
		{
		  printf("ETHER_RXRDY-packet okay %x\n", rr);
		}
#endif
	  return ((void *)1);

	case ETHER_CLOSE:
	  /* insure all packets are transmitted */
	  delay = 1000;
	  cr = TxFrmPtrCur[mac_ifindex];
	  rr = TxFrmPtrCur[mac_ifindex];
	  while (delay--)
		{
		  HitInvDCache(rr,sizeof(*rr));
		  if (rr->FD.FDCtl & MAC_OWNS_FD)
			udelay(1);
		  else
			{
			  rr = (FDBD *) rr->FD.FDNext;
			  if (cr == rr)
				break;
			}
		}
	  /* int ether_driver(ETHER_CLOSE,void,void) */
	  DisableTxRx(mac_ifindex);
	  Reset(mac_ifindex);
	  return ((void *)1);

	default : 
#ifdef VERBOSE
	  if (vflag)
		{
		  printf("mspeth_driver-bad op code %d\n", op);
		}
#endif
	  return(0);
	}
	return(0);
}
#else /* Tasking tools don't like empty files (sigh) */
mspeth_foobar() {}
#endif /* ETHERNET */
