/*************************************************************
 * File: lib/ether.c
 * Purpose: common ethernet functions
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	980615	Created from sonic.c
 *	981127	Moved the GETRXREC swapPkt here from drivers.
 */

/* This module contains all of the chip-independent code needed to
 * PMON/IMON with an Ethernet download facility.  The code is rather 
 * primitive, it only understands UDP and ARP messages, and is only
 * able to send two types of packet (UDP REPLY and ARP REPLY). The
 * ether_xxx functions are performed by either sonic.c or am79970.c.
 */

#include <defines.h>
#ifdef ETHERNET
/* #define VERBOSE */   	/* be verbose */

#define ETHER_INTR

#include <mon.h>
#include <termio.h>
#include <ether.h>


#ifdef CHECKS_ON
#include <assert.h>
#else
#define assert(x)
#endif

#define swap32(n) ( \
	((((Ulong)(n))>>24)&0x000000ff) | \
	((((Ulong)(n))>>8)&0x0000ff00) |  \
	((((Ulong)(n))<<8)&0x00ff0000) |  \
	((((Ulong)(n))<<24)&0xff000000)   \
	)

#define swap16(n) ( \
	((((Ushort)(n))>>8)&0x00ff) | \
	((((Ushort)(n))<<8)&0xff00)   \
	)

#define ether_driver(a,b,c)	(*ether_driver_ptr)(a,b,c)
/* This is how you call the chip-dependent code. ether_driver_ptr
 * should have been initialized by your cpu-dependent module (eg. c4101.c),
 * to point to the appropriate driver (eg. sonic_driver).
 */

vpFunc *ether_driver_ptr;

static Ushort last_udp_seqno;
static Ushort tx_ip_id;
static char *rx_byte_ptr;
static int rx_byte_cnt;
static int dotcnt;
static RXREC *rx_desc;
static int bcopy_direction;
static int tftp_started;
static char *dest_addr;
static char *orig_addr;
static Ulong orig_len;
static Ulong dest_len;
static Uchar EnetAddr[6]; /* the MAC address */
static Ulong myip;

static filbuf();
static putbuf();
static char *getMsg();

#ifdef VERBOSE
static printIp();
static printUdp();
static printArp();
#endif

extern int re_ether;
extern int vflag;

static int doingPing;
int pingCount;
unsigned short pingLen;
Ulong pingAddress;
Ulong pingInterval;
static Ushort pingIpId;
static Ulong pingSequence;
static int pingcnt;
Uchar pingDHost[6];

/*************************************************************
*  ether_open(etheraddr)
*	Just like the standard open() except that it opens the
*	Ethernet controller.
*	'etheraddr' is a string containing the MAC address in the
*	form aa:bb:cc:dd:ee:ff.
*/
ether_open(etheraddr,ifindex)
char *etheraddr;
unsigned long ifindex;
{
char tmp[30],*p;
int i,ipi[4],n;

#ifdef ETHER_INTR
if (etheraddr == 0 || ether_driver_ptr == 0) longjmp(intrbuf,1);
#else
if (etheraddr == 0 || ether_driver_ptr == 0) return(-1);
#endif

rx_byte_cnt = 0;
rx_desc = 0;

/*
 * use the converted value of heaptop as the load address if
 * we are doing a binary load and the client doesn't specify
 * a load address
 */
dest_addr = (char *) getheaptop();

/* get our IP address */
sscanf(getMonEnv("ipaddr"),"%d.%d.%d.%d", &ipi[0],&ipi[1],&ipi[2],&ipi[3]);
myip = ((ipi[0]<<24) | (ipi[1]<<16) | (ipi[2]<<8) | ipi[3]);

/* convert the mac address string into an array of 6 bytes */
strcpy(tmp,etheraddr);
for (i=0,p=strtok(tmp,":");p;p=strtok(0,":"),i++) {
	sscanf(p,"%x",&n);
	EnetAddr[i] = n; 
	}

#ifdef ETHER_INTR
if (!ether_driver(ETHER_INIT,EnetAddr,ifindex)) longjmp(intrbuf,1);
#else
if (!ether_driver(ETHER_INIT,EnetAddr,ifindex)) return(-1);
#endif
flush_cache(DCACHE);
return(ETHERFD);
}


/*************************************************************
*  ether_close()
*	Just like the standard close() except that it closes the
*	Ethernet controller.
*/
ether_close()
{
char *rxptr;
int len;

if (rx_desc) {
	rxptr = (char *)ether_driver(ETHER_GETRBA,rx_desc,&len);
	sendAck(rxptr,len);
	ether_driver(ETHER_RXDONE,rx_desc,0);
	}

ether_driver(ETHER_CLOSE, 0, 0);
}

/*************************************************************
*  ether_read(fd,ptr,size)
*	Just like the standard read() except that it reads from the
*	Ethernet controller.
*/
ether_read(fd,ptr,size)
int fd,size;
char *ptr;
{
int i,c;


for (i=0;i<size;) {
	c = ((rx_byte_cnt--)?*rx_byte_ptr++:filbuf(0));
	ptr[i++] = c;
	if (c == '\n') break;
	}
return(i);
}

/*************************************************************
*  ether_download()
*	supports "load -B", binary mode transfers
*/
ether_download(int fd)
{
	bcopy_direction = 0;			/* flag doing load from hostport */
	tftp_started = 0;			/* flag tftp transfer not yet started */	
	doingPing = 0;

	for (;;) {
	  filbuf(1);
	  bcopy(rx_byte_ptr,dest_addr,rx_byte_cnt);
	  dest_addr += rx_byte_cnt;
	  
	  /* when tftp sends a block less than 512 bytes, we are done */
	  if (rx_byte_cnt < 512)
		break;
	}
	close(fd);
	
	/* tell people what we copied */
	
	if (orig_addr == dest_addr)
	  printf("Nothing copied\n");
	else
	  printf("Loaded data from %x to %x, length %x (%d decimal)\n",
			 orig_addr, 
			 dest_addr - 1, 
			 dest_addr - orig_addr,
			 dest_addr - orig_addr);
}

/*************************************************************
*  ether_upload()
*	supports "dump -B", binary mode transfers
*/
ether_upload(adr,siz)
Ulong adr,siz;
{
  int fd;
  char *hostport;

  
  bcopy_direction = 1;		  /* flag doing dump to hostport */
  tftp_started = 0;			/* flag tftp transfer not yet started */	
  doingPing = 0;

  hostport = getMonEnv("hostport");
  fd = open(hostport,0,0);
  if (fd == -1) {
	printf("can't open ethernet\n");
  } 
  else {
	dest_addr = orig_addr = (char *) adr;
	dest_len = orig_len = siz;
	
	putbuf();

	close(fd);
	
	/* tell people what we copied */
	
	if (orig_addr == dest_addr)
	  printf("Nothing copied\n");
	else
	  printf("Dumped data from %x to %x, length %x (%d decimal)\n",
			 orig_addr, 
			 dest_addr - 1, 
			 dest_addr - orig_addr,
			 dest_addr - orig_addr);
  }
}

/*************************************************************
*  ether_ping()
*	supports ping request.
*/
ether_ping(int fd)
{
	tftp_started = 0;			/* flag tftp transfer not yet started */	
	doingPing = 1;			/* flag doing ping to hostport */
	pingcnt = 0;
	filbuf(1);
	close(fd);
	doingPing = 0;			/* flag not doing ping to hostport */
	printf("Done\n");
}

/*************************************************************
*  filbuf(no_rtn)
*	Called when the buffer runs out of bytes.
*	It gets a packet of data and transfers it to the
*	buffer.
*/
static filbuf(no_rtn)
int no_rtn;
{
RXREC *q;
struct ether_header *e;
char *rxptr;
int len;

if (rx_desc) {
	ether_driver(ETHER_RXDONE,rx_desc,0);
	}

while (1) {
	while (!ether_driver(ETHER_RXRDY,0,0)) {
		dotcnt--;
		if (dotcnt <= 0 || ! tftp_started) {
			scandevs();
			dotcnt = 100000;
			}
		if (doingPing) {
			tftp_started = 0;
			pingcnt--;
			if (pingcnt <= 0) {
				pingcnt = pingInterval;
				if (pingCount-- > 0) {
					sendPingRequest();
					}
				else
					return 0;
				}
			}
		}
	if (!(q=(RXREC *)ether_driver(ETHER_GETRXREC,0,0))) continue;
	e = (struct ether_header *)ether_driver(ETHER_GETRBA,q,&len);
#ifndef MIPSEB
	if (!swapPkt(phy2k1(e))) {
		ether_driver(ETHER_RXDONE,(void *)q,0);
		continue;
		}
#endif
#ifdef VERBOSE
	if (vflag) printPkt(e);
#endif
	if (e->ether_type == ETHERTYPE_ARP) {
		arpRequest(e);
		ether_driver(ETHER_RXDONE,q,0);
		continue;
		}
	if (e->ether_type != ETHERTYPE_IP) {
		ether_driver(ETHER_RXDONE,q,0);
		continue;
	}
	rx_byte_ptr = getMsg(e,len,&rx_byte_cnt);
	if (rx_byte_ptr) break;
	ether_driver(ETHER_RXDONE,q,0);
	}
assert(rx_byte_cnt != 0);

rx_desc = q;
dotcnt = 100000;
if (rx_byte_cnt == 512) {
	rxptr = (char *)ether_driver(ETHER_GETRBA,rx_desc,&len);
	sendAck(rxptr,len);
	}

if (no_rtn) return(0);
rx_byte_cnt--;
return(*rx_byte_ptr++);
}


/*************************************************************
*  putbuf()
*	Called to put the next buffer.
*	It gets an ack packet and sends a buffer of data.
*/
static putbuf()
{
RXREC *q;
struct ether_header *e;
char *rxptr;
int len;

while (1) {
	while (!ether_driver(ETHER_RXRDY,0,0)) {
		dotcnt--;
		if (dotcnt <= 0 || ! tftp_started) {
			scandevs();
			dotcnt = 100000;
			}
		}
	if (!(q=(RXREC *)ether_driver(ETHER_GETRXREC,0,0))) continue;
	e = (struct ether_header *)ether_driver(ETHER_GETRBA,q,&len);
#ifndef MIPSEB
	if (!swapPkt(phy2k1(e))) {
		ether_driver(ETHER_RXDONE,(void *)q,0);
		continue;
		}
#endif
#ifdef VERBOSE
	if (vflag) printPkt(e);
#endif
	if (e->ether_type == ETHERTYPE_ARP) {
		arpRequest(e);
		ether_driver(ETHER_RXDONE,q,0);
		continue;
		}
	if (e->ether_type != ETHERTYPE_IP) {
		ether_driver(ETHER_RXDONE,q,0);
		continue;
	}
	rx_byte_ptr = getMsg(e,len,&rx_byte_cnt);
	ether_driver(ETHER_RXDONE,q,0);
	if (dest_len == 0) break;
}
}


/*************************************************************
*  sendAck(re,len)
*	send an ACK back to the host
*/
sendAck(re,len)
struct ether_header *re;
Ushort len;
{
int i,n;
Ushort *sp;
Ulong t;
struct ether_header *te;
struct ip *ti,*ri;
struct udphdr *tu,*ru;

if (vflag) {
	printf("sendAck(%08x,%d)\n",re,len);
	printMem((char *)re,len);
	}
assert(re != 0);
n = 64;
te = (struct ether_header *) ether_driver(ETHER_GETTBA,0,&n);
te->ether_shost = re->ether_dhost;
te->ether_dhost = re->ether_shost;
te->ether_type = re->ether_type;

ti = (struct ip *) (te+1);
ri = (struct ip *) (re+1);
ti->ip_vhl = ri->ip_vhl;
ti->ip_tos = ri->ip_tos;
ti->ip_len = sizeof(struct ip)+sizeof(struct udphdr);
ti->ip_id = tx_ip_id++;
ti->ip_off = 0;
ti->ip_ttl = UDP_TTL;
ti->ip_p = IP_UDP;
bcopy(&(ri->ip_dst.s_addr),&(ti->ip_src.s_addr),sizeof(struct in_addr));
bcopy(&(ri->ip_src.s_addr),&(ti->ip_dst.s_addr),sizeof(struct in_addr));

tu = (struct udphdr *) (ti+1);
ru = (struct udphdr *) (ri+1);
tu->uh_sport = ru->uh_sport;
tu->uh_dport = ru->uh_sport;
tu->uh_ulen = 12;
tu->uh_sum = 0; /* haven't figured this out, try zero */
tu->uh_type = UDP_ACK;
tu->uh_seqno = last_udp_seqno;

#ifndef MIPSEB
unswapPkt(te);
#endif

/* compute csum of ip hdr -> t */
ti->ip_sum = 0;
sp = (Ushort *) ti;
for (i=0,t=0;i<(sizeof(struct ip)/sizeof(Ushort));i++,sp++) t += *sp;
t = (t&0xffff)+(t>>16);
ti->ip_sum = ~t;

if (vflag) printMem((char *)te,64);
#ifdef VERBOSE
if (vflag) printPkt(te,64);
#endif
if (re_ether) swap32n(te,64);
ether_driver(ETHER_TBARDY,0,0);
}

/*************************************************************
*  sendData(re,len)
*	send data back to the host
*/
sendData(re,len)
struct ether_header *re;
Ushort len;
{
int i,n;
Ushort *sp;
Ulong t;
struct ether_header *te;
struct ip *ti,*ri;
struct udphdr *tu,*ru;
char *ptr;

if (vflag) {
	printf("sendData(%08x,%d)\n",re,len);
	printMem((char *)re,len);
	}
assert(re != 0);
n = (dest_len > 512) ? 512 : dest_len;
n += 64;
te = (struct ether_header *) ether_driver(ETHER_GETTBA,0,&n);
n = (dest_len > 512) ? 512 : dest_len;
te->ether_shost = re->ether_dhost;
te->ether_dhost = re->ether_shost;
te->ether_type = re->ether_type;

ti = (struct ip *) (te+1);
ri = (struct ip *) (re+1);
ti->ip_vhl = ri->ip_vhl;
ti->ip_tos = ri->ip_tos;
ti->ip_len = sizeof(struct ip)+sizeof(struct udphdr)+n;
ti->ip_id = tx_ip_id++;
ti->ip_off = 0;
ti->ip_ttl = UDP_TTL;
ti->ip_p = IP_UDP;
bcopy(&(ri->ip_dst.s_addr),&(ti->ip_src.s_addr),sizeof(struct in_addr));
bcopy(&(ri->ip_src.s_addr),&(ti->ip_dst.s_addr),sizeof(struct in_addr));

tu = (struct udphdr *) (ti+1);
ru = (struct udphdr *) (ri+1);
tu->uh_sport = ru->uh_sport;
tu->uh_dport = ru->uh_sport;
tu->uh_ulen = 12+n;
tu->uh_sum = 0; /* haven't figured this out, try zero */
tu->uh_type = UDP_DATA;
tu->uh_seqno = last_udp_seqno;
ptr = (char *) (tu+1);
bcopy(dest_addr, ptr, n);

#ifndef MIPSEB
unswapPkt(te);
#endif

/* compute csum of ip hdr -> t */
ti->ip_sum = 0;
sp = (Ushort *) ti;
for (i=0,t=0;i<(sizeof(struct ip)/sizeof(Ushort));i++,sp++) t += *sp;
t = (t&0xffff)+(t>>16);
ti->ip_sum = ~t;

if (vflag) printMem((char *)te,64);
#ifdef VERBOSE
if (vflag) printPkt(te,64);
#endif
if (re_ether) swap32n(te,64);
ether_driver(ETHER_TBARDY,0,0);
}

/*************************************************************
*  char *getMsg(e,size,cnt)
*	Interprets the packet. 
*	Is only able to understand UDP packets of type UDP_RRQ, UDP_WRQ, 
*	and UDP_DATA.
*/
static char *getMsg(e,size,cnt)
struct ether_header *e;
Ushort size;
int *cnt;
{
struct ip *i;
struct udphdr *u;
int j,minlen;
Ulong ipaddr,t;
Ushort *sp;

*cnt = 0;
assert(e != 0);
minlen = sizeof(struct ether_header)+sizeof(struct ip)+sizeof(struct udphdr);
assert(size >= minlen);
i = (struct ip *)(e+1);
if (getIP_V(i->ip_vhl) != 4 || getIP_HL(i->ip_vhl) != 5)
	return 0;		/* wrong version or has options, and we don't do options */
bcopy(&(i->ip_dst.s_addr),&ipaddr,4);
if (pingRequest(e)) return(0);
if (pingReply(e)) return(0);
if (ipaddr != myip)
	 return 0;							/* if not my ip address */
if (i->ip_p != IP_UDP) return(0);

sp = (Ushort *) i;
for (j=0,t=0;j<(sizeof(struct ip)/sizeof(Ushort));j++,sp++) t += *sp;
t = (t&0xffff)+(t>>16);
if (t != 0xffff) return(0);			/* bad checksum */

u = (struct udphdr *)(i+1);

if (u->uh_sum != 0) {					/* check udp checksum */
	sp = (Ushort *) &i->ip_src.s_addr;
	for (j=0,t=0;j<(sizeof(Ulong)/sizeof(Ushort));j++,sp++) t += *sp;
	for (j=0;j<(sizeof(Ulong)/sizeof(Ushort));j++,sp++) t += *sp;
	t += IP_UDP;
	t += u->uh_ulen;
	sp = (Ushort *) u;
	for (j=0;j<(u->uh_ulen/2);j++,sp++) t += *sp;
	if (u->uh_ulen&1) t += (*sp&0xff00);
	t = (t&0xffff)+(t>>16);
	if (t != 0xffff) return(0);			/* bad checksum */
}

switch (u->uh_type) {
case UDP_WRQ:
case UDP_DATA:
  if (bcopy_direction) {
	printf("Expecting upload, host tried download, ignored\n");
	return 0;
  }
  break;
case UDP_RRQ:
case UDP_ACK:
  if (!bcopy_direction) {
	printf("Expecting download, host tried upload, ignored\n");
	return 0;
  }
  break;
}

switch (u->uh_type) {
	case UDP_WRQ : 
#ifndef MIPSEB
		u->uh_seqno = swap16(u->uh_seqno);
#endif
		t = 0;
		sscanf(&(u->uh_seqno),"%x",&t);
		if (t != 0)
		  dest_addr = (char *) t;
		orig_addr = dest_addr;
		last_udp_seqno = 0;
		tftp_started = 1;
		tx_ip_id = i->ip_id;
		sendAck(e,size);
		return(0);
	case UDP_DATA :
		if (u->uh_seqno == ((last_udp_seqno+1) & 0xffff)) {
			last_udp_seqno = u->uh_seqno;
			*cnt = u->uh_ulen-sizeof(struct udphdr);
			return((char *)(u+1));
			}
		else {
			/* retry packet */
			sendAck(e,size);
			}
		break;
	case UDP_RRQ :
	  last_udp_seqno = 1;		/* last block sent */
	  tftp_started = 1;
	  dest_addr = orig_addr;
	  dest_len = orig_len;

	  u->uh_seqno = 0;			/* preset peer is asking for first block */

	  /* fall through */
	case UDP_ACK :
		if (u->uh_seqno == last_udp_seqno)
		{		/* if acknowldged previous block, advance to next */

		  if (dest_len < 512) {	/* if short block acknowledged, done */
			dest_addr += dest_len;
			dest_len = 0;
			return 0;			/* done */
		  }

		  dest_addr += 512;
		  dest_len -= 512;
		  last_udp_seqno++;
		}
	  sendData(e,size);
	  return ((char *)(u+1));
	  break;
	}
return(0);
}

/*************************************************************
*  sendPingRequest(e)
*/
sendPingRequest()
{
struct ether_header *te;
struct ip *i,*ti;
struct icmp *tm;
Ushort *sp;
Ulong t;
int n, icmpheaderlen;

/* Build a ping request */
icmpheaderlen = (Ulong) &tm->icmp_data - (Ulong) &tm->icmp_type;
n = pingLen + icmpheaderlen + sizeof(struct ip) + sizeof(struct ether_header);
te = (struct ether_header *) ether_driver(ETHER_GETTBA,0,&n);
bcopy(pingDHost,&(te->ether_dhost),6);
bcopy(EnetAddr,&(te->ether_shost),6);
te->ether_type = ETHERTYPE_IP;

/* build ip header */
ti = (struct ip *) (te+1);
ti->ip_vhl = 0x45;
ti->ip_tos = 0;
ti->ip_len = sizeof(struct ip) + pingLen + icmpheaderlen;
ti->ip_id = ++pingIpId;
ti->ip_off = 0;
ti->ip_ttl = 250;
ti->ip_p = IP_ICMP;
bcopy(&(myip),&(ti->ip_src.s_addr),sizeof(struct in_addr));
bcopy(&(pingAddress),&(ti->ip_dst.s_addr),sizeof(struct in_addr));

/* build icmp header */
tm = (struct icmp *) (ti+1);
tm->icmp_type = ICMP_ECHO;
tm->icmp_code = 0;
tm->icmp_cksum = 0;
tm->icmp_seq = ++pingSequence;
tm->icmp_id = pingSequence >> 16;
for (n = 0; n < pingLen; n++)
	tm->icmp_data[n] = n;

sp = (Ushort *) tm;
for (n=0,t=0;n<((pingLen+icmpheaderlen)/sizeof(Ushort));n++,sp++) 
	t += *sp;
if (pingLen & 1) t += (*sp & 0xff00);
t = (t&0xffff)+(t>>16);
tm->icmp_cksum = ~t;


#ifndef MIPSEB
unswapPkt(te);
#endif

/* compute csum of ip hdr -> t */
ti->ip_sum = 0;
sp = (Ushort *) ti;
for (n=0,t=0;n<(sizeof(struct ip)/sizeof(Ushort));n++,sp++) t += *sp;
t = (t&0xffff)+(t>>16);
ti->ip_sum = ~t;

 printf("Sent ping request, length %d, id %d, seq %d", 
		ti->ip_len, tm->icmp_id, tm->icmp_seq); 
 if (vflag) printMem((char *)te,ti->ip_len+sizeof(struct ether_header));
#ifdef VERBOSE
if (vflag) printPkt(te,64);
#endif
if (re_ether) swap32n(te,ti->ip_len+sizeof(struct ether_header));
ether_driver(ETHER_TBARDY,0,0);
printf("\n");
}

/*************************************************************
*  pingRequest(e)
*/
pingRequest(e)
struct ether_header *e;
{
struct ether_header *te;
struct ip *i,*ti;
struct icmp *m,*tm;
Ushort *sp;
Ulong t;
int n,len;

if (e->ether_type != ETHERTYPE_IP) return(0);
i = (struct ip *)(e+1);
if (i->ip_p != IP_ICMP) return(0);
m = (struct icmp *)(i+1);
if (m->icmp_type != ICMP_ECHO) return(0);

/* This is a ping request */
len = i->ip_len+sizeof(struct ether_header);
te = (struct ether_header *) ether_driver(ETHER_GETTBA,0,&len);
te->ether_dhost = e->ether_shost;
bcopy(EnetAddr,&(te->ether_shost),6);
te->ether_type = e->ether_type;

/* build ip header */
ti = (struct ip *) (te+1);
ti->ip_vhl = i->ip_vhl;
ti->ip_tos = i->ip_tos;
ti->ip_len = i->ip_len;
ti->ip_id = i->ip_id;
ti->ip_off = i->ip_off;
ti->ip_ttl = i->ip_ttl;
ti->ip_p = i->ip_p; /* proto */
bcopy(&(myip),&(ti->ip_src.s_addr),sizeof(struct in_addr));
bcopy(&(i->ip_src.s_addr),&(ti->ip_dst.s_addr),sizeof(struct in_addr));

/* build icmp header */
tm = (struct icmp *) (ti+1);
tm->icmp_type = ICMP_ECHOREPLY;
tm->icmp_code = m->icmp_code;
tm->icmp_cksum = 0;
tm->icmp_id = m->icmp_id;
tm->icmp_seq = m->icmp_seq;
bcopy(m->icmp_data,tm->icmp_data,i->ip_len-sizeof(struct ip));

/* this is a hack. The real checksum calc appears to be 16-bit add
starting at icmp_type and continuing for 62 bytes. Then add 4. Write
the one's comp value into the chksum field. */
tm->icmp_cksum = ~((~m->icmp_cksum) - 0x800);

#ifndef MIPSEB
unswapPkt(te);
#endif

/* compute csum of ip hdr -> t */
ti->ip_sum = 0;
sp = (Ushort *) ti;
for (n=0,t=0;n<(sizeof(struct ip)/sizeof(Ushort));n++,sp++) t += *sp;
t = (t&0xffff)+(t>>16);
ti->ip_sum = ~t;

if (vflag) printMem((char *)te,i->ip_len+sizeof(struct ether_header));
#ifdef VERBOSE
if (vflag) printPkt(te,64);
#endif
if (re_ether) swap32n(te,i->ip_len+sizeof(struct ether_header));
ether_driver(ETHER_TBARDY,0,0);
return(1);
}

/*************************************************************
*  pingReply(e)
*/
pingReply(e)
struct ether_header *e;
{
struct ip *i;
struct icmp *m;
Ushort *sp;

if (e->ether_type != ETHERTYPE_IP) return(0);
i = (struct ip *)(e+1);
if (i->ip_p != IP_ICMP) return(0);
m = (struct icmp *)(i+1);
if (m->icmp_type != ICMP_ECHOREPLY) return(0);

/* This is a ping reply */

 printf("Received ping reply, length %d, id %d, seq %d", 
		i->ip_len, m->icmp_id, m->icmp_seq); 
if (vflag) printMem((char *)e,i->ip_len+sizeof(struct ether_header));
#ifdef VERBOSE
if (vflag) printPkt(e,64);
#endif
printf("\n");
return(1);
}

/*************************************************************
*  arpRequest(e)
*/
arpRequest(re)
struct ether_header *re;
{
struct arphdr *ra,*ta;
Ulong tpa;
struct ether_header *te;
int len;

assert(re != 0);
ra = (struct arphdr *)(re+1);
if (!(ra->ar_hrd == ARPHRD_ETHER && ra->ar_pro == 0x800 && ra->ar_hln == 6 && 
      ra->ar_pln == 4 && ra->ar_op == ARPOP_REQUEST)) return;

/* check the IP address */
bcopy(ra->ar_tpa,&tpa,4);
if (tpa != myip) return; /* not me */

len = 64;
te = (struct ether_header *) ether_driver(ETHER_GETTBA,0,&len);
te->ether_dhost = re->ether_shost;
bcopy(EnetAddr,&(te->ether_shost),6);
te->ether_type = re->ether_type;

ta = (struct arphdr *) (te+1);
ta->ar_hrd = ra->ar_hrd;
ta->ar_pro = ra->ar_pro;
ta->ar_hln = ra->ar_hln;
ta->ar_pln = ra->ar_pln;
ta->ar_op = ARPOP_REPLY;

bcopy(EnetAddr,ta->ar_sha,6);
bcopy(ra->ar_tpa,ta->ar_spa,4);
bcopy(ra->ar_sha,ta->ar_tha,6);
bcopy(ra->ar_spa,ta->ar_tpa,4);

#ifndef MIPSEB
unswapPkt(te);
#endif

if (vflag) printMem((char *)te,64);
#ifdef VERBOSE
if (vflag) printPkt(te,64);
#endif
if (re_ether) swap32n(te,64);
ether_driver(ETHER_TBARDY,0,0);
}

/*************************************************************
*  printMem(p,n)
*	print 'n' bytes of memory starting at 'p'.
*/
printMem(p,n)
char *p;
int n;
{
int i,len;
char ascii[17];

#ifdef HDRONLY
if (n > 80) n = 80;
#endif
printf("\n%08x  ",p);
len = 0;
for (i=0;;i++) {
	if (i && i%16 == 0) {
		ascii[len] = 0;
		printf("  %s\n",ascii);
		if (i>=n) break;
		len = 0;
		printf("%08x  ",&p[i]);
		}
	if (i>=n) {
		printf("   ");
		ascii[len++] = ' ';
		}
	else {
		printf("%02x ",0xff&p[i]);
		ascii[len++] = (isprint(p[i]))?p[i]:'.';
		}
	}
}

#ifdef VERBOSE
/*************************************************************
*  printPkt(e,len)
*	print the contents of an Ethernet header
*	If the next item is an IP header, print that also.
*/
printPkt(e,len)
struct ether_header *e;
int len;
{

printMem((char *)e,len);
printf("dhost=%02x:%02x:%02x:%02x:%02x:%02x   ",
	e->ether_dhost.ether_addr_octet[0],
	e->ether_dhost.ether_addr_octet[1],
	e->ether_dhost.ether_addr_octet[2],
	e->ether_dhost.ether_addr_octet[3],
	e->ether_dhost.ether_addr_octet[4],
	e->ether_dhost.ether_addr_octet[5]);

printf("shost=%02x:%02x:%02x:%02x:%02x:%02x\n",
	e->ether_shost.ether_addr_octet[0],
	e->ether_shost.ether_addr_octet[1],
	e->ether_shost.ether_addr_octet[2],
	e->ether_shost.ether_addr_octet[3],
	e->ether_shost.ether_addr_octet[4],
	e->ether_shost.ether_addr_octet[5]);


switch (e->ether_type) {
	case ETHERTYPE_IP : printIp(e+1); break;
	case ETHERTYPE_PUP: printf("type=PUP, not impl\n"); break;
	case ETHERTYPE_ARP: printArp(e+1); break;
	case ETHERTYPE_REVARP: printf("type=REVARP, not impl\n"); break;
	default : printf("type=%04x\n",e->ether_type);
	}
}

/*************************************************************
*  printIp(p)
*	print the contents of an IP header
*	If the next item is a UDP header, print that also.
*/
static printIp(p)
struct ip *p;
{
struct ip q;

bcopy(p,&q,sizeof(struct ip));
printf("ip: ver=%02x len=%04x id=%04x ttl=%02x proto=%02x src=%08x dst=%08x\n", 
		getIP_V(q.ip_vhl),q.ip_len,q.ip_id,q.ip_ttl,q.ip_p,
		q.ip_src.s_addr,q.ip_dst.s_addr);

switch (q.ip_p) {
	case IP_IP:	printf("ip protocol\n"); break;
	case IP_ICMP:	printf("icmp protocol\n"); break;
	case IP_IGMP:	printf("igmp protocol\n"); break;
	case IP_GGP:	printf("ggp protocol\n"); break;
	case IP_TCP:	printf("tcp protocol\n"); break;
	case IP_PUP:	printf("pup protocol\n"); break;
	case IP_UDP:	printUdp(p+1); break;
	default : 
		printf("%02x: unknown ip protocol\n",q.ip_p);
		return(1);
	}
}

/*************************************************************
*  printUdp(p)
*	print the contents of a UDP header
*/
static printUdp(p)
struct udphdr *p;
{

printf("udp: sport=%04x dport=%04x ulen=%04x sum=%04x type=%04x seq=%04x ",
	p->uh_sport,p->uh_dport,p->uh_ulen,p->uh_sum,p->uh_type,p->uh_seqno);

switch (p->uh_type) {
	case UDP_RRQ : printf("open read \"%s\"\n",&(p->uh_seqno)); break;
	case UDP_WRQ : printf("open write \"%s\"\n",&(p->uh_seqno)); break;
	case UDP_ACK : printf("ack\n"); break;
	case UDP_DATA : printf("data\n"); break;
	default : printf("%4x: unknown udp type\n",p->uh_type);
	}
}

/*************************************************************
*  printArp(p)
*	print the contents of a ARP header
*/
static printArp(p)
struct arphdr *p;
{
printf("arp: hrd=%04x pro=%04x hln=%02x pln=%02x op=%04x \n",
	p->ar_hrd,p->ar_pro,p->ar_hln,p->ar_pln,p->ar_op);
printf("sha=%02x:%02x:%02x:%02x:%02x:%02x ",
	p->ar_sha[0], p->ar_sha[1], p->ar_sha[2], p->ar_sha[3],
	p->ar_sha[4], p->ar_sha[5]);
printf("spa=%02x.%02x.%02x.%02x\n",
	p->ar_spa[0], p->ar_spa[1], p->ar_spa[2], p->ar_spa[3]);
printf("tha=%02x:%02x:%02x:%02x:%02x:%02x ",
	p->ar_tha[0], p->ar_tha[1], p->ar_tha[2], p->ar_tha[3],
	p->ar_tha[4], p->ar_tha[5]);
printf("tpa=%02x.%02x.%02x.%02x\n",
	p->ar_tpa[0], p->ar_tpa[1], p->ar_tpa[2], p->ar_tpa[3]);
}
#endif

/*************************************************************
*  swap32n(p,n)
*	Byte-swap the contents of 'n', 32-bit words pointed to by 'p'.
*/
swap32n(p,n)
long *p;
int n;
{
int i;

if (n%4) n = (n/4)+1; /* 971115 */
else n /= 4;
for (i=0;i<n;i++) p[i] = swap32(p[i]);
}

#ifndef MIPSEB
/*************************************************************
*  int swapPkt(e)
*	Byte-swap the headers of an Ethernet packet
*/
int swapPkt(e)
struct ether_header *e;
{
struct ip *i;
struct udphdr *u;

swapEh(e);
if (e->ether_type == ETHERTYPE_ARP) {
	swapArp((struct arphdr *)(e+1));
	return(1);
	}
if (e->ether_type != ETHERTYPE_IP) return(0);

i = (struct ip *)(e+1);
swapIp(i);
if (i->ip_p == IP_ICMP) swapIcmp((struct icmp *)(i+1));
else if (i->ip_p == IP_UDP) swapUdp((struct udphdr *)(i+1));
else printf("swapPkt: %02x: bad ip_p\n",i->ip_p);
return(1);
}

/*************************************************************
*  unswapPkt(e)
*	unByte-swap the headers of an Ethernet packet
*/
unswapPkt(e)
struct ether_header *e;
{
struct ip *i;
struct udphdr *u;

if (e->ether_type == ETHERTYPE_ARP) {
	swapArp((struct arphdr *)(e+1));
	swapEh(e);
	return;
	}
if (e->ether_type != ETHERTYPE_IP) {
	printf("unswapPkt: %04x: bad ether_type\n",e->ether_type);
	return;
	}
i = (struct ip *)(e+1);
if (i->ip_p == IP_ICMP) {
	swapIcmp((struct icmp *)(i+1));
	swapIp(i);
	swapEh(e);
	return;
	}
if (i->ip_p != IP_UDP) {
	printf("unswapPkt: %02x: bad ip_p\n",i->ip_p);
	return;
	}
u = (struct udphdr *)(i+1);
swapUdp(u);
swapIp(i);
swapEh(e);
}

/*************************************************************
*  swapEh(e)
*	Byte-swap the Ethernet header
*/
swapEh(e)
struct ether_header *e;
{
e->ether_type = swap16(e->ether_type);
}

/*************************************************************
*  swapIp(i)
*	Byte-swap the IP header
*/
swapIp(i)
struct ip *i;
{
Ulong t;

i->ip_len = swap16(i->ip_len);
i->ip_id = swap16(i->ip_id);
i->ip_off = swap16(i->ip_off);
i->ip_sum = swap16(i->ip_sum);
bcopy(&(i->ip_src.s_addr),&t,4);
t = swap32(t);
bcopy(&t,&(i->ip_src.s_addr),4);
bcopy(&(i->ip_dst.s_addr),&t,4);
t = swap32(t);
bcopy(&t,&(i->ip_dst.s_addr),4);
}

/*************************************************************
*  swapUdp(u)
*	Byte-swap the UDP header
*/
swapUdp(u)
struct udphdr *u;
{
u->uh_sport = swap16(u->uh_sport);
u->uh_dport = swap16(u->uh_dport);
u->uh_ulen = swap16(u->uh_ulen);
u->uh_sum = swap16(u->uh_sum);
u->uh_type = swap16(u->uh_type);
u->uh_seqno = swap16(u->uh_seqno);
}

/*************************************************************
*  swapArp(a)
*/
swapArp(a)
struct arphdr *a;
{
Ulong t;

a->ar_hrd = swap16(a->ar_hrd);
a->ar_pro = swap16(a->ar_pro);
a->ar_op = swap16(a->ar_op);
bcopy(a->ar_spa,&t,4);
t = swap32(t);
bcopy(&t,a->ar_spa,4);
bcopy(a->ar_tpa,&t,4);
t = swap32(t);
bcopy(&t,a->ar_tpa,4);
}

/*************************************************************
*  swapIcmp(a)
*/
swapIcmp(m)
struct icmp *m;
{

m->icmp_cksum = swap16(m->icmp_cksum);
m->icmp_id = swap16(m->icmp_id);
m->icmp_seq = swap16(m->icmp_seq);
}
#endif
#else /* Tasking tools don't like empty files (sigh) */
ether_foobar() {}
#endif /* ETHERNET */
