/*************************************************************
 * File: lib/crt1.s
 * Purpose: Part of C runtime library
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970410	Added code to skip the timing check if a1 == zero
 *      981012  Added word align code to sbrk. BSS_END can be unaligned.
 */

#include <mips.h>
#include <defines.h>

/*
 * Start-up code for executing a C program under LSI Logic's Prom Monitor 
 */

#define TBLBASE 0xbfc00500
#define TIMEREQ 0x74		/* 't' */

#define TBLENTRY(n,m) 			\
	.globl m; 			\
	.ent m;				\
m:	li	t0,TBLBASE+(4*n);	\
	lw	t0,(t0);		\
	j	t0;			\
	.end m

#ifdef TIMING
	.comm _timing,8
#endif

#ifdef BSO_TASKING
	.globl _filebase, 4
	.lcomm _filebase,4
#else
	.comm _filebase,4
#endif

	.globl _exit
	.globl _start
	.ent _start
_start:
	# note that clrbss must not use a3
	jal	clrbss

	# set the global data pointer
	la	gp,_gp

	# save file base address
	sw	a3,_filebase

#ifdef TIMING
	# is this a timing request?
	beq	a1,zero,1f
	lw	t0,(a1)
	lbu	t0,(t0)
	bne	t0,TIMEREQ,1f

	move	s0,a0
	move	s1,a1
	li	a0,0
	jal	time
	la	t4,_timing
	li	t0,1
	sw	t0,(t4)
	sw	v0,4(t4)
	move	a0,s0
	move	a1,s1
#endif

1: 	# call the main C routine
	jal	main
_exit:

#ifdef TIMING
	# was this a timing request?
	la	s4,_timing
	lw	t0,(s4)
	beq	t0,zero,1f

	li	a0,0
	jal	time
	lw	t0,4(s4)
	subu	a1,v0,t0
	la	a0,timefmt
	jal	printf
#endif

1:
	li	t0,TBLBASE+(4*8)
	lw	t0,(t0)
	j	t0
	.end _start


/*************************************************************
*  sbrk(size)
*	returns a pointer to a block of memory of the requested size.
*	Returns zero if heap overflow is detected. Heap overflow occurs
*	when the upper limit of the requested size, overlaps the stack
*	pointer.
*/
	.globl sbrk
	.ent sbrk
sbrk:
	li	v0,0
	la	t0,allocp1
	lw	t6,(t0)

	# force word align
	addu	t6,3
	and	t6,~3

	# check for heap overflow
	and	t1,t6,0xe0000000	# get seg
	and	t2,sp,~0xe0000000
	or	t1,t2			# heap limit
	addu	t7,t6,a0
	blt	t7,t1,1f
	j	ra

1:	# ok - pass value back to caller
	sw	t7,(t0)
	subu	v0,t7,a0
	j	ra
	.end sbrk

#ifdef CYGNUS
	.globl __main
	.ent __main
__main:
	j	ra
	.end __main
#endif

/*************************************************************
*  PMON entry table
*/
TBLENTRY(0,_read)
TBLENTRY(1,_write)
TBLENTRY(2,_open)
TBLENTRY(3,_close)
TBLENTRY(4,_ioctl)
TBLENTRY(5,printf)
TBLENTRY(6,vsprintf)
TBLENTRY(7,ttctl)
TBLENTRY(9,getenv)
TBLENTRY(10,onintr)
TBLENTRY(11,flush_cache)
TBLENTRY(17,adr2symoff)
TBLENTRY(18,sym2adr)
TBLENTRY(19,getclkfreq)
TBLENTRY(20,_clkinit)
TBLENTRY(21,scandevs)
TBLENTRY(22,do_cmd)

	.data
	.globl allocp1
#ifndef BSO_TASKING
	.word _ftext
	.word etext
#endif
allocp1: .word  BSS_END

#ifdef TIMING
timefmt: .asciiz "time %d secs\n"
#endif

