/*************************************************************
 * File: include/sonic.h
 * Purpose: Constant definitions for National Sonic Ethernet Chip
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	960924	Added DW32 to support the SDP1000 board
 */

#define SONIC_CR	(*(SONIC_BASE+0))
#define SONIC_DCR	(*(SONIC_BASE+2))
#define SONIC_RCR	(*(SONIC_BASE+4))
#define SONIC_TCR	(*(SONIC_BASE+6))
#define SONIC_IMR	(*(SONIC_BASE+8))
#define SONIC_ISR	(*(SONIC_BASE+10))
#define SONIC_UTDA	(*(SONIC_BASE+12))
#define SONIC_CTDA	(*(SONIC_BASE+14))
#define SONIC_TPS	(*(SONIC_BASE+16))
#define SONIC_TFC	(*(SONIC_BASE+18))
#define SONIC_TSA0	(*(SONIC_BASE+20))
#define SONIC_TSA1	(*(SONIC_BASE+22))
#define SONIC_TFS	(*(SONIC_BASE+24))
#define SONIC_URDA	(*(SONIC_BASE+26))
#define SONIC_CRDA	(*(SONIC_BASE+28))
#define SONIC_CRBA0	(*(SONIC_BASE+30))
#define SONIC_CRBA1	(*(SONIC_BASE+32))
#define SONIC_RBWC0	(*(SONIC_BASE+34))
#define SONIC_RBWC1	(*(SONIC_BASE+36))
#define SONIC_EOBC	(*(SONIC_BASE+38))
#define SONIC_URRA	(*(SONIC_BASE+40))
#define SONIC_RSA	(*(SONIC_BASE+42))
#define SONIC_REA	(*(SONIC_BASE+44))
#define SONIC_RRP	(*(SONIC_BASE+46)) 
#define SONIC_RWP	(*(SONIC_BASE+48))
#define SONIC_TRBA0	(*(SONIC_BASE+50))
#define SONIC_TRBA1	(*(SONIC_BASE+52))
#define SONIC_TBWC0	(*(SONIC_BASE+54))
#define SONIC_TBWC1	(*(SONIC_BASE+56))
#define SONIC_ADDR0	(*(SONIC_BASE+58))
#define SONIC_ADDR1	(*(SONIC_BASE+60))
#define SONIC_LLFA	(*(SONIC_BASE+62))
#define SONIC_TTDA	(*(SONIC_BASE+64))
#define SONIC_CEP	(*(SONIC_BASE+66))
#define SONIC_CAP2	(*(SONIC_BASE+68))
#define SONIC_CAP1	(*(SONIC_BASE+70))
#define SONIC_CAP0	(*(SONIC_BASE+72))
#define SONIC_CE	(*(SONIC_BASE+74))
#define SONIC_CDP	(*(SONIC_BASE+76))
#define SONIC_CDC	(*(SONIC_BASE+78))
#define SONIC_SR	(*(SONIC_BASE+80))
#define SONIC_WT0	(*(SONIC_BASE+82))
#define SONIC_WT1	(*(SONIC_BASE+84))
#define SONIC_RSC	(*(SONIC_BASE+86))
#define SONIC_CRCT	(*(SONIC_BASE+88))
#define SONIC_FAET	(*(SONIC_BASE+90))
#define SONIC_MPT	(*(SONIC_BASE+92))
#define SONIC_MDT	(*(SONIC_BASE+94))

/* isr */
#define ISR_BR		(1<<14)
#define ISR_HBL		(1<<13)
#define ISR_LCD		(1<<12)
#define ISR_PINT	(1<<11)
#define ISR_PKTRX	(1<<10)
#define ISR_TXDN	(1<<9)
#define ISR_TXER	(1<<8)
#define ISR_TC		(1<<7)
#define ISR_RDE		(1<<6)
#define ISR_RBE		(1<<5)
#define ISR_RBAE	(1<<4)
#define ISR_CRC		(1<<3)
#define ISR_FAE		(1<<2)
#define ISR_MP		(1<<1)
#define ISR_RFO		(1<<0)

/* cr */
#define CR_LCAM		(1<<9)
#define CR_RRRA		(1<<8)
#define CR_RST		(1<<7)
#define CR_ST		(1<<5)
#define CR_STP		(1<<4)
#define CR_RXEN		(1<<3)
#define CR_RXDIS	(1<<2)
#define CR_TXP		(1<<1)
#define CR_HTX		(1<<0)

/* rcr */
#define RCR_ERR		(1<<15)
#define RCR_RNT		(1<<14)
#define RCR_BRD		(1<<13)
#define RCR_PRO		(1<<12)
#define RCR_AMC		(1<<11)
#define RCR_MAC		(1<<9)	/* MAC loopback */
#define RCR_ENDEC	(2<<9)	/* ENDEC loopback */
#define RCR_TXCVR	(3<<9)	/* Transceiver loopback */
#define RCR_MC		(1<<8)
#define RCR_BC		(1<<7)
#define RCR_LPKT	(1<<6)
#define RCR_CRS		(1<<5)
#define RCR_COL		(1<<4)
#define RCR_CRC		(1<<3)
#define RCR_FAE		(1<<2)
#define RCR_LBK		(1<<1)
#define RCR_PRX		(1<<0)

/* tcr */
#define TCR_PTX		(1<<0)

/* dcr */
#define DCR_PO1		(1<<12)
#define DCR_PO0		(1<<11)
#define DCR_STERM	(1<<10)
#define DCR_WAIT0	(0<<6)
#define DCR_WAIT1	(1<<6)
#define DCR_WAIT2	(2<<6)
#define DCR_WAIT3	(3<<6)
#define DCR_DW32	(1<<5)
#define DCR_TFT1	(1<<1)
#define DCR_TFT0	(1<<0)

/* imr */
#define IMR_BREN	(1<<14)
#define IMR_HBLEN	(1<<13)
#define IMR_LCDEN	(1<<12)
#define IMR_PINTEN	(1<<11)
#define IMR_PRXEN	(1<<10)
#define IMR_PTXEN	(1<<9)
#define IMR_TXEREN	(1<<8)
#define IMR_TCEN	(1<<7)
#define IMR_RDEEN	(1<<6)
#define IMR_RBEEN	(1<<5)
#define IMR_RBAEEN	(1<<4)
#define IMR_CRCEN	(1<<3)
#define IMR_FAEEN	(1<<2)
#define IMR_MPEN	(1<<1)
#define IMR_RFOEN	(1<<0)

/*************** transmit descriptors **************************/
#ifdef SONIC_DW32
#ifdef MIPSEB
typedef struct TXrec {
	Ushort d1;
	Ushort status;
	Ushort d2;
	Ushort config;
	Ushort d3;
	Ushort pkt_size;
	Ushort d4;
	Ushort frag_count;
	Ushort d5;
	Ushort ptr0;
	Ushort d6;
	Ushort ptr1;
	Ushort d7;
	Ushort frag_size;
	Ushort d8;
	Ushort link;
	} TXrec;
#else
typedef struct TXrec {
	Ushort status;
	Ushort d1;
	Ushort config;
	Ushort d2;
	Ushort pkt_size;
	Ushort d3;
	Ushort frag_count;
	Ushort d4;
	Ushort ptr0;
	Ushort d5;
	Ushort ptr1;
	Ushort d6;
	Ushort frag_size;
	Ushort d7;
	Ushort link;
	Ushort d8;
	} TXrec;
#endif
#else
typedef struct TXrec {
	Ushort status;
	Ushort config;
	Ushort pkt_size;
	Ushort frag_count;
	Ushort ptr0;
	Ushort ptr1;
	Ushort frag_size;
	Ushort link;
	} TXrec;
#endif

#define k1TXrec(p)	((TXrec *)k02k1(p))
#define getTXptr(p)	((char *)phy2k1(((k1TXrec(p)->ptr1<<16)| \
			k1TXrec(p)->ptr0)))
#define putTXptr(p,q)	(k1TXrec(p)->ptr1=gethi(log2phy(q)), \
			 k1TXrec(p)->ptr0=getlo(log2phy(q)) )
#define getTXlink(p)	((TXrec *)phy2k1(((SONIC_UTDA<<16)| \
			(k1TXrec(p)->link&~1))))
#define putTXlink(p,q,f) (k1TXrec(p)->link=(getlo(log2phy(q))|f))
#define getCTDA()	((TXrec *)phy2k1((SONIC_UTDA<<16)|(SONIC_CTDA&~1)))
#define putCTDA(p)	(SONIC_UTDA=gethi(log2phy(p)), \
			 SONIC_CTDA=getlo(log2phy(p)))

/*************** receive descriptors **************************/
#ifdef SONIC_DW32
#ifdef MIPSEB
typedef struct RXrec {			/* Receive descriptor */
	Ushort d1;
	Ushort status;
	Ushort d2;
	Ushort byte_count;
	Ushort d3;
	Ushort ptr0;	/* ls half off addr */
	Ushort d4;
	Ushort ptr1;	/* ms half off addr */
	Ushort d5;
	Ushort seq_no;
	Ushort d6;
	Ushort link;	/* lsb = 1 = End of List */
	Ushort d7;
	Ushort in_use;	/* 0=full 0xffff=empty */
	} RXrec;
#else
typedef struct RXrec {			/* Receive descriptor */
	Ushort status;
	Ushort d1;
	Ushort byte_count;
	Ushort d2;
	Ushort ptr0;	/* ls half off addr */
	Ushort d3;
	Ushort ptr1;	/* ms half off addr */
	Ushort d4;
	Ushort seq_no;
	Ushort d5;
	Ushort link;	/* lsb = 1 = End of List */
	Ushort d6;
	Ushort in_use;	/* 0=full 0xffff=empty */
	Ushort d7;
	} RXrec;
#endif
#else
typedef struct RXrec {			/* Receive descriptor */
	Ushort status;
	Ushort byte_count;
	Ushort ptr0;	/* ls half off addr */
	Ushort ptr1;	/* ms half off addr */
	Ushort seq_no;
	Ushort link;	/* lsb = 1 = End of List */
	Ushort in_use;	/* 0=full 0xffff=empty */
	} RXrec;
#endif

#define k1RXrec(p)	((RXrec *)k02k1(p))
#define EMPTY   0xffff 	/* in_use=empty, avail for use by SONIC */
#define FULL 	0 	/* in_use=full, ready for processing */
#define getRXptr(p)	((struct ether_header *) \
				phy2k1(((k1RXrec(p)->ptr1<<16)| \
				k1RXrec(p)->ptr0)))
#define getRXlink(p)	((RXrec *)phy2k1(((SONIC_URDA<<16)|k1RXrec(p)->link)))
#define putRXlink(p,q)	(k1RXrec(p)->link=getlo(log2phy(q)))
#define EOL 		0x0001
#define LPKT		(1<<6)	/* status bit */
#define getCRDA()	((RXrec *)phy2k1((SONIC_URDA<<16)|(SONIC_CRDA)))
#define putCRDA(p)	(SONIC_URDA=gethi(log2phy(p)), \
			 SONIC_CRDA=getlo(log2phy(p)))
#define getRXpseq(p)	((k1RXrec(p)->seq_no) & 0xff)
#define getRXrseq(p)	((k1RXrec(p)->seq_no)>>8)

/*************** receive resources **************************/
#ifdef SONIC_DW32
#ifdef MIPSEB
typedef struct RRrec {			/* Receive Resource */
	Ushort d1;
	Ushort ptr0;
	Ushort d2;
	Ushort ptr1;
	Ushort d3;
	Ushort wc0;
	Ushort d4;
	Ushort wc1;
	} RRrec;
#else
typedef struct RRrec {			/* Receive Resource */
	Ushort ptr0;
	Ushort d1;
	Ushort ptr1;
	Ushort d2;
	Ushort wc0;
	Ushort d3;
	Ushort wc1;
	Ushort d4;
	} RRrec;
#endif
#else
typedef struct RRrec {			/* Receive Resource */
	Ushort ptr0;
	Ushort ptr1;
	Ushort wc0;
	Ushort wc1;
	} RRrec;
#endif


#define k1RRrec(p)	((RRrec *)k02k1(p))
#define putRRptr(p,q)	(k1RRrec(p)->ptr1 = gethi(log2phy(q)), \
			 k1RRrec(p)->ptr0 = getlo(log2phy(q)) )
#define getRRptr(p)	((char *)phy2k1(((k1RRrec(p)->ptr1<<16)|(p)->ptr0)))
#define putRRwc(p,n)	(k1RRrec(p)->wc1 = (((Ulong)(n))>>17), \
			 k1RRrec(p)->wc0 = ((((Ushort)(n))>>1) & 0xffff) )

#define getRSA()	((RRrec *)phy2k1((SONIC_URRA<<16)|(SONIC_RSA)))
#define getREA()	((RRrec *)phy2k1((SONIC_URRA<<16)|(SONIC_REA)))
#define getRWP()	((RRrec *)phy2k1((SONIC_URRA<<16)|(SONIC_RWP)))
#define getRRP()	((RRrec *)phy2k1((SONIC_URRA<<16)|(SONIC_RRP)))
#define putRSA(p)	(SONIC_URRA=gethi(log2phy(p)), \
			 SONIC_RSA=getlo(log2phy(p)))
#define putREA(p)	(SONIC_URRA=gethi(log2phy(p)), \
			 SONIC_REA=getlo(log2phy(p)))
#define putRWP(p)	(SONIC_URRA=gethi(log2phy(p)), \
			 SONIC_RWP=getlo(log2phy(p)))
#define putRRP(p)	(SONIC_URRA=gethi(log2phy(p)), \
			 SONIC_RRP=getlo(log2phy(p)))

