/*************************************************************
 * File: include/brecis.h
 * Purpose: Define BRECIS hardware related items
 */

#ifndef _BRECIS_H_
#define _BRECIS_H_

#define DEVICEID "deviceid"
#define FEATURES "features"
#define PHYADDR "phyaddr"

#define CPU_DEVID_MIIPCIDISABLED	(0x04000000)
#define CPU_DEVID_MIIMACMODE		(0x18000000)
#define CPU_DEVID_PCIHOST			(0x00100000)	
#define CPU_DEVID_PCIMUXMODE		(0x00080000)
#define CPU_DEVID_SUBFAMILY			(0x00070000)
#define CPU_DEVID_FAMILY			(0x0000ff00)
#define CPU_DEVID_REVISION			(0x000000ff)

#define	FPGA_IS_ZEUS(revision)  ((revision < 0x80))
#define FPGA_IS_POLO(revision)	((revision >= 0xb0))
#define FPGA_IS_5000(revision)	((revision <= 0x90))
#define FPGA_IS_DUET(revision)	(((revision >= 0xa0) && (revision < 0xb0)))

#define DUET_REVA				0x01

#define POLO_SUBFAMILY_2005		0x01
#define POLO_SUBFAMILY_2007		0x02
#define POLO_SUBFAMILY_2010		0x03
#define POLO_SUBFAMILY_2020		0x05
#define POLO_SUBFAMILY_2100		0x06

#define FAMILY_FPGA		0x0000
#define	FAMILY_ZEUS		0x1000
#define FAMILY_POLO		0x2000
#define FAMILY_DUET		0x4000
#define FAMILY_TRIAD	0x5000

#define MAX_POLO_ENET_INTERFACES	3

/* Use the following to create a string of features of the form:
 * <key><flag><key><flag>...
 * where key can be any of the xxx_KEY
 * and flag can be FEATURE_EXIST, FEATURE_NOEXIST, or special character.
 * <key> can occur more than once in a string such as ethernet where more
 * than one ethernet interface can exist (or not exist) as the case may be.
 */
#define ENET_KEY		'E'
#define ENETTXD_KEY		'e'
#define PCI_KEY			'P'
#define PCIMUX_KEY		'p'
#define SEC_KEY			'S'
#define SPAD_KEY		'D'
#define TDM_KEY			'T'
#define ZSP_KEY			'Z'

#define FEATURE_NOEXIST	'-'
#define FEATURE_EXIST	'+'

#define ENET_MII		'M'
#define ENET_RMII		'R'

#define	ENETTXD_FALLING	'F'
#define ENETTXD_RISING	'R'

#define PCI_HOST		'H'
#define PCI_PERIPHERAL	'P'

#define PCIMUX_FULL		'F'
#define PCIMUX_SINGLE	'S'

#define SEC_DUET		'D'
#define SEC_POLO		'P'
#define SEC_SLOW		'S'
#define SEC_TRIAD		'T'

#define SPAD_POLO		'P'

#define TDM_DUET		'D'		/* DUET TDMs might exist */
#define TDM_POLO		'P'		/* POLO TDMs might exist */
#define TDM_TRIAD		'T'		/* TRIAD TDMs might exist */

#define ZSP_DUET		'D'		/* one DUET zsp engine */
#define ZSP_TRIAD		'T'		/* two TRIAD zsp engines */

/*--------------------------------------------------------------------------*/
/* MIPS Interrupt Cause Register C0_CAUSE				    				*/
/* 15  14  13  12  11 10  9  8						    					*/
/* |   |   |   |   |  |   |  |						    					*/
/* |   |   |   |   |  |   |  --> IP 0 = Software Interrupt 0		    	*/
/* |   |   |   |   |  |   -----> IP 1 = Software Interrupt 1		    	*/
/* |   |   |   |   |  ---------> IP 2 = INT0: MAC 0					    	*/
/* |   |   |   |   ------------> IP 3 = INT1: MAC 1				    		*/
/* |   |   |   ----------------> IP 4 = INT2: VE, FE or SEC Programable     */
/* |   |   --------------------> IP 5 = INT3: VE, FE or SEC Programable	    */
/* |   ------------------------> IP 6 = INT4: SLM				    		*/
/* ----------------------------> IP 7 = INT5: MIPS Timer			        */
/*--------------------------------------------------------------------------*/
#define	SW0_INT_ID	SR_IBIT1			/* Sw0 interrupt ID */
#define SW1_INT_ID	SR_IBIT2			/* SW1 interrupt ID */

#define INT_MAC0_ID	SR_IBIT3			/* MAC0 Interrupt ID */
#define INT_MAC1_ID	SR_IBIT4			/* MAC1 Interrupt ID */
#define INT_VFS_ID2	SR_IBIT5			/* VOICE, FE OR SEC  Interrupt ID Option 2*/
#define INT_VFS_ID3	SR_IBIT6		   	/* VOICE, FE OR SEC  Interrupt ID Option 3*/
#define INT_SLM_ID  SR_IBIT7			/* SLM  Interrupt ID */
#define INT_COU_ID	SR_IBIT8			/* MIPS INTERNAL TIMER INTERRUPT */

		
#define	MEM_CTL_BASE	0xB7F00000	/* Base address to memory contlr   */
#define MEM_CNFG_REG	(MEM_CTL_BASE + 0x00)	/* memory configuration reg */
#define MEM_RFT_REG		(MEM_CTL_BASE + 0x04)	/* refresh timer reg */
#define MEM_RFC_REG		(MEM_CTL_BASE + 0x08)	/* refresh counter reg */
#define MEM_TMGM_REG	(MEM_CTL_BASE + 0x0C)	/* Timing param reg */
#define MEM_INIT_REG	(MEM_CTL_BASE + 0x10)	/* initialization reg */
#define SDRAMADDRESS	0xa0000000		/* first address non-cached sdram */
#define SDRAM32MBYTE	0x2000000		/* 32 mbyte */
#define SDRAM8MBYTE		0x0800000		/*  8 mbyte */
		
#define	SREG_BASE	0xBC000000	/* Base address to SLM Block       */
#ifdef LANGUAGE_ASSEMBLY
#define	DEV_ID_REG	(SREG_BASE + 0x000)  /* Device-ID Register         */
#define	FWR_ID_REG	(SREG_BASE + 0x004)  /* Firmware-ID Register       */
#define	SYS_RST_REG	(SREG_BASE + 0x010)  /* system Reset Status Register RO */
#define	RST_SET_REG	(SREG_BASE + 0x014)  /* System set Reset Register    WO */
#define	RST_CLR_REG	(SREG_BASE + 0x018)  /* System Clear Reset Register  WO */
#define	PCM0_CLK_REG	(SREG_BASE + 0x020)  /* PCM0 Clock Generator       */
#define	PCM1_CLK_REG	(SREG_BASE + 0x024)  /* PCM1 Clock Generator       */
#define	PCM2_CLK_REG	(SREG_BASE + 0x028)  /* PCM2 Clock Generator       */
#define PCM3_CLK_REG	(SREG_BASE + 0x02C)  /* PCM3 Clock Generator	   */
#define	PLL1_DIV_REG	(SREG_BASE + 0x02C)  /* PLL1 Divider Value         */
#define	PLL0_DIV_REG	(SREG_BASE + 0x030)  /* PLL0 Divider Value         */
#define	MIPS_CLK_REG	(SREG_BASE + 0x034)  /* MIPS Clock Generator       */
#define	VE_CLK_REG	(SREG_BASE + 0x038)  /* Voice Engine Clock Gen     */
#define	FE_CLK_REG	(SREG_BASE + 0x03C)  /* Framer Engine Clock Gen    */
#define SEC_CLK_REG (SREG_BASE + 0x3C)		/* Polo SEC clock engine */
#define	DVB_CLK_REG	(SREG_BASE + 0x040)  /* DV-Bus Clock Generator     */
#define	SMAC_CLK_REG	(SREG_BASE + 0x044)  /* SEC & MAC Clock Generator  */
#define	PERF_CLK_REG	(SREG_BASE + 0x048)  /* Peripheral & ADPCM Clk Gen */
#define	INT_STA_REG	(SREG_BASE + 0x070)  /* Interrupt Status Register  */
#define	INT_MSK_REG	(SREG_BASE + 0x074)  /* Interrupt Mask Register    */
#define ELB_CLKWAIT_REG (SREG_BASE + 0x0FC)	/* ELB Clock/Wait Register */
#else
#define	SYS_RST_REG	((volatile unsigned int *) (SREG_BASE + 0x010))
#define	RST_SET_REG	((volatile unsigned int *) (SREG_BASE + 0x014))
#define	RST_CLR_REG	((volatile unsigned int *) (SREG_BASE + 0x018))
/* Queue Flush Register 1 */
#define QFLUSH_REG_1 ((volatile unsigned int *) (MEM_CTL_BASE + 0x40000))
#define QFLUSH_PCI_BIT	0x02			/* if set, flush PCI data queue */
#define QFLUSH_SCR_BIT	0x01			/* if set, flush scratch RAM queue */
#endif

/*****************************************************************************/
/* System reset register definetion											 */
/* System Reset Status register												 */
/*																			 */
/*  31-------10 9  8  7  6  5  4  3  2  1  0								 */
/*Reserved      |  |  |  |  |  |  |  |  |  |								 */
/*	           |  |  |  |  |  |  |  |  |  ----> Global Reset				 */
/* 	          |  |  |  |  |  |  |  |  --> Mips Processor SLM And SDRAM		 */
/* 	         |  |  |  |  |  |  |  --> Voice Engine DMA						 */
/* 	        |  |  |  |  |  |  --> Voice Engine Processor					 */
/* 	       |  |  |  |  |  --> Framer Engine DMA								 */
/* 	      |  |  |  |  ----> Framer Engine Processor							 */
/* 	     |  |  |  --------> Ethernet 0										 */
/* 	    |  |  ------------> Ethernet 1										 */
/*     |   ---------------> Security engine									 */
/* 	   -------------------> Peripheral Block								 */
/*****************************************************************************/
#define	GLOBAL_RST	0x1						/* Global Reset */
#define MIPS_RST	0x2						/* MIPS reset   */
#define VE_DMA_RST	0x4						/* Voice Engine DMA reset  */
#define VE_CPU_RST	0x8						/* Voice Processor Reset   */
#define FE_DMA_RST	0x10					/* Framer Engine DMA reset */
#define FE_CPU_RST	0x20					/* Framer Engine Processor */
#define MAC0_RST	0x40					/* Ethernet 0 Reset */
#define MAC1_RST	0x80					/* Ethernet 1 Reset */
#define SE_RST		0x100					/* Security Engine Reset */
#define PER_RST		0x200					/* Peripheral Block Reset */
#define MAC2_RST	0x400					/* Ethernet 2 Reset */
		
/************************************/
/* define VERSION ID of each Target */
/************************************/

#define CPU_DEVID_MASK			0xff00
#define CPU_DEVID_FAM_MASK		0xf000
#define CPU_MSP_5000			0x50
#define CPU_MSP_4000			0x40
#define CPU_MSP_3000			0x30
#define CPU_MSP_2000			0x20
#define CPU_MSP_1000			0x10
#define CPU_MSP_FPGA_2000		0x02
#define CPU_MSP_5000_FAMILY		5000
#define	CPU_MSP_4000_FAMILY		4000
#define CPU_MSP_3000_FAMILY		3000
#define	CPU_MSP_2000_FAMILY		2000
#define CPU_MSP_1000_FAMILY		1000

#define MSP5_RESET_VAL  (MAC0_RST | MAC1_RST | VE_DMA_RST       \
                        | VE_CPU_RST | FE_DMA_RST | FE_CPU_RST  \
                        | SE_RST | PER_RST)

#define MSP4_RESET_VAL  (MAC0_RST | MAC1_RST | MAC2_RST | VE_DMA_RST    \
                        | VE_CPU_RST | FE_DMA_RST | FE_CPU_RST          \
                        | SE_RST | PER_RST)
#define	MSP1_RESET_VAL	(MAC0_RST | MAC2_RST | PER_RST)

#define MSP_CRESET_VAL  (PER_RST)


#define ELB_BASE	0xBC000080
#define	CS0_CNFG_REG	(ELB_BASE + 0x00)  /* ELB CS0 Configuration Reg    */
#define	CS0_BA_REG		(ELB_BASE + 0x04)  /* ELB CS0 Base Address Reg     */
#define	CS0_MSK_REG		(ELB_BASE + 0x08)  /* ELB CS0 Mask Register        */

#define	CS1_CNFG_REG	(ELB_BASE + 0x10)  /* ELB CS1 Configuration Reg    */
#define	CS1_BA_REG		(ELB_BASE + 0x14)  /* ELB CS1 Base Address Reg     */
#define	CS1_MSK_REG		(ELB_BASE + 0x18)  /* ELB CS1 Mask Register        */

#define	CS2_CNFG_REG	(ELB_BASE + 0x20)  /* ELB CS2 Configuration Reg    */
#define	CS2_BA_REG		(ELB_BASE + 0x24)  /* ELB CS2 Base Address Reg     */
#define	CS2_MSK_REG		(ELB_BASE + 0x28)  /* ELB CS2 Mask Register        */

#define	CS3_CNFG_REG	(ELB_BASE + 0x30)  /* ELB CS3 Configuration Reg    */
#define	CS3_BA_REG		(ELB_BASE + 0x34)  /* ELB CS3 Base Address Reg     */
#define	CS3_MSK_REG		(ELB_BASE + 0x38)  /* ELB CS3 Mask Register        */

#define	CS4_CNFG_REG	(ELB_BASE + 0x40)  /* ELB CS4 Configuration Reg    */
#define	CS4_BA_REG		(ELB_BASE + 0x44)  /* ELB CS4 Base Address Reg     */
#define	CS4_MSK_REG		(ELB_BASE + 0x48)  /* ELB CS4 Mask Register        */

#define	CS5_CNFG_REG	(ELB_BASE + 0x50)  /* ELB CS5 Configuration Reg    */
#define	CS5_BA_REG		(ELB_BASE + 0x54)  /* ELB CS5 Base Address Reg     */
#define	CS5_MSK_REG		(ELB_BASE + 0x58)  /* ELB CS5 Mask Register        */

#define	CS6_CNFG_REG	(ELB_BASE + 0x60)  /* ELB CS6 Configuration Reg    */
#define	CS6_BA_REG		(ELB_BASE + 0x64)  /* ELB CS6 Base Address Reg     */
#define	CS6_MSK_REG		(ELB_BASE + 0x68)  /* ELB CS6 Mask Register        */

#define	CS7_CNFG_REG	(ELB_BASE + 0x70)  /* ELB CS7 Configuration Reg    */
#define	CS7_BA_REG		(ELB_BASE + 0x74)  /* ELB CS7 Base Address Reg     */
#define	CS7_MSK_REG		(ELB_BASE + 0x78)  /* ELB CS7 Mask Register        */
		
#define		CS_SIZE_8MEG		0x0
#define 	CS_SIZE_4MEG		0x40
#define 	CS_SIZE_2MEG		0x60
#define 	CS_SIZE_1MEG		0x70
#define 	CS_SIZE_512K		0x78
#define 	CS_SIZE_256K		0x7c
#define 	CS_SIZE_128K		0x7e
#define		CS_SIZE_64K			0x7f

#define CS0_BASE_ADD_VAL	0x180			/* base 0x1f800000 */		
#define CS0_CNFG_REG_VAL	0x765			/* Time = Max, Size 8bits, Mode Intel */
#define CS0_ADD_SPACE		CS_SIZE_8MEG	/* Last 8 MegBytes */

#define CS1_BASE_ADD_VAL	0x0				/* base 0x1E000000 */		
#define CS1_CNFG_REG_VAL	0x765			/* Time = Max, Size 8bits, Mode Intel */
#ifdef _FLASH8MB
#define CS1_ADD_SPACE		CS_SIZE_8MEG	/* Address Space 8 MegBytes */
#else
#define CS1_ADD_SPACE		CS_SIZE_4MEG	/* Address Space 4 MegBytes */
#endif

#define CS2_BASE_ADD_VAL	0x40			/* base 0x1E400000 */		
#define CS2_CNFG_REG_VAL	0x5				/* Size 8bits, Mode Intel */
#define CS2_ADD_SPACE		CS_SIZE_4MEG	/* Address Space 4 MegBytes */

#define CS3_BASE_ADD_VAL	0x80			/* base 0x1E800000 */		
#define CS3_CNFG_REG_VAL	0x5				/* Size 8bits, Mode Intel */
#define CS3_ADD_SPACE		CS_SIZE_4MEG	/* Address Space 4 MegBytes */

#define CS4_BASE_ADD_VAL	0xc0			/* base 0x1EC00000 */		
#define CS4_CNFG_REG_VAL	0x765			/* Time = Max, Size 8bits, Mode Intel */
#define CS4_ADD_SPACE		CS_SIZE_4MEG	/* Address Space 4 MegBytes */

#define CS5_BASE_ADD_VAL	0x100			/* base 0x1F000000 */		
#define CS5_CNFG_REG_VAL	0x5				/* Size 8bits, Mode Intel */
#define CS5_ADD_SPACE		CS_SIZE_4MEG	/* Address Space 4 MegBytes */

#define CS6_BASE_ADD_VAL	0x140			/* base 0x1F400000 */		
#define CS6_CNFG_REG_VAL	0x5				/* Size 8bits, Mode Intel */
#define CS6_ADD_SPACE		CS_SIZE_2MEG	/* Address Space 2 MegBytes */

#define CS7_BASE_ADD_VAL	0x160			/* base 0x1F600000 */		
#define CS7_CNFG_REG_VAL	0x5				/* Size 8bits, Mode Intel */
#define CS7_ADD_SPACE		CS_SIZE_2MEG	/* Address Space 2 MegBytes	*/

#define	PER_BASE			0xB8400000		
#ifdef LANGUAGE_ASSEMBLY
#define	GPIO_CFG2_REG		(PER_BASE + 0xE8)
#define GPIO_CFG3_REG		(PER_BASE + 0x170)	
#else
#define GPIO_CFG7_REG		((volatile unsigned int *)(PER_BASE + 0x198))
#define POLO_MAC2_ENABLE	(0x20000000)		
#endif
#define ICACHESIZE		16384
#define	DCACHESIZE		16384
#define	ICACHELINESIZE	16		
#define	DCACHELINESIZE	16		
#define icache_indinv(r)	.word	(0xbc000000|((r)<<21))
#define icache_fetchandlock(r)	.word	(0xbc1c0000|((r)<<21))
#define dcache_indinv(r)	.word	(0xbc010000|((r)<<21))
#define dcache_hitinv(r)	.word	(0xbc150000|((r)<<21))

/*************** Security Engine stuff ******************************/
#define	SEC_BASE	0xB8800000	/* Base address to Security Engine */
#ifndef LANGUAGE_ASSEMBLY
#define DES_IFS_REG   (volatile unsigned int *) (SEC_BASE + 0x03C)
#endif

/*************** Ethernet stuff ******************************/

#define MSP_ETH_MAX_UNITS	3
#define MAX_PKT_SIZE		1532
#define	MAX_BUFF_SIZE		0x6F0
#if (MAX_PKT_SIZE > MAX_BUFF_SIZE)
#error "Code allocating space "
#endif

#define	MAC0_BASE	0xB8600000	/* Base address to MAC0 Controller */
#define	MAC1_BASE	0xB8700000	/* Base address to MAC1 Controller */
#define	MAC2_BASE	0xB8900000	/* Base address to MAC2 Controller */

/*--------------------------------------------------------------------------+
| Ethernet MAC-0 and MAC-1 Registers Addresses
+--------------------------------------------------------------------------*/

#ifndef LANGUAGE_ASSEMBLY

typedef volatile unsigned long MACREG;

typedef struct {
	MACREG	DMA_CTL;		/* DMA Control                     */
	MACREG	TXFRMPTR;		/* Transmit Frame Pointer          */
	MACREG	TXTHRSH;		/* Transmit Threshold              */
	MACREG	TXPOLLCTR;		/* Transmit Polling Counter        */

	MACREG	BLFRMPTR;		/* Buffer List Frame Pointer       */
	MACREG	RXFRAGSIZE;		/* Receive Fragment Size           */
	MACREG	INT_EN;			/* Interrupt Enable                */
	MACREG	FDA_BAS;		/* Free Descriptor Area Base       */

	MACREG	FDA_LIM;		/* Free Descriptor Area Limit      */
	MACREG	INT_SRC;		/* Interrupt Source                */
	MACREG	RM_BBR;			/* RMON Burst Base Register        */
	MACREG	RM_BCTL;		/* RMON Burst Control              */

	MACREG	PAUSECNT;		/* Pause Count                     */
	MACREG	REMPAUCNT;		/* Remote Pause Count              */
	MACREG	TXCONFRMSTAT;		/* Transmit Control Frame Status   */
	MACREG	unused1;

	MACREG MAC_CTL;			/* MAC Control                     */
	MACREG ARC_CTL;			/* ARC Control                     */
	MACREG TX_CTL;			/* Transmit Control                */
	MACREG TX_STAT;			/* Transmit Status                 */

	MACREG RX_CTL;			/* Receive Control                 */
	MACREG RX_STAT;			/* Receive Status                  */
	MACREG MD_DATA;			/* Station Management Data         */
	MACREG MD_CA;			/* Station Management Control&Addr */

	MACREG ARC_ADR;			/* ARC Address                     */
	MACREG ARC_DATA;		/* ARC Data                        */
	MACREG ARC_ENA;			/* ARC Enable                      */
	MACREG PROM_CTL;		/* PROM Control                    */

	MACREG PROM_DATA;		/* PROM data                       */
	MACREG unused2[2];
	MACREG MISS_CNT;		/* Missed Error Count              */

	MACREG CNTDATA;			/* Count Data Register             */
	MACREG CNTACC;			/* Count Access Register           */
	MACREG TXRMINTEN;		/* Transmit RMON Interrupt Enable  */
	MACREG RXRMINTEN;		/* Receive RMON Interrupt Enable   */

	MACREG TXRMINTSTAT;		/* Transmit RMON Interrupt Status  */
	MACREG RXRMINTSTAT;		/* Receive RMON Interrupt Status   */
	MACREG unused3[2];

	MACREG unused4[12];

	MACREG MAC_ID_REG;		/* Version #			   */
	MACREG MAC_BRCTRL_REG;	/* Bridge Control Register	   */

} MAC_BASE;

typedef MAC_BASE	PHY_BASE;

#endif

/*
 * The hardware supports only upto 28 BDs.
 * CAUTION: DO NOT CHANGE THIS VALUE !!
 */
#define HARDWARE_MAX_BD		28

#define MAX_BD_PER_FRAME	2	/* For scatter gather mechanism, several buffers could make up a packet. Keep it as even number to make the FDBD at 16byte boundry */
#if (MAX_BD_PER_FRAME > HARDWARE_MAX_BD)
#error "HARDWARE_MAX_BD can't be greater than what hardware can support !!!"
#endif

#define BD_PER_TX_FRAME		1	/* Number of BDs in the Transmit Queue per frame. */

#define TX_FD_COUNT		2
#define RX_BD_COUNT		6			
#define RX_FDA_COUNT	20		

/*
 *	Interrupt Source Register (0x24). 
 */
#define INTR_BLEx		0x00001000	/* BL exhausted during reception.  */

/*
 *	Receive Status (Rx_Stat 0x54) Register. Triad - Ethernet MAC... Page 4-33
 */
#define RCVD_LEN_ERR	0x00000010	/* Error Pkt Received.*/
#define RCVD_CTL_FRM	0x00000020	/* Error Pkt Received.*/
#define RCVD_GOOD_PKT	0x00004000	/* Good  Pkt Received with no error.*/
#define RCVD_ALIGN_ERR	0x00000100	/* Error Pkt Received.*/
#define RCVD_CRC_ERR	0x00000200	/* Error Pkt Received.*/
#define RCVD_OVRFLW_ERR	0x00000400	/* Error Pkt Received.*/
#define RCVD_LONG_ERR	0x00000800	/* Error Pkt Received.*/
#define RCVD_PAR_ERR	0x00002000	/* Error Pkt Received.*/

#define RCVD_OK_PKT		(RCVD_GOOD_PKT | RCVD_LEN_ERR)

/*
 *	MD_CA Register (0x5c)
 */
/*  Addr                     bits 0 to 4   */
/*  Phy Address              bits 5 to 9   */
#define MD_CA_PhyShift	(5)
#define MD_CA_Wr		0x0400	/* Set to Write - clear for read */
#define MD_CA_BUSY_BIT 	0x0800
#define MD_CA_PreSup	0x1000	/* Suppress Preamble sent to Phy */
#define MD_MAX_PHY		32		/* maximum number of PHY per MII */

/*
 * MAC_BRCTRL_REG (0xd4)
 */
#define RMII_Reset	0x04	/* RMII Reset */
#define RMII_10MBIT	0x02	/* 1 if 10 Mbs, 0 if 100 Mbs */
#define TxDClockRising	0x01	/* 0 if TxD generated off falling edge,
				 * 1 if generated off rising edge of Tx-CLK */

#define MSPEth_100MBS		100000000		/* bits per sec	*/
#define MSPEth_10MBS	 	 10000000		/* bits per sec	*/

/*
 * Frame Descriptor, Buffer Descriptor related parameters.
 */
#define FD_SYS			0x04		/* Offset for FDSystem in FD. */
#define FD_STAT			0x08		/* Offset for FDStat in FD. */
#define FD_CTL_LEN		0x0C		/* Offset for (FDCtl|FDLength) in FD. */
#define FD_BD_Buff		0x10		/* Offset for (BDs BuffData) in FD+BD. */
#define	FDA_1BLK_SIZE	16			/* Size of one memory block in FDA area */

#define MAC_OWNS_FD		0x8000		/* Controller (MAC) owns the Frame Descriptor.*/
#define MAC_OWNS_BD		0x80		/* Controller (MAC) owns the Buffer Descriptor.*/
#define MAC_OWNS_FDA	0x80000000	/* Controller (MAC) owns the Frame Descriptor.*/

#define SW_OWNS_FD		0x0000		/* Software owns the Frame Descriptor.*/
#define SW_OWNS_BD		0x00		/* Software owns the Buffer Descriptor.*/

#define FDCTL_BD_COUNT	0x001F		/* Bit 0-4 of the FDCtl field of the FD represents # of BDs */


#define ENA_TX_INT		0x00004000	/* Enable Transmit Interrupt, OR */
#define ENA_RX_INT		0x00005200	/* Enable Receive Interrupt for Good Packet and LenErr packet, OR  */
#define DISA_TX_INT		0xFFFFBFFF	/* Disable Transmit Interrupt, for AND */
#define DISA_RX_INT		0xFFFFBFFF	/* Disable Receive Interrupt, for AND */



#define ENA_FdaFul_INT	0x00000001	/* Enable FDA Exhaust Interrupt, OR */
#define DISA_FdaFul_INT	0xFFFFFFFE	/* Disable FDA Exhaust Interrupt, OR */
#define ENA_BlFul_INT	0x00000002	/* Enable BL Exhaust Interrupt, OR */
#define DISA_BlFul_INT	0xFFFFFFFD	/* Disable BL Exhaust Interrupt, OR */


#define ENA_RX_VLAN		0x00000004  /* Enable Receiving of VLAN Packet(1522 bytes), OR operation */
#define DISA_RX_VLAN	0xFFFFFFFB  /* Disable Receiving of VLAN Packet(1522 bytes), AND operation */
#define MAC_CTL_FullDuplex	0x00000008	/* set if MII_High for Full Duplex */
#define SPEED_AUTO		0xFFFFFF9F  /* ,Set the MAC speed to Automatic mode. AND operation */
#define SPEED_10		0xFFFFFFBF  /* ,Set the MAC speed to Automatic mode. OR operation */


/* PHY Register Addresses */  

#define PHYG_CONTROL							0
#define PHYG_STATUS_REG_ONE						1
#define PHYG_PHY_ID_REG_ONE						2
#define PHYG_PHY_ID_REG_TWO						3
#define PHYG_AUTO_NEG_ADVERTISING				4
#define PHYG_AUTO_NEG_LINK_PARTNER_ABILITY		5
#define PHYG_AUTO_NEG_EXPANSION					6
#define PHYG_AUTO_NEG_NEXT_PAGE_TX				7
#define PHYG_AUTO_NEG_LINK_PARTNER_RX_NEXT		8


/* Bit Defines for PHYG_CONTROL  			*/
/* ---------------------------------------- */
#define PHYGb_CONTROL_RESET						0x8000
#define PHYGb_CONTROL_LOOPBACK					0x4000
#define PHYGb_CONTROL_SELECT_SPEED				0x2000
#define PHYGb_CONTROL_AUTONEG					0x1000
#define PHYGb_CONTROL_POWERDOWN					0x0800
#define PHYGb_CONTROL_ISOLATE					0x0400
#define PHYGb_CONTROL_RESTART_AUTONEG			0x0200
#define PHYGb_CONTROL_DUPLEXMODE				0x0100
#define PHYGb_CONTROL_COLLISION_TEST			0x0080
#define PHYGb_CONTROL_SPEED_10_HALF				0x0000
#define PHYGb_CONTROL_SPEED_10_FULL				(PHYGb_CONTROL_DUPLEXMODE)
#define PHYGb_CONTROL_SPEED_100_HALF			(PHYGb_CONTROL_SELECT_SPEED )
#define PHYGb_CONTROL_SPEED_100_FULL			(PHYGb_CONTROL_SELECT_SPEED | PHYGb_CONTROL_DUPLEXMODE)
 
/* Bit Defines for PHYG_STATUS_REG_ONE 	*/
/* -------------------------------------	*/
#define PHYGb_STATUS_REG_ONE_100BASE_T4_OK		0x8000
#define PHYGb_STATUS_REG_ONE_100BASE_TX_FULL	0x4000
#define PHYGb_STATUS_REG_ONE_100BASE_TX_HALF	0x2000
#define PHYGb_STATUS_REG_ONE_10BASE_T_FULL		0x1000
#define PHYGb_STATUS_REG_ONE_10BASE_T_HALF		0x0800
	/* not used									0x0400	*/
	/* not used									0x0200	*/
	/* not used									0x0100	*/
	/* not used									0x0080	*/
#define PHYGb_STATUS_REG_ONE_NO_PREAMBLE		0x0040
#define PHYGb_STATUS_REG_ONE_AUTO_NEG_DONE		0x0020
#define PHYGb_STATUS_REG_ONE_REMOTE_FAULT		0x0010
#define PHYGb_STATUS_REG_ONE_AUTO_NEG_ABILITY	0x0008
#define PHYGb_STATUS_REG_ONE_LINK_STATUS		0x0004
#define PHYGb_STATUS_REG_ONE_JABBER_DETECT		0x0002
#define PHYGb_STATUS_REG_ONE_EXT_CAPABILITY		0x0001

#define PHYGb_MEDIAMASK							  \
		(PHYGb_STATUS_REG_ONE_100BASE_T4_OK		| \
		PHYGb_STATUS_REG_ONE_100BASE_TX_FULL	| \
		PHYGb_STATUS_REG_ONE_100BASE_TX_HALF	| \
		PHYGb_STATUS_REG_ONE_10BASE_T_FULL		| \
		PHYGb_STATUS_REG_ONE_10BASE_T_HALF		| \
		PHYGb_STATUS_REG_ONE_AUTO_NEG_ABILITY)

/* Bit Defines for PHYG_AUTO_NEG_ADVERTISING 	*/
/* -------------------------------------------- */
#define PHYGb_AUTO_NEG_ADVERT_NEXT_PAGE			0x8000
	/* not used									0x4000	*/
#define PHYGb_AUTO_NEG_ADVERT_REMOTE_FAULT		0x2000
	/* not used									0x1000	*/
	/* not used									0x0800	*/
#define PHYGb_AUTO_NEG_ADVERT_REMOTE_PAUSE		0x0400
#define PHYGb_AUTO_NEG_ADVERT_100BASE_T4		0x0200
#define PHYGb_AUTO_NEG_ADVERT_100BASE_TX_FULL	0x0100
#define PHYGb_AUTO_NEG_ADVERT_100BASE_TX		0x0080
#define PHYGb_AUTO_NEG_ADVERT_10BASE_T_FULL		0x0040
#define PHYGb_AUTO_NEG_ADVERT_10BASE_T			0x0020
	/* selector field 			bits 0 - 4				*/

/* Bit Defines for PHYG_AUTO_NEG_LINK_PARTNER_ABILITY	*/
/* ---------------------------------------------------- */
#define PHYGb_AUTO_NEG_LINK_NEXT_PAGE			0x8000
#define PHYGb_AUTO_NEG_LINK_ACKNOWLEDGE			0x4000
#define PHYGb_AUTO_NEG_LINK_REMOTE_FAULT		0x2000
	/* PAUSE field 			bits 10 - 11			*/
#define PHYGb_AUTO_NET_LINK_100BASE_T4			0x0200
#define PHYGb_AUTO_NET_LINK_100BASE_TX_FULL		0x0100
#define PHYGb_AUTO_NET_LINK_100BASE_TX			0x0080
#define PHYGb_AUTO_NEG_LINK_10BASE_T_FULL		0x0040
#define PHYGb_AUTO_NEG_LINK_10BASE_T			0x0020
	/* selector field 			bits 0 - 4				*/


/* Operation Delays */  
/* ---------------- */

#define MD_REGISTER_UDELAY			(1)	/* 1 us delay */
#define PHYG_UDELAY_FOR_RESPONSE		(50)	/* 50 us delay max */
#define PHYG_MAX_UDELAY_FOR_AUT_NEG		(3)	/* 3 us delay max */
#define PHYG_MAX_UDELAY_FOR_EXIT_RESET		(5)	/* 5 us delay */
#define PHYG_MAX_MDELAY_FOR_TO_AUTONEG		(300)	/* 300 ms */
#define PHYG_MAX_MDELAY_FOR_EXIT_RESET		(300)	/* 300 ms */

#define MAX_TIME_TO_WAIT_FOR_LINK		(10)	/* seconds */

#define MAC0_RST	0x40			/* Ethernet 0 Reset */
#define MAC1_RST	0x80			/* Ethernet 1 Reset */
#define MAC2_RST	0x400			/* Ethernet 2 Reset */

#ifndef LANGUAGE_ASSEMBLY
/*
 * For the MAC Mode Type.
 */
#define SPEEDDUPLEX "speedduplex"
#define STRING10HALFANY		"10h"
#define STRING10FULLANY		"10f"
#define STRING100HALFANY	"100h"
#define STRING100FULLANY	"100f"
#define STRING10HALFSWITCH	"10hs"
#define STRING10FULLSWITCH	"10fs"
#define STRING100HALFSWITCH	"100hs"
#define STRING100FULLSWITCH	"100fs"
#define STRINGAUTO		"auto"

typedef enum _eth_mode
{
	ETH_AUTO			=1,
	ETH_10Mbps_HALF		=2,
	ETH_10Mbps_FULL		=3,
	ETH_100Mbps_HALF	=4,
	ETH_100Mbps_FULL	=5		
} ETH_MODE;

/*
 * Frame Descriptors for Ethernet MAC of Msp
 */
typedef struct _FD
	{
		struct _FD	* FDNext;
		Ulong		FDSystem;
		Ulong		FDStat;
		Ushort		FDCtl;
		Ushort		FDLength;
	} FD;


/* 
 * Buffer Descriptor for Ethernet MAC of Msp
 */
typedef struct
    {
		Uchar		* BuffData;
		Uchar		BDCtl;
		Uchar		BDStat;
		Ushort		BuffLength;
	} BD;



/* 
 * Structure of FD and BDs to be used Tx and FDA of Rx.
 */
typedef struct
    {
		FD		FD;
		BD		BD[MAX_BD_PER_FRAME];
	} FDBD;

#define FDAFDBD FDBD


/* 
 * Structure of FD and BDs to be used in the BL Area for the Rx.
 */
typedef struct
    {
		FD		FD;
		BD		BD[RX_BD_COUNT];
	} BLFDBD;

#endif

#define MAC_DPX_MODE_HALF		0
#define MAC_DPX_MODE_FULL		1

#endif /* _BRECIS_H_ */

