/******************************************************************/
/* Copyright (c) 2001 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

/* mspeth.c*/

/************************************************************************
 * modification history
 * --------------------
 *
 *	01/22/01		Jey	K Surier		Original.
 *	11/04/01		Jey K Surier		Added the Structres for FDs, BDs.
 *
 *
 ************************************************************************/

#include <string.h>
#include <utypes.h>
#include <mips.h>

#define VERBOSE
#ifdef VERBOSE
#include <mon.h>
#endif
#ifdef CHECKS_ON
#include <assert.h>
#else
#define assert(x)
#endif

#define	OK		0
#define	ERROR	1

#define MSP_ETHERINIT	1
#define MSP_ETHERXMIT	2
#define MSP_ETHERRECV	3
#define MSP_ETHERRCMP	4

#define MSP_ETH_MAX_UNITS	2
#define MAX_PKT_SIZE		1532
#define	MAX_BUFF_SIZE		0x6F0
#if (MAX_PKT_SIZE > MAX_BUFF_SIZE)
#error "Code allocating space "
#endif

#define	MAC0_BASE	0xB8600000	/* Base address to MAC0 Controller */
#define	MAC1_BASE	0xB8700000	/* Base address to MAC1 Controller */
#define	MAC2_BASE	0xB8900000	/* Base address to MAC2 Controller */

/*--------------------------------------------------------------------------+
| Ethernet MAC-0 and MAC-1 Registers Addresses
+--------------------------------------------------------------------------*/
#define DMA_CTL		0x00		/* DMA Control                     */
#define TXFRMPTR	0x04		/* Transmit Frame Pointer          */
#define TXTHRSH		0x08		/* Transmit Threshold              */
#define TXPOLLCTR	0x0c		/* Transmit Polling Counter        */
#define BLFRMPTR	0x10		/* Buffer List Frame Pointer       */
#define RXFRAGSIZE	0x14		/* Receive Fragment Size           */
#define INT_EN		0x18		/* Interrupt Enable                */
#define FDA_BAS		0x1c		/* Free Descriptor Area Base       */
#define FDA_LIM		0x20		/* Free Descriptor Area Limit      */
#define INT_SRC		0x24		/* Interrupt Source                */

#define RM_BBR		0x28		/* RMON Burst Base Register        */
#define RM_BCTL		0x2c		/* RMON Burst Control              */

/*
 * MAC Control and Status Register
 */
#define PAUSECNT	0x30		/* Pause Count                     */
#define REMPAUCNT	0x34		/* Remote Pause Count              */
#define TXCONFRMSTAT	0x38	/* Transmit Control Frame Status   */

#define MAC_CTL		0x40		/* MAC Control                     */
#define ARC_CTL		0x44		/* ARC Control                     */
#define TX_CTL		0x48		/* Transmit Control                */
#define TX_STAT		0x4c		/* Transmit Status                 */
#define RX_CTL		0x50		/* Receive Control                 */
#define RX_STAT		0x54		/* Receive Status                  */
#define MD_DATA		0x58		/* Station Management Data         */
#define MD_CA		0x5c		/* Station Management Control&Addr */
#define ARC_ADR		0x60		/* ARC Address                     */
#define ARC_DATA	0x64		/* ARC Data                        */
#define ARC_ENA		0x68		/* ARC Enable                      */
#define PROM_CTL	0x6c		/* PROM Control                    */
#define PROM_DATA	0x70		/* PROM data                       */
#define MISS_CNT	0x7c		/* Missed Error Count              */

#define CNTDATA		0x80		/* Count Data Register             */
#define CNTACC		0x84		/* Count Access Register           */
#define TXRMINTEN	0x88		/* Transmit RMON Interrupt Enable  */
#define RXRMINTEN	0x8c		/* Receive RMON Interrupt Enable   */
#define TXRMINTSTAT	0x90		/* Transmit RMON Interrupt Status  */
#define RXRMINTSTAT	0x94		/* Receive RMON Interrupt Status   */
#define MAC_ID_REG	0xd0		/* Version # */

/*
 * The hardware supports only upto 28 BDs.
 * CAUTION: DO NOT CHANGE THIS VALUE !!
 */
#define HARDWARE_MAX_BD		28

#define MAX_BD_PER_FRAME	2	/* For scatter gather mechanism, several buffers could make up a packet. Keep it as even number to make the FDBD at 16byte boundry */
#if (MAX_BD_PER_FRAME > HARDWARE_MAX_BD)
#error "HARDWARE_MAX_BD can't be greater than what hardware can support !!!"
#endif

#define BD_PER_TX_FRAME		1	/* Number of BDs in the Transmit Queue per frame. */

#define TX_FD_COUNT		6
#define RX_BD_COUNT		10			
#define RX_FDA_COUNT	16		

/*************** Ethernet stuff ******************************/

/*
 *	Receive Status (Rx_Stat 0x54) Register. Triad - Ethernet MAC... Page 4-33
 */
#define RCVD_LEN_ERR	0x00000010	/* Error Pkt Received.*/
#define RCVD_CTL_FRM	0x00000020	/* Error Pkt Received.*/
#define RCVD_GOOD_PKT	0x00004000	/* Good  Pkt Received with no error.*/
#define RCVD_ALIGN_ERR	0x00000100	/* Error Pkt Received.*/
#define RCVD_CRC_ERR	0x00000200	/* Error Pkt Received.*/
#define RCVD_OVRFLW_ERR	0x00000400	/* Error Pkt Received.*/
#define RCVD_LONG_ERR	0x00000800	/* Error Pkt Received.*/
#define RCVD_PAR_ERR	0x00002000	/* Error Pkt Received.*/

#define RCVD_OK_PKT		(RCVD_GOOD_PKT | RCVD_LEN_ERR)

/*
 *	MD_CA Register
 */
#define MD_CA_BUSY_BIT 0x0800

/*
 * Frame Descriptor, Buffer Descriptor related parameters.
 */
#define FD_SYS			0x04		/* Offset for FDSystem in FD. */
#define FD_STAT			0x08		/* Offset for FDStat in FD. */
#define FD_CTL_LEN		0x0C		/* Offset for (FDCtl|FDLength) in FD. */
#define FD_BD_Buff		0x10		/* Offset for (BDs BuffData) in FD+BD. */
#define	FDA_1BLK_SIZE	16			/* Size of one memory block in FDA area */

#define MAC_OWNS_FD		0x8000		/* Controller (MAC) owns the Frame Descriptor.*/
#define MAC_OWNS_BD		0x80		/* Controller (MAC) owns the Buffer Descriptor.*/
#define MAC_OWNS_FDA	0x80000000	/* Controller (MAC) owns the Frame Descriptor.*/

#define SW_OWNS_FD		0x0000		/* Software owns the Frame Descriptor.*/
#define SW_OWNS_BD		0x00		/* Software owns the Buffer Descriptor.*/

#define FDCTL_BD_COUNT	0x001F		/* Bit 0-4 of the FDCtl field of the FD represents # of BDs */


#define ENA_TX_INT		0x00004000	/* Enable Transmit Interrupt, OR */
#define ENA_RX_INT		0x00005200	/* Enable Receive Interrupt for Good Packet and LenErr packet, OR  */
#define DISA_TX_INT		0xFFFFBFFF	/* Disable Transmit Interrupt, for AND */
#define DISA_RX_INT		0xFFFFBFFF	/* Disable Receive Interrupt, for AND */



#define ENA_FdaFul_INT	0x00000001	/* Enable FDA Exhaust Interrupt, OR */
#define DISA_FdaFul_INT	0xFFFFFFFE	/* Disable FDA Exhaust Interrupt, OR */
#define ENA_BlFul_INT	0x00000002	/* Enable BL Exhaust Interrupt, OR */
#define DISA_BlFul_INT	0xFFFFFFFD	/* Disable BL Exhaust Interrupt, OR */


#define ENA_RX_VLAN		0x00000004  /* Enable Receiving of VLAN Packet(1522 bytes), OR operation */
#define DISA_RX_VLAN	0xFFFFFFFB  /* Disable Receiving of VLAN Packet(1522 bytes), AND operation */

#define SPEED_AUTO		0xFFFFFF9F  /* ,Set the MAC speed to Automatic mode. AND operation */
#define SPEED_10		0xFFFFFFBF  /* ,Set the MAC speed to Automatic mode. OR operation */


#define PHY971_RESET		0x8000
#define PHY971_LINK_UP		0x0400
#define PHY971_MODE_100		0x4000 /* for 100 mbps*/
#define PHY971_FULL_DUPLEX	0x0200

#define MAC0_RST	0x40					/* Ethernet 0 Reset */
#define MAC1_RST	0x80					/* Ethernet 1 Reset */

#define	SREG_BASE	0xBC000000	/* Base address to SLM Block       */
#define	SYS_RST_REG	((volatile unsigned int *) (SREG_BASE + 0x010))
#define	RST_SET_REG	((volatile unsigned int *) (SREG_BASE + 0x014))
#define	RST_CLR_REG	((volatile unsigned int *) (SREG_BASE + 0x018))

/*
 * Frame Descriptors for Ethernet MAC of Msp
 */
typedef struct _FD
	{
		struct _FD	* FDNext;
		Ulong		FDSystem;
		Ulong		FDStat;
		Ushort		FDCtl;
		Ushort		FDLength;
	} FD;


/* 
 * Buffer Descriptor for Ethernet MAC of Msp
 */
typedef struct
    {
		Uchar		* BuffData;
		Uchar		BDCtl;
		Uchar		BDStat;
		Ushort		BuffLength;
	} BD;



/* 
 * Structure of FD and BDs to be used Tx and FDA of Rx.
 */
typedef struct
    {
		FD		FD;
		BD		BD[MAX_BD_PER_FRAME];
	} FDBD;

#define FDAFDBD FDBD


/* 
 * Structure of FD and BDs to be used in the BL Area for the Rx.
 */
typedef struct
    {
		FD		FD;
		BD		BD[RX_BD_COUNT];
	} BLFDBD;


/************************************************************************
 * Receive related Buffer and Frames Descriptors pointers.              *
 ************************************************************************/
static BLFDBD	*BLFrmPtr[MSP_ETH_MAX_UNITS];
static FDBD		*FDAPtr[MSP_ETH_MAX_UNITS];
static FDBD		*FDAPtrCur[MSP_ETH_MAX_UNITS];

static Ulong	currentBlBD[MSP_ETH_MAX_UNITS];

/************************************************************************
 * Transmit related Buffer and Frames Descriptors pointers.             *
 ************************************************************************/
static FDBD		*TxFrmPtr[MSP_ETH_MAX_UNITS];
static FDBD		*TxFrmPtrCur[MSP_ETH_MAX_UNITS];

/************************************************************************
 * Internal Function Declarations.                                      *
 ************************************************************************/

/*******************************************


 +--------+-----------+---------+---------+
  Byte 3     Byte 2     Byte 1     Byte 0
 +--------+-----------+---------+---------+
            FRAME DESCRIPTOR

+------------------------------------------+
|               FDNext					   | 0x00
+------------------------------------------+
|              FDSystem				       | 0x04
+------------------------------------------+
|               FDStat					   | 0x08	
+------------------+-----------------------+
|   FDCtl          |          FDLength     | 0x0c
+------------------+-----------------------+




            BUFFER DESCRIPTOR
+------------------------------------------+
|    BuffData (pointer to the buffer)      | 0x00
+------------------------------------------+
|              BStatus                     | 0x04
+------------------------------------------+


**********************************************/


/******************************************************************************
 *	Function Name:      Write_CSR_Reg
 *
 *  Purpose:            
 *		To write to the Control and Status Register of the specified MAC.
 *
 *  Inputs:       
 *		ifIndex	-	interface number 0 or 1
 *		regNo	-	Which register to write
 *      value	-	The value you wish to put into the register
 *
 *   Return Value:
 *
 *****************************************************************************/
static void Write_CSR_Reg(Ulong ifIndex, Ulong regNo, Ulong value)
{  
	if(ifIndex==0)
	{
		*((volatile unsigned int *)(MAC0_BASE + regNo)) = value;
	}
	else if (ifIndex==1)
	{
		*((volatile unsigned int *)(MAC1_BASE + regNo)) = value;
	}
}



/******************************************************************************
 *	Function Name:      Read_CSR_Reg
 *
 *  Purpose:            
 *		To read from the Control and Status Register of the specified MAC.
 *
 *  Inputs:       
 *		ifIndex	-	interface number 0 or 1
 *		regNo	-	Which register to write
 *
 *
 *   Return Value:
 *		data read.  If interface is bad, always read ~0.
 *
 *****************************************************************************/
static Ulong Read_CSR_Reg(Ulong ifIndex, Ulong regNo)
{
	if(ifIndex==0)
	{
		return (*(volatile unsigned int *)(MAC0_BASE + regNo));
	}
	else if (ifIndex==1)
	{
		return (*(volatile unsigned int *)(MAC1_BASE + regNo));
	}
	else
      		return (~0);
}

static void *xalloc(int size)
{
	void *v;

	v = (void *) malloc(size + 15);
	if (v == 0)
	{
#ifdef VERBOSE
		printf("xalloc-malloc returned zero\n");
#endif
		return v;
	}
	v = (void *)(((Ulong) v + 15 ) & 0xfffffff0);
	return v;
}

static void initRmonCounter(unsigned int ifIndex, unsigned int CounterOffset)
{
	Write_CSR_Reg(ifIndex, CNTDATA, 0x00000000);
	Write_CSR_Reg(ifIndex, CNTACC, 0x00000100|CounterOffset);

	return;
}


static void initAllRmonCounters(unsigned int ifIndex)
{
	unsigned int i;

	for(i=0; i<0x38; i++)
	{
		initRmonCounter(ifIndex, i);
	}
}

static void rmonSwitchBank(unsigned int ifIndex)
{

	Write_CSR_Reg(ifIndex, DMA_CTL, ((Read_CSR_Reg(ifIndex, DMA_CTL))| 0x01000000) );  /* switch bank */
	while((Read_CSR_Reg(ifIndex, DMA_CTL))& 0x01000000);
}

static void RmonInit(unsigned int ifIndex)
{
	unsigned int CONTINUE=1, data;

	data = Read_CSR_Reg(ifIndex, DMA_CTL);
	Write_CSR_Reg(ifIndex, DMA_CTL, data|0x01300000);

	while(CONTINUE)
	{
		if(!((Read_CSR_Reg(ifIndex, DMA_CTL))& 0x01300000)) 
		{
			CONTINUE=0;
			break;
		}
	}


	initAllRmonCounters(ifIndex);
	rmonSwitchBank(ifIndex);
}

/*************************************************************************
 *
 * SetupPHY971 - initialize and configure the PHY device.
 *
 * This routine initialize and configure the PHY device 
 *
 * Needs to be modified according to the PHY, and 
 * also there are two different PHYs connected to MAC 1 and 2.....
 * needs to be modified according to the SPEC of Level1 971 and 905.
 *  
**************************************************************************/
static void SetupPHY971(unsigned int ifIndex,unsigned int speed) 
{
	unsigned int i=0;
	unsigned int j;

	while((Read_CSR_Reg(ifIndex, MD_CA))&MD_CA_BUSY_BIT)
	{
		i++;
		if(i==1000000)
		{
#ifdef VERBOSE
			printf("1. PHY still BUSY !!!!\n");
#endif
			return;
		}
	}

   	/*
	 * To write to the PHY, MD_Data register at 0x58 must be setup with
	 * the data that is to be written to the PHY, and then configure
	 * the MD_CA register at 0x5C with the PHY Addr and write/read 
	 * command.
	 */
	Write_CSR_Reg(ifIndex, MD_DATA, 0x8000);	/* reset PHY    */
	Write_CSR_Reg(ifIndex, MD_CA, 0x0C00);		/* send command */
 	i=0;
	while((Read_CSR_Reg(ifIndex, MD_CA))&MD_CA_BUSY_BIT)
	{
		i++;
		if(i==1000000)
		{
#ifdef VERBOSE
			printf("2. PHY still BUSY !!!!\n");
#endif
			return;
		}
	}

	/*
	 * Read the Reset bit(0.15) of the PHY to see if the reset is completed.
	 */
	for (j = 0; j < 1000000; j++)
	{
		Write_CSR_Reg(ifIndex, MD_CA,   0x0800);	/* send READ command  */

		i=0;
		while((Read_CSR_Reg(ifIndex, MD_CA))&MD_CA_BUSY_BIT)
		{
			i++;
			if(i==1000000)
			{
#ifdef VERBOSE
				printf("PHY still in RESET mode !!!!\n");
#endif
				return;
			}
		}

		if ((Read_CSR_Reg(ifIndex, MD_DATA) & PHY971_RESET) == 0)
		{
#ifdef VERBOSE
			printf("PHY reset\n");
#endif
			break;
		}
	}

	/*
	 * LevelOne LXT971
	 * Loopback=disabled
	 * Speed = 100/10
	 * Auto-Negotioation = Disabled
	 * Power Down = 0 normal
	 * Isolate = 0 normal
	 * Restart Auto-Negotiation = 0
	 * Duplex = 1 Full / 0 Half
	 * Collision Test = 0 Disabled
	 *
	 * For Full Dup 10Mbps:  0x0100
	 * For Half Dup 10Mbps:  0x0000
	 * For Full Dup 100Mbps: 0x2100
	 * For Half Dup 100Mbps: 0x2000
	 */
	if(speed==100) {
#ifdef VERBOSE
		/* printf("enter setup 100MPS\n");*/
#endif
		Write_CSR_Reg(ifIndex, MD_DATA, 0x2100);		/* configure PHY  for 100 MPS full Duplex*/
	}
	else if (speed == 10) {
		Write_CSR_Reg(ifIndex, MD_DATA, 0x0100);	/* 0x0100= configure PHY  Half Duplex 10 Mbps*/
	}
	else {
		Write_CSR_Reg(ifIndex, MD_DATA, 0x1000);	/* 0x0000= configure PHY auto */
	}

	Write_CSR_Reg(ifIndex, MD_CA,   0x0C00);		/* WRITE send command  */

	while((Read_CSR_Reg(ifIndex, MD_CA))&MD_CA_BUSY_BIT)
	{
		i++;
		if(i==1000000)
		{
#ifdef VERBOSE
			printf("3. PHY still BUSY !!!!\n");
#endif
			return;
		}
	}
}

/*************************************************************************
*
* PHY971CheckCable - Check for Cable connection to the PHY 971.
*
*
* RETURNS: OK, or ERROR
* Needs to be modified according to the PHY, and 
* also there are two different PHYs connected to MAC 1 and 2.....
* needs to be modified according to the SPEC of Level1 971 and 905.
*  
**************************************************************************/
static Ulong PHY971CheckCable(Ulong ifIndex)
{
	Ulong data, i=0;
	Ulong status;

	Write_CSR_Reg(ifIndex, MD_CA,   0x0811);	/* READ at offset 0x11 of 971 PHY */
	while((Read_CSR_Reg(ifIndex, MD_CA))&MD_CA_BUSY_BIT)
	{
		i++;
		if(i==1000000)
		{
#ifdef VERBOSE
				printf("PHY971 still BUSY !!!!\n");
#endif
			return(ERROR);
		}
	}

/*	for(i=0; i<100000; i++); */
	data = Read_CSR_Reg(ifIndex, MD_DATA);
	

	if(data&PHY971_LINK_UP)
		status = OK;
	else
		status = ERROR;

#ifdef VERBOSE
	if (status == OK)
	{
		if(data&PHY971_MODE_100)
			printf("PHY971 is running at 100 Mbps, ");
		else
			printf("PHY971 is set for 10 Mbps,  ");
		
		if(data&PHY971_FULL_DUPLEX)
			printf("FULL Duplex Mode.\n");
		else
			printf("HALF Duplex Mode. \n");
	}
#endif

	return status;
}

/**********************************************************************
 * Function Name:	EnableTxRx
 *
 *   Purpose:       Starts transmission and reception.
 *   
 *   Parameters:	MAC Interface Index.
 *
 *   Return Value:	None    
 *
 ***********************************************************************/
static void EnableTxRx(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, TX_CTL, ((Read_CSR_Reg(ifIndex, TX_CTL)) | 0x00000001)); /* Enable Transmission */
	Write_CSR_Reg(ifIndex, RX_CTL, ((Read_CSR_Reg(ifIndex, RX_CTL)) | 0x00000001)); /* Enable Reception    */
}



/*************************************************************************
*
* PhyInit - initialize and configure the PHY device.
*
* This routine initialize and configure the PHY device 
*
* RETURNS: OK, or ERROR
* Needs to be modified according to the PHY, and 
* also there are two different PHYs connected to MAC 1 and 2.....
* needs to be modified according to the SPEC of Level1 971 and 905.
*  
**************************************************************************/
static int PhyInit(Ulong ifIndex)
{
	int status=OK, speed=0;				/* 0 means autobaud */
	int i;
	
	
	switch(ifIndex)
	{
		case 0:
			SetupPHY971(ifIndex, speed);
			break;

		case 1:
			SetupPHY971(ifIndex, speed);
			break;

		default:
			status = ERROR;
	}

	if (status == OK)
	{
		for (i = 0; i < 1000000; i++)
		{
			status = PHY971CheckCable(ifIndex);
			if (status == OK)
			{
#ifdef VERBOSE
				printf("Cable detected\n");
#endif
				break;
			}
		}

#ifdef VERBOSE
		if (status != OK)
		{
			printf("We believe NO Cable connected to MAC %d\n", ifIndex);
		}
#endif
	}
	
	if (status == OK)
		EnableTxRx(ifIndex);

   return (status);
} 


/*********************************************************************
 *	Function Name:  SetupTxBDs
 *
 *  Purpose:		To allocate memory for transmit frame and buffer descriptors
 *                  and set up initial values for these.
 *
 *  Inputs:			interface number: 0 or 1.
 *
 *   Return Value:
 *					OK
 *					ERROR
 *
 *********************************************************************/
static Ulong SetupTxBDs(short ifIndex)
{
	int i, j;
	int firsttime = 0;
	FDBD *tempFdBd;
  
	if (TxFrmPtr[ifIndex] == 0)
	{
		/* not yet set up - set up only first time */

		firsttime = 1;

		/*
		 * Set up the TXQ. 
		 * But to make sure 16 byte aligned for each FD.
		 * Allocate memory for transmit frame and  buffer descriptors.
		 */
		if(!(TxFrmPtr[ifIndex] = (FDBD *) xalloc(TX_FD_COUNT * sizeof(FDBD)))) 
		{ 
			return(ERROR);
		}
	}

	tempFdBd = TxFrmPtr[ifIndex];

	for(i = 0; i < TX_FD_COUNT; i++) 
	{  
		tempFdBd->FD.FDSystem		= 0x11111111;		/* A pattern for debugging ! */
		tempFdBd->FD.FDStat			= 0x00000000;
		tempFdBd->FD.FDCtl			= SW_OWNS_FD|BD_PER_TX_FRAME;
		tempFdBd->FD.FDLength		= 0x0000;
		
		for(j = 0; j < MAX_BD_PER_FRAME; j++)
		{
			if (firsttime)
				tempFdBd->BD[j].BuffData = (Uchar *)xalloc(MAX_BUFF_SIZE+15);
			tempFdBd->BD[j].BDCtl		= 0x00;
			tempFdBd->BD[j].BDStat		= 0x33;
			tempFdBd->BD[j].BuffLength	= 0x4444;
		}

		tempFdBd->FD.FDNext			= (FD *)((Ulong)tempFdBd+sizeof(FDBD));
		tempFdBd++;
	}	
		
	/* Set the FDNext in the last FD, to point to the first.*/
	tempFdBd--;
	tempFdBd->FD.FDNext	= (FD *)TxFrmPtr[ifIndex]; 


	/* Set the TxFrmPtrCur to the base of the TXQ. */
	TxFrmPtrCur[ifIndex]		= TxFrmPtr[ifIndex];

#ifdef VERBOSE
	printf("SetupTxBDs-for %d, TxFrmPtr %x, TxFrmPtrCur %x\n",
		   ifIndex, &TxFrmPtr[ifIndex], &TxFrmPtrCur[ifIndex]);
#endif

	return(OK);
}

/****************************************************************************
 *	Function Name:	SetupRxBDs                              
 *                                                                          
 *  Purpose:        Prepare memory for receive buffers and frame descriptors
 *                  and buffer descriptors.                                
 *                                                                         
 *  Inputs:			
 *			ifIndex		 - Interface Index, 0 for MAC0 and 1 for MAC1
 *
 *	Return Value:
 *			OK
 *			ERROR
 *
 *****************************************************************************/
static Ulong SetupRxBDs(Ulong ifIndex)
{
	int i;
	int firsttime = 0;
	Ulong fdaSize;	
	FD		*tempFd;

	if (BLFrmPtr[ifIndex] == 0)
	{
		/* buffers not set up yet - set up buffers first time */

		firsttime = 1;

		/*
		 * Buffer List: (which conisits of single FD and multiple BDs)
		 * There will be RX_BD_COUNT of BDs, 
		 * and each BD points to one RecBuffSpace
		 * of size of MAX_BUFF_SIZE. (add 16 for memeory boundary adjustment).
		 *
		 */
		if(!(BLFrmPtr[ifIndex] = (BLFDBD *) xalloc(sizeof(BLFDBD)))) 
		{ 
			return(ERROR);
		}
	
		/*
		 * Allocate 16 byte memory for (FDA)descriptors for received buffers.
		 * For each frame received, one FD+BD will be written in the FDA if the
		 * rx buffer that is pointed by the BD could fit the entire packet.
		 */
		fdaSize = RX_FDA_COUNT * FDA_1BLK_SIZE; 
		if(!(FDAPtr[ifIndex] = (FDBD *) xalloc(fdaSize))) 
		{ 
			return(ERROR);
		}
	}

	/*
	 * Initialize the FDA Area. Give the ownership of each 16 byte block
	 * to the controller. (The patterns of 0xBBB etc is for debugging only!)
	 */
	tempFd = &(FDAPtr[ifIndex]->FD);
	for(i = 0; i < RX_FDA_COUNT; i++)
	{
		tempFd->FDSystem = 0xBBBBBBBB;
		tempFd->FDStat   = 0xCCCCCCCC;
		tempFd->FDCtl    = MAC_OWNS_FD;
		tempFd->FDLength = 0xEEEE;
		tempFd++;
	}

	/*
	 * Initialize the BL Area. We have only one FD with BDs here, 
	 * so point to self as FDNext
	 */
	BLFrmPtr[ifIndex]->FD.FDNext	 = (FD *)BLFrmPtr[ifIndex];
	BLFrmPtr[ifIndex]->FD.FDSystem	 = 0xAAAAAAAA;		/* A pattern for debugging ! */
	BLFrmPtr[ifIndex]->FD.FDStat	 = 0x00000000;
	/*
	 * Set the number of BDs in the FDLength field of BLFrmPtr according to the
	 * spec.
	 *
     * Controller owns the FD, BDCount left blank as the controller
     * uses FDLength to show the number of BDs available in the Buffer
	 */
	BLFrmPtr[ifIndex]->FD.FDCtl	    = MAC_OWNS_FD;
	BLFrmPtr[ifIndex]->FD.FDLength  = RX_BD_COUNT;	/* Total Number of BDs in the BL area */
	
	/*
	 * Each BD points to a large enough buffer so that for each incoming packet
	 * only one BD per packet will be used.
	 *
	 */
	for(i = 0; i < RX_BD_COUNT; i++) 
	{
		/* Get a buf and attach to the BD.*/

		if (firsttime)
			BLFrmPtr[ifIndex]->BD[i].BuffData = (Uchar *)xalloc(MAX_BUFF_SIZE);

		/* Set all of the BDs to Controller ownership.*/
		BLFrmPtr[ifIndex]->BD[i].BDCtl		= MAC_OWNS_BD;
		BLFrmPtr[ifIndex]->BD[i].BuffLength	= MAX_PKT_SIZE;
	}


	FDAPtrCur[ifIndex] = FDAPtr[ifIndex];
	currentBlBD[ifIndex] = 0;

#ifdef VERBOSE
	printf("SetupRxBDs-for %d, BLFrmPtr %x, FDAPtr %x, Cur %x, curBD %x\n",
		   ifIndex, &BLFrmPtr[ifIndex], 
		   &FDAPtr[ifIndex], &FDAPtrCur[ifIndex], &currentBlBD[ifIndex]);
#endif

	return(OK);
}


/**********************************************************************
 * Function Name:	DisableTxRx
 *
 *   Purpose:       Stops transmission and reception.
 *   
 *   Parameters:	MAC Interface Index.
 *
 *   Return Value:	None    
 *
 ***********************************************************************/
static void DisableTxRx(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, TX_CTL, ((Read_CSR_Reg(ifIndex, TX_CTL)) & 0xFFFFFFF2)); /* Cancel Transmission */
	Write_CSR_Reg(ifIndex, RX_CTL, ((Read_CSR_Reg(ifIndex, RX_CTL)) & 0xFFFFFFF2)); /* Cancel Reception    */
}

static void ResetInternal(int bit)
{
	*RST_SET_REG = bit;
	while (!(*SYS_RST_REG & bit))
		;
	*RST_CLR_REG = bit;
	while (*SYS_RST_REG & bit)
		;
}

/*************************************************************************
 *
 * Function Name:		Reset
 *
 * Purpose:				To reset the MAC.
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static short Reset(Ushort ifIndex)
{
	if(ifIndex == 0) 
	{
		ResetInternal(MAC0_RST);
	}
	else if(ifIndex == 1) 
	{
		ResetInternal(MAC1_RST);
	}
	else
	    return(ERROR);
	
	return (OK);
}


static void SetupRegistersTx(unsigned int ifIndex)
{
	Write_CSR_Reg(ifIndex, DMA_CTL,  0x0008C040);
	Write_CSR_Reg(ifIndex, MAC_CTL,  0x00000000); 	/* auto Duplex 0x000000 */
	Write_CSR_Reg(ifIndex, TXTHRSH,  0x00000000); 
	Write_CSR_Reg(ifIndex, TXPOLLCTR,0x00000fff);
	Write_CSR_Reg(ifIndex, TX_CTL,   0x00000000);
}

static void SetupRegistersRx(unsigned int ifIndex)
{
	Write_CSR_Reg(ifIndex, RXFRAGSIZE, 0x00); 
	Write_CSR_Reg(ifIndex, FDA_BAS, (Ulong) FDAPtr[ifIndex]); 

	Write_CSR_Reg(ifIndex, FDA_LIM, ((RX_FDA_COUNT-1)<<4));  

	Write_CSR_Reg(ifIndex, ARC_CTL, 0x08);			/* setting to promiscuous mode*/
	Write_CSR_Reg(ifIndex, ARC_ADR, 0x4);			/* setting to promiscuous mode*/
	Write_CSR_Reg(ifIndex, ARC_DATA, 0x02040405);	/* setting to promiscuous mode*/
	Write_CSR_Reg(ifIndex, ARC_ENA, 0x00); 

	Write_CSR_Reg(ifIndex, RX_CTL, 0x00000000);		/*originally 0x205 enabling VLAN, or 0x4A11, 0x4211*/

	Write_CSR_Reg(ifIndex, BLFRMPTR, (unsigned int)BLFrmPtr[ifIndex]); 
}


static void SetUpTxFrmPtrRegister(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, TXFRMPTR, ((Ulong)TxFrmPtr[ifIndex] & 0x1fffffff));
}

/*************************************************************************
 *
 *	Function Name:		TxRxInit
 *
 *   Purpose:           Set up the Ethernet to receive and transmit data.
 *						Same as BootMspEthInit(), but includes the ifIndex as
 *						parameter.
 *
 *
 *   Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong TxRxInit(Ulong ifIndex)
{
	
	Reset(ifIndex);
    
	if(SetupRxBDs(ifIndex) == ERROR)
		return(ERROR);
    
	if(SetupTxBDs(ifIndex) == ERROR)
		return(ERROR);

	SetupRegistersTx(ifIndex);
	
	SetupRegistersRx(ifIndex);

	DisableTxRx(ifIndex);

	SetUpTxFrmPtrRegister(ifIndex);

	RmonInit(ifIndex);

    return (PhyInit(ifIndex));
}


/*************************************************************************
 *
 * Function Name:		EnableVLAN
 *
 * Purpose:				To enable the VLAN functionalities, so that MAC 
 *						could handle packets as long as 1522 bytes(instead of 1518 bytes).
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong EnableVLAN(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, RX_CTL, ((Read_CSR_Reg(ifIndex, RX_CTL)) | ENA_RX_VLAN) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		DisableVLAN
 *
 * Purpose:				To disble the VLAN functionalities of the MAC.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong DisableVLAN(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, RX_CTL, ((Read_CSR_Reg(ifIndex, RX_CTL)) & DISA_RX_VLAN) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		SetSpeedAuto
 *
 * Purpose:				To Set the MAC in Automatic mode.
 *						NOTE: The actual speed is determined by the PHY
 *						that is connected to this MAC.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong SetSpeedAuto(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, MAC_CTL, ((Read_CSR_Reg(ifIndex, MAC_CTL)) & SPEED_AUTO) );

	return(OK);
}

/*************************************************************************
 *
 * Function Name:		SetSpeed10
 *
 * Purpose:				To Set the MAC in 10 MBPS mode.
 *						NOTE: The speed of the PHY should match this speed.
 *						
 *
 *
 * Return Value:		ERROR on failure
 *                      OK on success
 **************************************************************************/
static Ulong SetSpeed10(Ulong ifIndex)
{
	Write_CSR_Reg(ifIndex, MAC_CTL, ((Read_CSR_Reg(ifIndex, MAC_CTL)) & SPEED_AUTO) | 0xFFFFFFBF );

	return(OK);
}

/*************************************************************
*  void *mspeth_driver(int op,void *vp1,void *vp2)
*	Main entry point for this driver. Everything else in this
*	module should be static.
*/
static void *mspeth_driver(int interfaceNum, int op,char **bufp,int *lenp)
{
	BLFDBD		*blptr;
	BD			*bdptr;
	char		*buf;
	FDBD		*rr;
	FDBD		*tempfd;
    unsigned int flushToSDRAM;
	
	switch (op) {
	case MSP_ETHERINIT :
		/* int ether_driver(int,MSP_ETHERINIT,void,void) */
		if (TxRxInit(interfaceNum) != OK)
			return(0);
		return (void *) 1;
		
	case MSP_ETHERXMIT :
		/* int ether_driver(int,MSP_ETHERXMIT,char **bufp,int *lenp) */
		rr = TxFrmPtrCur[interfaceNum];
		if (rr->FD.FDCtl & MAC_OWNS_FD)
			return 0;					/* cannot xmit packet */
		
		buf = *bufp;					/* swap packets with sender */
		*bufp = rr->BD[0].BuffData;
		rr->BD[0].BuffLength = *lenp;
		rr->BD[0].BuffData = buf;
		
		/* give new packet to chip harware */
		rr->BD[0].BDCtl = MAC_OWNS_BD;
		rr->FD.FDCtl = MAC_OWNS_FD | BD_PER_TX_FRAME;
		TxFrmPtrCur[interfaceNum] = (FDBD *) rr->FD.FDNext;
		Sync();
		Write_CSR_Reg(interfaceNum, DMA_CTL, 
					  ((Read_CSR_Reg(interfaceNum, DMA_CTL))| 0x00010000) );
		return((void *)1);

	case MSP_ETHERRECV :
		/* int ether_driver(int,MSP_ETHERRECV,char **bufp,int *lenp) */
	  
		rr = FDAPtrCur[interfaceNum];
		HitInvDCache(rr,sizeof(*rr));
		if (rr->FD.FDCtl & MAC_OWNS_FD)
			return 0;					/* no packet available */
		
		HitInvDCache(rr->BD[0].BuffData, rr->FD.FDLength);
		*bufp = rr->BD[0].BuffData;
		*lenp = rr->BD[0].BuffLength-4;

		/* Give FD area produced by chip back to chip */
		tempfd = (FDBD *)FDAPtrCur[interfaceNum]->FD.FDNext;
		FDAPtrCur[interfaceNum]->BD[0].BDCtl = MAC_OWNS_BD;
		FDAPtrCur[interfaceNum]->FD.FDCtl = MAC_OWNS_FD;
		FDAPtrCur[interfaceNum] = tempfd;
		return (void*) 1;					/* give packet to receiver */
		
	case MSP_ETHERRCMP:
		/* int ether_driver(int,MSP_ETHERRCMP,char **bufp,int *lenp) */
		blptr = BLFrmPtr[interfaceNum];
		bdptr = &blptr->BD[currentBlBD[interfaceNum]];
		bdptr->BuffData = *bufp;
		bdptr->BuffLength = MAX_PKT_SIZE;
		blptr->BD[currentBlBD[interfaceNum]].BDCtl = MAC_OWNS_BD;
		currentBlBD[interfaceNum] = 
			(currentBlBD[interfaceNum] + 1) % RX_BD_COUNT;
		return (void *) 1;

	default : 
		printf("mspeth_driver-bad op code %d\n", op);
		return(0);
	}
	return(0);
}

main()
{
	char		*bufp;
	int			len;
	int			n = 0;
	int			noactivity = 0;

	if (!mspeth_driver(0, MSP_ETHERINIT, 0, 0) ||
		!mspeth_driver(1, MSP_ETHERINIT, 0, 0))
		return 0;

	while (1) 
	{
		if (mspeth_driver(n, MSP_ETHERRECV, &bufp, &len))
		{
			mspeth_driver(1 - n, MSP_ETHERXMIT, &bufp, &len);
			mspeth_driver(n, MSP_ETHERRCMP, &bufp, &len);
			noactivity = 0;
		}
		else if (noactivity++ > 100000)
		{
			scandevs();
			noactivity = 0;
		}

		n = 1 - n;
	}
	
}
