 /*************************************************************
 * File: bsps/iceif.c
 * Purpose: serialIce interface code
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	961220	Created
 *	970216	Switched over to addCmdRec for extra commands.
 *	970218	Added dflush_needed logic.
 *	970224	Added dbx logic. works for bda.
 *	970225	Flush not needed if write is to internal register.
 *	970226	Delay flush until just before RUN.
 *	970304	Changed type of ch to Uchar in wakeup() (CYGNUS)
 *	970310	Moved brkInstall and brkRemove from here
 *	970310	Changed verbose cmd to use a level
 *	970310	Rewrote setbp_target()
 *	970311	Created bptReq() to handle bp requests from xvw and gdb
 *	970313	Rationalized the calls to init and wakeup (hopefully).
 *	970401	Added unaligned read_target to support MIPS16 disassembly
 *	970605	Changed is_writeable to use a byte (for 4101)
 *	970817	Added Hi and LO as PC+1 and PC+2
 *	970905	Switched over to new send_buffer
 *	970905	Added A1 case to write_target and read_target
 *	970905	Changed to new version of swap32n
 *	970905	p in send_buffer changed to Uchar
 *	970917	Added perens to self bra test in setTrcbp.
 *	970917	Added _ctrl(22) target_stopped.
 *	971023	Added REV1_WIGGLER switch.
 *	971112	Added more cmds to "portconfig".
 *	980127	Added no_gpr30 cmd. Needed for multi in "debug 16" mode.
 *	980127	Added send_ack_each_word cmd. Removed REV1_WIGGLER switch.
 *	980127	Removed USE_SIM_TARG switch.
 *	980127	Changed printDiag to use logfp. Eliminated logfile2.
 *	980127	Changed logfile to not append.
 *	980128	Added siemens_mbd cmd
 *	980128	Added ocm support
 *	980130	Removed 'verbose'. Now using 'logging'.
 *	980202	Added 'breakcode'.
 *	980204	Renamed bptReq, is_bptReq.
 *	980226	Fixed a bunch of stuff incl SRUN.
 *	980305	Changed siemens_mbd to force_wakeup
 *	980309	Added resync
 *	980309	Changed force_wakeup to force_resync
 *	980309	Added force_getsap
 *	980312	Merged iceif.c for dll and imon
 *	980323	Removed send_buffer from start of send_instr (recursive)
 *	980704	Changed write_target hword to bypass the buffer.
 *	980706	Added support for Siemens mbd kernel (force_sap).
 *	980707	Added support for ocm routines on dll w v1 kernel.
 *	980720	Added nobrkRemove. Needed for force_getsap.
 *	980727	Added resyncOk.
 *	980803	Added mallocOcmRec and freeOcmRec.
 *	980818	Added getSAP before write_target_byte calls.
 *	980821	Added undoc1.
 *	980903	Started adding ejtag support.
 *	981214	Made setbp_target less dbx-specific. Removed dcs member.
 *	981216	Added mask support to ibpts in setbp_target.
 *	981222	Fixed problem with corrupted downloads on 4011.
 *	990121	Added ejtag_block.
 *	990203	Fixed reg nums for LO and HI. Was swapped in several places.
 *	990311	Fixed prob with dup bpts. Added 'type' to which_bpt().
 *	990319	Fixed prob w writes to addr zero. 
 *	990331	Added dma op for send_bytes w ejtag.
 *	990331	Removed ejtag_block.
 *	990406	Fixed prob w ejtag fill. Byte not replicated across word.
 */

#include "iceif.h"

/*#define VERBOSE_DOWNLOAD /* */

#define BUFSZ	4096
#define MAXRL	65535
#define RLTHRESHOLD 84	/* rlcnt of > the value use set_bytes() */
#define MAXRETRIES 5    /* max number of attempts to contact target */

char *excodes[] = { /* includes r4000 codes */
	"Int","MOD","TLBL","TLBS","AdEL","AdES","IBE","DBE",
        "Sys", "Bp","RI","CpU","Ov","Tr","Resv","FPE",
	"Resv","Resv", "Resv","Resv","Resv","Resv","Resv", "WATCH",
	"Resv","Resv","Resv","Resv","Resv","Resv", "Resv","Resv",
	0};


char *xtcodes[] = { "PC","MEM","GPR","CP0", "CP1","CP1C",
	"CP2","CP2C", "CP3","CP3C", "DBX",0};

Ulong current_r8,current_r9;
int instr_buffer_cnt;
Ulong instr_buffer[300]; /* must be larger than the largest ibs */
int lendian_host;
int lendian_target;
int ejtag_mode;

int dcache_dirty;
int has_wb_dcache;		/* cpu has a write-back (copy-back) dcache */
int need_initial_flush = 1; /* default is yes */
int target_stopped;
int iflush_needed;
int dflush_needed;
int ice_baud;
char *ice_device;
int nonrt;
int slow_target;
int dbx_needs_pa;	/* the dbx module needs physical addresses */
int resyncOk;		/* kernel accepts 55 at any time */
int undoc1; 		/* undocumented cmd for test use only */
int noforcek1;		/* don't force addresses to kseg1 */

/* For version 2 kernels, ibs and regmap comes from the kernel.
 * For version 1 kernels, you must specify it in the config file.
 * However, it is only critical for cases where an ocm routine is provided 
 */
int kernel_ibs;		/* kernel ibs value */
Ulong kernel_map;	/* kernel regmap */

struct {
	int rlcnt;
	int bcnt;
	Uchar *buf;
	Ulong start_addr;
	Ulong expected_addr;
	Uchar prev_v;
	} ice;

OcmRec *ocmChain,*freeOcmChain;

BrkList brkList[MAX_BPT];

void gdbstop(int type);
void stop(int type);
void *malloc(Ulong size);
void free(void *ptr);

int has_nhwib;	/* has N h/w instr break regs */
int has_nhwdb;	/* has N h/w data break regs */

int icache_size;
int dcache_size;
int cache_line_size;
int has_ilock;	/* true or false */
int _cputype;

vFunc *brkInstall_ptr;
Func *ilockReq_ptr;
vFunc *flush_target_ptr;
Func *brkRemove_ptr;
vFunc *setFlushneeded_ptr;

int no_gpr30;
int send_ack_each_word;
int force_getsap;
Ulong breakcode;
int nobrkRemove;

SaveArea savearea; /* info about the serialice kernel */

#include "defs.h"

/*************************************************************
*  int wakeup(void) DEF
*/
int wakeup(void)
{
printDiag(1,"wakeup() ");
resync();
printDiag(1,"wakeup: connection established.\n");
dcache_dirty = 1;
brkRemove(0);
return(1);
}

/*************************************************************
*  int resync(void);
*	This routine only works for version 2 or greater kernels.
*/
int resync(void)
{
int i,timeout,ackcnt;
Uchar ch;
static int resync_active;

if (resync_active) return(0); /* avoid unintended recursion */
resync_active = 1;

printDiag(1,"resync() ");
current_r8 = current_r9 = 0;

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	extern char *init_fname;
	if (target_stopped) {
		resync_active = 0;
		return(1);
		}
	printDiag(3,"EjtagControl(WAKEUP)\n");
	if (!EjtagControl(EJTAG_WAKEUP,0,0)) {
		printDiag(1,"resync failure.\n");
		resync_active = 0;
		return(0);
		}
	printDiag(1,"resync achieved.\n");
	target_stopped = 2;
	if (init_fname) {
		sload(init_fname);
		init_fname = 0;
		}
	resync_active = 0;
	return(1);
	}
#endif

flushInput();
/* Try a bunch of times. Use a short timeout for the first 4 attempts.
 * Then switch to 1 sec for the remaining 15. We also require 2 
 * successful acks in a row (paranoia). The mbd kernel actually needs 
 * this. 
 */
timeout = 500;
for (i=ackcnt=0;i<4+15;i++) {
	ch = ATTN;
	PUT_BYTES(&ch,1);
	if (i == 4) timeout = 1000;
	if (GET_BYTE(&ch,timeout) && ch == ACK) {
		ackcnt++;
		printDiag(1,"%02x ",ch);
		if (ackcnt >= 2) break;
		}
	else ackcnt = 0;
	}
if (i >= 4+15) { 
	printDiag(0,"resync failure.\n");
	resync_active = 0;
	return(0);
	}
printDiag(1,"resync achieved.\n");
target_stopped = 2;

if (force_getsap && !checkSAP()) {
	deleteOcmRec((Ulong)savearea.sap);
	savearea.sap = 0;
	getSAP();
	}
resync_active = 0;
return(1);
}

/*************************************************************
*  int send_instr(Ulong v) DEF
*/
int send_instr(Ulong v)
{
char buf[100];

printDiag(8,"send_instr(%08x)\n",v);

if (diaglevel >= 4 && v != 0x12345678 && v != 0x87654321 && v != 0xdeadbeef) {
	disasm(buf,instr_buffer_cnt*4,v);
	printDiag(4,buf);
	printDiag(4,"\n");
	}

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	if (v == 0x87654321) { /* RUN_MODE */
		if (!EjtagControl(EJTAG_RUN,0,0)) {
			printDiag(1,"EjtagControl: RUN: failed.\n");
			return(0);
			}
#if 0
		else target_stopped = 2;
#endif
		}
	else {
		if (!EjtagControl(EJTAG_WRITE,v,0)) {
			printDiag(1,"EjtagControl: WRITE: failed.\n");
			return(0);
			}
		}
	return(1);
	}
#endif

instr_buffer[instr_buffer_cnt++] = v;
if (lendian_host) swap32n((Uchar *)&v,1);
PUT_BYTES((Uchar *)&v,4);
if (instr_buffer_cnt == 1) printDiag(2,"buffer_cnt="); 
printDiag(2,"%d ",instr_buffer_cnt); 
printDiag(3,"%08x ",v);
return(1);
}

/*************************************************************
*  int readWord(Ulong *vp);
*/
int readWord(Ulong *vp)
{
int i;
Ulong v;
Uchar ch;

*vp = v = 0;
for (i=0;i<4;i++) {
	if (!GET_BYTE(&ch,1000)) {
		printDiag(1,"readWord: timeout\n");
		return(0);
		}
	v <<= 8;
	v |= ch;
	}
v =  (v<<24)|((v<<8)&0x00ff0000)|((v>>8)&0x0000ff00)|(v>>24);
if (send_ack_each_word) PUT_BYTES((Uchar *)"0",1);

printDiag(1,"v=%08x\n",v);
*vp = v;
return(1);
}

/*************************************************************
*  Ulong *save_instr_buffer()
*/
Ulong *save_instr_buffer()
{
int i;
Ulong *p;

printDiag(1,"save_instr_buffer: cnt=%d\n",instr_buffer_cnt);
if (instr_buffer_cnt == 0) return(0);

p = malloc((instr_buffer_cnt+1)*4);
p[0] = instr_buffer_cnt;

for (i=0;i<instr_buffer_cnt;i++) p[i+1] = instr_buffer[i];
instr_buffer_cnt = 0;
return(p);
}

/*************************************************************
*  void restore_instr_buffer(Ulong *p)
*/
void restore_instr_buffer(Ulong *p)
{
int i;

if (p==0) return;
printDiag(1,"restore_instr_buffer cnt=%d\n",p[0]);

instr_buffer_cnt = p[0];
for (i=0;i<instr_buffer_cnt;i++) instr_buffer[i] = p[i+1];
free(p);
}

/*************************************************************
*  Ulong readA0(void) DEF
*/
Ulong readA0(void)
{
Ulong v;
int i,n,sbc;
Ulong *buf;

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	send_instr(0); /* nop */
	if (!EjtagControl(EJTAG_EXE,0,&v)) {
		printDiag(0,"EjtagControl: EXE: failed.\n");
		return(0);
		}
	printDiag(1,"v=%08x\n",v);
	return(v);
	}
#endif

flushInput();
send_instr(SENDA0);
if (savearea.ibs && instr_buffer_cnt > ((int)savearea.ibs)) {
	printDiag(0,"ibuf overflow ibc=%d\n",instr_buffer_cnt);
	instr_buffer_cnt = 0;
	return(0);
	}

for (n=0;!readWord(&v);n++) {
	printDiag(1,"readA0: readWord failed\n");
#if 0
	if (savearea.vers < 2) { v = 0; break;}
#else
	if (!resyncOk) { v = 0; break;}
#endif
	if (n >= MAXRETRIES) {v=0; break;}
	printDiag(1,"readA0: resending...\n");
	buf = save_instr_buffer();
	resync();
	restore_instr_buffer(buf);
	sbc = instr_buffer_cnt;
	instr_buffer_cnt = 0;
	for (i=0;i<sbc;i++) send_instr(instr_buffer[i]);
	}

instr_buffer_cnt = 0;
if (force_getsap) {
	target_stopped = 1; /* force wakeup at start of each command */
	deleteOcmRec((Ulong)savearea.sap);
	savearea.sap = 0; /* force reread of SAP */
	}

return(v);
}

/*************************************************************
*  void swap32n(Uchar *p,int n) DEF
*/
void swap32n(Uchar *p,int n)
{
Uchar v;

for (;n>0;n--) {
	v = p[0]; p[0] = p[3]; p[3] = v;
	v = p[1]; p[1] = p[2]; p[2] = v;
	p += 4;
	}
}

/*************************************************************
*/
Ulong read_unaligned_long(Uchar *a)
{
#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode)
	return (((a)[3]<<24)|((a)[2]<<16)| ((a)[1]<<8)|((a)[0])); 
#endif
return (((a)[0]<<24)|((a)[1]<<16)| ((a)[2]<<8)|((a)[3])); 
}

/*************************************************************
*  void flushInput(void) DEF
*/
void flushInput(void)
{
int i;
Uchar ch;

printDiag(2,"flushInput ");
for (i=0;;i++) if (!GET_BYTE(&ch,1)) break;
if (i) printDiag(1,"%d bytes flushed.\n",i);
}

/*************************************************************
*  int checkSAP(void)
*/
int checkSAP(void)
{
Ulong v;

printDiag(1,"checkSAP ");
flushInput();
for (;;) {
	send_instr(SENDSAP);
	if (readWord(&v)) break;
	}
v &= ~1;
if (!noforcek1) v |= K1BASE;
instr_buffer_cnt = 0;
if (savearea.sap && v == (Ulong)savearea.sap) return(1);
printDiag(1,"checkSAP: need new SAP value %08x vs %08x\n",savearea.sap,v);
return(0);
}

/*************************************************************
*  void getSAP(void) DEF
*/
void getSAP(void)
{
Ulong v,sav,m;
int sz,i;
static int getsap_active;

/* avoid unintended recursion and unnecessary operations */
if (getsap_active || savearea.sap) return; 
getsap_active = 1;

printDiag(1,"getSAP ");

current_r8 = current_r9 = 0;
if (target_stopped != 2) wakeup(); /* 980407 */

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	printDiag(4,"EjtagControl(SAP)\n");
	if (!EjtagControl(EJTAG_SAP,0,&v)) {
		printf("EjtagControl: SAP: failed.\n");
		getsap_active = 0;
		return;
		}
	printDiag(2,"sap = %08x\n",v);
	}
else {
#endif
	flushInput();
	for (;;) {
		instr_buffer_cnt = 0;
		send_instr(SENDSAP);
		if (readWord(&v)) break;
		}
#if defined(EJTAG) && !defined(PMCC)
	}
#endif
instr_buffer_cnt = 0;
if (!noforcek1) v |= K1BASE;
target_stopped = 2;

if (v&1) {
	printDiag(1,"new style savearea\n");
	v &= ~1;
	savearea.sap = (Ulong *)v;
	/* need to read header info */
	/* check to see if it's the right version */
	if ((sav=read_target(XT_MEM,(Ulong)(ICE_SAV+savearea.sap),4)) != 2) {
		printDiag(1,"bad savearea version %d\n",sav);
		savearea.sap = 0;
		getsap_active = 0;
		return; /* wrong version */
		}
	savearea.vers = sav;
	resyncOk = 1;
	savearea.hsize = read_target(XT_MEM,(Ulong)(ICE_SAH+savearea.sap),4);
	savearea.map = read_target(XT_MEM,(Ulong)(ICE_MAP+savearea.sap),4);
	savearea.ibs = read_target(XT_MEM,(Ulong)(ICE_IBS+savearea.sap),4);
	if (savearea.hsize > 7)
	    savearea.le = read_target(XT_MEM,(Ulong)(ICE_LE+savearea.sap),4);
	printDiag(1,"getSAP vers=%d hsize=%d map=%08x ibs=%d le=%d\n",
			savearea.vers,savearea.hsize,savearea.map,
			savearea.ibs,savearea.le);
	if (savearea.le) lendian_target = 1;
	/* 980303 turn off flush requirement when writing to savearea */
	sz = savearea.hsize;
	for (i=0,m=1;i<32;i++,m<<=1) if (m&savearea.map) sz++;
	addOcmRec((Ulong)savearea.sap,sz*4,0,0);
	}
else {
	printDiag(1,"old style savearea\n");
	savearea.sap = (Ulong *)v;
	savearea.vers = 1;
	addOcmRec((Ulong)savearea.sap,12*4,0,0);
	}
getsap_active = 0;
}

/*************************************************************
*  int hwibReq(Ulong addr) DEF
*	verify that other hw ibpts don't conflict with this one.
*	i.e. More than one. Return -1 if not possible.
*	Else return hw resource number.
*/
int hwibReq(addr)
Ulong addr;
{
int i,n;

/* if (is_ocm(addr)) return(-1); 980805 */
if (!has_nhwib) return(-1);

n = 0;
/* count how many are already set */
for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (brkList[i].method != BRK_METHOD_HW) continue;
	if (brkList[i].type == BPTYPE_DATA) continue;
	n++;
	}
if (n > (has_nhwib-1)) return(-1);
return(n);
}

/*************************************************************
*  int hwdbReq(Ulong addr) DEF
*	verify that other hw dbpts don't conflict with this one.
*	i.e. More than one. Return -1 if not possible.
*	Else return hw resource number.
*/
int hwdbReq(addr)
Ulong addr;
{
int i,n;

/* if (is_ocm(addr)) return(-1); 980805 */
if (!has_nhwdb) return(-1);

n = 0;
for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (brkList[i].method != BRK_METHOD_HW) continue;
	if (brkList[i].type != BPTYPE_DATA) continue;
	n++;
	}
if (n > (has_nhwdb-1)) return(-1);
return(n);
}

/*************************************************************
*  void printBpts()
*/
void printBpts()
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	switch (brkList[i].method) {
		case BRK_METHOD_HW :
			printDiag(1,"BRK_METHOD_HW %08x\n",brkList[i].addr);
			break;
		case BRK_METHOD_RAM :
			printDiag(1,"BRK_METHOD_RAM %08x\n",brkList[i].addr);
			break;
		case BRK_METHOD_ROM :
			printDiag(1,"BRK_METHOD_ROM %08x\n",brkList[i].addr);
			break;
		default: 
			printDiag(0,"%d: BRK_METHOD unknown\n",
				brkList[i].method);
		}
	}
}

/*************************************************************
*  int is_cacheable(addr)
*/
int is_cacheable(addr)
Ulong addr;
{
if (IS_K0SEG(addr)) return(1);
return(0);
}

/*************************************************************
*  int setbp_target(int n,int type,Ulong addr,Ulong addr2,Ulong value) DEF
*	type: 1=bpc 2=bda 3=itemp 4=sstep 5=nonrt
*	returns bp number, or -1 on error.
*/
int setbp_target(n,type,addr,addr2,value)
int n,type;
Ulong addr,addr2,value;
{
int i,method,atype,code;
Ulong info,x;
long mask; /* must be signed */

printDiag(1,"setbp_target(%d,%x,%08x,%08x,%08x)\n",n,type,addr,addr2,value); 
atype = type>>4;
type &= 0xf;
code = 0;
if (type == BPTYPE_NONRT) {
	printDiag(0,"warning: this breakpoint requires non real-time execution\n");
	nonrt = 1;
	return(0);
	}

if ((n=which_bpt(addr,type)) != -1) {
	printDiag(1,"setbp_target: %08x duplicate bpt\n",addr); 
	return(0-BP_E_ERR);
	}

if (type == BPTYPE_PC || type == BPTYPE_ITMP || type == BPTYPE_TRACE) {
	if (is_writeable_target(addr)) method = BRK_METHOD_RAM;
	else if (is_cacheable(addr) && ilockReq(addr)) method = BRK_METHOD_ROM;
	else if ((x=hwibReq(addr)) != -1) {
                method = BRK_METHOD_HW;
		if (atype&8) mask = addr2; /* 8 = addr2 is a mask */
		else if (addr2) { /* compute mask */
			for (mask=0x80000000;(addr&mask) == (addr2&mask);mask>>=1) ;
			mask <<= 1; /* oops! one step too many */
			code |= BP_W_MSK;
			}
		else mask = 0xffffffff; /* all-enable */
#if 0 /* 981214 */
                dcs = DCS_TR|DCS_UD|DCS_KD|DCS_DE|DCS_PCE;
		/* Note: some dbx implementations require a Physical Address */
#endif
		info = (x<<16);
                }
	else {
		printDiag(0,"%08x: can't set bpt\n",addr);
		return(0-BP_E_ERR);
		}
	}
else if (type == BPTYPE_DATA) {
	if ((x=hwdbReq(addr)) == -1) {
		printDiag(0,"%08x: can't set bpt\n",addr);
		return(0-BP_E_ERR);
		}
        method = BRK_METHOD_HW;
#if 0 /* 981214 */
        dcs = DCS_TR|DCS_UD|DCS_KD|DCS_DE|DCS_DAE;
 
        if (atype&1) dcs |= DCS_DR; /* 1 = read */
        if (atype&2) dcs |= DCS_DW; /* 2 = write */
#else
	info = (x<<16);
        if (atype&1) info |= 2; /* 2 = read */
        if (atype&2) info |= 1; /* 1 = write */
#endif

	if (atype&4) return(0-BP_E_VAL);
	/* 4 is used by the gdb interface to indicate that a value
	 * has been specified. See debug.c. 
	 */
		
 
	if (atype&8) mask = addr2; /* 8 = addr2 is a mask */
	else if (addr2) { /* compute mask */
                for (mask=0x80000000;(addr&mask) == (addr2&mask);mask>>=1) ;
                mask <<= 1; /* oops! one step too many */
		code |= BP_W_MSK;
                }
        else mask = 0xffffffff; /* all-enable */
        }

if (n == -1) {
	for (i=0;i<MAX_BPT;i++) if (brkList[i].type == 0) break;
	if (i >= MAX_BPT) {
		printDiag(0,"fatal error: out of bpts\n");
		return(0-BP_E_ERR);
		}
	n = i;
	}
if (n < 0 || n >= MAX_BPT) {
	printDiag(0,"%d: bad bpt number\n",n);
	return(0-BP_E_ERR);
	}
brkList[n].type = type;
brkList[n].addr = addr;
brkList[n].method = method;
brkList[n].mask = mask;
brkList[n].aux[0] = info;
return((code<<16)|n);
}

/*************************************************************
*  int clrbp_target(int n) DEF
*	return 0 on error; else 1
*/
int clrbp_target(n)
int n;
{
printDiag(1,"clrbp_target(%d)\n",n);
if (n < 0 || n >= MAX_BPT) {
	printDiag(0,"%d: bad bpt number\n",n);
	return(0);
	}
if (brkList[n].type==0) {
	printDiag(0,"%d: bpt is not set\n",n);
	return(0);
	}
brkList[n].type = 0;
return(1);
}


/*************************************************************
*/
int which_bpt(Ulong addr,int type)
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (type && type != brkList[i].type) continue;
	if (brkList[i].addr == addr) return(i);
	}
return(-1);
}

/*************************************************************
*  int which_dbpt(int hwbptno) DEF
*/
int which_dbpt(int hwbptno)
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (brkList[i].type != BPTYPE_DATA) continue;
	if (brkList[i].method != BRK_METHOD_HW) continue;
	if (((int)(brkList[i].aux[0]>>16)) == hwbptno) return(i);
	}
return(-1);
}

#define RS_(x)     (((x) >> 21) & ((1L <<  5) - 1))
#define SEX16(x)   ((long)(x<<16)>>16)
#define SIMM16(x)  (SEX16(UIMM16(x)))
#define UIMM16(x)  ((x)&0xffff)
#define IS_LDST(n) ((((n)>>26)>=0x20)&&(((n)>>26)<=0x2e))

/*************************************************************
*  int is_bpt(addr) DEF
*/
int is_bpt(addr)
Ulong addr;
{
int i;
Ulong instr,ea;

for (i=0;i<MAX_BPT;i++) {
	if (brkList[i].type==0) continue;
	if (brkList[i].addr == addr) return(brkList[i].type);
	if (brkList[i].type == BPTYPE_DATA) {
		instr = read_target(XT_MEM,addr,4); /* read instr */
		/* if lw/sw and addr eval == addr return type */
		if (!IS_LDST(instr)) continue;
		ea = read_target(XT_GPR,RS_(instr),4) + SIMM16(instr);
		if ((ea&brkList[i].mask) == brkList[i].addr) {
			printDiag(2,"is_bpt: %08x will hwdb\n",addr);
			return(brkList[i].type);
			}
		}
	}
return(0);
}


/*************************************************************
*  void brkDelete(Ulong type) DEF
*/
void brkDelete(type)
Ulong type;
{
int i;

for (i=0;i<MAX_BPT;i++) {
	if (type == brkList[i].type) brkList[i].type = 0;
	}
}

/*************************************************************
*  int is_writeable_target(Ulong adr) DEF
*/
int is_writeable_target(adr)
Ulong adr;
{
Uchar x,y;
OcmRec *p;

if ((p=is_ocm(adr)) && p->readonly) {
        printDiag(1,"\nocm not writeable(%08x)\n",adr);
        return(0);
        }

if (!noforcek1) adr |= K1BASE;
x = (Uchar)read_target(XT_MEM,adr,1);
y = ~x;

writeGpr(8,y);
writeGpr(9,adr);
send_instr(SB(8,0,9));
readA0();

/* flush_target(DCACHE); */
if (read_target(XT_MEM,adr,1) != y) {
        printDiag(1,"\nnot writeable(%08x)\n",adr);
	return(0);
	}

printDiag(1,"%08x is writeable\n",adr);
/* restore the value */
writeGpr(8,x);
writeGpr(9,adr);
send_instr(SB(8,0,9));
readA0();

return(1);
}


/*************************************************************
*  void send_buffer(void) DEF
*	write_target operations simply write to a buffer. Then
*	from time to time the buffer gets flushed down to the
*	target by calling this function.
*/
void send_buffer(void)
{
int wds;
Ulong addr,eadr,v;
static send_buffer_active;
Uchar *p;

if (send_buffer_active) return;
if (target_stopped != 2) wakeup();

if (ice.buf==0) ice.buf = (Uchar *)malloc(BUFSZ);
if (ice.bcnt==0 && ice.rlcnt==0) return;
send_buffer_active=1;
printDiag(3,"send_buffer()\n");
if (is_cacheable(ice.start_addr)) dcache_dirty = 1;

printDiag(1,"send_buffer: sadr=%08x bcnt=%d rlcnt=%d\n",
	ice.start_addr,ice.bcnt,ice.rlcnt);

if (ice.rlcnt && ice.rlcnt <= RLTHRESHOLD) /* expand rlcnt in buffer */
	for (;ice.rlcnt;ice.rlcnt--) ice.buf[ice.bcnt++] = ice.prev_v;

addr = ice.start_addr;
eadr = addr+ice.bcnt;
p = ice.buf;

if (addr == 0xbfff0000 && ice.bcnt == 4)
	printDiag(1,"[%02x,%02x,%02x,%02x]\n",p[0],p[1],p[2],p[3]);

/* send bytes until word aligned address */
while (addr&3 && addr < eadr) send_byte(addr++,*p++);

/* compute number of complete words */
if (addr == eadr) wds = 0;
else wds = ((eadr&~3) - addr)/4;

printDiag(1,"send_buffer: eadr=%08x wds=%d\n",eadr,wds);

if (wds > 3) {
#if defined(EJTAG) && !defined(PMCC)
	if (ejtag_mode) swap32n(p,wds);
#endif
	if (lendian_target) swap32n(p,wds);
	send_words(addr,(Ulong *)p,wds);
	addr += wds*4;
	p += wds*4;
	}
else for (;wds;wds--,addr += 4,p += 4) {
	v = read_unaligned_long(p);
#if defined(EJTAG) && !defined(PMCC)
	if (ejtag_mode) swap32n((Uchar *)&v,1);
#endif
	if (lendian_target) swap32n((Uchar *)&v,1);
	send_bhw(addr,v,4);
	}

/* send trailing bytes */
while (addr < eadr) send_byte(addr++,*p++);

if (ice.rlcnt > RLTHRESHOLD) set_bytes(addr,ice.prev_v,ice.rlcnt);

setFlushneeded(ice.start_addr,ice.rlcnt+ice.bcnt);
ice.rlcnt = ice.bcnt = 0;
send_buffer_active=0;
}

/*************************************************************
*  void write_target_byte(Ulong addr,Ulong v) DEF
*	write_target() calls this function to add bytes to the
*	buffer. send_buffer() is used to actually send the bytes
*	to the target.
*/
void write_target_byte(addr,v)
Ulong addr,v;
{

printDiag(5,"write_target_byte(%08x,%02x)\n",addr,v&0xff);
if (addr != ice.expected_addr) send_buffer();

v &= 0xff;
if (v == ice.prev_v && ice.bcnt) {
	ice.rlcnt++;
	if (ice.rlcnt >= MAXRL) send_buffer();
	}
else {
	if (ice.rlcnt) {
		if (ice.rlcnt <= RLTHRESHOLD) { /* expand in place */
			for (;ice.rlcnt;ice.rlcnt--) 
				ice.buf[ice.bcnt++] = ice.prev_v;
			if (ice.bcnt >= BUFSZ) send_buffer();
			}
		else send_buffer();
		}
	if (ice.bcnt == 0) ice.start_addr = addr;
	ice.buf[ice.bcnt++] = ice.prev_v = (Uchar)v;
	if (ice.bcnt+RLTHRESHOLD >= BUFSZ) send_buffer();
	}
ice.expected_addr = addr+1;
}

/*************************************************************
*  void send_byte(Ulong addr,Uchar v) DEF
*/
void send_byte(Ulong addr,Uchar v)
{
int offs,diff;
OcmRec *p;

printDiag(5,"send_byte(%08x,%02x)\n",addr,v);

if ((p=is_ocm(addr)) && p->func) {
	run_ocm(p,1,addr,1,v);
	return;
	}
offs=0;
diff = addr-current_r9;
if (diff < -32768 || diff > 32767) writeGpr(9,addr);
else offs = (addr-current_r9)&0xffff;
writeGpr(8,v);
send_instr(SB(8,offs,9));
readA0();
}


/*************************************************************
*  void set_bytes(Ulong addr,Uchar v,int n) DEF
*	Set multiple bytes e.g. fill memory
*	May also be used during download.
*/
void set_bytes(Ulong addr,Uchar v,int n)
{
OcmRec *p;
int offset;

printDiag(4,"set_bytes(%08x,%02x,%d)\n",addr,v,n);

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	if (addr&3) {
		writeGpr(9,addr);
		writeGpr(2,v);
		for (offset=0;addr&3;addr++,offset++,n--) {
			send_instr(SB(2,offset,9));
			}
		readA0();
		}
	if (!EjtagDownload(addr,n/4)) {
		printDiag(0,"EjtagDownload: failed\n");
		return;
		}
	if (!EjtagControl(EJTAG_FILL,(v<<24)|(v<<16)|(v<<8)|v,0)) {
		printDiag(0,"EjtagControl: FILL: failed\n");
		return;
		}
	addr += n;
	n &= 3;
	if (n>0) {
		writeGpr(9,addr);
		writeGpr(2,v);
		for (offset=0;n>0;offset++,n--) {
			send_instr(SB(2,offset,9));
			}
		readA0();
		}
	return;
	}
#endif

if ((p=is_ocm(addr)) && p->func) {
	for (;n>0;n--) run_ocm(p,1,addr++,1,v);
	return;
	}

writeGpr(9,addr);
writeGpr(8,n);
writeGpr(2,v);
send_instr(SB(2,0,9));
send_instr(SUBIU(8,1));
send_instr(BNE(8,0,-3)); /* self=-1 */
send_instr(ADDIU(9,9,1));
readA0();
current_r8 = current_r9 = 0;
}

/*************************************************************
*  void send_bhw(Ulong addr,Ulong v,int sz) DEF
*	980704 Created.
*
*	This was created to provide a quick solution for a specific
*	problem. The problem is that my write buffer discards info
*	about the size of the write access (bhw). This is not normally a
*	problem for memory. But can be a big problem for device registers.
*
*	The correct solution is to rewrite the buffer so that it remembers
*	what size data it contains. Writes of a different size would
*	cause a send_buffer(). But this is complicated to get the RL
*	encoding right for this scheme. Especially the expand-in-place.
*
*	So for now I am handling write hword as a special case. I always
*	send_buffer(), then I write the hword directly, bypassing the
*	buffer.
*/
void send_bhw(addr,v,sz)
Ulong addr;
Ulong v;
int sz;
{
int offs,diff;
OcmRec *p;

#ifdef VERBOSE_DOWNLOAD
printDiag(1,"send_bhw(%08x,%08x,%d)\n",addr,v,sz);
#endif

if ((p=is_ocm(addr)) && p->func) {
	run_ocm(p,1,addr,sz,v);
	return;
	}
offs=0;
diff = addr-current_r9;
if (diff < -32768 || diff > 32767) writeGpr(9,addr);
else offs = (addr-current_r9)&0xffff;
writeGpr(8,v);
switch (sz) {
	case 1 : send_instr(SB(8,offs,9)); break;
	case 2 : send_instr(SH(8,offs,9)); break;
	case 4 : send_instr(SW(8,offs,9)); break;
	}
readA0();
}

#if 0 /* 980704 Replaced by send_bhw */
/*************************************************************
*  void send_word(Ulong addr,Ulong v) DEF
*/
void send_word(addr,v)
Ulong addr;
Ulong v;
{
int offs,diff;
OcmRec *p;

#ifdef VERBOSE_DOWNLOAD
printDiag(1,"send_word(%08x,%08x)\n",addr,v);
#endif

if ((p=is_ocm(addr)) && p->func) {
	run_ocm(p,1,addr,4,v);
	return;
	}
offs=0;
diff = addr-current_r9;
if (diff < -32768 || diff > 32767) writeGpr(9,addr);
else offs = (addr-current_r9)&0xffff;
writeGpr(8,v);
send_instr(SW(8,offs,9));
readA0();
}
#endif

/*************************************************************
*  void send_words(Ulong addr,Ulong *vp,int n) DEF
*	send multiple words i.e. download
*/
void send_words(addr,vp,n)
Ulong addr;
Ulong *vp;
int n;
{
int gwp_offs;
OcmRec *p;
Ulong v;

printDiag(1,"send_words(%08x,%08x..,%d)\n",addr,*vp,n);

#if defined(EJTAG) && !defined(PMCC)
if (ejtag_mode) {
	printDiag(1,"EjtagDownload(%08x,%d)\n",addr,n);
	if (!EjtagDownload(addr,n)) {
		printDiag(0,"EjtagDownload: failed\n");
		return;
		}
	printDiag(1,"EjtagControl(BLOCK,0,%08x)\n",vp);
	if (!EjtagControl(EJTAG_BLOCK,0,vp)) {
		printDiag(0,"EjtagControl: BLOCK: failed\n");
		return;
		}
#if 0
	printDiag(1,"EjtagControl:WORD");
	for (;n>0;n--) {
		printDiag(1,"%08x ",*vp);
		if (!EjtagControl(EJTAG_WORD,*vp++,0)) {
			printDiag(0,"EjtagControl: WORD: failed\n");
			return;
			}
		}
	printDiag(1,"\n");
#endif
	return;
	}
#endif

if ((p=is_ocm(addr)) && p->func) {
	for (;n>0;n--,addr += 4) run_ocm(p,1,addr,1,*vp++);
	return;
	}

getSAP();
if (savearea.vers > 1) gwp_offs = ICE_GWP;
else gwp_offs = ICE_OLD_GWP;

if (savearea.gwp==0) {
	savearea.gwp = read_target(XT_MEM,(Ulong)(gwp_offs+savearea.sap),4);
	if (savearea.gwp==0) {
		printDiag(0,"fatal error: can't get address of get_word\n");
		return;
		}
	printDiag(1,"gwp %08x\n",savearea.gwp);
	if (!noforcek1) savearea.gwp |= K1BASE;
	}

send_instr(MOVE(10,31)); /* save RA -- non-leaf */
writeGpr(9,addr);	/* start address */
writeGpr(8,n);		/* word count */
writeGpr(4,savearea.gwp);

/* 4: */
send_instr(JALR_R4);     /* jal get_word -> v0 */
send_instr(SUBIU(8,1));   /* word count-- */
send_instr(SW(2,0,9));    /* store the word */
send_instr(BNE(8,0,-4)); /* self=-1 */
send_instr(ADDIU(9,9,4));   /* next word address */

send_instr(MOVE(31,10)); /* restore RA */

send_instr(SENDA0);	 /* execute it */
instr_buffer_cnt = 0;
PUT_BYTES((Uchar *)vp,n*4);  /* send data */
if (!readWord(&v)) {	 /* wait for done */
	printDiag(0,"send_words: readWord failed\n");
	}
current_r8 = current_r9 = 0;
}

/*************************************************************
*  int is_bptReq(Ulong addr,Ulong v) DEF
*	Changes a write_target into a setbp or clrbp
*/
is_bptReq(addr,v)
Ulong addr,v;
{
int n;

#if 0 /* old */
if (breakcode && (v == breakcode)) setbp_target(-1,BPTYPE_PC,addr,0,0);
#else /* 980825 new */
#ifdef PMCC
if (!(is_xvwmode() || is_gdbmode())) return(0);

if ((is_xvwmode() && v == XVWBREAKCODE) ||
    (is_gdbmode() && (v == DBXBREAKCODE || v == GDBBREAKCODE))
    ) setbp_target(-1,BPTYPE_PC,addr,0,0);
#else /* dll */
if (breakcode && (v == breakcode)) setbp_target(-1,BPTYPE_PC,addr,0,0);
#endif
#endif
else if ((n=which_bpt(addr,0)) != -1) clrbp_target(n);
else return(0);
printDiag(1,"is_bptReq(%08x,%08x)\n",addr,v);
return(1);
}

#if 0
if (xvwmode && v == XVWBREAKCODE
gdbmode && v == GDBBREAKCODE
if (!(is_xvwmode() || is_gdbmode())) return(0);

if ((is_xvwmode() && v == XVWBREAKCODE) ||
    (is_gdbmode() && (v == DBXBREAKCODE || v == GDBBREAKCODE))
    ) setbp_target(-1,BPTYPE_PC,addr,0,0);
#endif

/*************************************************************
*  void write_target(int type,Ulong addr,Ulong v,int sz) DEF
*/
void write_target(int type,Ulong addr,Ulong v,int sz)
{
int reg;

printDiag(1,"write_target(%s,%08x,%08x,%d)\n", xtcodes[type],addr,v,sz);

reg = addr;
switch (type) {
	case XT_PC :
	   switch (reg) {
		case 0 :
		   getSAP();
		   if (savearea.vers > 1)
		   write_target(XT_MEM,(Ulong)(ICE_EPC+savearea.sap),v,4);
		   else
		   write_target(XT_MEM,(Ulong)(ICE_OLD_EPC+savearea.sap),v,4);
		   break;
		case 1 :
			writeGpr(8,v);
			send_instr(OP_MTHI|(8<<21));
			break;
		case 2 :
			writeGpr(8,v);
			send_instr(OP_MTLO|(8<<21));
			break;
		}
		break;  /* 990319 */
	case XT_MEM :
		if (!savearea.sap) getSAP(); /* 980818 */
		/* Be sure that sap info is known before starting to
		 * fill the buffer. Otherwise download can fail.
		 */
#if 0  
	/* 981222 This screws up 4011 downloads. The 3 bytes at the beginning
	of each block gets corrupted. I can't see why I thought I needed it. */
		if (is_cacheable(addr) && has_wb_dcache && dcache_dirty)
			flush_target(DCACHE);
#endif
		switch (sz) {
			case 1 : write_target_byte(addr,v); break;
			case 2 : /* 980704 This is a hack */
				send_buffer();
				send_bhw(addr,v,2);
				break;
			case 4 : 
				if (is_bptReq(addr,v)) break;

				if (lendian_target) swap32n((Uchar *)&v,1);
				write_target_byte(addr,v>>24);
				write_target_byte(addr+1,v>>16);
				write_target_byte(addr+2,v>>8);
				write_target_byte(addr+3,v);
				break;
			}
		break;
	case XT_GPR : 
		writeUgpr(reg,v);
		break;
	case XT_CP0 : 
		writeGpr(8,v);
		send_instr(MTC0(8,reg));
		v = readA0();
		break;
	case XT_CP1 :
		writeGpr(8,v);
		send_instr(OP_MTC1|(8<<16)|(reg<<11));
		v = readA0();
		break;
	case XT_CP1C :
		writeGpr(8,v);
		send_instr(OP_CTC1|(8<<16)|(reg<<11));
		v = readA0();
		break;
	case XT_DBX :
		writeGpr(8,v);
		send_instr(MTD(8,reg));
		v = readA0();
		break;
	case XT_CP2 :
	case XT_CP2C :
	case XT_CP3 : 
	case XT_CP3C : 
	default : printDiag(0,"%08x: write_target: bad type\n",type);
	}
}

/*************************************************************
*  int writeGpr(int reg,Ulong v) DEF
*/
int writeGpr(reg,v)
int reg;
Ulong v;
{
long n;
int t,cnt;

printDiag(2,"writeGpr(%d,%08x)\n",reg,v);
if (reg == 8) current_r8 = v;
else if (reg == 9) current_r9 = v;

t = cnt = 0;
n = (long)v;
send_buffer();
if (n >= -32768 && n <= 32767) {
	send_instr(OP_ADDIU|(reg<<16)|(0<<21)|v&0xffff);
	cnt++;
	}
else {
	if (v&0xffff0000) {
		send_instr(OP_LUI|(reg<<16)|(v>>16));
		t = reg;
		cnt++;
		}
	if (v&0xffff) {
		send_instr(OP_ORI|(reg<<16)|(t<<21)|(v&0xffff));
		cnt++;
		}
	}
return(cnt);
}

/*************************************************************
*  Ulong read_target(int type,Ulong addr,int sz) DEF
*/
Ulong read_target(int type,Ulong addr,int sz)
{
int reg,offs,diff;
Ulong v;
OcmRec *p;

printDiag(1,"read_target(%s,%08x,%d) ",xtcodes[type],addr,sz);
send_buffer();
reg = addr;
switch (type) {
	case XT_PC :
		switch (reg) {
		   case 0 : /* PC */
			getSAP();
			if (savearea.vers > 1)
			return read_target(XT_MEM,(Ulong)(ICE_EPC+savearea.sap),4);
			else
			return read_target(XT_MEM,(Ulong)(ICE_OLD_EPC+savearea.sap),4);
		   case 1 : /* HI */
			send_instr(OP_MFHI|(4<<11));
			return readA0();
		   case 2 : /* LO */
			send_instr(OP_MFLO|(4<<11));
			return readA0();
		   }
	case XT_MEM :
		if ((p=is_ocm(addr)) && p->func) {
			return run_ocm(p,0,addr,sz,v);
			}
		if (sz == 4 && addr&3) { /* 970401 half-word aligned */
			addr &= ~1; /* clear LS bit */
			v = read_target(XT_MEM,addr,2)<<16;
			v |= read_target(XT_MEM,addr+2,2);
			return(v);
			}
		offs=0;
		diff = addr-current_r9;
		if (diff < -32768 || diff > 32767) writeGpr(9,addr);
		else offs = (addr-current_r9)&0xffff;
		switch (sz) {
			case 1 : send_instr(OP_LBU|(4<<16)|(9<<21)|offs); break;
			case 2 : send_instr(OP_LHU|(4<<16)|(9<<21)|offs); break;
			case 4 : send_instr(LW(4,offs,9)); break;
			}
		return readA0();
	case XT_GPR :
		return readUgpr(reg);
	case XT_CP0 : 
		send_instr(MFC0(4,reg));
		return readA0();
	case XT_CP1 :
		send_instr(OP_MFC1|(4<<16)|(reg<<11));
		return readA0();
	case XT_CP1C :
		send_instr(OP_CFC1|(4<<16)|(reg<<11));
		return readA0();
	case XT_DBX :
		send_instr(MFD(4,reg));
		return readA0();
		break;
	case XT_CP2 :
	case XT_CP2C :
	case XT_CP3 : 
	case XT_CP3C : 
	default : printDiag(0,"%08x: read_target: bad type\n",type);
	}
return(0);
}

/*************************************************************
*  Ulong readUgpr(int reg);
*	Read user's GP register
*/
Ulong readUgpr(int reg)
{
Ulong msk,v;
int i,n;

printDiag(1,"readUgpr(%d)\n",reg);

getSAP();
send_buffer();
if (savearea.vers == 1) {
	switch (reg) {
	    case 2 : /* v0 */
		v = read_target(XT_MEM,(Ulong)(ICE_V0+savearea.sap),4);
		break;
	    case 4 : /* a0 */
		v = read_target(XT_MEM,(Ulong)(ICE_A0+savearea.sap),4);
		break;
	    case 5 : /* a1 */
		v = read_target(XT_MEM,(Ulong)(ICE_A1+savearea.sap),4);
		break;
	    case 8 : /* t0 */
		v = read_target(XT_MEM,(Ulong)(ICE_T0+savearea.sap),4);
		break;
	    case 9 : /* t1 */
		v = read_target(XT_MEM,(Ulong)(ICE_T1+savearea.sap),4);
		break;
	    case 10 : /* t2 */
		v = read_target(XT_MEM,(Ulong)(ICE_T2+savearea.sap),4);
		break;
	    case 16: /* s0 */
		v = read_target(XT_MEM,(Ulong)(ICE_S0+savearea.sap),4);
		break;
	    case 31: /* ra */
		v = read_target(XT_MEM,(Ulong)(ICE_RA+savearea.sap),4);
		break;
	    case 32 :
		send_instr(OP_MFHI|(4<<11));
		v = readA0();
		break;
	    case 33 :
		send_instr(OP_MFLO|(4<<11));
		v = readA0();
		break;
	    default :
		if (reg == 30 && no_gpr30) return(0);
		send_instr(OP_OR|(4<<11)|(0<<16)|(reg<<21));
		v = readA0();
		}
	return(v);
	}

if (reg == 30 && no_gpr30) return(0);
else if (reg == 32) {
	send_instr(OP_MFHI|(4<<11));
	v = readA0();
	}
else if (reg == 33) {
	send_instr(OP_MFLO|(4<<11));
	v = readA0();
	}
else if (savearea.map&(1<<reg)) { /* reg is in save area */
	msk = 1;
	for (i=n=0;i<reg;i++) {
		if (savearea.map&msk) n++;
		msk <<= 1;
		}
	printDiag(1,"readUgpr hsize=%d n=%d\n",savearea.hsize,n); 
	v = read_target(XT_MEM,(Ulong)(savearea.hsize+n+savearea.sap),4);
	}
else { /* reg not in save area */
	send_instr(OP_OR|(4<<11)|(0<<16)|(reg<<21));
	v = readA0();
	}
return(v);
}

/*************************************************************
*  void writeUgpr(int reg,Ulong v);
*	Write user's GP register
*/
void writeUgpr(int reg,Ulong v)
{
Ulong msk;
int i,n;

printDiag(1,"writeUgpr(%d,%08x)\n",reg,v);

getSAP();
send_buffer();
if (savearea.vers == 1) {
	switch (reg) {
	    case 2 : /* v0 */
		write_target(XT_MEM,(Ulong)(ICE_V0+savearea.sap),v,4);
		break;
	    case 4 : /* a0 */
		write_target(XT_MEM,(Ulong)(ICE_A0+savearea.sap),v,4);
		break;
	    case 5 : /* a1 */
		write_target(XT_MEM,(Ulong)(ICE_A1+savearea.sap),v,4);
		break;
	    case 8 : /* t0 */
		write_target(XT_MEM,(Ulong)(ICE_T0+savearea.sap),v,4);
		break;
	    case 9 : /* t1 */
		write_target(XT_MEM,(Ulong)(ICE_T1+savearea.sap),v,4);
		break;
	    case 10 : /* t2 */
		write_target(XT_MEM,(Ulong)(ICE_T2+savearea.sap),v,4);
		break;
	    case 16: /* s0 */
		write_target(XT_MEM,(Ulong)(ICE_S0+savearea.sap),v,4);
		break;
	    case 31: /* ra */
		write_target(XT_MEM,(Ulong)(ICE_RA+savearea.sap),v,4);
		break;
	    case 32 :
		writeGpr(8,v);
		send_instr(OP_MTHI|(8<<21));
		break;
	    case 33 :
		writeGpr(8,v);
		send_instr(OP_MTLO|(8<<21));
		break;
	    default :
		if (reg == 30 && no_gpr30) break;
		writeGpr(reg,v);
		}
	readA0();
	return;
	}

if (reg == 30 && no_gpr30) return;
else if (reg == 32) {
	writeGpr(8,v);
	send_instr(OP_MTHI|(8<<21));
	readA0();
	}
else if (reg == 33) {
	writeGpr(8,v);
	send_instr(OP_MTLO|(8<<21));
	readA0();
	}
else if (savearea.map&(1<<reg)) { /* reg is in save area */
	msk = 1;
	for (i=n=0;i<reg;i++) {
		if (savearea.map&msk) n++;
		msk <<= 1;
		}
	write_target(XT_MEM,(Ulong)(savearea.hsize+n+savearea.sap),v,4);
	}
else { /* reg not in save area */
	writeGpr(reg,v);
	readA0();
	}
}

/*************************************************************
*  void freeOcmRec(OcmRec *p)
*	add the OcmRec to freeOcmChain
*	I do this because the IMON interface in crt3.s doesn't
*	  support free();
*/
void freeOcmRec(OcmRec *p)
{
p->next = freeOcmChain;
freeOcmChain = p;
}

/*************************************************************
*  OcmRec *mallocOcmRec(void)
*	Try to get an OcmRec from freeOcmChain. else use malloc.
*/
OcmRec *mallocOcmRec(void)
{
OcmRec *p;

if (freeOcmChain) {
	p = freeOcmChain;
	freeOcmChain = p->next;
	return(p);
	}
return (OcmRec *)malloc(sizeof(OcmRec));
}

/*************************************************************
*  OcmRec *addOcmRec(Ulong addr,Ulong size,Ulong *func,int funclen);
*	arg0 = start address of Ocm region
*	arg1 = size (in bytes) of Ocm region
*	arg2 = optional address of access function. else 0.
*	arg3 = optional length (in words) of access function. else 0.
*/
OcmRec *addOcmRec(Ulong addr,Ulong size,Ulong *func,int funclen)
{
OcmRec *p;

printDiag(1,"addOcmRec(%08x,%08x,%08x,%d)\n",addr,size,func,funclen);
p = mallocOcmRec();
if (!p) {
        printDiag(0,"addOcmRec: malloc failure. Out of memory.\n");
        return(0);
        }
p->addr = addr;
p->size = size;
p->func = func;
p->funclen = funclen;
p->regmap_ok = 0;
p->readonly = 0;
p->map = 0;
p->next = ocmChain;
ocmChain = p;
return(p);
}

/*************************************************************
*  int deleteOcmRec(Ulong addr)
*/
int deleteOcmRec(Ulong addr)
{
OcmRec *p;

printDiag(2,"deleteOcmRec(%08x)\n",addr);
p = ocmChain;
if (!p) return(0);
if (p->addr == addr) { /* 1st item in chain */
	ocmChain = p->next;
	freeOcmRec(p);
	return(1);
	}
for (p=ocmChain;p->next;p=p->next) {
	if (p->next->addr == addr) {
		p->next = p->next->next;
		freeOcmRec(p->next);
		return(1);
		}
	}
return(0);
}

/*************************************************************
*  OcmRec *is_ocm(Ulong addr) DEF
*       return addr of OcmRec if addr falls within an ocm region.
*/
OcmRec *is_ocm(addr)
Ulong addr;
{
OcmRec *p;

for (p=ocmChain;p;p=p->next) {
        if (addr >= p->addr && addr < p->addr+p->size) {
		printDiag(2,"%08x=ocm\n",addr);
		return(p);
		}
        }
return(0);
}

/*************************************************************
*  Ulong run_ocm(OcmRec *ocm,int mode,Ulong addr,int sz,Ulong value);
*	Run the designated ocm routine.
*	mode: 0=read 1=write
*/
Ulong run_ocm(OcmRec *ocm,int mode,Ulong addr,int sz,Ulong value)
{
int i;
Ulong need,r;
Ulong saved[32];

printDiag(1,"run_ocm(x,%s,%08x,%d,%08x)\n",(mode)?"write":"read",addr,sz,value);

/* Can't do this if there's no function */
if (ocm->func == 0 || ocm->funclen == 0) {
	printDiag(1,"run_ocm: ERROR: func=%08x funclen=%d\n",
		ocm->func,ocm->funclen);
	return(0);
	}

#if 0
if (savearea.vers == 1) {
	printDiag(1,"ocm %08x needs new format savearea.\n",addr);
	return(0);
	}
#endif

/* return if this is a write and ocm region is readonly */
if (mode == 1 && ocm->readonly) {
	printDiag(1,"run_ocm: ERROR: write to ro ocm\n");
	return(0);
	}

/* must know regmap and ibs of kernel */
if (savearea.map == 0) savearea.map = kernel_map;
if (savearea.ibs == 0) savearea.ibs = kernel_ibs;
if (savearea.map == 0 || savearea.ibs == 0) {
	printDiag(1,"run_ocm: ERROR: map=%08x ibs=%d\n",
		savearea.map,savearea.ibs);
	return(0);
	}

if (ocm->funclen > ((int)savearea.ibs)) {
	printDiag(1,"ocm %08x func too large for ibuf.\n",addr);
	return(0);
	}

if (!ocm->regmap_ok) {
	send_buffer();
	if (ocm->map == 0) {
		writeGpr(4,2); /* regmap function */
		for (i=0;i<ocm->funclen;i++) send_instr(ocm->func[i]);
		ocm->map = readA0();
		}
	printDiag(1,"ocm %08x samap %08x ocmmap %08x.\n",
				addr,savearea.map,ocm->map);
	/* savearea.map is what the kernel saves */
	/* ocm.map is what the ocm routine needs */
	need = (~(savearea.map&ocm->map)) & ocm->map;
	/* need are those extra registers that must be saved */
	if (need) {
#if 0
	if ((savearea.map&regmap) != regmap) {
		/* incompatible regmaps */
		printDiag(1,"ocm %08x bad regmap %08x %08x.\n",
				addr,savearea.map,regmap);
		return(0);
#endif
		printDiag(1,"ocm %08x need %08x\n",addr,need);
		send_buffer();
		for (i=0;i<32;i++) {
			if (need&(1<<i)) {
				send_instr(OP_OR|(4<<11)|(0<<16)|(i<<21));
				saved[i] = readA0();
				printDiag(1,"ocm %08x save $%d %08x\n",
					addr,i,saved[i]);
				}
			}
		}
	if (!need) ocm->regmap_ok = 1;
	/* regmap_ok indicates that no extra saves are necessary */
	}

send_buffer();
writeGpr(4,mode);
writeGpr(5,addr);
writeGpr(6,sz);
writeGpr(7,value);
for (i=0;i<ocm->funclen;i++) send_instr(ocm->func[i]);
r = readA0();
/* restore extra regs */
if (need) {
	for (i=0;i<32;i++) if (need&(1<<i)) {
		printDiag(1,"ocm %08x restore $%d %08x\n",
			addr,i,saved[i]);
		writeGpr(i,saved[i]);
		}
	readA0();
	}
return(r);
}

/*************************************************************
*  Ulong domt1(Ulong sadr,Ulong eadr)
*	Do Memory test part1
*/
Ulong domt1(Ulong sadr,Ulong eadr)
{
/* domt1: The gpr[10]s in this func were 26 (k0) */
/* domt1: The gpr[11]s in this func were 27 (k1) */
send_buffer();
/* part1: walking ones */
writeGpr(4,sadr);
writeGpr(9,eadr);
/* 1: */
writeGpr(10,1);
/* 2: */
send_instr(SW(10,0,4));
send_instr(LW(11,0,4));
send_instr(0);
send_instr(BNE(10,11,6)); /* 1f */
send_instr(0);
send_instr(SLLV(10,1));
send_instr(BNE(10,0,-7)); /* 2b */
send_instr(0);
send_instr(ADDIU(4,4,4));
send_instr(BNE(4,9,-11)); /* 1b */
/* 1: */
return readA0();
}

/*************************************************************
*  int domt2(Ulong sadr,Ulong eadr)
*	Do Memory test part2
* 	part2: write address to each word
*/
void domt2(Ulong sadr,Ulong eadr)
{
send_buffer();
writeGpr(4,sadr);
writeGpr(9,eadr);
/* 1: */
send_instr(SW(4,0,4));
send_instr(ADDIU(4,4,4));
send_instr(BNE(4,9,-3)); /* 1b */
readA0();
}

/*************************************************************
*  Ulong domt3(Ulong sadr,Ulong eadr)
*	Do Memory test part3
* 	part3: check each word contains address
*/
Ulong domt3(Ulong sadr,Ulong eadr)
{

/* domt3: The gpr[10]s in this func were 26 (k0) */
send_buffer();
writeGpr(4,sadr);
writeGpr(9,eadr);
/* 1: */
send_instr(LW(10,0,4));
send_instr(0);
send_instr(BNE(10,4,3)); /* 1f */
send_instr(0);
send_instr(ADDIU(4,4,4));
send_instr(BNE(4,9,-6)); /* 1b */
/* 1: */
return readA0();
}

