/*************************************************************
* File: i4102.s
* Purpose: An EJTAG initialization file for the BDMR4102 eval board.
* Author: Phil Bunce (pjb@carmel.com)
* Revision History:
*	981210	Created from k4102.s
*	990317	Added mrally's 990315 changes
*
*/

/*
 * This module is an example initialization file for use with the
 * BDMR4102 with an EJTAG connection when there is no software
 * installed on the board.
 *
 * This module is necessary in such cases because otherwise the
 * RAM memory will not be accessable by the debugger.
 *
 * This module should be built using the following command line:
 *
 *	pmcc -crt0 -T 0 -o i4102 i4102.s
 *
 * This will generate a file named i4102.rec. This file should be
 * named as the third argument to the 'conn' command in the 
 * serialice1.cfg configuration file. eg.
 *
 * 	conn ejtag i4102.rec
 *
 */

/* This file establishes the following memory map:
 *
 *                           phy          keg0            kseg1
 * ROM    2MB  starting at 1fc00000	bfc00000	9fc00000
 * SDRAM  16MB starting at 00000000	80000000 	a0000000
 * SRAM  128KB starting at 0e000000	8e000000 	ae000000
 *
 */

/* enable this to enable the UART and 7-seg display via GP4 */
#define UART_BASE	0xbe000000 /* define the UART base address */

#ifndef LR4102
#define LR4102
#endif

#include <mips.h>


/*************************************************************
*	This is where the CPU-specific init code lives.
*	This implementation is for the bdmr4102 (4102 eval board).
*/
	.globl _start
	.ent _start
_start:

 # ============ Set up FLASH in addr space assign to CS0 ===================
 #
 # Note: FACMP0 and FACMP1 are initialized on startup to a 2M space starting
 # at address 0x1fc00000.   These registers only need to be changed if you
 # have a boot flash larger than 2 Meg in size.

        li      t1, M_FBUSTA
	li	t2, 0x00160111
        sw      t2, (t1)               # Set FBUSTA = 1 turn around time

        li      t1, M_FACFG0
        li      t0, 0x02949517         # Set 8 bit, 11 wait states     
        sw      t0, (t1)               # Store Flash Timing settings

 # ============ Setup SRAM in addr space assign to CS3    ==================

        li      t1, M_FACMP3     # Assign CS3 to SRAM in address range
        li      t0, 0x00010e00         # 0x0e000000 to 0x0e01ffff
        sw      t0, (t1)               # Assign CS3 address range

        li      t1, M_FACFG3     
        li      t0, 0xc2100005         # Set 32 bit, 3 wait states
        sw      t0, (t1)               # Store SRAM timing settings
        lw      zero, (t1)

 # Setup BBCC System Configuration Register 
 #
 # Initialize Bus Unit interface.  This routine enables all caches.

                li      s1, M_SCR1        
                li      s0, 0x300010db   # Enable Caches,
					 # snooping, turn off TLB
                sw      s0, (s1)         # Configure SCR1, pg size = 256
                                
 # Initialize SDRAM controller
 #
 # This routine inits the SDRAM controller on the 4102.  It begins by
 # configuring the address range for the SDRAM and all of the timing
 # parameters.  It then waits for 100 usec using the timer0 as a count
 # down.  Finally it issues a precharge to bank 0, 1, 2 & 3 followed by 2
 # refreshes and configures the SDRAM for 1 word burst.

 # The SDRAM controller strobes out a new address with each read request.

 # The SDRAM must start on a 32MB boundary.


	# Set SDRAM clock
	li      s0, M_SCR2        # Set bclkp to run at 
					# pclk speed
	lw	s2, (s0)		# Read current PLL jumper status
	and	s2, 0x00000010		# Mask off all but the PLL bit
	li      s1, 0x000200a8          # and dclkp to run at 
	or	s1, s2			# bclk or 100MHz 
	sw      s1, (s0)                # and enable 4101 
					# compatibility mode

	# Issue COMMAND INHIBIT by not reading or writing to 
	# DRAM and wait for 100 us

	li      s0, M_TMR4001+O_TIC0
	li      s1, M_TMR4001+O_TCC0

	li      s2, 0x00000100          # Setup Timer to wait 
					# 100us  0x2710
	sw      s2, (s0)                # Store data
	li      s0, M_TMR4001+O_TMODE
	li      s2, 0x00000011          # Enable Timer 0 Disable 
					# Timer 1
	sw      s2, (s0)                # Store in Register

1:
	lw      s2, (s1)                # Read from Counter
	bne     zero, s2, 1b            # Loop till done

	# Enable SDRAM in addr range 2

	li      s0, M_FACMP2		# Set FACMP2 to 0x00ff0100
	li      s2, 0x07ff0000          # Address range 2 covers 
					# 0x00000000 to
	sw      s2, (s0)                # 0x00ffffff, 16MB Dram

	# Set SDRAM Configuration

	li      s0, M_FSDRAM
	li      s2, 0x000d8609          # Set SDRAM to 4 banks, 
					# 15.6 usec refresh
	sw      s2, (s0)                # 8 bit page size, 4 Meg 
					# bank size 

	# Enable SDRAM
	li      s0, M_FACFG2
	li      s1, 0xd0000001          # enable SDRAM 32 bit 
					# wide bus         
	sw      s1, (s0)        

	# Set FSDRAMT timing register

	li      s0, M_FSDRAMT     # Set SDRAM Timing for 
					# Lat = 3 cks
	li      s2, 0x000057a4          # tRC = 2 cks, tCL = 3 cks, 
					# tRAS = 7 cks
	sw      s2, (s0)                # Set Init Bit so that next 
					# read is a bank 
					# Precharge.  Set for MODE 
					# Register write 
					# on next Write to SDRAM
						
	# Issue a Precharge to each bank of SDRAM

	li      s1, 0xa0000000          # Execute dummy read to 
					# uncached SDRAM
	lw      zero, (s1)              # Read Banks in order to 
					# precharge
 
	# Issue 2 Refresh cycles to SDRAM

	li      s0, M_FSDRAM
	li      s2, 0x000d8609          # Set SDRAM Config

	sw      s2, (s0)                # Store in FSDRAM to cause 
					# a refresh
	lw      zero, (s0)              # Flush write buffers
	ori     s0, s0, 0x0             # Flush write buffers

	sw      s2, (s0)                # Store in FSDRAM to cause 
					# a second refresh    
	lw      zero, (s0)              # Flush write buffers
	ori     s0, s0, 0x0             # Flush write buffers

	# Set SDRAM mode register
	li      t0, M_FSDRAMT
	lw	s2,(t0)
	or	s2,0x8000
	sw	s2,(t0)

	li      s1, 0xa008c000          # Set SDRAM Mode Reg to 
	lb	zero,(s1)
	li      t0, M_FSDRAMT
	and	s2,~0xc000
	sw	s2,(t0)

#ifdef UART_BASE

	# Setup Uart and Seven Segment Display assign to CS4

        la      t1, M_FACMP4           # Assign CS4 to UART/7SEG in addr range
	li	t0, ((UART_BASE>>16)&0x1fff)
        sw      t0, (t1)               # Set FACMP3 to 0x00001fbf

        la      t1, M_FACFG4
        li      t0, 0x0a50901f         # Set 8 bit, 15 wait states
        sw      t0, (t1)
        lw      zero, (t1)             # Flush Write buffer

        li      t0, 0xf8               # Light a 7
        la      t4, (UART_BASE+0x20)   # Led port address
        sw      t0, (t4)               # Light it!
#endif

	.end _start
