/*************************************************************
 * File: util/setupenv.c
 * Purpose: set up the environment variables
 * Copyright: BRECIS Communications Corporation
 * Revision History:
 *	200205	Start of revision history
 */

#include <mips.h>
#include <defines.h>
#include <brecis.h>
#include <string.h>
#include <flashconfig.h>

char *argv[MAXARGC];
char *envp[MAXENVP];

char cmdline[MAXCMDLINE];
char envparea[MAXENVPAREA];
char tmpcmdline[MAXCMDLINE];
char *tmpargv[MAXARGC];

int pcsetflag;

extern unsigned char FLASH_CONFIG_ADDRESS;

extern void identify_features(char *);

static int getline(char *flashloc, int flashindex, char *dest, int maxdest)
{
	int index = 0;	
	char c;
	char quote = 0;

	maxdest--;			/* leave room for zero byte */

	while (flashindex < FLASHCFGMAXLEN && index < maxdest)
	{
		c = flashloc[flashindex++];

		if (quote != 0)
		{
			/* ending quote */
			dest[index++] = c;
			if (quote == c)
			{
				quote = 0;
			}
			continue;
		}

		if (c == '\'' || c == '"')
		{
			/* beginning quote */
			quote = c;
			dest[index++] = c;
			continue;
		}

		if (c == '\n' || c == '\r' || c == '\0')
		{
			dest[index++] = '\0';
			return flashindex;
		}

		dest[index++] = c;
	}

	dest[index++] = '\0';

	return -1;			/* something is wrong with last line, return EOF */
}

struct defaultEnvpStruct {
		int		bit;
		char	*name;
};

#define NEEDHEAPTOP		(1<<0)
#define	NEEDCLKFREQ		(1<<1)
#define	NEEDMEMSIZE		(1<<2)
#define	NEEDCPUTYPE		(1<<3)
#define	NEEDETHERADDR0	(1<<4)
#define	NEEDETHERADDR1	(1<<5)
#define	NEEDETHERADDR2	(1<<6)
#define NEEDDEVICEID	(1<<7)
#define NEEDFEATURES	(1<<8)
#define NEEDPHYADDR0	(1<<9)
#define NEEDPHYADDR1	(1<<10)
#define NEEDPHYADDR2	(1<<11)
#define NEEDSERIALNO	(1<<12)
#define NEEDREALMEMSIZE	(1<<13)
#define	NEEDCLKFREQHZ	(1<<14)
#define	NEEDUARTFREQHZ	(1<<15)

/* order of entries must match order of bits */

static struct defaultEnvpStruct defaultEnvps[] = {
		{NEEDHEAPTOP, "heaptop"},
		{NEEDCLKFREQ, "clkfreq"},
		{NEEDMEMSIZE, "memsize"},
		{NEEDCPUTYPE, "cputype"},
		{NEEDETHERADDR0, "ethaddr0"},
		{NEEDETHERADDR1, "ethaddr1"},
		{NEEDETHERADDR2, "ethaddr2"},
		{NEEDDEVICEID, "deviceid"},
		{NEEDFEATURES, "features"},
		{NEEDPHYADDR0, "phyaddr0"},
		{NEEDPHYADDR1, "phyaddr1"},
		{NEEDPHYADDR2, "phyaddr2"},
		{NEEDSERIALNO, "serialno"},
		{NEEDREALMEMSIZE, "realmemsize"},
		{NEEDCLKFREQHZ, "clkfreqhz"},
		{NEEDUARTFREQHZ, "uartfreqhz"},
		{0, ""}			/* end of table */
};

#define MAXETHER		3

static unsigned char etherAddrIndex[MAXETHER] = {
		FLASHCONFIG_ETHERMAC0,
		FLASHCONFIG_ETHERMAC1,
		FLASHCONFIG_ETHERMAC2
};

static unsigned char phyAddrIndex[MAXETHER] = {
		FLASHCONFIG_ETHERPHY0,
		FLASHCONFIG_ETHERPHY1,
		FLASHCONFIG_ETHERPHY2
};

static int getFlashConfigString(flashConfigPtr *flashConfig, 
								flashConfigKey key,
								char *buf)
{
		flashConfigLength		valuelength;
		flashConfigValue		*valuep;

		if (getFlashConfigValue(flashConfig, key, &valuep, &valuelength))
		{
				return 0;		/* not found */
		}

		strcpy(buf, valuep);

		return 1;				/* found */
}

static int getEtherAddr(flashConfigPtr *flashConfig, int interface, char *buf)
{
		flashConfigLength		valuelength;
		flashConfigValue		*valuep;

		if (interface >= MAXETHER ||
			getFlashConfigValue(flashConfig,
								etherAddrIndex[interface],
								&valuep,
								&valuelength) ||
			valuelength != 6)
		{
				return 0;		/* not found */
		}

		sprintf(buf, "%02x:%02x:%02x:%02x:%02x:%02x",
				valuep[0], valuep[1], valuep[2],
				valuep[3], valuep[4], valuep[5]);

		return 1;				/* found */
}

static char *getDefaultString(int index)
{
		int i;

		for (i = 0; defaultEnvps[i].bit != 0; i++)
		{
				if (index == defaultEnvps[i].bit)
						return defaultEnvps[i].name;
		}

		return "unknown default";
}

static void addDynamicDefaultEnvps(unsigned long flags, char *ev, int envc)
{
		char			buf[80];
		flashConfigPtr	*flashConfig;

		flashConfig = validateFlashConfig(&FLASH_CONFIG_ADDRESS);

/* add by hook 2003/03/24	*/
		
		DBGREG[R_PC] = 0xbfc70010;
		pcsetflag = 1;      /* flag pc has been set */
				
		envp[envc++] = ev;
		sprintf(ev, "ethaddr0=00:50:18:00:00:01");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;

   		envp[envc++] = ev;
		sprintf(ev, "ethaddr1=00:50:18:00:00:02");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;
										
        envp[envc++] = ev;
		sprintf(ev, "ethaddr2=00:50:18:00:00:03");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;
										
		// add by hook for ethernet 3
        envp[envc++] = ev;
		sprintf(ev, "ethaddr3=00:50:18:00:00:04");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;

		// add by hook for ethernet 4
        envp[envc++] = ev;
		sprintf(ev, "ethaddr4=00:50:18:00:00:05");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;
		
        envp[envc++] = ev;
		sprintf(ev, "modetty0=57600,n,8,1,hw");
		a5000StrOut(ev);
		a5000StrOut(" dynamically added" CRLF);
		ev += strlen(ev) + 1;								

/* end of add by hook */

/* mark by hook 2003/03/24
		if (flashConfig == NULL)
		{
				sprintf(buf, "No manufacturing flash config at %lx" CRLF, 
						(unsigned long) &FLASH_CONFIG_ADDRESS);
				a5000StrOut(buf);
		}
*/

		if (flags & NEEDHEAPTOP)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%lx", getDefaultString(NEEDHEAPTOP), 
						(unsigned long) &ftext);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDCLKFREQ)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%d", getDefaultString(NEEDCLKFREQ), 
						cpuclockrate() / 1000000);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDCLKFREQHZ)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%d", getDefaultString(NEEDCLKFREQHZ), 
						cpuclockrate());
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDMEMSIZE)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=0x%lx", getDefaultString(NEEDMEMSIZE),
						topClientMem);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDREALMEMSIZE)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=0x%lx", getDefaultString(NEEDREALMEMSIZE),
						topRealMem);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDCPUTYPE)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%ld", 
						getDefaultString(NEEDCPUTYPE), cpuType());
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDDEVICEID)
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%08lx", 
						getDefaultString(NEEDDEVICEID), deviceID());
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDETHERADDR0 && 
			flashConfig != NULL && 
			getEtherAddr(flashConfig, 0, buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDETHERADDR0), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDETHERADDR1 && 
			flashConfig != NULL && 
			getEtherAddr(flashConfig, 1, buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDETHERADDR1), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDETHERADDR2 && 
			flashConfig != NULL && 
			getEtherAddr(flashConfig, 2, buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDETHERADDR2), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDFEATURES)
		{
				identify_features(buf);
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDFEATURES), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDPHYADDR0 && 
			flashConfig != NULL && 
			getFlashConfigString(flashConfig, phyAddrIndex[0], buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDPHYADDR0), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDPHYADDR1 && 
			flashConfig != NULL && 
			getFlashConfigString(flashConfig, phyAddrIndex[1], buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDPHYADDR1), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDPHYADDR2 && 
			flashConfig != NULL && 
			getFlashConfigString(flashConfig, phyAddrIndex[2], buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDPHYADDR2), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDSERIALNO && 
			flashConfig != NULL && 
			getFlashConfigString(flashConfig, FLASHCONFIG_SERIAL, buf))
		{
				envp[envc++] = ev;
				sprintf(ev, "%s=%s", getDefaultString(NEEDSERIALNO), buf);
				a5000StrOut(ev);
				a5000StrOut(" dynamically added" CRLF);
				ev += strlen(ev) + 1;
		}

		if (flags & NEEDUARTFREQHZ)
		{
			extern unsigned int	uart_freq;

			envp[envc++] = ev;
			sprintf(ev, "%s=%d", getDefaultString(NEEDUARTFREQHZ), uart_freq);
			a5000StrOut(ev);
			a5000StrOut(" dynamically added" CRLF);
			ev += strlen(ev) + 1;
		}

	envp[envc] = NULL;
}

/* copy argv, envp info from flash, set up the argv, enpv arrays */
/* return the argc count */

int setupcallargs()
{
	int	argc;
	int envc;
	unsigned long envpflags;
	int flashindex;
	int index;
	int prevflashindex;
	int tmpargc;
	char *av;
	char *ev;
	char *flash;

	/* initialize some global variables and the return value in case we fail */

	argv[0] = NULL;
	cmdline[0] = '\0';
	argc = 0;
	argv[argc] = NULL;

	ev = envparea;
	envc = 0;
	envpflags = ~0;

	envp[envc] = NULL;

	a5000StrOut(CRLF "Processing flash configuration." CRLF);

	/* validate flash area */

	flash = (char *) FLASHCFG;

	if (strncmp(flash, FLASHCFGSTRING, strlen(FLASHCFGSTRING)) != 0)
	{
		a5000StrOut("Flash configuration needs to start with ");
		a5000StrOut(FLASHCFGSTRING);
		a5000StrOut(" at address ");
		a5000PrintWordReg((unsigned long) flash);
		a5000StrOut(CRLF);
		addDynamicDefaultEnvps(envpflags, ev, envc);
		return argc;
	}

	flashindex = 0;
	
	while (flashindex < FLASHCFGMAXLEN)
	{
		prevflashindex = flashindex;

		flashindex = getline(flash, flashindex, tmpcmdline, MAXCMDLINE);
		a5000StrOut(tmpcmdline);
		a5000StrOut(CRLF);
		
		if (flashindex < 0)
		{
			break;
		}

		/* we have a command to process */

		tmpargc = argvize(tmpargv, MAXARGC-1, tmpcmdline);

		if (tmpargc <= 0)
		{
			continue;
		}

		if (tmpargv[0][0] == '#')
		{
			/* if a comment */

			if (strcmp(FLASHCFGENDSTR, tmpargv[0]) == 0)
			{
				break;		/* if EOF */
			}

			continue;
		}

		if (strcmp(FLASHCMDSETARGS, tmpargv[0]) == 0)
		{
			argc = 0;
			av = cmdline;
			for (index = 1; index < tmpargc; index++)
			{
				/* copy command arguments to permanent command line */

				argv[argc++] = av;
				strcpy(av, tmpargv[index]);
				av += strlen(tmpargv[index]) + 1;
			}

			argv[argc] = NULL;	/* insure NULL pointer ends array */

			continue;
		}

		if (strcmp(FLASHCMDSET, tmpargv[0]) == 0) switch (tmpargc)
		{
		case 4:
			/* Set environment variable to value of another var */
			if (tmpargv[1][0] != '-' || tmpargv[1][1] != 'i')
				break;
			{
				int	len = strlen(tmpargv[3]);

				index = 0;
				while (envp[index])
				{
					if (strncmp(envp[index], tmpargv[3], len) == 0 &&
						envp[index][len] == '=')
					{
						tmpargv[1] = tmpargv[2];
						tmpargv[2] = &envp[index][len+1];
						index = -1;
						break;
					}
					++index;
				}
			}
			if (index != -1)
				break;
			/* Fall through to normal set */
		case 3:
			/* add environment variable to environment array */
			envp[envc++] = ev;	/* set array pointer */
			envp[envc] = NULL;	/* insure NULL pointer ends array */

			/* copy variable=data to environment area */
			
			strcpy(ev, tmpargv[1]);
			ev += strlen(tmpargv[1]);
			*ev++ = '=';
			strcpy(ev, tmpargv[2]);
			ev += strlen(tmpargv[2]) + 1;

			for (index = 0; defaultEnvps[index].bit != 0; index++)
			{
					if (strcmp(tmpargv[1], defaultEnvps[index].name) == 0)
					{
							envpflags &= ~defaultEnvps[index].bit;
							break;
					}
			}
			
			continue;

		default:
			break;
		}

		if (strcmp(FLASHCMDSETPC, tmpargv[0]) == 0 && tmpargc == 2)
		{
			/* set pc for kernel to start */

			char c;
			unsigned long pc = 0;

			for (index = 0; tmpargv[1][index] != 0; index++)
			{
				c = tmpargv[1][index];

				if (c >= '0' && c <= '9')
				{
					pc = pc * 16 + c - '0';
				}
				else if (c >= 'a' && c <= 'f')
				{
					pc = pc * 16 + c - 'a' + 10;
				}
				else if (c >= 'A' && c <= 'F')
				{
					pc = pc * 16 + c - 'A' + 10;
				}
			}

			DBGREG[R_PC] = pc;
			pcsetflag = 1;		/* flag pc has been set */

			continue;
		}

		a5000StrOut("flash configuration command <");
		a5000StrOut(tmpargv[0]);
		a5000StrOut("> not understood" CRLF "      found at flash location ");
		a5000PrintWordReg((unsigned long) &flash[prevflashindex]);
		a5000StrOut(CRLF "      Rest of flash configuration ignored." CRLF);
		break;
	}

	addDynamicDefaultEnvps(envpflags, ev, envc);

	a5000StrOut("Finished processing flash configuration." CRLF);
	return argc;
}


/*************************************************************
*  setQFlush()
*/
void setQFlush()
{
	char **envpp = envp;
	char *ev = NULL;
	char featurepci = FEATURE_NOEXIST;
	char featurespad = FEATURE_NOEXIST;
	char *features = NULL;
	int	featureslen = strlen(FEATURES);

	a5000StrOut("Finish initializing environment" CRLF);

	while (*envpp != NULL)
	{
		ev = *envpp;
		if (strncmp(ev, FEATURES, featureslen) == 0 &&
			ev[featureslen] == '=')
		{
			features = &ev[featureslen+1];
			break;
		}
		envpp++;
	}

	if (features == NULL)
		a5000StrOut(FEATURES " environment variable not found." CRLF);
	else
	{
		a5000StrOut(ev);
		a5000StrOut(" environment variable found." CRLF);
	}

	while (*features)
	{
		if (*features == SPAD_KEY)
			featurespad = features[1];
		else if (*features == PCI_KEY)
			featurepci = features[1];

		features += 2;
	}

	if (featurepci == FEATURE_NOEXIST && featurespad == FEATURE_NOEXIST)
		return;

	*QFLUSH_REG_1 &= ~(QFLUSH_PCI_BIT | QFLUSH_SCR_BIT);
	*QFLUSH_REG_1 |= ((featurepci != FEATURE_NOEXIST ? QFLUSH_PCI_BIT : 0) |
					  (featurespad != FEATURE_NOEXIST ? QFLUSH_SCR_BIT : 0));
}



