#include <defines.h>

extern int printf(char *fmt, ...);

/* nanodelay - Routine to perform short-duration delays using the MIPS
 *      count register.
 *
 * Written by: Mark Rustad, Brecis Communications, 07/31/01.
 */


/* In-line macro for COP0 count register access */

#define GetCOP0Count()  \
({ unsigned long __value;       \
        __asm__ volatile ("mfc0 %0, $9" : "=r" (__value));    \
        __value; })

#define OVERHEAD    3   /* Partial overhead adjustment */

static struct
{
    unsigned long   numerator;
    unsigned long   denominator;
}   nanoperiod;

/***************************************************************************
*
* NanoDelay - Delay for a short time.
*
* Input:
*  nanos - Number of nanoseconds to delay
*
* Notes:
*  NanoInit must have been called to establish the CPU clock rate
*  before calling this function.
*
*  The caller probably should have interrupts disabled before calling
*  this to have any expectation of real accuracy.
*
* RETURNS: N/A
*
*/

void NanoDelay
	(
	unsigned int nanos
	)
	{
	unsigned long   count;
	long    difference;

	count = GetCOP0Count(); /* Get the count before doing calculations */
	count += ((nanos * nanoperiod.numerator) / nanoperiod.denominator)
		- OVERHEAD;
	do
		difference = count - GetCOP0Count();
	while (difference > 0);
	}



/***************************************************************************
*
* NanoInit - Set timing parameters for NanoDelay.
*
*
* Input:
*  freq - CPU clock frequency in Hz.
*
* Notes:
*  This routine takes the 2:1 ratio between the cpu clock and count
*  register into account, so the caller uses only the actual cpu clock
*  period to initialialize this routine.
*
* RETURNS: N/A
*
*/
void NanoInit
	(
	)
	{
	unsigned long   denominator;
	unsigned int freq = (CPU_CLOCK_RATE);

	/* The following provides for the 2:1 ratio between the cpu clock
	 * and the count register rate.
	 */
	denominator = 2000000000;

	/*
	 * Reduce the numbers. Note that because the denominator
	 * is a constant and only has prime factors of two
	 * and 5, we only need to divide them out. Cool, eh?
	 *
	 * Look at the object code to see how GCC gets all the quotients
	 * and remainders without using any divide instructions. Ah,
	 * just like on the Cyber...
	 */

	while ((denominator & 1) == 0 && (freq & 1) == 0)
		denominator >>= 1, freq >>= 1;

	while ((denominator % 5) == 0 && (freq % 5) == 0)
		denominator /= 5, freq /= 5;

	nanoperiod.numerator = freq;
	nanoperiod.denominator = denominator;
	}

typedef unsigned long U32;

#define	PREG_BASE	0xB8400000	/* Base address to Peripheral Reg  */
#define PER_BASE 	PREG_BASE

/**********************************************************/
/* GPIO registers										  */
/**********************************************************/
#define GPIO_DATA_REG	((volatile U32 *)(PER_BASE + 0xE0))	/* GPIO Data reg */
#define GPIO_CFG1_REG	((volatile U32 *)(PER_BASE + 0xE4)) /* GPIO CFG1 reg */
#define GPIO_CFG2_REG	((volatile U32 *)(PER_BASE + 0xE8))	/* GPIO CFG2 reg */

#define MAXLEDSAVAILABLE	8

/******************************************************************************
*
* initLED - initialize GPIO pins for the LED
*
* Input :  gpioPin to be used as LED (pin is set to output)
* 
******************************************************************************/

void initLED (int gpioPin)
{
    /* set the appropriate GPIO pins to be outputs */
    /* *GPIO_CFG1_REG controls GPIO 0-7 */
    /* *GPIO_CFG2_REG controls GPIO 8-15 */
    if (gpioPin < 8)
    {
        *GPIO_CFG1_REG &= ~(0xF << (gpioPin * 4));
        *GPIO_CFG1_REG |= 0x8 << (gpioPin * 4);
    }
    else
    {
        *GPIO_CFG2_REG &= ~(0xF << (gpioPin * 4));
        *GPIO_CFG2_REG |= 0x8 << (gpioPin * 4);
    }
}

/*******************************************************************************
*
* turnOnLED - turn LED on associated with GPIO pin
*
* Input :  gpioPin used as LED (pin is set to output
* 
********************************************************************************/
void turnOnLED (int gpioPin)
{	
    *GPIO_DATA_REG |= 1 << gpioPin;    /* turns LED on */
}

/*******************************************************************************
*
* turnOffLED - turn LED off associated with GPIO pin
*
* Input :  gpioPin used as LED (pin is set to output
* 
********************************************************************************/
void turnOffLED (int gpioPin)
{
    *GPIO_DATA_REG &= ~(1 << gpioPin);    /* turns LED off */
}


void kernel_entry(int argc, char **argv, char **envp, int *prom)
{
	int i;
	int lastled = 0;
	int led;
	int delaycount = 1000000000;
	unsigned long selectedleds = 0xff;

	printf("\nI am in kernel_entry\n");

	printf("Calling arguments: argc %d, argv %lx, envp %lx, prom %lx\n",
		   argc, (unsigned long) argv, 
		   (unsigned long) envp, 
		   (unsigned long) prom);

	for (i = 0; i < argc; i++)
	{
		printf("argv[%d] at address %lx is <%s>\n", 
			   i, (unsigned long) argv[i], argv[i]);
	}

	if (argv[argc] == 0)
	{
		printf("argv[argc] is %ld\n", (unsigned long) argv[argc]);
	}
	else
	{
		printf("argv[argc] should have been 0, but was %ld\n", 
			   (unsigned long) argv[argc]);
	}

	for (i = 0; envp[i] != 0; i++)
	{
		printf("envp[%d] at address %lx is <%s>\n", 
			   i, (unsigned long) envp[i], envp[i]);
	}

	NanoInit();

	if (selectedleds == 0)
	{
		printf("No LEDs selected\n");
		while (1) {}
	}
	
	printf("delay count selected %d nanoseconds\n", delaycount);

	for (led = 0; led < MAXLEDSAVAILABLE; led++)
	{
		if (selectedleds & (1<<led))
		{
			initLED(led);
			turnOffLED(led);
			lastled = led;
			printf("led %d selected\n", led);
		}
	}

	/* turn on/off leds */

	while (1)
	{
		turnOffLED(lastled);
		led = lastled + 1;
		while (led != lastled)
		{
			if (led < MAXLEDSAVAILABLE)
			{
				if (selectedleds & (1<<led))
					lastled = led;
				else
					led++;
			}
			else
				led = 0;
		}
		turnOnLED(lastled);
		NanoDelay(delaycount);
	}

}
