#include "at.h"

#define RUN_REALTIME 0

char at_driver_name[] = "ATL2";
char at_driver_string[] = "Attansic(R) L2 Ethernet Network Driver";
#define DRV_VERSION "1.0.40.0"
char at_driver_version[] = DRV_VERSION;
char at_copyright[] = "Copyright (c) 1999-2007 Attansic Corporation.";


/* 
 * at_pci_tbl - PCI Device ID Table
 *
 * Wildcard entries (PCI_ANY_ID) should come last
 * Last entry must be all 0s
 *
 * { Vendor ID, Device ID, SubVendor ID, SubDevice ID,
 *   Class, Class Mask, private data (not used) }
 */
static struct pci_device_id at_pci_tbl[] = {
    ATTANSIC_ETHERNET_DEVICE(0x2048),
    /* required last entry */
    {0,}
};

MODULE_DEVICE_TABLE(pci, at_pci_tbl);

int32_t at_up(struct at_adapter *adapter);
void at_down(struct at_adapter *adapter);
int at_reset(struct at_adapter *adapter);
int32_t at_setup_ring_resources(struct at_adapter *adapter);
void at_free_ring_resources(struct at_adapter *adapter);

/* Local Function Prototypes */
static int at_init_module(void);
static void at_exit_module(void);
static int at_probe(struct pci_dev *pdev, const struct pci_device_id *ent);
static void __devexit at_remove(struct pci_dev *pdev);
static int at_sw_init(struct at_adapter *adapter);
static int at_open(struct net_device *netdev);
static int at_close(struct net_device *netdev);
static int at_xmit_frame(struct sk_buff *skb, struct net_device *netdev);
static struct net_device_stats * at_get_stats(struct net_device *netdev);
static int at_change_mtu(struct net_device *netdev, int new_mtu);
static void at_set_multi(struct net_device *netdev);
static int at_set_mac(struct net_device *netdev, void *p);
static int at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
static void at_tx_timeout(struct net_device *dev);
static irqreturn_t at_intr(int irq, void *data, struct pt_regs *regs);
static void at_intr_rx(struct at_adapter* adapter);
static void at_intr_tx(struct at_adapter* adapter);

static void at_watchdog(unsigned long data);
static void at_phy_config(unsigned long data);
static void at_tx_timeout_task(struct net_device *netdev);
static void at_check_for_link(struct at_adapter* adapter);
static void at_link_chg_task(struct net_device* netdev);
static uint32_t at_check_link(struct at_adapter* adapter);
void init_ring_ptrs(struct at_adapter *adapter);
static uint32_t at_configure(struct at_adapter *adapter);


#ifdef SIOCGMIIPHY
static int at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd);
#endif


#ifdef NETIF_F_HW_VLAN_TX
static void at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp);
static void at_vlan_rx_add_vid(struct net_device *netdev, uint16_t vid);
static void at_vlan_rx_kill_vid(struct net_device *netdev, uint16_t vid);
static void at_restore_vlan(struct at_adapter *adapter);
#endif

static int at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p);
static int at_suspend(struct pci_dev *pdev, uint32_t state);
#ifdef CONFIG_PM
static int at_resume(struct pci_dev *pdev);
#endif


struct notifier_block at_notifier_reboot = {
    .notifier_call  = at_notify_reboot,
    .next       = NULL,
    .priority   = 0
};

/* Exported from other modules */

extern void at_check_options(struct at_adapter *adapter);
//extern int at_ethtool_ioctl(struct net_device *netdev, struct ifreq *ifr);
#ifdef SIOCDEVPRIVATE
extern int at_priv_ioctl(struct net_device* netdev, struct ifreq* ifr);
#endif

static struct pci_driver at_driver = {
    .name     = at_driver_name,
    .id_table = at_pci_tbl,
    .probe    = at_probe,
    .remove   = __devexit_p(at_remove),
    /* Power Managment Hooks */
#ifdef CONFIG_PM
    .suspend  = at_suspend,
    .resume   = at_resume
#endif
};

MODULE_AUTHOR("Attansic Corporation, <xiong_huang@attansic.com>");
MODULE_DESCRIPTION("Attansic 1000M Ethernet Network Driver");
MODULE_LICENSE("ATTANSIC");
MODULE_VERSION(DRV_VERSION);

/**
 * at_init_module - Driver Registration Routine
 *
 * at_init_module is the first routine called when the driver is
 * loaded. All it does is register with the PCI subsystem.
 **/

static int __init
at_init_module(void)
{
    int ret;
    printk(KERN_INFO "%s - version %s\n",
           at_driver_string, at_driver_version);

    printk(KERN_INFO "%s\n", at_copyright);

    ret = pci_module_init(&at_driver);
    if(ret >= 0) {
        register_reboot_notifier(&at_notifier_reboot);
    }
    return ret;
}

module_init(at_init_module);

/**
 * at_exit_module - Driver Exit Cleanup Routine
 *
 * at_exit_module is called just before the driver is removed
 * from memory.
 **/

static void __exit
at_exit_module(void)
{
    unregister_reboot_notifier(&at_notifier_reboot);
    pci_unregister_driver(&at_driver);
}

module_exit(at_exit_module);


/**
 * at_probe - Device Initialization Routine
 * @pdev: PCI device information struct
 * @ent: entry in at_pci_tbl
 *
 * Returns 0 on success, negative on failure
 *
 * at_probe initializes an adapter identified by a pci_dev structure.
 * The OS initialization, configuring of the adapter private structure,
 * and a hardware reset occur.
 **/

static int __devinit
at_probe(struct pci_dev *pdev,
            const struct pci_device_id *ent)
{
    struct net_device *netdev;
    struct at_adapter *adapter;
    static int cards_found = 0;
    unsigned long mmio_start;
    int mmio_len;
    boolean_t pci_using_64 = TRUE;
    int err;
//    uint16_t eeprom_data;

    DEBUGFUNC("at_probe !");

    if((err = pci_enable_device(pdev)))
        return err;

    if((err = pci_set_dma_mask(pdev, PCI_DMA_64BIT))) {
        if((err = pci_set_dma_mask(pdev, PCI_DMA_32BIT))) {
            AT_ERR("No usable DMA configuration, aborting\n");
            return err;
        }
        pci_using_64 = FALSE;
    }

    
    // Mark all PCI regions associated with PCI device 
    // pdev as being reserved by owner at_driver_name
    if((err = pci_request_regions(pdev, at_driver_name)))
        return err;

    // Enables bus-mastering on the device and calls 
    // pcibios_set_master to do the needed arch specific settings
    pci_set_master(pdev);

    netdev = alloc_etherdev(sizeof(struct at_adapter));
    if(!netdev) {
        err = -ENOMEM;
        goto err_alloc_etherdev;
    }

    SET_MODULE_OWNER(netdev);
    SET_NETDEV_DEV(netdev, &pdev->dev);

    pci_set_drvdata(pdev, netdev);
    adapter = netdev_priv(netdev);
    adapter->netdev = netdev;
    adapter->pdev = pdev;
    adapter->hw.back = adapter;

    mmio_start = pci_resource_start(pdev, BAR_0);
    mmio_len = pci_resource_len(pdev, BAR_0);

    AT_DBG("base memory = %lx memory length = %x \n", 
        mmio_start, mmio_len);
    adapter->hw.mem_rang = (uint32_t)mmio_len;
    adapter->hw.hw_addr = ioremap_nocache(mmio_start, mmio_len);
    if(!adapter->hw.hw_addr) {
        err = -EIO;
        goto err_ioremap;
    }
    

    netdev->open = &at_open;
    netdev->stop = &at_close;
    netdev->hard_start_xmit = &at_xmit_frame;
    netdev->get_stats = &at_get_stats;
    netdev->set_multicast_list = &at_set_multi;
    netdev->set_mac_address = &at_set_mac;
    netdev->change_mtu = &at_change_mtu;
    netdev->do_ioctl = &at_ioctl;
#ifdef HAVE_TX_TIMEOUT
    netdev->tx_timeout = &at_tx_timeout;
    netdev->watchdog_timeo = 5 * HZ;
#endif
#ifdef NETIF_F_HW_VLAN_TX
    netdev->vlan_rx_register = at_vlan_rx_register;
    netdev->vlan_rx_add_vid = at_vlan_rx_add_vid;
    netdev->vlan_rx_kill_vid = at_vlan_rx_kill_vid;
#endif

    netdev->mem_start = mmio_start;
    netdev->mem_end = mmio_start + mmio_len;
    //netdev->base_addr = adapter->io_base;
    adapter->bd_number = cards_found;
    adapter->pci_using_64 = pci_using_64;

    /* setup the private structure */

    if((err = at_sw_init(adapter)))
        goto err_sw_init;

    
#ifdef NETIF_F_HW_VLAN_TX
    netdev->features |= 
               (NETIF_F_HW_VLAN_TX | 
                NETIF_F_HW_VLAN_RX );
#endif

    if(pci_using_64) {
        netdev->features |= NETIF_F_HIGHDMA;
        AT_DBG("pci using 64bit address\n");
    }
#ifdef NETIF_F_LLTX
    netdev->features |= NETIF_F_LLTX;
#endif


    /* reset the controller to 
     * put the device in a known good starting state */
    
    if (at_reset_hw(&adapter->hw)) {
        err = -EIO;
        goto err_reset;
    }

    /* copy the MAC address out of the EEPROM */

    at_read_mac_addr(&adapter->hw);
    memcpy(netdev->dev_addr, adapter->hw.mac_addr, netdev->addr_len);

    if(!is_valid_ether_addr(netdev->dev_addr)) {
        err = -EIO;
        goto err_eeprom;
    }

    AT_DBG("mac address : %02x-%02x-%02x-%02x-%02x-%02x\n",
        adapter->hw.mac_addr[0],
        adapter->hw.mac_addr[1],
        adapter->hw.mac_addr[2],
        adapter->hw.mac_addr[3],
        adapter->hw.mac_addr[4],
        adapter->hw.mac_addr[5] );

    at_check_options(adapter);
    
    /* pre-init the MAC, and setup link */

    if ((err = at_init_hw(&adapter->hw))) {
        err = -EIO;
        goto err_init_hw;
    }
    
    /* assume we have no link for now */

    netif_carrier_off(netdev);
    netif_stop_queue(netdev);
    
    init_timer(&adapter->watchdog_timer);
    adapter->watchdog_timer.function = &at_watchdog;
    adapter->watchdog_timer.data = (unsigned long) adapter;
    
    init_timer(&adapter->phy_config_timer);
    adapter->phy_config_timer.function = &at_phy_config;
    adapter->phy_config_timer.data = (unsigned long) adapter;
    adapter->phy_timer_pending = FALSE;
    
    INIT_WORK(&adapter->tx_timeout_task,
        (void (*)(void *))at_tx_timeout_task, netdev);
        
    INIT_WORK(&adapter->link_chg_task, 
        (void (*)(void *))at_link_chg_task, netdev);
   
    INIT_WORK(&adapter->pcie_dma_to_rst_task,
        (void (*)(void *))at_tx_timeout_task, netdev);

    if((err = register_netdev(netdev)))
        goto err_register;

    
    cards_found++;

    return 0;

err_init_hw:
err_reset:
err_register:
err_sw_init:
err_eeprom:
    iounmap(adapter->hw.hw_addr);
err_ioremap:
    free_netdev(netdev);
err_alloc_etherdev:
    pci_release_regions(pdev);
    return err;
}

/**
 * at_remove - Device Removal Routine
 * @pdev: PCI device information struct
 *
 * at_remove is called by the PCI subsystem to alert the driver
 * that it should release a PCI device.  The could be caused by a
 * Hot-Plug event, or because the driver is going to be removed from
 * memory.
 **/

static void __devexit
at_remove(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter;

    DEBUGFUNC("at_remove");

    /* Device not available. Return. */
    if (!netdev)  
        return;

    adapter = netdev_priv(netdev);
    
    unregister_netdev(netdev);

    iounmap(adapter->hw.hw_addr);
    pci_release_regions(pdev);

    free_netdev(netdev);
}



static int
at_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
    struct pci_dev *pdev = NULL;

    DEBUGFUNC("at_notify_reboot !");

    switch(event) {
    case SYS_DOWN:
    case SYS_HALT:
    case SYS_POWER_OFF:
        while((pdev = pci_find_device(PCI_ANY_ID, PCI_ANY_ID, pdev))) {
            if(pci_dev_driver(pdev) == &at_driver)
                at_suspend(pdev, 3);
        }
    }
    return NOTIFY_DONE;
}


static int
at_suspend(struct pci_dev *pdev, uint32_t state)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw * hw = &adapter->hw;
    uint32_t ctrl = 0;
    uint32_t wufc = adapter->wol;

    DEBUGFUNC("at_suspend !"); 

    netif_device_detach(netdev);

    if(netif_running(netdev))
        at_down(adapter);

    at_read_phy_reg(hw, MII_BMSR, (uint16_t*)&ctrl);
    at_read_phy_reg(hw, MII_BMSR, (uint16_t*)&ctrl);
    if(ctrl & BMSR_LSTATUS)
        wufc &= ~AT_WUFC_LNKC;
        
    // reduce speed to 10/100M
    if (wufc) {
        at_phy_enter_power_saving(hw);
        // if resume, let driver to re- setup link
        hw->phy_configured = FALSE; 
        set_mac_addr(hw);
        at_set_multi(netdev);
        
        ctrl = 0;
        /* turn on magic packet wol */
        if (wufc & AT_WUFC_MAG) {
            ctrl = WOL_MAGIC_EN|WOL_MAGIC_PME_EN;
        }
        // turn on Link change WOL */
        if (wufc & AT_WUFC_LNKC) {
            ctrl |= (WOL_LINK_CHG_EN|WOL_LINK_CHG_PME_EN);
        }
        AT_WRITE_REG(hw, REG_WOL_CTRL, ctrl);
            
        /* turn on all-multi mode if wake on multicast is enabled */
        ctrl = AT_READ_REG(hw, REG_MAC_CTRL);
        ctrl &= ~MAC_CTRL_PROMIS_EN;
        if(wufc & AT_WUFC_MC) {
            ctrl |= MAC_CTRL_MC_ALL_EN;
        } else {
            ctrl &= ~MAC_CTRL_MC_ALL_EN;
        }
        /* turn on broadcast mode if wake on-BC is enabled */
        if (wufc & AT_WUFC_BC) {
            ctrl |= MAC_CTRL_BC_EN;
        } else {
            ctrl &= ~MAC_CTRL_BC_EN;
        }
        
        // enable RX
        ctrl |= MAC_CTRL_RX_EN;
        AT_WRITE_REG(hw, REG_MAC_CTRL, ctrl);
        
        pci_enable_wake(pdev, 3, 1);
        pci_enable_wake(pdev, 4, 1); /* 4 == D3 cold */
    } else {
        AT_WRITE_REG(hw, REG_WOL_CTRL, 0);
        pci_enable_wake(pdev, 3, 0);
        pci_enable_wake(pdev, 4, 0); /* 4 == D3 cold */
    }
        
    pci_save_state(pdev);
    pci_disable_device(pdev);

    state = (state > 0) ? 3 : 0;
    pci_set_power_state(pdev, state);

    return 0;
}

static int
at_resume(struct pci_dev *pdev)
{
    struct net_device *netdev = pci_get_drvdata(pdev);
    struct at_adapter *adapter = netdev_priv(netdev);
    uint32_t ret_val;

    DEBUGFUNC("at_resume !");

    pci_set_power_state(pdev, 0);
    pci_restore_state(pdev);
    
    ret_val = pci_enable_device(pdev);
    pci_enable_wake(pdev, 3, 0);
    pci_enable_wake(pdev, 4, 0); /* 4 == D3 cold */
    
    AT_WRITE_REG(&adapter->hw, REG_WOL_CTRL, 0);
    at_reset(adapter);
    

    if(netif_running(netdev))
        at_up(adapter);

    netif_device_attach(netdev);

    return 0;
}

/**
 * at_irq_enable - Enable default interrupt generation settings
 * @adapter: board private structure
 **/

inline void
at_irq_enable(struct at_adapter *adapter)
{
    if(0 == atomic_dec_and_test(&adapter->irq_sem)) {
        AT_WRITE_REG(&adapter->hw, REG_IMR, IMR_NORMAL_MASK);
    }
}

/**
 * at_irq_disable - Mask off interrupt generation on the NIC
 * @adapter: board private structure
 **/

inline void
at_irq_disable(struct at_adapter *adapter)
{
    atomic_inc(&adapter->irq_sem);
    AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
    
    synchronize_irq(adapter->pdev->irq);
}

/**
 * at_sw_init - Initialize general software structures (struct at_adapter)
 * @adapter: board private structure to initialize
 *
 * at_sw_init initializes the Adapter private data structure.
 * Fields are initialized based on PCI device information and
 * OS network device settings (MTU size).
 **/

static int __devinit
at_sw_init(struct at_adapter *adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct pci_dev *pdev = adapter->pdev;

    /* PCI config space info */

    hw->vendor_id = pdev->vendor;
    hw->device_id = pdev->device;
    hw->subsystem_vendor_id = pdev->subsystem_vendor;
    hw->subsystem_id = pdev->subsystem_device;

    pci_read_config_byte(pdev, PCI_REVISION_ID, &hw->revision_id);

    pci_read_config_word(pdev, PCI_COMMAND, &hw->pci_cmd_word);

    adapter->wol = 0;

    adapter->ict = 50000;  // 100ms
    
    adapter->link_speed = SPEED_0;   // hardware init
    adapter->link_duplex = FULL_DUPLEX; //

  
    hw->phy_configured = FALSE;
    hw->preamble_len = 7;
    hw->ipgt = 0x60;
    hw->min_ifg = 0x50;
    hw->ipgr1 = 0x40;
    hw->ipgr2 = 0x60;
    hw->retry_buf = 2;
    
    hw->max_retry = 0xf;
    hw->lcol = 0x37;
    hw->jam_ipg = 7;
    
    hw->fc_rxd_hi = 0;
    hw->fc_rxd_lo = 0; 
    
    atomic_set(&adapter->irq_sem, 0);
    spin_lock_init(&adapter->stats_lock);
    spin_lock_init(&adapter->tx_lock);
//    spin_lock_init(&adapter->mb_lock);

    return 0;
}

int
at_reset(struct at_adapter *adapter)
{
    int ret;
    
    if (AT_SUCCESS != (ret = at_reset_hw(&adapter->hw)))
        return ret;

    return at_init_hw(&adapter->hw);
}

/**
 * at_open - Called when a network interface is made active
 * @netdev: network interface device structure
 *
 * Returns 0 on success, negative value on failure
 *
 * The open entry point is called when a network interface is made
 * active by the system (IFF_UP).  At this point all resources needed
 * for transmit and receive operations are allocated, the interrupt
 * handler is registered with the OS, the watchdog timer is started,
 * and the stack is notified that the interface is ready.
 **/

static int
at_open(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    int err;

    DEBUGFUNC("at_open !");

    /* allocate transmit descriptors */

    if((err = at_setup_ring_resources(adapter)))
        return err;

    if((err = at_up(adapter)))
        goto err_up;

    return 0;

err_up:
    at_reset(adapter);

    return err;
 
}

/**
 * at_close - Disables a network interface
 * @netdev: network interface device structure
 *
 * Returns 0, this is not allowed to fail
 *
 * The close entry point is called when an interface is de-activated
 * by the OS.  The hardware is still under the drivers control, but
 * needs to be disabled.  A global MAC reset is issued to stop the
 * hardware, and all transmit and receive resources are freed.
 **/

static int
at_close(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    DEBUGFUNC("at_close!");

    at_down(adapter);

    at_free_ring_resources(adapter);

    return 0;
}

/**
 * at_setup_mem_resources - allocate Tx / RX descriptor resources 
 * @adapter: board private structure
 *
 * Return 0 on success, negative on failure
 **/

int32_t
at_setup_ring_resources(struct at_adapter *adapter)
{
    struct pci_dev *pdev = adapter->pdev;
    int size;
    uint8_t offset = 0;

    DEBUGFUNC("at_setup_ring_resources");

    /* real ring DMA buffer */
    adapter->ring_size = size =   
	      adapter->txd_ring_size * 1   + 7         // dword align
	    + adapter->txs_ring_size * 4   + 7         // dword align
            + adapter->rxd_ring_size * 1536+ 127;    // 128bytes align
    
    adapter->ring_vir_addr = 
                    pci_alloc_consistent(pdev, size, &adapter->ring_dma);
    if (!adapter->ring_vir_addr) {
        DEBUGOUT1("pci_alloc_consistent failed, size = D%d", size);
        return -ENOMEM;
    }
 
    if (adapter->pci_using_64) { 
        // test whether HIDWORD dma buffer is not cross boundary
        if (    ((adapter->ring_dma       &0xffffffff00000000ULL)>>32)
             != (((adapter->ring_dma+size)&0xffffffff00000000ULL)>>32) ) {
            pci_free_consistent(
                     pdev, 
                     adapter->ring_size, 
                     adapter->ring_vir_addr, 
                     adapter->ring_dma);
            DEBUGOUT("memory allocated cross 32bit boundary !");
            return -ENOMEM;
        }
    }

//    DEBUGOUT("memory allocated successfully !");    
    
    memset(adapter->ring_vir_addr, 0, adapter->ring_size);

      // Init TXD Ring
      
      adapter->txd_dma = adapter->ring_dma ;
      offset = (adapter->txd_dma & 0x7) ? (8 - (adapter->txd_dma & 0x7)) : 0;
      adapter->txd_dma += offset;
      adapter->txd_ring = (tx_pkt_header_t*) (adapter->ring_vir_addr + offset);
      
      // Init TXS Ring
      
      adapter->txs_dma = adapter->txd_dma + adapter->txd_ring_size;
      offset = (adapter->txs_dma & 0x7) ? (8- (adapter->txs_dma & 0x7)) : 0;
      adapter->txs_dma += offset;
      adapter->txs_ring = (tx_pkt_status_t*) 
                (((uint8_t*)adapter->txd_ring) + (adapter->txd_ring_size+offset));
                
      // Init RXD Ring
      adapter->rxd_dma = adapter->txs_dma + adapter->txs_ring_size*4;
      offset = (adapter->rxd_dma & 127) ? (128 - (adapter->rxd_dma & 127)) : 0;
      if (offset > 7) {
	  offset -= 8;
      } else {
	  offset += (128 - 8);
      }
      adapter->rxd_dma += offset;
      adapter->rxd_ring = (rx_desc_t*)
                (((uint8_t*)adapter->txs_ring) + 
		    (adapter->txs_ring_size*4 + offset));


      // Read / Write Ptr Initialize:
  //      init_ring_ptrs(adapter);

    return AT_SUCCESS;
}


void
init_ring_ptrs(struct at_adapter *adapter)
{
        // Read / Write Ptr Initialize:
    adapter->txd_write_ptr = 0;
    atomic_set(&adapter->txd_read_ptr, 0);

    adapter->rxd_read_ptr = 0;
    adapter->rxd_write_ptr = 0;
    
    atomic_set(&adapter->txs_write_ptr, 0);
    adapter->txs_next_clear = 0;
}

/**
 * at_free_ring_resources - Free Tx / RX descriptor Resources
 * @adapter: board private structure
 *
 * Free all transmit software resources
 **/

void
at_free_ring_resources(struct at_adapter *adapter)
{
    struct pci_dev *pdev = adapter->pdev;
    
    DEBUGFUNC("at_free_ring_resources");

    pci_free_consistent(
         pdev, 
         adapter->ring_size,
         adapter->ring_vir_addr,
         adapter->ring_dma);
         
}


int32_t
at_up(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    int err;

//    DEBUGFUNC("at_up !"); 
    
    /* hardware has been reset, we need to reload some things */

    at_set_multi(netdev);
    init_ring_ptrs(adapter);

#ifdef NETIF_F_HW_VLAN_TX
    at_restore_vlan(adapter);
#endif

    if (at_configure(adapter)) {
        err = -EIO;
        goto err_up;
    }
   
    if ((err = request_irq(adapter->pdev->irq, 
                          &at_intr,
                          SA_SHIRQ | SA_SAMPLE_RANDOM,
                          netdev->name, netdev)))
        goto err_up;
        
    mod_timer(&adapter->watchdog_timer, jiffies); 
    
    at_irq_enable(adapter);
    
    at_check_link(adapter);

    return 0;

    // free irq
    // disable any interrupt
    AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
    free_irq(adapter->pdev->irq, netdev);
    
err_up:
   
    return err;    
}

inline void
at_setup_mac_ctrl(struct at_adapter* adapter)
{
    uint32_t value;
    struct at_hw* hw = &adapter->hw;
    struct net_device* netdev = adapter->netdev;
    
    /* Config MAC CTRL Register */
    value = MAC_CTRL_TX_EN | 
	    MAC_CTRL_RX_EN |
            MAC_CTRL_MACLP_CLK_PHY;
    // duplex
    if (FULL_DUPLEX == adapter->link_duplex)    
        value |= MAC_CTRL_DUPLX;
    // flow control
    value |= (MAC_CTRL_TX_FLOW|MAC_CTRL_RX_FLOW);

    // PAD & CRC
    value |= (MAC_CTRL_ADD_CRC|MAC_CTRL_PAD);
    // preamble length
    value |= (((uint32_t)adapter->hw.preamble_len
                  &MAC_CTRL_PRMLEN_MASK)<< MAC_CTRL_PRMLEN_SHIFT);
    // vlan 
    if (adapter->vlgrp)     
        value |= MAC_CTRL_RMV_VLAN;
        
    // filter mode
    value |= MAC_CTRL_BC_EN;
    if (netdev->flags & IFF_PROMISC) 
        value |= MAC_CTRL_PROMIS_EN;
    else if (netdev->flags & IFF_ALLMULTI)
        value |= MAC_CTRL_MC_ALL_EN;

    // half retry buffer
    value |= (((uint32_t)(adapter->hw.retry_buf
                &MAC_CTRL_HALF_LEFT_BUF_MASK)) 
                    << MAC_CTRL_HALF_LEFT_BUF_SHIFT);

    AT_WRITE_REG(hw, REG_MAC_CTRL, value);
}


static uint32_t
at_check_link(struct at_adapter* adapter)
{
    struct at_hw *hw = &adapter->hw;
    struct net_device * netdev = adapter->netdev;
    uint32_t ret_val;
    uint16_t speed, duplex, phy_data;
    int reconfig = 0;

//    DEBUGFUNC("at_check_link !");
	// MII_BMSR must read twise
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    at_read_phy_reg(hw, MII_BMSR, &phy_data);
    if (!(phy_data&BMSR_LSTATUS)) { // link down
		if (netif_carrier_ok(netdev)) { // old link state: Up
			DEBUGOUT("NIC Link is Down");
            adapter->link_speed = SPEED_0;
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
        return AT_SUCCESS;  
    }
    
    // Link Up
	ret_val = at_get_speed_and_duplex(hw, &speed, &duplex);
	if (ret_val)  return ret_val;
	switch( hw->MediaType )
	{
	case MEDIA_TYPE_100M_FULL:
		if (speed  != SPEED_100 || duplex != FULL_DUPLEX)
			reconfig = 1;
		break;
	case MEDIA_TYPE_100M_HALF:
		if (speed  != SPEED_100 || duplex != HALF_DUPLEX)
			reconfig = 1;
		break;
	case MEDIA_TYPE_10M_FULL:
		if (speed != SPEED_10 || duplex != FULL_DUPLEX)
			reconfig = 1;
	        break;	
	case MEDIA_TYPE_10M_HALF:
		if (speed  != SPEED_10 || duplex != HALF_DUPLEX)
			reconfig = 1;
		break;
	}
	// link result is our setting
	if (0 == reconfig)
	{
		if (adapter->link_speed != speed ||
            adapter->link_duplex != duplex ) {
			adapter->link_speed = speed;
			adapter->link_duplex = duplex;
			at_setup_mac_ctrl(adapter); 
			printk(KERN_INFO
                   "%s: %s NIC Link is Up<%d Mbps %s>\n",
		   			at_driver_name,
                    netdev->name, adapter->link_speed,
                    adapter->link_duplex == FULL_DUPLEX ?
 					"Full Duplex" : "Half Duplex"); 
		}
		
		if (!netif_carrier_ok(netdev)) { // Link down -> Up
			netif_carrier_on(netdev);
			netif_wake_queue(netdev);
		}
		return AT_SUCCESS;
	}
	
	// change orignal link status
	if (netif_carrier_ok(netdev)) { 
		adapter->link_speed = SPEED_0;
    	netif_carrier_off(netdev);
    	netif_stop_queue(netdev);
    }
    
    if (hw->MediaType != MEDIA_TYPE_AUTO_SENSOR) {
    	switch (hw->MediaType)
    	{
    	case MEDIA_TYPE_100M_FULL:
    		phy_data = MII_CR_FULL_DUPLEX|MII_CR_SPEED_100|MII_CR_RESET;
    		break;
    	case MEDIA_TYPE_100M_HALF:
    		phy_data = MII_CR_SPEED_100|MII_CR_RESET;
    		break;
    	case MEDIA_TYPE_10M_FULL:
    		phy_data = MII_CR_FULL_DUPLEX|MII_CR_SPEED_10|MII_CR_RESET;
    		break;
    	default: // MEDIA_TYPE_10M_HALF:
    		phy_data = MII_CR_SPEED_10|MII_CR_RESET;
    		break;
    	}
    	at_write_phy_reg(hw, MII_BMCR, phy_data);
    	return AT_SUCCESS;
    }

	// auto-neg, insert timer to re-config phy
    if (!adapter->phy_timer_pending) {
		adapter->phy_timer_pending = TRUE;
		mod_timer(&adapter->phy_config_timer, jiffies + 3 * HZ);
	}

    return AT_SUCCESS;
}


void
at_down(struct at_adapter *adapter)
{
    struct net_device *netdev = adapter->netdev;
    
//    DEBUGFUNC("at_down !");

    del_timer_sync(&adapter->watchdog_timer);
    del_timer_sync(&adapter->phy_config_timer);
    adapter->phy_timer_pending = FALSE;

    at_irq_disable(adapter);    
    free_irq(adapter->pdev->irq, netdev);
    at_reset_hw(&adapter->hw);

    adapter->link_speed = SPEED_0;
    adapter->link_duplex = -1;
    netif_carrier_off(netdev);
    netif_stop_queue(netdev);
}



/**
 * at_set_multi - Multicast and Promiscuous mode set
 * @netdev: network interface device structure
 *
 * The set_multi entry point is called whenever the multicast address
 * list or the network interface flags are updated.  This routine is
 * responsible for configuring the hardware for proper multicast,
 * promiscuous mode, and all-multi behavior.
 **/

static void
at_set_multi(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct at_hw *hw = &adapter->hw;
    struct dev_mc_list *mc_ptr;
    uint32_t rctl;
    uint32_t hash_value;

//    DEBUGFUNC("at_set_multi !");

    /* Check for Promiscuous and All Multicast modes */

    rctl = AT_READ_REG(hw, REG_MAC_CTRL);

    if(netdev->flags & IFF_PROMISC) {
        rctl |= MAC_CTRL_PROMIS_EN;
    } else if(netdev->flags & IFF_ALLMULTI) {
        rctl |= MAC_CTRL_MC_ALL_EN;
        rctl &= ~MAC_CTRL_PROMIS_EN;
    } else {
        rctl &= ~(MAC_CTRL_PROMIS_EN | MAC_CTRL_MC_ALL_EN);
    }

    AT_WRITE_REG(hw, REG_MAC_CTRL, rctl);

    /* clear the old settings from the multicast hash table */
    AT_WRITE_REG(hw, REG_RX_HASH_TABLE, 0);
    AT_WRITE_REG_ARRAY(hw, REG_RX_HASH_TABLE, 1, 0);

    /* comoute mc addresses' hash value ,and put it into hash table */

    for(mc_ptr = netdev->mc_list; mc_ptr; mc_ptr = mc_ptr->next) {
        hash_value = at_hash_mc_addr(hw, mc_ptr->dmi_addr);
        at_hash_set(hw, hash_value);
    }
}

#ifdef NETIF_F_HW_VLAN_TX
static void
at_vlan_rx_register(struct net_device *netdev, struct vlan_group *grp)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    uint32_t ctrl;

 //   DEBUGFUNC("at_vlan_rx_register !");    

    at_irq_disable(adapter);
    adapter->vlgrp = grp;

    if(grp) {
        /* enable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl |= MAC_CTRL_RMV_VLAN; 
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    } else {
        /* disable VLAN tag insert/strip */

        ctrl = AT_READ_REG(&adapter->hw, REG_MAC_CTRL);
        ctrl &= ~MAC_CTRL_RMV_VLAN;
        AT_WRITE_REG(&adapter->hw, REG_MAC_CTRL, ctrl);
    }

    at_irq_enable(adapter);
}

static void
at_vlan_rx_add_vid(struct net_device *netdev, uint16_t vid)
{
    /* We don't do Vlan filtering */
//    DEBUGFUNC("at_vlan_rx_add_vid !");
    return ;
}

static void
at_vlan_rx_kill_vid(struct net_device *netdev, uint16_t vid)
{
    struct at_adapter *adapter = netdev_priv(netdev);

//    DEBUGFUNC("at_vlan_rx_kill_vid !");
    at_irq_disable(adapter);

    if(adapter->vlgrp)
        adapter->vlgrp->vlan_devices[vid] = NULL;

    at_irq_enable(adapter);

    /* We don't do Vlan filtering */

    return;
}

static void
at_restore_vlan(struct at_adapter *adapter)
{
//    DEBUGFUNC("at_restore_vlan !");
    at_vlan_rx_register(adapter->netdev, adapter->vlgrp);

    if(adapter->vlgrp) {
        uint16_t vid;
        for(vid = 0; vid < VLAN_GROUP_ARRAY_LEN; vid++) {
            if(!adapter->vlgrp->vlan_devices[vid])
                continue;
            at_vlan_rx_add_vid(adapter->netdev, vid);
        }
    }
}
#endif

/**
 * at_configure - Configure Transmit&Receive Unit after Reset
 * @adapter: board private structure
 *
 * Configure the Tx /Rx unit of the MAC after a reset.
 **/

static uint32_t
at_configure(struct at_adapter *adapter)
{
    struct at_hw * hw = &adapter->hw;
    uint32_t value;
    
//    DEBUGFUNC("at_configure !");

    // clear interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0xffffffff);

    // set MAC Address
    value = (((uint32_t)hw->mac_addr[2]) << 24) |
            (((uint32_t)hw->mac_addr[3]) << 16) |
            (((uint32_t)hw->mac_addr[4]) << 8 ) |
            (((uint32_t)hw->mac_addr[5])      ) ;
    AT_WRITE_REG(hw, REG_MAC_STA_ADDR, value);
    value = (((uint32_t)hw->mac_addr[0]) << 8 ) |
            (((uint32_t)hw->mac_addr[1])      ) ;
    AT_WRITE_REG(hw, (REG_MAC_STA_ADDR+4), value);

    // tx / rx ring :
    
    // HI base address
    AT_WRITE_REG(
          hw, 
          REG_DESC_BASE_ADDR_HI, 
          (uint32_t)((adapter->ring_dma&0xffffffff00000000ULL) >>32));
    // LO base address
    AT_WRITE_REG(
          hw, 
          REG_TXD_BASE_ADDR_LO, 
          (uint32_t)(adapter->txd_dma&0x00000000ffffffffULL));
    AT_WRITE_REG(
          hw, 
          REG_TXS_BASE_ADDR_LO, 
          (uint32_t)(adapter->txs_dma& 0x00000000ffffffffULL));
    AT_WRITE_REG(hw, 
                 REG_RXD_BASE_ADDR_LO, 
                 (uint32_t)(adapter->rxd_dma& 0x00000000ffffffffULL));
  
    // element count
    AT_WRITE_REGW(hw, REG_TXD_MEM_SIZE, (uint16_t)(adapter->txd_ring_size/4));
    AT_WRITE_REGW(hw, REG_TXS_MEM_SIZE, (uint16_t)adapter->txs_ring_size);
    AT_WRITE_REGW(hw, REG_RXD_BUF_NUM,  (uint16_t)adapter->rxd_ring_size);
    DEBUGOUT1("txd ring size:%d, txs ring size:%d, rxd ring size:%d",
		    adapter->txd_ring_size/4,
		    adapter->txs_ring_size,
		    adapter->rxd_ring_size);
    
    /* config Internal SRAM */
/*
    AT_WRITE_REGW(hw, REG_SRAM_TXRAM_END, sram_tx_end);
    AT_WRITE_REGW(hw, REG_SRAM_TXRAM_END, sram_rx_end);    
*/
   
   
    /* config IPG/IFG */
    value = 
        (((uint32_t)hw->ipgt&MAC_IPG_IFG_IPGT_MASK) 
              <<MAC_IPG_IFG_IPGT_SHIFT) |
        (((uint32_t)hw->min_ifg &MAC_IPG_IFG_MIFG_MASK) 
              <<MAC_IPG_IFG_MIFG_SHIFT) |
        (((uint32_t)hw->ipgr1&MAC_IPG_IFG_IPGR1_MASK)
              <<MAC_IPG_IFG_IPGR1_SHIFT)|
        (((uint32_t)hw->ipgr2&MAC_IPG_IFG_IPGR2_MASK)
              <<MAC_IPG_IFG_IPGR2_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_IPG_IFG, value);
//    DEBUGOUT1("init ipg/ifg with 0x%x", value);
    
    /* config  Half-Duplex Control */
    value = 
      ((uint32_t)hw->lcol&MAC_HALF_DUPLX_CTRL_LCOL_MASK) |
      (((uint32_t)hw->max_retry&MAC_HALF_DUPLX_CTRL_RETRY_MASK)
          <<MAC_HALF_DUPLX_CTRL_RETRY_SHIFT) |
      MAC_HALF_DUPLX_CTRL_EXC_DEF_EN   |
      (0xa<<MAC_HALF_DUPLX_CTRL_ABEBT_SHIFT) |
      (((uint32_t)hw->jam_ipg&MAC_HALF_DUPLX_CTRL_JAMIPG_MASK)
          <<MAC_HALF_DUPLX_CTRL_JAMIPG_SHIFT);
    AT_WRITE_REG(hw, REG_MAC_HALF_DUPLX_CTRL, value);
//    DEBUGOUT1("init Half Duplex with 0x%x", value);
    
    
    /* set Interrupt Moderator Timer */
    AT_WRITE_REGW(hw, REG_IRQ_MODU_TIMER_INIT, adapter->imt);
    AT_WRITE_REG(hw, REG_MASTER_CTRL, MASTER_CTRL_ITIMER_EN);
//    DEBUGOUT1("init Irq Modurator Timer with 0x%x", adapter->imt);
    
    /* set Interrupt Clear Timer */
    AT_WRITE_REGW(hw, REG_CMBDISDMA_TIMER, adapter->ict);
//    DEBUGOUT1("init Irq Clear Timer with 0x%x", adapter->ict);
    
    /* set MTU */
    AT_WRITE_REG(hw, REG_MTU, 
		    adapter->netdev->mtu +
		    ENET_HEADER_SIZE + 
		    VLAN_SIZE +
		    ETHERNET_FCS_SIZE);
//    DEBUGOUT1("init MTU with 0x%x", hw->max_frame_size); 
   
    /* 1590 */
    AT_WRITE_REG(hw, 
		 REG_TX_CUT_THRESH,
		 0x177);
    
     /* flow control */
    AT_WRITE_REGW(hw, REG_PAUSE_ON_TH, hw->fc_rxd_hi);
    AT_WRITE_REGW(hw, REG_PAUSE_OFF_TH, hw->fc_rxd_lo);
    
    /* Init mailbox */
    AT_WRITE_REGW(hw, REG_MB_TXD_WR_IDX, (uint16_t)adapter->txd_write_ptr);
    AT_WRITE_REGW(hw, REG_MB_RXD_RD_IDX, (uint16_t)adapter->rxd_read_ptr);
    
    /* enable DMA read/write */
    AT_WRITE_REGB(hw, REG_DMAR, DMAR_EN);
    AT_WRITE_REGB(hw, REG_DMAW, DMAW_EN);
    
    
    value = AT_READ_REG(&adapter->hw, REG_ISR);
    if ((value&ISR_PHY_LINKDOWN) != 0) {
        value = 1; // config failed 
    } else {
        value = 0;
    }

    // clear all interrupt status
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0x3fffffff);
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
    return value;
}

/**
 * at_set_mac - Change the Ethernet Address of the NIC
 * @netdev: network interface device structure
 * @p: pointer to an address structure
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_set_mac(struct net_device *netdev, void *p)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct sockaddr *addr = p;

    DEBUGFUNC("at_set_mac !");
    
    if (netif_running(netdev))
        return -EBUSY; 

    if(!is_valid_ether_addr(addr->sa_data))
        return -EADDRNOTAVAIL;

    memcpy(netdev->dev_addr, addr->sa_data, netdev->addr_len);
    memcpy(adapter->hw.mac_addr, addr->sa_data, netdev->addr_len);
    
    set_mac_addr(&adapter->hw);


    return 0;
}



/**
 * at_change_mtu - Change the Maximum Transfer Unit
 * @netdev: network interface device structure
 * @new_mtu: new value for maximum frame size
 *
 * Returns 0 on success, negative on failure
 **/

static int
at_change_mtu(struct net_device *netdev, int new_mtu)
{
    DEBUGFUNC("at_change_mtu !");
    
    if ((new_mtu < 40) || (new_mtu > (ETH_DATA_LEN + VLAN_SIZE)))
        return -EINVAL;

      netdev->mtu = new_mtu;
      return 0;
}



void
at_read_pci_cfg(struct at_hw *hw, uint32_t reg, uint16_t *value)
{
    struct at_adapter *adapter = hw->back;

    pci_read_config_word(adapter->pdev, reg, value);
}

void
at_write_pci_cfg(struct at_hw *hw, uint32_t reg, uint16_t *value)
{
    struct at_adapter *adapter = hw->back;

    pci_write_config_word(adapter->pdev, reg, *value);
}


/**
 * at_get_stats - Get System Network Statistics
 * @netdev: network interface device structure
 *
 * Returns the address of the device statistics structure.
 * The statistics are actually updated from the timer callback.
 **/

static struct net_device_stats *
at_get_stats(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    uint32_t drop_rxd, drop_rxs;
    unsigned long flags;
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    drop_rxd = AT_READ_REG(&adapter->hw, REG_STS_RXD_OV);
    drop_rxs = AT_READ_REG(&adapter->hw, REG_STS_RXS_OV);
    adapter->net_stats.rx_over_errors += (drop_rxd+drop_rxs);

    spin_unlock_irqrestore(&adapter->stats_lock, flags);
        
    return &adapter->net_stats;
}       

/**
 * at_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
//    DEBUGFUNC("at_ioctl !");
    switch (cmd) {
#ifdef SIOCGMIIPHY
    case SIOCGMIIPHY:
    case SIOCGMIIREG:
    case SIOCSMIIREG:
        return at_mii_ioctl(netdev, ifr, cmd);
#endif

/*
#ifdef ETHTOOL_OPS_COMPAT
    case SIOCETHTOOL:
        return at_ethtool_ioctl(ifr);
#endif
*/
    
#ifdef SIOCDEVPRIVATE
    case SIOCDEVPRIVATE:
        return at_priv_ioctl(netdev, ifr);
#endif

    default:
        return -EOPNOTSUPP;
    }
}

#ifdef SIOCGMIIPHY
/**
 * at_mii_ioctl -
 * @netdev:
 * @ifreq:
 * @cmd:
 **/

static int
at_mii_ioctl(struct net_device *netdev, struct ifreq *ifr, int cmd)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr->ifr_data;
    unsigned long flags;

//    DEBUGFUNC("at_mii_ioctl !");

    switch (cmd) {
    case SIOCGMIIPHY:
        data->phy_id = 0;
        break;
    case SIOCGMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        spin_lock_irqsave(&adapter->stats_lock, flags);
        if (at_read_phy_reg(&adapter->hw, data->reg_num & 0x1F, &data->val_out)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
    case SIOCSMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        if (data->reg_num & ~(0x1F))
            return -EFAULT;
         
        spin_lock_irqsave(&adapter->stats_lock, flags);
    DEBUGOUT1("<at_mii_ioctl> write %x %x", 
            data->reg_num, 
            data->val_in);
        if (at_write_phy_reg(&adapter->hw, data->reg_num, data->val_in)) {
            spin_unlock_irqrestore(&adapter->stats_lock, flags);
            return -EIO;
        }
        // ......
        spin_unlock_irqrestore(&adapter->stats_lock, flags);
        break;
        
    default:
        return -EOPNOTSUPP;
    }
    return AT_SUCCESS;
}

#endif

/**
 * at_tx_timeout - Respond to a Tx Hang
 * @netdev: network interface device structure
 **/

static void
at_tx_timeout(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);

    DEBUGFUNC("at_tx_timeout !");

    /* Do the reset outside of interrupt context */
    schedule_work(&adapter->tx_timeout_task);
}

/*
 * If TPD Buffer size equal to 0, PCIE DMAR_TO_INT
 * will assert. We do soft reset <0x1400=1> according 
 * with the SPEC. BUT, it seemes that PCIE or DMA 
 * state-machine will not be reset. DMAR_TO_INT will
 * assert again and again.
 */
static void
at_tx_timeout_task(struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    //struct at_hw* hw = &adapter->hw;
    
    //uint32_t val1, val2;
    
    DEBUGFUNC("at_tx_timeout_task !");

    
    netif_device_detach(netdev);
    
    /*******   disp debug info **********
    val1 = AT_READ_REG(hw, 0x15f0);
    DEBUGOUT1("<15f0> = 0x%x", val1);
    val1 = AT_READ_REG(hw, 0x1540);
    val2 = AT_READ_REG(hw, 0x1544);
    DEBUGOUT1("<1540> = 0x%x <1544> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1548);
    val2 = AT_READ_REG(hw, 0x154c);
    DEBUGOUT1("<1548> = 0x%x <154c> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1550);
    val2 = AT_READ_REG(hw, 0x1554);
    DEBUGOUT1("<1550> = 0x%x <1554> = 0x%x", val1, val2);
    val1 = AT_READ_REG(hw, 0x1558);
    val2 = AT_READ_REG(hw, 0x155c);
    DEBUGOUT1("<1558> = 0x%x <155c> = 0x%x", val1, val2);
    
    DEBUGOUT1("tpd next to clean %d, tpd next to use %d",
        (uint16_t) atomic_read(&adapter->tpd_ring.next_to_clean),
                (uint16_t) atomic_read(&adapter->tpd_ring.next_to_use));

    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 0),
               *((uint32_t*)(adapter->rfd_ring.desc) + 1),
               *((uint32_t*)(adapter->rfd_ring.desc) + 2),
               *((uint32_t*)(adapter->rfd_ring.desc) + 3),
               *((uint32_t*)(adapter->rfd_ring.desc) + 4),
               *((uint32_t*)(adapter->rfd_ring.desc) + 5));

    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 6),
               *((uint32_t*)(adapter->rfd_ring.desc) + 7),
               *((uint32_t*)(adapter->rfd_ring.desc) + 8),
               *((uint32_t*)(adapter->rfd_ring.desc) + 9),
               *((uint32_t*)(adapter->rfd_ring.desc) + 10),
               *((uint32_t*)(adapter->rfd_ring.desc) + 11));
    
    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 12),
               *((uint32_t*)(adapter->rfd_ring.desc) + 13),
               *((uint32_t*)(adapter->rfd_ring.desc) + 14),
               *((uint32_t*)(adapter->rfd_ring.desc) + 15),
               *((uint32_t*)(adapter->rfd_ring.desc) + 16),
               *((uint32_t*)(adapter->rfd_ring.desc) + 17));
    
    DEBUGOUT1("rfd ring: 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x",
               *((uint32_t*)(adapter->rfd_ring.desc) + 18),
               *((uint32_t*)(adapter->rfd_ring.desc) + 19),
               *((uint32_t*)(adapter->rfd_ring.desc) + 20),
               *((uint32_t*)(adapter->rfd_ring.desc) + 21),
               *((uint32_t*)(adapter->rfd_ring.desc) + 22),
               *((uint32_t*)(adapter->rfd_ring.desc) + 23));
    */
    at_down(adapter);
    at_up(adapter);
    netif_device_attach(netdev);
}
/**
 * at_link_chg_task - deal with link change event Out of interrupt context
 * @netdev: network interface device structure
 **/
static void
at_link_chg_task(struct net_device* netdev)
{
    struct at_adapter * adapter = netdev_priv(netdev);
    unsigned long flags;
    DEBUGFUNC("at_link_chg_task !");
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    
    at_check_link(adapter);
    
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
}

static void
at_check_for_link(struct at_adapter* adapter)
{
    struct net_device *netdev = adapter->netdev;
    uint16_t phy_data = 0;

    DEBUGFUNC("at_check_for_link!");
    
    spin_lock(&adapter->stats_lock);
    adapter->phy_timer_pending = FALSE;
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    at_read_phy_reg(&adapter->hw, MII_BMSR, &phy_data);
    spin_unlock(&adapter->stats_lock);
    
    DEBUGOUT1("MII_BMSR=%x <at_check_for_link>", phy_data);
    
    // notify upper layer link down ASAP
    if (!(phy_data&BMSR_LSTATUS)) { // Link Down
        if (netif_carrier_ok(netdev)) { // old link state: Up
            printk(KERN_INFO
                   "%s: %s NIC Link is Down\n",
		   			at_driver_name,
                    netdev->name );
            adapter->link_speed = SPEED_0;
            netif_carrier_off(netdev);
            netif_stop_queue(netdev);
        }
    }
    schedule_work(&adapter->link_chg_task);
}

static inline void
at_clear_phy_int(struct at_adapter* adapter)
{
    uint16_t phy_data;
        spin_lock(&adapter->stats_lock);
    at_read_phy_reg(&adapter->hw, 19, &phy_data);
	spin_unlock(&adapter->stats_lock);
}


/**
 * at_intr - Interrupt Handler
 * @irq: interrupt number
 * @data: pointer to a network interface device structure
 * @pt_regs: CPU registers structure
 **/

static irqreturn_t
at_intr(int irq, void *data, struct pt_regs *regs)
{
    struct at_adapter *adapter = ((struct net_device*)data)->priv;
    uint32_t status;
    
    status = AT_READ_REG(&adapter->hw, REG_ISR);
    if (0 == status)    
        return IRQ_NONE;
       
    // link event
    if (status&ISR_PHY) {
        at_clear_phy_int(adapter);
    }
    
    // clear ISR status, and Enable CMB DMA/Disable Interrupt
    AT_WRITE_REG(&adapter->hw, REG_ISR, status|ISR_DIS_INT);

    
    // check if PCIE PHY Link down
   if (status&ISR_PHY_LINKDOWN) {
        DEBUGOUT1("pcie phy linkdown %x", status);
        if(netif_running(adapter->netdev)) { // reset MAC
            AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
            AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
            schedule_work(&adapter->pcie_dma_to_rst_task);
            return IRQ_HANDLED; 
        }
    }

    // check if DMA read/write error ?
    if (status&(ISR_DMAR_TO_RST|ISR_DMAW_TO_RST)) 
    {
        DEBUGOUT1("PCIE DMA RW error (status = 0x%x) !", status);
        //AT_WRITE_REG(&adapter->hw, REG_MASTER_CTRL, MASTER_CTRL_SOFT_RST);
        AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
        AT_WRITE_REG(&adapter->hw, REG_IMR, 0);
        schedule_work(&adapter->pcie_dma_to_rst_task);
        return IRQ_HANDLED; 
    }
    
    // link event
    if (status&ISR_PHY)
    {
        adapter->net_stats.tx_carrier_errors++;
        at_check_for_link(adapter);
    }
    
    
    // transmit event
    if ( status&ISR_TX_EVENT ) {
        at_intr_tx(adapter);
    }
    
    // rx exception
    if ( status& ISR_RX_EVENT ) {
        at_intr_rx(adapter);
    }
    
    // re-enable Interrupt
    AT_WRITE_REG(&adapter->hw, REG_ISR, 0);
    return IRQ_HANDLED;
}

static void
at_intr_tx(struct at_adapter* adapter)
{
    uint32_t txd_read_ptr;
    uint32_t txs_write_ptr;
    tx_pkt_status_t* txs;
    tx_pkt_header_t* txph;
    int free_hole = 0;  
    
//    DEBUGFUNC("at_intr_tx"); 

    do {
        txs_write_ptr = (uint32_t) atomic_read(&adapter->txs_write_ptr);
        txs = adapter->txs_ring + txs_write_ptr;
        if (!txs->update) 
            break; // tx stop here
            
        free_hole = 1;
        txs->update = 0;
       	
        if (++txs_write_ptr == adapter->txs_ring_size)
            txs_write_ptr = 0;
        atomic_set(&adapter->txs_write_ptr, (int)txs_write_ptr);
                
        txd_read_ptr = (uint32_t) atomic_read(&adapter->txd_read_ptr);
        txph = (tx_pkt_header_t*) 
                    (((uint8_t*)adapter->txd_ring) + txd_read_ptr);
        
        if ( txph->pkt_size != txs->pkt_size) {
		
            tx_pkt_status_t* old_txs = txs;
            printk(KERN_WARNING
                    "%s: txs packet size do not coinsist with txd"
		    " txd_:0x%08x, txs_:0x%08x!\n",
		    adapter->netdev->name,
		    *(uint32_t*)txph, *(uint32_t*)txs);
	    printk(KERN_WARNING
			    "txd read ptr: 0x%x\n",
			    txd_read_ptr);
	    txs = adapter->txs_ring + txs_write_ptr;
            printk(KERN_WARNING
			    "txs-behind:0x%08x\n",
			    *(uint32_t*)txs);
            if (txs_write_ptr < 2) {
		    txs = adapter->txs_ring + (adapter->txs_ring_size+
				               txs_write_ptr - 2);
	    } else {
		    txs = adapter->txs_ring + (txs_write_ptr - 2);
	    }
	    printk(KERN_WARNING
			    "txs-before:0x%08x\n",
			    *(uint32_t*)txs);
	    txs = old_txs;
        }
        
        txd_read_ptr += (((uint32_t)(txph->pkt_size)+7)& ~3);//4for TPH
        if (txd_read_ptr >= adapter->txd_ring_size)
            txd_read_ptr -= adapter->txd_ring_size;
        
        atomic_set(&adapter->txd_read_ptr, (int)txd_read_ptr);
        
        // tx statistics:
        if (txs->ok)            adapter->net_stats.tx_packets++;
        else                    adapter->net_stats.tx_errors++;
        if (txs->defer)         adapter->net_stats.collisions++;
        if (txs->abort_col)     adapter->net_stats.tx_aborted_errors++;
        if (txs->late_col)      adapter->net_stats.tx_window_errors++;
        if (txs->underun)       adapter->net_stats.tx_fifo_errors++;
    } while (1);
       
    if (free_hole) {
        if(netif_queue_stopped(adapter->netdev) && 
            netif_carrier_ok(adapter->netdev))
            netif_wake_queue(adapter->netdev);
    }
}       


static void
at_intr_rx(struct at_adapter* adapter)
{
    struct net_device *netdev = adapter->netdev;
    rx_desc_t* rxd;
    struct sk_buff* skb;
    
    
    do {
        rxd = adapter->rxd_ring+adapter->rxd_write_ptr;
        if (!rxd->status.update)
            break; // end of tx
	// clear this flag at once
        rxd->status.update = 0; 
        
        if (rxd->status.ok && rxd->status.pkt_size >= 60) {
            int rx_size = (int)(rxd->status.pkt_size-4);
            // alloc new buffer
            skb = dev_alloc_skb(rx_size+NET_IP_ALIGN);
            if (NULL == skb) {
                printk(KERN_WARNING"%s: Memory squeeze, deferring packet.\n",
                        netdev->name);
                /* We should check that some rx space is free.
                  If not, free one and mark stats->rx_dropped++. */
                adapter->net_stats.rx_dropped++;
                break;
            }
/*           
	    if (rx_size > 1400) {
                int s,c;
		c = 0;
		printk("rx_size= %d\n", rx_size);
		for (s=0; s < 800; s++) {
		    if (0 == c) {
		        printk("%04x ", s);
		    }
		    printk("%02x ", rxd->packet[s]);
		    if (++c == 16) {
			c = 0;
			printk("\n");
		    }
		}
		printk(KERN_WARNING"\n");
	    }
*/
	    
            skb_reserve(skb, NET_IP_ALIGN);
            skb->dev = netdev;
	    eth_copy_and_sum(
	        skb, 
		rxd->packet,
		rx_size, 0);
	    skb_put(skb, rx_size);
	    /*
            memcpy(skb_put(skb, rx_size), 
                    rxd->packet,
                    rx_size);
            */
            skb->protocol = eth_type_trans(skb, netdev);
#ifdef NETIF_F_HW_VLAN_TX
        
	    if(adapter->vlgrp && (rxd->status.vlan)) {
                uint16_t vlan_tag = 
                    (rxd->status.vtag>>4)        |
                    ((rxd->status.vtag&7) << 13) |
                    ((rxd->status.vtag&8) << 9)  ;
                DEBUGOUT1("RXD VLAN TAG<RRD>=0x%04x", rxd->status.vtag);
 	        vlan_hwaccel_rx(skb, adapter->vlgrp, vlan_tag);
            } else 
#endif
            netif_rx(skb);
            adapter->net_stats.rx_bytes += rx_size;
            adapter->net_stats.rx_packets++;
	    netdev->last_rx = jiffies;
	    
        } else { 
            
            adapter->net_stats.rx_errors++; 
            
            if (rxd->status.ok && rxd->status.pkt_size <= 60) {
                adapter->net_stats.rx_length_errors++;
            }
            if (rxd->status.mcast)  adapter->net_stats.multicast++;
            if (rxd->status.crc)    adapter->net_stats.rx_crc_errors++;
            if (rxd->status.align)  adapter->net_stats.rx_frame_errors++;
        }
        
        // advance write ptr
        if (++adapter->rxd_write_ptr == adapter->rxd_ring_size) 
            adapter->rxd_write_ptr = 0;
    } while (1);
    
     
    // update mailbox ?
    adapter->rxd_read_ptr = adapter->rxd_write_ptr;
    AT_WRITE_REGW(&adapter->hw, REG_MB_RXD_RD_IDX, adapter->rxd_read_ptr);
}

inline 
int
TxsFreeUnit(struct at_adapter* adapter)
{
    uint32_t txs_write_ptr = (uint32_t) atomic_read(&adapter->txs_write_ptr);
        
    return 
        (adapter->txs_next_clear >= txs_write_ptr) ?
        (int) (adapter->txs_ring_size - adapter->txs_next_clear 
                        + txs_write_ptr - 1) :
        (int) (txs_write_ptr - adapter->txs_next_clear - 1);    
}

inline
int
TxdFreeBytes(struct at_adapter* adapter)
{
    uint32_t txd_read_ptr = (uint32_t)atomic_read(&adapter->txd_read_ptr);
    
    return (adapter->txd_write_ptr >= txd_read_ptr) ?
            (int) (adapter->txd_ring_size - adapter->txd_write_ptr 
                        + txd_read_ptr - 1):
            (int) (txd_read_ptr - adapter->txd_write_ptr - 1);
}



static int
at_xmit_frame(struct sk_buff *skb, struct net_device *netdev)
{
    struct at_adapter *adapter = netdev_priv(netdev);
    unsigned long flags;
    tx_pkt_header_t* txph;
    uint32_t offset, copy_len;
    int txs_unused;
    int txbuf_unused;
    
    //DEBUGFUNC("at_xmit_frame");    
    
    if(skb->len <= 0) {
        dev_kfree_skb_any(skb);
        return NETDEV_TX_OK;
    }
    
#ifdef NETIF_F_LLTX
    local_irq_save(flags);
    if(!spin_trylock(&adapter->tx_lock)) {
        /* Collision - tell upper layer to requeue */
        local_irq_restore(flags);
        return NETDEV_TX_LOCKED;
    }
#else
    spin_lock_irqsave(&adapter->tx_lock, flags);
#endif

    txs_unused = TxsFreeUnit(adapter);
    txbuf_unused = TxdFreeBytes(adapter);

    if (txs_unused < 1 || skb->len > txbuf_unused) {
        // no enough resource
        netif_stop_queue(netdev);
        spin_unlock_irqrestore(&adapter->tx_lock, flags);
	DEBUGOUT("tx busy!!");
        return NETDEV_TX_BUSY;
    }

    offset = adapter->txd_write_ptr;

    txph = (tx_pkt_header_t*) 
            (((uint8_t*)adapter->txd_ring)+offset);
    
    *(uint32_t*)txph = 0; 
    txph->pkt_size = skb->len;

    offset += 4;
    if (offset >= adapter->txd_ring_size)
	    offset -= adapter->txd_ring_size;
    copy_len = adapter->txd_ring_size - offset;
    if (copy_len >= skb->len) {
        memcpy(((uint8_t*)adapter->txd_ring)+offset,
		skb->data,
		skb->len);
	offset += ((uint32_t)(skb->len+3)&~3);
    } else {
	memcpy(((uint8_t*)adapter->txd_ring)+offset,
		skb->data,
		copy_len);
	memcpy((uint8_t*)adapter->txd_ring,
		skb->data+copy_len,
		skb->len-copy_len);
	offset = ((uint32_t)(skb->len-copy_len+3)&~3);
    }
    
#ifdef NETIF_F_HW_VLAN_TX
    if (adapter->vlgrp && vlan_tx_tag_present(skb)) {
        uint16_t vlan_tag = vlan_tx_tag_get(skb);
        vlan_tag = (vlan_tag << 4)          |
                   (vlan_tag >> 13)         |
                   ((vlan_tag >>9) & 0x8)   ;
        txph->ins_vlan = 1;
	txph->vlan = vlan_tag;
        DEBUGOUT1("TXD VLAN TAG<TPD>=%04x", vlan_tag);
    }
#endif    
    if (offset >= adapter->txd_ring_size)
	    offset -= adapter->txd_ring_size;
    adapter->txd_write_ptr = offset;

    // clear txs before send
    adapter->txs_ring[adapter->txs_next_clear].update = 0;
    if (++adapter->txs_next_clear == adapter->txs_ring_size)
        adapter->txs_next_clear = 0;
/*
    printk("txd header: 0x%08x, txd write ptr: %08x\n", 
		    *(uint32_t*)txph,
		    adapter->txd_write_ptr);
*/		    
    
    AT_WRITE_REGW(  &adapter->hw, 
                    REG_MB_TXD_WR_IDX, 
                    (adapter->txd_write_ptr>>2));
                    
    spin_unlock_irqrestore(&adapter->tx_lock, flags);
    
    netdev->trans_start = jiffies;
    dev_kfree_skb_any(skb);
    return NETDEV_TX_OK;
}

/**
 * at_phy_config - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_phy_config(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
    struct at_hw *hw = &adapter->hw;  
    unsigned long flags;

     DEBUGFUNC("at_phy_reconfig!");
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    adapter->phy_timer_pending = FALSE;
    at_write_phy_reg(hw, MII_ADVERTISE, hw->mii_autoneg_adv_reg);
    DEBUGOUT("4 register written");
    at_write_phy_reg(hw, MII_BMCR, MII_CR_RESET|MII_CR_AUTO_NEG_EN);
    spin_unlock_irqrestore(&adapter->stats_lock, flags);
}


/**
 * at_watchdog - Timer Call-back
 * @data: pointer to netdev cast into an unsigned long
 **/

static void
at_watchdog(unsigned long data)
{
    struct at_adapter *adapter = (struct at_adapter *) data;
    uint32_t drop_rxd, drop_rxs;
    unsigned long flags;
    
    spin_lock_irqsave(&adapter->stats_lock, flags);
    drop_rxd = AT_READ_REG(&adapter->hw, REG_STS_RXD_OV);
    drop_rxs = AT_READ_REG(&adapter->hw, REG_STS_RXS_OV);
    adapter->net_stats.rx_over_errors += (drop_rxd+drop_rxs);

    spin_unlock_irqrestore(&adapter->stats_lock, flags);
    
    /* Reset the timer */
    mod_timer(&adapter->watchdog_timer, jiffies + 4 * HZ);
}



