/** 
 * @file IxOsServicesMem.c
 * 
 * @brief Implementation of memory mapping/unmapping routines
 * 
 * @par
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 1999-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

/* Access to the global memory map in IxOsServicesMemMap.h 
   is only allowed in this file */
#define IxOsServicesMem_C

#include <IxOsServices.h>
#include <IxOsCacheMMU.h>
#include <IxOsServicesMemMap.h>
 
#define SEARCH_PHYSICAL_ADDRESS (1)
#define SEARCH_VIRTUAL_ADDRESS  (2)

#ifdef DEBUG
#define IX_OSSERV_LOG ixOsServLog
#else
#define IX_OSSERV_LOG if(0) ixOsServLog
#endif /* DEBUG */

/*
 * private function prototypes
 */
PRIVATE IxOsServMemoryMap *
ixOsServMemMapFind(UINT32 requestedAddress, UINT32 size, UINT32 searchCriteria, UINT32 requestedCoherency);

#ifdef __linux

/* Linux-specific map/unmap functions to be used with dynamic maps */
#include <asm/io.h>
#include <linux/ioport.h>

PUBLIC void
ixOsServLinuxMemMap(IxOsServMemoryMap *map)
{
  map->virtualAddress = (UINT32) ioremap(map->physicalAddress, map->size);
}

PUBLIC void
ixOsServLinuxMemUnmap(IxOsServMemoryMap *map)
{
  iounmap((void *) map->virtualAddress);
  map->virtualAddress = 0;
}

#endif /* __linux */

/*
 * Searches for map using one of the following criteria:
 * 
 * - enough room to include a zone starting with the physical "requestedAddress" of size "size" (for mapping)
 * - includes the virtual "requestedAddress" in its virtual address space (already mapped, for unmapping)
 * - correct coherency
 *
 * Returns a pointer to the map or NULL if a suitable map is not found.
 */
PRIVATE IxOsServMemoryMap *
ixOsServMemMapFind(UINT32 requestedAddress, UINT32 size, UINT32 searchCriteria, UINT32 requestedCoherency)
{
  UINT32 mapIndex;
  
  for (mapIndex = 0 ; mapIndex < IX_OSSERV_MEM_MAP_ELEMENTS ; mapIndex++)
  { 
    IxOsServMemoryMap *map = &ixOsServGlobalMemoryMap[mapIndex];

    if (searchCriteria == SEARCH_PHYSICAL_ADDRESS
        && requestedAddress >= map->physicalAddress && (requestedAddress + size) <= (map->physicalAddress + map->size)
        && (map->coherency & requestedCoherency) != 0)
    {
      return map;
    }
    else if (searchCriteria == SEARCH_VIRTUAL_ADDRESS
             && requestedAddress >= map->virtualAddress && requestedAddress <= (map->virtualAddress + map->size)
             && (map->coherency & requestedCoherency) != 0)
    {
      return map;
    }
    else if (searchCriteria == SEARCH_PHYSICAL_ADDRESS)
    {
      IX_OSSERV_LOG(LOG_USER, "OsServices: Checking [phys addr 0x%x:size 0x%x:coherency %d]\n", map->physicalAddress, map->size, map->coherency, 0, 0, 0);
    }
  }

  /* not found */
  return NULL;
}

PUBLIC UINT32
ixOsServMemVirtToPhys(UINT32 virtualAddress, UINT32 requestedCoherency)
{
  IxOsServMemoryMap *map = ixOsServMemMapFind(virtualAddress, 0, SEARCH_VIRTUAL_ADDRESS, requestedCoherency);
  
  if (map != NULL)
  {
    return map->physicalAddress + virtualAddress - map->virtualAddress;
  }
  else
  {
    return (UINT32) IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(virtualAddress);
  }
}

/*
 * This function maps an I/O mapped physical memory zone of the given size
 * into a virtual memory zone accessible by the caller and returns a cookie - 
 * the start address of the virtual memory zone. 
 * IX_MMU_PHYS_TO_VIRT_TRANSLATION should NOT therefore be used on the returned 
 * virtual address.
 * The memory zone is to be unmapped using ixOsServMemUnmap once the caller has
 * finished using this zone (e.g. on driver unload) using the cookie as 
 * parameter.
 * The IX_OSSERV_READ/WRITE_LONG/SHORT macros should be used to read and write 
 * the mapped memory, adding the necessary offsets to the address cookie.
 *
 * Note: this function is not to be used directly. Use IX_OSSERV_MEM_MAP 
 * instead.
 */
PUBLIC void *
ixOsServMemMap(UINT32 requestedAddress, UINT32 size, UINT32 requestedCoherency)
{ 
  IxOsServMemoryMap *map;

  IX_OSSERV_LOG(LOG_USER, "OsServices: Mapping [addr 0x%x:size 0x%x:coherency %d]\n", requestedAddress, size, requestedCoherency, 0, 0, 0);

  map = ixOsServMemMapFind(requestedAddress, size, SEARCH_PHYSICAL_ADDRESS, requestedCoherency);

  if (map != NULL)
  {
    UINT32 offset = requestedAddress - map->physicalAddress;

    IX_OSSERV_LOG(LOG_USER, "OsServices: Found map [", 0, 0, 0, 0, 0, 0);
    IX_OSSERV_LOG(LOG_USER, map->name, 0, 0, 0, 0, 0, 0);
    IX_OSSERV_LOG(LOG_USER, ":addr 0x%x: virt 0x%x:size 0x%x:ref %d:coherency %d]\n", map->physicalAddress, map->virtualAddress,
        map->size, map->refCount, map->coherency, 0);

    if (map->type == IX_DYNAMIC_MAP && map->virtualAddress == 0)
    {
      if (map->mapFunction != NULL)
      {
        map->mapFunction(map);

        if (map->virtualAddress == 0)
        {
          /* failed */
          ixOsServLog(LOG_FATAL, "OsServices: Remap failed - [addr 0x%x:size 0x%x:coherency %d]\n", requestedAddress, size, requestedCoherency, 0, 0, 0);
          return NULL;
        }
      }
      else
      {
        /* error, no map function for a dynamic map */
        ixOsServLog(LOG_FATAL, "OsServices: No map function for a dynamic map - [addr 0x%x:size 0x%x:coherency %d]\n", requestedAddress, size, requestedCoherency, 0, 0, 0);

        return NULL;
      }
    }

    /* increment reference count */
    map->refCount++;

    return (void *)(map->virtualAddress + offset);
  }

  /* requested address is not described in the global memory map */
  ixOsServLog(LOG_FATAL, "OsServices: No mapping found - [addr 0x%x:size 0x%x:coherency %d]\n", requestedAddress, size, requestedCoherency, 0, 0, 0);
  return NULL;
}


/*
 * This function unmaps a previously mapped I/O memory zone using
 * the cookie obtained in the mapping operation. The memory zone in question
 * becomes unavailable to the caller once unmapped and the cookie should be
 * discarded.
 *
 * This function cannot fail if the given parameter is correct and does not
 * return a value.
 *
 * Note: this function is not to be used directly. Use IX_OSSERV_MEM_UNMAP
 * instead.
 */
PUBLIC void
ixOsServMemUnmap(UINT32 requestedAddress, UINT32 coherency)
{
  IxOsServMemoryMap *map;

  if (requestedAddress == 0)
  {
    /* invalid virtual address */
    return;
  }
  
  map = ixOsServMemMapFind(requestedAddress, 0, SEARCH_VIRTUAL_ADDRESS, coherency);
  
  if (map != NULL)
  {
    if (map->refCount > 0)
    {
      /* decrement reference count */
      map->refCount--;
      
      if (map->refCount == 0)
      {
        /* no longer used, deallocate */
        if (map->type == IX_DYNAMIC_MAP && map->unmapFunction != NULL)
        {
          map->unmapFunction(map);
        }
      }
    }
  }
  else
  {
    ixOsServLog(LOG_WARNING, "OsServices: ixOsServMemUnmap didn't find the requested map [virt addr 0x%x: coherency %d], ignoring call\n", 
        requestedAddress, coherency, 0, 0, 0, 0);
  }
}
