/** @file IxEthDB.h
 *
 * @brief this file contains the public API of @ref IxEthDB component
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 */
 
#ifndef IxEthDB_H
#define IxEthDB_H

#include <IxOsBuffMgt.h>
#include <IxTypes.h>

/**
 * @defgroup IxEthDB IXP425 Ethernet Database (IxEthDB) API
 *
 * @brief ethDB is a library that does provides a MAC address database learning/filtering capability
 *
 *@{
 */

#define INLINE __inline__

#define IX_ETH_DB_PRIVATE PRIVATE /* imported from IxTypes.h */

#define IX_ETH_DB_PUBLIC /* nothing */

/**
 * @brief port ID => message handler NPE id conversion (0 => NPE_B, 1 => NPE_C)
 */
#define IX_ETH_DB_PORT_ID_TO_NPE(id) (id + 1)

/**
 * @def IX_ETH_DB_NPE_TO_PORT_ID(npe)
 * @brief message handler NPE id => port ID conversion (NPE_B => 0, NPE_C => 1)
 */
#define IX_ETH_DB_NPE_TO_PORT_ID(npe) (npe - 1)

/**
 * @def IX_IEEE803_MAC_ADDRESS_SIZE
 * @brief The size of the MAC address
 */
#define IX_IEEE803_MAC_ADDRESS_SIZE (6)

/**
 * @enum IxEthDBStatus
 * @brief Ethernet database status
 */
typedef enum /* IxEthDBStatus */
{
    IX_ETH_DB_SUCCESS = IX_SUCCESS,/**< Return success */
    IX_ETH_DB_FAIL = IX_FAIL,/**< Return fail */
    IX_ETH_DB_INVALID_PORT,/**< invalid port */
    IX_ETH_DB_PORT_UNINITIALIZED, /**< port uninitialized */
    IX_ETH_DB_MAC_UNINITIALIZED, /**< MAC uninitialized */
    IX_ETH_DB_INVALID_ARG, /**< invalid arg */
  IX_ETH_DB_NO_SUCH_ADDR, /**< Address not found for search or delete operations */
  IX_ETH_DB_NOMEM,        /**< Learning database memory full */
  IX_ETH_DB_BUSY,         /**< Learning database is busy */
  IX_ETH_DB_END           /**< Database browser passed the end */
} IxEthDBStatus;
    
/**
 *
 * @brief The IEEE 802.3 Ethernet MAC address structure.
 * 
 * The data should be packed with bytes xx:xx:xx:xx:xx:xx 
 * @note
 * The data must be packed in network byte order.
 */
typedef struct  
{
   UINT8 macAddress[IX_IEEE803_MAC_ADDRESS_SIZE];
} IxEthDBMacAddr;

/**
 * @ingroup IxEthDB
 *
 * @brief Definition of an IXP425 port.
 */
typedef UINT32 IxEthDBPortId;

/**
 * @ingroup IxEthDB
 *
 * @brief Port dependency map definition
 */
typedef UINT32 IxEthDBPortMap;
    
/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBInit(void)
 *
 * @brief Initializes the Ethernet learning/filtering database
 *
 * @retval IX_ETH_DB_SUCCESS initialization was successful
 * @retval IX_ETH_DB_FAIL initialization failed (OSSL error)
 */
IX_ETH_DB_PUBLIC IxEthDBStatus ixEthDBInit(void);

/**
 * @ingroup IxEthDB
 *
 * @fn void ixEthDBPortInit(IxEthDBPortId portID)
 *
 * @brief Initializes a port
 *
 * This function is called automatically by the Ethernet Access
 * component top-level portInit() routine and should be manually
 * called for any user-defined port (any port that is not one of
 * the two Ethernet NPEs). 
 *
 * @param portID ID of the port to be initialized
 *
 * @retval void
 *
 * @see IxEthDBPortDefs.h for port definitions
 */
IX_ETH_DB_PUBLIC void ixEthDBPortInit(IxEthDBPortId portID);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBPortEnable(IxEthDBPortId portID)
 *
 * @brief enable a port
 *
 * This function is called automatically from the Ethernet Access component
 * top-level portEnable() routine and should be manually
 * called for any user-defined port (any port that is not one of
 * the two Ethernet NPEs). 
 *
 * @param portID ID of the port to enable processing on
 *
 * @retval IX_ETH_DB_SUCCESS if enabling is successful
 * @retval IX_ETH_DB_FAIL if the enabling was not successful due to
 * a message handler error
 * @retval IX_ETH_DB_MAC_UNINITIALIZED the MAC address of this port was
 * not initialized (only for Ethernet NPEs)
 *
 * @pre ixEthDBPortAddressSet() needs to be called prior to enabling the port events
 * for Ethernet NPEs
 *
 * @see ixEthDBPortAddressSet()
 *
 * @see IxEthDBPortDefs.h for port definitions
 */
IX_ETH_DB_PUBLIC IxEthDBStatus ixEthDBPortEnable(IxEthDBPortId portID);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBPortDisable(IxEthDBPortId portID)
 *
 * @brief disable processing on a port
 *
 * This function is called automatically from the Ethernet Access component
 * top-level portDisable() routine and should be manually
 * called for any user-defined port (any port that is not one of
 * the two Ethernet NPEs).
 *
 * Note: After Ethernet NPEs are disabled they are stopped therefore
 * when re-enabled they need to be reset, downloaded with microcode and started.
 * For learning to restart working the user needs to call again 
 * ixEthAccPortUnicastMacAddressSet or ixEthDBUnicastAddressSet
 * with the respective port MAC address.
 * Residual MAC addresses learnt before the port was disable will take up to
 * IX_ETH_DB_MAINTENANCE_TIME to be downloaded in the NPE MAC tree after
 * learning is restarted on the port. The dynamic addresses do not dissapear
 * after the disable-enable sequence. They also do not age during the time the
 * port is disabled.
 *
 * @param portID ID of the port to disable processing on
 *
 * @retval IX_ETH_DB_SUCCESS if disabling is successful
 * @retval IX_ETH_DB_FAIL if the disabling was not successful due to
 * a message handler error
 */
IX_ETH_DB_PUBLIC IxEthDBStatus ixEthDBPortDisable(IxEthDBPortId portID);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBPortAddressSet(IxEthDBPortId portID, IxEthDBMacAddr *macAddr)
 *
 * @brief set the port MAC address
 *
 * This function is to be called from the Ethernet Access component top-level
 * ixEthDBUnicastAddressSet(). Event processing cannot be enabled for a port
 * until its MAC address has been set.
 *
 * @param portID ID of the port whose MAC address is set
 * @param macAddr port MAC address
 *
 * @retval IX_ETH_DB_SUCCESS MAC address was set successfully
 * @retval IX_ETH_DB_FAIL MAC address was not set due to a message handler failure
 * @retval IX_ETH_DB_INVALID_PORT if the port is not an Ethernet NPE
 *
 * @see IxEthDBPortDefs.h for port definitions
 */
IX_ETH_DB_PUBLIC IxEthDBStatus ixEthDBPortAddressSet(IxEthDBPortId portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringPortMaximumFrameSizeSet(IxEthDBPortId portID, UINT32 maximumFrameSize)
 *
 * @brief Set the maximum frame size supported on the given port ID
 *
 * This functions set the maximum frame size supported on a specific port ID 
 * 
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to configure
 * @param maximumFrameSize maximum frame size to configure
 *
 * @retval IX_ETH_DB_SUCCESS the port is configured
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_INVALID_ARG parameter is out of range
 *
 * @note
 * This maximum frame size is used to filter the frames based on their
 * destination addresses and the capabilities of the destination port.
 * The mximum value that can be set for a NPE port is 16320.
 * (IX_ETHNPE_ACC_FRAME_LENGTH_MAX)
 */
IxEthDBStatus ixEthDBFilteringPortMaximumFrameSizeSet(IxEthDBPortId portID, UINT32 maximumFrameSize);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringStaticEntryProvision(IxEthDBPortId portID, IxEthDBMacAddr *macAddr)
 *
 * @brief Populate the Ethernet learning/filtering database with a static MAC address
 *
 * Populates the Ethernet learning/filtering database with a static MAC address. The entry will not be subject to aging.
 * If there is an entry (static or dynamic) with the corresponding MAC address on any port this entry will take precedence.
 * Any other entry with the same MAC address will be removed.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 * 
 * @param portID port ID to add the static address to
 * @param macAddr static MAC address to add
 * 
 * @retval IX_ETH_DB_SUCCESS the add was successful
 * @retval IX_ETH_DB_FAIL failed to populate the database entry
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port is not initialized
 */
IxEthDBStatus ixEthDBFilteringStaticEntryProvision(IxEthDBPortId portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringDynamicEntryProvision(IxEthDBPortId portID, IxEthDBMacAddr *macAddr)
 *
 * @brief Populate the Ethernet learning/filtering database with a dynamic MAC address
 *
 * Populates the Ethernet learning/filtering database with a dynamic MAC address. This entry will be subject to normal 
 * aging function, if aging is enabled on its port.
 * If there is an entry (static or dynamic) with the same MAC address on any port this entry will take precedence.
 * Any other entry with the same MAC address will be removed.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to add the dynamic address to
 * @param macAddr static MAC address to add
 *
 * @retval IX_ETH_DB_SUCCESS the add was successful
 * @retval IX_ETH_DB_FAIL failed to populate the database entry
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port is not initialized
 */
IxEthDBStatus ixEthDBFilteringDynamicEntryProvision(IxEthDBPortId portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringEntryDelete(IxEthDBMacAddr *macAddr)
 *
 * @brief Remove a MAC address entry from the Ethernet learning/filtering database
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param macAddr MAC address to remove
 *
 * @retval IX_ETH_DB_SUCCESS the removal was successful
 * @retval IX_ETH_DB_NO_SUCH_ADDR failed to remove the address (not in the database)
 */
IxEthDBStatus ixEthDBFilteringEntryDelete(IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringPortSearch(IxEthDBPortId portID, IxEthDBMacAddr *macAddr)
 *
 * @brief Search the Ethernet learning/filtering database for the given MAC address and port ID
 *
 * This functions searches the database for a specific port ID and MAC address. Both the port ID
 * and the MAC address have to match in order for the record to be reported as found.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to search for
 * @param macAddr MAC address to search for
 *
 * @retval IX_ETH_DB_SUCCESS the record exists in the database
 * @retval IX_ETH_DB_NO_SUCH_ADDR the record was not found in the database
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port ID is not initialized
 */
IxEthDBStatus ixEthDBFilteringPortSearch(IxEthDBPortId portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringDatabaseSearch(IxEthDBPortId *portID, IxEthDBMacAddr *macAddr)
 *
 * @brief Search the Ethernet learning/filtering database for a MAC address and return the port ID
 *
 * Searches the database for a MAC address. The function returns the portID for the 
 * MAC address record, if found. If no match is found the function returns IX_ETH_DB_NO_SUCH_ADDR. 
 * The portID is only valid if the function finds a match.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID the address belongs to (populated only on a successful search)
 * @param macAddr MAC address to search for
 *
 * @retval IX_ETH_DB_SUCCESS the record exists in the database
 * @retval IX_ETH_DB_NO_SUCH_ADDR the record was not found in the database
 */
IxEthDBStatus ixEthDBFilteringDatabaseSearch(IxEthDBPortId *portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringPortUpdatingSearch(IxEthDBPortId *portID, IxEthDBMacAddr *macAddr)
 *
 * @brief Search the filtering database for a MAC address, return the port ID and reset the record age
 *
 * Searches the database for a MAC address. The function returns the portID for the 
 * MAC address record and resets the entry age to 0, if found. 
 * If no match is found the function returns IX_ETH_DB_NO_SUCH_ADDR. 
 * The portID is only valid if the function finds a match.
 *
 * - Reentrant      - yes
 * - ISR Callable   - no
 *
 * @retval IX_ETH_DB_SUCCESS the MAC address was found
 * @retval IX_ETH_DB_NO_SUCH_ADDR the MAC address was not found
 */
IxEthDBStatus ixEthDBFilteringPortUpdatingSearch(IxEthDBPortId *portID, IxEthDBMacAddr *macAddr);

/**
 * @ingroup IxEthDB
 *
 * @def IX_ETH_DB_MAINTENANCE_TIME
 *
 * @brief The @ref ixEthDBDatabaseMaintenance must be called by the user at a frequency of 
 * IX_ETH_DB_MAINTENANCE_TIME
 *
 */
#define IX_ETH_DB_MAINTENANCE_TIME (1 * 60) /* 1 Minute */

/**
 * @ingroup IxEthDB
 *
 * @def IX_ETH_DB_LEARNING_ENTRY_AGE_TIME
 *
 * @brief The define specifies the filtering database age entry time. Static entries older than
 * IX_ETH_DB_LEARNING_ENTRY_AGE_TIME +/- IX_ETH_DB_MAINTENANCE_TIME shall be removed.
 *
 */
#define IX_ETH_DB_LEARNING_ENTRY_AGE_TIME (15 * 60 ) /* 15 Mins */

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBPortAgingDisable(IxEthDBPortId portID)
 *
 * @brief Disable the aging function for a specific port
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to disable aging on
 *
 * @retval IX_ETH_DB_SUCCESS aging disabled successfully
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port ID is not initialized
 */
IxEthDBStatus ixEthDBPortAgingDisable(IxEthDBPortId port);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBPortAgingEnable(IxEthDBPortId portID)
 *
 * @brief Enable the aging function for a specific port
 * 
 * Enables the aging of dynamic MAC address entries stored in the learning/filtering database
 * 
 * @note The aging function relies on the @ref ixEthDBDatabaseMaintenance being called with a period of 
 * @ref IX_ETH_DB_MAINTENANCE_TIME seconds.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to enable aging on
 *
 * @retval IX_ETH_DB_SUCCESS aging enabled successfully
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port ID is not initialized
 */
IxEthDBStatus ixEthDBPortAgingEnable(IxEthDBPortId portID);

/**
 * @ingroup IxEthDB
 *
 * @fn void ixEthDBDatabaseMaintenance(void)
 *
 * @brief Performs a maintenance operation on the Ethernet learning/filtering database
 * 
 * In order to perform a database maintenance this function must be called every
 * @IX_ETH_DB_MAINTENANCE_TIME seconds. It should be called regardless of whether learning is
 * enabled or not.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 * 
 * @retval void
 */
void ixEthDBDatabaseMaintenance(void);

/**
 * @ingroup IxEthDB
 *
 * @fn IxEthDBStatus ixEthDBFilteringDatabaseShow(IxEthDBPortId  portID)
 *
 * @brief This function displays the Mac Ethernet MAC address filtering tables.
 *
 * It displays the MAC address, port ID, entry type (dynamic/static),and age for 
 * the given port ID.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portID port ID to display the MAC address entries
 *
 * @retval IX_ETH_DB_SUCCESS operation completed successfully
 * @retval IX_ETH_DB_INVALID_PORT portID is invalid
 * @retval IX_ETH_DB_PORT_UNINITIALIZED port ID is not initialized
 */
IxEthDBStatus ixEthDBFilteringDatabaseShow(IxEthDBPortId portID);

/** 
 * @ingroup IxEthDB
 *
 * @fn void ixEthDBFilteringDatabaseShowAll(void)
 *
 * @brief Displays the MAC address recorded in the filtering database for all registered
 * ports (see IxEthDBPortDefs.h), grouped by port ID.
 *
 * @retval void
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 */
void ixEthDBFilteringDatabaseShowAll(void);

/**
 * @}
 */

#endif /* IxEthDB_H */
